/*-----------------------------------------------------------------------

                         SYRTHES version 3.4
                         -------------------

     This file is part of the SYRTHES Kernel, element of the
     thermal code SYRTHES.

     Copyright (C) 1988-2008 EDF S.A., France

     contact: syrthes-support@edf.fr


     The SYRTHES Kernel is free software; you can redistribute it
     and/or modify it under the terms of the GNU General Public License
     as published by the Free Software Foundation; either version 2 of
     the License, or (at your option) any later version.

     The SYRTHES Kernel is distributed in the hope that it will be
     useful, but WITHOUT ANY WARRANTY; without even the implied warranty
     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.


     You should have received a copy of the GNU General Public License
     along with the Code_Saturne Kernel; if not, write to the
     Free Software Foundation, Inc.,
     51 Franklin St, Fifth Floor,
     Boston, MA  02110-1301  USA

-----------------------------------------------------------------------*/
/*============================================================================*
 * Traitement de fichiers binaires Fortran de maniere portable sur les        *
 * architectures 'big endian' et 'little endian' (en prenant le cas           *
 * 'big endian' comme reference) depuis des programmes en C ou Fortran        *
 * utilisant la librairie C standard                                          *
 *                                                                            *
 * Bibliotheque : Code_Saturne, SYRTHES                    Copyright EDF 2000 *
 *============================================================================*/

/* Includes systeme */

#include <assert.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* Includes librairie */

#include "fortran_c.h"
#include "fic_bin_f.h"


/* Positionner le code d'erreur pour un fichier donne */
#define FIC_ERR_TST(ficptr) \
((integer_t)(ferror(ficptr) == 0 ? \
(feof(ficptr) == 0 ? 0 : FIC_ERR_FIN) : ferror(ficptr)))


#ifdef FORTRAN_WRAPPER /* Si le code doit etre appele depuis du Fortran */

/* Variable globale : tableau pour correspondance unite logique/pointeur */

fic_bin_t  fic_bin_f_ptr_unit[100];

#endif /* FORTRAN_WRAPPER */


/*============================================================================*/
/*                        Prototypes de fonctions privees                     */
/*============================================================================*/

/*----------------------------------------------------------------------------*
 * Permutation des octets pour passage de "little endian" a "big endian"      *
 *----------------------------------------------------------------------------*/

static void fic_bin_f__endswap
(
 void  *buf,   /* Tampon contenant les elements                               */
 size_t size,  /* Taille d'un element                                         */
 size_t nitems /* Nombre d'elements                                           */
);


/*============================================================================*/
/*                              Fonctions publiques                           */
/*============================================================================*/

/*----------------------------------------------------------------------------*
 * Ouverture d'un fichier                                                     *
 *----------------------------------------------------------------------------*/

#ifdef FORTRAN_WRAPPER

/* Equivalent Fortran :

   subroutine openbf(unit, ierror)

   char(*) nomfic   !  -> Nom du fichier
   integer lnom     !  -> Longueur du nom du fichier
   char    modfic   !  -> Mode d'ouverture demande ('r', 'w', ou 'a')
   integer lmod     !  -> Longueur du nom du mode d'ouverture
   integer unit     !  -> Numero d'unite logique du fichier (0 a 99)
   integer ierror   ! <-  Code de retour (different de 0 en cas d'erreur)
*/

#if defined(CHAINE_F_VERS_C_LEN_FIN)
void fsymbol(openbf, OPENBF)
(
 char      *nomfic, /*  -> Nom du fichier                                     */
 integer_t *lnom  , /*  -> Longueur du nom de fichier                         */
 char      *modfic, /*  -> Mode d'ouverture demande ('r', 'w', ou 'a')        */
 integer_t *lmod  , /*  -> Longueur de la chaine de mode d'ouverture          */
 integer_t *unit  , /* <-  Numero d'unite logique du fichier (0 a 99)         */
 integer_t *ierror, /* <-  Code de retour                                     */
 integer_t  lennom, /*  -> Longueur du nom de fichier                         */
 integer_t  lenmod  /*  -> Longueur de la chaine de mode d'ouverture          */
)
#elif defined(CHAINE_F_VERS_C_STR_NUL)
void fsymbol(openbf, OPENBF)
(
 char      *nomfic, /*  -> Nom du fichier                                     */
 integer_t *lnom  , /*  -> Longueur du nom de fichier                         */
 char      *modfic, /*  -> Mode d'ouverture demande ('r', 'w', ou 'a')        */
 integer_t *lmod  , /*  -> Longueur de la chaine de mode d'ouverture          */
 integer_t *unit  , /* <-  Numero d'unite logique du fichier (0 a 99)         */
 integer_t *ierror  /* <-  Code de retour                                     */
)
#endif
{
  int        i, lmaxch = LEN_MAX_NOM_FIC_BIN_F;
  char       chaine[LEN_MAX_NOM_FIC_BIN_F + 1];
  fic_bin_t  fic;

  if (*lnom < lmaxch)
    lmaxch = *lnom;

  for (i = 0;
       i <  lmaxch && (nomfic[i] != ' ' && nomfic[i] != '\0');
       i++) chaine[i] = nomfic[i];

  if (i < LEN_MAX_NOM_FIC_BIN_F) {

    chaine[i] = '\0';
    *ierror = ouvre_fic_bin_f(chaine, *modfic, &fic);

    fic_bin_f_ptr_unit[(int)(*unit)] = fic;

  }
  else {

    *ierror = FIC_ERR_LEN_NOM;
    fic.ptr = NULL;
    fic.end_swap = 0;
    fic_bin_f_ptr_unit[(int)(*unit)] = fic;

  }

}

#endif /* FORTRAN_WRAPPER */


integer_t ouvre_fic_bin_f
(
 char      *nomfic, /*  -> Nom du fichier                                     */
 char       modfic, /*  -> Mode d'ouverture demande ('r', 'w', ou 'a')        */
 fic_bin_t *fic     /* <-  Pointeur sur le descripteur de fichier             */
)
{
  int int_endian;

  switch (modfic) {

  case 'r':

    fic->ptr = fopen(nomfic, "rb");
    break ;

  case 'w':

    fic->ptr = fopen(nomfic, "wb");
    break ;

  case 'a':

    fic->ptr = fopen(nomfic, "ab");
    break ;

  default:

    assert (modfic == 'r' || modfic == 'w' || modfic == 'a');

  }

  /* Detection systeme "big-endian/litte-endian" */

  int_endian = 0 ;
  *((char *) (&int_endian)) = '\1' ;

  if (int_endian == 1)
    fic->end_swap = 1;
  else
    fic->end_swap = 0;

  /* Retour selon erreur ou reussite */

  if (fic->ptr != NULL)
    return 0;
  else
    return errno;
}


/*----------------------------------------------------------------------------*
 * Fermeture d'un fichier                                                     *
 *----------------------------------------------------------------------------*/

#ifdef FORTRAN_WRAPPER

/* Equivalent Fortran :

   subroutine closbf(unit, ierror)
   
   integer unit     !  -> Numero d'unite logique du fichier (0 a 99)
   integer ierror   ! <-  Code de retour (different de 0 en cas d'erreur)
*/

void fsymbol(closbf, CLOSBF)
(
 integer_t *unit  , /*  -> Numero d'unite logique du fichier (0 a 99)         */
 integer_t *ierror  /* <-  Code de retour                                     */
)
{
  fic_bin_t fic = fic_bin_f_ptr_unit[(int)(*unit)];

  assert (*unit > -1 && *unit < 100);

  *ierror = ferme_fic_bin_f(&fic);

  if (*ierror == 0)
    fic_bin_f_ptr_unit[(int)(*unit)] = fic;
}

#endif /* FORTRAN_WRAPPER */


integer_t ferme_fic_bin_f
(
 fic_bin_t *fic     /*  -> Pointeur sur le descripteur de fichier             */
)
{
  fic->end_swap = 0;

  return (integer_t)fclose(fic->ptr);
}


/*----------------------------------------------------------------------------*
 * Revenir au debut d'un fichier                                              *
 *----------------------------------------------------------------------------*/

#ifdef FORTRAN_WRAPPER

/* Equivalent Fortran :

   subroutine rewdbf(unit, ierror)
   
   integer unit     !  -> Numero d'unite logique du fichier (0 a 99)
   integer ierror   ! <-  Code de retour (different de 0 en cas d'erreur)
*/

void fsymbol(rewdbf, REWDBF)
(
 integer_t *unit  , /*  -> Numero d'unite logique du fichier (0 a 99)         */
 integer_t *ierror  /* <-  Code de retour                                     */
)
{
  fic_bin_t fic = fic_bin_f_ptr_unit[(int)(*unit)];

  *ierror = debut_fic_bin_f(&fic);
}

#endif /* FORTRAN_WRAPPER */

integer_t debut_fic_bin_f
(
 fic_bin_t *fic     /*  -> Pointeur sur le descripteur de fichier             */
)
{
  if (fseek(fic->ptr, 0L, SEEK_SET) == 0)
    return 0;
  else
    return errno;
}


/*----------------------------------------------------------------------------*
 * Tester si un fichier binaire est "big endian" ou "little endian".          *
 * On suppose que l'on connait la longueur du premier enregistrement.         *
 * Cette fonction doit s'appeller juste apres l'ouverture du fichier, et      *
 * configure le fichier de maniere a ce que les lectures ulterieures se       *
 * se fassent avec ou sans permutation des octets en fonction de ce test.     *
 *----------------------------------------------------------------------------*/

#ifdef FORTRAN_WRAPPER

/* Equivalent Fortran :

   subroutine tendbf(unit, ierror)

   integer unit     !  -> Numero d'unite logique du fichier (0 a 99)
   integer taille   !  -> Taille d'un element
   integer ierror   ! <-  Code de retour (different de 0 en cas d'erreur)
*/

void fsymbol(tendbf, OPENBF)
(
 integer_t *unit  , /*  -> Numero d'unite logique du fichier (0 a 99)         */
 integer_t *taille, /*  -> Taille des elements                                */
 integer_t *ierror  /* <-  Code de retour                                     */
)
{
  fic_bin_t fic = fic_bin_f_ptr_unit[(int)(*unit)];

  assert (*unit > -1 && *unit < 100);

  *ierror = test_end_fic_bin_f(&fic, *taille);
  fic_bin_f_ptr_unit[(int)(*unit)]=fic;
}

#endif /* FORTRAN_WRAPPER */


integer_t test_end_fic_bin_f
(
 fic_bin_t *fic,    /*  -> Pointeur sur le descripteur de fichier             */
 int        trec    /*  -> Taille du premier l'enregistrement                 */
)
{
  integer_4_t   itmp;

  if (fread(&itmp, 4, 1, fic->ptr) != 1)
    return FIC_ERR_TST(fic->ptr);

  fic->end_swap = 0;

  if ((int)itmp != trec) {

    fic_bin_f__endswap(&itmp, sizeof(integer_4_t), 1);

    if ((int)itmp == trec)
      fic->end_swap = 1;
    else
      return FIC_ERR_CONTENU;

  }

  return debut_fic_bin_f(fic);

}


/*----------------------------------------------------------------------------*
 * Lecture d'un enregistrement binaire IEEE de type Fortran ;                 *
 *                                                                            *
 * On rappelle qu'un enregistrement commence et se termine par un marqueur    *
 * (entier code sur 4 octets indiquant la taille en octets des donnees        *
 * contenues dans l'enregistrement), permettant certaines operations de type  *
 * lecture a vide ou retour en arriere d'un enregistrement.                   *
 *                                                                            *
 * On fournit en argument un parametre mode indiquant le type de lecture :    *
 *                                                                            *
 *  mode = 0 : Lecture complete de l'enregistrement                           *
 *         1 : Lecture partielle de 'nombre' d'elements                       *
 *         2 : Saut a la fin de l'enregistement                               *
 *                                                                            *
 * On doit donc indiquer pour les modes 1 et 2 le nombre d'elements du meme   *
 * enregistrement deja lus lors d'appels precedents a ce sous-programme, et   *
 * si ce nombre est non nul, le nombre total d'elements de l'enregistrement   *
 * (calcule a la premiere lecture).                                           *
 *----------------------------------------------------------------------------*/

#ifdef FORTRAN_WRAPPER

/* Equivalent Fortran :

   subroutine readbf(unit, taille, nombre, mode, nbrlus, nbrtot, elems, ierror)
   
   integer unit     !  -> Numero d'unite logique du fichier (0 a 99)
   integer taille   !  -> Taille d'un element
   integer nombre   !  -> Nombre d'elements a lire au cours de cet appel
   integer mode     !  -> Mode de lecture (0, 1, ou 2)
   integer nbrlus   ! <-> Nombre d'elements de l'enregistrement deja lus
   integer nbrtot   ! <-> Nombre total d'elements de l'enregistrement
   ?       elems    ! <-> Tableau des elements
   integer ierror   ! <-  Code de retour (different de 0 en cas d'erreur)
*/

void fsymbol(readbf, READBF)
(
 integer_t *unit  , /*  -> Numero d'unite logique du fichier (0 a 99)         */
 integer_t *taille, /*  -> Taille des elements                                */
 integer_t *nombre, /*  -> Nombre d'elements a lire au cours de cet appel     */
 integer_t *mode  , /*  -> Mode de lecture de l'enregistrement (0, 1, ou 2)   */
 integer_t *nbrlus, /* <-> Nombre d'elements de l'enregistrement deja lus     */
 integer_t *nbrtot, /* <-> Nombre total d'elements de l'enregistrement        */
 void      *elems , /* <-> Tableau des elements                               */
 integer_t *ierror  /* <-  Code de retour                                     */
)
{
  fic_bin_t fic = fic_bin_f_ptr_unit[(int)(*unit)];

  assert (*unit > -1 && *unit < 100);

  *ierror = lit_fic_bin_f(&fic, *taille, *nombre, *mode,
			  nbrlus, nbrtot, elems);
}

#endif /* FORTRAN_WRAPPER */


integer_t lit_fic_bin_f
(
 fic_bin_t *fic   , /*  -> Pointeur sur le descripteur de fichier             */
 integer_t  taille, /*  -> Taille des elements                                */
 integer_t  nombre, /*  -> Nombre d'elements a lire au cours de cet appel     */
 integer_t  mode  , /*  -> Mode de lecture de l'enregistrement (0, 1, ou 2)   */
 integer_t *nbrlus, /* <-> Nombre d'elements de l'enregistrement deja lus     */
 integer_t *nbrtot, /* <-> Nombre total d'elements de l'enregistrement        */
 void      *elems   /* <-> Tableau des elements                               */
)
{
  integer_t     nbr;
  integer_4_t   itmp;
  integer_4_t  *ptmp = &itmp;

  /* Lecture indicateur de dimension avant enregistrement */

  if (mode == 0 || *nbrlus == 0) {

    if (fread(ptmp, 4, 1, fic->ptr) != 1)
      return FIC_ERR_TST(fic->ptr);

    if (fic->end_swap != 0)
      fic_bin_f__endswap(ptmp, sizeof(integer_4_t), 1);

    nbr = (integer_t) (*ptmp / taille);

    if (mode == 1) {
      *nbrlus = 0 ;
      *nbrtot = (integer_t) nbr ;
    }

  }

  /* Verification que la taille correspond */

  if ((mode == 0 && nombre != nbr) ||
      (mode == 1 && nombre > (*nbrtot - *nbrlus)))
    return FIC_ERR_SZE_ENR;

  /* Lecture de l'enregistrement proprement dit */

  if (mode == 0 || mode == 1) {

    if ((integer_t)fread(elems, (size_t)taille, (size_t)nombre, fic->ptr)
	< nombre)
      return FIC_ERR_TST(fic->ptr);

    if (fic->end_swap != 0)
      fic_bin_f__endswap(elems, taille, nombre);

    /* Mise a jour position si mode 1 */

    if (mode == 1) *nbrlus += nombre;

  }
  else if (mode == 2) {

    if (fseek(fic->ptr, (long)((*nbrtot - *nbrlus) * taille), SEEK_CUR) != 0)
      return errno;

    *nbrlus = *nbrtot;

  }

  /* Lecture indicateur de dimension apres enregistrement */

  if (! (mode == 1 && (*nbrlus < *nbrtot))) {

    if (fread(ptmp, 4, 1, fic->ptr) != 1)
      return FIC_ERR_TST(fic->ptr);

    if (fic->end_swap != 0)
      fic_bin_f__endswap(ptmp, sizeof(integer_4_t), 1);

    nbr = (integer_t) (*ptmp / taille);

    /* Verification que la taille correspond toujours */

    if ((mode == 0 &&  nombre != nbr) ||
	(mode == 1 && *nbrlus != nbr))
      return FIC_ERR_SZE_ENR;

  }

  return 0;

}


/*----------------------------------------------------------------------------*
 * Ecriture d'un enregistrement binaire IEEE de type Fortran ;                *
 *                                                                            * 
 * On rappelle qu'un enregistrement commence et se termine par un marqueur    *
 * (entier code sur 4 octets indiquant la taille en octets des donnees        *
 * contenues dans l'enregistrement), permettant certaines operations de type  *
 * lecture a vide ou retour en arriere d'un enregistrement.                   *
 *                                                                            *
 * On fournit en argument un parametre mode indiquant le type d'ecriture :    *
 *                                                                            *
 *  mode = 0 : Ecriture complete de l'enregistrement                          *
 *         1 : Ecriture partielle de 'nombre' d'elements                      *
 *                                                                            *
 * On doit donc indiquer pour le mode 1 le nombre d'elements du meme          *
 * enregistrement deja ecrits lors d'appels precedents a ce sous-programme,   *
 * ainsi que le nombre total d'elements a ecrire.                             *
 *----------------------------------------------------------------------------*/

#ifdef FORTRAN_WRAPPER

/* Equivalent Fortran :

   subroutine writbf(unit, taille, nombre, mode, nbrecr, nbrtot, elems, ierror)
   
   integer unit     !  -> Numero d'unite logique du fichier (0 a 99)
   integer taille   !  -> Taille d'un element
   integer nombre   !  -> Nombre d'elements a ecrire au cours de cet appel
   integer mode     !  -> Mode de lecture (0 ou 1)
   integer nbrecr   ! <-> Nombre d'elements de l'enregistrement deja ecrits
   integer nbrtot   ! <-> Nombre total d'elements de l'enregistrement
   ?       elems    ! <-> Tableau des elements
   integer ierror   ! <-  Code de retour (different de 0 en cas d'erreur)
*/

void fsymbol(writbf, WRITBF)
(
 integer_t *unit  , /*  -> Numero d'unite logique du fichier (0 a 99)         */
 integer_t *taille, /*  -> Taille des elements                                */
 integer_t *nombre, /*  -> Nombre d'elements                                  */
 integer_t *mode  , /*  -> Mode d'ecriture de l'enregistrement (0 ou 1)       */
 integer_t *nbrecr, /* <-> Nombre d'elements de l'enregistrement deja ecrits  */
 integer_t *nbrtot, /* <-> Nombre total d'elements de l'enregistrement        */
 void      *elems,  /*  -> Tableau des elements                               */
 integer_t *ierror  /* <-  Code de retour                                     */
)
{
  fic_bin_t fic = fic_bin_f_ptr_unit[(int)(*unit)];

  assert (*unit > -1 && *unit < 100);

  *ierror = ecr_fic_bin_f(&fic, *taille, *nombre, *mode,
			  nbrecr, nbrtot, elems);
}

#endif /* FORTRAN_WRAPPER */


integer_t ecr_fic_bin_f
(
 fic_bin_t *fic   , /*  -> Pointeur sur le descripteur de fichier             */
 integer_t  taille, /*  -> Taille des elements                                */
 integer_t  nombre, /*  -> Nombre d'elements a ecrire au cours de cet appel   */
 integer_t  mode  , /*  -> Mode d'ecriture de l'enregistrement (0 ou 1)       */
 integer_t *nbrecr, /* <-> Nombre d'elements de l'enregistrement deja ecrits  */
 integer_t *nbrtot, /* <-> Nombre total d'elements de l'enregistrement        */
 void      *elems   /* <-> Tableau des elements                               */
)

{
  integer_4_t   itmp;
  integer_4_t  *ptmp = &itmp;


  /* Ecriture indicateur de dimension avant enregistrement */

  if (mode == 0 || *nbrecr == 0) {

    if (mode == 0)
      *ptmp = (integer_4_t)(nombre * taille);
    else
      *ptmp = (integer_4_t)((*nbrtot) * taille);

    if (fic->end_swap != 0)
      fic_bin_f__endswap(ptmp, 4, 1);

    if (fwrite(ptmp, 4, 1, fic->ptr) != 1)
      return FIC_ERR_TST(fic->ptr);

    if (mode == 1) *nbrecr = 0 ;

  }


  /* Ecriture de l'enregistrement proprement dit */
  if (fic->end_swap != 0)
    fic_bin_f__endswap(elems, taille, nombre);

  if ((integer_t)fwrite(elems, (size_t)taille, (size_t)nombre, fic->ptr)
      < nombre)
    return FIC_ERR_TST(fic->ptr);

  if (fic->end_swap != 0)
    fic_bin_f__endswap(elems, taille, nombre);


  /* Mise a jour position si mode 1 */

  if (mode == 1) *nbrecr += nombre;


  /* Ecriture indicateur de dimension apres enregistrement */

  if (mode == 0 || *nbrecr == *nbrtot) {

    if (mode == 0)
      *ptmp = (integer_4_t)(nombre * taille);
    else
      *ptmp = (integer_4_t)((*nbrtot) * taille);


    if (fic->end_swap != 0)
      fic_bin_f__endswap(ptmp, 4, 1);

    if (fwrite(ptmp, 4, 1, fic->ptr) != 1)
      return FIC_ERR_TST(fic->ptr);

  }

  return 0;

}


/*----------------------------------------------------------------------------*/
/* Ecriture d'une chaine de caracteres correspondant a un numero d'erreur     */
/*----------------------------------------------------------------------------*/

#ifdef FORTRAN_WRAPPER

/* Equivalent Fortran :

   subroutine strebf(chaine, lchn, ierror)
   
   char(*) chaine   ! <-> Chaine de caracteres receptrice
   integer lchn     !  -> Longueur de la chaine receptrice
   integer ierror   !  -> Code d'erreur
*/

#ifdef CHAINE_F_VERS_C_LEN_FIN 
void fsymbol(strebf, STREBF)
(
 char      *chaine, /* <-  Chaine de caracteres receptrice                    */
 integer_t *lchn  , /*  -> Longueur de la chaine receptrice                   */
 integer_t *ierror, /*  -> Code d'erreur                                      */
 integer_t  lenchn  /*  -> Longueur de la chaine receptrice                   */
)
#elif defined(CHAINE_F_VERS_C_STR_NUL)
void fsymbol(strebf, STREBF)
(
 char      *chaine, /* <-  Chaine de caracteres receptrice                    */
 integer_t *lchn  , /*  -> Longueur de la chaine receptrice                   */
 integer_t *ierror  /*  -> Code d'erreur                                      */
)
#endif
{
  int        i;
  integer_t lmaxch = LEN_MAX_NOM_FIC_BIN_F;

  if (*lchn < lmaxch)
    lmaxch = *lchn;

  err_fic_bin_f(chaine, *ierror, lmaxch - 1);

  for (i = strlen(chaine); i < (int)lmaxch - 1; i++) *(chaine + i) = ' ';

}

#endif /* FORTRAN_WRAPPER */


void err_fic_bin_f
(
 char      *chaine, /* <-> Chaine de caracteres receptrice                    */
 integer_t  ierror, /*  -> Code d'erreur                                      */
 integer_t  lmaxch  /*  -> Longueur maximale de la chaine receptrice          */
)
{

  switch(ierror) {

  case FIC_ERR_FIN:

    strncpy(chaine, FIC_ERR_FIN_MSG, lmaxch);
    break;

  case FIC_ERR_SZE_ENR:

    strncpy(chaine, FIC_ERR_SZE_ENR_MSG, lmaxch);
    break;

  case FIC_ERR_LEN_NOM:

    strncpy(chaine, FIC_ERR_LEN_NOM_MSG, lmaxch);
    break;

  default:

    strncpy(chaine, strerror((int)ierror), lmaxch);

  }

  chaine[lmaxch] = '\0';

}


/*============================================================================*/
/*                               Fonctions privees                            */
/*============================================================================*/

/*----------------------------------------------------------------------------*/
/* Permutation des octets pour passage de "little endian" a "big endian"      */
/*----------------------------------------------------------------------------*/

static void fic_bin_f__endswap
(
 void  *buf,   /* Tampon contenant les elements                               */
 size_t size,  /* Taille d'un element                                         */
 size_t nitems /* Nombre d'elements                                           */
)
{
  char  tmpswap;
  char *ptr = (char *)buf;

  size_t i, j, shift;
  for (j = 0; j < nitems; j++) {
    shift = j * size;
    for (i = 0; i < (size / 2); i++) {
      tmpswap = *(ptr + shift + i);
      *(ptr + shift + i ) = *(ptr + shift + (size - 1) - i);
      *(ptr + shift + (size - 1) - i) = tmpswap;
    }
  }
}

