use core:geometry;

/**
 * Describes a layout for a number of components.
 *
 * Layouts can be nested inside each other to describe more complex interactions. If a layout
 * requires to maintain additional information about each component, it should contain a class named
 * 'Info' for that purpose, and 'add' should return an instance associated with the newly added component.
 */
class Layout extends Component {
	// Our allotted space.
	protected Rect space;

	// Information on a child.
	class Info {}

	// Get the position of the layout.
	Rect pos() {
		space;
	}

	// Set the allocated space. This will trigger a re-layout.
	assign pos(Rect p) {
		space = p;
		layout();
	}

	// Get the minimum size of the layout.
	Size minSize() {
		Size(0, 0);
	}

	// Re-compute the layout here.
	void layout() : abstract;

	// Add a child to this layout.
	Info add(Component child) {
		Info();
	}

	// Make sure derived classes override 'findAll'. We provide a body so that it is callable as super:findAll().
	void findAll(fn(Component)->void fn) : abstract {
		super:findAll(fn);
	}
}

// Default border/space in the Ui.
Size defaultBorder() {
	Size(8, 8);
}


/**
 * A simple layout that adds a border around the single component within.
 */
class Border extends Layout {
	// The one and only component inside here.
	private Component? child;

	// Size of the border.
	Size border;

	// Initialize.
	init() {
		init() { border = defaultBorder; }
	}

	// Set the border size.
	void border(Float w, Float h) {
		border = Size(w, h);
	}

	// Add/set the child.
	Layout:Info add(Component c) {
		child = c;
		Layout:Info();
	}

	// Compute the minimum size.
	Size minSize() {
		Size s;
		if (c = child)
			s = c.minSize;
		s + border*2;
	}

	// Re-compute the layout.
	void layout() {
		unless (c = child)
			return;

		Rect p = pos;

		p.p0 += border;
		p.p1 -= border;

		c.pos = p;
	}

	// Traverse.
	void findAll(fn(Component)->void fn) : override {
		super:findAll(fn);
		if (child)
			child.findAll(fn);
	}

}

/**
 * A simple layout that stacks components on top of each other.
 */
class Stack extends Layout {
	// Our components.
	private Component[] components;

	// Get our minimum size.
	Size minSize() : override {
		Size s;
		for (c in components) {
			s = max(s, c.minSize);
		}
		s;
	}

	// Do layout.
	void layout() : override {
		Rect p = pos;
		for (c in components) {
			c.pos = p;
		}
	}

	// Add a child to the layout.
	Layout:Info add(Component c) {
		components << c;
		Layout:Info();
	}

	// Find all elements.
	void findAll(fn(Component)->void fn) {
		super:findAll(fn);
		for (c in components) {
			c.findAll(fn);
		}
	}
}
