#  Copyright (C) 1999-2012
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc PlotDef {} {
    global pap
    global iap

    set iap(tt) {ap}
    set iap(windows) {}
    set iap(unique) 0

    set pap(graph,x,grid) 1
    set pap(graph,x,log) 0
    set pap(graph,x,flip) 0
    set pap(graph,y,grid) 1
    set pap(graph,y,log) 0
    set pap(graph,y,flip) 0

    set pap(titleFont) helvetica
    set pap(titleSize) 12
    set pap(titleWeight) normal
    set pap(titleSlant) roman

    set pap(textlabFont) helvetica
    set pap(textlabSize) 9
    set pap(textlabWeight) normal
    set pap(textlabSlant) roman

    set pap(numlabFont) helvetica
    set pap(numlabSize) 9
    set pap(numlabWeight) normal
    set pap(numlabSlant) roman

    set pap(discrete) 0
    set pap(discrete,symbol) circle
    set pap(discrete,fill) 1
    set pap(discrete,color) red

    set pap(linear) 1
    set pap(linear,width) 1
    set pap(linear,color) black
    set pap(linear,dash) no

    set pap(step) 0
    set pap(step,width) 1
    set pap(step,color) black
    set pap(step,dash) no

    set pap(quadratic) 0
    set pap(quadratic,width) 1
    set pap(quadratic,color) black
    set pap(quadratic,dash) no

    set pap(error) 1
    set pap(error,width) 1
    set pap(error,color) red

    set pap(bar) 0
    set pap(bar,color) black
}

proc PlotAxisFormat {varname axis w nn} {
    upvar #0 $varname var
    global $varname

    return [format $var(graph,$axis,format) $nn]
}

proc PlotClearData {varname} {
    upvar #0 $varname var
    global $varname

    global ds9

    if {$var(data,total) == 0} {
	return
    }

    # first set can be external
    set clear $var(1,manage)

    for {set nn 1} {$nn<=$var(data,total)} {incr nn} {
	if {$var($nn,manage)} {
	    # delete elements
	    foreach el [$var(graph) element names] {
		set f [split $el -]
		if {[lindex $f 1] == $nn} {
		    $var(graph) element delete $el
		}
	    }

	    # delete markers
	    foreach mk [$var(graph) marker names "m-${nn}*"] {
		$var(graph) marker delete $mk
	    }

	    # destroy vectors
	    blt::vector destroy $var($nn,xdata) $var($nn,ydata)
	    switch $var($nn,dim) {
		xy {}
		xyxe {blt::vector destroy $var($nn,xedata)}
		xyye {blt::vector destroy $var($nn,yedata)}
		xyxeye {blt::vector destroy $var($nn,xedata) $var($nn,yedata)}
	    }

	    foreach x [array names $varname] {
		set f [split $x ,]
		if {([lindex $f 0] == $nn)} {
		    unset ${varname}($x)
		}
	    }
	}
    }

    if {$clear} {
	set var(data,total) 0
	set var(data,current) 0

	set var(xdata) {}
	set var(ydata) {}
	set var(xedata) {}
	set var(yedata) {}

	# reset other variables
	set var(graph,title) {}
	set var(graph,xaxis) {}
	set var(graph,yaxis) {}

	set var(graph,x,auto) 1
	set var(graph,x,min) {}
	set var(graph,x,max) {}
	set var(graph,y,auto) 1
	set var(graph,y,min) {}
	set var(graph,y,max) {}
	
	$var(mb).dataset delete $ds9(menu,start) end

	$var(proc,updategraph) $varname
	PlotStats $varname
	PlotList $varname
    } else {
 	set var(data,total) 1
 	set var(data,current) 1

	$var(mb).dataset delete [expr $ds9(menu,start)+1] end
 	PlotCurrentData $varname
	$var(proc,updategraph) $varname
    }
}

proc PlotCurrentData {varname} {
    upvar #0 $varname var
    global $varname

    if {$var(data,total) > 0} {
	set nn $var(data,current)

	set var(manage) $var($nn,manage)
	set var(dim) $var($nn,dim)

	set var(xdata) $var($nn,xdata)
	set var(ydata) $var($nn,ydata)
	set var(xedata) $var($nn,xedata)
	set var(yedata) $var($nn,yedata)

	set var(discrete) $var($nn,discrete) 
	set var(discrete,symbol) $var($nn,discrete,symbol) 
	set var(discrete,fill) $var($nn,discrete,fill) 
	set var(discrete,color) $var($nn,discrete,color) 

	set var(linear) $var($nn,linear) 
	set var(linear,width) $var($nn,linear,width) 
	set var(linear,color) $var($nn,linear,color) 
	set var(linear,dash) $var($nn,linear,dash) 

	set var(step) $var($nn,step) 
	set var(step,width) $var($nn,step,width) 
	set var(step,color) $var($nn,step,color) 
	set var(step,dash) $var($nn,step,dash) 

	set var(quadratic) $var($nn,quadratic) 
	set var(quadratic,width) $var($nn,quadratic,width) 
	set var(quadratic,color) $var($nn,quadratic,color) 
	set var(quadratic,dash) $var($nn,quadratic,dash) 

	set var(error) $var($nn,error) 
	set var(error,width) $var($nn,error,width) 
	set var(error,color) $var($nn,error,color) 

	set var(bar) $var($nn,bar)
	set var(bar,color) $var($nn,bar,color) 
    }

    PlotStats $varname
    PlotList $varname
}

proc PlotDataSet {varname dim data} {
    upvar #0 $varname var
    global $varname

    switch -- $dim {
	4 {
	    # first data set
	    PlotDataSetOne $varname "4.1" $data

	    # set color
	    set dc $var(discrete,color)
	    set lc $var(linear,color)
	    set sc $var(step,color)
	    set qc $var(quadratic,color)

	    set var(discrete,color) [PlotNextColor $var(discrete,color)]
	    set var(linear,color) [PlotNextColor $var(linear,color)]
	    set var(step,color) [PlotNextColor $var(step,color)]
	    set var(quadratic,color) [PlotNextColor $var(quadratic,color)]

	    # second data set
	    PlotDataSetOne $varname "4.2" $data

	    # rest colors
	    set var(discrete,color) $dc
	    set var(linear,color) $lc
	    set var(step,color) $sc
	    set var(quadratic,color) $qc
	}
	5 {
	    # first data set
	    PlotDataSetOne $varname "5.1" $data

	    # set color
	    set dc $var(discrete,color)
	    set lc $var(linear,color)
	    set sc $var(step,color)
	    set qc $var(quadratic,color)

	    set var(discrete,color) [PlotNextColor $var(discrete,color)]
	    set var(linear,color) [PlotNextColor $var(linear,color)]
	    set var(step,color) [PlotNextColor $var(step,color)]
	    set var(quadratic,color) [PlotNextColor $var(quadratic,color)]

	    # second data set
	    PlotDataSetOne $varname "5.2" $data

	    # rest colors
	    set var(discrete,color) $dc
	    set var(linear,color) $lc
	    set var(step,color) $sc
	    set var(quadratic,color) $qc
	}
	default {PlotDataSetOne $varname $dim $data}
    }
}

proc PlotDataSetOne {varname dim data} {
    upvar #0 $varname var
    global $varname

    # look for no data
    if {[string length $data] == 0} {
	return
    }

    # total length
    set ll [llength $data]
    set ii 0

    while {$ii<$ll} {
	# incr count
	incr ${varname}(data,total) 
	set nn $var(data,total)
	set var(data,current) $nn

	# new vector names
	set xdata ap${varname}xx${nn}
	set ydata ap${varname}yy${nn}
	set xedata ap${varname}xe${nn}
	set yedata ap${varname}ye${nn}

	# basics xy
	set var(manage) 1
	set var(xdata) $xdata
	set var(ydata) $ydata
	global $var(xdata) $var(ydata)
	blt::vector create $var(xdata) $var(ydata)

	# substitute all separtors
	regsub -all {[\n\r\t, ]+} $data { } data
	# remove all non-numeric data
	regsub -all {[^0-9.e\- ]+} $data {} data

	set ox [lindex $data $ii]
	set x {}
	set y {}
	set xe {}
	set ye {}
	switch -- $dim {
	    2 -
	    xy {
		set var(dim) xy
		set var(xedata) {}
		set var(yedata) {}

		for {} {$ii<$ll} {incr ii 2} {
		    set tx [lindex $data $ii]
		    if {$var(seq)} {
			if {$ox<=$tx} {
			    set ox $tx
			    lappend x $tx
			    lappend y [lindex $data [expr $ii+1]]
			} else {
			    break
			}
		    } else {
			lappend x $tx
			lappend y [lindex $data [expr $ii+1]]
		    }
		}
		$var(xdata) set $x
		$var(ydata) set $y
	    }

	    xyex {
		set var(dim) xyxe
		set var(xedata) $xedata
		set var(yedata) {}

		global $var(xedata)
		blt::vector create $var(xedata)

		for {} {$ii<$ll} {incr ii 3} {
		    set tx [lindex $data $ii]
		    if {$var(seq)} {
			if {$ox<=$tx} {
			    set ox $tx
			    lappend x $tx
			    lappend y [lindex $data [expr $ii+1]]
			    lappend xe [lindex $data [expr $ii+2]]
			} else {
			    break
			}
		    } else {
			lappend x $tx
			lappend y [lindex $data [expr $ii+1]]
			lappend xe [lindex $data [expr $ii+2]]
		    }
		}
		$var(xdata) set $x
		$var(ydata) set $y
		$var(xedata) set $xe
	    }

	    3 -
	    xyey {
		set var(dim) xyye
		set var(xedata) {}
		set var(yedata) $yedata

		global $var(yedata)
		blt::vector create $var(yedata)

		for {} {$ii<$ll} {incr ii 3} {
		    set tx [lindex $data $ii]
		    if {$var(seq)} {
			if {$ox<=$tx} {
			    set ox $tx
			    lappend x $tx
			    lappend y [lindex $data [expr $ii+1]]
			    lappend ye [lindex $data [expr $ii+2]]
			} else {
			    break
			}
		    } else {
			lappend x $tx
			lappend y [lindex $data [expr $ii+1]]
			lappend ye [lindex $data [expr $ii+2]]
		    }
		}
		$var(xdata) set $x
		$var(ydata) set $y
		$var(yedata) set $ye
	    }

	    xyexey {
		set var(dim) xyxeye
		set var(xedata) $xedata
		set var(yedata) $yedata

		global $var(xedata) $var(yedata)
		blt::vector create $var(xedata) $var(yedata)

		for {} {$ii<$ll} {incr ii 4} {
		    set tx [lindex $data $ii]
		    if {$var(seq)} {
			if {$ox<=$tx} {
			    set ox $tx
			    lappend x $tx
			    lappend y [lindex $data [expr $ii+1]]
			    lappend xe [lindex $data [expr $ii+2]]
			    lappend ye [lindex $data [expr $ii+3]]
			} else {
			    break
			}
		    } else {
			lappend x $tx
			lappend y [lindex $data [expr $ii+1]]
			lappend xe [lindex $data [expr $ii+2]]
			lappend ye [lindex $data [expr $ii+3]]
		    }
		}
		$var(xdata) set $x
		$var(ydata) set $y
		$var(xedata) set $xe
		$var(yedata) set $ye
	    }

	    4.1 {
		set var(dim) xyye
		set var(xedata) {}
		set var(yedata) $yedata

		global $var(yedata)
		blt::vector create $var(yedata)

		for {} {$ii<$ll} {incr ii 4} {
		    set tx [lindex $data $ii]
		    if {$var(seq)} {
			if {$ox<=$tx} {
			    set ox $tx
			    lappend x $tx
			    lappend y [lindex $data [expr $ii+1]]
			    lappend ye [lindex $data [expr $ii+2]]
			} else {
			    break
			}
		    } else {
			lappend x $tx
			lappend y [lindex $data [expr $ii+1]]
			lappend ye [lindex $data [expr $ii+2]]
		    }
		}
		$var(xdata) set $x
		$var(ydata) set $y
		$var(yedata) set $ye
	    }

	    4.2 {
		set var(dim) xy
		set var(xedata) {}
		set var(yedata) {}

		for {} {$ii<$ll} {incr ii 4} {
		    set tx [lindex $data $ii]
		    if {$var(seq)} {
			if {$ox<=$tx} {
			    set ox $tx
			    lappend x $tx
			    lappend y [lindex $data [expr $ii+3]]
			} else {
			    break
			}
		    } else {
			lappend x $tx
			lappend y [lindex $data [expr $ii+3]]
		    }
		}
		$var(xdata) set $x
		$var(ydata) set $y
	    }

	    5.1 {
		set var(dim) xyye
		set var(xedata) {}
		set var(yedata) $yedata

		global $var(yedata)
		blt::vector create $var(yedata)

		for {} {$ii<$ll} {incr ii 5} {
		    set tx [lindex $data $ii]
		    if {$var(seq)} {
			if {$ox<=$tx} {
			    set ox $tx
			    lappend x $tx
			    lappend y [lindex $data [expr $ii+1]]
			    lappend ye [lindex $data [expr $ii+2]]
			} else {
			    break
			}
		    } else {
			lappend x $tx
			lappend y [lindex $data [expr $ii+1]]
			lappend ye [lindex $data [expr $ii+2]]
		    }
		}
		$var(xdata) set $x
		$var(ydata) set $y
		$var(yedata) set $ye
	    }

	    5.2 {
		set var(dim) xyye
		set var(xedata) {}
		set var(yedata) $yedata

		global $var(yedata)
		blt::vector create $var(yedata)

		for {} {$ii<$ll} {incr ii 5} {
		    set tx [lindex $data $ii]
		    if {$var(seq)} {
			if {$ox<=$tx} {
			    set ox $tx
			    lappend x $tx
			    lappend y [lindex $data [expr $ii+3]]
			    lappend ye [lindex $data [expr $ii+4]]
			} else {
			    break
			}
		    } else {
			lappend x $tx
			lappend y [lindex $data [expr $ii+3]]
			lappend ye [lindex $data [expr $ii+4]]
		    }
		}
		$var(xdata) set $x
		$var(ydata) set $y
		$var(yedata) set $ye
	    }
	}

	# set menu options
	set var($nn,manage) 1
	set var($nn,dim) $var(dim)

	set var($nn,xdata) $var(xdata) 
	set var($nn,ydata) $var(ydata) 
	set var($nn,xedata) $var(xedata) 
	set var($nn,yedata) $var(yedata) 

	set var($nn,discrete) $var(discrete) 
	set var($nn,discrete,symbol) $var(discrete,symbol) 
	set var($nn,discrete,fill) $var(discrete,fill) 
	set var($nn,discrete,color) $var(discrete,color) 

	set var($nn,linear) $var(linear) 
	set var($nn,linear,width) $var(linear,width) 
	set var($nn,linear,color) $var(linear,color) 
	set var($nn,linear,dash) $var(linear,dash) 

	set var($nn,step) $var(step) 
	set var($nn,step,width) $var(step,width) 
	set var($nn,step,color) $var(step,color) 
	set var($nn,step,dash) $var(step,dash) 

	set var($nn,quadratic) $var(quadratic) 
	set var($nn,quadratic,width) $var(quadratic,width) 
	set var($nn,quadratic,color) $var(quadratic,color) 
	set var($nn,quadratic,dash) $var(quadratic,dash) 

	set var($nn,error) $var(error) 
	set var($nn,error,width) $var(error,width) 
	set var($nn,error,color) $var(error,color) 

	set var($nn,bar) $var(bar) 
	set var($nn,bar,color) $var(bar,color) 

	# update data set menu
	$var(mb).dataset add radiobutton -label "[msgcat::mc {Dataset}] $nn" \
	    -variable ${varname}(data,current) -value $nn \
	    -command [list PlotCurrentData $varname]

	$var(proc,createelement) $varname
	$var(proc,updateelement) $varname
    }
}

proc PlotDupData {varname mm} {
    upvar #0 $varname var
    global $varname

    if {$var(data,total) == 0} {
	return
    }

    # incr count
    incr ${varname}(data,total) 
    set nn $var(data,total)
    set pp [expr $nn-1]

    # new vector names
    set var($nn,xdata)  ap${varname}xx${nn}
    set var($nn,ydata)  ap${varname}yy${nn}
    set var($nn,xedata) ap${varname}xe${nn}
    set var($nn,yedata) ap${varname}ye${nn}
    global $var($mm,xdata) $var($mm,ydata) $var($mm,xedata) $var($mm,yedata)
    global $var($nn,xdata) $var($nn,ydata) $var($nn,xedata) $var($nn,yedata)
    
    $var($mm,xdata) dup $var($nn,xdata)
    $var($mm,ydata) dup $var($nn,ydata)
    if {$var($mm,xedata) != {}} {
	$var($mm,xedata) dup $var($nn,xedata)
    } else {
	set var($nn,xedata) {}
    }
    if {$var($mm,yedata) != {}} {
	$var($mm,yedata) dup $var($nn,yedata)
    } else {
	set var($nn,yedata) {}
    }

    # set menu options
    set var($nn,manage) 1
    set var($nn,dim) $var($mm,dim)

    set var($nn,discrete) $var($mm,discrete) 
    set var($nn,discrete,symbol) $var($mm,discrete,symbol) 
    set var($nn,discrete,fill) $var($mm,discrete,fill) 
    set var($nn,discrete,color) [PlotNextColor $var($pp,discrete,color)]

    set var($nn,linear) $var($mm,linear) 
    set var($nn,linear,width) $var($mm,linear,width) 
    set var($nn,linear,color) [PlotNextColor $var($pp,linear,color)]
    set var($nn,linear,dash) $var($mm,linear,dash) 

    set var($nn,step) $var($mm,step) 
    set var($nn,step,width) $var($mm,step,width) 
    set var($nn,step,color) [PlotNextColor $var($pp,step,color)]
    set var($nn,step,dash) $var($mm,step,dash) 

    set var($nn,quadratic) $var($mm,quadratic) 
    set var($nn,quadratic,width) $var($mm,quadratic,width) 
    set var($nn,quadratic,color) [PlotNextColor $var($pp,quadratic,color)]
    set var($nn,quadratic,dash) $var($mm,quadratic,dash) 

    set var($nn,error) $var($mm,error) 
    set var($nn,error,width) $var($mm,error,width) 
    set var($nn,error,color) $var($mm,error,color) 

    set var($nn,bar) $var($mm,bar)
    set var($nn,bar,color) [PlotNextColor $var($pp,bar,color)]

    # update data set menu
    $var(mb).dataset add radiobutton -label "[msgcat::mc {Dataset}] $nn" \
	-variable ${varname}(data,current) -value $nn \
	-command [list PlotCurrentData $varname]

    # make current
    set var(data,current) $nn

    set var(manage) $var($nn,manage)
    set var(dim) $var($nn,dim)

    set var(xdata) $var($nn,xdata)
    set var(ydata) $var($nn,ydata)
    set var(xedata) $var($nn,xedata)
    set var(yedata) $var($nn,yedata)

    set var(discrete) $var($nn,discrete) 
    set var(discrete,symbol) $var($nn,discrete,symbol) 
    set var(discrete,fill) $var($nn,discrete,fill) 
    set var(discrete,color) $var($nn,discrete,color) 

    set var(linear) $var($nn,linear) 
    set var(linear,width) $var($nn,linear,width) 
    set var(linear,color) $var($nn,linear,color) 
    set var(linear,dash) $var($nn,linear,dash) 

    set var(step) $var($nn,step) 
    set var(step,width) $var($nn,step,width) 
    set var(step,color) $var($nn,step,color) 
    set var(step,dash) $var($nn,step,dash) 

    set var(quadratic) $var($nn,quadratic) 
    set var(quadratic,width) $var($nn,quadratic,width) 
    set var(quadratic,color) $var($nn,quadratic,color) 
    set var(quadratic,dash) $var($nn,quadratic,dash) 

    set var(error) $var($nn,error) 
    set var(error,width) $var($nn,error,width) 
    set var(error,color) $var($nn,error,color) 

    set var(bar) $var($nn,bar) 
    set var(bar,color) $var($nn,bar,color) 

    $var(proc,createelement) $varname
    $var(proc,updateelement) $varname
    $var(proc,updategraph) $varname
    PlotStats $varname
    PlotList $varname
}

proc PlotDestroy {varname} {
    upvar #0 $varname var
    global $varname
    
    global iap

    # see if it still is around
    if {![PlotPing $varname]} {
 	return
    }
    
    for {set nn 1} {$nn<=$var(data,total)} {incr nn} {
 	if {$var($nn,manage)} {
 	    switch $var($nn,dim) {
 		xy {
		    blt::vector destroy $var($nn,xdata) $var($nn,ydata)
		}
 		xyxe {
		    blt::vector destroy $var($nn,xdata) $var($nn,ydata) \
			$var($nn,xedata)
		}
 		xyye {
		    blt::vector destroy $var($nn,xdata) $var($nn,ydata) \
			$var($nn,yedata)
		}
 		xyxeye {
		    blt::vector destroy $var($nn,xdata) $var($nn,ydata) \
			$var($nn,xedata) $var($nn,yedata)
		}
 	    }
 	}
    }
    
    # in the rare case where the user has start a zoom, then closes the dialog
    # we must clean up his mess
    global zoomInfo
    if [info exists zoomInfo($var(graph),corner)] {
	if {$zoomInfo($var(graph),corner) == {B}} {
	    blt::SetZoomPoint $var(graph) 0 0
	}
    }

    destroy $var(top)
    destroy $var(mb)

    # stats window?
    if {$var(stats)} {
	SimpleTextDestroy "${varname}stats"
    }

    # list window?
    if {$var(list)} {
	SimpleTextDestroy "${varname}list"
    }

    # delete it from the xpa list
    set ii [lsearch $iap(windows) $varname]
    if {$ii>=0} {
	set iap(windows) [lreplace $iap(windows) $ii $ii]
    }

    unset $varname
}

proc PlotExternal {varname} {
    upvar #0 $varname var
    global $varname

    # incr count
    incr ${varname}(data,total) 
    set nn $var(data,total)
    set var(data,current) $nn

    # set menu options
    set var($nn,manage) $var(manage)
    set var($nn,dim) $var(dim)

    set var($nn,xdata) $var(xdata) 
    set var($nn,ydata) $var(ydata) 
    set var($nn,xedata) $var(xedata) 
    set var($nn,yedata) $var(yedata) 

    set var($nn,discrete) $var(discrete) 
    set var($nn,discrete,symbol) $var(discrete,symbol) 
    set var($nn,discrete,fill) $var(discrete,fill) 
    set var($nn,discrete,color) $var(discrete,color) 

    set var($nn,linear) $var(linear) 
    set var($nn,linear,width) $var(linear,width) 
    set var($nn,linear,color) $var(linear,color) 
    set var($nn,linear,dash) $var(linear,dash) 

    set var($nn,step) $var(step) 
    set var($nn,step,width) $var(step,width) 
    set var($nn,step,color) $var(step,color) 
    set var($nn,step,dash) $var(step,dash) 

    set var($nn,quadratic) $var(quadratic) 
    set var($nn,quadratic,width) $var(quadratic,width) 
    set var($nn,quadratic,color) $var(quadratic,color) 
    set var($nn,quadratic,dash) $var(quadratic,dash) 

    set var($nn,error) $var(error) 
    set var($nn,error,width) $var(error,width) 
    set var($nn,error,color) $var(error,color) 

    set var($nn,bar) $var(bar) 
    set var($nn,bar,color) $var(bar,color) 

    # update data set menu
    $var(mb).dataset add radiobutton -label "[msgcat::mc {Dataset}] $nn" \
	-variable ${varname}(data,current) -value $nn \
	-command "PlotCurrentData $varname"
}

proc PlotList {varname} {
    upvar #0 $varname var
    global $varname

    if {!$var(list)} {
	return
    }

    set rr {}
    if {$var(xdata) != {}} {
	global $var(xdata) $var(ydata) $var(xedata) $var(yedata)
	set ll [$var(xdata) length]
	set xx [$var(xdata) range]
	set yy [$var(ydata) range]

	switch $var(dim) {
	    xy {
		for {set ii 0} {$ii<$ll} {incr ii} {
		    append rr "[lindex $xx $ii] [lindex $yy $ii]\n"
		}
	    }
	    xyxe {
		set xe [$var(xedata) range]
		for {set ii 0} {$ii<$ll} {incr ii} {
		    append rr "[lindex $xx $ii] [lindex $yy $ii] [lindex $xe $ii]\n"
		}
	    }
	    xyye {
		set ye [$var(yedata) range]
		for {set ii 0} {$ii<$ll} {incr ii} {
		    append rr "[lindex $xx $ii] [lindex $yy $ii] [lindex $ye $ii]\n"
		}
	    }
	    xyxeye {
		set xe [$var(xedata) range]
		set ye [$var(yedata) range]
		for {set ii 0} {$ii<$ll} {incr ii} {
		    append rr "[lindex $xx $ii] [lindex $yy $ii] [lindex $xe $ii] [lindex $ye $ii]\n"
		}
	    }
	}
    }

    SimpleTextDialog "${varname}list" [msgcat::mc {Data}] 40 20 insert top $rr PlotListDestroyCB $varname
}

proc PlotListDestroyCB {varname} {
    upvar #0 $varname var
    global $varname

    set var(list) 0
}

proc PlotLoadConfig {varname} {
    upvar #0 $varname var
    global $varname

    PlotLoadConfigFile $varname [OpenFileDialog apconfigfbox]
}

proc PlotLoadConfigFile {varname filename} {
    upvar #0 $varname var
    global $varname

    if {$filename != {}} {
	source $filename
	array set $varname [array get analysisplot]
	unset analysisplot

	# backward compatibility
	if {[info exists var(grid)]} {
	    set var(graph,x,grid) $var(grid)
	    set var(graph,y,grid) $var(grid)
	    unset var(grid)
	}
	if {[info exists var(format)]} {
	    set var(graph,format) $var(format)
	    set var(graph,x,format) $var(format,x)
	    set var(graph,y,format) $var(format,y)
	    unset var(format)
	    unset var(format,x)
	    unset var(format,y)
	}

	if {[info exists var(grid,log)]} {
	    switch $var(grid,log) {
		linearlinear {
		    set var(graph,x,log) 0
		    set var(graph,y,log) 0
		}
		linearlog {
		    set var(graph,x,log) 0
		    set var(graph,y,log) 1
		}
		loglinear {
		    set var(graph,x,log) 1
		    set var(graph,y,log) 0
		}
		loglog {
		    set var(graph,x,log) 1
		    set var(graph,y,log) 1
		}
	    }
	    unset var(grid,log)
	}

	$var(proc,updategraph) $varname
    }
}

proc PlotLoadData {varname} {
    upvar #0 $varname var
    global $varname

    set filename [OpenFileDialog apdatafbox]
    if {$filename != {}} {
	set dim xy
	if {[PlotDataFormatDialog dim]} {
	    PlotLoadDataFile $varname $filename $dim
	}
    }
}

proc PlotLoadDataFile {varname filename dim} {
    upvar #0 $varname var
    global $varname

    set ch [open $filename]
    set data [read $ch]
    close $ch

    PlotRaise $varname

    PlotDataSet $varname $dim $data
    $var(proc,updategraph) $varname
    PlotStats $varname
    PlotList $varname
}

proc PlotNextColor {which} {
    switch -- $which {
	black {return white}
	white {return red}
	red {return green}
	green {return blue}
	blue {return cyan}
	cyan {return magenta}
	magenta {return yellow}
	yellow {return black}
    }
}

proc PlotPing {varname} {
    upvar #0 $varname var
    global $varname

    if {[info exists var(top)]} {
	if {[winfo exists $var(top)]} {
	    return 1
	}
    }
    return 0
}

proc PlotRaise {varname} {
    upvar #0 $varname var
    global $varname

    if {[PlotPing $varname]} {
	raise $var(top)
	return 1
    }
    return 0
}

proc PlotSaveConfig {varname} {
    upvar #0 $varname var
    global $varname

    PlotSaveConfigFile $varname [SaveFileDialog apconfigfbox]
}

proc PlotSaveConfigFile {varname filename} {
    upvar #0 $varname var
    global $varname

    if {$filename == {}} {
	return
    }

    set ch [open $filename w]

    set analysisplot(graph,x,grid) $var(graph,x,grid)
    set analysisplot(graph,x,log) $var(graph,x,log) 
    set analysisplot(graph,x,flip) $var(graph,x,flip) 
    set analysisplot(graph,y,grid) $var(graph,y,grid)
    set analysisplot(graph,y,log) $var(graph,y,log) 
    set analysisplot(graph,y,flip) $var(graph,y,flip) 

    set analysisplot(titleFont) $var(titleFont) 
    set analysisplot(titleSize) $var(titleSize) 
    set analysisplot(titleWeight) $var(titleWeight) 
    set analysisplot(titleSlant) $var(titleSlant) 

    set analysisplot(textlabFont) $var(textlabFont) 
    set analysisplot(textlabSize) $var(textlabSize) 
    set analysisplot(textlabWeight) $var(textlabWeight) 
    set analysisplot(textlabSlant) $var(textlabSlant) 

    set analysisplot(numlabFont) $var(numlabFont) 
    set analysisplot(numlabSize) $var(numlabSize) 
    set analysisplot(numlabWeight) $var(numlabWeight)
    set analysisplot(numlabSlant) $var(numlabSlant)

    set analysisplot(graph,x,format) $var(graph,x,format)
    set analysisplot(graph,y,format) $var(graph,y,format)

    set analysisplot(discrete) $var(discrete) 
    set analysisplot(discrete,symbol) $var(discrete,symbol) 
    set analysisplot(discrete,fill) $var(discrete,fill) 
    set analysisplot(discrete,color) $var(discrete,color) 

    set analysisplot(linear) $var(linear) 
    set analysisplot(linear,width) $var(linear,width) 
    set analysisplot(linear,color) $var(linear,color) 
    set analysisplot(linear,dash) $var(linear,dash) 

    set analysisplot(step) $var(step) 
    set analysisplot(step,width) $var(step,width) 
    set analysisplot(step,color) $var(step,color) 
    set analysisplot(step,dash) $var(step,dash) 

    set analysisplot(quadratic) $var(quadratic) 
    set analysisplot(quadratic,width) $var(quadratic,width) 
    set analysisplot(quadratic,color) $var(quadratic,color) 
    set analysisplot(quadratic,dash) $var(quadratic,dash) 

    set analysisplot(error) $var(error) 
    set analysisplot(error,width) $var(error,width) 
    set analysisplot(error,color) $var(error,color) 

    set analysisplot(bar) $var(bar) 
    set analysisplot(bar,color) $var(bar,color) 

    puts $ch "array set analysisplot \{ [array get analysisplot] \}"
    close $ch
}

proc PlotSaveData {varname} {
    upvar #0 $varname var
    global $varname

    if {$var(xdata) == {}} {
	return
    }

    PlotSaveDataFile $varname [SaveFileDialog apdatafbox]
}

proc PlotSaveDataFile {varname filename} {
    upvar #0 $varname var
    global $varname

    if {$var(xdata) == {}} {
	return
    }

    if {$filename == {}} {
	return
    }

    global $var(xdata) $var(ydata) $var(xedata) $var(yedata)
    set ll [$var(xdata) length]
    set xx [$var(xdata) range]
    set yy [$var(ydata) range]

    set ch [open $filename w]
    switch $var(dim) {
	xy {
	    for {set ii 0} {$ii<$ll} {incr ii} {
		puts $ch "[lindex $xx $ii] [lindex $yy $ii]"
	    }
	}
	xyxe {
	    set xe [$var(xedata) range]
	    for {set ii 0} {$ii<$ll} {incr ii} {
		puts $ch "[lindex $xx $ii] [lindex $yy $ii] [lindex $xe $ii]"
	    }
	}
	xyye {
	    set ye [$var(yedata) range]
	    for {set ii 0} {$ii<$ll} {incr ii} {
		puts $ch "[lindex $xx $ii] [lindex $yy $ii] [lindex $ye $ii]"
	    }
	}
	xyxeye {
	    set xe [$var(xedata) range]
	    set ye [$var(yedata) range]
	    for {set ii 0} {$ii<$ll} {incr ii} {
		puts $ch "[lindex $xx $ii] [lindex $yy $ii] [lindex $xe $ii] [lindex $ye $ii]"
	    }
	}
    }
    close $ch

    PlotRaise $varname
}

proc PlotStats {varname} {
    upvar #0 $varname var
    global $varname

    if {!$var(stats)} {
	return
    }

    set min {}
    set max {}
    set mean {}
    set median {}
    set varr {}
    set sdev {}

    if {$var(ydata) != {}} {
	if {[$var(ydata) length] > 0} {
	    set min [format "%6.3f" [blt::vector expr min($var(ydata))]]
	    set max [format "%6.3f" [blt::vector expr max($var(ydata))]]
	    set mean [format "%6.3f" [blt::vector expr mean($var(ydata))]]
	    set median [format "%6.3f" [blt::vector expr median($var(ydata))]]
	    set varr [format "%6.3f" [expr [blt::vector expr var($var(ydata))]]]
	    set sdev [format "%6.3f" [expr [blt::vector expr sdev($var(ydata))]]]
	}
    }
    
    set rr {}
    append rr "min $min\n"
    append rr "max $max\n"
    append rr "mean $mean\n"
    append rr "median $median\n"
    append rr "var $varr\n"
    append rr "sdev $sdev\n"

    SimpleTextDialog "${varname}stats" [msgcat::mc {Statistics}] 40 20 insert top $rr PlotStatsDestroyCB $varname
}

proc PlotStatsDestroyCB {varname} {
    upvar #0 $varname var
    global $varname

    set var(stats) 0
}

proc PlotTitle {varname title xaxis yaxis} {
    upvar #0 $varname var
    global $varname

    set var(graph,title) "$title"
    set var(graph,xaxis) "$xaxis"
    set var(graph,yaxis) "$yaxis"
}

proc PlotUpdateGraph {varname} {
    upvar #0 $varname var
    global $varname

    global ds9

    if {$var(graph,format)} {
	if {$var(graph,x,format) != {}} {
	    $var(graph) xaxis configure -command [list PlotAxisFormat $varname x]
	} else {
	    $var(graph) xaxis configure -command {}
	}
	if {$var(graph,y,format) != {}} {
	    $var(graph) yaxis configure -command [list PlotAxisFormat $varname y]
	} else {
	    $var(graph) yaxis configure -command {}
	}
    }

    # Menus
    if {$var(xdata) != {}} {
	$var(mb).file entryconfig "[msgcat::mc {Save Data}]..." -state normal
	$var(mb).file entryconfig [msgcat::mc {Clear Data}] -state normal
	$var(mb).file entryconfig [msgcat::mc {Statistics}] -state normal
	$var(mb).file entryconfig [msgcat::mc {List Data}] -state normal

	if {$var(1,manage)} {
	    $var(mb).file entryconfig [msgcat::mc {Duplicate Data}] \
		-state disabled
	} else {
	    $var(mb).file entryconfig [msgcat::mc {Duplicate Data}] \
		-state normal
	}
    } else {
	$var(mb).file entryconfig "[msgcat::mc {Save Data}]..." -state disabled
	$var(mb).file entryconfig [msgcat::mc {Clear Data}] -state disabled
	$var(mb).file entryconfig [msgcat::mc {Duplicate Data}] -state disabled
	$var(mb).file entryconfig [msgcat::mc {Statistics}] -state disabled
	$var(mb).file entryconfig [msgcat::mc {List Data}] -state disabled
    }

    # Graph
    $var(graph) configure -plotpadx 0 -plotpady 0 \
	-title $var(graph,title) \
	-font "{$ds9($var(titleFont))} $var(titleSize) $var(titleWeight) $var(titleSlant)"

    $var(graph) xaxis configure \
	-grid $var(graph,x,grid) -logscale $var(graph,x,log) \
	-title $var(graph,xaxis) \
	-tickfont "{$ds9($var(numlabFont))} $var(numlabSize) $var(numlabWeight) $var(numlabSlant)" \
	-titlefont "{$ds9($var(textlabFont))} $var(textlabSize) $var(textlabWeight) $var(textlabSlant)"
    $var(graph) yaxis configure \
	-grid $var(graph,y,grid) -logscale $var(graph,y,log) \
	-title $var(graph,yaxis) \
	-tickfont "{$ds9($var(numlabFont))} $var(numlabSize) $var(numlabWeight) $var(numlabSlant)" \
	-titlefont "{$ds9($var(textlabFont))} $var(textlabSize) $var(textlabWeight) $var(textlabSlant)"

    # windows: we're on top of the image window, so set focus to this window
    switch $ds9(wm) {
	x11 {}
	win32 {focus $var(top)}
	aqua {}
    }
}

