/*
 * QuteCom, a voice over Internet phone
 * Copyright (C) 2010 Mbdsys
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "QtChatHistoryWidget.h"

#include <presentation/qt/chat/QtChatTheme.h>
#include <presentation/qt/chat/QtChatUtils.h>

#include <model/config/Config.h>
#include <model/config/ConfigManager.h>
#include <model/profile/AvatarList.h>

#include <control/chat/CChatHandler.h>

#include <imwrapper/Account.h>

#include <util/Logger.h>
#include <util/WebBrowser.h>

#include <QtCore/QFile>
#include <QtCore/QFileInfo>
#include <QtCore/QTime>

#include <QtGui/QFileDialog>
#include <QtGui/QScrollBar>

// If previous message from last sender is more than MAX_SECONDS_BETWEEN_CONSECUTIVE_MESSAGES
// old, don't group new message with it.
static int MAX_SECONDS_BETWEEN_CONSECUTIVE_MESSAGES = 3 * 60;

QtChatHistoryWidget::QtChatHistoryWidget(QWidget * parent) 
: QWebView(parent) 
{
	_translate = true;
	//connect (this, SIGNAL(anchorClicked(const QUrl &)), this, SLOT(urlClicked(const QUrl &)));
	setContextMenuPolicy(Qt::NoContextMenu);
	Config & config = ConfigManager::getInstance().getCurrentConfig();

	_theme = new QtChatTheme();
	
	QWebFrame *	frame = page()->mainFrame (); 
	frame->setScrollBarPolicy(Qt::Horizontal,Qt::ScrollBarAlwaysOff);

	QString path = QString::fromUtf8(config.getResourcesDir().c_str()) + "pics/avatars/default-avatar.png";
#if defined(OS_WINDOWS)
	path = path.replace("\\","/");
	_avatarMap["defaut"] = path;
#else
	_avatarMap["defaut"] = "file://"+path;
#endif

	page()->setLinkDelegationPolicy(QWebPage::DelegateAllLinks);
	connect(this,SIGNAL(linkClicked(const QUrl &)),this,SLOT(urlClicked(const QUrl &)));
	
	//load(QUrl("/Users/laurent/Desktop/qutecom-2.2/debug/debug/QuteCom.app/Contents/Resources/chat/base.html"));
}

QtChatHistoryWidget::~QtChatHistoryWidget() {
	delete _theme;
}

void QtChatHistoryWidget::setTheme(const QString& themeDir,const QString & variant) {
	_theme->load(themeDir);
	Config & config = ConfigManager::getInstance().getCurrentConfig();
	
	QString Resources_path;
	
#if defined(OS_WINDOWS)
	Resources_path = _theme->getCurrentThemePath();
#else
	Resources_path = "file://"+_theme->getCurrentThemePath();
#endif

	Resources_path += "Contents/Resources/";
	
	page()->mainFrame()->addToJavaScriptWindowObject("parent",this);
	QFile file_base(QString::fromUtf8(config.getResourcesDir().c_str())+"chat/base.html");
	file_base.open(QIODevice::ReadOnly);
	
	QString base = file_base.readAll();
	
	if (variant.isEmpty())
		base.replace("%cssPath%","");
	else 
		base.replace("%cssPath%","Variants/"+variant+".css");

	setHtml(base,QUrl(Resources_path));
}

void QtChatHistoryWidget::saveHistoryAsHtml() {
	Config & config = ConfigManager::getInstance().getCurrentConfig();
	QString contentToSave =_data;

	QString filePath = QFileDialog::getSaveFileName(this, tr("Save As"), 
		QString::fromUtf8(config.getLastChatHistorySaveDir().c_str()), "HTML (*.htm *.html)");

	if(filePath.length() > 0) {

		if(!filePath.endsWith(QString(".htm"), Qt::CaseInsensitive) && 
			!filePath.endsWith(QString(".html"), Qt::CaseInsensitive)) {

			filePath.append(QString(".html"));
		}

		QFile fileToSave(filePath);
		config.set(Config::LAST_CHAT_HISTORY_SAVE_DIR_KEY, std::string(QFileInfo(fileToSave).absolutePath().toUtf8()));
		fileToSave.open(QIODevice::WriteOnly);
		fileToSave.write(contentToSave.toUtf8().data(), (long long)contentToSave.length());
		fileToSave.close();
	}
}

void QtChatHistoryWidget::urlClicked(const QUrl & link) {
	//setSource(QUrl(QString::null));
	QString url = link.toString();
	if(url.contains(_theme->getCurrentThemePath()))
		url = url.remove(_theme->getCurrentThemePath());

	WebBrowser::openUrl(std::string(url.toUtf8()));
}

void QtChatHistoryWidget::insertMessage(const QString & contactId, const QString & senderName, const QString & message, const QTime & time) {
	if (contactId == "self" || !_translate)
		insertMessage2(contactId,senderName,message,time);
	else
	{
		QString text = message;
		text = text.replace('\"',"\\\"");
		text = text.replace('\'',"\\\'");
		
		Config & config = ConfigManager::getInstance().getCurrentConfig();
		QString lang = QString::fromUtf8(config.getTranslationReceived().c_str());
		QString script = QString("translate(\"%1\",\"%2\",\"%3\",\"%4\",\"%5\")").arg(contactId).arg(senderName).arg(text).arg(time.toString()).arg(lang);
		page ()->currentFrame ()->evaluateJavaScript(script);
	}
}

void QtChatHistoryWidget::insertMessage2(const QString & contactId, const QString & senderName, const QString & message, const QTime & time) {
	// Prepare message
	QString htmlMessage = QtChatUtils::decodeMessage(_protocol, message);
	

	// Get html
	QtChatTheme::Position position;
	int timeDelta = _lastMessageTime.secsTo(time);
	if (_lastSenderName == senderName && timeDelta < MAX_SECONDS_BETWEEN_CONSECUTIVE_MESSAGES) 
	{
		position = QtChatTheme::Next;
	} 
	else 
	{
		if (!_lastSenderName.isEmpty()) 
		{
			QString footer =  _theme->getFooter();
			footer = footer.replace('\n',"");
			footer = footer.replace('\r',"");
			footer = footer.replace('\t',"");
			footer = footer.replace('\"',"\\\"");
			footer = footer.replace('\'',"\\\'");
			page ()->currentFrame ()->evaluateJavaScript("insertMessage(\"\")");
			page ()->currentFrame ()->evaluateJavaScript("insertMessage(\""+footer+"\")");
		}
		position = QtChatTheme::First;
		_lastSenderName = senderName;
	}
	_lastMessageTime = time;

	QtChatTheme::Direction direction;

	if (contactId == "self") { // FIXME: do not duplicate "self"
		direction = QtChatTheme::Outgoing;
	} else {
		direction = QtChatTheme::Incoming;
		
	}

	QString html = _theme->getContent(direction, position);
	QtChatTheme::setKeywordValue(html, "senderId", contactId);
	QtChatTheme::setKeywordValue(html, "sender", senderName);
	QtChatTheme::setKeywordValue(html, "message", "<div id='magic_id'>"+htmlMessage+"</div>");

	QRegExp reg("\\%(time|shortTime)(\\%|\\{.*\\}\\%)");
	html = html.replace(reg,time.toString("HH:mm"));

	QRegExp reg2("\\%timeOpened(\\%|\\{.*\\}\\%)");
	html = html.replace(reg2,QTime::currentTime().toString("HH:mm"));

	QtChatTheme::setKeywordValue(html, "service", "");

	QString avatar_path = _avatarMap["defaut"];
	if(_avatarMap.contains(contactId))
		avatar_path = _avatarMap[contactId];
	QtChatTheme::setKeywordValue(html, "userIconPath", avatar_path);

	//updateHtml();

	html = html.replace('\n',"");
	html = html.replace('\r',"");
	html = html.replace('\t',"");
	html = html.replace('\"',"\\\"");
	html = html.replace('\'',"\\\'");
	
	if(position == QtChatTheme::First)
		page ()->currentFrame ()->evaluateJavaScript("insertMessage(\"\")");
	
	page ()->currentFrame ()->evaluateJavaScript("insertMessage(\""+html+"\")");
	
	//delay for QWebView to update itself
	QTimer::singleShot(200,this,SLOT(scrollToBottom()));
}

void QtChatHistoryWidget::translationFinishedSlot(const QVariant  & traduction, const QVariant  & contactId, const QVariant  & senderName, const QVariant  & message, const QVariant  & time)
{
	QString tmp = traduction.toString().replace("&#39;","'");
	if(tmp.isEmpty())
	{
		insertMessage2(contactId.toString(),senderName.toString(),message.toString(),time.toTime());
	}
	else
	{
		
		QTextDocument doc;
		doc.setHtml(message.toString());
		
		QString str = "<div>"+tmp+"</div><div style=\"color:grey\">"+doc.toPlainText()+"</div>";
		insertMessage2(contactId.toString(),senderName.toString(),str,time.toTime());
	}
}

void QtChatHistoryWidget::insertStatusMessage(const QString & message, const QTime& time) {
	// Make sure we close any opened message group
	if (!_lastSenderName.isEmpty()) {
		QString footer = _theme->getFooter();
		page ()->currentFrame ()->evaluateJavaScript("insertMessage(\"\")");
		page ()->currentFrame ()->evaluateJavaScript("insertMessage(\""+footer+"\")");
		_lastSenderName = "";
	}
	QString html = _theme->getStatusMessage();
	QtChatTheme::setKeywordValue(html, "message", message);
	QtChatTheme::setKeywordValue(html, "time", time.toString("HH:mm"));
	
	html = html.replace('\n',"");
	html = html.replace('\r',"");
	html = html.replace('\t',"");
	html = html.replace('\"',"\\\"");
	html = html.replace('\'',"\\\'");
	page ()->currentFrame ()->evaluateJavaScript("insertMessage(\"\")");
	page ()->currentFrame ()->evaluateJavaScript("insertMessage(\""+html+"\")");
	
	//_data += html;
	
}

void QtChatHistoryWidget::setAvatarPixmap(const QString& name, const QPixmap& pixmap) 
{
	QDir dir(QString::fromUtf8(Config::getConfigDir().c_str()));
	
	if(!dir.exists("cache"))
		dir.mkdir("cache");
	dir.cd("cache");
	
	//QString path = dir.absolutePath()+QDir::separator()+name+".png";
	QString path = dir.absolutePath()+"/"+name+".png";

	pixmap.save(path,"PNG");
	
#if defined(OS_WINDOWS)
	_avatarMap[name] = path;
#else
	_avatarMap[name] = "file://"+path;
#endif
}

void QtChatHistoryWidget::scrollToBottom() {
	
	QWebFrame *	frame = page()->mainFrame ();
	frame->setScrollBarValue(Qt::Vertical,frame->scrollBarMaximum(Qt::Vertical));
}

void QtChatHistoryWidget::clear() 
{
	_data  = "";
	_lastSenderName = "";
}


void QtChatHistoryWidget::updateHtml()
{
#if defined(OS_WINDOWS)
	setHtml(_data,QUrl(_theme->getCurrentThemePath()));
#else
	setHtml(_data,QUrl("file://"+_theme->getCurrentThemePath()));
#endif
}
