/*
 * QuteCom, a voice over Internet phone
 * Copyright (C) 2010 Mbdsys
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "QuteComAccount.h"

#include "QuteComBuildId.h"
#include "QuteComAccountParser.h"

#include <model/config/ConfigManager.h>
#include <model/config/Config.h>
#include <model/network/NetworkObserver.h>
#include <model/network/NetworkProxyDiscovery.h>

#include <thread/Thread.h>
#include <util/StringList.h>
#include <util/Logger.h>
#include <http/HttpRequest.h>

#include <sstream>
#include <exception>

using namespace std;

unsigned short QuteComAccount::_testStunRetry = 2;


QuteComAccount::QuteComAccount()
	: SipAccount(), QuteComWebService(this) {

	Config & config = ConfigManager::getInstance().getCurrentConfig();
	_protocol = EnumIMProtocol::IMProtocolQuteCom;
	_ssoRequestOk = false;
	_qutecomLoginOk = false;
	_ssoWithSSL = false;
	_discoveringNetwork = false;
	_needsHttpTunnel = false;
	_stunServer = config.getNetworkStunServer();
	_lastNetworkDiscoveryState = NetworkDiscoveryStateError;

	_ssoTimer.timeoutEvent += boost::bind(&QuteComAccount::ssoTimeoutEventHandler, this);
	_ssoTimer.lastTimeoutEvent += boost::bind(&QuteComAccount::ssoLastTimeoutEventHandler, this);
}

QuteComAccount::QuteComAccount(const std::string & login, const std::string & password, bool rememberPassword)
	: SipAccount(),
	QuteComWebService(this) {

	Config & config = ConfigManager::getInstance().getCurrentConfig();
	_protocol = EnumIMProtocol::IMProtocolQuteCom;
	_qutecomLogin = login;
	_qutecomPassword = password;
	_rememberPassword = rememberPassword;
	_ssoRequestOk = false;
	_qutecomLoginOk = false;
	_ssoWithSSL = false;
	_needsHttpTunnel = false;
	_stunServer = config.getNetworkStunServer();
	_discoveringNetwork = false;
	_lastNetworkDiscoveryState = NetworkDiscoveryStateError;

	_ssoTimer.timeoutEvent += boost::bind(&QuteComAccount::ssoTimeoutEventHandler, this);
	_ssoTimer.lastTimeoutEvent += boost::bind(&QuteComAccount::ssoLastTimeoutEventHandler, this);
}

QuteComAccount::QuteComAccount(const QuteComAccount & qutecomAccount)
	: SipAccount(*this),
	QuteComWebService(this) {
	copy(qutecomAccount);
	
	_ssoTimer.timeoutEvent += boost::bind(&QuteComAccount::ssoTimeoutEventHandler, this);
	_ssoTimer.lastTimeoutEvent += boost::bind(&QuteComAccount::ssoLastTimeoutEventHandler, this);
}

QuteComAccount & QuteComAccount::operator = (const QuteComAccount & qutecomAccount) {
	copy(qutecomAccount);

	return *this;
}

bool QuteComAccount::operator == (const SipAccount & other) const {
	if (other.getType() != SipAccount::SipAccountTypeQuteCom) {
		return false;
	}

	const QuteComAccount & gother = dynamic_cast<const QuteComAccount &>(other);

	return (	_qutecomLogin == gother._qutecomLogin &&
				_qutecomPassword == gother._qutecomPassword);
}

QuteComAccount * QuteComAccount::clone() const {
	return new QuteComAccount(*this);
}

void QuteComAccount::copy(const QuteComAccount & qutecomAccount) {
	SipAccount::copy(qutecomAccount);

	_qutecomLogin = qutecomAccount._qutecomLogin;
	_qutecomPassword = qutecomAccount._qutecomPassword;
	_rememberPassword = qutecomAccount._rememberPassword;
	_ssoRequestOk = qutecomAccount._ssoRequestOk;
	_qutecomLoginOk = qutecomAccount._qutecomLoginOk;
	_ssoWithSSL = qutecomAccount._ssoWithSSL;
	_stunServer = qutecomAccount._stunServer;
	_discoveringNetwork = false;
	_lastNetworkDiscoveryState = qutecomAccount._lastNetworkDiscoveryState;
}

QuteComAccount::~QuteComAccount() {
	Mutex::ScopedLock lock(_mutex);

	_ssoTimer.stop();
}

EnumSipLoginState::SipLoginState QuteComAccount::discoverNetwork() {
	static const unsigned LOGIN_TIMEOUT = 10000;
	static const unsigned LIMIT_RETRY = 5;

	if (!discoverForSSO()) {
		LOG_ERROR("error while discovering network for SSO");
		_lastNetworkDiscoveryState = NetworkDiscoveryStateHTTPError;
		return EnumSipLoginState::SipLoginStateNetworkError;
	}

	_ssoTimerFinished = false;
	_ssoTimer.start(0, LOGIN_TIMEOUT, LIMIT_RETRY);
	while (!_ssoTimerFinished) {
		Thread::msleep(100);
	}

	if (!_ssoRequestOk) {
		LOG_ERROR("error while doing SSO request");
		_lastNetworkDiscoveryState = NetworkDiscoveryStateError;
		return EnumSipLoginState::SipLoginStateNetworkError;
	} else if (_ssoRequestOk && !_qutecomLoginOk) {
		LOG_ERROR("SSO request Ok but login/password are invalid");
		_lastNetworkDiscoveryState = NetworkDiscoveryStateOk;
		return EnumSipLoginState::SipLoginStatePasswordError;
	}

	if (!discoverForSIP()) {
		LOG_ERROR("error while discovering network for SIP");
		_lastNetworkDiscoveryState = NetworkDiscoveryStateSIPError;
		return EnumSipLoginState::SipLoginStateNetworkError;
	}

	LOG_DEBUG("initialization Ok");
	_lastNetworkDiscoveryState = NetworkDiscoveryStateOk;
	return EnumSipLoginState::SipLoginStateReady;
}

bool QuteComAccount::discoverForSSO() {
	//
	// Please contact network@openqutecom.com before any modifications.
	//

	LOG_DEBUG("discovering network parameters for SSO connection");

	Config & config = ConfigManager::getInstance().getCurrentConfig();

	string url = config.getQuteComServerHostname() + ":" + String::fromNumber(443) + config.getQuteComSSOPath();
	if (_networkDiscovery.testHTTP(url, true)) {
		_ssoWithSSL = true;
		LOG_DEBUG("SSO can connect on port 443 with SSL");
		return true;
	}

	url = config.getQuteComServerHostname() + ":" + String::fromNumber(80) + config.getQuteComSSOPath();
	if (_networkDiscovery.testHTTP(url, false)) {
		_ssoWithSSL = false;
		LOG_DEBUG("SSO can connect on port 80 without SSL");
		return true;
	}

	LOG_ERROR("SSO cannot connect");
	return false;
}

bool QuteComAccount::discoverForSIP() {
	//
	// Please contact network@openqutecom.com before any modifications.
	//

	LOG_DEBUG("discovering network parameters for SIP connection");

	Config & config = ConfigManager::getInstance().getCurrentConfig();

	_localSIPPort = _networkDiscovery.getFreeLocalPort();
	
	if (!config.getNetWorkTunnelNeeded()) {
		LOG_DEBUG("SIP will use " + String::fromNumber(_localSIPPort) + " as local SIP port");

		// Stun test
		unsigned short iTestStun;
		for (iTestStun = 0; iTestStun < _testStunRetry; iTestStun++) {
			LOG_DEBUG("testUDP (Stun): " + String::fromNumber(iTestStun + 1));
			if (_networkDiscovery.testUDP(_stunServer)) {
				break;
			}
		}

		if (iTestStun == _testStunRetry) {
			// Stun test failed
			_networkDiscovery.setNatConfig(EnumNatType::NatTypeFullCone);
		}
		////

		// SIP test with UDP
		for (unsigned short i = 0; i < _testSIPRetry; i++) {
			LOG_DEBUG("testSIP test number: " + String::fromNumber(i + 1));

			if (_networkDiscovery.testSIP(_sipProxyServerHostname,_username, _sipProxyServerPort, _localSIPPort)) {
				LOG_DEBUG("SIP can connect via UDP");
				_needsHttpTunnel = false;
				return true;
			}
		}
		////
	}

	LOG_DEBUG("cannot connect via UDP");

	// SIP test with Http
	if (_networkDiscovery.testSIPHTTPTunnel(_httpTunnelServerHostname, 80, false,
		_sipProxyServerHostname, _sipProxyServerPort)) {

		_needsHttpTunnel = true;
		_httpTunnelServerPort = 80;
		_httpTunnelWithSSL = false;

		LOG_DEBUG("SIP can connect via a tunnel on port 80 without SSL");
		return true;
	}

	if (_networkDiscovery.testSIPHTTPTunnel(_httpTunnelServerHostname, 443, false,
		_sipProxyServerHostname, _sipProxyServerPort)) {

		_needsHttpTunnel = true;
		_httpTunnelServerPort = 443;
		_httpTunnelWithSSL = false;

		LOG_DEBUG("SIP can connect via a tunnel on port 443 without SSL");
		return true;
	}

	if (_networkDiscovery.testSIPHTTPTunnel(_httpsTunnelServerHostname, 443, true,
		_sipProxyServerHostname, _sipProxyServerPort)) {

		_needsHttpTunnel = true;
		_httpsTunnelServerPort = 443;
		_httpTunnelWithSSL = true;

		LOG_DEBUG("SIP can connect via a tunnel on port 443 with SSL");
		return true;
	}
	////

	LOG_ERROR("SIP cannot connect");
	return false;
}

void QuteComAccount::ssoTimeoutEventHandler() {
	Config & config = ConfigManager::getInstance().getCurrentConfig();
	setHostname(config.getQuteComServerHostname());
	setGet(false);
	setHttps(_ssoWithSSL);
	setServicePath(config.getQuteComSSOPath());
	setQuteComAuthentication(true);

	LOG_DEBUG("setting proxy settings for SSO request");
	NetworkProxy networkProxy = NetworkProxyDiscovery::getInstance().getNetworkProxy();

	HttpRequest::setProxy(networkProxy.getServer(), networkProxy.getServerPort(),
		networkProxy.getLogin(), networkProxy.getPassword());

	if (_ssoWithSSL) {
		setPort(443);
		LOG_DEBUG("sending SSO request with SSL");
	} else {
		setPort(80);
		LOG_DEBUG("sending SSO request without SSL");
	}

	call(this);
}

void QuteComAccount::ssoLastTimeoutEventHandler() {
	_ssoTimerFinished = true;
}

void QuteComAccount::answerReceived(const std::string & answer, int requestId) {
	if (!answer.empty()) {
		LOG_DEBUG("SSO request has been processed successfully");
		_ssoRequestOk = true;
		QuteComAccountParser parser(*this, answer);
		if (parser.isLoginPasswordOk()) {
			LOG_DEBUG("login/password Ok");
			_qutecomLoginOk = true;
			_ssoTimer.stop();
			_ssoTimerFinished = true;
			//SIP connection test can now be launched
		} else {
			LOG_DEBUG("login/password not Ok");
			_qutecomLoginOk = false;
			_ssoTimer.stop();
			_ssoTimerFinished = true;
		}
	}
}

bool QuteComAccount::isEmpty() const {
	return (_qutecomLogin.empty() || _qutecomPassword.empty());
}
