
##############################################################################
#
# Copyright (c) 2003-2017 by The University of Queensland
# http://www.uq.edu.au
#
# Primary Business: Queensland, Australia
# Licensed under the Apache License, version 2.0
# http://www.apache.org/licenses/LICENSE-2.0
#
# Development until 2012 by Earth Systems Science Computational Center (ESSCC)
# Development 2012-2013 by School of Earth Sciences
# Development from 2014 by Centre for Geoscience Computing (GeoComp)
#
##############################################################################

from __future__ import print_function, division

__copyright__="""Copyright (c) 2003-2017 by The University of Queensland
http://www.uq.edu.au
Primary Business: Queensland, Australia"""
__license__="""Licensed under the Apache License, version 2.0
http://www.apache.org/licenses/LICENSE-2.0"""
__url__="https://launchpad.net/escript-finley"

"""
test for util operations for unary operations without tagged data

:remark: use see `test_util`
:var __author__: name of author
:var __copyright__: copyrights
:var __license__: licence agreement
:var __url__: url entry point on documentation
:var __version__: version
:var __date__: date of the version
"""

__author__="Lutz Gross, l.gross@uq.edu.au"

import esys.escriptcore.utestselect as unittest
import numpy
import math
import cmath
from esys.escript import *
from test_util_base import Test_util_base

haveLapack = hasFeature('lapack')

class Test_util_unary_no_tagged_data(Test_util_base):
   """
   test for unary operations. No tagged data are tested.
   """
   def iterateops(self, ops, vals):
      for p in ops:
          o,c,z=p
          for v in vals:
            res=o(v)
            if isinstance(v,complex):
               ref=z(v)
            else:
               ref=c(v)
            self.assertTrue(isinstance(res,type(ref)),"wrong type of result.")
            self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
            d=Data(v)
            res=o(d)
            self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result for data")

   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_complex_nonzero(self):
      #Compare results of unary ops provided by util and their python equivalents
      ops=[(sin,math.sin,cmath.sin), (cos,math.cos,cmath.cos), (tan,math.tan,cmath.tan), (log,math.log,cmath.log), (log10, math.log10, cmath.log10), (Abs, abs, abs),
(acos,math.acos,cmath.acos), (acosh,math.acosh,cmath.acosh), (asin,math.asin,cmath.asin), (asinh, math.asinh,cmath.asinh),
(cosh, math.cosh, cmath.cosh), (exp, math.exp, cmath.exp), (sinh, math.sinh, cmath.sinh), (sqrt, math.sqrt, cmath.sqrt)]
      vals=[1+0j,-1+0j,1j, -1j, math.pi*1j,3+4j]
      self.iterateops(ops,vals)
      ops=[(atan,math.atan,cmath.atan)]
      vals=[1+0j,-1+0j, math.pi*1j,3+4j]
      self.iterateops(ops,vals)
      ops=[(atanh,math.atanh,cmath.atanh)]
      vals=[1j, -1j, math.pi*1j,3+4j]
      self.iterateops(ops,vals)

   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log10_float_rank0(self):
      arg=52.2519689858
      res=log10(arg)
      ref=1.7181026604
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log10_array_rank0(self):
      arg=numpy.array(45.1686859861)
      res=log10(arg)
      ref=numpy.array(1.65483745657)
      if not isinstance(res,float):
         self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
         self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log10_array_rank1(self):
      arg=numpy.array([13.954560287699826, 23.475641658812407])
      res=log10(arg)
      ref=numpy.array([1.1447161562933539, 1.3706174716547848])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log10_array_rank2(self):
      arg=numpy.array([[25.915081952650038, 58.337204652970946, 15.450038105653986, 38.930671788011558, 87.005181288770302], 
[3.6208586171272144, 84.49833125689257, 69.261223337588021, 54.98351177434435, 58.379918137373565], [99.26000663643724, 
45.338259591938282, 53.685949933590322, 95.061307572866596, 47.187722694142664], [5.6737763969659527, 44.873622862711144, 
97.056192800604094, 55.273029732289778, 85.494701869969745]])
      res=log10(arg)
      ref=numpy.array([[1.4135525865703951, 1.7659456151718125, 1.1889295548970895, 1.5902918979721861, 1.9395451162751614], 
[0.55881156734663273, 1.9268481322268585, 1.8404901580778943, 1.7402324746037254, 1.7662634816675511], [1.9967742998978548, 
1.6564648447687793, 1.7298606420366613, 1.9780037837413516, 1.6738290185543834], [0.75387221628934742, 1.6519911336241999, 
1.9870232513506001, 1.7425132706239763, 1.9319392022143682]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log10_array_rank3(self):
      arg=numpy.array([[[61.549373713195962, 41.673709308293191], [18.017632188926811, 13.171469559653669]], 
[[50.290930361991521, 14.626533261319382], [15.969153538380304, 56.425345549785291]], [[90.382517465043676, 
98.988183466908609], [33.988560486114899, 61.488838120931966]], [[27.205433883156655, 77.280843084053075], [42.537512400205529, 
20.752110100745131]], [[2.7510631963546697, 37.227526331006125], [17.709630263115471, 77.944115343628809]], 
[[15.848941645935733, 77.546800032084079], [44.6765341255255, 59.885570390022544]]])
      res=log10(arg)
      ref=numpy.array([[[1.7892236381887503, 1.6198621580149473], [1.2556977170065395, 1.1196342325201316]], 
[[1.7014896699685444, 1.1651414030979794], [1.203281896517113, 1.7514742273848101]], [[1.9560844337638301, 1.9955833545833324], 
[1.5313327713445872, 1.7887962867926523]], [[1.434655656599509, 1.8880718513054029], [1.6287720887060415, 1.3170622629043851]], 
[[0.43950056694834205, 1.5708641794769067], [1.2482094942016551, 1.8917833322219468]], [[1.2000002663849119, 
1.8895638813619857], [1.6500794744582843, 1.7773221904207173]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log10_array_rank4(self):
      arg=numpy.array([[[[49.041458121649541, 12.453861805081642, 22.352539133207308, 35.856582214340698], [47.729104473308858, 
67.27333693612141, 48.837394467178136, 9.6698413270700367], [64.583260503019162, 46.411663014624139, 85.71255093334365, 
36.363958914509439]], [[80.036726405255635, 40.387280979035751, 15.276817994504155, 59.062184549366151], [59.167871518749848, 
21.930421722965473, 0.23764880334288624, 24.921348290633865], [35.111521136514753, 93.225181218111516, 48.850751790389111, 
51.591988234960553]]], [[[47.509673966789677, 54.481428169403792, 36.464944550283505, 90.799221097408463], [46.169280587237125, 
47.324905848019064, 87.626395482722373, 98.929522440449787], [61.773097039759605, 82.412988902042329, 4.5402354395996527, 
41.873043607198412]], [[36.545836359312922, 49.472492642337876, 54.796411622124296, 9.1573852487394394], [94.483164799633187, 
59.013891170601127, 81.50036423642824, 4.9151253236939265], [53.788789868327207, 58.332866884869581, 96.756911476142037, 
37.843173549368686]]], [[[80.58432601195986, 21.263772240505372, 63.6235430125706, 35.501558596702829], [5.6711036927454339, 
74.217470766535399, 40.89063067086159, 30.690400010085241], [35.999004111325121, 20.274838444258663, 70.564641379269005, 
99.740954368148309]], [[31.256278345889452, 17.526864558417355, 12.375337394164927, 31.216749887776125], [31.235942054585749, 
4.4714795388662898, 16.839093071461757, 98.544202438777774], [7.1017853296580133, 80.078083678719835, 33.756128516966982, 
25.658869355266614]]]])
      res=log10(arg)
      ref=numpy.array([[[[1.6905633743319768, 1.0953040422410532, 1.3493268638843343, 1.5545688910211573], [1.6787832859072254, 
1.8278429703764396, 1.6887524857660539, 0.98541934778069684], [1.8101199667066123, 1.6666271302620568, 1.9330444205462154, 
1.5606711584094208]], [[1.9032893171804688, 1.6062446158451591, 1.1840329045447144, 1.771309506296336], [1.7720859464227217, 
1.3410469832596377, -0.62406436821075306, 1.3965715347504626], [1.5454496448138082, 1.969533236248409, 1.6888712516966586, 
1.7125822648927624]]], [[[1.6767820501011672, 1.7362484835956555, 1.5618755574979692, 1.9580821230304646], [1.6643531073987192, 
1.6750897586162579, 1.9426349473298661, 1.9953259126255949], [1.7907993755575464, 1.9159956651428329, 0.65707837432577154, 
1.6219345294073721]], [[1.5628379052561994, 1.6943637925865513, 1.738752119365097, 0.96177148522339684], [1.9753544319382004, 
1.7709542514498018, 1.9111595496665708, 0.69153459575815146], [1.7306917737033041, 1.7659133212214235, 1.98568199703713, 
1.5779875494313049]]], [[[1.9062505779281724, 1.3276403118294007, 1.8036178500584357, 1.5502474199590055], 
[0.75366758817392854, 1.8705061500821059, 1.6116238088855386, 1.4870025489414542], [1.5562904864634381, 1.3069574023136983, 
1.8485871387211481, 1.998873519435157]], [[1.4949372657476212, 1.2437042305963351, 1.0925570481249276, 1.4943876847840263], 
[1.4946546085032237, 0.65045124779595043, 1.2263186973378477, 1.9936310789157055], [0.85136754045869678, 1.9035136716426462, 
1.5283526316700879, 1.4092375155000518]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log10_constData_rank0(self):
      arg=Data(32.7103924665,self.functionspace)
      res=log10(arg)
      ref=Data(1.51468575492,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log10_constData_rank1(self):
      arg=Data(numpy.array([38.13113991980368, 3.9574358995083889]),self.functionspace)
      res=log10(arg)
      ref=Data(numpy.array([1.5812797885802696, 0.59741388911487381]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log10_constData_rank2(self):
      arg=Data(numpy.array([[94.835349573618515, 17.653263208696131, 62.502954853493407, 19.234729096475913, 
43.672005218091151], [15.048159867737148, 98.672070942815608, 52.576472580061754, 49.935023943546007, 26.60036022461162], 
[6.721528672936774, 75.440376910024227, 63.249503917897741, 63.687804000958671, 32.436637160710369], [68.994703422400121, 
20.092325091226218, 54.808697863191284, 46.361425111883918, 79.716043096984109]]),self.functionspace)
      res=log10(arg)
      ref=Data(numpy.array([[1.9769702493968937, 1.2468249965531741, 1.7959005492838003, 1.28408607404631, 1.6402031331890035], 
[1.1774833963984679, 1.9941942433294959, 1.7207914453990389, 1.6984052624540111, 1.4248875179281264], [0.82746805560156844, 
1.8776038495269229, 1.8010571235780615, 1.804056274373053, 1.5110358227817056], [1.8388157521471691, 1.3030301963730122, 
1.7388494842917164, 1.6661567773501147, 1.9015457332824068]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log10_constData_rank3(self):
      arg=Data(numpy.array([[[60.505360366376287, 95.665159725837654], [45.535060469642595, 8.7120300613775825]], 
[[85.791369386166309, 55.438032107346338], [38.874954040860693, 0.032497264299415379]], [[56.341710995337245, 
51.491208922462491], [88.530577620189433, 12.245118647390212]], [[2.7511376879554934, 34.207282964733224], [30.122953960922267, 
38.812210597586919]], [[86.660286634007576, 99.90443738735307], [17.396122538609294, 56.668511719223552]], 
[[36.718231261528388, 50.782102049171812], [88.979825368021281, 69.352170891166125]]]),self.functionspace)
      res=log10(arg)
      ref=Data(numpy.array([[[1.7817938519155039, 1.9807538009560601], [1.658345917646564, 0.9401193653092289]], 
[[1.9334436000107917, 1.7438078056569788], [1.5896698885992053, -1.4881531974726712]], [[1.7508300315839129, 
1.7117330884128066], [1.9470932977951618, 1.0879629975176723]], [[0.43951232634857773, 1.5341185801691359], 
[1.4788975580341073, 1.5889683792252702]], [[1.9378201211553368, 1.999584778416601], [1.2404524581824454, 1.7533418069486038]], 
[[1.5648817528162677, 1.7057106738803878], [1.9492915490656653, 1.8410600600927651]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log10_constData_rank4(self):
      arg=Data(numpy.array([[[[7.1178859886623451, 14.637867086244679, 36.35859711294826, 37.7723554870581], 
[44.508345724124766, 25.973407994350175, 49.071421558624728, 42.354776612571754], [91.85968925895304, 94.407818198616539, 
43.315126893833394, 75.615056315689174]], [[6.2918566984022908, 13.126791472139409, 5.4081624176919272, 67.758267396244534], 
[6.3749374576688131, 28.038450788587877, 57.310863148910528, 40.510465526946518], [93.706477842024938, 56.835264912992365, 
40.743079267474769, 55.242148230373495]]], [[[83.448958312190257, 44.295439593214205, 2.2473909991848409, 76.158191501497342], 
[89.137647290369543, 55.865574608149423, 34.00060537625113, 37.86772086689443], [12.457637095864028, 82.46283305226747, 
8.2656512835811657, 64.122937473337672]], [[54.25835998622415, 31.151251379554999, 71.605899030813731, 23.688427104278972], 
[93.960338747791468, 24.330099432103534, 39.06736609968543, 33.657862341058689], [4.584970065153624, 11.444794976152613, 
92.427741892222471, 86.464234082604506]]], [[[73.237800540520212, 40.200704386760357, 61.394401252086411, 10.275754214437864], 
[77.622206392059113, 6.1972834106359214, 63.472374671706035, 80.460731554212771], [75.404584514400938, 93.73627374151657, 
66.710853225948313, 60.813357606195645]], [[45.25139607197994, 74.577688908930867, 95.208049640850376, 14.393702972960131], 
[65.862630893385798, 71.877942440445722, 27.710516935466455, 70.580992209569118], [67.600547228807855, 78.088363930903228, 
38.219137036754042, 28.047017615696081]]]]),self.functionspace)
      res=log10(arg)
      ref=Data(numpy.array([[[[0.85235102737513713, 1.1654777993891423, 1.5606071177331162, 1.5771740683100366], 
[1.6484414528258733, 1.4145289375397965, 1.6908286393162306, 1.6269023955286619], [1.9631249719620456, 1.9750079610345306, 
1.636639591024321, 1.8786082799426707]], [[0.79877882269221356, 1.1181585862396579, 0.73304972584189554, 1.8309622923953988], 
[0.80447592841225835, 1.4477540138397433, 1.7582369493404233, 1.6075672339172933], [1.9717696143008383, 1.7546178886111818, 
1.6100538488746503, 1.7422705588721348]]], [[[1.9214209197798999, 1.6463590160322708, 0.35167863717923287, 1.8817166224792843], 
[1.9500611670813865, 1.7471442700825282, 1.5314866496665132, 1.5782691671562759], [1.0954356752650989, 1.9162582511278587, 
0.91727707904742606, 1.8070134091786803]], [[1.7344666626342538, 1.4934754974259916, 1.8549488017910998, 1.3745362248168791], 
[1.9729445738360003, 1.386143883806513, 1.5918141323168808, 1.527086529852433], [0.66133650454670789, 1.0586080170609664, 
1.9658023428913385, 1.9368364987628319]]], [[[1.8647352932012256, 1.6042336627512135, 1.788128768271634, 1.0118137078374039], 
[1.8899859832929091, 0.79220135751176601, 1.8025847467388194, 1.9055839768869141], [1.8773977512912479, 1.9719076852022974, 
1.8241964952763503, 1.7839989821922779]], [[1.6556319823700489, 1.872608920864252, 1.9786736686240998, 1.1581725363856989], 
[1.8186390746853227, 1.8565956366123635, 1.442644627555113, 1.8486877592678144], [1.8299502115849742, 1.8925863237992646, 
1.5822808767428842, 1.4478866872710365]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log10_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(96.1097445547)+(1.-msk_arg)*(15.7194730216)
      res=log10(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(1.98276742296)+(1.-msk_ref)*(1.19643798269)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log10_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([67.188570100204331, 83.852182235203458])+(1.-msk_arg)*numpy.array([12.970539131997421, 
55.494059090853888])
      res=log10(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([1.8272953985804494, 1.9235143695106758])+(1.-msk_ref)*numpy.array([1.112958028295743, 
1.7442464922729273])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log10_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[45.930333300115343, 70.753859497865847, 22.04664793382257, 79.15163171404815, 
75.405024274312993], [93.445968445291669, 30.36616470571791, 6.5043375890994559, 31.426752793912506, 21.34925350080275], 
[55.608242339925667, 60.535785665483225, 34.569731451078027, 21.386089927826287, 3.9072078681127631], [19.365485690555648, 
32.043862755374363, 58.335981689330417, 80.08022377912441, 29.309322772405462]])+(1.-msk_arg)*numpy.array([[25.745723389769719, 
64.392268446036255, 85.501520058738976, 45.362572337757612, 97.503453294299518], [2.8517470840863286, 14.034055548650054, 
33.880935693055591, 29.696281993337909, 92.433531843558612], [37.125936326201604, 37.497435106305446, 91.813274646371681, 
77.728476618025766, 45.331358346905688], [71.362205298919307, 94.671089912047734, 25.688825508752888, 56.379550500341139, 
4.1755783242915809]])
      res=log10(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[1.6620995969501617, 1.849750134838158, 1.3433425668419521, 1.898459872288744, 
1.8774002840914414], [1.9705605692811208, 1.4823899433069487, 0.81320307398795078, 1.4972995093490966, 1.3293826940612745], 
[1.7451391681548531, 1.7820121832883282, 1.5386960058146677, 1.3301313886858133, 0.59186651682317182], [1.2870283936579181, 
1.5057448629748622, 1.7659365106572738, 1.9035252780814409, 1.4670057836514949]])+(1.-msk_ref)*numpy.array([[1.410705098912199, 
1.8088337249250597, 1.9319738357486911, 1.6566976737600356, 1.9890199974431519], [0.45511100615307598, 1.1471831911912613, 
1.5299553957767158, 1.47270207858137, 1.9658295475498999], [1.5696774154882256, 1.5740015622004013, 1.9629054773764056, 
1.8905801561543689, 1.6563987328183856], [1.853468262533325, 1.9762173770124165, 1.4097442488078882, 1.7511216090311554, 
0.62071663446741465]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log10_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[22.388258065770664, 51.715962865840815], [65.432613978623507, 8.3209401983813631]], 
[[88.431488726297857, 89.785784570369586], [3.2671698372353806, 14.647277245749521]], [[68.153001107883583, 
19.918374467307398], [62.741803756290338, 32.019573139182008]], [[67.165624226669038, 2.8759095839159929], [73.719522685630153, 
94.724688180711894]], [[48.844734513795146, 28.150257589983699], [42.654320101950383, 50.476435941718051]], 
[[37.462663753206783, 82.654691621504796], [52.152313992909285, 
3.1027027366472075]]])+(1.-msk_arg)*numpy.array([[[1.8076283580543755, 4.753292603600527], [6.7876691754431446, 
88.85524686992207]], [[20.447379893795944, 10.467051425047025], [48.020952347676165, 29.139882960928777]], 
[[32.353404155614214, 49.731545888765915], [50.076198518791642, 12.61911728105847]], [[50.812406051522565, 24.959142420006426], 
[32.119283687346766, 27.446174244224839]], [[78.480705204997335, 51.479031400667225], [80.539043834605735, 
72.176036709299979]], [[65.812691581415578, 73.827132926837777], [61.203847954462837, 57.906796690274135]]])
      res=log10(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[1.3500203043046504, 1.7136246149415175], [1.8157942703490402, 0.92017240079787166]], 
[[1.9466069363367877, 1.953207581975233], [0.51417171098507863, 1.1657569020250118]], [[1.8334849846886128, 
1.2992538928886661], [1.7975570000190917, 1.5054155379351397]], [[1.8271470553686036, 0.45877522808649995], [1.86758251461448, 
1.9764631843179037]], [[1.6888177534210882, 1.4494823732334281], [1.6299630238135752, 1.7030886824981748]], 
[[1.573598654304283, 1.9172675099038647], [1.7172735827923957, 
0.49174016874627252]]])+(1.-msk_ref)*numpy.array([[[0.25710914591980127, 0.6769945494552323], [0.83172066706037573, 
1.9486830778292807]], [[1.3106376658608263, 1.0198243578793709], [1.6814307686997538, 1.4644878031125961]], 
[[1.5099199829403827, 1.6966319593433428], [1.6996313524512594, 1.1010289766693471]], [[1.7059697599596824, 1.397229659189178], 
[1.5067658512361108, 1.4384818161662032]], [[1.8947628968344679, 1.711630366878371], [1.9060064693262986, 1.8583930306392809]], 
[[1.8183096527670626, 1.8682160029445969], [1.7867787275843738, 1.7627295411341]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log10_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[51.471579704080902, 1.402847457439917, 48.558207798874633, 89.648969258612524], 
[86.525832839744794, 67.293178311636296, 36.293476074354814, 28.027447132690767], [10.633474820618499, 52.422445043400536, 
16.892564122675008, 91.588281668647483]], [[69.006862669320284, 50.632932996111499, 20.819307520773513, 31.458894256989172], 
[34.054758838486975, 67.499730360187598, 74.656828581187256, 25.901320636052535], [69.566935512371245, 69.265794475105736, 
49.758452930544813, 28.779752034461762]]], [[[72.22373251870431, 78.006053500246296, 31.060760583589399, 13.641516246702956], 
[19.276675259486392, 50.172392044390563, 63.569136215761631, 41.789254015859484], [77.72177399967164, 17.666602305012276, 
46.52250820815128, 99.530462341866837]], [[77.258789628594741, 13.343315652586506, 70.821037532780295, 52.515534097664812], 
[78.878814482604369, 32.375661085180987, 46.503085042716158, 45.997964823991765], [12.861656668310804, 1.89832465373334, 
25.450959358908168, 91.571383479978465]]], [[[1.2791555124757796, 61.193118582110273, 75.060163212570316, 55.342467763428438], 
[71.762210006922828, 84.536864679041486, 73.714524745683775, 11.850589611465555], [75.639151162482378, 64.29578756080727, 
33.11720623917023, 37.534286641607828]], [[45.364350792710475, 93.649088149678789, 32.500785342140865, 3.4368793436320275], 
[19.133668646879922, 58.565788824469315, 98.054154704002983, 23.793410565976142], [77.647872455877192, 54.998750332692239, 
19.370566893444124, 63.765617721508789]]]])+(1.-msk_arg)*numpy.array([[[[63.565214992977019, 60.995683199558883, 
10.071281252885441, 99.995081907465718], [41.291004315699134, 3.2589259834977984, 64.44648163924586, 36.675923847946201], 
[53.577021195107271, 66.247159776521869, 11.509715616484302, 32.399165283870758]], [[25.855485901204684, 43.242105604769712, 
13.014136372280724, 59.917489517777284], [72.87071809369273, 47.58552363917326, 25.297863133765659, 42.034501839579164], 
[94.309551997346077, 68.920541976900637, 1.692465072456105, 75.023262610656431]]], [[[63.471572962233353, 64.798741064964048, 
43.72068323448616, 8.6997778638889542], [94.546430747727896, 88.016904759109764, 14.58256144742673, 74.45351675626388], 
[52.003630381495853, 38.537771628548576, 69.220538944397731, 25.695279099495448]], [[83.306032767743361, 47.257525060077683, 
79.090659443546016, 86.984706241324432], [32.514084780833812, 41.274617166342345, 81.606897997605458, 40.058066677423589], 
[16.582991113576046, 26.477777900260783, 70.754349706548751, 36.162958070109205]]], [[[70.623876696227569, 88.724541508204311, 
35.433617134818753, 24.504940423114135], [79.892676517300728, 54.161986455126332, 21.400020573937091, 45.043365143126863], 
[65.200095846905569, 45.852516046794086, 13.178288768375849, 66.486188034920545]], [[36.310614216780785, 59.353182927411325, 
93.107604989020089, 34.366396717715467], [45.413466526703367, 0.13231504642780872, 85.586369506934446, 56.883390161446918], 
[34.343943871748365, 91.343033836337057, 8.5221756532344575, 80.981820970431841]]]])
      res=log10(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[1.7115674973008292, 0.14701044936490915, 1.6862626492899628, 1.9525453006146518], 
[1.9371457882142409, 1.8279710408692522, 1.5598285655445743, 1.4475835420458216], [1.0266752070276506, 1.7195172730793549, 
1.227695576143472, 1.9618399110892391]], [[1.8388922830736614, 1.7044330853392986, 1.318466280174116, 1.4977434536169738], 
[1.5321778091382223, 1.8293020379670764, 1.8730695371715311, 1.4133219081093693], [1.8424028727067936, 1.8405188199206457, 
1.6968668689863722, 1.4590870477476021]]], [[[1.8586799289868094, 1.8921283065333185, 1.4922120860741765, 1.1348626445857921], 
[1.2850321311761868, 1.700464807165174, 1.8032463102305676, 1.6210646185787188], [1.890542704813432, 1.2471530326945834, 
1.6676631211809771, 1.9979560214717456]], [[1.8879479000068351, 1.1252637599178532, 1.8501622849109463, 1.7202877867394817], 
[1.8969603747078481, 1.5102186451268347, 1.6674817652036087, 1.6627386167845508], [1.1092969122559071, 0.27837048800046127, 
1.4057041574557758, 1.9617597756499436]]], [[[0.10692334674954952, 1.7867025866905135, 1.8754095044181454, 1.743058520763453], 
[1.8558958048831982, 1.9270461365869012, 1.867553069886098, 1.0737399586698149], [1.8787466464617588, 1.8081825203732931, 
1.5200536926633199, 1.5744281662911923]], [[1.6567147000879612, 1.9715035530567897, 1.5118938553061785, 0.53616428590079812], 
[1.2817982486650861, 1.7676439971874796, 1.9914660001128803, 1.3764566986081392], [1.8901295606012043, 1.7403528216800304, 
1.2871423308426, 1.8045865712170828]]]])+(1.-msk_ref)*numpy.array([[[[1.8032195202182379, 1.7852991001099465, 
1.0030847243432686, 1.9999786404702637], [1.6158554462910679, 0.51307449691082996, 1.8091992126689456, 1.5643810623111436], 
[1.7289785637983148, 1.8211672634416693, 1.0610645931591225, 1.5105338214017157]], [[1.4125527038651586, 1.635906833067311, 
1.1144153532983052, 1.7775536085729702], [1.8625530492364937, 1.6774748527239138, 1.4030838386300655, 1.6236059048612139], 
[1.974555681811033, 1.838348683991484, 0.22851971487795045, 1.8751959468180956]]], [[[1.8025792611996256, 1.8115665683111739, 
1.6406869401634994, 0.93950816368541279], [1.9756451383018918, 1.9445660919061505, 1.1638338151190002, 1.871885216148214], 
[1.7160336628581898, 1.5858865988143969, 1.8402349763459562, 1.4098533393090875]], [[1.9206764528218314, 1.6744709733078493, 
1.8981251966246733, 1.9394429011373768], [1.5120715338251509, 1.6156830537503111, 1.9117268699754428, 1.6026899851076344], 
[1.2196628680091728, 1.422881534916125, 1.8497531437791539, 1.5582639478260578]]], [[[1.8489515532379794, 1.948043763770869, 
1.5494154882493205, 1.3892536509847744], [1.9025069708797608, 1.7336945842346827, 1.3304141908792402, 1.653630828745982], 
[1.8142482341636932, 1.6613631715274912, 1.1198590197647234, 1.8227314335250477]], [[1.5600335953381868, 1.7734440137833716, 
1.9689851554161102, 1.5361339991826606], [1.6571846539810831, -0.87839076647757686, 1.9324046044051812, 1.7549854717676672], 
[1.5358501657051966, 1.9606754320191659, 0.93055048133642204, 1.9083875381587312]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_wherePositive_float_rank0(self):
      arg=-70.4202098913
      res=wherePositive(arg)
      ref=0.0
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_wherePositive_array_rank0(self):
      arg=numpy.array(-15.0739210922)
      res=wherePositive(arg)
      ref=numpy.array(0.0)
      if isinstance(res,numpy.ndarray):
         self.assertEqual(res.shape,(),"wrong shape of result.")
      else:
         self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_wherePositive_array_rank1(self):
      arg=numpy.array([28.906077447952242, -23.10248488222922])
      res=wherePositive(arg)
      ref=numpy.array([1.0, 0.0])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_wherePositive_array_rank2(self):
      arg=numpy.array([[67.494612236577638, 80.909854174047979, 47.70625909870725, -71.777921046997903, -98.760598538077588], 
[-91.46372992135052, -96.471924292992298, 2.2231997478348262, -86.769574545358523, 27.58841468272739], [41.637665565063259, 
43.273086569741736, -91.492851720531291, -1.0475761142533457, 2.0960045464434245], [66.692822223370541, -81.550480374891549, 
-16.03214690197197, 40.933392894460582, -55.932865200691054]])
      res=wherePositive(arg)
      ref=numpy.array([[1.0, 1.0, 1.0, 0.0, 0.0], [0.0, 0.0, 1.0, 0.0, 1.0], [1.0, 1.0, 0.0, 0.0, 1.0], [1.0, 0.0, 0.0, 1.0, 
0.0]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_wherePositive_array_rank3(self):
      arg=numpy.array([[[-78.74335358357429, 61.656365045394921], [-42.157193730553708, -38.883346265013287]], 
[[-88.068489379591128, 78.654807861894938], [65.794256223486343, 0.30259701842065567]], [[-34.476517576931769, 
-70.04973714943732], [-40.160695156882142, -18.154859425407395]], [[84.416246128010471, -50.99152387283268], 
[62.940159515920811, -0.11087897257154111]], [[24.387079920060856, 48.804666891859824], [-84.793512049461796, 
-45.707677832173552]], [[-56.719649784696678, 41.392713601723557], [-61.888885999508261, -26.917437791111837]]])
      res=wherePositive(arg)
      ref=numpy.array([[[0.0, 1.0], [0.0, 0.0]], [[0.0, 1.0], [1.0, 1.0]], [[0.0, 0.0], [0.0, 0.0]], [[1.0, 0.0], [1.0, 0.0]], 
[[1.0, 1.0], [0.0, 0.0]], [[0.0, 1.0], [0.0, 0.0]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_wherePositive_array_rank4(self):
      arg=numpy.array([[[[72.100249984910391, -57.956175146094189, -98.913296387113945, -94.28075564808627], 
[-20.338451587905567, 2.4621360465247619, -80.82947875967011, 14.027370399640887], [-48.035910797270517, 64.795111774229667, 
-82.329489801042541, 86.211936552870952]], [[37.185934763312503, 49.50849631983408, 52.839672777504035, 56.377399244546268], 
[-7.234268113362134, 63.462611511320603, -52.156035067909997, -43.754752300758845], [-14.612026578032641, 53.353096426791865, 
-30.372778094043994, 97.860579226476858]]], [[[-82.319953411897728, -2.3001105395993733, -12.470606338277989, 
-63.185118541205121], [-94.960301922301298, -5.6565221271575297, 84.191028153250841, -70.636345764666828], 
[-49.130664317092567, -63.768657233437096, -36.120318903149595, 9.5139407723962393]], [[-92.191650374208336, 
20.072457509379518, 81.652491381337285, -23.646733247590561], [81.342975568236085, 8.3577532005437547, 90.97695735629847, 
-73.840918529210441], [55.181621714112651, 38.330720467911931, -44.40476584033042, -83.817032958509117]]], 
[[[51.616654622513693, -14.450479331389161, -5.8020122717656477, -12.213830658544154], [-51.343817002573424, 
18.583775895090156, 30.9554845465818, -87.095470001919878], [18.943349747730863, -70.349447902193702, -88.554802232088136, 
-28.827441779916782]], [[93.287427705419901, -70.256526015900903, -42.22715235102428, -19.855219969615817], 
[83.720263265665665, 37.151450334636081, -9.298456402444728, -68.274978717591296], [-18.390940148111184, -34.890690302498555, 
-23.885346500749165, -41.804510053113894]]]])
      res=wherePositive(arg)
      ref=numpy.array([[[[1.0, 0.0, 0.0, 0.0], [0.0, 1.0, 0.0, 1.0], [0.0, 1.0, 0.0, 1.0]], [[1.0, 1.0, 1.0, 1.0], [0.0, 1.0, 
0.0, 0.0], [0.0, 1.0, 0.0, 1.0]]], [[[0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 1.0, 0.0], [0.0, 0.0, 0.0, 1.0]], [[0.0, 1.0, 1.0, 0.0], 
[1.0, 1.0, 1.0, 0.0], [1.0, 1.0, 0.0, 0.0]]], [[[1.0, 0.0, 0.0, 0.0], [0.0, 1.0, 1.0, 0.0], [1.0, 0.0, 0.0, 0.0]], [[1.0, 0.0, 
0.0, 0.0], [1.0, 1.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_wherePositive_constData_rank0(self):
      arg=Data(-37.6753904615,self.functionspace)
      res=wherePositive(arg)
      ref=Data(0.0,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_wherePositive_constData_rank1(self):
      arg=Data(numpy.array([5.9258449909595186, -10.188430268215654]),self.functionspace)
      res=wherePositive(arg)
      ref=Data(numpy.array([1.0, 0.0]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_wherePositive_constData_rank2(self):
      arg=Data(numpy.array([[-83.114314035576911, -97.085921526582439, -2.9217465777721827, -1.1787890915813506, 
-20.648618676160794], [71.555185206951847, -36.727722158005328, -65.518358513104715, -10.283766355353151, -88.62623357705364], 
[-87.443527444393794, 60.959525199376515, -36.134372946418772, 29.770497973959067, -0.56000819620498987], [-49.731344859570029, 
75.300101947703638, -19.333618937761287, -76.395441717604839, -82.674517477413232]]),self.functionspace)
      res=wherePositive(arg)
      ref=Data(numpy.array([[0.0, 0.0, 0.0, 0.0, 0.0], [1.0, 0.0, 0.0, 0.0, 0.0], [0.0, 1.0, 0.0, 1.0, 0.0], [0.0, 1.0, 0.0, 
0.0, 0.0]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_wherePositive_constData_rank3(self):
      arg=Data(numpy.array([[[-40.29961143870586, -64.342711281295834], [-64.836836776535421, -41.307920796211107]], 
[[-61.487633921118331, -7.3019368673698182], [-81.595580375270529, -80.989974279719092]], [[13.759821229996419, 
-43.651255501857598], [50.571853247937838, 14.873409423642684]], [[32.599560708951344, 55.599891743152568], 
[-1.8584615650637915, 23.421438487813944]], [[47.743729631487469, -36.78569238955729], [-28.252743553505184, 
22.915574068362659]], [[-0.67504474651151725, 28.5961011553226], [77.940744385806681, -89.35268267418266]]]),self.functionspace)
      res=wherePositive(arg)
      ref=Data(numpy.array([[[0.0, 0.0], [0.0, 0.0]], [[0.0, 0.0], [0.0, 0.0]], [[1.0, 0.0], [1.0, 1.0]], [[1.0, 1.0], [0.0, 
1.0]], [[1.0, 0.0], [0.0, 1.0]], [[0.0, 1.0], [1.0, 0.0]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_wherePositive_constData_rank4(self):
      arg=Data(numpy.array([[[[71.612692363554316, -80.902829681816598, -6.0794925791431496, 41.078313564593316], 
[84.057541602542102, -97.902796065627044, 76.245927278333113, -96.120505502491469], [-99.105019921573785, -34.847444130381049, 
88.667555631999448, -61.769631633660339]], [[40.254500143735328, 70.315581126398541, 10.456067243123286, -17.984811825653452], 
[-27.229768774609923, 75.270523605621804, 51.732035936211417, 14.053481511556541], [-29.429004067130975, 13.931520718869521, 
-30.222269455495066, 53.099744636438828]]], [[[66.658594737374756, 26.443778675942383, 9.5217968199842886, 68.676346716535676], 
[-4.3009384691851125, -50.115873372077658, 40.775834478113808, 82.265077077049256], [-7.2654412117834681, 81.465859170947255, 
-40.570736265673958, 56.150931410619563]], [[15.146701299125894, -64.769075114245851, 77.291597540960367, -14.009811712530947], 
[32.952062015880443, 80.976651596673207, -12.913148087348574, 91.160708451911319], [11.314142446960517, 82.128551417863406, 
9.3083839762397673, 84.293128609776659]]], [[[-18.320762412080938, -27.151169441911918, 68.650437109092991, 
99.874147618429021], [68.713441258274884, 57.829825185320146, 97.052922881235219, -75.918331274766771], [35.001416215619628, 
-31.489897641770497, -92.684784085487038, -60.095802530301491]], [[92.145606292141849, -48.839544388877989, 
-81.576565094059148, -25.403820705830697], [-35.642188462105295, 93.371365566245657, 86.533657118913908, -64.737341204155797], 
[80.204843593716333, -55.196561402571298, -91.04571841996929, -36.30116120361]]]]),self.functionspace)
      res=wherePositive(arg)
      ref=Data(numpy.array([[[[1.0, 0.0, 0.0, 1.0], [1.0, 0.0, 1.0, 0.0], [0.0, 0.0, 1.0, 0.0]], [[1.0, 1.0, 1.0, 0.0], [0.0, 
1.0, 1.0, 1.0], [0.0, 1.0, 0.0, 1.0]]], [[[1.0, 1.0, 1.0, 1.0], [0.0, 0.0, 1.0, 1.0], [0.0, 1.0, 0.0, 1.0]], [[1.0, 0.0, 1.0, 
0.0], [1.0, 1.0, 0.0, 1.0], [1.0, 1.0, 1.0, 1.0]]], [[[0.0, 0.0, 1.0, 1.0], [1.0, 1.0, 1.0, 0.0], [1.0, 0.0, 0.0, 0.0]], [[1.0, 
0.0, 0.0, 0.0], [0.0, 1.0, 1.0, 0.0], [1.0, 0.0, 0.0, 0.0]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_wherePositive_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(60.0229181114)+(1.-msk_arg)*(-96.3764768528)
      res=wherePositive(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(1.0)+(1.-msk_ref)*(0.0)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_wherePositive_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([-22.419211375644196, 61.516723654676611])+(1.-msk_arg)*numpy.array([-47.697327344809139, 
-28.757233801998126])
      res=wherePositive(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.0, 1.0])+(1.-msk_ref)*numpy.array([0.0, 0.0])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_wherePositive_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[75.742071937578856, 0.92373368970095271, -52.139002610910602, -90.166031609404286, 
5.5135768493946244], [-85.866582176624462, -5.8971094180698174, -44.235765048437599, 66.234147934377376, 17.435204749442562], 
[33.989217424407883, -90.944874166814358, 95.204126153994281, -48.494625981425976, -47.866621860789692], [-51.66387411792315, 
23.408933865995692, -38.274649597022268, 94.35801540027137, 
-22.729990325215482]])+(1.-msk_arg)*numpy.array([[-77.74312490692175, 73.695397504354787, -25.165673520548154, 
28.581554535703447, 16.809417381385657], [14.13939656000484, -30.054842741776127, 80.5114214445926, -91.608466400819012, 
74.642682764121503], [24.373158682098676, 88.149830810457473, -87.043665488082226, -41.342480458547229, 47.83512981806436], 
[-23.988019538589128, -97.754385926732397, 64.136066737433026, 93.308550605904486, 50.264032149431102]])
      res=wherePositive(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[1.0, 1.0, 0.0, 0.0, 1.0], [0.0, 0.0, 0.0, 1.0, 1.0], [1.0, 0.0, 1.0, 0.0, 0.0], [0.0, 1.0, 0.0, 
1.0, 0.0]])+(1.-msk_ref)*numpy.array([[0.0, 1.0, 0.0, 1.0, 1.0], [1.0, 0.0, 1.0, 0.0, 1.0], [1.0, 1.0, 0.0, 0.0, 1.0], [0.0, 
0.0, 1.0, 1.0, 1.0]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_wherePositive_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[-77.3470085491482, 23.703501363968414], [-78.457555629030821, -30.460470615708402]], 
[[40.586688169139535, 85.979119968548844], [14.391931326799096, 98.012599760182638]], [[-25.877266251828445, 
92.469507013732766], [28.446352130979278, -29.047890229715236]], [[85.045265454383923, -79.743098568103363], 
[-67.998286974132469, 51.27842688771014]], [[-26.856044632583959, -46.616707014967403], [-13.323366086721251, 
5.0944903333334395]], [[86.74261544510577, -64.927377055330226], [-96.292217645844659, 
73.280353362666261]]])+(1.-msk_arg)*numpy.array([[[-73.854272948424978, -62.224959421353731], [69.003679176662558, 
55.120389456685018]], [[-66.302549803605586, 41.107120553488897], [35.942390589413179, -49.629959161552975]], 
[[87.070085680653023, -7.4957980304887002], [62.24913427711212, 70.671000810016523]], [[-62.720150429412101, 
-67.208518801718355], [1.0758801572739856, -7.2411545422204995]], [[-46.322912286667474, -95.449711884536555], 
[-41.530447460581279, 23.396402682152996]], [[61.760793037463145, 26.046559417690631], [20.210361159406972, 
-6.0045794744297183]]])
      res=wherePositive(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.0, 1.0], [0.0, 0.0]], [[1.0, 1.0], [1.0, 1.0]], [[0.0, 1.0], [1.0, 0.0]], [[1.0, 0.0], [0.0, 
1.0]], [[0.0, 0.0], [0.0, 1.0]], [[1.0, 0.0], [0.0, 1.0]]])+(1.-msk_ref)*numpy.array([[[0.0, 0.0], [1.0, 1.0]], [[0.0, 1.0], 
[1.0, 0.0]], [[1.0, 0.0], [1.0, 1.0]], [[0.0, 0.0], [1.0, 0.0]], [[0.0, 0.0], [0.0, 1.0]], [[1.0, 1.0], [1.0, 0.0]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_wherePositive_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[43.80370896979781, -1.8783559803067504, 68.652439234878358, -54.13722802579484], 
[-16.330009797654171, -4.9840146257985225, -82.461337583892941, 20.478464419172909], [2.5283355218952437, -38.317141099135156, 
74.439904322815721, -32.370032619039009]], [[75.661807530547463, 14.806511977463515, 92.913122066573948, -57.392835755828209], 
[-24.746344559568371, 58.577607900172268, 19.638932598152707, -56.683628155430846], [-23.63916211487431, -31.732325498364517, 
2.9992635782867012, -51.737397326020073]]], [[[14.055418901807641, 49.78668125052377, 68.35230222025416, 1.1913826311549229], 
[-92.159227348973076, 77.751907937221432, 44.535941263032896, -69.410161289906654], [27.701588449716269, 33.203546994102169, 
-52.284409227888084, 6.1028377961118991]], [[-5.3335556081772069, -9.4816270741597748, 21.074022334438197, 
-82.638909781192709], [-38.201386351163571, -60.608459482874963, -10.140896441708108, -51.937205523614985], 
[67.836075729072832, 27.271013305993591, 37.943674493436077, 47.122338195390938]]], [[[94.80320858794326, 78.422597672517611, 
-47.362034814216145, -67.005672757399992], [13.984708106797612, -80.851401714264526, -66.341143846358705, 23.096516376649774], 
[19.354469887332982, -32.204625876582298, -90.448693280849412, -83.438145584116057]], [[-20.430907359037946, 
34.265521783285777, 36.932356914307803, 94.749144039157954], [-23.740364230369423, -24.861180581515924, -23.653628334874384, 
-76.457114676244316], [-49.29499131919097, 97.199643030562271, 86.725540680528979, 
-20.724580882040343]]]])+(1.-msk_arg)*numpy.array([[[[-1.6865270561507373, -48.38266559451008, -21.021725580733076, 
-13.790499838190257], [50.521044913720118, -11.838528039107359, 48.1628346881065, -89.550279456158279], [74.235807932697497, 
-13.100435601018347, 2.3652821326138422, 30.026883928678501]], [[-69.738936297086411, 88.100714549880394, -31.068488721700788, 
-75.688640201758659], [74.214023515172073, 24.090808172915288, 60.52301851018504, -87.853385795062763], [-87.508927874364957, 
-1.5837298917809335, -28.713029983952225, -57.250723533626036]]], [[[-65.075934563641624, 11.614682951153725, 
45.215631592128346, -74.364837387252123], [50.728658050670447, -58.927486265154741, 52.743491813692458, -35.976343344650601], 
[79.119298474238235, 97.80256840023921, -59.11443388001851, 54.649824887730659]], [[-99.518412000665691, -42.977729536858433, 
20.676967689422, -90.750676595060924], [38.4658987940185, -16.645573644802568, -78.554599571785232, -59.494115705700779], 
[9.3463745210543578, 1.8679075108029082, -47.098256469256448, 83.374869892965677]]], [[[-69.480204147992879, 
22.097990324074402, 64.676628015428008, -57.273925044302757], [-22.854234700698981, -91.099345045115427, 13.492331841283175, 
64.248512967461465], [35.277064985517995, -94.753538497730474, 4.898846453131128, 75.023190487572577]], [[-90.672626500706244, 
-96.193464484471392, -13.003806772626731, -36.673991242489045], [-91.46754262338321, -48.493735714203744, -52.961005505255308, 
-75.058035570591386], [-39.614658185133521, 43.427235596994592, -79.256978699402055, -23.592401540492915]]]])
      res=wherePositive(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[1.0, 0.0, 1.0, 0.0], [0.0, 0.0, 0.0, 1.0], [1.0, 0.0, 1.0, 0.0]], [[1.0, 1.0, 1.0, 0.0], 
[0.0, 1.0, 1.0, 0.0], [0.0, 0.0, 1.0, 0.0]]], [[[1.0, 1.0, 1.0, 1.0], [0.0, 1.0, 1.0, 0.0], [1.0, 1.0, 0.0, 1.0]], [[0.0, 0.0, 
1.0, 0.0], [0.0, 0.0, 0.0, 0.0], [1.0, 1.0, 1.0, 1.0]]], [[[1.0, 1.0, 0.0, 0.0], [1.0, 0.0, 0.0, 1.0], [1.0, 0.0, 0.0, 0.0]], 
[[0.0, 1.0, 1.0, 1.0], [0.0, 0.0, 0.0, 0.0], [0.0, 1.0, 1.0, 0.0]]]])+(1.-msk_ref)*numpy.array([[[[0.0, 0.0, 0.0, 0.0], [1.0, 
0.0, 1.0, 0.0], [1.0, 0.0, 1.0, 1.0]], [[0.0, 1.0, 0.0, 0.0], [1.0, 1.0, 1.0, 0.0], [0.0, 0.0, 0.0, 0.0]]], [[[0.0, 1.0, 1.0, 
0.0], [1.0, 0.0, 1.0, 0.0], [1.0, 1.0, 0.0, 1.0]], [[0.0, 0.0, 1.0, 0.0], [1.0, 0.0, 0.0, 0.0], [1.0, 1.0, 0.0, 1.0]]], [[[0.0, 
1.0, 1.0, 0.0], [0.0, 0.0, 1.0, 1.0], [1.0, 0.0, 1.0, 1.0]], [[0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0], [0.0, 1.0, 0.0, 
0.0]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNegative_float_rank0(self):
      arg=60.8682840238
      res=whereNegative(arg)
      ref=0.0
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNegative_array_rank0(self):
      arg=numpy.array(12.3306392667)
      res=whereNegative(arg)
      ref=numpy.array(0.0)
      if not isinstance(res,float):
         self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
         self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNegative_array_rank1(self):
      arg=numpy.array([-96.137063268028797, -76.658277780494259])
      res=whereNegative(arg)
      ref=numpy.array([1.0, 1.0])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNegative_array_rank2(self):
      arg=numpy.array([[34.527255357681298, 46.742030193793738, -10.068808745305603, 33.451041357426902, 52.37433370591711], 
[-1.1274024836940555, 2.5419221930968519, 32.984481014462432, 76.341642878243078, 36.384014089381708], [-98.496974526985866, 
-65.01368641484936, 50.789914135856463, -16.286375703249959, 45.813382104794698], [18.983852245009942, -39.656344508041677, 
-87.540891441344542, 22.257065860125209, -56.083642049697957]])
      res=whereNegative(arg)
      ref=numpy.array([[0.0, 0.0, 1.0, 0.0, 0.0], [1.0, 0.0, 0.0, 0.0, 0.0], [1.0, 1.0, 0.0, 1.0, 0.0], [0.0, 1.0, 1.0, 0.0, 
1.0]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNegative_array_rank3(self):
      arg=numpy.array([[[27.452945477282029, 10.079247782917776], [73.463914059433876, 89.31763644890097]], 
[[-39.00630564129397, -56.727418650360839], [99.956113750577458, -63.201931425083899]], [[23.800879753842111, 
-71.802714984390661], [20.122988022110079, 50.095040283423629]], [[17.391754513203111, -87.274468281986955], 
[-69.061025469838569, -95.107053552460357]], [[-38.397282711106314, -73.428763082157076], [-24.751594707412835, 
93.224119395489396]], [[-71.420548157545795, -10.204520855028434], [8.2682413993733377, 69.215290928341716]]])
      res=whereNegative(arg)
      ref=numpy.array([[[0.0, 0.0], [0.0, 0.0]], [[1.0, 1.0], [0.0, 1.0]], [[0.0, 1.0], [0.0, 0.0]], [[0.0, 1.0], [1.0, 1.0]], 
[[1.0, 1.0], [1.0, 0.0]], [[1.0, 1.0], [0.0, 0.0]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNegative_array_rank4(self):
      arg=numpy.array([[[[52.851635463931416, -42.757919744122042, -43.221594386333393, -71.627489318948307], 
[63.051185031362394, 88.122130054764369, 3.0294368603839104, -80.279495847366533], [22.202897744980945, -9.5679057475316398, 
0.6747259697796153, -32.77109355448286]], [[7.4103317420963606, 56.685726337130205, -5.6243186818793873, 30.690249505075911], 
[-40.26391967567595, 56.69805771317823, 38.011050108017827, -67.152404819687916], [11.988018990285681, 15.418047294689202, 
-35.853437208267195, 9.7603269619072535]]], [[[62.818258893227807, -68.93898305799496, -59.052468240174804, 
-26.916691703402719], [-93.556184474191454, 54.562368122195096, -69.599681490131715, -79.832141645610747], 
[-68.174504363157169, 15.388412833294524, 92.958241545241805, 48.717089383312725]], [[80.362433523267072, 85.459156338229405, 
48.410142376841151, 92.900690384172805], [-13.477883213840627, -48.877024108923784, -42.918741473403841, 30.897642075312746], 
[6.738002126296621, 12.415597112960256, 39.201682409893436, -18.884889673699519]]], [[[-62.408880043825611, 
-72.590378070784652, -82.098861995291813, -34.475953721594863], [85.281229062252805, -53.982038453937164, 31.566642378708877, 
10.785407426910879], [67.392800607262615, -44.131930493424655, -79.616463869751612, 69.356492589536515]], [[20.091109964478719, 
-76.324154832707023, 76.477460746877824, -45.456803948000314], [77.912305288034929, -24.195046812358953, -28.192001907231074, 
87.18708113620994], [31.883641084515517, 58.423371148498177, 68.91426566250999, -4.6562642110223038]]]])
      res=whereNegative(arg)
      ref=numpy.array([[[[0.0, 1.0, 1.0, 1.0], [0.0, 0.0, 0.0, 1.0], [0.0, 1.0, 0.0, 1.0]], [[0.0, 0.0, 1.0, 0.0], [1.0, 0.0, 
0.0, 1.0], [0.0, 0.0, 1.0, 0.0]]], [[[0.0, 1.0, 1.0, 1.0], [1.0, 0.0, 1.0, 1.0], [1.0, 0.0, 0.0, 0.0]], [[0.0, 0.0, 0.0, 0.0], 
[1.0, 1.0, 1.0, 0.0], [0.0, 0.0, 0.0, 1.0]]], [[[1.0, 1.0, 1.0, 1.0], [0.0, 1.0, 0.0, 0.0], [0.0, 1.0, 1.0, 0.0]], [[0.0, 1.0, 
0.0, 1.0], [0.0, 1.0, 1.0, 0.0], [0.0, 0.0, 0.0, 1.0]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNegative_constData_rank0(self):
      arg=Data(71.0461471154,self.functionspace)
      res=whereNegative(arg)
      ref=Data(0.0,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNegative_constData_rank1(self):
      arg=Data(numpy.array([-80.731119617568183, 91.086434642832756]),self.functionspace)
      res=whereNegative(arg)
      ref=Data(numpy.array([1.0, 0.0]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNegative_constData_rank2(self):
      arg=Data(numpy.array([[-30.078161181135357, -83.774816691205274, 31.234430645356866, -88.643364099015969, 
-27.327692899224587], [39.87080777988453, -90.362613632345415, -88.178393380316351, 0.78766148538969105, 17.467833815617212], 
[-41.019873514226759, 85.969686883649359, -71.810931481288037, -64.605161148773874, 73.291192617723908], [59.382800564915414, 
72.112470346114435, 64.321292886976977, 83.840884620483592, -18.887748452642555]]),self.functionspace)
      res=whereNegative(arg)
      ref=Data(numpy.array([[1.0, 1.0, 0.0, 1.0, 1.0], [0.0, 1.0, 1.0, 0.0, 0.0], [1.0, 0.0, 1.0, 1.0, 0.0], [0.0, 0.0, 0.0, 
0.0, 1.0]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNegative_constData_rank3(self):
      arg=Data(numpy.array([[[52.467962714577851, -10.755811269205751], [-49.824884301105229, -87.898834770673545]], 
[[-84.887027502945656, 39.276449348108372], [-0.28784075758221661, -43.719102909037424]], [[5.0483861425380212, 
77.163995546260338], [-82.618948195844013, 1.7083645155553171]], [[-85.252812042880379, 47.920337911790767], 
[-89.706782276303997, -9.5992153716397723]], [[-53.487541003352291, -37.042887199510169], [-39.924749090729648, 
-97.861135144144257]], [[14.313959182168418, -64.680421169049538], [-18.772463792577526, 
-22.16535015919969]]]),self.functionspace)
      res=whereNegative(arg)
      ref=Data(numpy.array([[[0.0, 1.0], [1.0, 1.0]], [[1.0, 0.0], [1.0, 1.0]], [[0.0, 0.0], [1.0, 0.0]], [[1.0, 0.0], [1.0, 
1.0]], [[1.0, 1.0], [1.0, 1.0]], [[0.0, 1.0], [1.0, 1.0]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNegative_constData_rank4(self):
      arg=Data(numpy.array([[[[-56.738900482413811, -11.202980729285244, -71.631743919679849, -14.501263562534177], 
[43.190705972099948, 28.379497298551968, -13.569119185364585, 65.358348068303656], [-91.736472018303189, 45.571509175250782, 
20.814396687393597, 53.332854517832743]], [[32.567368220643601, 3.4928426287876135, 58.278182327340858, -96.270406402097166], 
[30.357738302076029, -34.075847316972101, 88.565145570585258, 68.632732598053138], [-43.333644849377052, 98.698365574236476, 
99.086221332142486, 79.256177564404851]]], [[[83.658967348768414, 73.87622641143912, 26.664285223308923, -24.236702163040988], 
[33.966505242244011, -47.630267479564296, -41.486678856495061, -68.553303600684572], [-75.108951411161826, 42.384054423970923, 
-48.286880035511935, -66.006863305952038]], [[-1.946984825710345, 12.753883934941655, 99.463784625818676, -86.199776629612472], 
[43.644149588437671, -18.088721159515316, 82.264706299718881, -66.686685578089453], [-1.6489398255121301, 17.662742039769299, 
-39.062788718667996, 43.366643090778211]]], [[[7.0838003526987166, 84.164965010157346, 94.39131652727707, 42.082953004169639], 
[-55.773938700136782, 52.435811047719028, 53.070615567554626, -64.791867065020568], [-11.109776598239193, 76.314724953648579, 
-47.359305606426716, 52.614937206019448]], [[18.800439656297712, 93.558627961377653, 13.541712656625251, 4.6114826390651018], 
[63.601298072608671, 48.408056729632989, 49.122562180543326, -0.17795085269705169], [35.457986798476099, -74.518767618135072, 
11.99706514332901, 40.603347309608296]]]]),self.functionspace)
      res=whereNegative(arg)
      ref=Data(numpy.array([[[[1.0, 1.0, 1.0, 1.0], [0.0, 0.0, 1.0, 0.0], [1.0, 0.0, 0.0, 0.0]], [[0.0, 0.0, 0.0, 1.0], [0.0, 
1.0, 0.0, 0.0], [1.0, 0.0, 0.0, 0.0]]], [[[0.0, 0.0, 0.0, 1.0], [0.0, 1.0, 1.0, 1.0], [1.0, 0.0, 1.0, 1.0]], [[1.0, 0.0, 0.0, 
1.0], [0.0, 1.0, 0.0, 1.0], [1.0, 0.0, 1.0, 0.0]]], [[[0.0, 0.0, 0.0, 0.0], [1.0, 0.0, 0.0, 1.0], [1.0, 0.0, 1.0, 0.0]], [[0.0, 
0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 1.0], [0.0, 1.0, 0.0, 0.0]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNegative_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(-30.5711237435)+(1.-msk_arg)*(69.0316558372)
      res=whereNegative(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(1.0)+(1.-msk_ref)*(0.0)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNegative_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([-71.631541638025368, -34.933285862461716])+(1.-msk_arg)*numpy.array([94.532432395056986, 
-8.4861064431438251])
      res=whereNegative(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([1.0, 1.0])+(1.-msk_ref)*numpy.array([0.0, 1.0])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNegative_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[-81.246161725962281, -0.80928674630524711, 64.245598914154385, 25.019488668613903, 
-20.02114893086133], [-16.925215481515338, -19.119399037651519, -33.929581396089105, 49.036506368260234, -19.653199853598636], 
[-71.356363589257569, 60.190772929399429, -80.879381891956257, -25.014458239978964, 53.593957929715629], [-99.037250890485964, 
2.1499053353822575, 24.458899543815619, 17.352712366156538, 
87.281727012725412]])+(1.-msk_arg)*numpy.array([[24.876343327252769, -97.810881158075432, -98.976581314248165, 
-58.828345137646295, -47.996308815032116], [88.870044806649133, 83.497827171390071, -49.848159713628817, -79.747992284629902, 
-86.622423551338159], [-44.407345444816215, -19.420350667571313, 30.593026030666437, -92.522432388627806, -27.411015715113336], 
[-70.620933054656803, 95.630924114745426, -6.4891593001671595, 30.067888220202349, 54.633003267596763]])
      res=whereNegative(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[1.0, 1.0, 0.0, 0.0, 1.0], [1.0, 1.0, 1.0, 0.0, 1.0], [1.0, 0.0, 1.0, 1.0, 0.0], [1.0, 0.0, 0.0, 
0.0, 0.0]])+(1.-msk_ref)*numpy.array([[0.0, 1.0, 1.0, 1.0, 1.0], [0.0, 0.0, 1.0, 1.0, 1.0], [1.0, 1.0, 0.0, 1.0, 1.0], [1.0, 
0.0, 1.0, 0.0, 0.0]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNegative_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[-13.226116878406287, -82.199364713243199], [10.802887584692229, -65.686743331979756]], 
[[-53.361630863787582, 99.378467998876744], [-3.7915436947351395, -26.517844578338142]], [[93.096723454132416, 
-93.695818457215509], [31.554012577630004, 39.96605767563608]], [[-42.995245201378005, -29.890999774133903], 
[18.30723099164662, 88.621280116942501]], [[1.7197935367676394, 25.587678904490673], [48.238578489671426, 
-98.542525990226594]], [[-37.454079112118485, 7.2826272286010294], [76.911623094137013, 
88.86281750539041]]])+(1.-msk_arg)*numpy.array([[[-76.308681189268725, 78.036351430824794], [29.442414274483752, 
32.970131475820949]], [[-11.522188039672287, -29.450015896780172], [-71.490304555395142, 86.623804825605646]], 
[[29.9575694239212, 48.244193165489548], [12.890596118737662, -44.72251029422889]], [[87.880930152139086, 77.877282206925827], 
[52.458002105709056, -26.128933933435519]], [[-10.037415450512867, 62.849269823732953], [-13.341389821511029, 
-66.474146337225662]], [[-66.827808490764482, 55.123032168800961], [11.770865332062257, -65.792755427561076]]])
      res=whereNegative(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[1.0, 1.0], [0.0, 1.0]], [[1.0, 0.0], [1.0, 1.0]], [[0.0, 1.0], [0.0, 0.0]], [[1.0, 1.0], [0.0, 
0.0]], [[0.0, 0.0], [0.0, 1.0]], [[1.0, 0.0], [0.0, 0.0]]])+(1.-msk_ref)*numpy.array([[[1.0, 0.0], [0.0, 0.0]], [[1.0, 1.0], 
[1.0, 0.0]], [[0.0, 0.0], [0.0, 1.0]], [[0.0, 0.0], [0.0, 1.0]], [[1.0, 0.0], [1.0, 1.0]], [[1.0, 0.0], [0.0, 1.0]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNegative_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[22.736350434258242, 83.810869845847236, 19.347903964541985, -80.166931850749208], 
[45.452393364563619, 50.301718349604471, 67.66905146372207, -16.354259444700304], [47.685594996358162, -15.412627032981447, 
-33.973100957852424, -28.421171446812849]], [[28.765923294757869, 81.948159836174653, -61.002964447142837, 
-60.377845446919643], [-11.088814072045068, 77.109491110526534, -28.820136591829765, 54.89579817587898], [88.727197576669454, 
-61.758232727685126, 31.11303916232518, 22.425386560120742]]], [[[-37.668698306499728, -93.367508938543509, -96.69001608717312, 
46.73287429891991], [-17.086614002398235, 77.117506515236443, -89.609359206118882, -51.116796499884963], [-11.331078933065797, 
82.655593610768733, -51.263800662903833, -45.417082762140161]], [[-38.448402748623288, 40.952337637478053, 8.3792678972500312, 
-28.811191611059755], [-46.223465500265526, -44.801884964741333, 87.075157001648392, 72.791216981596563], 
[-0.90078100649908777, -60.044301966082593, -24.235637315239387, 86.769389985854133]]], [[[-85.232530260963387, 
8.41880287115751, -80.819791741701195, 1.6073205456352753], [43.848963529258867, -84.768973989806852, 45.149836932049368, 
-54.480107023795263], [-90.354256577100813, -28.667837787830337, -74.038580399961162, -44.842435509105826]], 
[[-13.569853286180006, -39.870360955262484, -43.80119994880998, 98.821718640347399], [-74.739576772222932, -10.481845098843181, 
-19.428165239121967, -64.409567816025458], [-84.960799515544096, 44.601968055026731, -53.945696380018205, 
-62.508824901005354]]]])+(1.-msk_arg)*numpy.array([[[[98.331226651596097, -45.908896808474253, -76.328427133254692, 
9.0732880852006019], [77.537791077921838, 81.188001463478429, -77.573617910825661, -70.53070043922385], [-60.093184821750675, 
-51.652608941265157, 17.555344559057744, 71.905673472708713]], [[60.587001914144139, -22.25480650550962, -15.669252605643379, 
81.086498408593513], [20.771698800848753, 88.031012565600975, -57.68613427820555, 33.563864544408631], [71.357793416800945, 
48.697344496300332, -56.302119150659721, 85.465915125881565]]], [[[71.34331579003441, -76.222881134514139, 70.657517114902021, 
-51.209553238176994], [-51.01048305469795, -54.010632435840478, 40.057721463180229, 90.017288091933807], [23.78119243806394, 
-65.128478845325958, -69.24071358054411, -35.156183078888859]], [[24.931984536066182, 29.218423454675872, 37.291762771095193, 
-44.301474105023033], [-27.534021386697475, 22.357601126944559, -32.007180009349128, -78.646515703268264], [70.214506130886946, 
57.390883214823788, -94.288983066193083, 68.156267247037135]]], [[[86.210144621283746, 47.931898737102983, 64.611807566463199, 
97.889272082903091], [-31.445648061764658, 80.771569964243128, -36.261273157065155, 95.317389789191196], [-53.571317526281348, 
-14.421189259125327, 59.637908000860364, -61.238575115717154]], [[-25.60102669901228, 53.006556636924671, -69.520681581420575, 
-2.0132709991870286], [22.968036096731808, 5.7145633292572597, 73.063684440985469, -57.182218254882436], [77.157618409964044, 
-31.84995932925176, -83.282695391829435, 77.690533538106138]]]])
      res=whereNegative(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.0, 0.0, 0.0, 1.0], [0.0, 0.0, 0.0, 1.0], [0.0, 1.0, 1.0, 1.0]], [[0.0, 0.0, 1.0, 1.0], 
[1.0, 0.0, 1.0, 0.0], [0.0, 1.0, 0.0, 0.0]]], [[[1.0, 1.0, 1.0, 0.0], [1.0, 0.0, 1.0, 1.0], [1.0, 0.0, 1.0, 1.0]], [[1.0, 0.0, 
0.0, 1.0], [1.0, 1.0, 0.0, 0.0], [1.0, 1.0, 1.0, 0.0]]], [[[1.0, 0.0, 1.0, 0.0], [0.0, 1.0, 0.0, 1.0], [1.0, 1.0, 1.0, 1.0]], 
[[1.0, 1.0, 1.0, 0.0], [1.0, 1.0, 1.0, 1.0], [1.0, 0.0, 1.0, 1.0]]]])+(1.-msk_ref)*numpy.array([[[[0.0, 1.0, 1.0, 0.0], [0.0, 
0.0, 1.0, 1.0], [1.0, 1.0, 0.0, 0.0]], [[0.0, 1.0, 1.0, 0.0], [0.0, 0.0, 1.0, 0.0], [0.0, 0.0, 1.0, 0.0]]], [[[0.0, 1.0, 0.0, 
1.0], [1.0, 1.0, 0.0, 0.0], [0.0, 1.0, 1.0, 1.0]], [[0.0, 0.0, 0.0, 1.0], [1.0, 0.0, 1.0, 1.0], [0.0, 0.0, 1.0, 0.0]]], [[[0.0, 
0.0, 0.0, 0.0], [1.0, 0.0, 1.0, 0.0], [1.0, 1.0, 0.0, 1.0]], [[1.0, 0.0, 1.0, 1.0], [0.0, 0.0, 0.0, 1.0], [0.0, 1.0, 1.0, 
0.0]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonNegative_float_rank0(self):
      arg=58.4734070432
      res=whereNonNegative(arg)
      ref=1.0
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonNegative_array_rank0(self):
      arg=numpy.array(-77.2396345376)
      res=whereNonNegative(arg)
      ref=numpy.array(0.0)
      if not isinstance(res,float):
         self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
         self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonNegative_array_rank1(self):
      arg=numpy.array([-96.481185201309529, 55.69393817245151])
      res=whereNonNegative(arg)
      ref=numpy.array([0.0, 1.0])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonNegative_array_rank2(self):
      arg=numpy.array([[33.751666255076515, -97.928265537757909, 14.362676388994842, 74.281280811324933, -52.516825657717888], 
[-20.461289856454684, -45.157730149915643, 31.896091176433487, 4.1317379123191529, -55.246764488338762], [19.700429048464358, 
16.679444563048492, -38.30023330559407, -77.041761864300582, -28.043364036650644], [5.0773359198315973, 91.9368391294513, 
20.710543350370259, -28.942481048294326, -16.348545481686472]])
      res=whereNonNegative(arg)
      ref=numpy.array([[1.0, 0.0, 1.0, 1.0, 0.0], [0.0, 0.0, 1.0, 1.0, 0.0], [1.0, 1.0, 0.0, 0.0, 0.0], [1.0, 1.0, 1.0, 0.0, 
0.0]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonNegative_array_rank3(self):
      arg=numpy.array([[[19.729007031849989, 28.339725664851557], [-47.773566720641256, -21.927219410096143]], 
[[46.094165089019498, -0.63223312390847752], [5.1418015828473358, -3.2223067899969919]], [[-93.553953569350369, 
88.717633043793484], [-78.87519362229618, 32.971725631423112]], [[-90.65660301934227, 6.6283631097893192], [90.193523289250635, 
-40.957164498019537]], [[83.152161885821187, -70.542346994808355], [30.571013881082479, -0.80033587662163086]], 
[[53.966121161234639, -32.727914208209413], [5.7429993927969889, -30.72691997703258]]])
      res=whereNonNegative(arg)
      ref=numpy.array([[[1.0, 1.0], [0.0, 0.0]], [[1.0, 0.0], [1.0, 0.0]], [[0.0, 1.0], [0.0, 1.0]], [[0.0, 1.0], [1.0, 0.0]], 
[[1.0, 0.0], [1.0, 0.0]], [[1.0, 0.0], [1.0, 0.0]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonNegative_array_rank4(self):
      arg=numpy.array([[[[67.819303150305586, 58.094451445762473, -56.716860642067182, -31.264742194049617], 
[-65.095373210856522, -76.708282393809441, 32.654968187506199, 85.420496372460292], [-6.1214503461968235, -12.93519491683017, 
-71.316345077104828, 50.676771270018918]], [[-90.117656138807604, -30.507925648793119, 29.786316442392916, 
-13.581636478986141], [-56.535310447507214, 68.422294613888425, -76.869431303022864, -92.136715817023898], 
[-69.545808694217229, -65.561773987757903, 80.884826568492457, -16.59346797909906]]], [[[28.446041688660898, 
-93.575372847332545, 51.967032569599922, 16.338681282927169], [-54.098652915975663, -64.462591681548645, 27.942122982761418, 
66.929439755872039], [44.711579155976636, -20.254353511943918, 5.2271920927414754, -32.665593100317622]], 
[[-28.264723368846106, 51.707175859789402, -70.695873526038469, 10.846862489367922], [-78.389576448699813, -61.161653853188945, 
15.927215052827066, -96.297305121328208], [-89.498120859989911, -83.699171123343334, -49.945976969918114, 
52.515091562529761]]], [[[98.432201063954437, 5.3470028633255424, -87.883945686853139, -82.398714780883893], 
[88.61112112218737, 65.282106288691892, -90.450701516842031, 11.372598817788031], [-1.2170253549431749, -11.147053059704916, 
-51.10796851690624, -21.568184900408241]], [[-58.41155791852222, 63.556679823020005, -33.777540011731944, -19.968219145362582], 
[62.331887732495801, 96.26963346060586, -47.912958914398573, 20.836893897762778], [83.530900729436809, 36.16042965799457, 
-99.099072331349603, -82.111303540526407]]]])
      res=whereNonNegative(arg)
      ref=numpy.array([[[[1.0, 1.0, 0.0, 0.0], [0.0, 0.0, 1.0, 1.0], [0.0, 0.0, 0.0, 1.0]], [[0.0, 0.0, 1.0, 0.0], [0.0, 1.0, 
0.0, 0.0], [0.0, 0.0, 1.0, 0.0]]], [[[1.0, 0.0, 1.0, 1.0], [0.0, 0.0, 1.0, 1.0], [1.0, 0.0, 1.0, 0.0]], [[0.0, 1.0, 0.0, 1.0], 
[0.0, 0.0, 1.0, 0.0], [0.0, 0.0, 0.0, 1.0]]], [[[1.0, 1.0, 0.0, 0.0], [1.0, 1.0, 0.0, 1.0], [0.0, 0.0, 0.0, 0.0]], [[0.0, 1.0, 
0.0, 0.0], [1.0, 1.0, 0.0, 1.0], [1.0, 1.0, 0.0, 0.0]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonNegative_constData_rank0(self):
      arg=Data(-5.63597616175,self.functionspace)
      res=whereNonNegative(arg)
      ref=Data(0.0,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonNegative_constData_rank1(self):
      arg=Data(numpy.array([52.597537475659237, 65.281264541702825]),self.functionspace)
      res=whereNonNegative(arg)
      ref=Data(numpy.array([1.0, 1.0]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonNegative_constData_rank2(self):
      arg=Data(numpy.array([[23.937746997604961, 12.178098197245674, 19.416176574130617, -13.724111061677206, 
73.879619373893036], [-64.162137997220782, 40.478876735342112, -58.917478343894622, -70.727192689590737, -44.735513365105881], 
[-84.885212174941756, -95.547022727178614, 29.595829863194666, 86.446737656525102, -84.567583366252606], [82.021223674901137, 
-37.06970380424206, 29.766667765490439, 36.637380378689386, 94.715161252546096]]),self.functionspace)
      res=whereNonNegative(arg)
      ref=Data(numpy.array([[1.0, 1.0, 1.0, 0.0, 1.0], [0.0, 1.0, 0.0, 0.0, 0.0], [0.0, 0.0, 1.0, 1.0, 0.0], [1.0, 0.0, 1.0, 
1.0, 1.0]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonNegative_constData_rank3(self):
      arg=Data(numpy.array([[[-84.874906350810292, -44.379505192158206], [28.596504409608315, 60.352404739121454]], 
[[17.592457984129254, 67.664065046718292], [46.795367470916034, 36.556332282490359]], [[6.2010626169601011, 
-33.585920455893998], [-47.715736291142342, -42.963154840594278]], [[-45.34095850737463, -26.483708610893302], 
[-39.974262905912994, 19.88771053468794]], [[88.691725161686577, -75.208046572141768], [-85.689747680137458, 
8.2377492978098985]], [[83.787631396620867, -49.123477152480156], [36.050783089982389, 
54.394056720653481]]]),self.functionspace)
      res=whereNonNegative(arg)
      ref=Data(numpy.array([[[0.0, 0.0], [1.0, 1.0]], [[1.0, 1.0], [1.0, 1.0]], [[1.0, 0.0], [0.0, 0.0]], [[0.0, 0.0], [0.0, 
1.0]], [[1.0, 0.0], [0.0, 1.0]], [[1.0, 0.0], [1.0, 1.0]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonNegative_constData_rank4(self):
      arg=Data(numpy.array([[[[-84.52613988075008, -64.947347041847252, 9.4101305201296839, 35.435694420253526], 
[-7.120286056474697, -6.0689912937658619, 61.501160707208356, -4.1907953448503577], [54.669009036752612, -98.597999209648663, 
79.057818995264938, 34.756927524520307]], [[-70.577681199984156, -52.492379254166764, 35.141902397747174, -16.226709548482205], 
[36.635918130607024, 42.223918562099385, -29.109256279694733, 54.62766676733915], [15.534773817682535, 87.38605319097698, 
1.625572546046854, 27.706127877563389]]], [[[-25.7664484244482, 47.928621558570313, 7.3034707294236796, 78.015951593044775], 
[-43.166701835432143, -38.442261925682033, 30.893468557397057, 76.670511167039166], [92.706771429777774, -55.126224012943425, 
67.321551454448951, 21.740500733205522]], [[-61.907644030699458, -38.306720971672583, 0.028473852938077471, 
-77.315845643856392], [-66.248821618301704, -68.743432146090242, -30.342588891529431, 74.988392905265613], [67.96408525986044, 
-30.961390473072626, -88.014243699266714, -7.3404633115983273]]], [[[82.80863756339042, -84.617444504028242, 
28.915306301329991, -73.0670924857328], [-24.942282815492604, -25.503886997001899, -47.670576507576847, 67.418759772866565], 
[-5.4429473873534846, -9.2842979893077171, 81.764457533224487, 72.156672638590123]], [[-90.515361748774609, 49.995483389181857, 
64.396138141130649, -33.448204883697159], [63.15104838581496, 45.549340467023313, -10.674194426909494, 33.4040207366196], 
[-29.719350476337425, 1.7739287529206678, 46.246160005609397, 56.425625128320206]]]]),self.functionspace)
      res=whereNonNegative(arg)
      ref=Data(numpy.array([[[[0.0, 0.0, 1.0, 1.0], [0.0, 0.0, 1.0, 0.0], [1.0, 0.0, 1.0, 1.0]], [[0.0, 0.0, 1.0, 0.0], [1.0, 
1.0, 0.0, 1.0], [1.0, 1.0, 1.0, 1.0]]], [[[0.0, 1.0, 1.0, 1.0], [0.0, 0.0, 1.0, 1.0], [1.0, 0.0, 1.0, 1.0]], [[0.0, 0.0, 1.0, 
0.0], [0.0, 0.0, 0.0, 1.0], [1.0, 0.0, 0.0, 0.0]]], [[[1.0, 0.0, 1.0, 0.0], [0.0, 0.0, 0.0, 1.0], [0.0, 0.0, 1.0, 1.0]], [[0.0, 
1.0, 1.0, 0.0], [1.0, 1.0, 0.0, 1.0], [0.0, 1.0, 1.0, 1.0]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonNegative_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(56.6120440687)+(1.-msk_arg)*(96.3965404442)
      res=whereNonNegative(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(1.0)+(1.-msk_ref)*(1.0)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonNegative_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([25.636357534486805, 67.685837005139604])+(1.-msk_arg)*numpy.array([-24.15576704470746, 
-84.048449027253042])
      res=whereNonNegative(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([1.0, 1.0])+(1.-msk_ref)*numpy.array([0.0, 0.0])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonNegative_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[-78.072975305635964, -30.667161396580141, 61.34202507179748, -11.419584402032768, 
39.71596012189525], [-25.684737576238177, 53.340398852687258, -3.7920141589733873, -35.952675355382183, -72.00887455038], 
[-66.687289808523516, -55.038138946819949, 87.076259533901833, -62.254642616503332, -29.28679422910929], [-72.59303136759938, 
93.821103037696304, -1.9135138142220853, 29.766163398776797, 
39.407165859633977]])+(1.-msk_arg)*numpy.array([[83.571115792006481, 21.888287744499536, -29.168589548635609, 
34.101960445545103, 25.366974991818296], [37.116659623741299, -20.551392540396819, 9.4877933474515999, -61.562905177089135, 
92.89134307705973], [-52.907637411387178, 55.499587467889029, 48.295777148962998, -48.29039188616369, -41.329781691943211], 
[-89.407168408621132, 48.502208126113885, 40.222625009294035, -62.616466083297404, -70.677377752043967]])
      res=whereNonNegative(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.0, 0.0, 1.0, 0.0, 1.0], [0.0, 1.0, 0.0, 0.0, 0.0], [0.0, 0.0, 1.0, 0.0, 0.0], [0.0, 1.0, 0.0, 
1.0, 1.0]])+(1.-msk_ref)*numpy.array([[1.0, 1.0, 0.0, 1.0, 1.0], [1.0, 0.0, 1.0, 0.0, 1.0], [0.0, 1.0, 1.0, 0.0, 0.0], [0.0, 
1.0, 1.0, 0.0, 0.0]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonNegative_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[40.268335457118411, -86.265565082439849], [-77.060652900660841, -35.613541831532686]], 
[[-11.130159180791651, -50.801802035975065], [77.616966406185526, -70.955846219724322]], [[-59.237474200638005, 
30.293686762130278], [8.9399890546122833, 18.022819538905438]], [[-16.735446376444173, 81.491526325502974], 
[-59.435886845211414, -23.576205091650152]], [[-19.93446385501656, 28.158357589195418], [-81.904756903369673, 
37.850741195421818]], [[-23.170136003419174, -88.117336264200148], [-39.721430862707805, 
-37.158697868374823]]])+(1.-msk_arg)*numpy.array([[[-89.484033408544164, 3.6290659328079471], [55.5035691803443, 
35.04365064842699]], [[-39.575786434207672, 24.075061751850129], [-51.028410946050862, -72.168381468693894]], 
[[8.7978652953924126, 33.743730472745966], [48.813391295251847, 57.556748922574599]], [[-50.67819357472851, 
98.297612205036046], [65.754745719803822, 94.701073496612821]], [[-7.8627589244033089, -77.587700068588745], 
[-78.102433124642317, 80.568138398059347]], [[20.626187739191067, 72.868306997375896], [-92.253636917468327, 
-92.26606957875822]]])
      res=whereNonNegative(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[1.0, 0.0], [0.0, 0.0]], [[0.0, 0.0], [1.0, 0.0]], [[0.0, 1.0], [1.0, 1.0]], [[0.0, 1.0], [0.0, 
0.0]], [[0.0, 1.0], [0.0, 1.0]], [[0.0, 0.0], [0.0, 0.0]]])+(1.-msk_ref)*numpy.array([[[0.0, 1.0], [1.0, 1.0]], [[0.0, 1.0], 
[0.0, 0.0]], [[1.0, 1.0], [1.0, 1.0]], [[0.0, 1.0], [1.0, 1.0]], [[0.0, 0.0], [0.0, 1.0]], [[1.0, 1.0], [0.0, 0.0]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonNegative_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[50.118052228953417, -88.730765495976627, -84.378037931711788, -23.415595349324121], 
[-67.440289645809926, 93.404419793514279, -59.787425377620963, -32.374614750659873], [72.996508166755063, 28.673559835094807, 
-41.279856362022961, -98.826373041981256]], [[34.161465745827087, 60.408826978493238, -55.569419121217997, 98.526543458136359], 
[89.581255428071103, -7.9093712080281477, 19.046497284788245, 46.656250120588084], [-89.213555622028167, -60.048706949978836, 
-96.023118800400738, -38.560321211470217]]], [[[30.610073607783136, -83.484858651904204, -17.939350518275972, 
-19.569367705996626], [-74.45115543876399, 6.2068488889173636, -6.9764471417878156, -9.6106071030669113], [-34.169979148291318, 
39.422153163908035, -82.520572748630272, -97.826881500667696]], [[-43.805063575145113, 52.820087341249064, -11.425891732077531, 
91.225666306308483], [-65.7372703434865, -62.123010875670246, -34.999218025158257, 82.649036492923187], [26.621421005843573, 
-24.208721371623469, 36.104693970302208, -26.311040499741821]]], [[[-73.614891770734744, -85.837005758293458, 
-53.722074500413754, 80.21071183957946], [46.723144872292266, -6.4559074348415066, 76.314206389016562, 49.12876304442392], 
[-24.203320133532188, 92.593031809694111, 95.529348278263171, 79.699349442454661]], [[47.068575118893051, 38.079994995473555, 
49.966665821438681, 26.424916515149306], [-87.637253126554185, -12.958087998221799, -84.146782819641402, -1.3189534835551484], 
[-78.523734070290715, 46.584675916144505, 12.785152169211571, 
-39.544671827810497]]]])+(1.-msk_arg)*numpy.array([[[[-44.483784104558957, -28.006494802891964, 92.253214471020186, 
-15.814708339708289], [-34.727972062008462, 18.489845474296487, -8.684561294250372, -39.459862391993617], [3.2532135157034503, 
97.032451670690847, -5.0819510089139328, 78.319214911261668]], [[-98.398026812312395, 59.505846535267978, 10.784341452949846, 
-14.250704963278665], [16.595393811048226, 4.8049629604389565, -76.61905163551242, -96.404190597114223], [11.885589053645433, 
-73.332964281192645, 53.445417740070354, 93.807480385985798]]], [[[66.632634303970747, 28.645399611787298, -83.987062053435622, 
-62.89109800936339], [-24.23393761880088, 48.42862684926024, 40.07211030469216, -81.012080582820943], [50.754863806056903, 
-68.507318003236549, -25.055399867857432, 62.59244861144785]], [[-11.047268692459596, -90.177853490375483, -11.20089156470867, 
35.53859445005051], [59.047803703205204, 23.854271835098714, -89.220352115110501, -54.108930266124823], [-70.781715085187784, 
30.727426768606961, 19.981466099151106, -26.098236640973766]]], [[[71.538263486211775, -85.096322580377006, 79.221899163626887, 
-59.858169598469793], [47.338661263600414, 93.057272248217714, 75.394754283706391, 58.169107710198773], [73.772225431452796, 
-59.557034519677707, -26.119548712323578, 31.027715412640816]], [[61.119765212242982, 42.526393080611683, 69.940051844556393, 
-54.370980511290881], [-75.41802697240314, -87.096596680306959, 90.532406636535342, -46.482642233552987], [-46.532222608169491, 
73.733664853166999, -93.120651954554361, 42.436012745957726]]]])
      res=whereNonNegative(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[1.0, 0.0, 0.0, 0.0], [0.0, 1.0, 0.0, 0.0], [1.0, 1.0, 0.0, 0.0]], [[1.0, 1.0, 0.0, 1.0], 
[1.0, 0.0, 1.0, 1.0], [0.0, 0.0, 0.0, 0.0]]], [[[1.0, 0.0, 0.0, 0.0], [0.0, 1.0, 0.0, 0.0], [0.0, 1.0, 0.0, 0.0]], [[0.0, 1.0, 
0.0, 1.0], [0.0, 0.0, 0.0, 1.0], [1.0, 0.0, 1.0, 0.0]]], [[[0.0, 0.0, 0.0, 1.0], [1.0, 0.0, 1.0, 1.0], [0.0, 1.0, 1.0, 1.0]], 
[[1.0, 1.0, 1.0, 1.0], [0.0, 0.0, 0.0, 0.0], [0.0, 1.0, 1.0, 0.0]]]])+(1.-msk_ref)*numpy.array([[[[0.0, 0.0, 1.0, 0.0], [0.0, 
1.0, 0.0, 0.0], [1.0, 1.0, 0.0, 1.0]], [[0.0, 1.0, 1.0, 0.0], [1.0, 1.0, 0.0, 0.0], [1.0, 0.0, 1.0, 1.0]]], [[[1.0, 1.0, 0.0, 
0.0], [0.0, 1.0, 1.0, 0.0], [1.0, 0.0, 0.0, 1.0]], [[0.0, 0.0, 0.0, 1.0], [1.0, 1.0, 0.0, 0.0], [0.0, 1.0, 1.0, 0.0]]], [[[1.0, 
0.0, 1.0, 0.0], [1.0, 1.0, 1.0, 1.0], [1.0, 0.0, 0.0, 1.0]], [[1.0, 1.0, 1.0, 0.0], [0.0, 0.0, 1.0, 0.0], [0.0, 1.0, 0.0, 
1.0]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonPositive_float_rank0(self):
      arg=-45.8140100626
      res=whereNonPositive(arg)
      ref=1.0
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonPositive_array_rank0(self):
      arg=numpy.array(-63.8959249588)
      res=whereNonPositive(arg)
      ref=numpy.array(1.0)
      if not isinstance(res,float):
         self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
         self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonPositive_array_rank1(self):
      arg=numpy.array([54.415838481100479, -35.5624835416114])
      res=whereNonPositive(arg)
      ref=numpy.array([0.0, 1.0])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonPositive_array_rank2(self):
      arg=numpy.array([[-19.977776356765318, 5.9786051244587952, -56.499280299922752, -34.772432329184056, 
-42.376287015366223], [-84.487175491423102, -47.807388262784926, -2.1549567968020398, 65.070219308351994, -81.156967947597551], 
[39.362869981047908, 82.897480274445286, -25.251661283915567, -14.554050325024775, 99.441792996707022], [-87.49041530809086, 
-32.750499747633071, 40.491174827110797, 68.288080492730728, 21.562840881641691]])
      res=whereNonPositive(arg)
      ref=numpy.array([[1.0, 0.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 0.0, 1.0], [0.0, 0.0, 1.0, 1.0, 0.0], [1.0, 1.0, 0.0, 0.0, 
0.0]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonPositive_array_rank3(self):
      arg=numpy.array([[[-80.816933807988647, 18.410516218736745], [-84.025394562335066, 73.817364376601347]], 
[[61.338668393303806, -19.309105906918774], [60.531656152601499, -62.629597501263888]], [[98.405881825725487, 
-87.157404259910081], [-77.87716088817686, 54.080880261791691]], [[44.017293267806849, 81.605183414497702], 
[25.005545230944733, 43.880992994339664]], [[-9.0200660048142396, -52.950279136546015], [33.391394798474323, 
77.366312324785582]], [[-12.930426357736494, 26.724479555287701], [76.853844695874045, -73.905941842749797]]])
      res=whereNonPositive(arg)
      ref=numpy.array([[[1.0, 0.0], [1.0, 0.0]], [[0.0, 1.0], [0.0, 1.0]], [[0.0, 1.0], [1.0, 0.0]], [[0.0, 0.0], [0.0, 0.0]], 
[[1.0, 1.0], [0.0, 0.0]], [[1.0, 0.0], [0.0, 1.0]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonPositive_array_rank4(self):
      arg=numpy.array([[[[84.854900822533551, -70.314859921760501, 91.167657851483199, 58.634317483028553], 
[-45.674981783852253, -91.344582047164224, -45.296293433418299, -11.820391409783412], [7.5520922261311085, 33.939791936902679, 
-30.78101779452625, 92.217815104797097]], [[16.487093948363324, 46.549440093303588, -35.737509999956288, -79.69614030357792], 
[28.449486617318911, -62.884436564134603, 40.844828284827628, 10.366425813044785], [79.665416005370531, 39.56742434170576, 
-27.721667918864227, -81.954150375477553]]], [[[47.951911230948269, -25.788610534552575, -16.885257596759516, 
82.068516597608067], [-0.51652980482228372, -66.573004416820339, -72.185890143047729, -63.647624985740258], 
[93.906834018256092, -58.422363414411606, -39.708973666342608, -7.1654247643074314]], [[-82.284191996980809, 
63.235007339504534, -78.292387412733092, -52.126182522828877], [16.684565256071693, 57.832220397954075, -38.766429611788354, 
-62.311953863196749], [-60.30897110518432, -42.386777033904146, -60.890434946023241, 78.663607764550704]]], 
[[[-47.287040896177388, -20.906728008055325, 48.334998443867818, 18.78306004141406], [-74.950637186434733, 84.51043393945514, 
14.145744299214428, 76.234540063777558], [18.666731132635476, 83.354101505920454, 48.433353335770846, -50.240885823292778]], 
[[-72.060114740388656, -55.762235089994427, -35.520941042497213, 49.448425790974369], [-0.16498976632010454, 
3.6711888832083588, 0.44840601553353565, -4.6970780448722991], [-94.786520432934168, 18.525824825003696, 61.101943285402172, 
-87.622236927860868]]]])
      res=whereNonPositive(arg)
      ref=numpy.array([[[[0.0, 1.0, 0.0, 0.0], [1.0, 1.0, 1.0, 1.0], [0.0, 0.0, 1.0, 0.0]], [[0.0, 0.0, 1.0, 1.0], [0.0, 1.0, 
0.0, 0.0], [0.0, 0.0, 1.0, 1.0]]], [[[0.0, 1.0, 1.0, 0.0], [1.0, 1.0, 1.0, 1.0], [0.0, 1.0, 1.0, 1.0]], [[1.0, 0.0, 1.0, 1.0], 
[0.0, 0.0, 1.0, 1.0], [1.0, 1.0, 1.0, 0.0]]], [[[1.0, 1.0, 0.0, 0.0], [1.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 1.0]], [[1.0, 1.0, 
1.0, 0.0], [1.0, 0.0, 0.0, 1.0], [1.0, 0.0, 0.0, 1.0]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonPositive_constData_rank0(self):
      arg=Data(76.8341760885,self.functionspace)
      res=whereNonPositive(arg)
      ref=Data(0.0,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonPositive_constData_rank1(self):
      arg=Data(numpy.array([5.110430775637127, -8.5929188047610552]),self.functionspace)
      res=whereNonPositive(arg)
      ref=Data(numpy.array([0.0, 1.0]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonPositive_constData_rank2(self):
      arg=Data(numpy.array([[73.186196525129645, 63.106873510282924, -37.93591985539193, -4.9728512739999076, 
19.376979391439363], [4.5452345913447658, 46.103234583667671, 18.062219251009836, 99.310495136151133, 51.526446572677145], 
[-94.648587297912812, -79.599690191233478, 95.310895065698446, -91.742395450846828, -95.503130322854204], [54.132238583769265, 
-6.2958941894595881, 80.387872082718758, -56.594075571445003, -0.087162886623445957]]),self.functionspace)
      res=whereNonPositive(arg)
      ref=Data(numpy.array([[0.0, 0.0, 1.0, 1.0, 0.0], [0.0, 0.0, 0.0, 0.0, 0.0], [1.0, 1.0, 0.0, 1.0, 1.0], [0.0, 1.0, 0.0, 
1.0, 1.0]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonPositive_constData_rank3(self):
      arg=Data(numpy.array([[[35.640522044117716, 35.479584214315452], [-83.66008428402651, -12.353315661361748]], 
[[-46.695985576916875, 56.815876089929901], [14.328557434943392, 54.490435657692615]], [[38.157071729732564, 
-63.060450376793042], [-56.69840568574287, 45.584077281388033]], [[96.777622099398059, -5.1796500781218384], 
[89.674102699704235, 41.472615569686411]], [[59.694746638725746, -37.205367017231382], [-29.955192328084593, 
55.672175132947785]], [[35.375171131639576, -93.866403218358215], [23.27567992190464, 
-49.139452886576862]]]),self.functionspace)
      res=whereNonPositive(arg)
      ref=Data(numpy.array([[[0.0, 0.0], [1.0, 1.0]], [[1.0, 0.0], [0.0, 0.0]], [[0.0, 1.0], [1.0, 0.0]], [[0.0, 1.0], [0.0, 
0.0]], [[0.0, 1.0], [1.0, 0.0]], [[0.0, 1.0], [0.0, 1.0]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonPositive_constData_rank4(self):
      arg=Data(numpy.array([[[[78.120351293960368, 45.517827635106755, 13.754081165958112, 53.648332704104661], 
[-95.730532610383534, 64.863790698266143, -56.063132630021649, 66.592428370803646], [63.399839434838213, -73.022434587860104, 
-67.904116954138885, -50.280183675978037]], [[-29.930362494006559, 32.052615321911333, 72.250187735124626, 33.654605438190998], 
[-65.127089193367851, -32.722417502163452, 38.596924965290668, -81.264280989691599], [-26.325082622716451, 92.723726612008733, 
-6.9104901305947948, -56.732407303629607]]], [[[81.299873121289409, -0.60752785487785843, 81.02975074643345, 
5.6594823714348053], [17.236459205051176, 2.9463513679599771, 96.07565713416264, -70.792732773851498], [87.611958793760635, 
99.359964258540117, 10.130252954092242, -67.297975693155252]], [[-71.553505655481999, -90.989697336378043, -57.310147425889625, 
-4.7051290563562276], [45.896460083362314, 79.487272101784242, 33.393465498093576, 90.509868144157366], [-68.059017250831175, 
98.172331068427752, 76.21324396944388, -18.114975003809363]]], [[[30.18165573581669, -95.406985301029422, -41.697668617320851, 
-88.452149593062245], [-26.45265307152971, 7.5605523928963692, 87.21992505390017, -83.416286385195633], [-25.719143977748814, 
57.849131833666064, 16.280200193493343, -48.38523212914918]], [[22.977807767062089, -16.18840952303114, 99.764245118344576, 
-31.703462585340404], [-11.504718586929556, -24.746181326375378, 30.058139117771191, 93.910717466047032], [-79.520098835545866, 
30.192855384126148, 99.121945637672439, -59.524886303278301]]]]),self.functionspace)
      res=whereNonPositive(arg)
      ref=Data(numpy.array([[[[0.0, 0.0, 0.0, 0.0], [1.0, 0.0, 1.0, 0.0], [0.0, 1.0, 1.0, 1.0]], [[1.0, 0.0, 0.0, 0.0], [1.0, 
1.0, 0.0, 1.0], [1.0, 0.0, 1.0, 1.0]]], [[[0.0, 1.0, 0.0, 0.0], [0.0, 0.0, 0.0, 1.0], [0.0, 0.0, 0.0, 1.0]], [[1.0, 1.0, 1.0, 
1.0], [0.0, 0.0, 0.0, 0.0], [1.0, 0.0, 0.0, 1.0]]], [[[0.0, 1.0, 1.0, 1.0], [1.0, 0.0, 0.0, 1.0], [1.0, 0.0, 0.0, 1.0]], [[0.0, 
1.0, 0.0, 1.0], [1.0, 1.0, 0.0, 0.0], [1.0, 0.0, 0.0, 1.0]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonPositive_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(-21.294938937)+(1.-msk_arg)*(-72.0328127018)
      res=whereNonPositive(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(1.0)+(1.-msk_ref)*(1.0)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonPositive_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([14.001065620474165, 51.060850237675112])+(1.-msk_arg)*numpy.array([-6.4845892236093334, 
73.407748235087723])
      res=whereNonPositive(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.0, 0.0])+(1.-msk_ref)*numpy.array([1.0, 0.0])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonPositive_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[-64.52120230042874, -31.827528475750682, 10.004997743630327, 91.575189079620088, 
50.375139386687209], [41.720379535781916, 63.159827023202382, 49.57432401463214, 9.3070620932102202, 60.232643522299185], 
[25.240200875891901, -96.938260964323163, 62.0452615917537, -42.644345796361179, -82.263178482606222], [-75.954278388926681, 
14.813887736649605, 8.7669517953596028, -76.802613030476522, 
4.3723418171869923]])+(1.-msk_arg)*numpy.array([[3.6464123199156973, -30.094451732015187, -38.084149034576512, 
16.553157123976646, 49.784666124112704], [-66.53566180083385, -1.0279761734758068, 36.193921499854014, 27.465955037373504, 
8.41610652840032], [45.914465105438126, 47.11191355094499, -78.536609881550731, -28.478652266113855, -36.443325125458003], 
[86.47075792022045, -8.4827911972243299, 36.645811902720908, 45.3318814313877, -93.9890221530123]])
      res=whereNonPositive(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[1.0, 1.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0, 0.0], [0.0, 1.0, 0.0, 1.0, 1.0], [1.0, 0.0, 0.0, 
1.0, 0.0]])+(1.-msk_ref)*numpy.array([[0.0, 1.0, 1.0, 0.0, 0.0], [1.0, 1.0, 0.0, 0.0, 0.0], [0.0, 0.0, 1.0, 1.0, 1.0], [0.0, 
1.0, 0.0, 0.0, 1.0]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonPositive_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[3.4932152236425225, -36.146311025025412], [-79.511782256523205, 43.209384016358598]], 
[[-12.911756606412169, 84.296575347249728], [-18.796706505874795, -65.624653714469986]], [[20.077128133134423, 
82.197624733378518], [-10.896782554696998, 8.3500680112954484]], [[-55.918585807034724, -71.322632870917687], 
[-79.500335685116255, -66.109585462512442]], [[5.3425876719804819, 59.506331837634548], [4.3592655512523493, 
-44.497382825601115]], [[-8.9240915992721028, 56.420248561123145], [10.275532725278708, 
-14.724999384740372]]])+(1.-msk_arg)*numpy.array([[[-50.654219085401195, -20.564018798701042], [-6.8157495083358981, 
-8.7152789021806001]], [[67.612800754357124, 20.241913145493101], [68.465229152050711, 32.266053812298566]], 
[[-0.88016538230590413, 41.688901064752685], [27.230966724318776, 32.520603018797004]], [[26.061980470503499, 
-54.87980460183941], [7.6346427424732042, 20.663565250317689]], [[-96.61073536198441, -73.190601459898843], 
[-61.479505152207395, -99.707856346427221]], [[-9.3807883210439371, -9.3488757322235188], [-6.7769146729327758, 
-73.95899955952676]]])
      res=whereNonPositive(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.0, 1.0], [1.0, 0.0]], [[1.0, 0.0], [1.0, 1.0]], [[0.0, 0.0], [1.0, 0.0]], [[1.0, 1.0], [1.0, 
1.0]], [[0.0, 0.0], [0.0, 1.0]], [[1.0, 0.0], [0.0, 1.0]]])+(1.-msk_ref)*numpy.array([[[1.0, 1.0], [1.0, 1.0]], [[0.0, 0.0], 
[0.0, 0.0]], [[1.0, 0.0], [0.0, 0.0]], [[0.0, 1.0], [0.0, 0.0]], [[1.0, 1.0], [1.0, 1.0]], [[1.0, 1.0], [1.0, 1.0]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonPositive_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[-58.89835447211469, -6.0893328346566733, -1.0023693518593575, -61.539230319459627], 
[60.870457135837682, -83.305095813845867, -38.116740546736374, -64.075745680293039], [79.081243725668998, -3.1565899739248522, 
-69.359266580736985, -25.939574951041493]], [[44.840056891487734, 9.4190204246416016, 33.086497185256661, 45.011451755319172], 
[94.214589826739939, -5.702363967501924, -19.973298938018431, 76.184762328377644], [7.5290921100338721, 48.192190457350875, 
80.630439347990887, -10.677496472704931]]], [[[11.221137534077187, 7.1347297082242846, 25.87988065153138, -3.7157692974099064], 
[-76.043759143725808, 69.02633203511769, -98.860304021235777, 53.531068740423734], [-34.89611427266945, 27.156902812036535, 
72.743609137679357, 95.314597465726479]], [[78.038851407351615, 56.72377621086261, -20.725276919441342, -58.75384173399285], 
[-26.213835341147828, -86.937961787952219, -45.889334819778796, 31.050065322925832], [75.255475636162998, -54.929307087299996, 
-42.283656056524642, 68.6586886677525]]], [[[-28.16883583872432, -40.372922117108565, -30.135564687244994, 50.246251468410065], 
[-7.2498686947860165, 94.45851868271248, -53.581025727792642, -32.424001887898868], [85.576621269199592, 58.609425585253945, 
78.085399899893304, 98.996509927055513]], [[-75.317618268425264, 11.26227410259915, -51.177413277829785, -34.929151547420361], 
[4.3965337305110097, 90.923268663940149, -89.483861691460845, -5.4115854061605972], [-49.940568614308049, -28.969569648394568, 
-95.982720688496244, 87.745557466302273]]]])+(1.-msk_arg)*numpy.array([[[[-74.771460648328443, 13.793031178508002, 
-41.976214738752859, -25.36331645721701], [24.01507156186311, 60.776951586528895, 8.3096968814290619, -60.613551059743067], 
[1.4391448885635327, -92.057454358544248, 13.907184752443811, -52.254719033883632]], [[-46.910170661904637, 
-33.334225663168681, -9.3791104242402952, -15.404344393378281], [69.025944878882456, -51.909070828632785, 31.053379001160948, 
62.760215014236422], [-74.91014748820912, 22.358213500298248, -91.88145013095253, 35.630042926552306]]], 
[[[-29.279785927448572, -98.509512781683981, -3.2141945191207668, -7.8787884397738992], [61.929082763373486, 
-13.285831454240963, -22.609607789839885, -95.781398772215994], [11.65303310062788, -55.087961160687016, -7.340663618342063, 
-51.681435909557429]], [[10.223206694434708, 52.870731191118949, 94.343773117331011, 54.502331701573127], [-74.304124617034418, 
37.5127280034894, 87.414312181666446, 5.5216671015867007], [-48.162344739211704, 40.109818025385323, 88.765369352144177, 
-3.2256322380404612]]], [[[-21.040797096187333, 38.819012415945906, -40.815118132374636, 57.1068994645087], 
[44.584730270097026, 63.853583439836456, 64.648203529995698, 19.794733417566988], [90.528792123051915, -65.456358000132781, 
10.691310766549861, 90.084242627365683]], [[-64.643286624216984, 36.640703954145948, 82.940348145540838, -74.20215167286193], 
[80.641128632128385, 7.801376809332325, 8.1922136533463004, 85.312711221718672], [60.727877225987015, -96.319606263671147, 
42.598308142226301, 27.966063247606769]]]])
      res=whereNonPositive(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[1.0, 1.0, 1.0, 1.0], [0.0, 1.0, 1.0, 1.0], [0.0, 1.0, 1.0, 1.0]], [[0.0, 0.0, 0.0, 0.0], 
[0.0, 1.0, 1.0, 0.0], [0.0, 0.0, 0.0, 1.0]]], [[[0.0, 0.0, 0.0, 1.0], [1.0, 0.0, 1.0, 0.0], [1.0, 0.0, 0.0, 0.0]], [[0.0, 0.0, 
1.0, 1.0], [1.0, 1.0, 1.0, 0.0], [0.0, 1.0, 1.0, 0.0]]], [[[1.0, 1.0, 1.0, 0.0], [1.0, 0.0, 1.0, 1.0], [0.0, 0.0, 0.0, 0.0]], 
[[1.0, 0.0, 1.0, 1.0], [0.0, 0.0, 1.0, 1.0], [1.0, 1.0, 1.0, 0.0]]]])+(1.-msk_ref)*numpy.array([[[[1.0, 0.0, 1.0, 1.0], [0.0, 
0.0, 0.0, 1.0], [0.0, 1.0, 0.0, 1.0]], [[1.0, 1.0, 1.0, 1.0], [0.0, 1.0, 0.0, 0.0], [1.0, 0.0, 1.0, 0.0]]], [[[1.0, 1.0, 1.0, 
1.0], [0.0, 1.0, 1.0, 1.0], [0.0, 1.0, 1.0, 1.0]], [[0.0, 0.0, 0.0, 0.0], [1.0, 0.0, 0.0, 0.0], [1.0, 0.0, 0.0, 1.0]]], [[[1.0, 
0.0, 1.0, 0.0], [0.0, 0.0, 0.0, 0.0], [0.0, 1.0, 0.0, 0.0]], [[1.0, 0.0, 0.0, 1.0], [0.0, 0.0, 0.0, 0.0], [0.0, 1.0, 0.0, 
0.0]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereZero_float_rank0(self):
      arg=36.9194219022
      res=whereZero(arg)
      ref=0.0
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereZero_array_rank0(self):
      arg=numpy.array(-93.2286970236)
      res=whereZero(arg)
      ref=numpy.array(0.0)
      if not isinstance(res,float):
         self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
         self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereZero_array_rank1(self):
      arg=numpy.array([-5.4758135185183079, -12.393482705182549])
      res=whereZero(arg)
      ref=numpy.array([0.0, 0.0])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereZero_array_rank2(self):
      arg=numpy.array([[-22.216037501798056, 4.9281618608239484, 24.076323373045312, -1.5837381383595073, 89.085883212985664], 
[-31.966408477169978, 59.125907926918728, 22.148581815123777, -3.4787216423014513, 70.486779662550191], [-38.434264024343953, 
-28.545406042409468, -86.569085677953652, 29.872458065808701, 82.968130635374706], [-0.12806513945584186, 5.2584442959378919, 
88.481457835654652, 52.526216268675853, -47.867118858740824]])
      res=whereZero(arg)
      ref=numpy.array([[0.0, 0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0, 
0.0]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereZero_array_rank3(self):
      arg=numpy.array([[[83.566843166821798, 83.428361682287743], [-27.142097287854597, -7.7449066894637042]], 
[[95.740285720896139, 58.956113534039076], [50.123535284541219, 51.623550969006601]], [[95.393321001956423, 
88.750537248542344], [-29.708780869260522, -90.320511258990138]], [[75.205523953075215, -81.374513795204507], 
[-70.55439045053609, 27.401171040594903]], [[23.344274043598048, -52.331497199756583], [77.956546937871479, 
25.315698227587191]], [[-73.700561398579438, -10.261392160687222], [-0.82001847170529629, -69.219536679954956]]])
      res=whereZero(arg)
      ref=numpy.array([[[0.0, 0.0], [0.0, 0.0]], [[0.0, 0.0], [0.0, 0.0]], [[0.0, 0.0], [0.0, 0.0]], [[0.0, 0.0], [0.0, 0.0]], 
[[0.0, 0.0], [0.0, 0.0]], [[0.0, 0.0], [0.0, 0.0]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereZero_array_rank4(self):
      arg=numpy.array([[[[25.646485297551976, 11.073836310281109, 54.114761331724992, -12.154046230984548], 
[55.324874795537227, 28.95659934737148, 94.639492916569793, 34.007912394649708], [63.557437176693213, 74.008668533501378, 
64.891082638452303, 75.02732907327038]], [[0.10812203896047379, -56.42920257387194, 25.284865245539166, 56.659069500901381], 
[-24.816276674907726, 54.234226668365494, 30.712957393064244, 99.206875660084734], [-70.610950252661524, -95.570055459069891, 
-15.080615988435554, -34.468629003757911]]], [[[56.891765767496025, -18.388156109946863, -69.598390029982539, 
-49.746422132267121], [-55.031142846420522, -76.066644780947399, -77.183345994632262, 57.117746506564544], 
[-90.739836243999989, -64.407621405298698, 47.917245532062736, 40.184055666825515]], [[-50.242499172011243, -8.87012437468087, 
43.485752684298006, 1.9802518704210286], [52.738617953516098, 67.416303731706478, -93.317491063936075, -10.148998504978351], 
[41.786805141066679, -0.75710798140093516, -0.44487284484937106, 52.422098163397976]]], [[[-56.523619961520424, 
-56.87408626775543, 86.257157391371067, -59.666478835501756], [-27.523677831571518, 26.978800410887743, 41.981358630995317, 
60.966684046224884], [-88.336158365149743, 25.253390160451673, 24.947726326096927, -69.88210288786064]], [[28.934555939829323, 
28.843750360082907, 52.298137157289517, 98.435696242381738], [-45.858063671063661, -24.386879374695596, -4.7125807697196933, 
18.897314951135954], [-68.600970437531487, 46.967350537404627, 24.491641179009946, -14.113138296837874]]]])
      res=whereZero(arg)
      ref=numpy.array([[[[0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0]], [[0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 
0.0, 0.0], [0.0, 0.0, 0.0, 0.0]]], [[[0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0]], [[0.0, 0.0, 0.0, 0.0], 
[0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0]]], [[[0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0]], [[0.0, 0.0, 
0.0, 0.0], [0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereZero_constData_rank0(self):
      arg=Data(28.1448958043,self.functionspace)
      res=whereZero(arg)
      ref=Data(0.0,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereZero_constData_rank1(self):
      arg=Data(numpy.array([-82.424192879607403, -37.742931006156489]),self.functionspace)
      res=whereZero(arg)
      ref=Data(numpy.array([0.0, 0.0]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereZero_constData_rank2(self):
      arg=Data(numpy.array([[88.228357156618614, -67.868123140983812, 1.512038393501669, -2.4090303677805451, 
-77.431662457170702], [-28.559471393034855, -15.937848887398616, -10.531725851274871, -1.7510038488192521, 
-44.964870198048246], [-39.261122217051955, 63.804285642486178, -92.35899221299897, 83.265740587577596, 25.680465013643357], 
[54.5460930839632, 60.917809009044504, -50.068271855067373, -27.753796832679427, 24.173974108605705]]),self.functionspace)
      res=whereZero(arg)
      ref=Data(numpy.array([[0.0, 0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 
0.0, 0.0]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereZero_constData_rank3(self):
      arg=Data(numpy.array([[[5.8006491078266436, 35.129891450052469], [72.468149481480708, 30.423431420167645]], 
[[-97.49296225256478, -95.545777693389056], [-19.272840387210778, -52.712492439965295]], [[-8.4318486054725099, 
-48.14775009081422], [15.485176866163101, 20.812291283387779]], [[59.148678501666808, 44.49011017020382], [68.060455906204766, 
-28.366297330479597]], [[-53.89552234182662, -56.019523425289734], [-12.787919401488608, -25.640418311378554]], 
[[-80.946202928069681, 98.68988212964004], [-86.271543759781096, 58.957148643203482]]]),self.functionspace)
      res=whereZero(arg)
      ref=Data(numpy.array([[[0.0, 0.0], [0.0, 0.0]], [[0.0, 0.0], [0.0, 0.0]], [[0.0, 0.0], [0.0, 0.0]], [[0.0, 0.0], [0.0, 
0.0]], [[0.0, 0.0], [0.0, 0.0]], [[0.0, 0.0], [0.0, 0.0]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereZero_constData_rank4(self):
      arg=Data(numpy.array([[[[-65.177620457916689, 67.29262327490423, -7.3415276413508082, -5.2405941808933392], 
[-48.758858112961768, 22.839891911492273, 79.281591854134689, -31.661211799895611], [-29.378697622508909, 97.445171039982398, 
-3.9519401978111972, 76.511678962031851]], [[-62.423828480664746, 56.125375059826581, -75.453141434267295, 
-58.099034647323307], [-97.405453924636291, 50.587164071913492, -62.207039945193785, 6.9380229297484561], [-97.178170849233254, 
95.128123507156232, 73.270487977455275, 43.015502805504582]]], [[[-18.640235019885793, -64.312133051653603, 94.354412363014035, 
22.993281228643994], [-91.306876358518196, 43.015613220689886, 33.570404531422213, 82.971175030422302], [31.335392435326156, 
-34.121087575670714, 78.788049923464627, -5.6671192929215692]], [[85.897768300402873, -12.59829275357778, 43.814744997102991, 
6.6323709267625048], [-43.13141600459609, -90.169076035549622, -61.538645551752701, -65.336406452922034], [25.4847655361812, 
79.757518347506931, 4.4404225559480039, -90.310854126467547]]], [[[-89.942170042909737, 48.83614806552265, 27.339850059944709, 
40.364684920253126], [82.979560873661626, -14.475961583560661, -62.784944309919325, -98.935571321987737], [-88.87373557335512, 
-85.243755112678102, -26.797719666615038, -78.779936332270836]], [[-97.025077640729592, 60.965387643532466, 
-96.879335350883025, -44.074960019423479], [84.871574286123774, 63.119314763292721, -64.659489888329944, 84.502657554937542], 
[65.086432289042222, -21.632326349787093, -4.2254524282816277, 84.264250212943978]]]]),self.functionspace)
      res=whereZero(arg)
      ref=Data(numpy.array([[[[0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0]], [[0.0, 0.0, 0.0, 0.0], [0.0, 
0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0]]], [[[0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0]], [[0.0, 0.0, 0.0, 
0.0], [0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0]]], [[[0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0]], [[0.0, 
0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereZero_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(32.6280408645)+(1.-msk_arg)*(95.9857415093)
      res=whereZero(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(0.0)+(1.-msk_ref)*(0.0)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereZero_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([78.912256081910982, 95.107329887542988])+(1.-msk_arg)*numpy.array([-93.129096936828716, 
-31.119112819271606])
      res=whereZero(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.0, 0.0])+(1.-msk_ref)*numpy.array([0.0, 0.0])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereZero_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[-4.5674063676684113, -89.281893625146708, 23.19996829396085, 18.836280966336673, 
48.069098115838699], [-22.012453235487612, 49.277998375509071, 36.357927774507999, -81.584591164471846, -86.383640243979073], 
[-11.398848874052291, 53.725683521115826, 57.996950911680813, -19.278919654298136, -7.6330215452677663], [23.212683532644604, 
47.218488529598716, 94.087207901705796, -19.986642121880791, 
-60.768388103484973]])+(1.-msk_arg)*numpy.array([[-91.766428267475405, -80.743485192118527, 49.721965309529139, 
-8.4541150165939598, 11.588341364396655], [-66.832555155876435, -41.812843111232766, -42.848862832974092, 38.718794135330427, 
97.120556706875561], [-1.7661898089937438, 9.3172028698214291, 64.632791057438652, 8.7177199255444151, 97.409280590459673], 
[-40.687002837505325, -44.151855951781215, 81.104463998584123, 68.501916362320713, 15.603376060294408]])
      res=whereZero(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.0, 0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 
0.0, 0.0]])+(1.-msk_ref)*numpy.array([[0.0, 0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0, 0.0], [0.0, 
0.0, 0.0, 0.0, 0.0]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereZero_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[-21.084709880958158, 60.058563461115341], [23.462135905224613, 4.1925110299263935]], 
[[-73.752445728764471, 98.917050977312613], [91.380933382588495, 96.936759674063694]], [[-42.119594542112758, 
-36.557259071778894], [-41.337250253991378, -23.488672516050045]], [[79.635013224815708, 34.63320785977092], 
[52.590069604255888, 21.512417656206551]], [[54.361795611776955, -45.247793152599435], [-90.021841110057906, 
80.87660694310236]], [[68.838235188657535, 53.277307015666736], [69.927085977405824, 
-86.715842974659438]]])+(1.-msk_arg)*numpy.array([[[35.824777853004889, -98.41139283617531], [46.481491612419063, 
73.744423180097272]], [[-62.535302751139653, 90.281500522248365], [75.01937602982872, -28.925552206472531]], 
[[-13.317676532324427, -65.881247444860747], [88.671022199931372, 73.998882504024408]], [[0.15025950906122887, 
-94.853839866383098], [46.402552203453979, -47.706832151504287]], [[-89.242628256571521, -34.33096936505919], 
[81.905543323362821, -16.972568236626273]], [[96.777671705474376, -54.187557400745568], [31.04774327078735, 
-10.538455973188675]]])
      res=whereZero(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.0, 0.0], [0.0, 0.0]], [[0.0, 0.0], [0.0, 0.0]], [[0.0, 0.0], [0.0, 0.0]], [[0.0, 0.0], [0.0, 
0.0]], [[0.0, 0.0], [0.0, 0.0]], [[0.0, 0.0], [0.0, 0.0]]])+(1.-msk_ref)*numpy.array([[[0.0, 0.0], [0.0, 0.0]], [[0.0, 0.0], 
[0.0, 0.0]], [[0.0, 0.0], [0.0, 0.0]], [[0.0, 0.0], [0.0, 0.0]], [[0.0, 0.0], [0.0, 0.0]], [[0.0, 0.0], [0.0, 0.0]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereZero_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[-32.662481234741009, 40.260928889330899, 75.847123834183577, 78.64906987881929], 
[-63.275395114512165, -31.782530467351066, -33.223085447198656, -41.760129832198146], [-74.215302896565618, 96.846057240589488, 
-42.720460482944176, -81.859139209222548]], [[-80.806077534597605, -71.830510003086573, -52.675707514171592, 
59.066414162745303], [-70.588014101153917, 51.148551968394855, -74.276699989547552, 73.901372979374713], [70.366974039594226, 
-9.8398879113183852, 36.910219459152358, 70.56418458925782]]], [[[-86.98957296102455, -81.086417993215889, -46.036348376772821, 
1.9830002078387707], [-81.396274030320797, -7.4045882449412943, 6.3580886221511008, 79.040288934545032], [-14.153401419846759, 
67.679531820133519, 51.592855364211687, 90.708140577398439]], [[-94.306912199804941, 98.63226780530934, -8.9357196058543309, 
4.7943903912119907], [-74.373684606723373, 91.169643203671882, 16.841492324960569, 93.632465846065628], [-99.088266439789805, 
45.694749451260208, -90.330018599691812, -60.130076035705656]]], [[[-7.2101696440878698, -34.214635113611934, 
-41.058199491627747, 16.37206535285118], [12.97372741143397, 61.291284156806938, 92.690533886945445, -90.198572175888316], 
[43.854839856732838, 69.937218292009533, -9.1804659284850771, -94.216640840953787]], [[68.181402600087836, 0.30694213217643096, 
-28.922839864406868, -23.615203338489295], [-25.458367043581916, -53.828155760722353, -87.161998896681524, 25.519257343379692], 
[79.178020952386134, -30.669991922330126, 69.968344114998871, 
-2.5714051990672857]]]])+(1.-msk_arg)*numpy.array([[[[85.208773947656681, 88.332585876380023, -44.90871447056162, 
-43.006236243527241], [-2.047920558204595, 89.27443100854407, -5.5594319239118448, -51.774859847771722], [-60.342290733130845, 
30.737338461190973, -11.740568632642152, 92.593235560303924]], [[-16.237930188618634, 85.904441626286911, 73.575939167565167, 
61.372847005304777], [-25.293050011864722, 86.444597934853704, 65.144765734125684, 24.221545914400892], [-45.788018695054447, 
20.030203845344701, -38.808816142235322, 95.412752620189195]]], [[[95.355286693057337, -20.930533826082325, 
-53.087896571538536, 58.552873058160117], [-9.8932393167384873, -63.401775116131475, -78.43041141947036, 8.9216142837929198], 
[56.310803454533783, -77.378150114889451, 54.241934023962244, -24.239451277709605]], [[-81.577621003821022, 
-20.575488788882396, -74.748333868053194, -77.650260041846195], [-36.715889937928893, 46.713294962560639, -32.726652799822858, 
-86.511792039144225], [-36.63521378668375, -6.6216634903382214, -38.109037159628009, 37.802959591625665]]], 
[[[-89.164784930058289, -16.893154586268452, -34.0483761542868, 45.087030820992595], [56.096068746482814, -47.841426397183099, 
62.786778552668522, -53.641715554093629], [68.655982343210781, -29.962711809331694, -37.999496293427541, 65.862456745659273]], 
[[-73.176583009461709, 49.80677997366908, 12.506512460993363, 35.473350491292592], [16.781388716717373, -8.5505718161589215, 
26.19539860058407, 46.761865879346487], [41.621990096276903, -38.514318974070669, -74.226339225637702, 41.167568383147398]]]])
      res=whereZero(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0]], [[0.0, 0.0, 0.0, 0.0], 
[0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0]]], [[[0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0]], [[0.0, 0.0, 
0.0, 0.0], [0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0]]], [[[0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0]], 
[[0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0]]]])+(1.-msk_ref)*numpy.array([[[[0.0, 0.0, 0.0, 0.0], [0.0, 
0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0]], [[0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0]]], [[[0.0, 0.0, 0.0, 
0.0], [0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0]], [[0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0]]], [[[0.0, 
0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0]], [[0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 
0.0]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonZero_float_rank0(self):
      arg=-56.0152542247
      res=whereNonZero(arg)
      ref=1.0
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonZero_array_rank0(self):
      arg=numpy.array(71.1859507003)
      res=whereNonZero(arg)
      ref=numpy.array(1.0)
      if not isinstance(res,float):
         self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
         self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonZero_array_rank1(self):
      arg=numpy.array([-17.3035326538253, -86.593915731827906])
      res=whereNonZero(arg)
      ref=numpy.array([1.0, 1.0])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonZero_array_rank2(self):
      arg=numpy.array([[41.61768486731961, 69.290611719671801, 48.401728321290705, -54.590972112334143, 11.564399558126269], 
[-1.8042524010560896, -92.702095284465159, 45.136484559735237, -72.362244352134866, 24.194710407260914], [-43.108695050778387, 
47.647489079294672, -61.701429681233932, -36.067537829718567, 99.180966430966492], [51.707301922404213, 7.5838051866514746, 
-43.604821883975163, 74.85674310120524, 91.181953108853094]])
      res=whereNonZero(arg)
      ref=numpy.array([[1.0, 1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0, 
1.0]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonZero_array_rank3(self):
      arg=numpy.array([[[21.449928615271091, -17.253570875796783], [23.691826817893485, 72.615672710486564]], 
[[-56.101868173239325, -50.789146263740115], [44.254260393918344, -77.113348468531143]], [[31.628186192244158, 
-29.567091682631514], [-25.637642791911858, 90.000932154214325]], [[-98.909566050888145, -15.524708628153604], 
[40.294638716411356, 19.662663209587222]], [[-8.9424053143323476, 44.033554341712147], [-28.592658918978913, 
57.701429147236496]], [[5.8373347600123964, -15.938309186009377], [47.118492474253401, -17.104874895088756]]])
      res=whereNonZero(arg)
      ref=numpy.array([[[1.0, 1.0], [1.0, 1.0]], [[1.0, 1.0], [1.0, 1.0]], [[1.0, 1.0], [1.0, 1.0]], [[1.0, 1.0], [1.0, 1.0]], 
[[1.0, 1.0], [1.0, 1.0]], [[1.0, 1.0], [1.0, 1.0]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonZero_array_rank4(self):
      arg=numpy.array([[[[36.811388311146146, -37.819440076221554, -70.946877208868955, -53.423749294710589], 
[-27.959990449110549, -13.19600531838357, 30.051887488086265, -62.951332747963093], [44.237322250517849, -49.199225003263706, 
80.483871168859963, -89.985441118504284]], [[44.560573770678758, -54.599849870638707, -88.477359629839185, 28.797401260182085], 
[-77.152989636288908, 20.342246911351296, 75.740577688308235, -19.451449067952353], [55.586372931716994, 46.066327045425965, 
-61.412715276495746, -0.57311906305483262]]], [[[69.342920685200568, 20.761986693929686, 65.355240394684017, 
-43.975994488286531], [-45.602468581331699, -60.702312875128705, -52.741215221192064, -50.152173529149891], 
[63.871292914570887, 50.308104012520829, 13.646656015678801, -2.2486349800344954]], [[9.7609011134064332, 3.3174725979807818, 
96.220499171807035, -20.264739672411267], [-26.826050737340353, 12.940873209136299, 54.903850018848289, -44.753514431245911], 
[-31.296365135620107, -53.852944671155981, 37.681503437560878, 50.394193944723071]]], [[[93.559245006066305, 
21.615439772198215, -46.999800142512726, -99.515382399330576], [-26.590613229769033, 65.009853745193624, -38.710017866404975, 
-34.390372809473206], [71.657928675928162, 85.018672069146191, 72.768640349535559, 66.342192378120103]], [[39.999657304603545, 
-96.748350051606195, 84.903024547152626, -24.425754771404911], [14.239472718681043, 21.23525425634169, 85.086308297188765, 
58.613396024449628], [58.879250368865002, 22.553655766939201, -40.509432689823655, 90.333538469165688]]]])
      res=whereNonZero(arg)
      ref=numpy.array([[[[1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0]], [[1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 
1.0, 1.0], [1.0, 1.0, 1.0, 1.0]]], [[[1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0]], [[1.0, 1.0, 1.0, 1.0], 
[1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0]]], [[[1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0]], [[1.0, 1.0, 
1.0, 1.0], [1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonZero_constData_rank0(self):
      arg=Data(81.267839677,self.functionspace)
      res=whereNonZero(arg)
      ref=Data(1.0,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonZero_constData_rank1(self):
      arg=Data(numpy.array([4.3097880462597686, 38.2746982582035]),self.functionspace)
      res=whereNonZero(arg)
      ref=Data(numpy.array([1.0, 1.0]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonZero_constData_rank2(self):
      arg=Data(numpy.array([[-57.872158775060825, 3.0665803612860714, 23.678752611770307, 27.706914354716304, 
-78.399825738474846], [83.27658948992422, -86.630812378705045, -94.818125169373133, 80.207597472222801, -59.996764104241066], 
[-48.920076163552359, -5.2793956419858716, -50.805323597125707, -2.4177320821246724, 14.787663281902354], [-91.456661324218942, 
-75.915552368352166, 9.029031640098097, -64.522060166953736, 45.606353207854369]]),self.functionspace)
      res=whereNonZero(arg)
      ref=Data(numpy.array([[1.0, 1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 
1.0, 1.0]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonZero_constData_rank3(self):
      arg=Data(numpy.array([[[-65.737279660632169, -69.58794855021317], [49.173019648663285, -75.324917930399081]], 
[[36.734465319459133, 39.356936322031572], [72.229844336186147, 27.313887429173732]], [[-46.346862025625811, 
-28.232476346878286], [67.290787988042212, 28.839469197769318]], [[22.138154835183599, -73.891269486450966], 
[44.435903802312509, 35.208760455185853]], [[17.707928227345704, -82.502312692751232], [-50.185359824329211, 
-75.512768380834984]], [[-11.011396097469046, 3.7914246847852837], [88.484294237998171, 
27.817718891298298]]]),self.functionspace)
      res=whereNonZero(arg)
      ref=Data(numpy.array([[[1.0, 1.0], [1.0, 1.0]], [[1.0, 1.0], [1.0, 1.0]], [[1.0, 1.0], [1.0, 1.0]], [[1.0, 1.0], [1.0, 
1.0]], [[1.0, 1.0], [1.0, 1.0]], [[1.0, 1.0], [1.0, 1.0]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonZero_constData_rank4(self):
      arg=Data(numpy.array([[[[-90.837005944640453, 58.32908469863068, -21.397220189906932, -59.882605473903048], 
[-60.155669460320048, 94.264197917567316, 5.2271187077156895, 7.6994454870920919], [45.007283205913836, -12.641863450474105, 
14.162620707917583, 19.00968469844095]], [[80.350439667173504, 62.976498386779269, -92.277088804255243, -22.02365795339658], 
[-77.586036199115824, 18.456470236740913, 47.615072286184358, 54.538747153369343], [91.74150400219861, 86.412378355903371, 
88.617980688509135, -46.604677919468649]]], [[[-53.574380602108441, -25.238541870158244, -73.591218463895842, 
-37.649627917290609], [-67.751440748047912, 75.941689285252011, -89.393524054767482, 67.428736009005092], [96.911150988908844, 
12.958697435254351, -66.109426273033193, -88.3308257725689]], [[62.378724670258009, -4.8176444206246885, 81.947676307785855, 
-48.326765737807989], [2.7384341578773075, -51.578329102849452, -8.1888405371007593, 40.041650279869941], [1.097138779089164, 
-94.669191888914668, -51.813633997289308, -54.599754268322151]]], [[[83.590503053969371, -74.086642770815402, 
-72.770948998639255, -97.314802633614931], [-86.027885055613837, 33.553797401705197, 31.053141327356713, 67.511970696929382], 
[52.366010839111311, -33.722724485456567, -85.983722002837638, -86.789587160415778]], [[-70.157132541078397, 
-77.562471328811228, 14.699980308694634, -18.99732113106478], [12.774619589743438, -87.237747438149697, -81.789726467975441, 
70.99819494952979], [-49.82446845622561, 29.476931575070466, 31.2380468420277, -80.56982756990962]]]]),self.functionspace)
      res=whereNonZero(arg)
      ref=Data(numpy.array([[[[1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0]], [[1.0, 1.0, 1.0, 1.0], [1.0, 
1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0]]], [[[1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0]], [[1.0, 1.0, 1.0, 
1.0], [1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0]]], [[[1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0]], [[1.0, 
1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonZero_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(55.457381569)+(1.-msk_arg)*(-23.468996897)
      res=whereNonZero(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(1.0)+(1.-msk_ref)*(1.0)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonZero_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([75.794421272424358, 53.341174509357501])+(1.-msk_arg)*numpy.array([62.218317595110733, 
28.772022216936335])
      res=whereNonZero(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([1.0, 1.0])+(1.-msk_ref)*numpy.array([1.0, 1.0])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonZero_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[25.693195010945445, 53.470932335076157, -68.183323445425884, -90.253837566532937, 
97.906247975957115], [58.50459295308795, -44.270817918326053, 44.734053038122738, 46.132305677124208, 15.093865803964562], 
[42.388010237041698, -7.0981379825856266, -2.7347840708205098, -29.594785545579597, -95.196484793137472], [38.36601032797401, 
-31.853113575844233, 22.743621849911833, -4.9889383642481846, 
-51.432166195627829]])+(1.-msk_arg)*numpy.array([[-21.251861279170669, 23.87089448722665, 54.546286256972479, 
-50.596083351973761, 71.290931136958989], [-85.431548064496582, 20.522672789852606, -90.730704002438671, -46.757529093371275, 
16.785961991598612], [22.174454395058632, 69.692756669654159, 87.615316871643131, -78.004806291849491, 3.6808126640922865], 
[-57.786804759801669, -87.557537170166015, 64.774622761137721, 90.823752701099721, -47.516231901181612]])
      res=whereNonZero(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[1.0, 1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 
1.0, 1.0]])+(1.-msk_ref)*numpy.array([[1.0, 1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0, 1.0], [1.0, 
1.0, 1.0, 1.0, 1.0]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonZero_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[86.149242248728058, -35.02751878546529], [40.087555658614548, -77.393517988885691]], 
[[26.117710543523316, 71.702690330409268], [98.666028493959146, 11.651670052091717]], [[-89.606381835492499, 
52.900856899025115], [-28.078601574503764, -95.119493866321875]], [[98.666959366780844, 23.275898081831897], 
[4.3281069846126172, -80.865475942459497]], [[-14.7551225313251, 97.396689501269265], [23.671398524796032, 
21.397118290457513]], [[23.732489190132, -59.259215202504677], [-83.242145340201688, 
-75.536810160429368]]])+(1.-msk_arg)*numpy.array([[[-9.7478095088222148, 11.661323310074323], [4.8146279047090559, 
-52.254294186764461]], [[-25.036812644543332, 73.198082013951563], [-9.0171269289220106, -47.016186545303171]], 
[[-28.848236212560835, 22.014297849998428], [-80.472580936591527, 35.926128036359103]], [[-50.335917685172404, 
10.849423531461483], [-41.037282141150008, 99.521035981305374]], [[3.7060647555777138, -50.51251638563128], 
[-33.000056918825209, 91.690666567142586]], [[-26.856701959084845, -3.2105214924453378], [96.598379912806394, 
-15.447865865744575]]])
      res=whereNonZero(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[1.0, 1.0], [1.0, 1.0]], [[1.0, 1.0], [1.0, 1.0]], [[1.0, 1.0], [1.0, 1.0]], [[1.0, 1.0], [1.0, 
1.0]], [[1.0, 1.0], [1.0, 1.0]], [[1.0, 1.0], [1.0, 1.0]]])+(1.-msk_ref)*numpy.array([[[1.0, 1.0], [1.0, 1.0]], [[1.0, 1.0], 
[1.0, 1.0]], [[1.0, 1.0], [1.0, 1.0]], [[1.0, 1.0], [1.0, 1.0]], [[1.0, 1.0], [1.0, 1.0]], [[1.0, 1.0], [1.0, 1.0]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_whereNonZero_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[-13.011952221782906, 60.146594640260929, -92.58084328474439, -63.668052583858525], 
[-29.604118765606316, -54.921705782621324, 63.355787794763557, 32.048187979650379], [52.690986117833347, -65.333226351220446, 
-80.457709522487477, -98.926180293535211]], [[-7.1796814873670343, 94.388137298771483, 95.299168868419684, 
-54.524320427950833], [-61.062722861775207, 83.2566734157692, 19.473153243447854, -17.683274427867616], [92.214390273881946, 
33.610188459781625, -81.830151098817836, 58.157655737818004]]], [[[63.469751000464782, -17.599047199042445, 78.031842669949526, 
-4.4970488112825819], [-66.54952900441117, 67.261106981137686, 54.23547006750232, 55.602853560488029], [45.038007193518467, 
-70.427944498069365, -69.825782007360743, -65.982920351293558]], [[-95.008913410852742, -39.213243291094948, 
72.194277685731862, 97.046556215652714], [37.536007158729404, 44.977826357229333, -74.984923076853008, -25.784263917813036], 
[16.469242924065597, 55.697269253699091, -14.794604017518907, -31.753199846344259]]], [[[56.500805440617142, 
35.623660964275814, -33.468639069785453, 55.151083545768927], [80.08078661666562, 86.285902320161256, 11.017641963291695, 
-68.900497305225002], [68.192503427277103, -86.515657932050203, 47.994984746708809, -43.370874890365265]], 
[[61.776516875254572, -18.579244394902929, -15.183358149836849, -74.888276150809219], [69.965669844543783, 5.4340404212045286, 
28.363989733719109, 15.311329580643189], [94.76411206365961, -11.043728008539119, -15.649591146513345, 
47.035633382894645]]]])+(1.-msk_arg)*numpy.array([[[[74.385412727423812, -37.538049462402647, 55.913400604665355, 
-70.457307068022459], [79.482613421908439, 83.515322164631527, -43.324348243704634, 14.84345334513965], [68.778457595687001, 
-56.301700989105122, -67.104809294858455, -93.503482897897513]], [[-23.750280106836712, 33.848737195440208, 14.997455364891138, 
97.834390144064486], [-6.4280342178465446, -82.354008114682159, 1.1797643810675993, -84.817435750943289], [4.0640627650950734, 
36.731149482102808, 4.4401253362509721, 50.187621844828357]]], [[[10.746049103776784, -0.50416504196932976, 6.1710864748900605, 
25.710101291704632], [-8.3240472300805806, 0.40060317937977175, -4.6344318693605686, 60.779785582340679], [-45.076412053791138, 
-1.378387041154653, 51.553519112651571, 26.371530000106787]], [[-46.325656386879729, 38.949682993329958, -96.678227888724777, 
-71.397182995030661], [-39.275033452116404, -20.946676674594329, -1.9183420721467428, -16.206004000684842], [9.166499552764293, 
55.367654830204089, 55.869168851337804, 31.8283590864053]]], [[[70.064271798572975, 25.227460810734044, -49.584790116912771, 
6.4927871739775185], [-14.735101219279272, -32.448581566238843, 92.390606133049715, -71.298582637949863], [-39.675482065031062, 
13.907314019034956, -47.48451389090782, -93.93888851018022]], [[-92.274017822855086, -10.260862968529636, 14.31392023943387, 
32.589318378815591], [84.922863364253459, 71.725715754215599, 47.377427817862753, -55.639887444292157], [63.461117774674562, 
89.548143331715664, 33.157691946888747, -56.134655663723642]]]])
      res=whereNonZero(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0]], [[1.0, 1.0, 1.0, 1.0], 
[1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0]]], [[[1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0]], [[1.0, 1.0, 
1.0, 1.0], [1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0]]], [[[1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0]], 
[[1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0]]]])+(1.-msk_ref)*numpy.array([[[[1.0, 1.0, 1.0, 1.0], [1.0, 
1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0]], [[1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0]]], [[[1.0, 1.0, 1.0, 
1.0], [1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0]], [[1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0]]], [[[1.0, 
1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0]], [[1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 
1.0]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sin_float_rank0(self):
      arg=26.6185049196
      res=sin(arg)
      ref=0.99638690326
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sin_array_rank0(self):
      arg=numpy.array(-19.1395275546)
      res=sin(arg)
      ref=numpy.array(-0.285925042532)
      if not isinstance(res,float):
         self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
         self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sin_array_rank1(self):
      arg=numpy.array([9.7813278790190594, -15.410822535775822])
      res=sin(arg)
      ref=numpy.array([-0.34904322255989967, -0.29278743949712893])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sin_array_rank2(self):
      arg=numpy.array([[7.5709622007726978, -96.010110171854322, 64.464875837548789, 18.074098255478049, -79.887891912588785], 
[-61.137259666889591, 24.376915700677898, 1.6520396981142653, 73.282870077726415, 37.390208438590236], [-18.018770402944398, 
-20.185849422983694, 16.072369015111192, 52.807262209935885, 19.287397979810578], [-77.896963892166468, 49.777355373857489, 
61.194119343070525, 50.551462513063115, 15.922964094659335]])
      res=sin(arg)
      ref=numpy.array([[0.9602166212011517, -0.98171332509549469, 0.99806455979119224, -0.70004296831489288, 
0.9753000889147212], [0.99234692329352858, -0.68588964365820848, 0.99670157217596123, -0.85535672018082665, 
-0.30401412851360171], [0.73846127084748792, -0.97262998476338458, -0.3563941257877799, 0.56448804411955977, 
0.42398607347515932], [-0.59948094483604641, -0.46897252800353989, -0.99776052848304986, 0.28209783791975507, 
-0.21334823597722449]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sin_array_rank3(self):
      arg=numpy.array([[[-58.576474897348007, 54.809262272793802], [-92.545446097205968, 85.575190211710776]], 
[[38.681288743038209, -11.603575878192785], [-95.597383200249268, -13.02556705264513]], [[61.706270910680416, 
85.490591964337312], [71.624027820036616, -25.863896412300605]], [[25.476947591593444, 70.435340495814557], 
[-30.073684565633215, 67.490356578240863]], [[4.9713423686298341, 91.911859760915974], [30.367874697055328, 
-84.144554970831109]], [[98.391390622536989, 31.732748787807651], [-14.934490963501432, -59.311188442325168]]])
      res=sin(arg)
      ref=numpy.array([[[-0.89737553935745262, -0.98581911829326108], [0.99136145080590388, -0.68323847477655364]], 
[[0.83170798407592028, 0.82079120409534756], [-0.97563646528573944, -0.44322793023104901]], [[-0.90251845539243469, 
-0.61909542454627942], [0.59124622666754811, -0.66773000309600883]], [[0.33744969840141792, 0.96879003389058782], 
[0.97399495168350392, -0.99854852911307401]], [[-0.96665851232305666, -0.72129680977545241], [-0.86645223153963946, 
-0.62758444363043242]], [[-0.84255979532259895, 0.31154854639750068], [-0.698623841417506, -0.37005848663264262]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sin_array_rank4(self):
      arg=numpy.array([[[[-50.615462747878006, -66.973812639371985, -77.708206606928769, -36.431802774059221], 
[84.078764335768454, -86.057175975235566, -4.2507864983636097, 32.605074036304671], [-35.47698277417679, -32.850292882440925, 
19.746311000390619, 96.575011359809082]], [[37.550695304429951, -85.192245059759713, -19.585580977316283, 4.9299086609782705], 
[70.955489529502842, 64.084022545666187, -35.537247087129771, -47.717715879912468], [3.8361995697958662, 95.532145761226531, 
-51.581690563318517, -23.90597768756011]]], [[[-85.885115626404513, 27.544257154700233, 57.776442658393961, 
-22.764439696740865], [9.7442723032797431, -47.221529831578344, -24.523464073641321, 22.843310070989517], [67.821301169614003, 
19.731335445973343, -74.187754964775991, -69.863316195387767]], [[64.051377288481888, 13.397049375815271, 7.2977357293966065, 
-53.46018591213759], [18.669622257829204, 64.3833982650888, 43.248464847568329, 55.630477540818163], [69.55974394554579, 
-47.17349597543474, 13.955474294777375, 57.58228196835239]]], [[[-67.48312543703419, -79.807110080995614, 29.982433513595652, 
-41.775471818361368], [-36.948208936105289, -81.305101266586661, -85.068091915731998, -25.025548528744793], 
[-18.039651899818338, 6.4567829828828138, 78.793240634732797, 32.996039522485205]], [[-99.940020762322376, 68.326160737955007, 
-61.082363029407126, 54.949813891014884], [80.038229033350177, -33.552375886074117, -31.608129371983892, 64.177422432822794], 
[32.303265514373805, -76.156866511153297, 64.174936832666873, 11.368893661666618]]]])
      res=sin(arg)
      ref=numpy.array([[[[-0.34287929276723833, 0.84166917263443375, -0.73901678422019978, 0.95430012717376367], 
[0.67741096998138572, 0.943875804457038, 0.89533992846215704, 0.92805179004897842], [0.79527653438049994, -0.99070785666439132, 
0.78130571381342773, 0.72728711253181411]], [[-0.1478722649927692, 0.3609099416305746, -0.67134722109833267, 
-0.97643572609341589], [0.96386290831076604, 0.94966646919227582, 0.83034577260054665, 0.55953618516020254], 
[-0.64008348060134623, 0.95925856748150318, -0.96776708303022096, 0.94140212104966459]]], [[[0.8733869978866744, 
0.66692680849003383, 0.94174275492293247, 0.69849419559794712], [-0.31408653242230877, 0.097484958579927491, 
0.57227483208076302, -0.75270519985308859], [-0.96186401066781391, 0.77187148397831629, 0.93578136651060684, 
-0.68037764724345384]], [[0.93893574912137301, 0.73838928073958066, 0.84924325972879255, 0.053085835853353053], 
[-0.17896430849100536, 0.99981470265224548, -0.66972046488018222, -0.79450391882013482], [0.43019210289246596, 
0.04958582917574688, 0.98353924979282858, 0.85915404611622637]]], [[[0.9981329609384284, 0.95429554635975633, 
-0.99058870054119763, 0.80446088766755564], [0.68229912906486578, 0.36748906525527336, 0.24264389693973754, 
0.10698753894470732], [0.72422099416087193, 0.17272706090204004, -0.25072034421713113, 0.99995660023908339]], 
[[0.55714530958135622, -0.70956285806092356, 0.98407671685665454, -0.99960641285478424], [-0.99738144842386833, 
-0.84423878131865737, -0.19102162937524009, 0.97474344445505001], [0.77539411931850055, -0.68793708162803491, 
0.97418533082369707, -0.93112187460985318]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sin_constData_rank0(self):
      arg=Data(35.9258166114,self.functionspace)
      res=sin(arg)
      ref=Data(-0.979567062569,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sin_constData_rank1(self):
      arg=Data(numpy.array([11.05411721386011, 92.760749350185449]),self.functionspace)
      res=sin(arg)
      ref=Data(numpy.array([-0.99828685225980784, -0.99649367379012466]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sin_constData_rank2(self):
      arg=Data(numpy.array([[86.368381659051408, -41.695780796282492, -58.183479792434831, -70.875178760715073, 
-20.678382721502217], [17.695341117085974, 2.740439328337672, 62.183987328150124, -22.389692204658516, 25.527065283434851], 
[-19.879704086428717, 97.930429727602103, 16.055583599572529, -16.208182451420882, 54.913306337155262], [-69.833724282794194, 
31.117621263412161, 97.441216860315848, 31.002527925222012, -33.156619640633039]]),self.functionspace)
      res=sin(arg)
      ref=Data(numpy.array([[-0.99967702286155391, 0.75462098315788761, -0.99795169465255495, -0.98212790490620516, 
-0.96689443052183988], [-0.91447752242728786, 0.39048036598212266, -0.60348598179363533, 0.3880765237690833, 
0.38418420577157436], [-0.85737525785576585, -0.5150427020633046, -0.34066144314268004, 0.47961787867933503, 
-0.99791639785660491], [-0.65839615788937589, -0.2939007480209867, -0.051821376961544853, -0.40172395652639609, 
-0.98560222500947936]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sin_constData_rank3(self):
      arg=Data(numpy.array([[[-9.383136985495085, 51.313988740117281], [-84.286741947414299, -12.349769403836447]], 
[[-58.98719875193035, -69.772217254427616], [25.488307874218165, 60.443485945243793]], [[80.648598553510482, 
92.976864439316103], [4.2817739078012664, 91.184638461427113]], [[-76.55157819489709, -16.719487135234104], 
[13.038521969338817, -14.062881841606639]], [[-60.572162238841166, -8.277832942135376], [90.508007133317363, 
9.9045178477461775]], [[43.319148489339796, -36.82759950317913], [23.772826290958989, 23.859083701318681]]]),self.functionspace)
      res=sin(arg)
      ref=Data(numpy.array([[[-0.041628942278232502, 0.86667902768507166], [-0.5109243145942064, 0.21491149901715179]], 
[[-0.64655635572868242, -0.61088574863445955], [0.34812162256726858, -0.68399529227053291]], [[-0.85874246895135953, 
-0.9553716024683695], [-0.90870917195229073, -0.078371058632619367]], [[-0.91412907684432942, 0.84764134621368814], 
[0.45480331572319638, -0.99724213056634026]], [[0.77194923151153583, -0.91151176131243561], [0.56313928014945791, 
-0.4615484409914466]], [[-0.6156012262898336, 0.76530325949300315], [-0.97784680058787665, 
-0.95617812015239778]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sin_constData_rank4(self):
      arg=Data(numpy.array([[[[-82.266761499442822, 10.258465683133593, 35.258711312343792, 83.260426498166652], 
[44.545759248512837, 78.847333090045225, -91.458863199213994, 88.135667192818687], [55.303615717334651, 57.343360606367497, 
72.386684244462259, -77.851237894363791]], [[7.6357829095114624, 58.043761182779804, 32.459127943514119, -46.057054845382581], 
[-62.173352222977194, -91.055460532286546, 1.908599997174349, 50.256424878237198], [80.737361916893349, -59.097170441109384, 
-11.392890659087485, -17.113974908227121]]], [[[-69.94898633515831, 41.553507654384049, -2.0229189980548909, 
11.130327214085185], [27.241980003875298, -92.430057302595131, -59.821389090442324, -10.836164150097432], [-97.960459786173644, 
77.056043492990739, 18.592655235604354, -99.221726011137193]], [[-62.79302507786295, -60.628776679562854, -92.135333094895984, 
-7.7874612902526792], [-59.349180405626001, 98.544187052373076, 25.967954318471186, 26.092417193137578], [15.354438427681245, 
-0.13204595002771669, 57.604938537040567, -78.12077453253174]]], [[[26.134134711410127, -21.654760380881413, 
91.027816009977528, -70.493423376080287], [50.18077929890714, -45.818068992158459, 17.514556104968705, -87.948924607482581], 
[-2.9358337467884752, 52.58088647758106, -88.281586106394585, -37.905045294843575]], [[9.2050228670356233, -53.760066650396411, 
-36.646477787406951, -50.989346886330431], [46.181223758729573, -56.454456456188559, 0.97893723365712049, 67.524717727099727], 
[76.315867619424694, -64.23482558405928, -9.4212307356945644, 94.459095295520825]]]]),self.functionspace)
      res=sin(arg)
      ref=Data(numpy.array([[[[-0.55249323661859384, -0.74041510223459006, -0.64512901510731857, 0.99996620630617827], 
[0.53411628907438291, -0.30269281624662714, 0.34541056811800569, 0.17023967780324173], [-0.94741280897254443, 
0.71364847302087853, -0.12968690532551203, -0.63544017109361273]], [[0.97628895760467316, 0.99713590303276967, 
0.86402042247071265, -0.87567650001557451], [0.61193184565404779, -0.050704669999494079, 0.94348483865887567, 
-0.00905745535309368], [-0.8099383976922776, -0.55892595044018734, 0.92210273022629663, 0.98645369822246709]]], 
[[[-0.74058999329446951, -0.65395701522909533, -0.89952178562120422, -0.99093455465566593], [0.8585084340096536, 
0.96966836773299581, 0.13075320819151812, 0.98732108746426506], [0.54054731770475606, 0.99621584600900559, 
-0.25408417591070465, 0.9659884230955289]], [[0.038818238402855587, 0.80668212053066524, 0.85685909391293247, 
-0.997788337658438], [-0.33450508434738102, -0.91472013442829736, 0.74143951598281521, 0.81900568441531063], 
[0.34620680919904595, -0.13166255596987642, 0.87052630534488828, -0.40688535076780291]]], [[[0.84222306949426196, 
-0.33007987762886654, 0.078290742973310345, -0.98154597956213518], [-0.084601909290267188, -0.96509891341258691, 
-0.97232857103810033, 0.015669051785311802], [-0.20431011725232823, 0.73535381865796079, -0.31170965722815802, 
-0.20448097728983067]], [[0.21799061313810039, 0.34570643936958911, 0.86873084472110407, -0.66228504096565033], 
[0.8091276300491409, 0.094072003936346266, 0.82990491677663747, -0.99980940641969562], [0.79417206042993815, 
-0.98595060503156651, -0.0035472176358126886, 0.209746502418206]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sin_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(64.9800250937)+(1.-msk_arg)*(-13.9362171446)
      res=sin(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(0.837897949793)+(1.-msk_ref)*(-0.979877440358)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sin_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([38.770149608840256, -60.012793688818753])+(1.-msk_arg)*numpy.array([-99.060327275798542, 
-82.301745794852508])
      res=sin(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.87769828831808394, 0.31697021887466476])+(1.-msk_ref)*numpy.array([0.99498831041282509, 
-0.58130924307676168])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sin_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[1.8437603079485996, -55.085629630896008, 0.76077789325732681, 75.303888016859787, 
44.134994661850783], [79.846819977879846, 55.190840097906744, 66.852695154059461, -2.7788078944115426, -33.754582563675072], 
[79.437692683252919, 25.719174268770573, 22.693852460282187, 73.984833469926286, 34.325902065505829], [36.438188095997873, 
-58.458802817368706, 35.872691199405381, -97.828193056307839, 
17.582825707314015]])+(1.-msk_arg)*numpy.array([[5.857972204383401, 68.980392763644375, -31.110427811729835, 
60.164069309555259, 59.562550189394926], [-79.365962953462713, -74.725764079778173, 85.216081299963463, -56.646488172185805, 
46.373883249354861], [-96.708272172455125, -53.912331939216031, 50.478527343574285, -97.278578429153086, -19.251159449913018], 
[89.592226731549431, -48.83081996413825, 86.246650594081075, -29.89076793492336, -45.154211051500191]])
      res=sin(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.96297607712265498, 0.99419970184494533, 0.68948508165976352, -0.094195812584939725, 
0.15210480698702625], [-0.96540800347437894, -0.97740775938701152, -0.77026043446389958, -0.35487911125423122, 
-0.719398961710221], [-0.78200505820620614, 0.55339355773313237, -0.64628337521816626, -0.98763721556473016, 
0.22955176403585492], [-0.95237243281291895, -0.94297498574434913, -0.96750562604917711, 0.42487228151059697, 
-0.954126976601816]])+(1.-msk_ref)*numpy.array([[-0.4125149172810601, -0.13423914232300688, 0.30076884437212431, 
-0.45627887875349671, 0.12736335490838124], [0.73532534193785637, 0.62291199948324294, -0.38303500671569563, 
-0.097664477681636913, 0.68164355585755676], [-0.62964804242596883, 0.48403219384114982, 0.21143692166914244, 
-0.11056730045170879, -0.39089478845779391], [0.99838526337700328, 0.9907480912937221, -0.98919334458602826, 
0.99895877973110281, -0.92149562271137819]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sin_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[-38.413084764990302, -51.052907554914874], [-70.505293089549951, 90.313652142673448]], 
[[14.910063396779577, 5.5459023128808838], [-17.734013723516611, -0.2879278829081926]], [[-22.114710637688262, 
46.626248466538811], [67.532808824706478, -59.397222523445059]], [[92.618471591726234, 73.621613593824065], 
[88.375553175968605, 5.7283336450617099]], [[53.683923353330755, -77.160809717487297], [-34.501807709057687, 
-33.073637412461139]], [[27.624848265822479, 15.259019682038172], [0.55157605306179391, 
-71.932924310466888]]])+(1.-msk_arg)*numpy.array([[[-34.335248638101717, 68.752336646733312], [-43.346716752254302, 
-8.6223337031949683]], [[-35.640511588324259, 29.644108535921475], [64.496990192063578, 38.316350896738442]], 
[[-42.749910167259664, 9.2631745284455604], [73.587108766995641, -96.934463397218067]], [[-74.560330219656223, 
2.8651995440780524], [65.274936254565944, -71.732320070952852]], [[66.378370922669262, -31.104012988746703], 
[38.402067771718777, 88.070193006735252]], [[-3.3040251330990884, -30.446998125502006], [53.53952042332989, 
49.143306316308468]]])
      res=sin(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.65484153130973721, -0.70853858647854739], [-0.98374658316761376, 0.71213510355429321]], 
[[0.71589133616462552, -0.67227900197685508], [0.89814931915579443, -0.28396601912126207]], [[0.1232478868102998, 
0.47735428568469945], [-0.99993464136673393, -0.28886194517886638]], [[-0.99828867944095934, -0.97889500442468336], 
[0.3994885492462833, -0.5268172219749746]], [[-0.27332526710635363, -0.98166466097630112], [-0.055682665640661121, 
-0.99622530762962569]], [[0.60477683330883625, 0.43401404656957493], [0.5240302030950843, 
-0.31808294342294718]]])+(1.-msk_ref)*numpy.array([[[-0.22044488395248701, -0.35480148713588266], [0.59364467642489349, 
-0.71905687697203813]], [[0.88336425850768363, -0.97986309124469417], [0.99555320691853921, 0.57878587408359439]], 
[[0.94328393594361259, 0.16090095326994339], [-0.97126219531557589, -0.43938044812272531]], [[0.74323543603717557, 
0.27288743671671112], [0.6430769524396025, -0.50061664432606201]], [[-0.39395000791253521, 0.30688041201024424], 
[0.64647568844428205, 0.10540255889338963]], [[0.16171914304468005, 0.82427947074375341], [-0.13205843112469617, 
-0.90104641666580576]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sin_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[-68.525560759670157, 72.014851464858708, -96.064109131678975, 77.226473627484438], 
[10.661045262101496, 17.374475984434937, -70.970539568996642, -86.5663539792791], [42.328708688260292, -49.606342449617813, 
-41.332552022078197, 5.781826685010131]], [[32.656794594741456, -45.69741225061923, -75.977367014659336, 65.992637114312402], 
[-24.968933149299261, -19.170628852284295, -82.992054567050218, -69.278260336258882], [45.308496224570149, 58.254105930225364, 
-52.053961481820508, -99.633451029223679]]], [[[57.389652364413365, -6.4497448547971459, 67.302050277783934, 
-64.487213635922004], [2.7850477397074656, -37.14171412837635, -45.831747383644263, -81.22619991798237], [28.555595155095858, 
6.3743955466930515, -9.7644609522321133, -2.1997743618038328]], [[35.568247785419004, -31.599877667329878, 82.028553557923146, 
-78.86370276205146], [-99.731804643220002, 69.776357240610565, -12.167123991837968, 33.790715856665315], [-10.562353591062944, 
-16.629309453963131, -77.001863431944969, 52.718875999701169]]], [[[-98.376736649980899, 29.119393841437898, 
75.366820567618731, -66.514039609932439], [-88.471818477178289, 47.880968729353668, 65.085879417509886, 84.296285545557623], 
[-86.151465152354675, 9.1305089222339433, 3.0045836139087356, -81.15026537989543]], [[19.458035685379343, -16.79612796438164, 
-62.697038154714392, 82.150007929305815], [51.125444489928753, -39.220742821773079, -31.48429119581813, -40.990991828400048], 
[-20.250208920559061, 52.655819512739868, 26.546619397350256, 
-77.359673716677449]]]])+(1.-msk_arg)*numpy.array([[[[-18.615245719283436, 96.201699795366437, -82.66494125542863, 
33.07492631817621], [82.443853712892633, -34.736058741884463, -95.351325060809899, 59.914700106027027], [-31.967579463417351, 
45.580214173517021, -85.610494042354162, 49.498206680862722]], [[47.852867738343264, -35.372573605242792, 2.511993838319043, 
94.155389339549941], [-1.38794622700091, 85.930835412964996, -85.707091771314651, -46.64999398418388], [57.879302040774746, 
-50.458699813236429, 25.001254046544503, -59.801937388884156]]], [[[99.915640912459565, 42.356808810363106, 
-15.646740182074836, 72.140752053033623], [22.725076309780334, -0.022312650774281906, -7.1186223183705266, 
-26.368838179359486], [66.038559283219001, -17.561521846617723, 4.9702769173480306, -59.711097000633437]], 
[[-78.084063698630573, -45.932568204284749, -45.100652927345664, 80.338502970189182], [50.249408685318571, 27.311139139809868, 
9.9265850486579552, -56.606805234548688], [7.2672257036688848, 49.011928816065478, -54.764618596546889, 70.942797505173303]]], 
[[[61.689516820950246, -58.515211403448731, 53.79986661743186, -81.589305129450977], [74.672959560216214, 61.905849543205051, 
-27.673934374496582, -43.264203000639313], [66.074534000186645, -29.171690529749995, 62.854710104264711, 28.934380662657873]], 
[[72.033136619176673, 0.8371610199688746, -21.89771655702684, 35.848349672772173], [14.782846328746871, 69.671698166617858, 
-71.840672845829047, -65.869180747022639], [-87.952973126148649, -58.34136644843764, 58.528431489251943, 
-45.940658422238933]]]])
      res=sin(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.55592687965297938, 0.23943081082144563, -0.97000785672325052, 0.96704147053511302], 
[-0.94456504679141606, -0.99542268261330857, -0.95974459076326801, 0.98514911686367079], [-0.99657468842237251, 
0.61243723818846874, 0.47225521641568707, -0.48061739888675808]], [[0.94606558536471286, -0.98960410832751255, 
-0.54730716207657015, -0.01919021088691918], [0.16307648225519095, -0.3155848432241144, -0.96635121146850689, 
-0.16249817830586416], [0.97023493524697935, 0.99094947267277556, -0.97640053267971305, 0.78177909697867587]]], 
[[[0.74529994268538491, -0.16579049701658166, -0.97081465164637015, -0.99642657514017952], [0.3490385329297237, 
0.52897960232366481, -0.96142656801469195, 0.4396501084403685], [-0.277567581500057, 0.091083824433171562, 0.33318821411150523, 
-0.80862917156767822]], [[-0.84721913137013771, -0.18291546159728064, 0.34021409508202199, 0.31825330538480789], 
[0.71677079367561358, 0.61415820889112394, 0.38872432525224898, 0.69383676427292507], [0.90761842336617005, 
0.79641644551081814, -0.99946070339480197, 0.63514722255050504]]], [[[0.83457694337317756, -0.74801090758055289, 
-0.031397957395935407, 0.51464528088708705], [-0.4857527846740865, -0.68680120896912167, 0.77553765270553499, 
0.50269724950691053], [0.97078081455322718, 0.29004038701884971, 0.13658079814321689, 0.5065197281697561]], 
[[0.57162074004370722, 0.88577662065266316, 0.1344069097928022, 0.45163669702654402], [0.75781779092430412, 
-0.99879162771218244, -0.06831141940693769, 0.1497222316080222], [-0.98556050825669073, 0.68255692976113103, 
0.98771358789813224, -0.92466033167086703]]]])+(1.-msk_ref)*numpy.array([[[[0.23217209197051894, 0.92750140233150447, 
-0.83245973502718251, 0.99611259648561323], [0.69069140539580076, 0.17759253084678614, -0.89280995913599226, 
-0.22256013812798214], [-0.52409567549761338, 0.99963225645305609, 0.70858607515885852, -0.69417690883737659]], 
[[-0.66610766977352276, 0.72776291956861339, 0.58882054224263247, -0.092258884266531088], [-0.98332944541626977, 
-0.89473334399200433, 0.77333845218638109, -0.45635622904471135], [0.97129944223643883, -0.19201736907284048, 
-0.13110863189212127, 0.11144498092658177]]], [[[-0.57722312856525515, -0.99850476462170312, -0.061184846303749152, 
0.11561981763652862], [-0.66979133087446185, -0.022310799411864708, -0.74158975309064767, -0.9445091028081305], 
[-0.065067556442604341, 0.96028840893223799, -0.96693079262155868, 0.020835074713991906]], [[-0.44013825745236268, 
-0.92885934187433605, -0.89938286674246037, -0.97414519119251186], [-0.016073079973959268, 0.82101963337083039, 
-0.4810106237620847, -0.058104725027164135], [0.83274117746688869, -0.95009916753271018, 0.97734765469722507, 
0.9671663248687612]]], [[[-0.90960662105081236, -0.92270873486286653, -0.38276882002523788, 0.091973697724973463], 
[-0.6633331176425904, -0.79922431918451375, -0.56497215635127529, 0.65795064607285259], [-0.10091619527458537, 
0.78168104056774534, 0.022855042267844725, -0.61315380741542369]], [[0.22163847286200966, 0.7427452080440865, 
-0.093296140968069, -0.96106487708897625], [0.7986911357951999, 0.5283532279670673, -0.40406659748310347, 
-0.10407616703187229], [0.011620912790781377, -0.97548053305667226, 0.9175319386483336, -0.92583209645347753]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cos_float_rank0(self):
      arg=12.8022737001
      res=cos(arg)
      ref=0.972303667601
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cos_array_rank0(self):
      arg=numpy.array(81.9030902279)
      res=cos(arg)
      ref=numpy.array(0.975529175116)
      if not isinstance(res,float):
         self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
         self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cos_array_rank1(self):
      arg=numpy.array([-3.1659404550478598, -78.770544011004517])
      res=cos(arg)
      ref=numpy.array([-0.9997036069247125, -0.97350024454745809])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cos_array_rank2(self):
      arg=numpy.array([[-30.794187778759778, 0.85080886747965678, 72.306186238999288, 6.9748772974918012, 79.516672956944063], 
[86.786469847342659, 3.7028708915669171, 80.37984411727453, 88.481994155573062, 44.230357294268288], [-58.302217071119557, 
-55.178139958451936, 87.593515438312323, -24.785367985815171, 98.359589229615466], [-31.854839257078595, -9.0018909167008729, 
4.7136540476817572, 16.002129052312014, -11.789662012845042]])
      res=cos(arg)
      ref=numpy.array([[0.81286694784924007, 0.65937524376097301, -0.99877239201012524, 0.77016789679198738, 
-0.55963036169023139], [0.38265829523948425, -0.84657543666548907, 0.26599065257267407, 0.8691082050444322, 
0.96939052643352663], [-0.18173739330410366, 0.1989324917250061, 0.93193666987235591, 0.94027018029592546, 
-0.56512054377873722], [0.90521424329995193, -0.91190791424406981, 0.0012650669596327702, -0.95704434909532998, 
0.7132244558805807]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cos_array_rank3(self):
      arg=numpy.array([[[50.387970299672958, 29.283929509977696], [-66.772009838509376, 60.429920999281819]], 
[[-8.5766208394698822, -60.883217299800329], [95.242982403475651, -86.115091741382543]], [[-52.324500845707703, 
25.632872154700763], [-0.58350729326133433, -24.097894724429821]], [[-56.715089150416162, 84.206037318419675], 
[-36.690358592980928, -19.579806089690194]], [[-13.309049137718802, -73.64720223221002], [-44.056513958126729, 
-89.338449548164448]], [[39.590677420492995, 71.64654237642219], [-60.51045883245412, -14.000945362814974]]])
      res=cos(arg)
      ref=numpy.array([[[0.99250773865265618, -0.53220311324739533], [-0.69773603296499365, -0.73869738226508386]], 
[[-0.6613665430403447, -0.36891317441110316], [0.54433278152098952, -0.27511203517551286]], [[-0.46905641157801509, 
0.87751978510932405], [0.83453542849264828, 0.51065791183165576]], [[0.9861838930452187, -0.81563853576330847], 
[0.53291609526312445, 0.74500754948473746]], [[0.73665981590171459, -0.17925105881386577], [0.99724719663050987, 
0.19567045990235213]], [[-0.31529666740670903, -0.81959722640487132], [-0.68207612410963148, 0.13580067388724723]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cos_array_rank4(self):
      arg=numpy.array([[[[8.8585796551860625, 56.230722191967914, 32.019025852785489, -76.240723304025721], 
[-47.749040562512789, -98.339434837051783, 70.859766363200237, -98.311574089761322], [-52.324580920881303, 96.326373563570314, 
69.773493973765369, 34.547548410615576]], [[-81.713903361911818, 96.89515505261167, 32.14011355478587, 40.833117226994347], 
[88.359411641871901, 99.260851582320299, 32.291978752467458, 34.648053452706648], [62.889324490972825, -83.646300835071457, 
22.610216303248336, 66.616016567138644]]], [[[-17.197294613142319, -3.6842582419897667, 78.922021411276916, 
35.992025233043194], [-92.506302343396186, -12.80399377292683, -18.033493882610045, 72.067566702745893], [21.035957164282863, 
70.855105287036082, -8.5253976863984349, 80.750151682605122]], [[-8.8417506220947359, 61.36510079425446, -25.644929133852216, 
40.188275611212987], [-25.669229350991557, -16.113815339165384, 43.92335634612391, 90.772865066514441], [26.034572422228308, 
-2.8711185835616675, -30.821975852409707, 99.67303892757721]]], [[[-12.673601706489663, 2.304354282387223, -44.91890224201309, 
-51.245125847269122], [-36.174794536889479, -6.8764825759013206, -68.713937504645145, -97.121693324391245], 
[-92.116548149907402, -51.150678601242895, -85.862075707093993, -26.492481960906773]], [[-9.3549875850648334, 
91.091860017849058, 19.150635549346703, -59.957632280501507], [-30.353193336019046, -97.973159096626318, 12.100803192254858, 
73.753754526126613], [-93.775512934495197, 2.0310829139235693, 31.781795131664779, 88.243508559195647]]]])
      res=cos(arg)
      ref=numpy.array([[[[-0.84394640242754804, 0.94987966856902739, 0.82358164777126441, 0.66559941934698164], 
[-0.81087490180541133, -0.58163218822498852, -0.173056013152627, -0.60406691123278633], [-0.46912712992167072, 
-0.48625395737776261, 0.79093818984192754, -0.99995029219615772]], [[0.99947210445754142, -0.8803402908496113, 
0.74903829114045251, -0.99997121680753465], [0.92306683750653218, 0.29617262700437857, 0.64018888147855402, 
-0.99590457207030214], [0.99834897250432109, -0.38397319491648929, -0.81441978564223916, -0.80055781398499271]]], 
[[[-0.081374903851665389, -0.85633516088809258, -0.92784447233066147, -0.13586874199713481], [-0.16985343142213913, 
0.97190021204609445, 0.68509513388229792, -0.98218051486600799], [-0.57745249597890147, -0.16846340013067879, 
-0.62209529663370222, 0.59682562466322531]], [[-0.83479976539592393, 0.10385643517733804, 0.8716743341110903, 
-0.7946115232954305], [0.85950897121254921, -0.91876633151615572, 0.99826349361266198, -0.94496069123950321], 
[0.62017450370660798, -0.96364433869084776, 0.82873619390006759, 0.65400783890141501]]], [[[0.99425625331526324, 
-0.66951670587906009, 0.59252620664860245, 0.55731867514789302], [0.046462287647983845, 0.82910169364382147, 
0.92063173530463938, -0.96438740023068792], [-0.53155481457863152, 0.6331376906760543, -0.50701857453496257, 
0.20949218767157707]], [[-0.9975656400559022, -0.99989737120426636, 0.95501688058642786, -0.96446857407183784], 
[0.4864859369011077, -0.83438138098161108, 0.89356697671985796, -0.073606205812436112], [0.89053945351718922, 
-0.44420488584319706, 0.93381336417866412, 0.96135492253562493]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cos_constData_rank0(self):
      arg=Data(-61.9498622145,self.functionspace)
      res=cos(arg)
      ref=Data(0.635615451425,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cos_constData_rank1(self):
      arg=Data(numpy.array([-89.363505318102554, 90.746151091366158]),self.functionspace)
      res=cos(arg)
      ref=Data(numpy.array([0.17104017933113105, -0.93588418955761465]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cos_constData_rank2(self):
      arg=Data(numpy.array([[-87.716817023452975, -13.508318618768172, -40.837878999841813, -94.821047816906486, 
20.238625532480697], [10.664222985785372, 41.421464805806608, -59.947824248490633, -78.555304048035651, -8.6157242378561278], 
[12.921775769760472, -50.786928458875181, 14.281561344574072, -62.970825984819953, 69.030431174425786], [33.649169420684956, 
31.961869743753539, 95.025401181456971, -69.38900568783086, 28.121974134384544]]),self.functionspace)
      res=cos(arg)
      ref=Data(numpy.array([[0.96945993825961785, 0.58821378024571513, -0.99999600828650015, 0.84013285161999507, 
0.1807281241902764], [-0.32532112085362436, -0.83604574058067016, -0.96701340164638327, -0.99988006784331296, 
-0.69018350050446609], [0.93750558375153104, 0.8670997832638041, -0.1438931618585739, 0.9903587968098958, 0.99642294505936524], 
[-0.61504777690368062, 0.85463793445943459, 0.71258422387118503, 0.96270510850591873, 
-0.98841568900027077]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cos_constData_rank3(self):
      arg=Data(numpy.array([[[-86.264864472603193, 7.0899163167813697], [62.034724520374027, -53.737765416027081]], 
[[21.737171707872307, -39.991208563910121], [41.050321790962244, -40.100213891371972]], [[80.79158486625019, 
9.5468715078936839], [20.045951470114602, -73.62813814925822]], [[90.373079869171505, 55.136085676347903], 
[-60.773850688066375, -19.49023282382187]], [[-16.279695430573497, 70.631083378881186], [87.762984375593931, 
-70.166718723420061]], [[-44.704810449858392, 60.228705435547226], [-72.118752328137177, 
-48.200833401477624]]]),self.functionspace)
      res=cos(arg)
      ref=Data(numpy.array([[[-0.1285765694430033, 0.69186243246368828], [0.69876368539675648, -0.9458184296187333]], 
[[-0.96792086970536395, -0.6603617578658344], [-0.97811062189215869, -0.73813766065052]], [[0.62954868271257325, 
-0.99255583718030116], [0.36571488292688958, -0.1979726592446612]], [[-0.74309878813858687, 0.15755500107466464], 
[-0.46815886672734131, 0.80169133119863356]], [[-0.84096498214967286, 0.054723976242368536], [0.97974546517631589, 
0.49611277631934148]], [[0.7501461251974979, -0.85850711663576951], [-0.99050978024869984, 
-0.47402177642967447]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cos_constData_rank4(self):
      arg=Data(numpy.array([[[[10.86769150637592, -11.766101946953086, -40.632054317141275, -44.331198232316773], 
[63.040749529539085, -18.188400617211855, -49.618899482969979, 16.74749768820601], [-99.222919092701332, 50.811854793623525, 
-98.706022950048023, -76.194952985075304]], [[-92.353764397766298, 0.88734782067865581, -6.0019342520621279, 
12.072469458988607], [86.40044972063265, 93.841407292060126, 62.04309009236411, -25.676293779358559], [-53.006995335453475, 
33.517916229155048, 62.667462961355739, -61.156034212813907]]], [[[-91.6272838267937, 24.711419376381642, 56.938687934439201, 
85.589689401573139], [27.872798937640738, -94.922800615920181, 24.187649616642489, -82.28813915263045], [-82.429183872811905, 
23.776052793476254, 49.178642418795391, 42.237566296771035]], [[-95.599653557158874, -3.2117874250670013, -4.6985177190430107, 
30.009412205188966], [-94.177295806719982, -67.100440854840656, 78.043053536597739, -17.760351739077223], [-82.421230180492344, 
-61.760191108767785, 83.752315525263384, -43.739701552250153]]], [[[17.27288088011116, 88.675054753497733, 6.787070381501394, 
69.810884667683069], [-55.219843507383715, 54.083266698745206, 86.063401087901809, -76.089326493737033], [-42.038461813845807, 
-49.675851649369406, -74.52809891911744, -26.645885023219407]], [[95.412577827443414, -31.818322125741673, -62.668782913250844, 
51.056651157908732], [29.626229191282619, -70.916672795563258, -93.812308419630284, 50.993951822668606], [97.536350190192877, 
-58.491790765244666, 40.487592210008074, 58.566434870771474]]]]),self.functionspace)
      res=cos(arg)
      ref=Data(numpy.array([[[[-0.12753450014394557, 0.69651395400439842, -0.97831140708599751, 0.93974896212096948], 
[0.9782603634601218, 0.78928336789060305, 0.79814708436766113, -0.50662172029142771], [0.25973763620098306, 
0.85441504186502082, -0.25141858195451089, 0.69904923600339386]], [[-0.31762040357237481, 0.63147074411941229, 
0.96070895016005275, 0.88049016444672401], [0.0066516978615375373, 0.9185608137809258, 0.70472349851785032, 
0.85587677865914791], [-0.92102992500063807, -0.50656262601085844, 0.98651834766531155, -0.10482957689473014]]], 
[[[-0.86727364348422242, 0.91254914364446615, 0.92490139133616656, -0.72021249801851428], [-0.92046217359465865, 
0.78069382297587053, 0.58566862525374508, 0.82151681807034405], [0.73320378466216263, 0.21247577462533909, 0.46528475996332708, 
-0.17305883912173914]], [[0.21717784735742893, -0.99753735846015812, -0.013870816513353947, 0.16354403647281665], 
[0.99751704509186812, -0.42937551918191064, -0.87912995574006803, 0.46319081359752184], [0.7385891182610298, 
0.47866569181287344, -0.47952224979636382, 0.97071772306861814]]], [[[-0.0058786807720651261, 0.75806165680370197, 
0.87571333970615173, 0.76751147584898971], [0.23961770902092361, -0.77996177708505321, -0.324418473652555, 
0.77054356811991798], [-0.36444710950988046, 0.83114602646711189, 0.64473117916407829, 0.057620599880895247]], 
[[0.3949359265321995, 0.92012546535486572, 0.98673349925454323, 0.70301464495912791], [-0.21715699760614957, 
-0.22879346324193936, 0.9066713780848985, 0.74619426288234914], [-0.98921817473160367, -0.36378367286102392, 
-0.93830096876607127, -0.43223588967640608]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cos_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(-36.0973683311)+(1.-msk_arg)*(-15.0675429856)
      res=cos(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-0.0309422456304)+(1.-msk_ref)*(-0.801844696357)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cos_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([13.22047315692194, 66.297986628531646])+(1.-msk_arg)*numpy.array([25.148700308715902, 
90.629815998628033])
      res=cos(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.79359430314944091, -0.94779722020057866])+(1.-msk_ref)*numpy.array([0.99987265658563174, 
-0.88866489544992711])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cos_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[-1.224755096301692, 97.114446084647994, 78.66934127829532, 20.492601387243852, 
-1.8468029073290637], [94.122385272674506, -83.420646649404915, 84.662371945918665, 96.044573177625836, -4.3173175013998133], 
[39.380839398982147, 49.750816368301798, -46.750527293456187, 96.2026545747689, 59.005783622713864], [-64.783479411930358, 
80.977519793545952, -9.0322065293006801, 59.07665091049779, 
18.567068128306488]])+(1.-msk_arg)*numpy.array([[-40.062758393821142, -77.335375405412861, 47.444663163105105, 
-22.075478512825967, 30.498519871683385], [-6.6058669139393373, -43.594483739360854, -79.124767745891376, 39.981887534339364, 
67.873590863604875], [-35.635014978859346, -91.956973584247208, 13.377356719156637, -71.146754763345243, -24.032516390161177], 
[-16.058009823221269, 66.650000191064265, 33.773123019379568, -35.806618515625303, -82.89935689255681]])
      res=cos(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.33917637019859703, -0.96244524151414412, -0.99162336600665246, -0.072186299308219973, 
-0.27251555769578545], [0.99214842647378776, -0.16764594205585706, -0.98712676481798567, -0.22407835776507445, 
-0.38487416268261437], [-0.11070385395641859, 0.87045713651526513, -0.93110614158646188, -0.37470506271731796, 
-0.7747499584525428], [-0.37169114559236371, 0.76233091786134399, -0.923928374612207, -0.8175749134173127, 
0.96036494926789795]])+(1.-msk_ref)*numpy.array([[-0.71235650110259441, -0.35821507052086493, -0.94899186128321478, 
-0.99644633755360834, 0.60788137929554054], [0.94838846229715867, 0.92573816396702235, -0.83373891839854186, 
-0.65333356768775441, 0.32342690776752892], [-0.47353549211311979, -0.65939196193876926, 0.68878387477507896, 
-0.44477233883471701, 0.45339573218511048], [-0.93935674812859138, -0.77973462685642325, -0.70781494235331721, 
-0.31617696002534756, 0.3455721479169635]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cos_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[55.483105445596493, -84.002238303792453], [28.575161935168694, 67.00969325541476]], 
[[-82.092190352486142, 70.81562364547608], [-74.349947649074409, -6.7333955028757941]], [[73.606514485922332, 
40.968585169340713], [-10.786045857308508, 65.198060327659135]], [[-50.170284604471327, -55.056223512041669], 
[77.506098391169076, -23.282530663895102]], [[-26.756072717949436, 84.143805447203476], [1.180711606109071, 
-52.027593969940369]], [[-4.2616554777689117, -84.42480221013362], [15.144723947311817, 
99.934532959080201]]])+(1.-msk_arg)*numpy.array([[[72.197802657304322, 30.031068553513251], [31.351829685111909, 
-36.235960482725837]], [[21.624265511345357, 23.64734280582843], [-89.88417079282425, 60.44136194543546]], 
[[10.996017730716773, 65.262227504852063], [-76.321146670174272, -52.668478887245065]], [[82.50391310015678, 
-20.135683721573756], [14.572063027442368, -0.6896670909694933]], [[60.842163889504008, 36.998458942261806], 
[-20.499215145000463, -72.797762608461667]], [[16.509505303768293, -79.949879255527264], [-6.9809554307616679, 
66.810788972572908]]])
      res=cos(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.48401222470712563, -0.68166289343156683], [-0.95509145527759087, -0.50945283155622245]], 
[[0.91680908584073406, -0.12942486038880624], [0.49906571322494375, 0.90035565457820055]], [[-0.21912040288715273, 
-0.99183440386679222], [-0.20799866689643615, -0.71415131202097681]], [[0.99547210549862564, 0.078271930973057291], 
[-0.51162790055459817, -0.27579265145672299]], [[-0.05251100009257479, -0.77807789151181295], [0.38026677268852599, 
-0.19015024920430146]], [[-0.43562589734555279, -0.92176067291186181], [-0.84552998640555266, 
0.8273450262977281]]])+(1.-msk_ref)*numpy.array([[[-0.99827011011545363, 0.18486878514750979], [0.99794650005510754, 
0.10743719861113013]], [[-0.93344994710516715, 0.08529414341350125], [-0.34175167339508283, -0.73093758146911281]], 
[[0.00044344313796299586, -0.75756723617489508], [0.60349204112184918, -0.73941438014513183]], [[0.68038820299051661, 
0.28083933369769526], [-0.42131616285948442, 0.77145788123141812]], [[-0.40674928195501753, 0.76442141404261976], 
[-0.078781176005662423, -0.85712634845985503]], [[-0.69559969265779853, -0.16004220706279867], [0.76627681039248841, 
-0.66943880063060068]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cos_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[81.658891253874316, 30.154394472077058, -36.265578949628633, -47.079147518937468], 
[-71.167146036277003, -41.9186826247917, -19.117903943939993, -26.079905327428762], [65.931361328358861, 90.197906187757837, 
-31.423335850518441, -28.286048187927378]], [[71.871283726884485, 97.6737686461978, 73.42998495160947, -14.865232484694445], 
[21.661875371448701, 84.396447257210269, -22.129504015834016, 42.427494938535261], [88.771175748186749, -75.408811105169974, 
-69.575485813748742, 5.5990892178879648]]], [[[-2.5470387198181186, 83.040539862229934, 27.081303077340337, 
51.185017387292476], [-47.666956331255705, -71.09454374103494, -12.656279276279207, 46.394025119360208], [-89.901778088496684, 
-17.767376720946132, -99.917741127313974, -47.717421519561462]], [[8.693865786048363, 14.971207834154555, -63.535614994602518, 
-44.328116514882751], [-51.286711883396954, 9.1258561937947604, 43.312231786364464, -33.34607470853696], [27.51825978940407, 
73.872039558971039, 84.752128628228576, -9.0124335912783948]]], [[[-26.128227021007319, 20.639092270432016, 
-22.680126332913247, 25.063812028694016], [-56.615261494527402, 67.210893028966325, 87.796053191372465, -81.471710946526699], 
[-89.314681542332949, 65.872995175957186, 43.779448960120646, 67.866612105077394]], [[-66.870791246045002, -45.354327014974331, 
-65.816476857544785, 53.473037113499004], [-41.393387472045397, 41.345634736576699, 40.192586700259739, 11.936195522492625], 
[-83.82311215875697, 79.874611048483359, -80.758910243743685, 
54.169687707747642]]]])+(1.-msk_arg)*numpy.array([[[[79.975844309170498, -48.345625034036722, 66.443940680510877, 
39.323120361210499], [9.3550298748197491, 52.272469669987004, 7.7830009762651002, -50.077500193710648], [-61.587099942318744, 
-66.863938143271454, -51.722120138546821, -43.001183411657685]], [[74.508937209465643, -17.801821857280032, 
-96.972937711273246, 72.41183780379302], [29.452848452635891, 94.133920463076976, -27.938357509488185, 4.098257930496942], 
[21.067732757629187, -52.606997531840641, 42.818983964361422, -20.470439762486563]]], [[[41.131011535410437, 
33.995686953582606, -59.091466947449334, 13.06823181504609], [-26.826785538691382, -36.157623980696727, 67.955195199568607, 
-26.327491349671178], [-66.437617402363486, -38.150104004241527, -10.872833925186612, -7.9714487224826769]], 
[[59.359911777606669, 90.541284511390501, -90.232513741603299, 79.142818011771851], [-58.547802522437806, 74.087580646014828, 
95.212298086166982, 58.740940692381656], [-97.360556330976905, 13.614895311489079, -16.297445160271735, -14.876606558474577]]], 
[[[-0.81239837735233777, -71.100446898779367, -74.082829351107051, 33.018490125485727], [8.0131639573274924, 
-29.224274631812278, 65.331408901937834, -93.121364679526437], [48.094133457963153, 94.659711075768058, -56.265304671957381, 
-44.775197816875092]], [[78.067699088113585, -50.740947996549203, -27.505788007906219, -48.105796725511851], 
[27.275728631570999, -82.611405981579054, 62.772330081477833, 91.643617134438784], [-79.364280665253176, -47.031342269127421, 
-98.190983248186029, -34.616820364422708]]]])
      res=cos(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.99974648641704578, 0.30435788687272552, 0.13683280344622692, -0.99899923093822218], 
[-0.46294192167477966, -0.47311060525720278, 0.96421021544423069, 0.58398751340702615], [-0.9991145824550095, 
-0.61510218317556875, 0.9999725511540013, -0.99993138830653794]], [[-0.92666794295836774, -0.9598311886992611, 
-0.38706137277129665, -0.66542688071327594], [-0.94627760692292984, -0.91039672543745431, -0.99044414397882241, 
0.015993433168101586], [0.69197041282698768, 0.99994395380283829, 0.89585377001765809, 0.77499061055278551]]], 
[[[-0.82839844046690569, 0.21008847929829561, -0.36884446429894552, 0.60619010167380138], [-0.85612803895715517, 
-0.39742502551382042, 0.99596093819165676, -0.74526463332938697], [-0.3582450109517854, 0.46940528357073097, 
0.81779695883550263, -0.82897059205453272]], [[-0.74456579079006469, -0.74065243965863614, 0.76241328374955808, 
0.94079803034423892], [0.52231795226174615, -0.95565457337977633, 0.78378107414507481, -0.35166755390596066], 
[-0.72753505953347086, 0.044597402857294056, -0.99748955870013933, -0.91618381584706621]]], [[[0.54409536262501013, 
-0.21699984096121355, -0.77189630709509238, 0.99762532313598207], [0.99778345689649628, -0.32720952301781842, 
0.98583053656377828, 0.97809381526902872], [0.21892156237938304, -0.99495908440040703, 0.97949675533888336, 
0.31681541317886081]], [[-0.62368710091572177, 0.19746023066230201, -0.98770566197447685, -0.99782529579539758], 
[-0.85111909839025046, -0.87520822274754506, -0.79722147176466518, 0.80792434147142878], [-0.54039529506920203, 
-0.23381695647535797, 0.60383025819594616, -0.72303366513856404]]]])+(1.-msk_ref)*numpy.array([[[[-0.13436077240726316, 
-0.34201567621685647, -0.89134402060793727, -0.053187082842302578], [-0.9975685881864238, -0.42249008090281082, 
0.07092106962798754, 0.98238330330531154], [0.32029719862810208, -0.62902930625367448, 0.11391085089048571, 
0.55609724451005038]], [[0.6299663251108153, 0.49953529411945946, -0.91453698294999897, -0.98797958829090204], 
[-0.38229784081228507, 0.99352504718605417, -0.94408885824716904, -0.5762485691730832], [-0.60309899769845521, 
-0.69665105534915361, 0.39629980402135667, -0.050066573862182912]]], [[[-0.95815603193412535, -0.84628043097907402, 
-0.82601627167478864, 0.8766887352613707], [-0.12293619592253698, 0.029304268667122303, 0.39948329810258998, 
0.36724583096741847], [-0.89419269786037481, 0.90001510332012979, -0.1224324098233948, -0.11719712955985295]], 
[[-0.94592931147917447, -0.84464083011827396, -0.64201466079431768, -0.82363702778858228], [-0.41535991838593167, 
0.25722868355816148, 0.5698126445332351, -0.58223630038953733], [-0.99958484980840756, 0.49885021791560108, 
-0.83122882247753749, -0.67387398078725091]]], [[[0.68775933759123087, -0.40283501156914814, 0.25263438090145968, 
-0.031761920026041855], [-0.15851092074962655, -0.58173128371482041, -0.80087771324083845, 0.42989944177895328], 
[-0.5650985275826873, 0.91634921141350756, 0.96012059545797757, 0.70178185948911376]], [[-0.89060741634512253, 
0.8890797163293207, -0.71892217832022665, -0.55543784153332254], [-0.54147542099155643, 0.59783639656790899, 
0.9982290297813865, -0.85902708455056043], [-0.67895034675987809, -0.99572053270346172, -0.69555015450633373, 
-0.99824220054368329]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tan_float_rank0(self):
      arg=-73.1683491266
      res=tan(arg)
      ref=-1.29094059165
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tan_array_rank0(self):
      arg=numpy.array(90.0547372602)
      res=tan(arg)
      ref=numpy.array(-1.74918567862)
      if not isinstance(res,float):
         self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
         self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tan_array_rank1(self):
      arg=numpy.array([-90.804921883575233, -8.7022315806742085])
      res=tan(arg)
      ref=numpy.array([0.3107229167523074, 0.88158317195667435])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tan_array_rank2(self):
      arg=numpy.array([[-2.3724941053440602, -48.601954124485871, -52.520368557893704, -44.648455594451299, 
-60.476894309616405], [-89.484319414032612, 78.388337444069492, -74.574676276650791, -19.378062084700787, 32.932487296528166], 
[67.090159567764232, -47.899977177175288, 39.127004012812932, 20.752555375265459, -5.8024944116525035], [-96.55061081056455, 
-83.700269788600039, -98.672699239564238, -75.043326009593869, 93.423514903830721]])
      res=tan(arg)
      ref=numpy.array([[0.96792080723658425, -10.752834845341308, 1.2263196589385075, -0.78602034428054013, 
-1.0024745151129619], [-19.563475424641489, -0.15264823373499628, 1.0793647009956275, -0.58391209678792599, 18.42000331956234], 
[2.0487598989891298, -0.98154967381469926, 6.9499973647659106, -2.8986487280973905, 0.52148931315161706], [1.1128560915479377, 
2.0804288822011725, -3.3822765215709079, 0.37058875127079471, -1.0809188670162906]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tan_array_rank3(self):
      arg=numpy.array([[[-13.399433060551161, 27.574431466092349], [-76.662808268438681, 36.369674227141502]], 
[[54.634677053530481, 17.778153988882053], [39.57242281363213, -49.419688517289885]], [[2.5196519826096875, 
21.846361806000075], [51.807079258019229, 28.599488083524648]], [[-45.73535151572019, 5.8762264632167245], [33.51395743807447, 
-2.4148231523229242]], [[-20.517037047992901, 7.3285200537442279], [88.889463770733954, -58.438339772220303]], 
[[2.5519971943378295, -12.277054039119491], [49.016285140330922, -92.240613404114342]]])
      res=tan(arg)
      ref=numpy.array([[[-1.100179422756191, -0.84212157960033707], [-3.1629996812482575, -4.0624436331973532]], 
[[2.7984939949030196, -1.8331254466158315], [-3.2041662607131003, 1.1287223918149247]], [[-0.71684284018998334, 
-0.14580705713117498], [34.237397294165511, 0.33711947331487158]], [[5.4258387964391019, -0.43102032282312885], 
[-1.7175985562162568, 0.88911654776901494]], [[10.310639070642971, 1.724623544309009], [1.3266168551858331, 
3.0289989204576986]], [[-0.66896990600621553, 0.29766865434801526], [-3.0015161363740916, -2.144296621195898]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tan_array_rank4(self):
      arg=numpy.array([[[[-59.015820056280212, 85.543699083815341, -87.819901160078444, 62.651006410716974], 
[21.720630773667352, 61.480886892675471, -7.7444639554287136, -82.349457666812413], [-84.308279246529352, -72.784808384901538, 
97.541402076032909, 2.136386578933184]], [[-83.714748392755013, -0.29729523775282019, -68.614409746518106, 20.217823136780112], 
[-56.222430473591615, 76.420888561902956, -56.827326752971288, -59.438720281308946], [-39.264531142940108, 99.419706224833533, 
90.500243572794204, -10.710457964248562]]], [[[-85.69521928028108, -22.585350896123828, -72.52927492957582, 
57.757817601793249], [51.580648782531853, -93.613693951567754, 28.171608879507488, 11.517980056351135], [-88.931829532207061, 
-56.249053451763899, -87.584984472030072, 43.491916565840654]], [[-94.53768335189487, 37.641179681991815, 11.144929267988203, 
-63.260602094839612], [-25.912108534124798, -91.229027842386785, -61.810591493115254, -10.734121781958876], 
[-6.4337815570971486, -33.651680944002351, 68.962023718952679, -16.396538431039602]]], [[[-84.799582111986084, 
64.276227805976703, -78.077367826663732, -3.6837152183603337], [78.427127328841493, 33.971410708893046, -43.793783555712373, 
82.136561293717904], [-81.186619273210155, -98.626300180003639, -14.683142006191005, -20.73255828974925]], 
[[-13.673139859757583, 8.9334351361798809, -73.591248043355648, -87.431184521884461], [5.6160360497661515, 76.88896697897772, 
88.152428349888851, -92.149709517358787], [-47.578365858255324, 99.523299278583835, 12.249212146098571, 22.654233876034084]]]])
      res=tan(arg)
      ref=numpy.array([[[[0.79950716680569922, 0.87830259423178569, 0.14571143922581117, -0.18284436052204769], 
[-0.2773156783220862, -4.475452720259427, -9.0944108025791461, -0.78908296406867962], [0.56557513792510949, 
-0.58347126143700012, 0.15321203751170501, -1.5753871708133473]], [[2.0055349111488323, -0.30637514187851106, 
0.54711901592328005, 4.8698667416051844], [0.33832609682780074, 1.6379017824275812, -0.28610298799539385, 0.2569831209871134], 
[-185.97458690416855, -2.0207202975932006, -0.69289767428481341, -3.4117825520347127]]], [[[-1.1906723226806171, 
-0.6756590963192679, -0.27960663329076402, 2.6435180360022512], [3.8263203588137005, 0.7353910856639071, -0.10308786718784783, 
-1.7368327181766676], [-1.4505841687645999, 0.30891370686855446, 0.39896037614204499, -0.53387710771185082]], 
[[-0.29830796218385497, -0.05799705759773028, -6.6455988221745717, -0.4571077918773665], [-0.98801044595711418, 
-0.12346252563819389, 1.6327456768282265, -3.7372358481221206], [-0.15174514246473089, 1.2753797489407113, 
-0.15422015341332149, -0.8229435103993028]]], [[[0.023423817551381263, 7.8678556515446925, 0.49850202748878536, 
-0.60231853913454181], [-0.11316845194916761, -0.6639341584934304, 0.19077889390802158, 0.48942552654706356], 
[0.53955636170037213, -2.8830666820829745, 1.6458713134899143, 3.0982616936665104]], [[-1.9981040768316087, 
-0.53511424608726788, -4.1550502673672733, 0.59050655093849269], [-0.78762455951792953, 12.465023179525426, 
0.19007470679790897, -1.7174437677344925], [-0.48858757161804556, -1.5839843117523376, -0.32823877780688571, 
0.78106049224881269]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tan_constData_rank0(self):
      arg=Data(-30.0025969993,self.functionspace)
      res=tan(arg)
      ref=Data(6.29796949778,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tan_constData_rank1(self):
      arg=Data(numpy.array([66.593964083030585, -35.487523356410918]),self.functionspace)
      res=tan(arg)
      ref=Data(numpy.array([0.71469184337462144, -1.3408854877899916]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tan_constData_rank2(self):
      arg=Data(numpy.array([[-89.42948815321698, -1.5051744754706249, 83.941583875472418, -48.610776414656542, 
-14.89474779545661], [-47.175391297680804, -77.33860359143749, -61.3839607538238, 66.140510322202005, 70.054304797535877], 
[14.550414584637011, 0.31585995633204789, 27.080035637773122, -15.081081992329715, 6.4886224582310774], [-55.22969140175875, 
-0.0062072766996408291, -78.154983206543847, 92.668632118052386, -59.267245301933528]]),self.functionspace)
      res=tan(arg)
      ref=Data(numpy.array([[-9.4073227815026499, -15.216946161620596, -1.213162501723507, -11.889587982778911, 
1.057241693422611], [-0.051547076449721334, 2.5814167755436053, 8.0954217841941016, 0.16863643744895468, 1.3671276946903863], 
[-2.2805130829493727, 0.32680097531525049, -2.5293544480957255, 0.72434888676539033, 0.20837690815500404], [3.8867940245995718, 
-0.0062073564235809653, 0.40502777001129742, 119.74101942010704, 0.45019384520522238]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tan_constData_rank3(self):
      arg=Data(numpy.array([[[-69.273104575358218, 60.711315608233178], [90.685907133619764, -45.641434304686058]], 
[[92.549207370647508, -99.36637222495655], [-75.196118321852992, -59.616136741559032]], [[84.889952408568945, 
26.048941169737844], [-52.364282868836121, -40.459441802176109]], [[-7.0655444131640905, 54.426855764089538], 
[-6.1744146665177624, 62.51685315076594]], [[65.054440371058178, 11.610655434102597], [-30.132725963507482, 
-86.727513476026445]], [[-52.059066692780533, 84.392425945932587], [-17.100234246718429, 
6.3858745488867896]]]),self.functionspace)
      res=tan(arg)
      ref=Data(numpy.array([[[-0.15939591097762928, 1.6319893400995349], [-0.44690821249314394, 11.290332058319448]], 
[[7.7835629384450797, 2.3249049792954355], [0.20490285117646773, 0.074259728712782316]], [[0.06705097479145887, 
1.3029611896239104], [1.7145628950189544, 0.40087759382256916]], [[-0.99394028234836151, 1.6273299043890241], 
[0.10920163748018658, -0.3258493564360137]], [[-1.3105571609810689, -1.4154099231637629], [3.3807047302720217, 
2.8844911798126636]], [[4.4140644020080124, -0.45931799763219394], [-5.5418101837864766, 
0.10305172619060445]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tan_constData_rank4(self):
      arg=Data(numpy.array([[[[-43.008982137736496, -25.894399589373563, -96.102640495608412, 70.701556910193403], 
[-18.469456478509215, -74.198222078995371, 50.46952388240129, 99.968474792527076], [-42.790367948913463, 54.701236433815069, 
-24.246273137282742, -14.875467532404272]], [[-44.892829832718562, -43.705289474025854, -47.057862077523581, 
-81.177279424926695], [95.7999944283047, 8.8762706963703835, 42.132319061115822, -95.238545354656907], [68.581933066285785, 
36.881232710145952, 87.685370512931911, -40.488130478060192]]], [[[-51.308975416693215, -92.095118469304978, 
5.2637313191304003, 6.9885669707750395], [19.1073439553074, 23.668175698701361, 22.087764980771567, -33.170111188421657], 
[61.986541559798155, 34.138046250580146, -54.921156309264816, -60.12480133574072]], [[58.074488876845777, -90.266731317557955, 
90.056867639013234, 49.897727241455726], [-48.782612786640797, -25.048704821644989, 49.190410654266998, 7.7122533322660161], 
[-71.796968169578918, -12.47843118206471, -68.255346712144132, 52.262722951222202]]], [[[-4.0858282568336506, 
6.5150811497092889, -19.551662689336055, -70.315260042367655], [62.129557315401343, 95.589557314400878, -65.507089673104929, 
-73.614933094780582], [5.2355918202245277, 46.317587800485484, 71.086123292723556, 65.939163538252046]], [[-13.169904684374075, 
53.635065866508398, -15.611206866849429, -86.274287528715348], [79.826628427580516, -55.99508950265124, -7.1459936366078409, 
81.199949922941698], [-75.362769167644529, -49.007947878888402, -66.176201908906279, 
-28.535807675974567]]]]),self.functionspace)
      res=tan(arg)
      ref=Data(numpy.array([[[[1.4696251520426851, -0.95361290052619752, 3.4251249825247791, -63.5990714681122], 
[0.39952803326161662, 2.5721638622470255, 0.20692100082167844, -0.63042387037828729], [2.511933490216212, 3.5221859611259023, 
1.2257227587962072, 1.0989275488982408]], [[-1.287784487254672, 0.28431734984364526, 0.066123846774426592, 
0.55167666777936852], [53.810756601709429, -0.61105323780837517, 3.4883481013315856, -1.5262232106875513], 
[-0.59009599213940689, -1.0671672178579088, -0.28671411763598054, 0.36794824042380658]]], [[[-1.7173268493397653, 
-1.520133438965094, -1.6261387973706098, 0.85153004464439552], [0.26365442928259597, -9.3780291255031187, 0.096918161766116923, 
5.3916428679849284], [-1.12762594029806, -0.44594052366037307, -17.613070683144521, -0.46412855455257801]], 
[[22.219474927979153, 1.1144111925066942, -1.7405691809751591, -0.38528291939832143], [11.34379133598736, 0.08423479250036954, 
-1.8492358146166747, 7.00844762379187], [0.49502894536647446, 0.088166824340488725, 1.1608317769691643, -2.201071056623654]]], 
[[[-1.3814821100547447, 0.23614403136436313, -0.84589620151352451, -2.5738407656301536], [-0.84622047092723696, 
4.2898493766160071, 0.50339015340869331, -4.634963805947022], [-1.7336356375315758, -1.0427066332493169, -2.3633206957149038, 
-0.034295623706834442]], [[-0.68933757169022136, 0.23202495854688529, 0.097059474136437562, -8.3275946720817853], 
[3.4261478840130195, 0.61803938267021241, -1.1681748275595498, -0.52246678578298333], [0.035469381697588412, 3.087108495241913, 
-0.20558108569758385, -0.2676002521215895]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tan_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(-82.7943909543)+(1.-msk_arg)*(75.4328929503)
      res=tan(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-2.02951083819)+(1.-msk_ref)*(0.0346831611459)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tan_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([-7.3248354120000698, -87.147464474235775])+(1.-msk_arg)*numpy.array([41.608442296069171, 
-78.560499218206715])
      res=tan(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-1.7100719792739603, 1.0655658466304159])+(1.-msk_ref)*numpy.array([0.96528867783115768, 
-0.020685828217283941])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tan_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[78.331320767264287, -97.011371817799485, -81.19726511567147, 61.388659133810904, 
-24.154257565058487], [-46.621846192504243, 29.720545610501091, 97.053910355407282, 24.667317167211337, 32.031716717012586], 
[-48.11366750755046, 57.519069963952319, 38.304765639255947, -10.181060763686418, 48.537532668036619], [-21.340370659113987, 
21.619546264455721, 21.709132593414495, 46.248663585742094, 
-87.301597217962069]])+(1.-msk_arg)*numpy.array([[51.454722056784107, 13.824859289128511, -41.076915438761439, 
-25.136410077592643, -54.203068907128582], [-7.7302813963067365, -98.922838690852572, 57.795135619451855, 57.707000303682662, 
-26.113694309943128], [25.925446964429838, -84.574396835248763, -58.22448848354702, -94.912695500510139, 4.7413358187785377], 
[42.616835448768825, 37.867732226137605, 63.188161743836616, 55.404492144175265, -45.751160534099313]])
      res=tan(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-0.2115701741437529, 0.39709602178317632, 0.52588927806508634, -7.7942635096775126, 
1.4860821999798721], [0.54895897913832459, 7.9851030992618588, -0.34863908165386559, -0.50222254209094175, 
0.70757259213073009], [-1.5229386136795577, 1.4604604563054311, 0.69246914322462005, -0.9434011801583404, 6.3107359742636362], 
[0.76143260773873467, -0.38970766837031495, -0.28973827292651183, -1.1979723522444043, 
0.78091846559021194]])+(1.-msk_ref)*numpy.array([[2.4924049010641216, 3.0971850388407622, -0.24070443344261469, 
-0.0036688653358406429, -1.0214190149835636], [-8.0427833045534989, -26.775735376827765, 2.9744186078649859, 
2.2853822737486791, -1.4940343775526681], [1.0147229960597459, 0.25385629561037121, 9.4865636983107855, -0.78401207028159592, 
-34.536437600193061], [-4.8014611899331134, 0.17023688871706086, 0.3721943678983648, -2.2000396353460703, 4.982599492212537]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tan_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[1.6102560966153305, 35.211412889683288], [-91.348383522015013, -43.857284315634139]], 
[[33.539004788531201, 8.8251891315726425], [51.330955600675225, -87.461115337251741]], [[90.686832781690981, 
-45.277807895003107], [-17.344779326601852, 74.101133091794566]], [[28.236736300559841, -48.0922384802859], 
[-69.594930186950705, -71.960357666661935]], [[62.854665874822416, 39.757063242854031], [94.915058237249042, 
76.347604801075533]], [[6.0764673077697182, -36.784818989774635], [4.9181469864972343, 
61.284822866682845]]])+(1.-msk_arg)*numpy.array([[[90.73596176800811, -31.215570947384649], [25.433399939639358, 
73.301832358500462]], [[-38.89836254920025, -64.916029536379071], [36.46922783395209, 58.453758155653674]], 
[[55.500707090549128, -47.499239289706253], [-13.773329408173723, 25.90545737823318]], [[-23.056721349061533, 
30.464629957469555], [35.060143579080574, 34.084769656498338]], [[79.356213483449466, 86.222912134948189], [-41.30052616314579, 
-71.566281173260137]], [[35.36133591328084, -0.62772684982286364], [45.373439624402266, 78.846680693659096]]])
      res=tan(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-25.329111806401379, 0.76636658169690774], [-0.24704607074007473, 0.12566817386772378]], 
[[-1.6227198571843844, -0.68353336156502986], [1.8075514996805491, 0.55082835717393452]], [[-0.44579814645029653, 
-3.5403624361062564], [15.124973719611226, -3.5618646683661388]], [[-0.037615307477773245, -1.4540461107144631], 
[-0.52047333601538337, 0.30525770073957043]], [[0.02281676129380757, -1.8877207961528162], [0.78783420934824355, 
1.3965550719789721]], [[-0.20971372967545993, 1.297829170232494], [-4.7912978354452571, 
-42.068778832035186]]])+(1.-msk_ref)*numpy.array([[[-0.38812224689308822, 0.20308026233909043], [0.31005813906306778, 
1.7240934091643856]], [[-2.5664560030424779, 1.7736633153110168], [-2.8187777975795356, -2.8791077986502325]], 
[[-1.7351073396694705, -0.39403026733253455], [-2.6261163344676994, 0.97495228485454377]], [[-1.8079767245857725, 
-1.4022215555778026], [0.54971501950449431, -0.51142973691805127]], [[1.0640024741404535, 5.7947868248621326], 
[-0.49522667978110607, 0.82592445153584237]], [[1.0375326627384083, -0.72563890838339118], [5.5062458315870551, 
0.31687356918461207]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tan_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[46.491077325888426, 29.447945316611367, -23.586457473126401, 78.69939563243048], 
[-36.620072103892795, -72.120158121127872, -9.1790882482629428, -32.045673298618311], [87.88459132385367, -26.056616429108502, 
-46.175019172437651, -25.222466232047466]], [[94.39339312856572, 95.93512857059423, 1.1544276807080109, 71.523417453984763], 
[81.642196442509885, 98.829647356363438, 16.41888816038761, 88.129224341307577], [99.019064223470053, -83.911779981643008, 
-18.183588610107179, 61.821703879571942]]], [[[-31.031924849456871, -12.837394270780791, -85.569840431879427, 
-25.045095880271646], [-48.476356109161969, -26.767989157305678, -92.458181314729472, 43.943889922385893], 
[-43.882838733308141, -57.543085380047842, -51.027180728714328, -47.670724706899705]], [[-48.579616580739724, 
63.70607585492786, -5.7697764547977926, -26.508051068464738], [87.895871897975695, -70.198617661527948, -55.595447190108715, 
33.96389855756027], [-94.131645641046816, 10.196260987949145, 35.960983077713763, 44.216698389509759]]], [[[23.258359787375511, 
93.483587852251645, -91.678626786292568, -27.968910950121057], [32.644797019074076, -29.649432737967047, 92.711941076427763, 
53.941442799366115], [-0.86301094415041746, 43.29692826471782, -59.552717665098911, 26.637950671845402]], [[40.667728859386244, 
-40.777619796397801, 81.309833140301237, 39.694822835400061], [78.540349363328545, 82.32321933868468, 51.776690636912804, 
63.487306940913385], [21.112464219695084, -3.5816562554041411, 52.108534339413495, 
41.290731086375132]]]])+(1.-msk_arg)*numpy.array([[[[-34.364881723672539, 87.687578661166697, 43.648764082558017, 
53.282256903259736], [-53.230587114313565, 38.974570407598577, 34.583114247146796, 93.582888491530355], [-3.6200179443561638, 
79.020818674349385, 70.078727626785792, -28.108566735771262]], [[90.69928625977667, -2.1565268391896808, -47.243800256190951, 
-83.178804302595779], [77.153867197608434, 68.660513660938534, 8.3217830648971045, 27.929305713042126], [10.196055190991544, 
58.151981582440385, 66.523273718547046, -6.6877710232844834]]], [[[60.903792844808748, 33.015388147600078, 4.2001855873327543, 
65.001401669469288], [-90.726693407996507, 4.4943970319144313, -23.080076858571459, 74.888412270995076], [30.259226971212826, 
-28.521562428787718, -12.993093176507301, 61.934973167544001]], [[20.538642074545436, -20.935126708996265, 69.180246563064031, 
21.519398436361328], [15.333534083569461, 27.630503865288574, -1.500969476128617, 90.330870767340997], [-33.056101278068823, 
1.7310360358059285, -48.781639323707402, 30.831490267631267]]], [[[-94.925303241927267, -84.897084601406391, 
-98.913684138256414, -15.535976111476174], [-13.988938568622373, 45.947782563866554, 66.111021486607058, -84.17642816992452], 
[-58.931536389236982, -70.315637041480841, 91.959253912296901, -3.1911032774115711]], [[-73.628963698291997, 
-29.488871813922742, 21.072434974335124, -47.163429466844129], [61.271656416986644, -16.795896961687035, -50.73883783992683, 
-99.733328467239858], [-90.272007621846399, 95.223199555065008, 86.994040638722311, 9.8533693129492406]]]])
      res=tan(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.73343120662525252, 2.3839105002511194, 40.787222808366948, 0.16094782608102345], 
[1.8669025194812907, 0.13732653419637611, 0.25075562441470961, -0.72872694051721498], [-0.080174100501534507, 
-1.3238763715093944, 1.395050031588724, -0.089966560555014355]], [[0.14665148735221328, -8.5409284144307644, 
2.2612972634669206, -0.9007215017868293], [-0.039232661248680731, 7.6180321764527887, 0.86113813844661358, 
0.16613366949677899], [-16.959550464823533, 1.2896176983531111, 0.78571117079365793, -1.5927335619554577]]], 
[[[0.404060252091059, -0.27786051202158374, -0.92570903371055602, 0.087870462140269959], [-4.507213144849521, 
15.494028370999409, -4.497175009718946, -0.038426124048842311], [0.099787666563255661, -1.5384497855699044, 
-0.95368910787364802, -0.6087587826180636]], [[-8.6520059627526305, 1.1955317828142094, 0.56384270447617579, 
-5.0501143130775983], [-0.068830793974709911, -1.8874380240156459, 1.4079440068912947, -0.67481219010045013], 
[0.11665890217631358, 0.97254992731388834, 5.9202456966132972, 0.23879071044341971]]], [[[3.1921468290216199, 
-0.95846183249471029, -0.64441616463570783, 0.31528813899144992], [2.8097371180779902, -5.0445284153583341, -28.59426924755839, 
0.59179921342809771], [-1.1686540503354326, -0.81757996000162092, 0.13841671213997156, 15.225082410023267]], 
[[-0.1747217143289867, 0.063168519328776943, -0.38967719286870423, -2.2100407587939466], [0.00053302363419388608, 
0.74736152538309053, 16.761993487641927, 0.76884603209500324], [-1.2064284582874256, -0.47085822776253189, -3.5818161403026476, 
0.48308786024645467]]]])+(1.-msk_ref)*numpy.array([[[[0.19505624946190481, -0.2843259566886906, -0.34647724580667361, 
-0.12547047820166515], [0.17834353548942933, 3.286930560390747, 0.025600648291005411, -0.78397206481850568], 
[-0.51861097048005378, 0.52188551279039708, 1.439633029347724, 0.16730238109610227]], [[-0.43095137196253414, 
1.5074095617966261, -0.12048848856897221, -13.599313538553698], [-5.3481175767151781, -0.48864785355494489, 
-1.9794018936539635, -0.35940434921762371], [0.97214955673987546, -30.741840194434616, 0.612868574375663, 
-0.42820918892409709]]], [[[2.6789375605862964, -34.875846819936136, 1.7785521656252401, -1.4656166707976426], 
[0.3988255914237408, 4.5144301583416366, -1.9120915598810979, -0.55911115208210205], [-2.2752580938171345, 
-0.25239185044421364, -0.45466017088407429, -1.2521150119968862]], [[-8.4143456305126385, 1.7678968748357526, 
0.065300765623003867, -0.51016958443706106], [-0.39296746508187924, -0.75051404592755078, -14.297855322237128, 
-0.9800366404508225], [14.390571255598807, -6.1871454817516467, 11.471441215914821, -0.66152749037896763]]], 
[[[-0.80457380870258743, -0.074218782108839948, -21.497073365851712, 0.17370323306664814], [-6.6968647716992491, 
-2.4006899112511384, 0.1384503528165294, 0.75481167789734471], [0.94802577140077926, -2.5767180463531125, 1.1453985305058376, 
-0.049551118689906332]], [[-4.9723768792651821, -2.6871814892327959, -1.3097646062444974, -0.039560281128975132], 
[-94.339008797193358, -1.907469445158082, -0.51219428963168179, 1.0247810836302187], [1.1026512201765113, 1.4762969700485067, 
-1.4609350848042868, 0.45691718981098589]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asin_float_rank0(self):
      arg=0.921542730641
      res=asin(arg)
      ref=1.17203520937
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asin_array_rank0(self):
      arg=numpy.array(-0.116366973393)
      res=asin(arg)
      ref=numpy.array(-0.116631212893)
      if not isinstance(res,float):
         self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
         self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asin_array_rank1(self):
      arg=numpy.array([0.59510006409554861, -0.26006099506635738])
      res=asin(arg)
      ref=numpy.array([0.63739015474464755, -0.26308537092922801])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asin_array_rank2(self):
      arg=numpy.array([[-0.49989439482181364, 0.57493953328558645, -0.22601342898621324, -0.83034953339491324, 
-0.60260563309327075], [-0.1183899214658567, 0.6688404400551391, -0.071516382310285787, -0.78037985627758721, 
-0.95192547390413962], [-0.053129700505135102, -0.15530697314779995, 0.92610227514436883, -0.37007640105796091, 
0.56505716602168943], [-0.62671081964525888, -0.15753169566378111, 0.91493774880171697, -0.90680349497265444, 
-0.40551773776530553]])
      res=asin(arg)
      ref=numpy.array([[-0.52347683753416852, 0.61253024363956177, -0.22798325969707625, -0.97973464653994857, 
-0.64676214381720187], [-0.11866824279449521, 0.73264789716466916, -0.071577485913038447, -0.89527306004515883, 
-1.2594613457722619], [-0.053154727752750201, -0.15593818938989257, 1.1839468899449852, -0.3790912593373173, 
0.60050253973749557], [-0.67732506399858938, -0.15819063839895628, 1.1553536519616823, -1.1356384378982181, 
-0.41754515818302307]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asin_array_rank3(self):
      arg=numpy.array([[[-0.10842716508524175, 0.5082117637587793], [0.58585987724409372, 0.90471202587596511]], 
[[-0.20200633220213926, 0.94124210243213469], [0.16461270307960985, -0.032077448048531854]], [[0.83741397268700912, 
0.063397104323260178], [0.88417010321942135, -0.60086994262517457]], [[-0.60498074744701535, 0.32804630789980416], 
[0.9543718794297158, -0.28951903491123265]], [[0.20176530654432256, 0.83818508768174582], [-0.57874532118996114, 
-0.78012661689222274]], [[0.93482043277232596, -0.42274083268354035], [0.56472167733846312, 0.51569699638917799]]])
      res=asin(arg)
      ref=numpy.array([[[-0.1086407500696193, 0.53310714556714378], [0.62594068801933334, 1.1307032773581887]], 
[[-0.20340605458647631, 1.2262894251988117], [0.16536534504568678, -0.032082951680059833]], [[0.99253454813338793, 
0.063439648847116878], [1.0847145607858641, -0.64458898108908502]], [[-0.64974168787116715, 0.33423469094145819], 
[1.2675492557523189, -0.29372431413119748]], [[0.20315996164410971, 0.99394690458257451], [-0.61718932564806839, 
-0.89486817742762526]], [[1.2077530763452442, -0.43646755933558173], [0.60009596889108507, 0.5418209656130919]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asin_array_rank4(self):
      arg=numpy.array([[[[0.88966005181042362, -0.29090442515415971, 0.45676523069227426, -0.1701472152064889], 
[-0.15433686972589178, -0.692562435723725, 0.68853169353761912, 0.077072679404003086], [-0.46022904059871383, 
0.26090747633090494, 0.55090661938911167, 0.57154126282973339]], [[0.24125763624481666, -0.34205570616986392, 
-0.37099113633403424, 0.1179598556953767], [-0.51470852431381331, -0.36723245932705617, -0.029819240904085831, 
-0.40762301554494829], [-0.30049438920777694, -0.63800972272934819, 0.57533920712090647, -0.5547685031617493]]], 
[[[-0.45428517526235013, -0.91591813179303816, 0.048138704633584828, 0.16781373520254439], [0.55928446578010882, 
-0.17996621670450408, -0.41269412581352494, -0.16388089460637645], [-0.69029985924238013, -0.81436668624702369, 
0.6544901000178458, -0.52000585988815029]], [[-0.48651827566799033, -0.92984762682329214, -0.82193504097585135, 
0.3187163608211292], [0.79286290660471681, 0.70068675467956432, -0.33330102833624109, 0.6421872627888765], 
[-0.9326448796142256, -0.85359123150797367, -0.22108314451399158, 0.72617047853626748]]], [[[-0.2278965599162287, 
0.2783546807423698, -0.054403887255599193, -0.52370326050531091], [-0.64076099399832764, -0.84493595964898371, 
0.45103869280140008, -0.34175422352324081], [-0.36452306370640453, -0.66647673460051138, 0.70930387643468928, 
0.30980967826785966]], [[-0.60933431763720147, 0.69296120123776173, 0.80122387712332555, 0.90915685991222528], 
[-0.64692029738039536, 0.29646040695346199, 0.030404099519964367, -0.33816747484983467], [0.0096626525135834473, 
-0.33985523430847275, -0.053345947133067062, 0.32340108068154527]]]])
      res=asin(arg)
      ref=numpy.array([[[[1.096600146212479, -0.29517200978672831, 0.47435553104518491, -0.17097906076855005], 
[-0.15495624603949892, -0.76503526228584184, 0.75946242730472591, 0.077149188591699849], [-0.47825316791617667, 
0.26396211924382607, 0.58345018405227489, 0.60838290486102364]], [[0.24366155889752167, -0.34910369584784678, 
-0.38007609635636896, 0.118235142196882], [-0.54066764777441745, -0.3760318296904806, -0.029823661820804641, 
-0.41984948575077341], [-0.30521095698992118, -0.69191080852804077, 0.6130188093590363, -0.58808470051499495]]], 
[[[-0.4715696479852326, -1.1577895055396825, 0.048157316298299481, 0.1686115337087383], [0.59352239389335559, 
-0.18095210710088311, -0.42540983845725161, -0.16462346123538527], [-0.76190341390320648, -0.95163709743774971, 
0.71350801552982279, -0.546857811070394]], [[-0.50810014137090853, -1.1939985079019542, -0.96480004280611553, 
0.3243749145377256], [0.91549265425679227, 0.77635959931468246, -0.33980264503789476, 0.69734826101164271], 
[-1.2016754217560228, -1.0228405540600332, -0.22292495796884845, 0.8127353450352377]]], [[[-0.229916847463042, 
0.28208066229523743, -0.054430760346312043, -0.55119222763060238], [-0.69548906875601646, -1.006445424494244, 
0.46792879299916307, -0.34878287902414212], [-0.37312057474105464, -0.72947286465013783, 0.78851017261506551, 
0.31499285551356021]], [[-0.65522078004289686, 0.76558820748693557, 0.92933779604320665, 1.1412549961378431], 
[-0.70353883417231877, 0.30098430617202493, 0.030408785774862383, -0.34496896929459286], [0.0096628028818117016, 
-0.34676296542532031, -0.053371281484292522, 0.32932151749631378]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asin_constData_rank0(self):
      arg=Data(-0.431396747373,self.functionspace)
      res=asin(arg)
      ref=Data(-0.446040426687,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asin_constData_rank1(self):
      arg=Data(numpy.array([-0.045428206170097463, -0.31923313806271669]),self.functionspace)
      res=asin(arg)
      ref=Data(numpy.array([-0.045443845896120716, -0.3249201744113962]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asin_constData_rank2(self):
      arg=Data(numpy.array([[0.30858622140952829, 0.55975622903552336, -0.72296555662115225, 0.29902784110302227, 
0.8947344332511491], [-0.28086673729874623, 0.36930400884433667, 0.055454221970449469, -0.97164417842046047, 
-0.075055313104506749], [0.93809396914032095, -0.57188228631310689, 0.45014552683962106, 0.91746601597092892, 
-0.51070243899383461], [0.51841908694667493, 0.39576002646208308, 0.18297603812902641, 0.63377426610316556, 
-0.83051619049372838]]),self.functionspace)
      res=asin(arg)
      ref=Data(numpy.array([[0.31370635716784467, 0.59409159492300612, -0.80808514330144787, 0.30367371766279994, 
1.1078362050462669], [-0.28469707958989926, 0.37825997479970791, 0.055482683240285852, -1.3320879147382605, 
-0.075125960524460869], [1.2170859487219492, -0.60879855676389416, 0.46692830451457823, 1.1616631850328356, 
-0.53600161208133779], [0.545001165256355, 0.4068953000547485, 0.18401274786360855, 0.68642286711820089, 
-0.9800337876373596]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asin_constData_rank3(self):
      arg=Data(numpy.array([[[0.21550495466289088, -0.41647705032555027], [-0.67886462371191669, 0.22474245692504513]], 
[[-0.96334469136781409, 0.95593870477481868], [0.66328477249108797, -0.80145077788727803]], [[0.89898917230385633, 
0.1774129768963526], [-0.05129063564194114, -0.053588501749134543]], [[-0.6428424400535997, -0.6775276306234792], 
[-0.19531987534983708, 0.5043002210160703]], [[0.019534435998354471, -0.52684414437015681], [-0.6055926650690967, 
0.69383797088632582]], [[0.96505883636811807, 0.44488881526651802], [-0.58391588073180656, 
-0.78409047454798708]]]),self.functionspace)
      res=asin(arg)
      ref=Data(numpy.array([[[0.21720890557728748, -0.42956685605504841], [-0.74621524799537076, 0.22667872366014621]], 
[[-1.2992030317775713, 1.2728409005456607], [0.72519952323795989, -0.92971709383444179]], [[1.1174560405729719, 
0.1783571011760767], [-0.051313150936083789, -0.053614183546871738]], [[-0.69820327890816269, -0.74439591420945306], 
[-0.19658359684931548, 0.52857140167195993]], [[0.01953567858295259, -0.55488333663246614], [-0.65051042137277548, 
0.76680500836780285]], [[1.305667822198445, 0.46105010951250719], [-0.62354398750998707, 
-0.90122931909666981]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asin_constData_rank4(self):
      arg=Data(numpy.array([[[[0.51329109257338223, 0.17638688863895813, 0.19744849395823549, 0.29970826785632898], 
[0.089617882800095305, -0.37724763017644924, 0.59818929907588569, 0.073704816796113404], [0.51902744583651805, 
0.82464364926529643, 0.88748655297225953, -0.85265192651803112]], [[-0.87001926728047962, 0.3115239678743027, 
-0.25453986022985609, -0.70581647691430527], [0.57415191967585866, 0.51196143677988815, 0.083722655439862459, 
-0.64179016127270039], [0.051555209592012652, 0.94524755109618686, -0.56223316652327482, 0.68020119939589163]]], 
[[[-0.62429913234626166, 0.68069439023091149, -0.75088135587997096, -0.8952032164472512], [-0.98378574742269875, 
0.095565317191834032, -0.1209650098901609, -0.76429525141130861], [-0.28776192762841646, -0.61083531299057392, 
0.2211123500220018, -0.067015454819630449]], [[-0.22790630542101575, -0.17754749697118044, -0.041038715396540693, 
0.49224268574691377], [0.91263635337803639, 0.0074790580830090825, -0.59640193079471981, -0.59729702493832493], 
[0.85233492612134532, -0.35846165293336951, -0.47620666195463901, 0.94342203393223345]]], [[[0.56723269228361684, 
-0.80821529187729158, 0.98679652604886114, 0.27468576367466624], [-0.85599127998888336, -0.9376189775985958, 
0.63709044853440533, -0.052295250533300086], [-0.55038394046099415, 0.50496293378490931, -0.35751955178831762, 
0.67752074651881133]], [[-0.43319700685010187, 0.12314979151568961, -0.6116075679468943, -0.56810680840205885], 
[-0.7069345719913811, -0.082489642682527009, -0.65674869784167045, -0.48692682645143048], [-0.92691310557280204, 
0.58262478280467001, -0.47397194661641007, 0.22217343281501023]]]]),self.functionspace)
      res=asin(arg)
      ref=Data(numpy.array([[[[0.53901522517510991, 0.17731457095913181, 0.19875448966163933, 0.30438685033273022], 
[0.089738277416217413, -0.38682253015972068, 0.64123964853566662, 0.073771712799479763], [0.54571274425398231, 
0.96957192870487008, 1.0918620613107901, -1.0210401242105613]], [[-1.055241399501305, 0.31679638615842398, 
-0.25737186146177238, -0.78357505844152286], [0.61156794655593194, 0.53746661381479266, 0.083820774003697224, 
-0.69683033453048715], [0.051578075390915357, 1.2383523064463466, -0.59708372062425452, 0.74803707776282535]]], 
[[[-0.67423402613433836, 0.7487101033953697, -0.84939557236661423, -1.1088869858972243], [-1.3904729504724711, 
0.095711380308911853, -0.12126197357074665, -0.86994776181052924], [-0.29188909512746969, -0.65711517302815758, 
0.22295490460665185, -0.067065718330701532]], [[-0.22992685635987592, -0.17849379129345105, -0.041050243539981428, 
0.51466432495832592], [1.149687769120979, 0.0074791278099158847, -0.63901106736411895, -0.64012665366330246], 
[1.0204337009655178, -0.36661951476761556, -0.49633576889089914, 1.2328032031186469]]], [[[0.60314176416042264, 
-0.94111513576231365, 1.4081148467409164, 0.27826287985156239], [-1.0274652123272283, -1.215717186843875, 0.69071757938722977, 
-0.052319116031394948], [-0.58282402577863912, 0.5293390213768131, -0.36561054423355616, 0.7443865544541004]], 
[[-0.44803687496570443, 0.12346321423915407, -0.65809090825146055, -0.60420354302721868], [-0.78515465246445915, 
-0.08258348099914431, -0.71649917687907549, -0.5085678280485314], [-1.1861017810533363, 0.62195450939266994, 
-0.49379613823657181, 0.22404305196597224]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asin_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(-0.935485281929)+(1.-msk_arg)*(-0.357063901388)
      res=asin(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-1.20962989131)+(1.-msk_ref)*(-0.365122692091)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asin_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([-0.63169403295190651, -0.83458513213454666])+(1.-msk_arg)*numpy.array([0.82699554920489238, 
0.62915303164526115])
      res=asin(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-0.68373650599948488, -0.98737925343052979])+(1.-msk_ref)*numpy.array([0.97374247172577488, 
0.68046307573541864])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asin_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[0.13492976012467373, 0.45039665300870291, 0.82745760804589152, -0.78032328525238159, 
-0.79535932380071572], [0.040200434802502683, -0.560657055826701, 0.75257355988623198, 0.17014171439389081, 
-0.62381482981900049], [-0.13673638445438074, -0.57207005371691855, -0.70885711527885409, -0.59521295394636686, 
0.27796938493484036], [0.28968060938295759, -0.81382014034948424, -0.57883480062175985, 0.97828524335498468, 
-0.47692242103358973]])+(1.-msk_arg)*numpy.array([[0.98463045928945925, -0.71016923240161356, 0.22724017107798478, 
-0.13420432795884052, -0.49522411318530396], [-0.24215065909068589, -0.47545453895201739, 0.16689573585393336, 
-0.884725036039959, -0.2448024385199935], [0.47239090503000214, -0.065838234321151323, -0.60401771019109174, 
0.94495449584511348, -0.57500867223012198], [0.026113254330203839, 0.84338192310488602, -0.68332490675398916, 
-0.90490822828641326, -0.79511524036967574]])
      res=asin(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.13534257398878016, 0.46720955487698312, 0.9745648331075083, -0.89518259576887582, 
-0.91960015750312418], [0.040211270503774127, -0.59517908662188101, 0.85196156667185596, 0.17097347856252973, 
-0.67361422108213365], [-0.13716609949526337, -0.6090274659653474, -0.78787658342813327, -0.63753063318255765, 
0.28167953568827331], [0.29389312240631499, -0.95069596761109842, -0.61729905219723025, 1.3620198752343233, 
-0.49714992765255567]])+(1.-msk_ref)*numpy.array([[1.3952453465226335, -0.78973855655962144, 0.22924277229757584, 
-0.13461048306641557, -0.51809278015141624], [-0.24458186898310855, -0.49548063364167766, 0.16768040234131235, 
-1.0859037723974954, -0.24731592596092278], [0.49200146764799862, -0.065885891877964342, -0.64853276181815889, 
1.2374555042108284, -0.61261474784852954], [0.026116223021538727, 1.0035465968695956, -0.75230693964273876, 
-1.1311640462762003, -0.91919757773497301]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asin_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[-0.88294562430405077, -0.58883850026769669], [0.78435414730135289, -0.39414060857807909]], 
[[-0.1571683926997115, -0.7975580613896821], [-0.2945111683422631, 0.41055434942130842]], [[-0.80681023399648044, 
-0.89237160017422534], [-0.35686026887708688, 0.89137098122307035]], [[0.58188145946961756, 0.98096002867963006], 
[-0.96703875848368448, 0.69253168197482706]], [[-0.95183876438652382, -0.8152627643759649], [0.3102730445853128, 
0.19886153789887206]], [[-0.55485031032008247, 0.35995694685484425], [0.6885493902744837, 
-0.095941786351267377]]])+(1.-msk_arg)*numpy.array([[[-0.74710543677102026, -0.22789353215452168], [-0.15973686026830802, 
-0.36962540900031582]], [[-0.50276936501545821, 0.41783521835723847], [0.63006184002494159, 0.06284672567721139]], 
[[0.67244742928322609, 0.61492056321962485], [0.75395602128802874, -0.63454839490795589]], [[-0.56678694778925376, 
-0.7609018889656215], [-0.48168621185231431, -0.43937248667302553]], [[-0.81696324056246894, -0.84056300912251525], 
[0.22986607490662969, 0.69055079459555779]], [[0.15385067709048927, 0.62820976038893495], [0.60245120466023039, 
-0.339411441556008]]])
      res=asin(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-1.082099943123908, -0.62962103347527654], [0.90165426879134403, -0.40513257681710863]], 
[[-0.15782275264006218, -0.92323629240894056], [-0.29894395981631167, 0.42306192755623429]], [[-0.93873307278326767, 
-1.1025731963686696], [-0.36490469787552221, 1.100360847855689]], [[0.62104022097574174, 1.3753445421127855], 
[-1.3133320610622126, 0.76499263002524065]], [[-1.2591784107070783, -0.95318280312402626], [0.31548023857866159, 
0.20019612027398354]], [[-0.58818302908839293, 0.36822174663823493], [0.75948682993754912, 
-0.096089587304573051]]])+(1.-msk_ref)*numpy.array([[[-0.84369670087251036, -0.22991373787470354], [-0.16042408470691499, 
-0.37860584724501922]], [[-0.52679952575473488, 0.43106126206126066], [0.68163284381925415, 0.062888170448777259]], 
[[0.73751053069511152, 0.66228522974939386], [0.85406347654141701, -0.68742406751509944]], [[-0.60260062231540579, 
-0.86470192892228293], [-0.50257784145479101, -0.4549000012992826]], [[-0.95612535890128514, -0.99832169691796069], 
[0.23194007064979238, 0.76225029541447975]], [[0.15446417622903663, 0.67925011914863354], [0.64656864810064951, 
-0.34629112552047575]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asin_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[-0.27511398230535411, -0.46662989989543591, -0.22441239953004122, 0.18150688623048961], 
[0.49941686769326732, 0.87457024376703973, -0.87380665594527707, 0.79121346821665162], [0.50231236792513156, 
0.1191706805278514, 0.65751052732157733, -0.83527921961876861]], [[0.42542627900989349, 0.95254516260310917, 
0.9797450311831879, -0.056086392265635943], [0.70562385261889182, -0.15238185937635873, -0.7520053238168829, 
-0.29012974466979236], [-0.49037878258834466, -0.37598859643327753, -0.28323067888119635, -0.64026010918540921]]], 
[[[-0.65005531302701525, -0.39405833873845286, -0.75904048565507809, -0.015003604383261648], [0.9090993871887163, 
0.58017246409081613, -0.26368358487187271, 0.32167448565884116], [0.26214130341082931, -0.98967967346675467, 
0.064360402500340586, -0.50611410136153889]], [[-0.42093480017575857, 0.70135268664243311, 0.71895184273718149, 
-0.86323319073763405], [-0.5767136644809493, 0.39583920332661826, 0.34268205824253761, 0.61372782577529894], 
[0.46752193036938783, 0.7154831469400309, -0.82255666245626213, 0.96516971353008452]]], [[[0.0019509455447228596, 
-0.78737473712788097, -0.43804967448687215, 0.64335035861888956], [-0.49542482696794432, -0.17151517826495466, 
-0.97868047818299519, 0.35390383795454494], [-0.7694409760951727, 0.58427831802745445, 0.00096330111127984619, 
0.75410408265331119]], [[0.71720235299347612, 0.86142952009247975, -0.15674648691300697, -0.56118354114754865], 
[0.9174804385553772, 0.60787171495325998, 0.3178492462700333, 0.57104165965956977], [-0.17174684362927917, 0.91599805188901029, 
0.63147510284287045, 0.55433424393878439]]]])+(1.-msk_arg)*numpy.array([[[[-0.62285256226667429, 0.94747808126172939, 
0.73484479595758656, 0.69147310694883268], [-0.022200619225001272, 0.50484210944387953, -0.36523592219393508, 
-0.92502655519871246], [0.33803049448399181, 0.6212699560750059, -0.50576790379229219, -0.21926800433967464]], 
[[0.23713462201122026, -0.89857553666784007, -0.52300843087669113, -0.81440210653268541], [-0.36907667472630967, 
-0.38580765289932628, -0.54018307162586165, -0.64545896783864887], [0.54718615470341425, 0.56596067096491853, 
-0.034187593672710626, 0.62062878010765443]]], [[[0.54215374118024462, -0.39927917095284304, 0.06193411181948294, 
-0.041196326853802367], [-0.76425260056210043, 0.63444559349493024, -0.16775721544684186, -0.91860693657908643], 
[0.49425201961547427, -0.95983545296803341, 0.88259670458618467, 0.43984665325503514]], [[-0.88310554812007747, 
0.69692150209492754, -0.62493853854404269, -0.41708169944900997], [-0.076309665994630294, -0.33991125352713836, 
0.95306631662451768, 0.46918567435242609], [0.2136605652056387, -0.82500614020397123, 0.33456477704955012, 
-0.71852588897756831]]], [[[0.66125441549764008, -0.35324644445977804, -0.27072683646971263, -0.61339724659399686], 
[-0.21460347895993015, 0.52662476392791757, -0.63192318334950093, 0.48443466061384877], [-0.61215005225351682, 
-0.065272774939972189, 0.29165000657400753, 0.49142027865503946]], [[0.76190555814182281, -0.088837714027288461, 
0.12896878495483866, -0.93453202997325036], [0.88788589890627034, -0.5840128766952335, -0.72168479661107432, 
0.92147989476843639], [-0.040219618430762605, -0.30331298680308938, -0.76494577650641415, 0.82340012787280714]]]])
      res=asin(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.27870825761839202, -0.4854765517566324, -0.22634001435677467, 0.18251857409233957], 
[0.52292556319079719, 1.0645488267204084, -1.0629763197185023, 0.91279074057775189], [0.52627093252767754, 0.11945456862335069, 
0.71750985905497855, -0.98864042223401249]], [[0.43943286361981204, 1.2614906926972653, 1.3691842345960543, 
-0.056115838973713819], [0.78330317865231025, -0.15297783166342702, -0.85109907743804869, -0.29436241112714734], 
[-0.51252432804152026, -0.38546342334403261, -0.28716105928761965, -0.69483683195426182]]], [[[-0.70765722559105781, 
-0.40504306237600363, -0.86183803602937437, -0.015004167345881437], [1.1411170118941574, 0.61894041864228966, 
-0.2668389538587293, 0.32749743707328327], [0.26524043584810481, -1.427003927323073, 0.064404918464174121, 
-0.53067323830189173]], [[-0.43447562136549622, 0.77729339954318932, 0.80229313038767136, -1.0416398462883221], 
[-0.61470024504554266, 0.40698151772109187, 0.34977033591812273, 0.66077362448906518], [0.4864853881881962, 
0.79731542525594112, -0.96589226956572438, 1.3060912939591347]]], [[[0.0019509467823360682, -0.90653883093385168, 
-0.45342796599562785, 0.69886654397680281], [-0.51832382875793648, -0.17236743258110099, -1.3639354598603213, 
0.36174179443241539], [-0.87796546151930466, 0.62399051275396267, 0.00096330126026229724, 0.85428889123119034]], 
[[0.79977936132339045, 1.038077685909877, -0.15739555185318704, -0.59581503847922357], [1.161699441126113, 0.65337749134827083, 
0.32346023265640822, 0.60777418739966049], [-0.17260258733578449, 1.1579886714720635, 0.68345412738926736, 
0.58756284808396553]]]])+(1.-msk_ref)*numpy.array([[[[-0.67238363036634785, 1.2452560758741194, 0.82543781751001044, 
0.76352624629541876], [-0.022202443290184014, 0.52919904495299541, -0.37388622091774154, -1.1811054880957892], 
[0.34482341746192496, 0.67036233983021343, -0.53027188149326687, -0.22106415384083244]], [[0.23941527683712491, 
-1.1165125176456334, -0.55037680917350096, -0.95169813260904035], [-0.37801535954330284, -0.39608309178171242, 
-0.57065463584988774, -0.70162403529315909], [0.57899875083300933, 0.60159803973044745, -0.034194256873020745, 
0.6695443574790968]]], [[[0.57299812163741748, -0.4107304919526254, 0.061973775152364977, -0.041207988399531562], 
[-0.86988162536296132, 0.68729106448183663, -0.16855420118021394, -1.164540710942632], [0.51697419956469148, 
-1.2864151404265336, 1.081357249605716, 0.455427915365544]], [[-1.0824406960699393, 0.77109579311860177, -0.67505280174259574, 
-0.43023203101282276], [-0.076383921371208918, -0.34682253070575986, 1.2632073895421627, 0.48836842974235006], 
[0.21532052757007339, -0.97021306508027971, 0.34114335485168745, -0.80168049416163079]]], [[[0.72248972448527826, 
-0.36103900502767217, -0.27414798362995207, -0.66035498229658951], [-0.21628583186162206, 0.55462525106866323, 
-0.68403213647001693, 0.50571682237127791], [-0.65877677118519107, -0.065319213521102537, 0.29595138600434567, 
0.513719778951374]], [[0.86625013769371306, -0.088954984258416803, 0.12932900957920046, -1.2069418165268366], 
[1.092729358658767, -0.62366347314555226, -0.80623313812206476, 1.1718734078259914], [-0.040230469659354465, 
-0.30816751348698795, -0.87095714062265983, 0.96737705491606174]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acos_float_rank0(self):
      arg=0.985740349094
      res=acos(arg)
      ref=0.169077910884
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acos_array_rank0(self):
      arg=numpy.array(0.826889176675)
      res=acos(arg)
      ref=numpy.array(0.597243033529)
      if not isinstance(res,float):
         self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
         self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acos_array_rank1(self):
      arg=numpy.array([-0.63213218712299135, -0.33089431344776521])
      res=acos(arg)
      ref=numpy.array([2.2550981644146875, 1.9080474441017223])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acos_array_rank2(self):
      arg=numpy.array([[-0.72843761973291299, 0.75132013167527423, 0.7305296658484246, -0.73186213125202215, 
0.33036465901341705], [-0.011208176384803981, -0.65633642180145313, 0.38054117728107384, -0.83534325169323898, 
-0.35013777170836047], [0.37919962966099341, 0.35129785438693784, 0.14766002332027672, 0.2656843305563632, 
0.26215051655943089], [-0.7790508472377522, 0.24408953678161049, -0.45074521189155492, 0.72586676838512587, 
0.95378041543165537]])
      res=acos(arg)
      ref=numpy.array([[2.3868350301985446, 0.72073613147570303, 0.75169906338238779, 2.3918468733230376, 1.2341064264134605], 
[1.5820047378608315, 2.2867489279353692, 1.1804148948913278, 2.5595532181308576, 1.9285145086775217], [1.1818651545651737, 
1.211839377066489, 1.4225943856005618, 1.3018826316034773, 1.3055463439220196], [2.4639468218018798, 1.324215606594773, 
2.0383963185742955, 0.75850262955101466, 0.30522149600461151]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acos_array_rank3(self):
      arg=numpy.array([[[0.28353692373664341, 0.61542192939104523], [0.25144155378928756, -0.25189993190960069]], 
[[0.94890847616218266, 0.70730421473080574], [-0.57715871575168876, 0.047216865889484927]], [[0.1937662730632963, 
0.79855371079478288], [0.77627184321710674, 0.086071303453728154]], [[-0.11492784335801087, -0.66261875443081841], 
[-0.48456164639434052, -0.18465393027718757]], [[-0.22634860309881022, 0.86740198797227008], [-0.26660949969627068, 
-0.24538144471295276]], [[-0.38788806426699496, 0.92171103803670973], [-0.39914180837375146, 0.62549283059295724]]])
      res=acos(arg)
      ref=numpy.array([[[1.2833159319955465, 0.90787516310583949], [1.3166269544767646, 1.8254393217715685]], 
[[0.32103771641723783, 0.78511891120696886], [2.1860414508284478, 1.5235618988119928]], [[1.375796593753503, 
0.64590773190377249], [0.68206620715851407, 1.4846183942534523]], [[1.6859786882046819, 2.2951062389220485], 
[2.0766583102325242, 1.7565160524662036]], [[1.7991236776292714, 0.52083900779847314], [1.8406698082845645, 
1.8187094743119285]], [[1.9691354795040374, 0.39832742167247659], [1.981377000145516, 0.89503330575333595]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acos_array_rank4(self):
      arg=numpy.array([[[[0.4605237049668125, 0.76196633583491491, 0.93510863936086652, -0.69820035968358751], 
[0.21097150692514655, -0.043621169146923822, -0.20763689909699656, 0.93225841623824435], [0.073277069608374434, 
-0.58861463416529336, 0.73927953678614866, -0.8289511534076508]], [[0.47532928484625048, 0.22944970439796419, 
-0.42905693495264019, -0.43308250083072042], [0.75523763102948771, 0.48849004497139625, 0.20086723429722753, 
-0.58044831503011873], [-0.56095902399592679, 0.84178187446480002, 0.32300794089709495, 0.09626694212411091]]], 
[[[0.65629908287470529, -0.76523354444005598, 0.85403151394732757, -0.84695311747879232], [-0.76679205459617394, 
-0.14520678538474674, -0.63600998222059424, 0.64867214798534878], [0.6775278006294092, 0.21642053094323344, 
0.96297464460226823, 0.45356109665772526]], [[-0.76901973544603275, 0.7332356543780969, -0.4770314077435972, 
0.42419333225758371], [-0.46792264020313468, 0.55407930669892003, -0.46235498652038054, -0.47207273783195491], 
[0.066253675527875089, -0.25918037754367318, -0.75735583573299514, -0.4974204559370714]]], [[[0.94978754550583888, 
-0.67695081420073167, -0.85685440601700802, 0.65392505071621665], [-0.64720060324861994, 0.58829412385566227, 
-0.58437449998834157, 0.078133777555144945], [0.97457697870363202, 0.68147996192198979, 0.66613566730491591, 
-0.13177598801009816]], [[-0.074483700182634616, -0.12718838574249325, -0.20630161796885882, 0.092900764686704607], 
[0.48745463135409994, -0.55617308916961572, -0.073204267220728814, -0.41814418719676072], [-0.60704289927616728, 
0.090976254866284023, -0.28913855901190699, -0.15322476619247993]]]])
      res=acos(arg)
      ref=numpy.array([[[[1.0922112264002402, 0.70445234602339968, 0.36223080812629399, 2.3436769249154126], 
[1.3582275965513071, 1.6144313415738889, 1.7799549146864544, 0.37019056830208641], [1.4974535210005628, 2.2001404121417258, 
0.73879648878197346, 2.548026182971693]], [[1.0754580639059634, 1.33928406128202, 2.0142447964896868, 2.0187061601339602], 
[0.71477973718762533, 1.0604378834551216, 1.3685532086175525, 2.1900754641307225], [2.1663401331271253, 0.57022066925151105, 
1.2418902459273333, 1.4743800716564994]]], [[[0.85489321497054882, 2.4422003649590081, 0.54711002291044963, 2.581024384402709], 
[2.4446248372338433, 1.7165182966079997, 2.2601129502851682, 0.86495791213688389], [0.82640018144123251, 1.3526497165739733, 
0.27296929486518567, 1.1000392999644573]], [[2.4481025403701864, 0.74772801168251413, 2.0680702557537685, 1.132725410127422], 
[2.0577350705169941, 0.98353975413701233, 2.0514456071228038, 2.0624368455559856], [1.5044940846405901, 1.8329698122219216, 
2.430050636165173, 2.0914190538366961]]], [[[0.31824012547968422, 2.3144082704072089, 2.5999333746678372, 0.85803541969229691], 
[2.2747028047825673, 0.94184865240084659, 2.1949053604869202, 1.4925828303917394], [0.22597114542923219, 0.82101333440479185, 
0.84178085420026694, 1.7029567058764927]], [[1.6453490698637583, 1.6983301518107314, 1.7785900813130933, 1.4777614096231593], 
[1.0616241158522846, 2.160570174047852, 1.6440661341493474, 2.0021976936961061], [2.2231304325420114, 1.4796941053234045, 
1.8641231648849237, 1.7246270815009563]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acos_constData_rank0(self):
      arg=Data(0.488701715793,self.functionspace)
      res=acos(arg)
      ref=Data(1.06019528363,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acos_constData_rank1(self):
      arg=Data(numpy.array([-0.7069884510260479, 0.066536456228094343]),self.functionspace)
      res=acos(arg)
      ref=Data(numpy.array([2.3560271600734319, 1.5042106785793099]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acos_constData_rank2(self):
      arg=Data(numpy.array([[0.56430292990442688, -0.98903682986190777, -0.3420413002282644, 0.86958069963922946, 
0.086189994221981925], [0.48061085978204687, 0.56992558470396526, -0.39143804022424022, 0.91538699421808523, 
-0.6247607993572315], [-0.30579045720477172, -0.53926308586964256, -0.96199444330867911, 0.86384631615783958, 
-0.3861295045166292], [-0.24830573389812416, -0.64277064528559213, -0.70223592832687909, 0.76791985554719511, 
0.27955652659046315]]),self.functionspace)
      res=acos(arg)
      ref=Data(numpy.array([[0.97120766943480641, 2.9933815827581096, 1.9198846919914925, 0.51644378785517808, 
1.4844992607673155], [1.0694451614217777, 0.96438103744225412, 1.9729901409391144, 0.41432815714697696, 2.2456214752472663], 
[1.8815648715040563, 2.1403581385994435, 2.8650107894440024, 0.52794063840509398, 1.9672283047104602], [1.8217271456511239, 
2.2689058823457233, 2.3493295735451714, 0.69520898269337095, 1.2874641379547431]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acos_constData_rank3(self):
      arg=Data(numpy.array([[[-0.25127641498339215, 0.47924391907707808], [0.9094250810728004, 0.81331707815527476]], 
[[-0.037478053221196239, 0.81219526051698732], [0.51171924435985527, 0.0021205041946392722]], [[0.16799305554640487, 
0.56547267023777792], [-0.26255430534477431, 0.83920615425295941]], [[-0.21798749346619972, 0.19401336601038488], 
[0.48284983878642973, -0.45527594054453502]], [[0.95806523218654926, 0.72278068055937794], [0.51804555776735994, 
0.7853765876520582]], [[-0.96140995213000935, 0.11631607026176338], [0.70463792320602381, 
-0.08196371083534959]]]),self.functionspace)
      res=acos(arg)
      ref=Data(numpy.array([[[1.8247950824773533, 1.0710032694468428], [0.4288968174324721, 0.62096551684740176]], 
[[1.6082831592065068, 0.62289104232028281], [1.0336116334699639, 1.5686758210110996]], [[1.4020028903514798, 
0.96979009027549468], [1.836464755730844, 0.57497452883712163]], [[1.7905482239837283, 1.3755447210967584], 
[1.0668901798666217, 2.0434784399142201]], [[0.29062407883675673, 0.7629787395042269], [1.026231915240418, 
0.66749207157886759]], [[2.8628783455342024, 1.4542163650659583], [0.78888358880502207, 
1.6528520889014222]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acos_constData_rank4(self):
      arg=Data(numpy.array([[[[-0.61506559019902962, -0.5745855795832846, -0.84628969134001653, -0.90588684384827134], 
[-0.34626018727959984, 0.21239810890652722, -0.89665370333061689, 0.49626317852593038], [0.053027849671565264, 
0.41476120419267581, 0.079246066899021139, 0.93764126103549206]], [[0.41591907659775873, -0.59733290277491746, 
-0.48495991625159296, -0.83767028966123103], [0.88436245938456315, -0.15432833326657736, 0.83512803661055357, 
-0.63830525301626739], [-0.82989438905878299, 0.072141070800046281, -0.72738350214592196, -0.30730913275609717]]], 
[[[0.40689716920808761, -0.71104587249807949, 0.038185725737092335, 0.72519525154690023], [0.16994966241130927, 
-0.65065670682723398, -0.9817125619282292, 0.24716251884795093], [0.24671804065054359, -0.31467773612302219, 
0.22546160861260089, -0.3587941989414849]], [[-0.12795500126836257, -0.87113152042082509, -0.47568884271877887, 
-0.81079637242974734], [-0.75665137722701226, -0.98769953006858657, 0.62512005540456372, -0.68341880150951506], 
[0.83252830304410286, 0.46952863639918307, 0.49031203045876337, 0.26584809064004156]]], [[[0.35259748760084042, 
-0.42818629651415374, -0.2775668738584729, 0.066202539052023779], [-0.94748831807720502, 0.5703896988471937, 
-0.41797854998348039, 0.89047050044501885], [0.89570103719563954, 0.26870565772981547, -0.049108042983056377, 
-0.7678497253743336]], [[0.9271467814771237, -0.5831398014900766, 0.21610298757442514, 0.49550005126441743], 
[-0.87632122307813454, -0.095018178911907558, 0.33126598775779859, 0.40088099997632765], [0.68662416943361415, 
-0.31774086596728757, -0.84047685768629388, 0.097968380312837056]]]]),self.functionspace)
      res=acos(arg)
      ref=Data(numpy.array([[[[2.2332654766472384, 2.1828940335193545, 2.5797778005152967, 2.7042653654927959], 
[1.9243780643320196, 1.3567679151065177, 2.6829487352607524, 1.0515071129448936], [1.5177435936546868, 1.1431159725994684, 
1.4914670812075259, 0.35501503976615867]], [[1.1418431182805178, 2.2109677155385556, 2.0771136596081119, 2.5637999997054917], 
[0.48566985343049385, 1.7257439328591164, 0.58243081234726091, 2.2630909878313621], [2.5497146906214776, 1.4985925346207736, 
2.3852976801199297, 1.883160362796465]]], [[[1.1517415817131644, 2.3617808392114039, 1.5326013148794557, 0.75947840290709789], 
[1.4000177385630743, 2.2792452459775898, 2.9500544073035377, 1.3210455058981643], [1.321504189200869, 1.890913450179291, 
1.3433795083710702, 1.937772084653625]], [[1.6991030827030646, 2.6282982474051981, 2.0665433124916572, 2.516307719502028], 
[2.4289725107533333, 2.9845845652547225, 0.89551099038517956, 2.3232318769732614], [0.58714030331920919, 1.0820394952604777, 
1.0583485908023083, 1.301712762607544]]], [[[1.2104509090741409, 2.0132811508655828, 1.8520568629978282, 1.5045453336331587], 
[2.8160844121687294, 0.96381610230997872, 2.0020153579539635, 0.4724182272406488], [0.46079103843953362, 1.2987473104137255, 
1.6199241293874687, 2.4462741930788185]], [[0.38407138871666829, 2.1933846612100569, 1.3529749566134648, 1.0523858968930593], 
[2.6389680175419801, 1.6659580676204133, 1.233151320896297, 1.1583180290803612], [0.81396094577727263, 1.8941422534606247, 
2.5689590083594278, 1.47267055219761]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acos_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(0.512310362971)+(1.-msk_arg)*(0.0862731439845)
      res=acos(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(1.03292346732)+(1.-msk_ref)*(1.48441580012)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acos_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([-0.090954384914536557, -0.68894829295793558])+(1.-msk_arg)*numpy.array([0.91418720592286107, 
0.84919614238108254])
      res=acos(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([1.6618765872656689, 2.3308333673386792])+(1.-msk_ref)*numpy.array([0.41729841612792568, 
0.55633513527397027])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acos_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[0.2804453407421259, 0.77763643772123725, -0.50891824703132826, -0.015399320487557766, 
-0.20651117857675527], [0.97588762362229819, -0.50360155484903713, 0.92192780249910689, 0.097606395857343653, 
-0.19386259923194082], [0.54971562849321631, -0.049469889075480888, 0.10690412480636202, -0.85184852479768358, 
0.10040115565114571], [0.040602640127320511, -0.65459682248570683, 0.090235567222326507, -0.68149527434267732, 
0.60825008223382726]])+(1.-msk_arg)*numpy.array([[0.34136624659615156, 0.75926309009511228, 0.77789894882021904, 
-0.066915699206772206, 0.93543046173981526], [-0.41010424930381473, 0.049565144609317047, 0.82019592131588825, 
0.26245784982848397, -0.65352513542162505], [-0.0078117768329551973, 0.79866127006155652, -0.72801744849359751, 
0.83420484116591287, -0.051374346873333865], [-0.95406083950882437, 0.75373063881370528, 0.67464131689396067, 
0.2753555135311001, 0.4517157493233297]])
      res=acos(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[1.286538289575955, 0.67989865828519735, 2.1047239879846535, 1.5861962559775056, 
1.7788042538375517], [0.22004502009943872, 2.0985588364087358, 0.39776820198487428, 1.4730342798824598, 1.7658942478252626], 
[0.9887725483012112, 1.6202864158192103, 1.4636875221119119, 2.5903007032171494, 1.4702257208543619], [1.5301825223044656, 
2.284445505059038, 1.48043785214544, 2.3206002426619823, 0.91694223653884477]])+(1.-msk_ref)*numpy.array([[1.2224262507178765, 
0.70861630308412027, 0.67948101395893024, 1.6377620650861413, 0.36132154092385393], [1.993364689647263, 1.5212108652389638, 
0.60904292180911768, 1.3052278589778357, 2.2830287571767944], [1.5786081830808261, 0.64572901814839645, 2.3862219417068564, 
0.58410704766047927, 1.6221932994714208], [2.8373057216055884, 0.71707586986534644, 0.83031765708968386, 1.2918368345268203, 
1.1021087852439346]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acos_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[-0.37586391282974818, 0.62470558872234805], [0.091302019840561632, -0.70798636307288287]], 
[[-0.37913740490435188, -0.3162754990700174], [-0.84728557794292914, -0.36970690721335442]], [[-0.10396238492584042, 
0.53284031294139389], [0.2546849214514959, 0.28665280820048045]], [[-0.055930271789216501, -0.35282792246465755], 
[0.070288522535513875, 0.56604590299217228]], [[-0.077693143121295494, 0.31210727536482596], [-0.59677252539507886, 
0.98219318593775795]], [[0.12324920601952205, -0.85346551275143934], [0.16535454799763261, 
-0.36883708728965514]]])+(1.-msk_arg)*numpy.array([[[-0.86966268727301044, -0.034919001731702815], [-0.58629732963526315, 
-0.77305630377201351]], [[0.23223965293963689, -0.55192572795012351], [-0.099691450122294301, 0.087663451356910338]], 
[[0.61838189516748576, -0.97950134325935756], [-0.22548412529394968, 0.28948848533220728]], [[0.70545350471541868, 
-0.64335891330722283], [0.88499058116939566, -0.26110567672575735]], [[0.77783347375492196, 0.90646573079083481], 
[-0.72080605785848495, -0.40721365883308991]], [[-0.18061072813696843, -0.489949285529519], [-0.69795568161587407, 
-0.22451539207761495]]])
      res=acos(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[1.9561251970070277, 0.89604188521688577], [1.4793669789025623, 2.3574391817750775]], 
[[1.9596602528361542, 1.8925971966694082], [2.5816500126164552, 1.9494898853123985]], [[1.6749469024493095, 
1.0088428069892188], [1.3132744604280966, 1.2800651354804158]], [[1.6267557998446578, 1.9313880062094919], [1.5004497987434593, 
0.96909490069994553]], [[1.6485678452031067, 1.2533860249105842], [2.210269166536222, 0.18899689709531206]], 
[[1.4472329348908231, 2.5933955995277178], [1.4046788295486632, 
1.9485539120237449]]])+(1.-msk_ref)*numpy.array([[[2.6253149271826195, 1.6057224287598164], [2.197276910076932, 
2.454441538223969]], [[1.3364166650095359, 2.1554681263892284], [1.6706536484888048, 1.4830202048367507]], 
[[0.9041142734798493, 2.9387672380782237], [1.7982362570422947, 1.2771039289999349]], [[0.78773352457340773, 
2.2696740450698036], [0.48432254519423468, 1.8349637633431017]], [[0.6795852022923633, 0.43595843634506076], 
[2.3757608567492108, 1.9901975667604868]], [[1.7524036823873717, 2.0828279033653345], [2.343335206121457, 1.7972420306407804]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acos_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[0.66221360632939441, -0.94251020652369566, -0.16594505003236304, -0.28503661164691907], 
[-0.066660859930379623, 0.28940811790412768, -0.75965868120338509, 0.59561410130815418], [-0.35297887870138689, 
-0.49809296419620158, -0.66719582366744257, 0.2888487191876552]], [[-0.9048544980725487, 0.78255262272828352, 
0.80716885940726457, -0.66711335230692681], [0.5719879950138449, -0.44392524878649109, 0.69420201942549764, 
-0.91987103797060366], [0.860697806067376, -0.097043843288551979, -0.65829452140047806, -0.075882212682259809]]], 
[[[-0.32982341080643829, -0.084866377288239314, 0.73033870298349712, 0.24821712499013882], [-0.81265372732308061, 
-0.84980911303706097, 0.58035909859198065, 0.68610226518717066], [-0.68481952195819296, 0.94699290376008172, 
0.40549290505719537, 0.46254221605299084]], [[-0.24798791324688474, 0.47117774066354889, -0.0016065208147061361, 
-0.36287063706994049], [0.13776897554884404, 0.92285226754586858, 0.075242619053802251, 0.33937185140995907], 
[-0.23675803548744545, -0.665851865880836, -0.13666010140032614, -0.11100463558549523]]], [[[0.37345959320309485, 
0.010491830978333283, 0.011525360014400077, -0.67516524575806636], [-0.17725357932420827, 0.71837839638216816, 
0.60129571297724071, -0.28312984496459193], [-0.78481350011419393, -0.16467516133226845, 0.62221885864428916, 
0.69592699892503629]], [[0.47769545340613684, -0.95744218798053149, -0.49683202632976137, -0.39114595690081333], 
[0.001244053049042404, -0.72191272228374548, 0.26053610625112911, 0.51841530534848657], [0.50576954948152753, 
-0.37248029627795387, 0.54411867943943903, 0.60294784384061839]]]])+(1.-msk_arg)*numpy.array([[[[0.52221341827102896, 
-0.60780868184235026, -0.34092413388118004, -0.23230852702474913], [-0.30548243576056, 0.52398282562525855, 
-0.19009250373883924, 0.41513151951491367], [0.94052918838634003, 0.26610936986107503, -0.04773529617513339, 
-0.39546462752067268]], [[0.53220756167580685, 0.35147292845038658, 0.96278515472351089, 0.953629757401252], 
[0.47928671316086935, 0.86223938287985447, 0.052350510890903479, 0.024615906909640772], [-0.1780157125313685, 
0.96802327240458097, 0.36421959064049347, -0.85231711536313515]]], [[[0.58412035000030249, -0.46750797178866166, 
-0.69382695817302276, 0.1453579221490815], [-0.37632488054907598, -0.34931346621012316, -0.40656475816581883, 
-0.91286095122651101], [0.25994571003325273, 0.55724454897911868, -0.2898841587730876, -0.65425567645610261]], 
[[0.30371493818768558, 0.82003183707366545, -0.45132628164553112, 0.51849673592669077], [-0.55738608493292752, 
0.37835120518731835, 0.16971053506112765, 0.14161802418133851], [-0.38528689194246568, 0.37073070584817169, 
0.84343267812664191, 0.34535656622260547]]], [[[0.43678944471691783, 0.92214005160463497, 0.089375488650982993, 
-0.10764991846375271], [0.9364986794087744, 0.6468038954106734, 0.44127038837700172, 0.5420692591338383], [0.4840716452132483, 
-0.54001284392825766, -0.72412701125398871, 0.5709486193583333]], [[-0.36094179884816935, 0.51648181058402409, 
-0.80111461540034723, -0.6013063152120085], [0.48722302729807487, -0.98789987861842987, 0.17796473593310957, 
0.88963369698139227], [-0.67302249976645145, -0.48662292523098893, 0.10667393342862663, -0.98513665981343168]]]])
      res=acos(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.84702723515587619, 2.8008603726019876, 1.7375125983683439, 1.8598409509035099], 
[1.6375066555229352, 1.2771878905141356, 2.4335844349495459, 0.93276639352923663], [1.9315493428866961, 2.0921944433718616, 
2.3012341472409212, 1.277772246915398]], [[2.7018341462054916, 0.67204096636285038, 0.63145599653959406, 2.301123435434639], 
[0.96186890394569635, 2.0307708170902243, 0.80348566434158086, 2.7385478856900369], [0.53415761573052334, 1.6679931377533226, 
2.2893472042745899, 1.6467515518456204]]], [[[1.906912839461848, 1.6557649078901451, 0.75197866454715023, 1.3199569805741449], 
[2.5194879142989217, 2.5864193627024221, 0.95162674769728761, 0.81467857779881891], [2.3251523298812491, 0.327053887444546, 
1.1532783351219611, 1.0899358804390948]], [[1.8213990633833639, 1.0801707753327729, 1.5724028483006509, 1.9421429926271587], 
[1.4325877704145551, 0.39537481527303237, 1.4954825291773979, 1.2245472895643386], [1.8098239786074506, 2.2994313467450143, 
1.7078854203610079, 1.6820302028463818]]], [[[1.1880606622511485, 1.5603043033194948, 1.5592707116054447, 2.3119850175059198], 
[1.7489914633989034, 0.76932787633977706, 0.92567459115910511, 1.8578522485394942], [2.4731914601206637, 1.7362249942498398, 
0.89922244680123919, 0.80108634954699232]], [[1.0727666891844105, 2.8488021258164755, 2.0907408881141678, 1.9726727510297446], 
[1.5695522734249572, 2.3773587882689151, 1.3072188819926083, 1.0257995837992171], [1.0405225376301781, 1.9524765363267209, 
0.99545800161588049, 0.9236052990200041]]]])+(1.-msk_ref)*numpy.array([[[[1.0213520058097405, 2.2240944372897569, 
1.9186960755023263, 1.8052467993125372], [1.8812413699551784, 1.0192758997479741, 1.7620526942277395, 1.1427089637461607], 
[0.34661161349687786, 1.301441720136945, 1.6185497703241629, 1.9773699878222433]], [[1.0095903733015195, 1.2116523782578037, 
0.27367129134220664, 0.30572244820038885], [1.0709545107103615, 0.53112187415254197, 1.5184218746075275, 1.5461779332349248], 
[1.7497659130830958, 0.25356898233737313, 1.1980016278635097, 2.5911959734801822]]], [[[0.94700044937846861, 
2.0572659244683966, 2.3375860426257709, 1.4249215994963234], [1.9566226899952648, 1.9276346414360024, 1.9894872036943363, 
2.7210338780904983], [1.3078303470341091, 0.97973266366845091, 1.8649021239193764, 2.2839943287845035]], [[1.2622069619200633, 
0.6093296819449785, 2.0390473728493759, 1.0257043549730003], [2.1620304533627004, 1.1827818860098001, 1.4002603908368234, 
1.4287006047461364], [1.9663150231147959, 1.1910006587093609, 0.56715526238387481, 1.2181776246838893]]], 
[[[1.1187697723925423, 0.39721990909757504, 1.4813014201460846, 1.6786552539835249], [0.35828761768088968, 0.86741012952585794, 
1.1137824723799394, 0.99789874233059173], [1.0654944133468176, 2.1412486964122239, 2.3805640658149647, 0.96313547140266142]], 
[[1.9400738994041635, 1.028059094469765, 2.4999515446017484, 2.2159313314358133], [1.0618893478966709, 2.9858710906398382, 
1.3918785442946064, 0.47425389410173352], [2.309084102143752, 2.0790162536749586, 1.4639190373633402, 2.9689641122733712]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atan_float_rank0(self):
      arg=44.9393163302
      res=atan(arg)
      ref=1.54854776859
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atan_array_rank0(self):
      arg=numpy.array(-21.3588712779)
      res=atan(arg)
      ref=numpy.array(-1.5240115377)
      if not isinstance(res,float):
         self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
         self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atan_array_rank1(self):
      arg=numpy.array([64.659395784790092, 50.751545798162624])
      res=atan(arg)
      ref=numpy.array([1.5553319032496276, 1.5510950428097428])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atan_array_rank2(self):
      arg=numpy.array([[-19.345979702683962, 27.7956741940298, 72.666853463162084, 89.015783187682274, 1.914561175913974], 
[-29.880971483935113, -32.439773694919836, -2.2300197916426185, -58.78797361957109, 34.421827800340168], [-69.105011015373762, 
-43.690515564887747, 23.517614349173058, -28.680900758571141, 99.244908881856816], [62.389263071731904, -50.697564225633208, 
-2.4907789862864291, 60.922845826536076, 19.281942417125308]])
      res=atan(arg)
      ref=numpy.array([[-1.5191519639610569, 1.5348350155532802, 1.557035762881066, 1.5595628365073395, 1.089458151101522], 
[-1.5373426981443643, -1.5399797300101612, -1.1492516836176661, -1.5537876853112542, 1.5417531620474145], [-1.5563266060988914, 
-1.5479120596318676, 1.5283006063280371, -1.5359440367088459, 1.5607205841614402], [1.5547693002290608, -1.5510740707828783, 
-1.1890140294197289, 1.5543835970174962, 1.5189807541224212]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atan_array_rank3(self):
      arg=numpy.array([[[48.591223797236211, -60.407409664124586], [55.42520498832377, -48.267357460855109]], 
[[5.6903603977796564, -65.7636692561384], [-44.072918417550589, -92.278010607272094]], [[35.431379763276283, 
92.755665039211408], [-38.139384841990001, -0.42133864948894484]], [[-0.47317187494142843, 50.559557908487108], 
[-47.848861820211596, 7.6441437272139581]], [[-3.3887693786038113, 74.706033098156013], [31.954703360495813, 
-19.709825824410032]], [[-29.055355038619268, 49.978915760572932], [5.2980397084653674, -19.280720220496733]]])
      res=atan(arg)
      ref=numpy.array([[[1.5502193834496876, -1.5542435781713584], [1.5527559512508307, -1.5500813547761727]], 
[[1.3968368202042565, -1.5555915344191245], [-1.5481105487289648, -1.5599599330469556]], [[1.5425802479905968, 
1.5600157317890007], [-1.5445827173583029, -0.39876536778996424]], [[-0.44195569401201562, 1.5510202513390183], 
[-1.5499002300274034, 1.4407159203348561]], [[-1.2838480009126936, 1.557411326435403], [1.5395122389393303, 
-1.5201036783370705]], [[-1.5363928430204041, 1.5507905589430886], [1.3842420139244003, -1.5189774754318326]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atan_array_rank4(self):
      arg=numpy.array([[[[-89.442950755915618, -9.0762811033631579, -7.0454555566523851, -13.147771060991261], 
[6.0944259236431151, -79.300702340918178, 72.374150902049507, -81.510672955512689], [-58.535649934880695, -98.160904270913647, 
87.239218036804743, -94.376814705699204]], [[41.687519837184624, 34.803890735024851, 68.625522071027774, -10.940910715896351], 
[-65.672541485671275, -78.188721259702575, 77.804811504092214, -8.5965672845346575], [-61.808521432469689, -15.897237585865895, 
13.048247043938659, 91.546119201454815]]], [[[69.40339294692518, -34.74825460092373, 7.800817649310261, 19.204443641086513], 
[36.781133506428347, 54.28143878984605, -74.021734945457411, -3.0372103275579718], [9.973784999344204, -20.111342719241847, 
54.047691446843288, -1.6135689486826266]], [[-69.845443188333476, 25.951262759248635, 8.7322367436408257, 4.6592412064706537], 
[93.567647035323262, -9.1575495967150005, -11.230628210551743, 87.470886223671613], [85.083728526065386, 87.067233308755874, 
35.369004542883175, 69.463950506918479]]], [[[-10.34056020168947, 33.46225204200465, -24.847260894791006, -59.109071961961021], 
[9.9710069753514006, 42.523042135448804, 67.711457203952875, -27.639404284628057], [9.737755874999678, 61.244149680548674, 
-61.938963304632331, 45.330763610056607]], [[18.040232820029928, 41.973090108002367, -31.623440803959312, -16.117089191252163], 
[54.422883289341826, 99.27589483877091, 65.925973018150671, -21.055334647009772], [50.393253951063969, -23.981532862985915, 
93.941046702532276, 6.780962535872078]]]])
      res=atan(arg)
      ref=numpy.array([[[[-1.5596164816732538, -1.461061638974225, -1.4298026342807506, -1.4948839640536289], 
[1.4081612076083487, -1.5581867663549249, 1.5569801181413903, -1.5585286102172273], [-1.5537143821177253, -1.5606093239803602, 
1.5593340942364142, -1.5602009005578581]], [[1.5468129317131352, 1.5420718097615786, 1.5562255199091977, -1.4796495069964613], 
[-1.5555704393798147, -1.5580074556522689, 1.5579443588719621, -1.454991297069079], [-1.5546187393491513, -1.5079750884693934, 
1.4943071984104082, 1.559873305288288]]], [[[1.5563888062923836, -1.5420258437105194, 1.4432999892296154, 1.5187720307489923], 
[1.5436151711093875, 1.5523759069280521, -1.5572876030199483, -1.2527257088649826], [1.4708674444093528, -1.5211140602920303, 
1.5522962597898045, -1.0159853925779969]], [[-1.5564799786675656, 1.5322816117586471, 1.4567748411849246, 1.3593765010727061], 
[1.56010927886331, -1.4620277808550317, -1.4819883238774678, 1.5593644495133814], [1.5590437393637295, 1.5593114547691729, 
1.5425305138321885, 1.5564013648023072]]], [[[-1.4743895586017877, 1.5409207982349873, -1.530572150057002, 
-1.5538800637181975], [1.4708397882570008, 1.5472839987375624, 1.5560288509872104, -1.5346318713043459], [1.4684619901052813, 
1.5544696874246791, -1.5546528030091864, 1.5487398302327025]], [[1.5154213396082203, 1.5469760444428287, -1.5391847483235195, 
-1.5088298196273806], [1.552423770618264, 1.5607237287858999, 1.5556289614902006, -1.5233380867581676], [1.5509550049483409, 
-1.5291217176563028, 1.5601517548442514, 1.4243799671358919]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atan_constData_rank0(self):
      arg=Data(77.1815475061,self.functionspace)
      res=atan(arg)
      ref=Data(1.55784058697,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atan_constData_rank1(self):
      arg=Data(numpy.array([24.127270900853333, 53.780128807479542]),self.functionspace)
      res=atan(arg)
      ref=Data(numpy.array([1.5293731596626932, 1.5522042408771073]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atan_constData_rank2(self):
      arg=Data(numpy.array([[75.066387999598078, -47.643583044234546, 99.162875214966277, -18.723008836579183, 
-95.743635891661796], [96.211541201427821, -27.847659208175884, 63.381003621292422, 47.734503447322254, -89.087971144685184], 
[96.740225385202592, 56.248126285276811, 56.630152580852155, 24.16393055846855, -59.980362473980023], [46.378736029545195, 
18.802526973583156, -80.609815333851003, 28.693222841438256, -81.257715706022225]]),self.functionspace)
      res=atan(arg)
      ref=Data(numpy.array([[1.5574755732798313, -1.5498102227640487, 1.5607122494484384, -1.5174368080814036, 
-1.5603521481282565], [1.5604029376053998, -1.5349020891427179, 1.5550200377220329, 1.5498501835119196, -1.5595719382364186], 
[1.5604597332971082, 1.5530198295282804, 1.5531397245479179, 1.5294359320064643, -1.5541257479425095], [1.5492380619260975, 
1.5176620464764465, -1.5583915259365355, 1.5359589916461613, -1.5584904243659321]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atan_constData_rank3(self):
      arg=Data(numpy.array([[[25.272363409809245, 72.710114255454272], [-73.916525013030395, 11.822717990243106]], 
[[-7.9567089480108422, -88.078886114129261], [-59.401029497479009, 10.144426127914841]], [[5.8669614003095063, 
43.606134108185074], [69.709483087677597, -42.605180979447724]], [[-43.097498031782337, -9.679455010215591], 
[21.035923477893689, 68.6961260439559]], [[-91.031239659782216, -1.2208283925224634], [15.250609642430319, 
-42.532509035449138]], [[13.071398406025111, 97.016272209092278], [59.455456659615635, 
-8.7278589133672284]]]),self.functionspace)
      res=atan(arg)
      ref=Data(numpy.array([[[1.5312480434052784, 1.5570439490574493], [-1.5572683775673588, 1.4864142558717994]], 
[[-1.4457717485345001, -1.5594433558340857], [-1.5539631918274199, 1.4725374754036429]], [[1.4019727460075528, 
1.5478677921959694], [1.5564520601503382, -1.5473293116727294]], [[-1.5475972864623277, -1.4678499472573472], 
[1.5232943598733732, 1.556240493264109]], [[-1.5598115288125254, -0.88450752197576599], [1.5053189119572299, 
-1.5472892301914358]], [[1.4944421454789365, 1.5604891426050933], [1.5539785984359276, 
-1.4567181436228189]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atan_constData_rank4(self):
      arg=Data(numpy.array([[[[-73.815492295086131, -92.500705558825771, 87.748960455022598, 93.869799859814776], 
[-68.796518711549481, 19.459631681379363, -50.169905794359202, -87.325365197342592], [-83.711161222625122, -79.114184980026408, 
97.077076710901508, 42.313681127800123]], [[59.543879948626454, -50.509108559523241, -10.284047733615836, 78.275475104885828], 
[31.29091315766081, 12.62909728852874, 36.248365949548798, -44.559136320242246], [51.160352394820393, 6.3621112110461127, 
-25.978291616523649, -69.408432020921907]]], [[[49.68584363169154, -9.6460711402444019, -70.474457925281399, 
-62.451927153649642], [-30.496406405855907, 1.6682876497169872, 30.235431456702315, 19.793086487669925], [-34.845954112412699, 
76.4987668335271, -18.170251109050611, -7.3957876503398126]], [[40.466196086470461, -60.574804857195666, -71.361552214051756, 
-85.773458179168131], [-30.74668658449437, 90.208188193051996, 54.181437977268985, 49.1283236787352], [-86.604405396153567, 
65.178789234154152, -65.760215730859869, 79.899645042192361]]], [[[-62.769285895686643, -75.277005948442763, 
77.727830035600476, -50.309740670989456], [-71.761746783409208, 31.612120067975837, -79.588343231201677, 58.612388999815863], 
[34.684376857908944, 88.391856821826991, 31.195885412178427, 48.037977520490216]], [[73.384550352684755, -99.068631804654188, 
-96.682332656221377, 22.829902353358193], [73.344567819447292, 92.222156476908793, 61.038612492870158, 65.13117809093913], 
[-39.260121441170106, 21.366310632200936, -0.55261871246865724, 88.344433566551317]]]]),self.functionspace)
      res=atan(arg)
      ref=Data(numpy.array([[[[-1.5572498638604184, -1.5599860195717461, 1.5594006737014565, 1.5601436762631382], 
[-1.5562617311557778, 1.5194530562019364, -1.5508666979873209, -1.5593454008301859], [-1.5588510566327916, -1.5581570415747426, 
1.5604955980822841, 1.5471677068655716]], [[1.5540035682413356, -1.5510005038189301, -1.4738630914786084, 1.5580216282009003], 
[1.5388490403060773, 1.4917789700660136, 1.5432158717743245, -1.5483580066014826], [1.551252429447717, 1.4148913624735471, 
-1.5323216444591623, -1.5563898521381769]]], [[[1.5506725866112558, -1.4674961873298999, -1.5566077409836079, 
-1.5547853789512975], [-1.5380173230435497, 1.0308056036111648, 1.5377345986833393, 1.5203165568056236], [-1.5421064647717848, 
1.5577249651606764, -1.5158167776794742, -1.4363992982962024]], [[1.5460893711004231, -1.5542893124754382, -1.5567840958304537, 
-1.559138236785127], [-1.5382839576640661, 1.5597113126679629, 1.5523419167053862, 1.5504442803027987], [-1.559250083200814, 
1.5554551159561958, -1.5555907360330348, 1.5582812800743679]]], [[[-1.5548663158686198, -1.5575128390990942, 
1.5579316317154102, -1.5509220773698864], [-1.5568622278594864, 1.5391734353257136, -1.5582323338673307, 1.5537367425435882], 
[1.5419728868200167, 1.5594835500759663, 1.5387517901757588, 1.5499824698864477]], [[1.5571703237042165, -1.5607026571758496, 
-1.5604535442558731, 1.5270221057996427], [1.55716289664247, 1.5599533705250226, 1.5544147200509701, 1.5554439032331284], 
[-1.5453306942953691, 1.5240278035525368, -0.50485151526843219, 1.5594774779006402]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atan_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(28.6230744709)+(1.-msk_arg)*(98.5640531084)
      res=atan(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(1.53587368291)+(1.-msk_ref)*(1.56065098822)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atan_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([15.120049440768895, 59.552438066657544])+(1.-msk_arg)*numpy.array([52.891203656552477, 
33.825939592184994])
      res=atan(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([1.5047551561658299, 1.5540059810291873])+(1.-msk_ref)*numpy.array([1.5518918436003104, 
1.5412418239818249])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atan_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[-76.573088506216536, 7.4752246848007786, 37.352472317776233, -14.429628586894935, 
9.2461911796326746], [15.877163349576634, -69.731025577972062, 98.462119397040425, 0.11086967101496725, 66.952552972891937], 
[21.4613988176026, -54.928766777193914, -7.7698409134170845, 57.293866613665671, -41.720062354240909], [-72.760335772018465, 
-24.61590982299748, 51.635067291854057, 26.171365038108533, 
-28.850522289435517]])+(1.-msk_arg)*numpy.array([[-2.9979196308199931, 17.928449691641319, 94.084138402514611, 
-30.706956428885945, -39.048688915545867], [81.455444225558637, 41.076133491728314, -12.348027684747322, -96.998584622607737, 
3.5494500007671661], [11.928348043289731, 29.617524133858467, -35.191570440387949, -5.5723015603326616, -8.6801462526229045], 
[-53.226075765935434, 81.324735529017204, 73.64439022653616, -63.410920204307629, 82.578775778502859]])
      res=atan(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-1.5577376507527787, 1.4378106287674151, 1.5440307306372216, -1.5016051015403054, 
1.463062437233243], [1.5078958698601415, -1.5564564910167438, 1.5606404858936544, 0.1104187191535006, 1.5558614870511973], 
[1.5242347190023855, -1.5525929408132795, -1.4427972108798626, 1.5533442235989503, -1.5468316320959736], [-1.5570534401939278, 
-1.5301945190809678, 1.5514320641385182, 1.5326052059257838, 
-1.5361487801364981]])+(1.-msk_ref)*numpy.array([[-1.2488376055640988, 1.5150767908604279, 1.5601679428556545, 
-1.5382419212729004, -1.5451928685007317], [1.5585202932487556, 1.5464560974462378, -1.4899880859926042, -1.5604872632344087, 
1.296180605863227], [1.4871579958148731, 1.5370453536464654, -1.542388075516685, -1.3932274015265744, -1.4560965540045021], 
[-1.5520107531779834, 1.5585005646539951, 1.5572183944805211, -1.5550274795696868, 1.558687269550189]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atan_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[-99.737276053966312, -48.356572199033245], [-46.543844781704344, 6.1259139416056456]], 
[[17.531330241371037, -68.076100025644365], [34.396809149103206, -42.739378052101664]], [[-69.901002539280711, 
-94.463586283456991], [33.925925982756581, -85.860630072851365]], [[-62.962532551429781, -7.3984681684334817], 
[1.5075059375863304, -49.036071243781798]], [[-81.754588035205671, 65.093527638906664], [42.963896911418999, 
51.734739152726718]], [[-49.020680649492611, -22.838913258247871], [49.004459500225579, 
76.342316315937097]]])+(1.-msk_arg)*numpy.array([[[-42.403013939001077, 54.642707383431059], [-97.145600635971149, 
-16.246689875098625]], [[99.311269190139114, 9.0171199802705502], [64.898476270374118, -29.039009382359367]], 
[[32.392343847543202, -52.386935943203184], [-31.895152513240205, -76.450552368784287]], [[39.846439433659185, 
82.933673871252665], [-34.611126003877658, 54.69675631686502]], [[-33.191937042133077, 91.814713073171191], 
[12.801345428982543, -72.185557226881315]], [[13.739705291094211, -29.048383232189295], [-79.964704298940319, 
-75.430544754497646]]])
      res=atan(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-1.5607703211487765, -1.5501195616535675], [-1.5493145137389956, 1.4089826212208438]], 
[[1.5138173326161095, -1.5561079400871647], [1.5417320492801183, -1.5474029686668451]], [[-1.5564913561647646, 
-1.5602106324892369], [1.5413288764382662, -1.5591500718467173]], [[-1.5549152005439371, -1.4364474073842823], 
[0.98509526524340307, -1.5504060022368278]], [[-1.5585652073339196, 1.5554350247039193], [1.5475251724114438, 
1.5514693619207949]], [[-1.5503996022401041, -1.52703935451643], [1.5503928525174224, 
1.5576981807016601]]])+(1.-msk_ref)*numpy.array([[[-1.5472174681385134, 1.552497665697705], [-1.5605028634291704, 
-1.5093228802853307]], [[1.5607273163700399, 1.4603474942982249], [1.5553888638429108, -1.5363734930655313]], 
[[1.5399346360439679, -1.5517099162482095], [-1.5394538672091014, -1.5577167224883579]], [[1.5457052483615532, 
1.5587390828093173], [-1.5419119188905757, 1.5525157436011672]], [[-1.5406776385730307, 1.5599052568590541], 
[1.4928378541404648, -1.5569440262029417]], [[1.4981426678331333, -1.5363845924701713], [-1.5582914612503311, 
-1.5575398744555249]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atan_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[7.471089114324613, 70.328976721663992, 91.135317485739535, -50.316614879212864], 
[-38.538554940368485, 41.026797041508161, -16.239214006993663, -39.572872500906044], [-74.54280888287002, 75.062639286125858, 
0.99547023184534567, 15.811608938675548]], [[40.606309118514929, 23.85663909327711, -45.132198979563313, 93.932967299681081], 
[-69.486170815875312, -50.383134509931573, -29.734176537280902, -90.025151324879289], [91.884098711206121, -50.386185022722252, 
-74.487832254660674, -44.309746966834936]]], [[[97.374283771484727, -66.608064038971463, 20.556858659059401, 
-75.191258905758858], [-10.256368763591524, -7.2861826654478676, 77.732211999338432, 30.599293579406776], [55.916279138906987, 
8.5083275857011245, -48.495588516870214, -69.09225826810605]], [[74.267020946021631, 42.452692352010985, -29.40484298542178, 
-85.89313887753039], [-39.649700676935204, -72.029896988320829, 20.256364767222095, -56.492181896586736], [-59.634793343404802, 
21.648445583603433, 12.210114044640804, -89.176350179376968]]], [[[-5.5991749272061782, 21.329511183984323, 6.0926073268593939, 
-94.393056620920817], [3.3116391941678103, 30.093617974227726, -71.897305369955319, 67.417049193001901], [36.365218857801182, 
23.118265360742242, 95.407153008307034, 2.7225248744416888]], [[80.718414649984396, 23.173759008791123, -7.0086824829779886, 
-97.528512728265042], [-25.090833274499147, -87.762952095286437, -81.216220936006508, -42.927107797260632], 
[33.348026549453522, 3.2907832520295415, 79.737697162730228, 
37.885498122078133]]]])+(1.-msk_arg)*numpy.array([[[[-63.004023819034472, -2.049652340368894, 13.190600755775677, 
37.175483016803781], [61.430771949859604, 89.362799499122417, -16.415132676351391, -68.917476266466338], [-52.541729778765877, 
2.9212104116696764, 78.88664706246476, 81.155895691801447]], [[59.584817779511297, 21.126360435622189, -52.787453990724153, 
34.253669429770639], [-31.619817034204885, 14.11543299407883, -69.786851140046053, 23.971700498307882], [42.782352614931341, 
-81.356718244078905, 80.573119461737491, -25.533831260854896]]], [[[-75.805548024930431, 53.104822974458017, 
-93.399330417793934, -50.394284117981194], [-58.14368385677335, 4.1508752532025426, 38.215255790502056, -88.395128321106114], 
[71.089913321867044, -33.168846041598243, -52.964062231899376, 33.057220685581456]], [[96.968768206307374, -95.903342115446605, 
-31.909536969887881, -40.547910578314884], [-76.297242962934348, 26.145592184621449, 93.812237807833355, 50.827305831279517], 
[-90.949731839773847, -1.2650248380202527, -99.019545756177749, 64.954388539210413]]], [[[78.805529185098521, 
-58.246018609649639, -87.263711664097542, 5.814815671712509], [73.487399951373732, -76.342128274633211, -64.022937188891206, 
-63.55305715525801], [31.124532850429318, -60.910366380188741, -20.616496245036856, -72.31471158193574]], 
[[-59.363841356629287, 24.814059041715893, 42.83305739367762, 49.568370342543034], [99.850581964437396, -86.94854964394041, 
-60.735234289466987, 97.594810737717751], [40.735595282229923, 80.925214481671617, 69.321783061786761, 35.596345133481123]]]])
      res=atan(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[1.4377378809945247, 1.5565783946920098, 1.5598240726003174, -1.5509247918562272], 
[-1.544854107151185, 1.5464268388761395, -1.5092946517234498, -1.5455318674958753], [-1.5573820212023413, 1.5574749081049686, 
0.78312814187483493, 1.5076357782094583]], [[1.5461745885263589, 1.5289037981371612, -1.5486428216553989, 1.5601508393479102], 
[-1.5564059673920825, -1.5509510208694877, -1.5371776647071664, -1.5596887767485348], [1.5599134805251607, -1.5509522220413452, 
-1.5573721218039416, -1.5482317593401189]]], [[[1.5605270358872381, -1.5557842574222749, 1.5221890790800454, 
-1.5574976925835151], [-1.4736031388779884, -1.4344023824539049, 1.5579323568511914, 1.53812746074776], [1.5529143535162204, 
1.453801159649831, -1.5501788163645516, -1.5563239357139196]], [[1.5573322136652075, 1.547245050062495, -1.5368014248996946, 
-1.5591544793188135], [-1.5455808010019469, -1.5569140945247899, 1.5214691733856718, -1.5530966108766573], 
[-1.5540291640430552, 1.5246364466139044, 1.4890793896011576, -1.5595830613361006]]], [[[-1.3940619703968145, 
1.5239472325159973, 1.4081135141449859, -1.5602027235429896], [1.2775375154612196, 1.5375789125601389, -1.5568884964790235, 
1.5559643713485154], [1.5433044517475791, 1.5275674342113457, 1.5603153161892209, 1.2187879940259665]], [[1.5584082137639628, 
1.5276708249529203, -1.4290727110253891, -1.5605432743276089], [-1.5309622162924836, -1.5594024903003154, -1.5584841377030505, 
-1.5475052359249772], [1.5408185281257161, 1.2757846057947446, 1.558255864579823, 
1.5444071301661151]]]])+(1.-msk_ref)*numpy.array([[[[-1.5549256572948229, -1.1168854906239309, 1.4951295091481112, 
1.5439033632048849], [1.5545192778079766, 1.5596064550863848, -1.5099521222861714, -1.5562872373453143], [-1.5517661333618258, 
1.2409762461903, 1.5581205891567347, 1.5584749866652237]], [[1.5540151035624956, 1.5234974007763187, -1.5518546971757805, 
1.5416106638802107], [-1.5391811279113796, 1.5000700485459202, -1.5564679605095357, 1.5291046439660905], [1.5474264585662092, 
-1.5585053978411494, 1.5583858769277745, -1.5316526043374064]]], [[[-1.557605445314644, 1.5519678707159599, 
-1.5600900210118984, -1.5509554104409442], [-1.5535992496248576, 1.334388206478472, 1.5446347369800053, -1.559483968725548], 
[1.5567305612900715, -1.5406566836778355, -1.5519178428571008, 1.5405549721326295]], [[1.5604840935793878, -1.5603695393818466, 
-1.5394679867888108, -1.546139141785815], [-1.557690443727713, 1.5325675959179943, 1.5601371404208322, 1.5511244007420097], 
[-1.5598016851901375, -0.9018760093171112, -1.5606976538739674, 1.555402124379679]]], [[[1.5581075428814881, 
-1.5536294579145471, -1.5593373112331161, 1.400487761069096], [1.5571893916785078, -1.5576981484427903, -1.5551781947019521, 
-1.5550627409037603], [1.5386783789701497, -1.5543802349462992, -1.5223294654398811, -1.5569687633091833]], 
[[-1.5539526487994846, 1.5305183872196484, 1.5474541132197985, 1.5506249077892615], [1.5607816974440236, -1.5592957794461904, 
-1.554332907134087, 1.5605502389333508], [1.5462527012877434, 1.5584398676633486, 1.5563718472205175, 1.5427109418260911]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sinh_float_rank0(self):
      arg=4.62054860905
      res=sinh(arg)
      ref=50.7699402792
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sinh_array_rank0(self):
      arg=numpy.array(2.05667480474)
      res=sinh(arg)
      ref=numpy.array(3.84602263608)
      if not isinstance(res,float):
         self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
         self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sinh_array_rank1(self):
      arg=numpy.array([-0.57299002361149398, 4.9701708508385654])
      res=sinh(arg)
      ref=numpy.array([-0.6048625470928668, 72.022277312851116])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sinh_array_rank2(self):
      arg=numpy.array([[3.6806932839026612, -1.4763292012289053, -4.3936702751486969, -3.0685976107319157, 3.2536285659124147], 
[-4.4095798168419371, -2.6144539630149879, 1.4410911915399787, -0.91715742174741433, 1.285974035391229], [1.6504732439114713, 
1.4296161835932377, -0.83808417692926795, -1.1986415642205728, 1.4438095972866583], [-4.907261588977204, -1.8936881350111214, 
0.62331381585711121, -3.7556965094494754, 2.2731655121682595]])
      res=sinh(arg)
      ref=numpy.array([[19.824342157621622, -2.0741873929957602, -40.462290011551872, -10.732613707052465, 12.922729043574737], 
[-41.111371117944849, -6.7932739287290032, 1.9943172760742081, -1.0512570948280311, 1.6709046198129101], [2.5087427690880735, 
1.9688474279020707, -0.93969753927909105, -1.507003088089597, 2.0003893791900094], [-67.630546672484869, -3.2466556097316586, 
0.66446683296690445, -21.37030540447623, 4.8035519806053335]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sinh_array_rank3(self):
      arg=numpy.array([[[0.57558125067040677, 2.6488701019751346], [-0.94123085641171578, 0.14874936109775838]], 
[[3.5154837836427042, 4.1522824187969203], [-2.7763707487596778, -3.5754913430690092]], [[-2.0295270640192631, 
3.3540665218001422], [-1.7128292970757633, 4.4547643869409317]], [[-1.4365986390192589, 0.77300449065532373], 
[2.272193134553091, -3.3481291932146071]], [[2.1824694818873027, -2.103879329425189], [-0.080040229811302765, 
-0.83170008251496785]], [[3.7337182427542999, -3.1467333469181047], [-3.3843902376309156, -0.74228681140730757]]])
      res=sinh(arg)
      ref=numpy.array([[[0.60789294711217534, 7.033661984918508], [-1.0864935017499182, 0.1492985155793059]], 
[[16.801230639766963, 31.781610064849605], [-7.9991814643697721, -17.842080882796203]], [[-3.7395442928147746, 
14.291967323955285], [-2.6821359548586048, 43.012127111270928]], [[-1.9843145434106473, 0.85232048578214947], 
[4.7987832434057767, -14.20715516610351]], [[4.3777081634408459, -4.0379642014059876], [-0.080125719324351308, 
-0.93095614552616213]], [[20.905232943393834, -11.608482894620423], [-14.733050467727624, -0.8123549578553696]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sinh_array_rank4(self):
      arg=numpy.array([[[[-1.8086582690314765, 1.39060486960134, 2.020725041947288, 0.29806182610831211], 
[-0.90573907424410738, -4.5980087007476946, 1.4993604973028605, -2.9888705719344988], [-4.6566597830640957, 
-1.5930957089459561, 2.964126955656865, 4.4423449003001245]], [[3.5904416017474592, 1.6747795677026378, 4.7218385023099216, 
4.8037225859645201], [0.54556039543419743, -0.16459548088977538, 3.0781387898621944, -1.3751574759805618], 
[-0.1361046360535445, 3.8960872464981566, 0.29486508533061784, 1.0143412603084743]]], [[[-3.3123285989151352, 
0.057612623691935205, -0.94240316200511565, 1.1886844652424209], [-0.39608611020759543, -4.2269788913375113, 
-1.8920952877787323, 3.2852579622543914], [-2.2708582338273917, 2.545081749744428, -4.5439024406060451, -1.6007533952128759]], 
[[0.49952067223890229, 2.9996285721058946, 3.8335883755464586, 1.6685766348145004], [-2.2573108015501075, -1.8050532135727124, 
-1.3045066155030138, -1.7204337250428514], [-4.5451936070022123, 3.3917459302062554, -0.32167433053103345, 
-3.9599150377642856]]], [[[2.2137287039231976, 0.77024373917856437, -2.8796152675487576, 4.2307034867917857], 
[4.0786213711543855, -2.7561249749471339, 3.9503844313151451, -3.9056297043252606], [-4.7562970841582182, -2.420085805999157, 
-1.3409791848332961, 1.2642726263021213]], [[3.4364481643749105, -0.95561538128355394, -2.9282688513030939, -1.40166028556445], 
[-4.9701506300903873, 4.0569445577269327, -3.0071817785187518, 3.3110511019131774], [-2.86903477304958, 0.50348566954613094, 
4.8491770644363932, -2.8549802820407812]]]])
      res=sinh(arg)
      ref=numpy.array([[[[-2.9691902446807963, 1.8841772781189157, 3.7056165990814112, 0.3024948163307673], 
[-1.0347582652447369, -49.638168918522005, 2.1277755180975699, -9.9064457285986887], [-52.637162614298219, -2.3578288679041957, 
9.6630863456380407, 42.48109770080066]], [[18.111247172821027, 2.5751344815427903, 56.182882571036622, 60.977697147674753], 
[0.5730290911041912, -0.16533968273097588, 10.835949112121765, -1.8514499093861441], [-0.1365252365202691, 24.594602952849424, 
0.29915656049286315, 1.1974525282016266]]], [[[-13.706268754028031, 0.057644500423540593, -1.0882253230306314, 
1.4890690199219379], [-0.40652426294480121, -34.247673402981455, -3.2412485512326437, 13.339222092678304], 
[-4.7922440084715996, 6.332901623598274, -47.023252703202353, -2.3775104769692659]], [[0.52055486290594588, 10.014136207856676, 
23.103246491408616, 2.5580484101645218], [-4.7263607385259849, -2.9579146447974853, -1.7072818667325922, -2.7039813200969438], 
[-47.084020487761762, 14.842071159280765, -0.32725061097147184, -26.216902256364982]]], [[[4.5202386396949903, 
0.84869625497928236, -8.8756322025137688, 34.375524345082184], [29.523527794424748, -7.8375995651853874, 25.968044476653894, 
-24.830613176244185], [-58.152913786734722, -5.5789551653496003, -1.7805977437680296, 1.629035928215584]], 
[[15.522102465167844, -1.1078474185501597, -9.3208743365655344, -1.907875233414877], [-72.020820842870791, 28.890071227909356, 
-10.090437619412899, 13.688723675517242], [-8.781624503724883, 0.52502901056844919, 63.813738078181551, -8.6582688332870443]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sinh_constData_rank0(self):
      arg=Data(-2.5657517285,self.functionspace)
      res=sinh(arg)
      ref=Data(-6.46678681426,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sinh_constData_rank1(self):
      arg=Data(numpy.array([1.1951913612740679, -1.3968858996198898]),self.functionspace)
      res=sinh(arg)
      ref=Data(numpy.array([1.5007719850531684, -1.8976126105016544]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sinh_constData_rank2(self):
      arg=Data(numpy.array([[2.9770279415500216, 1.7818636224413753, 4.3506744978116192, 0.3155572034638352, 
-2.3029367265386327], [-0.11993319399341562, 2.6857959326922929, 4.2109593790749873, -3.2040673460201399, 1.2577958899821411], 
[1.5868875135233314, 0.17819888395057326, -2.8845536156052543, -0.5945474834818123, 2.6685386599884584], [4.5823461838410111, 
-3.7661133399884203, -0.68696806955082312, -1.1111846895786726, -2.7810040165747942]]),self.functionspace)
      res=sinh(arg)
      ref=Data(numpy.array([[9.7892230758553609, 2.8862967869232636, 38.758920748662248, 0.32082034514838775, 
-4.9517760554609822], [-0.12022092011217191, 7.3008530250561883, 33.703179570386084, -12.295959469707883, 1.6166898534944676], 
[2.3419742439740063, 0.17914349541120622, -8.9198488889924992, -0.63019924529477911, 7.1747646992966505], [48.866609318112609, 
-21.594323333839892, -0.74229038016238635, -1.3543932651030797, -8.0366182941930546]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sinh_constData_rank3(self):
      arg=Data(numpy.array([[[-4.3802430708794402, 4.9005320124584024], [-4.2425192555513433, 4.3111478206038392]], 
[[3.880548263128091, 1.4341087171719682], [-3.5355229312811698, 3.0606487402799125]], [[-0.90726945620558741, 
3.5990377183086188], [-3.2341576671144177, -4.0933603799747811]], [[-2.3094794473310767, 0.74730852876805898], 
[3.0700184357300202, -1.6461977960389849]], [[-2.3355173128039217, -4.821113804751449], [-3.4522520329650974, 
-3.681432729284432]], [[-2.3871712850706315, -0.51799335858846263], [3.0104680965835424, 
-3.7262089557179436]]]),self.functionspace)
      res=sinh(arg)
      ref=Data(numpy.array([[[-39.922459875190185, 67.176899953095202], [-34.784278526191635, 37.256282092161399]], 
[[24.215066028041921, 1.9787879590622455], [-17.141904913593617, 10.647270204626349]], [[-1.0369616994349651, 
18.267841771904756], [-12.672793396893498, -29.962147518146278]], [[-4.9849344076868478, 0.81883510335371068], 
[10.747939760153812, -2.4972189533501159]], [[-5.1190222297287669, -62.047591400659577], [-15.769868972780067, 
-19.839025235072505]], [[-5.3953885782207864, -0.54147053612911866], [10.123814999959565, 
-20.748657990635127]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sinh_constData_rank4(self):
      arg=Data(numpy.array([[[[0.78892972729009436, -3.139922608777741, 3.6510878042457602, -1.2595049097810405], 
[4.4603290555941104, 1.8597590456468112, 3.5974905261408612, 2.6580657442379216], [-0.44612689815487716, 4.6594579342132576, 
3.4540258453851429, 2.9715841434203751]], [[-4.14364066046166, -3.8892374742486178, -2.7219443285590517, 2.2147405609622597], 
[1.2018440212795287, 0.48460940304713773, 1.2139357644421196, 3.7367171409701445], [-4.7403368299256519, -4.1697742881625235, 
-2.4543979901826929, 2.2824134079724097]]], [[[2.6115639398614316, -2.5135399865228147, -2.5472731083490641, 
-0.37771857440259282], [-0.85431180067155221, 3.7309801835051299, 2.7512806562564593, -3.1449752488933691], 
[-1.4484468001528428, -2.2897091084260337, 0.39876760605970318, 3.4929247099803167]], [[4.1779388003502387, 
-3.3169515851373066, -4.3146395799959212, -1.2167662535572865], [2.0685360031034978, -3.505111140969249, 4.190593726189265, 
-2.2083500699947498], [0.63892969367560681, 2.4791765066356666, -2.5576686573379481, -4.197608367045607]]], 
[[[3.4152452657396406, -3.9489074158380344, 2.8390880510982219, 4.0075809792544081], [3.3904728060630624, -2.9524191574209677, 
-1.3740060584983582, 4.9068504335830667], [1.9285646825264182, -4.0066312524956116, -3.925108248006989, 1.4182955199725997]], 
[[-3.4384713388796326, 4.5181071240802471, 2.9862225585272917, 3.3904358137110506], [3.7666081179571744, 0.28141999812769392, 
-0.28290165793087851, -0.10010227370964131], [1.7877178611185869, -1.0563226761013444, 2.4799227116148836, 
3.2561748429742288]]]]),self.functionspace)
      res=sinh(arg)
      ref=Data(numpy.array([[[[0.87335432985454431, -11.529396371837118, 19.24528942768201, -1.6199410110301038], 
[43.252207209304025, 3.1332394928338956, 18.239557447967837, 7.0992897475622954], [-0.46107358074185661, 52.784682186220351, 
15.797922770305087, 9.7357999799236268]], [[-31.508008461802877, -24.426571993521392, -7.5720598474439269, 4.5249253780668939], 
[1.5128028110983349, 0.503801500864545, 1.5348416168631165, 20.968091392813321], [-57.232008666409868, -32.342695481943238, 
-5.7767547450294474, 4.8491331861900733]]], [[[6.773457979590308, -6.1337927392073723, -6.3469664473175502, 
-0.38676446468335113], [-0.96209037579084766, 20.8480060183653, 7.7994157538254107, -11.588016388859527], [-2.0107816401041081, 
-4.8853848195334031, 0.40942033034978198, 16.425782829771663]], [[32.607964913195694, -13.769947758523358, -37.386646304103749, 
-1.5400328533252909], [3.893429518625795, -16.627549777894171, 33.023432382435878, -4.4954033518162637], [0.68329752955599155, 
5.9238113488375141, -6.4141047225228123, -33.256000449117735]]], [[[15.195771116937493, -25.929689156153675, 
8.5208424885397758, 27.497726523251767], [14.823144542918863, -9.5500083745996136, -1.8490282662596134, 67.60274268450803], 
[3.367136009302333, -27.471606330370332, -25.319410725211963, 1.9439742212176185]], [[-15.553603279701219, 45.825509675085875, 
9.8801147162858687, 14.822594963706324], [21.605021822983019, 0.28514934861235797, -0.28669038188405588, -0.10026953604836293], 
[2.9042288290549307, -1.2640220870915277, 5.9282959171248137, 12.955774192507791]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sinh_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(1.02674745587)+(1.-msk_arg)*(3.69865941758)
      res=sinh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(1.2169000064)+(1.-msk_ref)*(20.1841805257)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sinh_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([-3.1431830989496734, 4.6891324289258378])+(1.-msk_arg)*numpy.array([-1.2046366514798832, 
4.8411071879799383])
      res=sinh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-11.567190339695394, 54.374794115752316])+(1.-msk_ref)*numpy.array([-1.5178729899438197, 
63.300778158843258])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sinh_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[3.7727489767833493, -4.5886059679871094, -4.1991703482297975, -0.43926557142501288, 
-1.0884934269923296], [4.6885903672243554, -1.9112278253002679, 1.0612197554152285, 2.9865814162420907, -0.49009335031272805], 
[-1.8181758539140516, 2.7478984245834761, -3.2531506059766935, 3.8048256472758553, -0.88145263770013571], [3.1460548423241299, 
-2.6306504475411154, -4.9199047997205962, -3.5891331937957203, 
0.95425219682215623]])+(1.-msk_arg)*numpy.array([[2.7968784735891052, -4.3741407840581665, 4.4026875993616059, 
-4.9288157946675444, 4.0314388779315991], [-0.076278667719481774, -0.27600539649420419, 1.1277228638470005, 1.8605251302958408, 
4.5612894191480269], [-0.093755180748743783, 2.2321319722766075, 0.35760544645317527, -1.1417129632969725, 
-4.8786493784601621], [-1.5674041273061712, -3.9657107214259337, -0.42119705464689616, -3.5803016413271527, 
2.5061937268904577]])
      res=sinh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[21.738245452674651, -49.173527205228737, -33.308009764864849, -0.45352884552208983, 
-1.3165365375612967], [54.34532262531684, -3.3067434114495136, 1.2719301636791487, 9.8836790222577324, -0.50994969815464275], 
[-2.9991443806785316, 7.772864940729626, -12.916535507273551, 22.447577438808032, -1.0001117976692304], [11.600579986631331, 
-6.9053827014659408, -68.491135647234728, -18.087529674908627, 
1.1058139996497129]])+(1.-msk_ref)*numpy.array([[8.1661971134451647, -39.679506963896756, 40.82891300561824, 
-69.104252253480865, 28.162086662933127], [-0.076352659655629665, -0.27952306998107268, 1.3824228017234343, 3.1357600263359933, 
47.848178957164635], [-0.093892592997541796, 4.6062074299158802, 0.36527619266751105, -1.4064286710439649, -65.72269669955773], 
[-2.2928004925868599, -26.369398786294131, -0.43376187410671041, -17.928248065329345, 6.0883025002424862]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sinh_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[-1.303136739676761, -2.7823239757113827], [4.8931862179530139, 1.4774738754507224]], 
[[-2.4548874607777704, -2.2300892560322527], [0.4119505895635811, -3.4926836674010096]], [[1.0075299978071071, 
-0.044204027369439736], [2.8179795665745466, 4.6686018119988262]], [[0.79236361366614272, -1.5875122221704907], 
[1.5975687069538314, -3.5649573286862646]], [[-1.7048108220196898, 2.1472169539047652], [4.0122744502945054, 
-3.3531962362607235]], [[-1.6046076437374812, -4.0646346120008214], [-4.6305939583823905, 
-0.37046614076695317]]])+(1.-msk_arg)*numpy.array([[[2.0695132072285141, -0.016954066391089206], [3.1657136783850408, 
-3.9614359560820533]], [[-1.8683925142770419, 4.0548362962379194], [-1.1996626651097317, -3.3727181389251184]], 
[[-0.76116516025855141, -1.5777011960535239], [-1.6801040051784999, 3.1316186551385368]], [[-0.23815354213756113, 
3.6736179339321815], [-4.1490773312105267, 3.4034906583672502]], [[1.151326232544557, -3.0595502117065232], 
[-2.0858069753870256, 2.7046766283448296]], [[-1.1210863474348209, -2.4270501491589327], [2.3290012925175274, 
-3.6093493326917825]]])
      res=sinh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-1.7045730476109413, -8.0473151122299544], [66.685185604394277, 2.0768245505488694]], 
[[-5.7796250415463479, -4.5965886763363448], [0.4237014170005905, -16.421816663316527]], [[1.1868540148661137, 
-0.044218424525007148], [8.3416309438742626, 53.269638895522085]], [[0.87791860778638553, -2.343565543852741], 
[2.3693083967961788, -17.654823631285023]], [[-2.6592691239286865, 4.2220953829933077], [27.627174964934028, 
-14.279504232742436]], [[-2.3874692504024497, -29.11322692579412], [-51.282611158827883, 
-0.37899859692979027]]])+(1.-msk_ref)*numpy.array([[[3.8973595435324224, -0.016954878616607251], [11.831736093192438, 
-26.256835356905849]], [[-3.1617516606223406, 28.829191086286595], [-1.5088506439879201, -14.561688881576821]], 
[[-0.83682363449580677, -2.3186794493060443], [-2.5898797262802926, 11.433693759053044]], [[-0.24041116502191431, 
19.684394700529776], [-31.67985987513276, 15.017810134949846]], [[1.4230835364327377, -10.635528822023661], 
[-3.9634395206198274, 7.4412948594358639]], [[-1.3711299875399072, -5.6185637662039971], [5.0851445171502316, 
-18.457469290909831]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sinh_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[-1.1956120031013828, 0.020480384571296639, -2.1408598298154624, -1.7496823766374203], 
[4.7182903999426724, 2.0994984782904815, -3.1938651643851337, 4.0438335377971573], [0.95068639331084004, 1.794459420325226, 
4.3374765090612968, 1.8089220458721398]], [[-2.0074642736757364, -0.93930475870105745, 0.38582781909330865, 
-4.4284562113176928], [-2.7153125645725975, 3.1682240643764565, 1.3204304141721455, -3.8804297690033174], [4.0749661007464208, 
-0.80869682585387626, -4.6137126721387665, -2.6663050451966441]]], [[[-0.42109483610852561, -3.7685754036111563, 
4.8844221008209825, -4.0678774803234976], [4.4927386645977485, 2.7267609465270812, -0.059159901845545981, 3.4863176211843694], 
[3.8540725475829429, -1.4738511891772665, -0.066051969247521214, -3.8702560500498819]], [[-2.2843797278800326, 
-2.6294776774924289, 4.9296280437576421, 3.2076860802982772], [0.39912802996856822, -1.7084823211535314, 1.4250788208891363, 
-3.7283793073745661], [4.4624043052602378, -1.1497747358745682, -3.6996333445846807, 2.1454045115480831]]], 
[[[4.5987765981064825, -1.282241230862502, -1.5088437694731063, -0.13104013861338171], [-2.6269576167047379, 
3.5649742087127763, -0.235004087188039, 1.0808919804304642], [4.5294217311902525, 3.8216426668240207, -0.093487548521227204, 
-1.8888434322595158]], [[1.126896256967961, -3.0625210810934336, 1.8652508425716956, 1.928470212128758], [1.9903138371456759, 
0.6164484947775648, 3.1839446074343929, -1.9398977004551843], [1.9193477356678592, 2.3604150261013093, -2.6646690282195196, 
-1.1429482199921037]]]])+(1.-msk_arg)*numpy.array([[[[3.897761904145904, -1.4542350055844322, -0.83040361242227689, 
4.7834859344688425], [-2.5882823627554465, -0.54976371687761461, 1.988976883021401, 2.1365988152983029], [1.4119198786684137, 
-4.218122390413253, 1.6404997511034525, -1.8022421229503429]], [[-3.6561939914405226, 4.5746099316440301, 3.7110383911132878, 
-0.81459575889978808], [-1.7349250941674157, 0.90432296335839446, -2.8595181519238944, -2.4217842979915192], 
[-4.2769549866033838, -1.0280346693472051, 3.3963416040519085, 2.44420030008658]]], [[[2.3658119534953359, -1.9650168272699151, 
1.3070217485385403, -0.90034068472098028], [0.26047738281421218, 2.6840850081318299, 0.95308275860888791, -1.6802945154786739], 
[-1.8733797851138512, -1.9273631507925448, 2.3966655382694624, 1.3701972247481358]], [[-4.1110570128143733, 4.6702211045884052, 
-4.4827239984420855, -0.10124889321244979], [2.6807889393202222, -4.9917580357272424, 2.2672338274698323, -1.1709697004829875], 
[-3.669175248354184, 0.29467071817101775, 2.3689059046111813, -4.2768458599722372]]], [[[-4.2845220579622652, 
-3.9587395683711781, -1.5429058622960081, 1.2463326194245381], [0.37557031525832585, 0.21581378431796594, 4.6486919354968776, 
2.2659188299091113], [1.465372849868178, -3.898457597122349, -3.7403161266913552, 1.0593546626436154]], [[-2.3826515649392199, 
-4.9905936605620118, 3.4901410436949192, 2.3625573801922437], [-2.0130916687585856, 0.66379111863935858, 0.47197149963044183, 
-1.9640532330175264], [0.6501661999953976, 2.1833709465746951, -1.9572268695871422, -3.9688303001077951]]]])
      res=sinh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-1.501530710899408, 0.02048181633774105, -4.1945975590820384, -2.7894730115584774], 
[55.983861604581357, 4.0197787848410078, -12.170737414093134, 28.513538345952682], [1.1005047121467133, 2.9250021568686249, 
38.250571561139559, 2.9700167777380524]], [[-3.6550437632995201, -1.0836513608425915, 0.39547191033222612, 
-41.895005803076394], [-7.5215738564401873, 11.861581529273391, 1.7390062988916495, -24.212194409319796], [29.415746422043231, 
-0.89977115369258009, -50.42399639573491, -7.1586020138617341]]], [[[-0.43365045585121148, -21.647612412021161, 
66.103236687825486, -29.207846210707022], [44.677332097248318, 7.6089362217494747, -0.059194416782373985, 16.317413023338194], 
[23.581820851201702, -2.0684877307239979, -0.066100009004037785, -23.966905136784455]], [[-4.8588781526277831, 
-6.8972045458831737, 69.160410798061548, 12.340682795341275], [0.40980981912733172, -2.6697180784409937, 1.9588480374298647, 
-20.793791048479846], [43.342083525549121, -1.4203867289656362, -20.203872131099995, 4.2142382996652747]]], 
[[[49.676308310220008, -1.6636474064923956, -2.1501671630498622, -0.13141548708188369], [-6.8796633166378678, 
17.655122125365484, -0.23717316011991438, 1.3040072856230609], [46.347075146699282, 22.828645296180618, -0.09362378701940284, 
-3.2302353617786053]], [[1.381012923406312, -10.667311930553934, 3.1513490762423411, 3.3668041977457963], [3.590588738387507, 
0.65623970607267734, 12.050186972429607, -3.4071602393066369], [3.334904116111352, 5.2504834207543132, -7.1467862771242103, 
-1.4085614274706719]]]])+(1.-msk_ref)*numpy.array([[[[24.635859030734867, -2.0238140654222083, -0.92918560604521261, 
59.755948863056695], [-6.6158732675053997, -0.57787868891113314, 3.5856087953070865, 4.1762614433939218], [1.9300758438449881, 
-33.945568773278723, 2.4819316664894542, -2.9491490261873019]], [[-19.343943370189301, 48.489944314855222, 20.435899335234847, 
-0.90772215435572201], [-2.7460448848831613, 1.0327215167313188, -8.6979093992124348, -5.588590047644856], 
[-36.003458296579915, -1.2189284664931614, 14.910592096184065, 5.7172683939126765]]], [[[5.2794058713778194, 
-3.4974395620369241, 1.7122636846380859, -1.0270050159248734], [0.26343288986630808, 7.2882558674617384, 1.1040712235776156, 
-2.5904086745158206], [-3.1783294550751613, -3.3629180673847192, 5.4477301402041904, 1.8410350361248211]], 
[[-30.49739132489028, 53.355983101821046, -44.232024429531251, -0.10142197134686007], [7.2640477532059791, -73.594088233996828, 
4.7745322821384413, -1.457526198759536], [-19.597024321087066, 0.29895368791646082, 5.2960558288771846, -35.999528059624467]]], 
[[[-36.277037497735407, -26.186080785387141, -2.2322027986720072, 1.5950042683928387], [0.38446201805586855, 
0.21749296569934382, 52.219348493891225, 4.7681216785065281], [2.0490826074636868, -24.65301810166217, -21.043776986052105, 
1.2689147218897665]], [[-5.3706426419798809, -73.508439063873681, 16.380037856522499, 5.2619460707927139], 
[-3.6764260460798259, 0.71362284196631431, 0.48969020114152906, -3.4939360214109358], [0.69695009919163586, 4.3817579436686165, 
-3.4692086980710664, -26.451847774946053]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cosh_float_rank0(self):
      arg=1.33144608066
      res=cosh(arg)
      ref=2.0253050594
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cosh_array_rank0(self):
      arg=numpy.array(-3.22296171288)
      res=cosh(arg)
      ref=numpy.array(12.5710965319)
      if not isinstance(res,float):
         self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
         self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cosh_array_rank1(self):
      arg=numpy.array([-3.124143411079582, 1.667936944321772])
      res=cosh(arg)
      ref=numpy.array([11.392191076394347, 2.74492781349739])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cosh_array_rank2(self):
      arg=numpy.array([[-0.4118827236501339, 2.4356115379092493, 0.36303565470659649, -1.428998007076594, 
-0.78127117304184601], [-1.5454270798753855, 4.0874097264539682, 4.0585539164692506, 4.8643399088552002, 0.84820627709083496], 
[-0.68888904527317774, 2.5936370228586583, -1.1169027112109955, -3.2641893241483153, -2.9621244747947051], [4.9674078254801159, 
-4.9189888180570316, -4.5864043001585211, -1.4431331853638163, 2.0939536830741048]])
      res=cosh(arg)
      ref=numpy.array([[1.0860296672101115, 5.7551731242139974, 1.0666243758254017, 2.2070315450757585, 1.3210353628148421], 
[2.4515974912120857, 29.801065965979777, 28.953904975342382, 64.796544568895726, 1.3818181096321629], [1.2468177212237013, 
6.726544121469785, 1.6913340472315268, 13.098561960209436, 9.6953611683278016], [71.830494589790092, 68.435727575714537, 
49.075549571672738, 2.2350636799020163, 4.120071355774817]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cosh_array_rank3(self):
      arg=numpy.array([[[4.7949417234712257, 4.3333706597081427], [-0.57379488029691395, -2.4040814630842853]], 
[[-1.7407940816410692, -4.2074903580853462], [-4.1666438228123219, -2.0934453730681679]], [[-2.4677852578825221, 
-3.240209709220677], [-4.4018078212910545, 3.5291032709620715]], [[4.5170085692510202, -0.58012552663763728], 
[0.63134241461983898, -1.7982340771652385]], [[0.84126156865006685, 1.1687175188458019], [-4.4293017999745796, 
0.42243458993485472]], [[2.4658079668291855, 2.0840081067304483], [2.0354119411930025, 3.176286435475923]]])
      res=cosh(arg)
      ref=numpy.array([[[60.452803790759141, 38.106912027345942], [1.1691867830828482, 5.5793037264776855]], 
[[2.9386252268250264, 33.60129731369058], [32.257061929362344, 4.1180402377879721]], [[5.9405322772852749, 12.789116324335017], 
[40.805252812032904, 17.061356292079893]], [[45.786105142982791, 1.173045364518899], [1.2060050626854528, 3.1022823661607513]], 
[[1.3752286555988582, 1.7643141893120646], [41.942379628471855, 1.0905602739836204]], [[5.928965340918829, 4.0805232711595627], 
[3.8930158568153344, 11.999680179224233]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cosh_array_rank4(self):
      arg=numpy.array([[[[-1.6690197555589825, -3.8116492280090974, -2.7447433776447241, 2.0065825235384747], 
[4.2194306050613992, 1.7285538163969525, -4.5142338880194535, -2.1298537708576171], [3.0674206676824678, 3.8956823485336116, 
0.0682248125217777, -4.8147509066210734]], [[0.99841547002491993, 4.0399882383766226, -1.7127872008141098, -1.670183215365685], 
[4.2177801473885381, 2.1700833895117135, 4.3585832669653453, 2.9726630046621603], [0.41356438747781432, -2.7660861601790723, 
0.17542706127044383, 4.1360124558933649]]], [[[4.3892773320274632, 0.89926128541526129, 3.8350335273254217, 
-4.4302002307532646], [-0.7933685751409767, 3.8306387475795649, 0.283169225590445, 2.00705860160787], [4.1024590382863408, 
4.243480821787486, -1.558916779976002, 2.227593992149334]], [[-4.8358172757705926, 2.3901734753730892, -2.220061426050004, 
-0.020959058082217652], [-0.49298297131854607, -0.34751301688075387, -3.7924426845241186, 0.74032781979609474], 
[1.5109442652287441, -4.6977981024109416, -3.8647857175399092, -4.9568057895060722]]], [[[4.5536952609256218, 
2.4341771240463057, -2.4578655024474463, -1.9330061166977863], [-4.9203549018898629, 3.3607811390612259, -1.7955466098525799, 
2.0108468852649324], [4.2807650984569285, -2.639963637692909, 3.5667969846535019, 3.9584367322321459]], [[3.6624702496382255, 
-4.809363566726149, 1.3025554535824933, -3.8541299844527566], [-2.447622595413713, 3.9571504469866063, -1.215222598119353, 
4.5670313071703976], [-3.7225368767190661, -3.3715861873041031, -0.2464533710960346, 1.8348695506240889]]]])
      res=cosh(arg)
      ref=numpy.array([[[[2.7476974049097338, 22.623537676845487, 7.8124425036845562, 3.7861512650390625], [34.004732205263551, 
2.9050216821069799, 45.659269687915526, 4.2662455376223054], [10.766475795665887, 24.604967933823534, 1.0023282153924855, 
61.662100252034392]], [[1.5412204296518119, 28.421636059374627, 2.8623778422829349, 2.7506768680771714], [33.948679396555278, 
4.4365912581575495, 39.07956997936828, 9.7975313578483814], [1.0867436025875736, 7.979602137938631, 1.0154268290541089, 
31.284438767686826]]], [[[40.29728673928728, 1.4323284735201398, 23.15829020151552, 41.980068173990041], [1.331574902542511, 
23.056832860418506, 1.0403610223963287, 3.7878901903155278], [30.252690615971371, 34.832113347828837, 2.4820165018569504, 
4.6926524404266523]], [[62.97470513300955, 5.5035005597306235, 4.658249445410223, 1.0002196490983304], [1.1239971560221078, 
1.060992777316776, 22.193588657691386, 1.2867901250244131], [2.3758543631708742, 54.857230850023285, 23.857009484069554, 
71.07304166456899]]], [[[47.496637440331817, 5.7470493166781953, 5.8827358999429542, 3.5274822975797018], [68.529270421381526, 
14.423198081403774, 3.0944012978379041, 3.8017579238172221], [36.154782140465535, 7.0420289645244001, 17.715630567032527, 
26.197239052491749]], [[19.491563890591397, 61.330842464912244, 1.9752608079107055, 23.604368606038676], [5.823664119474965, 
26.163588152162625, 1.8338441717501983, 48.134156864495282], [20.696691126204136, 14.579511390203788, 1.0305236627181771, 
3.2119756202545435]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cosh_constData_rank0(self):
      arg=Data(-4.24083627673,self.functionspace)
      res=cosh(arg)
      ref=Data(34.7401579178,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cosh_constData_rank1(self):
      arg=Data(numpy.array([-1.9733026353741332, 3.2399516397570522]),self.functionspace)
      res=cosh(arg)
      ref=Data(numpy.array([3.666697404384252, 12.785826374671135]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cosh_constData_rank2(self):
      arg=Data(numpy.array([[2.0539793908707917, 2.8128119898843718, 1.2141527320457737, -3.243614151246673, 
-4.5362320919854486], [1.2296028833129871, -1.3994300352059472, 2.37351917667851, 0.82153102692914892, 0.53839836990063095], 
[3.0198442347602423, 1.6980775305344924, -1.1381084409768394, -4.9123521967338046, -3.7792775753055152], [-0.15027305906061628, 
-2.2170801519268291, 4.2112858997992575, -2.0616340232335908, -3.7383640756514138]]),self.functionspace)
      res=cosh(arg)
      ref=Data(numpy.array([[3.9635489211250197, 8.358363403230328, 1.8322006226186001, 12.832597024463693, 
46.674579868696753], [1.8561399268366547, 2.1498134298915885, 5.4141284730433741, 1.3568682770373275, 1.1484715005443755], 
[10.268454461146804, 2.823234534718607, 1.7206420197086019, 67.983097846218271, 21.903619280527362], [1.0113122599057027, 
4.6447063816889846, 33.72901828649443, 3.9930233323441375, 21.026485167172385]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cosh_constData_rank3(self):
      arg=Data(numpy.array([[[-0.36177154329420613, 1.4988683958821056], [4.1034695113171171, -1.7699726010526087]], 
[[-2.1029471002977274, -2.6032513600604776], [-2.1352009755962387, -3.4074238341414298]], [[0.99060029190973964, 
3.4724448192705566], [-4.0407926511560603, 2.8454364383593713]], [[2.6335037395322534, 0.99595624934796945], 
[-3.0732465478213342, -3.5206455532238579]], [[1.7827475086733999, 0.53498737966029175], [4.1326101933053501, 
-1.8346394293971535]], [[2.7799164377749896, 0.36372011419116301], [2.9400569003838584, 
-4.4190987814190237]]]),self.functionspace)
      res=cosh(arg)
      ref=Data(numpy.array([[[1.0661561632736485, 2.350001619492684], [30.28325888887721, 3.0205150821088833]], 
[[4.1561840002924653, 6.7908086087120969], [4.2884835830598309, 15.110251371798855]], [[1.5321020935485141, 
16.123224070640664], [28.444493828834755, 8.6335883138613081]], [[6.9971457981271632, 1.5383410174530738], [10.829111916250652, 
16.917912742521455]], [[3.057173300564425, 1.1465516876750839], [31.178236141991025, 3.2112732967707007]], 
[[8.0898588773584752, 1.066878603107577], [9.4848926912684721, 41.516738129835801]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cosh_constData_rank4(self):
      arg=Data(numpy.array([[[[-4.2409219794156749, -4.1069688839162035, 1.3082412185149552, 4.1943544672817907], 
[3.3983975397115671, 3.1865768709158644, 3.0568132716671812, -2.9029054352671513], [0.65039766551508205, 4.76543751632531, 
-4.8234631881249683, 2.3006186468104515]], [[-0.51352308613466491, 4.6209329831393955, 0.85710242205948273, 
1.4796698117867644], [3.63435878513822, 0.45798296596782961, -0.90448622863052819, -1.3024034388555936], [-1.6046507784748352, 
-4.5531553640666296, 0.16257113795205314, -3.5552552776456947]]], [[[-2.5775578394932017, -1.5896527729425625, 
1.6885217897360771, -1.5110436080547651], [0.43298055538480451, 2.884067452216045, -0.70210528747998691, -1.8186180117527662], 
[-0.11998812418442917, -2.6591322975083012, 1.5006767889797423, -3.3950584055228772]], [[-2.6607801216329809, 
1.9651497608404238, 4.0578450139145694, -1.0073514267856911], [4.6473427088186288, -2.7070166857094957, 1.2583395026820385, 
-3.3647179903432036], [1.2789252013032248, -1.8816583425499109, 3.8425866669526858, -1.1280058491098668]]], 
[[[-2.0618340296927364, 2.7633815803321582, -3.0952947605427159, -1.7366320521496159], [-1.5313856774033496, 
-1.9963279906705091, 1.0967687647204318, 0.037199950164313655], [-0.22913397286609349, 1.8807565335991141, 4.4821398097716507, 
-0.333814815960082]], [[2.9917107729337156, -2.1565622419208061, 0.31145584028410767, -0.74161455292077161], 
[4.0458374004588968, -0.80432227096545716, -1.2421846703327399, -1.3321179109587176], [-0.11954985488736281, 
1.3442737904922559, -2.7722305168561858, 1.8082366582307756]]]]),self.functionspace)
      res=cosh(arg)
      ref=Data(numpy.array([[[[34.743134136450763, 30.389359136961577, 1.9849780598949431, 33.162996184543296], 
[14.974774505357111, 12.123370105579067, 10.653368780754152, 9.1409447706893641], [1.2190704717273027, 58.695490979337386, 
62.201594183009171, 5.040275849070242]], [[1.1347761049156291, 50.799306072495348, 1.3903565839545473, 2.3096041427758909], 
[18.951979156745157, 1.1067201625845755, 1.4377060109706092, 1.9750018851118263], [2.5885406263195816, 47.471006760241508, 
1.0132438177609397, 17.512663397890641]]], [[[6.6204536176587645, 2.5530214710574506, 2.7981339878120091, 2.3760684738729165], 
[1.0952096713695527, 8.9713931733915615, 1.2567688251839493, 3.1627923642663403], [1.0072072156987255, 7.1769492478358226, 
2.353851226976039, 14.924966730746183]], [[7.1886699846984641, 3.6380580495021988, 28.933398997277276, 1.5517618150744819], 
[52.158514390814744, 7.5256204404079341, 1.9018489082538677, 14.479955344073044], [1.9355562168863731, 3.3583595516651834, 
23.33370710253033, 1.7065837606103507]]], [[[3.9937965926755647, 7.9582199641211337, 11.069505806354666, 2.9271499421727931], 
[2.4204081939123738, 3.7489031599985223, 1.6642114654953828, 1.0006919979415811], [1.0263662441050192, 3.3554696972909537, 
44.217494713085976, 1.0562354731278587]], [[9.984966487788963, 4.3785509139728607, 1.0488957201954647, 1.2878332289904297], 
[28.588263022798937, 1.3412860859750293, 1.8759618961716029, 2.0264887507809632], [1.0071545990434139, 2.048064603939975, 
8.0283960660376845, 3.1318125444277793]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cosh_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(3.88292275349)+(1.-msk_arg)*(0.322255519995)
      res=cosh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(24.2932723522)+(1.-msk_ref)*(1.05237522413)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cosh_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([-0.36748993195377189, -2.6556479970929328])+(1.-msk_arg)*numpy.array([-4.8059400561651007, 
2.6808889414732695])
      res=cosh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([1.0682877788671881, 7.152230048052421])+(1.-msk_ref)*numpy.array([61.121262592069399, 
7.3332832936048682])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cosh_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[-4.2350734618128261, 3.0775966972614928, 2.1762241845227361, 2.7331150990953654, 
0.86011520741826253], [3.8054452790923694, -3.599020068183989, 1.3015977329259982, 2.5006226881364677, -0.41851978226338371], 
[0.57831113034031034, -3.1330165445813316, 3.2509043185735713, 1.9045706018519368, 2.6480322511947332], [-0.24416697626465034, 
3.5377951168342818, 2.1760491560495767, 1.0008350796838563, 
-1.7852105791806117]])+(1.-msk_arg)*numpy.array([[-3.0431100804627098, -2.6429201434997429, -0.53601965051825928, 
-2.1337263749500632, 1.4700619727218802], [2.1927998477955244, -3.4863452752498891, 0.27229060712039832, -1.6446815214848733, 
-4.1533600742551036], [1.1118973572464323, -0.28412083875394156, 0.68039884981537213, 1.1356801298918828, -4.556167025511467], 
[0.67079036222744914, 2.1024678619974395, 4.4088037831176923, -3.0804994070005898, -4.3070991879278679]])
      res=cosh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[34.540615531256734, 10.876121499050756, 4.4632181903464634, 7.7228706974442156, 
1.3932731483732854], [22.483754064914642, 18.294869373290364, 1.9736303086488172, 6.1360580593292751, 1.0888652499931104], 
[1.1719346756070419, 11.493335108420753, 12.926206154395803, 3.4327047531688994, 7.0985024055090609], [1.029957144419003, 
17.210042374352401, 4.4624569287766871, 1.5440625596104669, 
3.0642983866706377]])+(1.-msk_ref)*numpy.array([[10.509023891408981, 7.0626685833451042, 1.147131273574826, 4.2823387712197167, 
2.2897079599373371], [4.5359346216944267, 16.348477667383587, 1.0373006984274089, 2.6862170861337429, 31.831606637332886], 
[1.6845276921942158, 1.0406345767253475, 1.240540069293381, 1.7172469258189496, 47.6141571362026], [1.2335433944879068, 
4.1542511880486943, 41.09163997398813, 10.907603859242382, 37.119168270354493]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cosh_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[-3.1231934177926366, -0.95840699316385347], [2.6810924499416933, 1.3941911400141072]], 
[[-3.8006774875062534, -4.6515689182263653], [-2.0049398036884645, -3.0124546543952393]], [[0.12374854843647221, 
1.0389217974863101], [-0.39433234438398213, 0.67650066423593636]], [[1.8006940917801231, -0.78690296802687243], 
[-0.69551037332237353, -0.33284068028093294]], [[-0.85379219583044019, -2.8555951774500765], [3.0681255302820052, 
1.0747175826907398]], [[0.39445323354999218, -3.3498299135446974], [2.0391632485289488, 
-4.2963552416313346]]])+(1.-msk_arg)*numpy.array([[[1.1255745951844194, -1.8621313216175817], [-3.8579232989929055, 
3.9124355364903085]], [[-1.1841415266191948, -4.9220286959496153], [-1.9298831136343697, 1.8414030150539906]], 
[[-4.372240580220601, 2.7400775386069585], [-2.088882268394566, -1.7610582480803041]], [[-4.1520921159707855, 
-0.4336829066071477], [-3.4479596966992068, 2.2264039504651301]], [[4.1770864330176298, 0.28396753576443245], 
[-1.6275356011749667, 4.5545515730082951]], [[2.0479796568591944, 0.15394157703128819], [0.015877955542630495, 
-1.9296546874071083]]])
      res=cosh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[11.381415485941703, 1.4955213290935458], [7.3347618899387506, 2.1398728712194552]], 
[[22.376917434690284, 52.379373132527377], [3.780157644371517, 10.19321524144935]], [[1.0076666278386768, 1.5900020862920177], 
[1.0787617199717443, 1.237687731707249]], [[3.1095160640037447, 1.3259177695768152], [1.2517758866481756, 1.0559047202892629]], 
[[1.3871666277018044, 8.7211513671070549], [10.774034551927908, 1.6352798469829397]], [[1.0788106434485849, 
14.266488224538655], [3.907157159997829, 36.722641396328491]]])+(1.-msk_ref)*numpy.array([[[1.7032266062313568, 
3.2963918080547536], [23.693997051786543, 25.020311279660248]], [[1.7869444979544635, 68.644058143142132], [3.5169350820753236, 
3.2319866791074112]], [[39.616778389585214, 7.7763756612698582], [4.0998544548431886, 2.9952272724428073]], 
[[31.791290986886569, 1.0955236372670905], [15.733998974254735, 4.687199581556472]], [[32.595512870024109, 1.0405904439948925], 
[2.6438626198653061, 47.537317788318902]], [[3.940609189276072, 1.0118724228797822], [1.0001260573844286, 3.5161649731416196]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_cosh_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[4.4268561537365105, -0.99633996327074215, -2.7354186303501349, -3.6014039393315778], 
[-0.40945829201719341, 4.1499656361584449, -0.1249574612485862, -0.88459308163129879], [1.7694709819392065, 
-3.7389385796510766, 0.9576164231264519, -4.6474900261076266]], [[0.70645854352401649, -4.8520668460414642, 
-2.5330933539038325, -0.549856898092286], [1.8381015373343175, -1.8529189538661672, -0.87427712481285891, 4.6750446068199683], 
[-3.5352767503987415, 2.045441799582516, 1.2776721016379122, 2.1728225798541629]]], [[[-0.56323580914173732, 
-3.7051661815256809, 1.2543544690069339, -1.6436198111460976], [-3.1086303342246815, -0.35069305734997691, -1.6195095281913452, 
4.2853348445927359], [-4.2137567624764145, 0.10924936849622746, -0.23582664483908911, -3.4634747068464602]], 
[[2.818505277171238, -2.6709571128015073, 0.60753220310522771, 1.8856497868117907], [2.3384261944842244, -3.8991650044075286, 
4.0624497170115976, -1.4520743015602799], [-0.80099419925215631, -2.5277264635204553, 0.84887563890653439, 
4.6308043178127072]]], [[[-4.5155938239725657, -4.319314463897741, -1.7586348790143669, -2.486077325081796], 
[-0.89873097999010998, 4.8390284754075061, -0.1211626176791043, -3.6260601922434854], [-4.0701835796414372, -1.47437113109953, 
-2.1611609550698918, 0.033932330292179458]], [[-0.89649129634609181, -1.7113532713986856, -4.6951489544984497, 
-1.8689282526061515], [-2.552427731526492, -2.5531140718789249, 3.6715490104752977, -1.6816493439517552], [3.3651846580962896, 
4.1391423929523121, 4.1805493499622219, -4.3484643062308539]]]])+(1.-msk_arg)*numpy.array([[[[-1.8196007567202397, 
4.4099981357561369, -4.8456764126230043, -1.5756741956767106], [1.1585261997134975, -0.028832364502445529, 1.2862094746382349, 
4.8688765048720128], [1.6746266716005964, -1.6901669417949905, -2.3713936790090173, 3.5340096469991238]], 
[[-3.8160271228624332, 1.6243636724700909, -0.28845829640247977, 1.7926773513803864], [3.9049300291451221, -2.596711942191956, 
1.0395206592269783, 3.5415783753689585], [-2.4331346970078203, 4.2458970755392773, 2.4435119508535399, 2.6004862182530886]]], 
[[[0.58789796900014757, -4.4275483873668078, -4.8627385861438146, 3.2334936281987545], [3.1577746863758733, 2.3350059274054358, 
-2.7232066521672529, 2.8091005171998624], [-1.8856346109478062, 2.3299051767583538, 0.93272533070350505, -2.7644165606259827]], 
[[-0.17911965915960426, -3.3761658564198074, 3.3130885250881015, -1.2135810779944691], [3.7591622751412164, 
-1.0523576999085238, -4.1413001914442695, -2.3045228388628738], [4.07788372173931, -4.0154821040822162, 1.0687978150348147, 
-1.0135661775004214]]], [[[2.2327912051434717, 4.6495691308844087, -0.15553759505009879, 0.94643632960029223], 
[2.8371944668672366, -3.1289628884919685, 0.88162720207343614, 3.43192458574919], [-3.1951007618980851, -0.26891643435667412, 
3.5540706427473943, -2.3034553204173669]], [[-0.064343492742324315, 3.1951416600776987, -1.450275526567737, 
-2.9511871906285094], [-2.1236353657941964, 1.577933721183614, -3.1867061074145022, -2.3112025656707722], [-3.1749577662793604, 
2.0952384852572337, 4.3808439963857246, 0.79703870380891573]]]])
      res=cosh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[41.839957894910235, 1.5387896811434534, 7.7405313094428516, 18.338468809137268], 
[1.0850058015507766, 31.723792728930643, 1.007817347535561, 1.4174403918176495], [3.0190857585889495, 21.038554768439571, 
1.4946426682365133, 52.166197395425506]], [[1.2600945639349359, 64.006247405726484, 6.3359058422853183, 1.1550186913533884], 
[3.2218575396745703, 3.2675947650031318, 1.4071526504919982, 53.62333780585503], [17.166828937548178, 3.9309485369198072, 
1.9334810742759783, 4.44844785699095]]], [[[1.1628551224852983, 20.340699647349787, 1.8954173098009086, 2.6835716038822715], 
[11.21750908517309, 1.0621256271682211, 2.6243041562668994, 36.320315185399423], [33.812424468101241, 1.0059736502186449, 
1.0279362148482973, 15.979522109153839]], [[8.4057438830957558, 7.2614912315417444, 1.1902942959750151, 3.3711829969901008], 
[5.2306953618190981, 24.690737158008943, 29.066856324000966, 2.2530254504946821], [1.338318561735534, 6.3024190181455282, 
1.3824567466461783, 51.303149041549133]]], [[[45.721390717205928, 37.575205220423015, 2.9883940069155637, 6.0486460818795296], 
[1.431784868763482, 63.177229024876837, 1.0073491740882192, 18.79557426783354], [29.292393849305842, 2.2986052363638692, 
4.3982008084731818, 1.0005757567603555]], [[1.4294934564455519, 2.8585349628531902, 54.712122403040581, 3.3178176207809655], 
[6.4580630496018596, 6.4624435396403683, 19.669095813163231, 2.7802398775453381], [14.486698115728975, 31.382460454320757, 
32.708530956839823, 38.686248847673632]]]])+(1.-msk_ref)*numpy.array([[[[3.1657426601157588, 41.140732667496167, 
63.598573819598698, 2.5204338087367897], [1.749591975889075, 1.0004156814167158, 1.9476793176780096, 65.091133094900329], 
[2.76209025861169, 2.8024371205175291, 5.4028309719525378, 17.145127502478068]], [[22.722701460749384, 2.6361127713385373, 
1.0418933791560909, 3.0860117888529324], [24.833374169499958, 6.7470296913582493, 1.5907426640360278, 17.275165726105953], 
[5.7411529479695185, 34.916343642719674, 5.8001298088654769, 6.7722617078694727]]], [[[1.1778470409566488, 41.868922674178009, 
64.692879780182039, 12.703774344296651], [11.780361267688194, 5.213165532570553, 7.6473711218607532, 8.3276218852101866], 
[3.3711341394041159, 5.1871359824701289, 1.4674529296279082, 7.9663955441826833]], [[1.0160848626105952, 14.646276608546929, 
13.753119716667834, 1.8313232962853927], [21.467882667309649, 1.6067551116635668, 31.450216204778748, 5.0596013290754431], 
[29.518688607164375, 27.73402794970962, 1.6276489527607436, 1.559166109201122]]], [[[4.7165444443869706, 52.274749283052188, 
1.0121203768318836, 1.4823165707257469], [8.563201898689849, 11.447016066701803, 1.4144672229112365, 15.484224693139931], 
[12.226797914685024, 1.0363764507230055, 17.491963418960516, 5.0543095384610668]], [[1.0020707568070522, 12.227296303400523, 
2.2493974724805725, 9.590463606278961], [4.2405377001104023, 2.5256678080441803, 12.124931649576444, 5.0928445274724501], 
[11.983802620165205, 4.1252099479469129, 39.958979825977963, 1.3348108617705192]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tanh_float_rank0(self):
      arg=0.0964706283536
      res=tanh(arg)
      ref=0.0961724676385
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tanh_array_rank0(self):
      arg=numpy.array(-1.1594273991)
      res=tanh(arg)
      ref=numpy.array(-0.820853186665)
      if not isinstance(res,float):
         self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
         self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tanh_array_rank1(self):
      arg=numpy.array([4.0095102506592379, -3.99056605857663])
      res=tanh(arg)
      ref=numpy.array([0.99934193207919786, -0.99931652927269499])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tanh_array_rank2(self):
      arg=numpy.array([[1.351548992744017, 2.0449129798935504, 2.8112467393827902, -2.5178321306129137, 3.8198348090942211], 
[2.0416535387114099, 0.56501340349619955, -4.4424177756608731, -1.2822074018373844, -3.5331502805649584], [-2.2633536901424911, 
3.5731671756862724, 4.1306742234161451, 0.6666546450788271, -4.100943801011284], [-2.2011208973423937, 3.2395332370634318, 
-1.4364780978255896, -4.3642531075688833, -1.3549108509495991]])
      res=tanh(arg)
      ref=numpy.array([[0.87441840333729182, 0.96706707367361455, 0.99279482234020389, -0.9870802471797735, 
0.99903848780463256], [0.96685525599798428, 0.51168769239271461, -0.99972309244228208, -0.85707193069948551, 
-0.99829468664799281], [-0.9785990155129688, 0.99842574731827016, 0.99948351253734291, 0.58277500666799609, 
-0.99945187901482491], [-0.97579679071081338, 0.99693422031776624, -0.89298651729304512, -0.99967624416618217, 
-0.87520743692522673]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tanh_array_rank3(self):
      arg=numpy.array([[[-3.0591224029112443, 0.23109198581316903], [-1.4994946112523233, 2.0691436308560363]], 
[[-4.8503401950417953, -4.4611329722086888], [-0.43747928219825649, -0.45946372541495428]], [[-3.2312319143988146, 
-3.2048024334570879], [-1.9828645867264907, -1.3460565150168149]], [[-2.3096454016957435, 1.6016237079990656], 
[4.2477315885716322, -1.1748931568902221]], [[3.3809571674213892, -0.80000881666223211], [-4.8272192192542809, 
-4.6824030740363218]], [[-3.4687708160885791, 2.6729251410703565], [3.2789743871688373, -1.4826589266887669]]])
      res=tanh(arg)
      ref=numpy.array([[[-0.99560504295624908, 0.2270642937269424], [-0.9050568847538446, 0.96860053300043447]], 
[[-0.99987752387555451, -0.99973326426349807], [-0.4115528471121348, -0.42964703203893384]], [[-0.99688297536967896, 
-0.99671405832263915], [-0.96279673715136715, -0.87311928929396732]], [[-0.98047296021739172, 0.92191260098834482], 
[0.99959129640418842, -0.82583448228521239]], [[0.9976886499723463, -0.66404169923957479], [-0.99987172772389166, 
-0.99982864011992778]], [[-0.99806057673022697, 0.99050945905679066], [0.99716643425886242, -0.90196495990114667]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tanh_array_rank4(self):
      arg=numpy.array([[[[1.2763803852500875, -4.7769098525734064, -4.2492051964063524, 1.4636066032733126], 
[-1.5804414838030989, -2.9939163303655691, -4.7257573726464122, 1.6977428145706943], [-4.8017089460422167, -1.9324890920322346, 
-3.2625954439350102, 0.63152881283044859]], [[4.3755289326276916, 3.1341019938551664, -4.8306255048273208, 1.2410084321149437], 
[1.9846840376006911, 0.06624291929865489, 1.664982845061389, 0.78912066472288167], [1.6758882108539801, 3.9609640254004557, 
2.0083703297777085, -4.8515786058850914]]], [[[4.3742797522930132, -0.25412700187898629, 0.70765842721248617, 
-1.3607898690159259], [-3.3005813104149908, 1.641994883259164, 1.2525009090276571, 1.0270005540723872], [1.8959450139668723, 
0.70683933191074644, -1.1191814815479892, 0.61913289513206848]], [[-0.093937912991567174, 0.75275158299922929, 
3.0423789984191743, -2.4684135460314516], [4.131568036363495, -3.2239604363377703, 3.0585299003077981, 4.5979690370181334], 
[0.094695795181266362, 2.7492890515721342, 2.6200377341127421, -4.1790135369376555]]], [[[-1.6495214577878556, 
2.9190007623353669, -1.6992499612390088, 2.7446564284324388], [-3.305670499164691, -4.777120049544151, 2.4492732433528648, 
-1.3909373022368232], [-4.8718687730684742, -2.2617999660952792, -2.1846334045644165, 4.613860993262044]], 
[[-1.651719042091778, 1.5626878037243461, 3.5128797794076387, 1.5136430058684782], [-0.12073849045584417, -0.27362112325412724, 
1.2096442534866476, 0.15145293383402336], [1.9680041544821645, -2.3885726711532298, -4.0445618910312504, 4.5608808246544328]]]])
      res=tanh(arg)
      ref=numpy.array([[[[0.85551753372245953, -0.99985815043337611, -0.99959249892349267, 0.89835080851200899], 
[-0.91867081167654774, -0.99499436716610368, -0.99984287142395978, 0.93512630359288218], [-0.99986501382698589, 
-0.9589341201002286, -0.99707221370724797, 0.55910402384246138]], [[0.99968346253987261, 0.9962159130238406, 
-0.99987259856253974, 0.8457429603422425], [0.96292936541956653, 0.066146195014493897, 0.9308852500919681, 
0.65791060623917763], [0.93232591781184393, 0.99927485832385166, 0.96461420348043592, -0.99987782683320392]]], 
[[[0.99968267085205875, -0.24879416616747818, 0.60920639389905007, -0.87657613912730947], [-0.99728611286680324, 
0.92775092584359353, 0.84898344594428987, 0.7727025690957362], [0.95588896168780346, 0.60869103437918337, -0.80728402104900643, 
0.55052401142723395]], [[-0.093662571736948461, 0.6367876397426887, 0.99545571801304755, -0.98574762347264655], 
[0.99948443476115711, -0.99683738559757407, 0.99559984328717055, 0.99979711928023451], [0.094413751804246548, 
0.99184818891396165, 0.98945616710502216, -0.9995310968773482]]], [[[-0.92879192501146057, 0.99418761775040898, 
-0.93531524254309617, 0.99177262154299362], [-0.9973135588162314, -0.99985821004931486, 0.98519557345055386, 
-0.88337693338910206], [-0.99988268512804812, -0.97853312454102914, -0.97499553009809325, 0.99980346559992717]], 
[[-0.92909313816777483, 0.91585482520374717, 0.99822419364703796, 0.90758341265569553], [-0.12015519166021936, 
-0.26699106952634971, 0.836572744941117, 0.15030545236693518], [0.96169592313668306, -0.98330060707783251, 
-0.99938647123791335, 0.99978149982113096]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tanh_constData_rank0(self):
      arg=Data(-1.90957268378,self.functionspace)
      res=tanh(arg)
      ref=Data(-0.957049516663,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tanh_constData_rank1(self):
      arg=Data(numpy.array([3.0450472296413356, 1.432126409576103]),self.functionspace)
      res=tanh(arg)
      ref=Data(numpy.array([0.99547984915077459, 0.89210154026459232]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tanh_constData_rank2(self):
      arg=Data(numpy.array([[4.8437919915555963, -2.5469599150035105, -3.7755998912281106, -1.5465789430165811, 
1.3273349435614961], [0.72340158670166854, 2.6925022924858908, 2.20210550418824, 2.319073394178778, -0.20807942301705395], 
[-3.8574040203634175, -0.58268769247525309, -1.2819123208454264, 1.3174562235992271, -2.5860229767494305], 
[-3.1267309667591361, -1.4431136221515386, -2.7381970155408011, 3.952041369681309, 4.2942188526069742]]),self.functionspace)
      res=tanh(arg)
      ref=Data(numpy.array([[0.99987590942925708, -0.98780693391160634, -0.99894959339106471, -0.91321926124675368, 
0.86859646755763642], [0.61901190326289346, 0.99087221066287723, 0.97584383001438246, 0.98083421672300164, 
-0.20512746216786271], [-0.99910805575394912, -0.52461615560491315, -0.85699358821784055, 0.86614993170102228, 
-0.9887181009625674], [-0.99615982249214008, -0.89432277166522001, -0.99166608972028158, 0.99926180661148267, 
0.9996275756969728]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tanh_constData_rank3(self):
      arg=Data(numpy.array([[[3.0939937552576353, -1.8704146834806048], [-2.7785386857585967, -4.1682991479195479]], 
[[3.1450575190517256, 3.723201281663794], [-4.2008063084053529, -3.837389398605342]], [[-3.3207490163046094, 
-0.52079028016587081], [-1.027307240963331, 4.6623116320500131]], [[3.9104413061754038, 1.6657769808907599], 
[2.5568581938816219, -1.3876462770909481]], [[-1.5955034588057582, 0.77528913504868058], [0.22348360858481886, 
4.4728687313666118]], [[0.17032760970709937, -4.4655936587553366], [-4.0945523558099781, 
-2.5681462508726605]]]),self.functionspace)
      res=tanh(arg)
      ref=Data(numpy.array([[[0.99590050785342554, -0.95363170332359115], [-0.99230960246320177, -0.99952094285763438]], 
[[0.99629777277076503, 0.99883360647693664], [-0.99955109079919224, -0.9990716445793616]], [[-0.99739326049900423, 
-0.47830972223023993], [-0.77282611339044227, 0.99982161479629295]], [[0.99919778717595975, 0.93099115131410992], 
[0.98804451235976753, -0.88265197052784417]], [[-0.92098889104700044, 0.64999447262250154], [0.21983584235596779, 
0.99973945124762165]], [[0.1686993495595813, -0.9997356330159971], [-0.99944482942821133, 
-0.98830983690678409]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tanh_constData_rank4(self):
      arg=Data(numpy.array([[[[-1.7587501139797999, 0.70194454539271689, 1.76847078377107, -2.466662569760365], 
[0.066061765372191772, 1.0586122083300964, -3.8178980531256714, 3.4955226415045928], [3.8478418636362619, -2.0617341159484806, 
-4.1056049952685338, 3.3407301203827462]], [[4.5746748424991779, 3.1710646110632918, 4.7182337989563887, 3.456863253654026], 
[-3.4329141540143135, -4.8208022233483465, -0.90996588534785872, -1.5386022921369813], [-4.8077467958524709, 
-3.6903248266921351, 3.4358784355735015, -3.9898051735999198]]], [[[-4.7653741409855748, -0.53179015099919713, 
-4.6427474600413241, -0.91880207281385395], [3.3668510463461807, 1.3484735466852174, 0.39999125968241156, -2.7533112197851595], 
[4.3553361830961386, 0.90082939331075362, -0.62690100720129038, -4.7214170567013971]], [[0.48373748860922028, 
3.1565435389195358, 1.0396158501191977, -4.9734407215385499], [3.7299503573359551, 1.0300972522425047, 4.3849990276443123, 
-4.7556624754842067], [3.3162968468734597, -3.425508029436215, -0.21773888181763468, -4.3880927836952601]]], 
[[[2.1342330123370887, -3.5234722159028662, 1.2361077932000626, 4.0261813496523384], [0.347201318555447, 0.60466080916009712, 
-0.87884646688452772, 0.45905059074119148], [-0.77948242669181056, 1.1055367390735773, -1.2958220450723621, 
2.7741742622958032]], [[4.0178678904347027, -0.70544059048622998, -1.8292324518298928, -4.7012312124829165], 
[0.63024314147784111, 0.56061907582271786, 1.983324928844338, 2.5148683685507569], [2.6372735980164048, -0.13877233962269209, 
3.7195370628459639, -0.22607222852178666]]]]),self.functionspace)
      res=tanh(arg)
      ref=Data(numpy.array([[[[-0.94236324620779599, 0.60560060666190929, 0.94344157797393802, -0.98569798237182327], 
[0.065965831535009775, 0.78513212690209422, -0.99903475795400698, 0.99816152299303551], [0.99909084160701811, 
-0.96813922630853111, -0.9994569636867312, 0.99749524967543468]], [[0.99978744478917347, 0.99648508889160248, 
0.99984048939929282, 0.99801388116654965], [-0.99791653603450747, -0.99987007097689962, -0.72111587972323532, 
-0.91188520371119564], [-0.99986663396890596, -0.99875438447619858, 0.99792883868613613, -0.99931548875193732]]], 
[[[-0.99985483995767821, -0.48674830189455665, -0.99981449718396165, -0.72533021495952599], [0.99762259167420397, 
0.87369251675153514, 0.37994148367565833, -0.99191323768852135], [0.99967041952147229, 0.7167014763292644, 
-0.55591462824928961, -0.99984150161994512]], [[0.4492318424159556, 0.99638169894919792, 0.77773632423384098, 
-0.99990425116892567], [0.9988492358531843, 0.77394734147665667, 0.99968940047810628, -0.99985199311671347], 
[0.99736997621857015, -0.99788547842636965, -0.21436189276423381, -0.99969131608724759]]], [[[0.97238027124818438, 
-0.99826138582875035, 0.84434186019388635, 0.99936350488638293], [0.33389119722039118, 0.54035779043081344, 
-0.70584096186572054, 0.42931010082295512], [-0.65240953199693885, 0.80247902978951224, -0.86064373511441739, 
0.99224244184521493]], [[0.99935283691652621, -0.60780978360851434, -0.94975093136037203, -0.99983497238948615], 
[0.5582196146482673, 0.50843661727008671, 0.96283033756278658, 0.98700393672386233], [0.98981161798604589, 
-0.13788833368173969, 0.99882503227709296, -0.22229795337603012]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tanh_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(0.55561767846)+(1.-msk_arg)*(1.96046025342)
      res=tanh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(0.504718696829)+(1.-msk_ref)*(0.961124935586)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tanh_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([-4.8590436658390379, -2.504637273202972])+(1.-msk_arg)*numpy.array([-4.5707080133524949, 
-1.0626844238645337])
      res=tanh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-0.99987963723507145, -0.98673705107256093])+(1.-msk_ref)*numpy.array([-0.99978575192297836, 
-0.7866891099753095])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tanh_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[-1.5725442958062521, 4.538869538960066, 4.3754844250523313, 1.7682178829554926, 
4.717327272322505], [4.0507231705475384, -0.85787948702801842, 1.3301712283504337, -2.3969760265995541, 0.53773055086014843], 
[1.9226661350288925, -3.9451091399026716, -1.3215015776486205, -1.0387585465107243, 0.54774831698657067], 
[-0.55676025766049619, 2.0662376942496028, -2.4031375986835615, 1.9075490278973835, 
-1.482227077851813]])+(1.-msk_arg)*numpy.array([[3.8888849420546503, 1.6742188836049978, -3.1046205107373792, 
3.1362367712810411, -0.31235473783172374], [0.41196253336889388, -0.50619544394277494, 1.4135311238239447, -4.5282756155814141, 
-1.7520137697571956], [-4.8550214050489258, 3.0649584783941322, 3.2283765740852317, 0.31326482436231373, -1.8790504341590064], 
[2.9458609811343504, 4.4593163675897252, 3.2043864360430181, 0.12930161818456121, 3.7146834279226546]])
      res=tanh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-0.91742952371278552, 0.99977166713747556, 0.99968343436644891, 0.94341377299128926, 
0.99984019995887796], [0.99939398281550451, -0.69516351705670476, 0.86929117608771767, -0.9835766381563682, 
0.49126815676322072], [0.95813642052430836, -0.99925150453808176, -0.86715686394788094, -0.77739735518157493, 
0.49883071966682674], [-0.50556972316304782, 0.96842040181253919, -0.98377615201656221, 0.95687908594641125, 
-0.90188440633607481]])+(1.-msk_ref)*numpy.array([[0.99916246002264297, 0.93210728251533603, -0.9959865440866712, 
0.99623200457586425, -0.30257777219675197], [0.39013777407641853, -0.46697557818172586, 0.88824163603089668, 
-0.99976677820989568, -0.94160429588553274], [-0.99987866512812817, 0.99565593256271212, 0.99686515209509829, 
0.30340430946131552, -0.95440757753116146], [0.99449078530720014, 0.99973229352409232, 0.9967113277958517, 0.12858581083779483, 
0.99881357780848878]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tanh_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[-4.6767827400115172, -1.5963811367614476], [4.9364113375148158, -4.2108626255357802]], 
[[-1.5740140112884236, -2.4236130676827439], [-1.833257388429165, 4.4126381030071364]], [[-0.43642959510515844, 
0.22400601845558121], [1.6902483195521025, 0.77595750478221959]], [[4.0956023626898368, 2.6332465969246215], 
[-0.65520337124814976, 2.2752478291998539]], [[1.9927504387300736, -0.11074372133587218], [0.42040702061010204, 
-4.375547694108862]], [[4.1681026769181351, 2.108905421865602], [1.114587696464624, 
1.7434232989549052]]])+(1.-msk_arg)*numpy.array([[[4.3823050751778343, 1.1602081323507107], [-0.37300374111213941, 
3.1841794365219283]], [[3.318019151829608, -2.7779967080540025], [-1.2737565871025005, -2.7615213851746647]], 
[[-0.44734994876607459, -1.040324431641074], [3.0842447353363571, -2.413167458259756]], [[-1.0484880977935775, 
-4.2052835073519956], [-1.7463433050778221, -3.8653342504676225]], [[1.8821201218206642, 1.6920387113663748], 
[1.3990428129984505, 3.2648709164526313]], [[0.15693804862696226, 3.4598385364560293], [2.6579177139838688, 
1.8688613037515935]]])
      res=tanh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.9998267032216801, -0.92112199690756524], [0.99989689132719251, -0.99956002738913241]], 
[[-0.91766190008410231, -0.98442204375811204], [-0.95014376335635053, 0.99970610146978378]], [[-0.41068057505080652, 
0.22033294817319152], [0.93417883127845103, 0.65038029327415559]], [[0.9994459937481307, 0.9897296524811936], 
[-0.57516242131125073, 0.97909684256290386]], [[0.96351179693474087, -0.11029320461733813], [0.39727326957708298, 
-0.99968347441519323]], [[0.99952075462408829, 0.97096598122713351], [0.80567809585499972, 
0.94062237830419282]]])+(1.-msk_ref)*numpy.array([[[0.9996877227430252, 0.82110769872592171], [-0.35661625582362178, 
0.99657592937687534]], [[0.99737900818750724, -0.99230129400139677], [-0.85481253944775304, -0.99204441685029865]], 
[[-0.41971822675478937, -0.77801614910962502], [0.99581996014240526, -0.98409579319724139]], [[-0.78121786689790695, 
-0.9995550916771333], [-0.94095792669208778, -0.99912208464398244]], [[0.95468030578241614, 0.93440638506397822], 
[0.88514457445732864, 0.99708548825619836]], [[0.15566217889742379, 0.99802565297084245], [0.99022169760849776, 
0.95349077933046067]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_tanh_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[0.46512297390958057, 3.4310299813813678, -3.4213264329023776, -3.8515212705647084], 
[2.300272283727379, 3.3568571208591784, 2.0650638505317236, -1.5079748476987209], [-1.829344545538186, 1.7301821883298061, 
2.419169960088178, -0.73664714086065874]], [[-4.3788271414916817, 1.1881110560422483, -4.4045192586630062, 1.2839357715463358], 
[-3.3238522486550148, -4.8336202307845788, -1.6811072379633973, 2.9689344148804722], [1.8427313680595026, -2.0768486203618233, 
3.8393062198119026, 3.2156249771134533]]], [[[-4.7780122854249738, -1.9401802230172938, -1.7722170720233485, 
1.6584743314341726], [0.99661425241374335, 3.2316007914088765, 4.187870093955949, 4.0478015251588921], [-4.4871933403715953, 
-0.98954033062758917, 0.10674981035324116, -1.7674952618670292]], [[-4.6147921046437244, -2.0254101992172844, 
0.38892999667285633, -4.6308884311784135], [3.1903230754408476, -4.8919390071761928, 2.3977740460276706, -0.59985966714503469], 
[0.34161772006919122, -2.4183087509886914, -4.3562650045199289, -4.8368170584764574]]], [[[-3.891390883408171, 
-1.6438819711102903, -4.4149873124009451, 3.9997248251446003], [-0.95400091142260468, -3.0027620275146161, 4.313087568992346, 
3.3076711975848259], [-1.1688623238556071, -2.4590024922764098, -1.976417655820156, -0.91339911883986691]], 
[[-3.1583623296878982, 3.5602612370099109, -2.9323185003182139, 4.685859200916747], [-1.0700064516939323, -4.7921697308655355, 
0.52425004382541829, -2.4086757834267636], [1.856953154322575, 0.47062771727135733, 2.9105240171267361, 
1.3810236783464749]]]])+(1.-msk_arg)*numpy.array([[[[4.6168419233162865, 1.8983765609663239, -0.95678492823518191, 
-0.81237214266949387], [-2.7219353684986034, 1.2216453848653588, -0.58701108060555729, -0.79293632254504498], 
[-3.0573993474474745, 0.74332583789675688, -4.5685641052283668, 1.4448539359407579]], [[-4.8712846765737492, 
3.1537647603289383, 3.5712214889684191, 4.7942261624515883], [-4.9805160693352422, -2.3513811699708853, 1.7879918196695979, 
-3.4455033085700846], [-4.9278459076163239, -0.45012753915495196, 1.7418470351422686, 1.4782775878400614]]], 
[[[2.2499676342027017, -4.4391434700784629, 2.7713520554145088, -4.8674476762089594], [1.5708072385681255, -2.935344697824791, 
-2.1694673812030594, -4.5613407439781684], [0.057335218925087972, -0.68444580449617121, 0.40194312175698688, 
-3.6115206989007067]], [[-1.3224492817811773, -0.88241832209333815, 0.90127434512467275, 4.6150077716858178], 
[4.6115135544633148, 4.7309526462830078, 1.3863306793827048, -1.9949257202006865], [-0.57491720281503511, 2.714003677314599, 
3.9174502565758491, -0.42994533591708972]]], [[[2.5507509535521784, -0.99209977151803486, -3.0261926092008675, 
-4.609434419812545], [0.022491091286896747, -3.1946478672116285, -1.1982202563970281, 2.3757581385329472], 
[-1.3160538439571532, 2.157765959784915, -3.6772738408070591, 2.2310660784842042]], [[-3.3883133058809722, -4.3504543520647276, 
-0.51185913029812014, 4.0298214526430183], [4.0799035252510532, -2.4885711104810948, 3.9311588519301282, -2.1089948285341475], 
[0.62940260904930678, 4.4483387530207086, -2.4430643541052555, 3.2750693351990119]]]])
      res=tanh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.43425036061868283, 0.99790867823649199, -0.9978677390523073, -0.99909750437031863], 
[0.9801071243916627, 0.99757465251187527, 0.96834735041758335, -0.90657900348502329], [-0.94976191236703289, 
0.93907745970292922, 0.98428409042239762, -0.62711516863274885]], [[-0.99968554335438531, 0.82999211396229111, 
-0.99970129099993732, 0.85753000888140551], [-0.9974093681417916, -0.99987335929853571, -0.93300509063877535, 
0.9947385878656464], [0.95105664361938846, -0.96907326419964457, 0.99907519510632048, 0.99678430546446539]]], 
[[[-0.99985846282596857, -0.95954829402995923, -0.94385191024496951, 0.9300113895918366], [0.76016855829263796, 
0.9968852705390796, 0.99953932758776365, 0.99939043239719882], [-0.99974680892160872, -0.75716625188828146, 
0.10634616023713185, -0.94333425182471586]], [[-0.99980383121432936, -0.96577953835651686, 0.37043742068494778, 
-0.99981004524840833], [0.99661767374495835, -0.99988730068162113, 0.98360261483782641, -0.53694970173332168], 
[0.32892086228123918, -0.98425721098523711, -0.9996710310953677, -0.99987416636180337]]], [[[-0.99916664542856282, 
-0.92801329642364028, -0.99970747888781741, 0.99932893064166783], [-0.74158899316782845, -0.99508192919542116, 
0.99964136567813999, 0.9973242725029553], [-0.82390711719578735, -0.98547878936201927, -0.96232303274695996, 
-0.72275972771854147]], [[-0.99639481322222423, 0.99838461718174143, -0.99433995736142011, 0.99982982041815593], 
[-0.78946365162461574, -0.99986241395799924, 0.48097354168124917, -0.98395342795676799], [0.95239648723201753, 
0.43870635915079276, 0.99408853128475416, 0.88118028846774887]]]])+(1.-msk_ref)*numpy.array([[[[0.9998046337105545, 
0.95609825966752082, -0.74283934344576275, -0.67089677180857032], [-0.99139177346460716, 0.84013886264693882, 
-0.52774254123395214, -0.66006924100969089], [-0.99558990477437848, 0.63115035094630911, -0.99978483139330487, 
0.89467061773942858]], [[-0.99988254800961063, 0.99636157073660292, 0.99841961422859726, 0.99986297862979123], 
[-0.99990559647656507, -0.98202268259324232, 0.94554822536080663, -0.99796828661439396], [-0.99989510986418051, 
-0.42200383695075122, 0.94044047659851315, 0.9011447829771283]]], [[[0.97802470791752738, -0.9997212733854266, 
0.99219870244889563, -0.99988164326924056], [0.91715406878425598, -0.99437401464306874, -0.97423539379421598, 
-0.99978170069170413], [0.057272474883844786, -0.59440203459853025, 0.38161034392895998, -0.9985419036024985]], 
[[-0.86739173841585271, -0.70762876292513488, 0.71691780474585354, 0.99980391580207295], [0.99980254081690045, 
0.99984449549680032, 0.88236098371742511, -0.96366731880799561], [-0.51896133832274605, 0.99125473087762472, 
0.99920894964793461, -0.40527562410310602]]], [[[0.98789847581581269, -0.75825624873092368, -0.99530655075593999, 
-0.9998017181004919], [0.02248729968733645, -0.99664675445833417, -0.83311094357033977, 0.98287079853999104], 
[-0.86579921351235412, 0.97363336067918338, -0.99872146446999754, 0.97718772843038626]], [[-0.99772236782692325, 
-0.99966718638660623, -0.47139247697324255, 0.99936812041651868], [0.99942832833671635, -0.98630692601415448, 
0.99923033517476023, -0.97097109708527496], [0.55764072832728684, 0.99972635178380265, -0.98501197546550456, 
0.9971442488931882]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asinh_float_rank0(self):
      arg=53.5640700041
      res=asinh(arg)
      ref=4.67411281178
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asinh_array_rank0(self):
      arg=numpy.array(-28.2897331664)
      res=asinh(arg)
      ref=numpy.array(-4.03595836682)
      if not isinstance(res,float):
         self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
         self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asinh_array_rank1(self):
      arg=numpy.array([8.2061468207208321, 52.348873857492663])
      res=asinh(arg)
      ref=numpy.array([2.8017226187898281, 4.6511688207733162])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asinh_array_rank2(self):
      arg=numpy.array([[-11.268379991778389, 26.078429537791692, 61.186675840076731, 43.272979008444878, 64.086666592131081], 
[-32.643925012772044, -31.707403483646914, 37.550955748115683, 93.407482972750955, 9.5153083383155348], [-94.278228397939728, 
-16.257853729518331, 10.214502896994532, 14.490526105756388, -65.657014075379408], [35.679819309731641, 31.939848706549725, 
24.582790367790096, 10.846308253277996, 61.732939591503111]])
      res=asinh(arg)
      ref=numpy.array([[-3.1171108337292814, 3.9546230975512011, 4.8071434015612393, 4.4608090604022976, 4.8534443779974552], 
[-4.1790404764480993, -4.1499459571368149, 4.3190232596184046, 5.2301472922687999, 2.9487988969877006], [-5.2394255930035643, 
-3.4826677710201546, 3.0193432805216656, 3.3678307396521299, -4.8776496042957262], [4.2679287450996357, 4.1572465575557356, 
3.8956072361629706, 3.0790902914356173, 4.8160304298624759]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asinh_array_rank3(self):
      arg=numpy.array([[[48.70380443069223, -0.58466485043287264], [30.531877850209469, 46.492832620000286]], 
[[-37.469319528711644, 45.031503409545195], [-16.1621580014899, 82.50592916194995]], [[91.832409136986996, 73.812740558496216], 
[-43.724635153785265, 13.399640237394948]], [[-12.060867028264838, 3.219383581499514], [26.012421261440721, 
-72.47023754814299]], [[41.890954648120527, -42.329788666104641], [-5.5337160472955134, 8.7841043115799238]], 
[[-9.2413080106654917, -25.484438640763926], [63.488971099727962, 29.421313862046617]]])
      res=asinh(arg)
      ref=numpy.array([[[4.5790097042651858, -0.55563071526196395], [4.1121865700239564, 4.5325609799632556]], 
[[-4.3168476542586962, 4.5006327624802518], [-3.476775458120315, 5.1060540635916398]], [[5.2131421000397804, 
4.9947244161010893], [-4.4711895955046534, 3.2897645071703892]], [[-3.1848274806877011, 1.8856293247930429], 
[3.9520906101104454, -4.9763707396598154]], [[4.4283595363790917, -4.4387777370000139], [-2.4120726095475979, 
2.8693153066743973]], [[-2.9197451844628475, -3.931599912174061], [4.84407540380219, 4.0751552409338272]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asinh_array_rank4(self):
      arg=numpy.array([[[[66.33188642986363, -12.551594647853094, -88.515657271508914, 34.236545814523623], 
[38.476261229201668, -82.542482081110506, 28.607151018292825, 2.7759547521209527], [-97.218772566865553, 69.312145077054311, 
-86.80660875753307, -86.068348260881194]], [[-29.57335409938635, 93.70369586756533, 10.152691790735375, -2.8617364538564942], 
[44.582719343737267, -99.14766458653294, -5.3397400314643306, 29.642673806358147], [-71.44258924316776, -72.66922811619672, 
29.46150819077684, -21.118617671168735]]], [[[-23.540079102899369, -12.418131621132076, 93.929079200347019, 
60.995280603796346], [15.86148681870965, 81.472332652216693, 43.161412126791504, -15.606741542598982], [55.648812200441313, 
-65.886430153679541, -0.084142920970450064, 45.980109011875697]], [[-7.6840442682781998, -58.235379889649977, 
80.961895857228001, -8.1269739988009633], [-27.124372691614667, 68.305656362227296, 16.315271320565913, -83.356016778093831], 
[-14.96671937954676, 24.050062406146225, 34.563313730729561, 27.070602538273647]]], [[[55.588142173379822, -58.423204686473355, 
65.025371614144234, -12.929599723397004], [-3.4559525035913623, -55.840063993594732, -61.262064979442243, -40.943217751898018], 
[-28.137549231855743, -28.617625880561633, 68.345543507200688, -65.487751042721513]], [[-61.552437326591168, 
-84.882950579149082, -59.327673501328547, -39.734301966929621], [-9.8040908765093775, 3.9501875492096872, 75.080957643504007, 
-47.772569788498267], [-67.715315201942161, 73.850963548190975, -78.099586704109555, 64.835384847591996]]]])
      res=asinh(arg)
      ref=numpy.array([[[[4.8878747181470601, -3.2245780107545325, -5.1763585417374545, 4.2266540618639565], 
[4.3433574681643439, -5.1064969668114468, 4.0471092484833049, 1.7451106649693606], [-5.2701374565896666, 4.9318193590641375, 
-5.1568631122315223, -5.1483226550713237]], [[-4.0803066650590241, 5.2333132839977603, 3.0133026433693662, 
-1.7737925470898555], [4.4906192607613198, -5.2897829116104855, -2.3769790138578233, 4.082646580023483], [-4.9620903372517127, 
-4.9791125411657937, 4.0765196853689218, -3.7438622584819665]]], [[[-3.8523024910800499, -3.2139220505578368, 
5.2357155363599723, 4.8040108644547939], [3.4580333539217274, 5.0934483277999592, 4.4582282106502404, -3.4418749723632036], 
[4.7122886329195044, -4.8811372702942597, -0.084043946896535215, 4.5214742994686485]], [[-2.736500637508644, 
-4.7577139610844599, 5.0871639406746967, -2.7920996637959798], [-3.9939194903447999, 4.9171933387885467, 3.4861866058094382, 
-5.1163039525189191], [-3.4000904029255126, 3.8737167147239719, 4.2361492087527814, 3.9919365170698424]]], 
[[[4.7111979840524763, -4.7609335674798388, 4.8679838270093869, -3.2541585151894044], [-1.9535487683584942, 
-4.7157189520928871, -4.8083745958434827, -4.4054824549214722], [-4.0305677545528882, -4.0474751204468307, 4.9177770566744119, 
-4.8750685870983039]], [[-4.8131026106584995, -5.1344551317981155, -4.7762940669516869, -4.3755203336238555], 
[-2.9785377368297734, 2.0825599131416936, 5.0117584928715218, -4.5597083260868443], [-4.9085140733297008, 4.9952420718825161, 
-5.0511729296519343, 4.8650581644874018]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asinh_constData_rank0(self):
      arg=Data(-72.2424060584,self.functionspace)
      res=asinh(arg)
      ref=Data(-4.97322229434,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asinh_constData_rank1(self):
      arg=Data(numpy.array([46.898808015843542, -67.367352228123565]),self.functionspace)
      res=asinh(arg)
      ref=Data(numpy.array([4.5412530831643014, -4.9033627742428001]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asinh_constData_rank2(self):
      arg=Data(numpy.array([[-63.575800675085191, -20.327495514604664, -17.286821365088372, 65.650121396592112, 
-13.696171285627571], [18.181905449895552, -21.10706483019564, -58.311888995805973, -20.846176390445351, -34.33971893954994], 
[96.959904721402324, -77.901702328696558, 84.114459513388198, -54.859545832039288, -11.004014888071637], [50.96559852814022, 
-61.826590476393228, 31.048295911499878, -94.961381672842379, 62.521584052421929]]),self.functionspace)
      res=asinh(arg)
      ref=Data(numpy.array([[-4.845441932600588, -3.7057260841841524, -3.5439271574219151, 4.8775446308780097, 
-3.3115935781449521], [3.5943294603669664, -3.7433156762353494, -4.7590266963403831, -3.7308925201194576, -4.2296617891819066], 
[5.2674713114455063, -5.0486361785480387, 5.125360997749203, -4.6980064453659383, -3.0934656201619717], [4.6243942798031714, 
-4.8175461144353617, 4.1289503415236481, -5.2466452030197175, 4.8287229722073493]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asinh_constData_rank3(self):
      arg=Data(numpy.array([[[71.728752084194639, -11.506640608653782], [-34.879891883081456, -58.102657132383854]], 
[[-17.548053222884548, -48.973728039931828], [-30.738225230024213, -28.819084735855057]], [[73.54720894485888, 
74.454450366648359], [-48.862945594197036, 32.329394312101698]], [[-49.873883925104593, 81.966840582824375], 
[-17.883545231602781, -11.086825397543549]], [[-80.780708169656947, 29.13871504516365], [-85.865666195127275, 
-94.184883818734491]], [[-41.824941354051994, 21.980890750044523], [-89.094109314099427, 
-88.016710451750853]]]),self.functionspace)
      res=asinh(arg)
      ref=Data(numpy.array([[[4.9660874403353885, -3.1379543459118104], [-4.245263106157104, -4.7554326228380086]], 
[[-3.5589010717773562, -4.584535391165895], [-4.1189186726633773, -4.0544858859680035]], [[4.9911208935831404, 
5.0033798095821851], [-4.5822712227511939, 4.1693631428228324]], [[-4.6027451693085375, 5.0994991711644913], 
[-3.5778089816523142, -3.1009323869675809]], [[-5.0849236662078621, 4.0655091961170697], [-5.1459651407775455, 
-5.2384350614343003]], [[-4.4267829068386364, 3.7838376804709597], [-5.1828718931958928, 
-5.170706137747004]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asinh_constData_rank4(self):
      arg=Data(numpy.array([[[[-76.987937257845388, 60.24336342375139, -13.812427169848476, -50.553162890520341], 
[-43.09546490990985, 75.90720580856555, 84.901785978874585, 75.001390212373678], [-24.304738447167679, 20.785867439057043, 
20.315493386719183, 59.610074366081733]], [[17.896170448153882, 61.34195700808894, -85.139017094148244, -10.037937479979561], 
[11.957661364919716, -61.000947811674557, 51.97117883980701, -90.662345864156308], [-41.333734272700084, 99.113649845065311, 
83.527266294937533, -30.60261078316131]]], [[[-44.414580616723008, -71.530119428284067, 85.155487873192016, 17.42760135675492], 
[13.988671289053215, -14.379613550878105, 71.474472659878785, 78.946427653133554], [16.674481178671385, -52.106041340346842, 
47.119206349912218, -63.326781321810131]], [[-97.564403219556198, -78.598780678133863, -33.590593704621497, 
-39.693723069145427], [-27.270263711318393, -54.061040105556145, 89.403786950718342, -29.50901081540691], [27.099297775814009, 
-47.686803319603399, 69.59896381683231, 45.930837685207791]]], [[[76.985838170494105, 33.71520889047278, 29.841604890159942, 
8.8876258578152232], [-42.903743906119686, 68.210485599821197, -0.22755771830003368, 48.932518150345913], [25.74594627973535, 
-14.843656366209345, -95.688709179092598, -83.647916965507079]], [[-8.5812519395967399, 96.63475692882426, 45.81705977916144, 
52.83407324550879], [-77.970721200203002, -41.862497656555675, -23.075708192961741, 76.427921817500646], [14.706347855724374, 
-10.967210043988686, 76.370395799734467, 16.182872110248894]]]]),self.functionspace)
      res=asinh(arg)
      ref=Data(numpy.array([[[[-5.0368381073249102, 4.7916084736161872, -3.3200237071503986, -4.6162705027962891], 
[-4.4566995323125527, 5.0227021842571817, 5.134676990294059, 5.0106982699238261], [-3.8842414534405334, 3.7279986183818683, 
3.7051361854992284, 4.781042121836629]], [[3.5785136009556808, 4.8096776756445863, -5.1374670836956726, -3.0019907986099641], 
[3.176263231166736, -4.8041037598917349, 4.6439290377520317, -5.2003197146345066], [-4.4149724545949738, 5.2894397986705437, 
5.1183561326668796, -4.114499344537915]]], [[[-4.4868416970281304, -4.963314649695941, 5.1376605091183958, 3.5520245234585164], 
[3.3326701245673558, -3.3601655252479343, 4.962536474267381, 5.0619567816729996], [3.507924604474149, -4.6465201462001131, 
4.5459404590531411, -4.8415178398262109]], [[-5.2736861490858651, -5.0575438321055328, -4.2076147522485696, 
-4.3744988794656887], [-3.9992800502035624, -4.683346490473677, 5.1863414972251363, -4.0781298234312411], [3.9929952490243465, 
-4.5579117990112463, 4.9359484661768755, 4.5204023995011875]]], [[[5.0368108441070154, 4.2113160779171839, 4.0893313539032663, 
2.880957177999683], [-4.4522420611547657, 4.9157992096430378, -0.22563820061084491, 4.5836937432006559], [3.941801313157649, 
-3.3918524879092016, -5.254274793049154, -5.1197994337119335]], [[-2.846104829046598, 5.2641124301019424, 4.5179227581366801, 
4.6603930372620894], [-5.0495216872698867, -4.4276801880837207, -3.832396814981069, 5.0295380752582473], [3.3825803357558231, 
-3.0901291349040911, 5.0287851730775639, 3.4780538385057014]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asinh_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(-87.3453801846)+(1.-msk_arg)*(25.3727619962)
      res=asinh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-5.16305009434)+(1.-msk_ref)*(3.92721152403)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asinh_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([57.709883829215471, 97.353803797457545])+(1.-msk_arg)*numpy.array([49.065563492095578, 
-43.584868424290612])
      res=asinh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([4.7486506931867414, 5.2715253615234374])+(1.-msk_ref)*numpy.array([4.5864084437844239, 
-4.4679887940475798])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asinh_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[-8.2838375890167413, -46.082349426477954, 33.660609854513183, 70.468970355904275, 
-75.306681226679316], [-15.055405689348092, 40.808996984124775, -74.651938940026838, 53.945528198505144, 16.080719059825825], 
[-49.660029033397791, -27.500518319221513, -35.521834679850215, -64.064013072646645, 40.446642406901049], [-4.1102908407429055, 
-55.378663259419689, -94.369513758126672, 56.345632467396513, 
-83.816608718825819]])+(1.-msk_arg)*numpy.array([[30.177563449380017, -50.278248780668818, -39.951925150494169, 
-73.708833494583345, 0.84857193383514584], [9.5646384440543528, -55.859488170839896, -25.995294599318711, 83.104478570666544, 
-71.966743358149074], [2.4813608403909058, 11.709046133864248, -7.5198554933995467, -23.132289816644771, 70.039626615177355], 
[-13.288179012776098, -21.25659752518672, -92.300997407122793, 63.784663709028166, 84.531709510693219]])
      res=asinh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-2.8110769177384465, -4.5236948862469388, 4.2096960602402556, 4.9483699964796619, 
-5.0147601198126512], [-3.4059854171148669, 4.4021998345951383, -5.0060285347847726, 4.6812078769839145, 3.4717335451272033], 
[-4.5984489037521188, -4.0076824352872533, -4.2634928200557498, -4.8530908759877853, 4.3932835977834159], [-2.1211206336498551, 
-4.7074230687321537, -5.2403933251012198, 4.7247306458753986, 
-5.1218139473004234]])+(1.-msk_ref)*numpy.array([[4.100520302248837, -4.6108186158927609, -4.3809806300878575, 
-4.9933158420137334, 0.77014895014981288], [2.9539416171689332, -4.7160666896471222, -3.9514324765011217, 5.1132819713023236, 
-4.9693995614601221], [1.6402864020397783, 3.155327393120436, -2.7150860739649971, -3.8348435224417496, 4.9422593157378687], 
[-3.2814348504773423, -3.7503673304902256, -5.2182314713973694, 4.848721403575226, 5.1303088896927269]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asinh_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[15.623960460661593, -35.249534440497257], [-25.958976764215237, 22.879533193334026]], 
[[76.789287387677348, 49.786025455292304], [-42.785409453058776, 23.572089448082068]], [[-91.891648940605066, 
32.704551732044507], [-65.669127593556624, -16.458623506481175]], [[21.702541613296788, 74.943182273881234], 
[-10.764328119683711, -50.875398241484085]], [[-90.2777587931824, -32.535158622196832], [21.706453957740408, 
-13.933723021467188]], [[97.494584124793761, 14.015670833289249], [-90.643934344892259, 
60.555907783285932]]])+(1.-msk_arg)*numpy.array([[[-27.223295143110235, 70.714402565642075], [-29.815396406942668, 
-1.3084150726888595]], [[93.090742224435218, -51.844022963418347], [48.788860852670325, 91.978888269270186]], 
[[79.928586202889562, 6.9514390428295769], [4.4602779426849537, 80.445270619872048]], [[-38.627313023373745, 
-64.5496847740406], [-45.427598785605468, -67.90762992204418]], [[-77.53734986575796, 8.1583665566308667], 
[-30.029464555445657, 16.486528778291486]], [[-78.359718374546361, 83.948692642123774], [70.955278035194368, 
20.873846721541582]]])
      res=asinh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[3.442975410007926, -4.2558006441102831], [-3.9500354417511514, 3.823867182228224]], 
[[5.0342547185007884, 4.600982358031505], [-4.4494808643282875, 3.8536601677093034]], [[-5.2137869398780285, 
4.1808950984926589], [-4.8778340628285903, -3.4949183672130166]], [[3.7711069223933311, 5.0099219462026587], 
[-3.0715355228641807, -4.6226232264085878]], [[-5.1960689802945099, -4.1757045804272357], [3.7712869863879734, 
-3.3287443922343871]], [[5.2729703099550402, 3.334593466283839], [-5.2001166284621743, 
4.7967823827903819]]])+(1.-msk_ref)*numpy.array([[[-3.9975573886919751, 4.9518464374998397], [-4.0884532082186729, 
-1.0835714064383526]], [[5.2267507683516703, -4.6414798328585469], [4.5807542160476062, 5.2147358050543513]], 
[[5.0743198741373314, 2.6372297547250776], [2.2006943435051318, 5.0807628947238781]], [[-4.3472743085280596, 
-4.8606424090728311], [-4.5093881252598598, -4.9113497873172571]], [[-5.0439485152033647, 2.7959262346019425], 
[-4.0956033497992133, 3.4966092988393047]], [[-5.0544978921330994, 5.1233884631819793], [4.9552466244829221, 
3.7322174760049376]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_asinh_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[-31.091081117097133, -38.902611105807708, -68.849530363866904, 86.470875054984447], 
[-3.6000643008149495, 85.280863472183455, 54.943724004241062, 34.498351219358057], [-36.885557788562686, -50.222816543320903, 
53.237623693753363, -25.230707976380245]], [[-7.9503183676549583, -16.952766519884221, -27.099473177202228, 
-90.795295294927755], [47.206889833323856, 69.938198741260692, 17.24324007370312, -41.997048369242492], [66.510208184761012, 
-42.956916830433478, -83.691258864758908, 20.972353682119021]]], [[[-39.716386812503288, 81.158514325818004, 
-97.40636462699031, -80.203383329805717], [-38.874140918366429, -43.927791030439181, -83.220214028135487, -92.975487525627415], 
[-12.343833616402677, 30.104070847974896, -16.72929055531209, -21.035324209470588]], [[-85.128161105148052, 96.827618637203756, 
41.878345800560282, -42.461985361177135], [11.803250255420906, -49.57219392589429, 60.379718365751842, -53.187092138172567], 
[-97.438353832502827, -45.705360363718796, 45.527940756318685, 27.454699252678736]]], [[[-42.787770043931552, 
8.3629919241016779, 85.854061698816395, 18.583639098807097], [-59.397251038518625, 47.593811879652719, -91.829566325533591, 
-43.830306240721796], [-44.285646723955253, 76.281808196938954, 20.392036034677432, -44.591228692627283]], 
[[-45.031872471154877, -30.263753065022385, -52.704337695312105, 78.277343708395449], [54.911999151252587, 47.971687288000368, 
88.947303894447856, -71.058452950429853], [91.428019924314839, 27.551749268560769, -45.903986292773304, 
73.520512031767652]]]])+(1.-msk_arg)*numpy.array([[[[-90.394593894893106, 23.797489220259223, 44.161446287029349, 
68.818173624062212], [-9.1254467828786261, 56.279334031571437, -46.611514994449557, -81.30463095138353], [28.654908904155462, 
25.171790416220844, -85.408033938837775, -65.433159636024726]], [[58.823220002503916, -43.205461002826297, 41.234800979887552, 
11.721741914892192], [88.744352931799085, -48.549328881830476, -74.574162687604414, -96.576084145907302], [-5.6675412231440845, 
-59.603581196903832, 69.55528570711482, 2.5666953805598212]]], [[[10.926200302932315, 37.842531966039871, 43.751136906648071, 
29.829781461426307], [-25.946071309634505, 29.806991667479878, 86.908200908296436, 99.869539801081118], [79.197582678741583, 
77.07063199562225, 86.278362693898714, 8.0848121153953798]], [[-77.05403568081239, 10.291535656769639, 32.891025448314934, 
-94.521421121488885], [-32.876225217590729, -56.544727788634709, 0.48813650952317289, 52.112626899786818], [51.07458469833179, 
-74.308056046350089, 66.279051868322256, 0.10364973306909064]]], [[[74.579979872901276, 89.894159768483661, 43.983145388402448, 
-22.520006446424361], [-84.131929079213876, 44.068062654014483, 28.662153233476573, 16.233091913166575], [74.864731841445092, 
99.967198226378372, -66.706458003423222, -66.444198424722089]], [[31.030180080196345, 14.701898947931909, -37.218808402153726, 
-88.939859627593648], [60.509017054630306, 62.347339050093694, -85.521147631762133, 69.936454921316397], [-43.051037517988846, 
4.5841614181120889, -10.154185340710626, 68.854476092497919]]]])
      res=asinh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-4.1303267012672142, -4.3543737010832446, -4.9251233201375237, 5.1529882664776583], 
[-1.9928529745156156, 5.139131638831711, 4.6995394457697666, 4.2342687065400328], [-4.301150966503946, -4.6097157167370852, 
4.668000734496391, -3.9216014908702914]], [[-2.7702911160624923, -3.5244469637363838, -3.9930017171447494, 
-5.2017849753863095], [4.5477991982343484, 4.9408102642195821, 3.5414071274612624, -4.4308882326016032], [4.8905591331443334, 
-4.4534803118500621, -5.1203174093115207, 3.7369201646290895]]], [[[-4.3750695009996701, 5.0895893429396093, 
-5.2720650823513591, -5.0777517430726071], [-4.3536418427889769, -4.4758238853805468, -5.1146735519267645, 
-5.2255119830425256], [-3.2079405337109761, 4.0980833337323963, -3.5112003691299263, -3.7399148231962243]], 
[[-5.1373395753845159, 5.2661061145760373, 4.4280585847064673, -4.4418950215400663], [3.1633117808326339, -4.596678968678253, 
4.7938690071531171, -4.66705128138471], [-5.2723934209480383, -4.515482420347988, 4.5115939889751964, 4.0060160324866061]]], 
[[[-4.4495360205448726, 2.8205189255518026, 5.1458299937041572, 3.616151870904126], [-4.7774659805206259, 4.555960279498187, 
-5.2131111448798215, -4.4736027906487053], [-4.4839352508007897, 5.0276246263158653, 3.7088922730942975, -4.4908100610954991]], 
[[-4.5006409560581986, -4.1033707539307827, -4.6579349307715106, 5.0534461875282854], [4.6989619683893444, 4.5638667866372504, 
5.181222881445172, -4.9566995079834921], [5.2087290820081149, 4.0095423837444022, -4.5198177622882332, 
4.9907578710936695]]]])+(1.-msk_ref)*numpy.array([[[[-5.1973622380447404, 3.8631684148472276, 4.481127498204482, 
4.924667825770114], [-2.9072027569990939, 4.7235535002663003, -4.535109841932754, -5.0913879735657872], [4.0487762787487247, 
3.9192654456665372, -5.1406216216387239, -4.874234724261008]], [[4.7677560984456671, -4.4592479783561147, 4.4125767640460127, 
3.1564071504902529], [5.178938720215891, -4.5758336004479441, -5.0049862335490518, -5.263505117994435], [-2.4355962981163364, 
-4.7809332038509984, 4.935320765611384, 1.6717203143889088]]], [[[3.0863983561030994, 4.3267553632597053, 4.4717953593412112, 
4.088935291599948], [-3.9495385388267277, 4.0881714343701425, 5.1580326777980261, 5.2970369772404249], [5.0651328128586943, 
5.0379115663158265, 5.1507596078442157, 2.7869374009169285]], [[-5.0376962222313804, 3.0268210112033329, 4.1865780304726847, 
-5.2420016487814909], [-4.1861281593878115, -4.7282573294574917, 0.47057570090180251, 4.6466465025862664], [4.6265300126066178, 
-5.0014118252744897, 4.8870779730638114, 0.10346503513752499]]], [[[5.0050642288672629, 5.1918110917671694, 4.4770828879127675, 
-3.8080438561162118], [-5.1255686495373425, 4.479011206595672, 4.0490289059034295, 3.4811464215435635], [5.0088746924612888, 
5.2980143104725164, -4.8935051286131399, -4.8895662770228778]], [[4.128367001254925, 3.3822784711073628, -4.3101418416178356, 
-5.1811391902304438], [4.796007850714485, 4.8259324817458369, -5.1419450465346097, 4.9407853327303348], [-4.4556683719898542, 
2.2274440517919998, -3.0134490330376598, 4.9251951438586685]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acosh_float_rank0(self):
      arg=95.1417913418
      res=acosh(arg)
      ref=5.2484878804
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acosh_array_rank0(self):
      arg=numpy.array(79.3820855421)
      res=acosh(arg)
      ref=numpy.array(5.0673802251)
      if not isinstance(res,float):
         self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
         self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acosh_array_rank1(self):
      arg=numpy.array([95.979908521154016, 99.149151150172116])
      res=acosh(arg)
      ref=numpy.array([5.2572589246874193, 5.289747042333623])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acosh_array_rank2(self):
      arg=numpy.array([[19.941916347574463, 81.942822615225168, 18.862699394909288, 95.241399023021671, 97.962580446844939], 
[84.709081263386764, 88.878493449655181, 67.750830408623983, 23.597652669131087, 44.978714490843913], [4.0038378434307251, 
99.472053979186185, 28.604990697303784, 57.652384728038086, 38.943153293387631], [82.839571127035029, 80.913347173225588, 
9.8784649118765273, 5.9811627902656213, 25.107382741276087]])
      res=acosh(arg)
      ref=numpy.array([[3.6853418064442032, 5.099131665193509, 3.6296301947436298, 5.2495343297913806, 5.2777067024102884], 
[5.1323351513008131, 5.1803857258317825, 4.9089294285942779, 3.8538451672569503, 4.4992129505326162], [2.0644274890952379, 
5.2929986537079792, 4.0464227111708029, 4.7475035683802611, 4.355085268948157], [5.1100166075265276, 5.0864877868236498, 
2.9809325119993271, 2.4746995083921228, 3.9159122946732663]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acosh_array_rank3(self):
      arg=numpy.array([[[29.647399929679686, 32.474911680184142], [44.526838192432805, 79.637008938614187]], 
[[72.715841217822671, 80.153440274709965], [38.266087043596222, 4.7954216484387624]], [[80.720144081287643, 39.06173313710277], 
[88.452371261185576, 80.118934430861216]], [[43.796426614291292, 71.797800504277689], [52.801600294326221, 
67.778126703742714]], [[94.78583480785197, 62.280117502670265], [86.063127142538647, 5.0076700376838357]], 
[[42.300095761336728, 1.919375325409987], [82.031544568166382, 25.888451781669893]]])
      res=acosh(arg)
      ref=numpy.array([[[4.0822370642938983, 4.1733778869196128], [4.4891131749850848, 5.0705866800337676]], 
[[4.9796591559792285, 5.0770510662901582], [4.3375404541689928, 2.2497557058148336]], [[5.084096970729246, 4.3581265894574424], 
[5.1755794546405056, 5.0766204427173367]], [[4.4725690494978059, 4.9669525216526287], [4.6595989976290477, 
4.9093322837286619]], [[5.2447393294862383, 4.824724955268727], [5.1481944898867047, 2.293996059917617]], [[4.4377967815879513, 
1.2691051669962143], [5.1002138899745546, 3.9465709474569688]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acosh_array_rank4(self):
      arg=numpy.array([[[[16.792102101980927, 91.498636287473161, 11.518341442768513, 91.424330235480824], [44.127628593731174, 
8.6243281294532679, 75.992237244111038, 34.990968486360124], [56.120963592223312, 96.557528034227232, 54.36139809111733, 
81.104624656804404]], [[96.74125402143487, 34.03689813428754, 51.970009580159989, 16.558031245105344], [41.023932817592474, 
10.585625156665863, 64.368155305544505, 99.601281261536215], [49.811807325209898, 68.016829008183024, 76.901653673343915, 
44.903002524854429]]], [[[5.0762863587430402, 57.380171943369561, 25.108085331940632, 21.168119189263425], [65.923679707194964, 
79.976059260896278, 31.792750138861589, 45.855626593971316], [25.406438012992005, 21.988197476051813, 86.579734069228479, 
52.854743026593262]], [[8.6296368958821308, 74.378864940070798, 52.459835396778402, 85.242821775932953], [53.541537192750589, 
58.540574508483921, 39.65409505205222, 71.832025732475799], [69.484251472926687, 90.396286589625589, 85.320935186569855, 
64.573460176306227]]], [[[72.866107155140597, 12.148055522675744, 16.92415772209884, 61.244189665313264], [79.111821680505756, 
43.292241178415132, 8.1125941630036014, 82.079260431999813], [62.411436598462224, 86.87977555326998, 75.40450774865792, 
89.512378743154642]], [[84.555634657540779, 20.069760044838688, 75.246783056253889, 11.170079445592965], [98.381154958058673, 
22.687741050623146, 8.5270170479126683, 5.013871249515736], [78.372637819722925, 74.030731720543329, 68.436329141321593, 
92.515312824937794]]]])
      res=acosh(arg)
      ref=numpy.array([[[[3.5131680576369808, 5.20944138598934, 3.1351981612753996, 5.2086289073895298], [4.4801048544932414, 
2.8443560064329492, 5.0237350799014111, 4.2480329159395227], [4.7205772203025127, 5.2632593412563891, 4.6887168801480463, 
5.0888491563138736]], [[5.2651603969760634, 4.2203764917444886, 4.6437214187070168, 3.4991053382072526], [4.4071542235833299, 
3.0504056023957906, 4.8577058641095938, 5.2942970076286855], [4.6012984590982464, 4.9128482971572902, 5.0356322848209585, 
4.4975278306450814]]], [[[2.3078811363553764, 4.7427700485220363, 3.9159402999319943, 3.7450850265110551], [4.8815873553735738, 
5.0748354230261628, 4.1521380354912942, 4.5185261753722328], [3.9277622579312643, 3.7831355264912392, 5.1541795985004573, 
4.6606051323985378]], [[2.8449755504472223, 5.0022738157412157, 4.6531041630327499, 5.1386166841381176], [4.6735177095346288, 
4.7627943176127898, 4.3731823775352279, 4.9674291433726072], [4.9341955252999519, 5.1973197739811443, 5.1395326910101735, 
4.8608907124614182]]], [[[4.9817237000298409, 3.1886179262952599, 3.52101526546384, 4.8079495046679428], [5.0639695495953703, 
4.4609871962950596, 2.782744468126837, 5.1007954410669294], [4.8268315300193976, 5.1576393307477906, 5.0159702664729151, 
5.1874929034151522]], [[5.1305219280334873, 3.6917401460496646, 5.0138761768694291, 3.1043761810448292], [5.2819706211117374, 
3.8144858749911945, 2.8329305941872227, 2.2952590432995263], [5.0545813355500533, 4.9975818620869017, 4.9189976091843617, 
5.2204921454148616]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acosh_constData_rank0(self):
      arg=Data(14.9455204745,self.functionspace)
      res=acosh(arg)
      ref=Data(3.39643769166,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acosh_constData_rank1(self):
      arg=Data(numpy.array([2.8523485114042471, 40.788348429719399]),self.functionspace)
      res=acosh(arg)
      ref=Data(numpy.array([1.7090402422267734, 4.4013933411633657]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acosh_constData_rank2(self):
      arg=Data(numpy.array([[79.481650262327435, 59.020926208304687, 90.28931897253382, 27.954716652897464, 
67.195562083935812], [90.036950169690485, 12.274702504667838, 72.436025578107177, 73.962626040391811, 88.250555458141903], 
[53.759883774411506, 80.410405574514911, 89.521362583744704, 80.682625807136674, 38.178186948699057], [13.804092473514778, 
48.344412908391462, 8.9612703451357074, 91.075511929025353, 95.528026784651487]]),self.functionspace)
      res=acosh(arg)
      ref=Data(numpy.array([[5.0686337852552823, 4.7709674678259208, 5.1961356820576352, 4.023413054057948, 
4.9006990128929235], [5.1933364837788405, 3.1990241991858435, 4.9758032971708914, 4.9966613896283585, 5.1732950689450714], 
[4.677588202166481, 5.0802521039200004, 5.1875932688674258, 5.0836320325429689, 4.3352399487641922], [3.3167977241937843, 
4.5713908570514512, 2.8829313823198848, 5.2048060034869277, 5.252539462603286]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acosh_constData_rank3(self):
      arg=Data(numpy.array([[[14.875101526833383, 81.129795397433639], [61.560580838326167, 78.780591792832624]], 
[[23.012235781415036, 88.913664171315588], [71.880749183318272, 48.429719356992621]], [[40.76295375585682, 45.647292528697577], 
[34.388448687134385, 62.574265422661533]], [[31.833448505811365, 76.539709839690985], [18.467751916790512, 
35.948422930103824]], [[5.0022196543192035, 59.49308179581412], [32.605045673145653, 53.45176836337177]], [[97.023140137955295, 
36.841405083427624], [22.157397947468496, 31.346818727452622]]]),self.functionspace)
      res=acosh(arg)
      ref=Data(numpy.array([[[3.3917041893569753, 5.089159480765499], [4.8131029500844447, 5.0597735664889401]], 
[[3.8287008235112894, 5.1807813894885877], [4.9681072761500804, 4.5731542351826651]], [[4.4007703636206301, 
4.5139714745998099], [4.2306564227018706, 4.8294374243676161]], [[4.153417964034225, 5.0309141925573932], [3.6084394312448533, 
4.2750388840152702]], [[2.2928846498751247, 4.778936573621194], [4.1773789854564694, 4.6718393888661263]], 
[[5.2680701300456523, 4.2995852897933293], [3.7908090084784094, 4.1380054482779247]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acosh_constData_rank4(self):
      arg=Data(numpy.array([[[[10.057805486881218, 33.573225336394927, 43.683538129591234, 32.375002093275889], 
[30.549727630737635, 84.942763694910099, 33.004746524947073, 90.665686521193777], [55.39955714564514, 13.102519958201794, 
55.996558403753724, 18.937582109556615]], [[33.915469012245779, 46.635210272348203, 11.233582595005759, 65.016283474046233], 
[69.405237667811079, 51.552268589130748, 35.219620002035647, 41.360581340322376], [43.873679246002759, 85.538713174871873, 
36.077574037779627, 58.180003710551553]]], [[[90.188704881404632, 82.211114385826193, 99.359630615455941, 74.6287097327738], 
[45.311510318835992, 14.442103234785934, 76.214185408677139, 6.5327923001186878], [27.9623643312644, 65.453523026113928, 
31.829905956756033, 41.677003978658497]], [[37.319932213963106, 6.7687916196156692, 16.454239198960497, 13.880483974314521], 
[21.765715630318926, 6.4144862698528797, 44.618033382443336, 7.2985384768453514], [31.390191953753334, 15.840877362125562, 
76.65645245433717, 60.598595719799697]]], [[[60.836169677671059, 20.378022280592532, 74.425024247596937, 87.150594086264206], 
[49.564250587575351, 58.120616410862162, 62.434855624448296, 41.212616778229503], [38.895441963759609, 57.338622361895681, 
29.91485402068011, 40.641145413934261]], [[97.757419995956738, 71.10230260301509, 13.286515159385161, 28.80653662107348], 
[74.092105403273905, 18.713666923474612, 13.188947602938496, 88.710933489690774], [64.556364563044554, 51.164091677566468, 
8.8193879801673933, 64.720379076780702]]]]),self.functionspace)
      res=acosh(arg)
      ref=Data(numpy.array([[[[2.9990156210605976, 4.2066541949625416, 4.4699874741140349, 4.1702951622734092], 
[4.112234972149035, 5.1350901914167739, 4.1895689843391581, 5.2002957336753903], [4.7076373137511496, 3.264492332984037, 
4.7183576736066009, 3.6335977762303808]], [[4.2168009906741952, 4.5353880509636983, 3.1100679188411142, 4.8677257868578963], 
[4.9330576133169979, 4.6356493157779566, 4.2545488889730851, 4.4153292950281493], [4.4743318570334578, 5.1420820703372963, 
4.2786265071380356, 4.7566150320747944]]], [[[5.1950206401498571, 5.1024006933950545, 5.2918677568512091, 5.0056275718948369], 
[4.5065864843219998, 3.3620941826554556, 5.0266517437512173, 2.564071585621829], [4.0236867655893969, 4.8744291396602941, 
4.1533066190230752, 4.4229527342531689]], [[4.3124951943794105, 2.5999680061204238, 3.4928056566761709, 3.3223309006743862], 
[3.772955108165104, 2.5455739778765687, 4.4911596905126512, 2.6760947213442674], [4.1393888514459389, 3.4547431819634191, 
5.0324384159506907, 4.7973508142031864]]], [[[4.8012641332313448, 3.707001590129293, 5.0028942763904789, 5.1607518522893852], 
[4.5963152185511857, 4.755593608662676, 4.827206743915518, 4.4117443995609467], [4.3538589597484583, 4.7420455656800211, 
4.0912228491562903, 4.3977767744969309]], [[5.2756101232359365, 4.9572174478647089, 3.2784776078432114, 4.0534481003137861], 
[4.998410623981961, 3.6216866482863757, 3.2710860400054655, 5.178498556764203], [4.8606258989022955, 4.6280896171077561, 
2.8668699230375667, 4.863163621132891]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acosh_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(69.1461040561)+(1.-msk_arg)*(30.2458367587)
      res=acosh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(4.9293166042)+(1.-msk_ref)*(4.10223233506)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acosh_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([7.9349992375790723, 16.058994767364677])+(1.-msk_arg)*numpy.array([18.084077328224183, 
86.692301520006765])
      res=acosh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([2.7604360707751892, 3.4684454815251367])+(1.-msk_ref)*numpy.array([3.5874137011259082, 
5.1554789999380386])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acosh_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[54.669868001095011, 90.182976252039182, 9.8744725701028209, 47.030638292916322, 
22.774817461242101], [2.1957097699843335, 69.698751214316658, 59.194528795498897, 54.071443794067434, 50.666451935735751], 
[47.597376725897199, 99.389548141560027, 90.7157750923935, 1.9766934537997654, 34.751329198386358], [77.422487636994219, 
40.049138574357805, 18.577999204991986, 41.492944566347425, 
51.678616374353147]])+(1.-msk_arg)*numpy.array([[73.137714974525537, 16.888239580310195, 46.734337750870637, 
47.840749536761876, 6.1940799031624323], [75.965598117005882, 52.990699440731987, 70.835012292379119, 83.78088451827368, 
89.950212877175801], [1.5019185131551049, 19.415784343062032, 73.14250595827896, 69.123754708851934, 78.763697202043161], 
[53.709425083701355, 14.428583725693096, 69.277955078245796, 90.530207066590208, 69.15791986544636]])
      res=acosh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[4.6943762227350545, 5.1949571159699293, 2.9805261962588196, 4.5438334033198586, 
3.8183202786556167], [1.4232248987097744, 4.9372781152146041, 4.7739049443572288, 4.6833678674671484, 4.6183137738298194], 
[4.5558144607922699, 5.2921688302080536, 5.2008480681562608, 1.3033960824141992, 4.2411579250566751], [5.0423827475998708, 
4.3830984418224315, 3.6144000950287598, 4.4185253429366149, 
4.6380976444474209]])+(1.-msk_ref)*numpy.array([[4.9854446111005677, 3.5188869836932413, 4.5375118754215382, 
4.5609157092183974, 2.5101604549559351], [5.0233844374365031, 4.6631745536828735, 4.9534507545748685, 5.1213164353418286, 
5.1923726077648311], [0.96413764888109166, 3.6585697030355568, 4.9855101214147686, 4.9289932989472112, 5.0595590750277184], 
[4.6766490050147187, 3.3611553711177127, 4.9312218340750391, 5.198800250186955, 4.929487489249385]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acosh_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[22.545022974532376, 48.532251571022009], [24.966619067233186, 75.022956186250653]], 
[[21.34982292829358, 33.176776950245866], [64.043803364734373, 82.123601971952056]], [[16.940305292808596, 74.038749744124445], 
[2.4616914596336992, 36.162095091114836]], [[27.762186896139401, 45.353540869915129], [36.519863360274407, 
33.551947706709313]], [[15.41834897505694, 12.384949195402966], [79.79531219493397, 86.417305466744608]], [[93.773965386966324, 
95.026528867303981], [60.89104630453015, 7.3569260967003274]]])+(1.-msk_arg)*numpy.array([[[47.533832992105872, 
60.452726557735176], [68.545634445442658, 4.0603884930954619]], [[50.79690844428918, 20.850774712992095], [50.114629531972511, 
71.004800270985612]], [[34.143476381728526, 15.100354000832803], [40.592210146937731, 94.217662366016782]], 
[[16.982825957022335, 52.169300870739129], [37.565690653677876, 4.6780958561374781]], [[35.002244673320504, 
71.421770285190988], [86.077136032139791, 13.212632247762436]], [[3.8009396634829575, 38.884316780367328], [62.06398844176811, 
37.636295499504236]]])
      res=acosh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[3.8081692914433791, 4.5752695814338393], [3.9102855640593375, 5.0108969095366813]], 
[[3.7536417064648036, 4.1947701163602336], [4.8526535001127797, 5.1013355635698421]], [[3.5219705918111086, 4.997690172793857], 
[1.5499254869390875, 4.2809674232597432]], [[4.0164975698326533, 4.5075138710823843], [4.2908159936557455, 
4.2060199446417528]], [[3.427652176670434, 3.2079952775178149], [5.0725726733929868, 5.1523016530004639]], 
[[5.2340060124168559, 5.2472755978955856], [4.8021658878411326, 
2.6841380488725473]]])+(1.-msk_ref)*numpy.array([[[4.5544782475559247, 4.794940445720032], [4.9205936876290419, 
2.0789051375519407]], [[4.6208857748269772, 3.7299627512059401], [4.6073605947309497, 4.9558450744483098]], 
[[4.2235042017084679, 3.4067671686209522], [4.3965716017998338, 5.2387266791850857]], [[3.5244818445928523, 
4.6475495273455776], [4.3190611281140923, 2.2244139011038695]], [[4.2483552555718305, 4.9617008964503997], [5.1483572622095215, 
3.2728853946467038]], [[2.0106237715436324, 4.3535727962682076], [4.8212481963238476, 4.3209395319370323]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_acosh_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[39.22051413728142, 16.451548271957513, 13.367814378708413, 52.236141473314255], 
[21.842674461155802, 84.401582507200715, 77.772308283088861, 64.502329490910327], [62.956465665643314, 16.784359788068979, 
63.269486908381872, 19.795193771033606]], [[79.709904390238293, 30.360730917426654, 21.35374482919757, 69.252077971092021], 
[26.152393156452423, 66.67165116195504, 72.586620155051122, 41.364022791714973], [29.508361682222571, 50.467456625714341, 
44.883636329856351, 38.791588946739324]]], [[[69.986784333169794, 73.592531860666284, 53.792543579884146, 74.384695035866386], 
[84.195963758294866, 70.740080300322916, 40.178493608507864, 49.247592556788085], [25.490786653965674, 45.913723295413945, 
74.010825417603215, 93.979527942473311]], [[82.9168363371287, 72.053654280911218, 82.575366256087264, 11.254338908132734], 
[51.732039139601504, 48.895272233811774, 4.2513448478433489, 84.706091350287537], [57.957192360619167, 35.19550065073188, 
61.499879481615615, 10.531437440198969]]], [[[49.506350615395696, 94.8771288599184, 10.855230017485445, 83.693813247976991], 
[22.02627066703063, 10.18193441002081, 85.43066373128606, 29.755020598191951], [88.818296086422151, 92.104708929283049, 
17.418979706307901, 5.628857473834227]], [[65.011731842827814, 88.191390441506101, 11.513308159247467, 98.701635779199549], 
[49.162700887163879, 43.341026765912439, 57.164538560741171, 32.578357578010383], [95.77104080393994, 45.087478446046603, 
75.966317669756705, 16.902032605795494]]]])+(1.-msk_arg)*numpy.array([[[[51.93524062094086, 55.56188555649679, 
34.346516246341118, 90.45148873891614], [88.998734483268095, 5.0509669397114241, 73.169458975107929, 29.721679180468957], 
[54.382914104598484, 91.052679902822433, 97.373746501584861, 66.925865557088102]], [[88.936152878022028, 13.311702729106779, 
27.540824187705063, 70.455294997849705], [4.6510994158635333, 98.181703570535632, 70.17055621983404, 69.585006891269572], 
[43.63587285735646, 69.850975118739044, 89.909176124186615, 3.2721960445789291]]], [[[13.636817210463459, 38.679684278762807, 
57.904110306753978, 18.254856931500502], [3.3436219840068189, 55.538115366274191, 52.794413593006333, 55.96935888839986], 
[37.302939486600472, 12.38069599992061, 15.63076018617684, 63.57624974886145]], [[30.109449278601893, 45.152132637988075, 
15.668987830474366, 98.429590860109116], [76.11525076716859, 5.109461461638773, 61.904178108582315, 13.655336687749809], 
[82.23517853943072, 66.372128885684944, 8.7767566079109347, 64.933931397043921]]], [[[84.82452843539825, 61.699816548514583, 
72.972789576967585, 98.786873672298967], [83.589908178439075, 68.083980125351772, 96.42838827478586, 26.800882134550694], 
[89.565022760769523, 4.9459920152379038, 56.089301900710488, 54.869105561743964]], [[82.953308235025403, 88.364359382459, 
17.209367530257058, 90.303584262450812], [44.936563675350584, 8.9574623009288974, 70.844782003420576, 4.5680109107409219], 
[15.201817150760727, 41.727334883447654, 60.635198492985197, 17.097307908536919]]]])
      res=acosh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[4.3621845481851631, 3.492641800322386, 3.2845951365080137, 4.6488301670778451], 
[3.7764883721179325, 5.1286982357423874, 5.0468912788135061, 4.8597884262289037], [4.8355275656366086, 3.5127060622185793, 
4.8404878955546717, 3.6779477387966684]], [[5.0715016798615231, 4.1060258812474935, 3.7538255884913405, 4.9308481995040587], 
[3.9567221577648555, 4.892870776739378, 4.9778803376661802, 4.4154125219628702], [4.0775336158817979, 4.614377715330801, 
4.4970963410189162, 4.351184446119948]]], [[[4.9414025659583505, 4.991644567951254, 4.6781956353143102, 5.0023522035231691], 
[5.1262588963289, 4.9521095372948354, 4.3863241473819956, 4.5899045699930827], [3.9310792920304141, 4.5197926224212637, 
4.9973129089595689, 5.2361958443894858]], [[5.1109489496525606, 4.9705100634069899, 5.106821921317942, 3.1119212602203246], 
[4.6391310537016492, 4.5827233038268904, 2.1262543285174993, 5.1322998519658052], [4.7527774214339056, 4.2538635507441196, 
4.8121162906864772, 3.0452502908617465]]], [[[4.5951461173250792, 5.2457020808174573, 3.075665797542356, 5.1202765470631366], 
[3.7848673464618128, 3.0113419681128586, 5.1408180211419525, 4.085862567627526], [5.1797081538970993, 5.2160437799085511, 
3.5498826217495281, 2.4130681979279331]], [[4.8676557685664195, 5.1726243801260221, 3.1347594285872793, 5.2852230370706357], 
[4.5881789532010986, 4.4621137519308363, 4.7390044182476823, 4.1765597380222541], [5.255080274070413, 4.5016287478559649, 
5.0233939102993865, 3.5197048078506481]]]])+(1.-msk_ref)*numpy.array([[[[4.6430520511217708, 4.7105636437885314, 
4.2294357860606553, 5.1979302932632567], [5.1817377668383751, 2.3027804148879656, 4.985878588007731, 4.084740774127126], 
[4.6891126645936509, 5.2045552635783325, 5.2716774439988328, 4.8966768833095626]], [[5.1810343009539483, 3.2803769144789663, 
4.0084868380761431, 4.9480752088000406], [2.2184886032997042, 5.2799411249731989, 4.9440251999827893, 4.9356446716896913], 
[4.468895442359555, 4.9394599840680531, 5.1919162591553993, 1.8543969291523428]]], [[[3.3045733893314155, 4.3482945477013208, 
4.7518609814017001, 3.5968273022747765], [1.8770503989976142, 4.7101356682193938, 4.6594628563023477, 4.7178717425860901], 
[4.3120396012444635, 3.2076506765410966, 3.4413631406165042, 4.8453252923078782]], [[4.0977103564565418, 4.5030620449951995, 
3.4438108245314702, 5.2824628544391485], [5.0253526752809741, 2.3145245102232059, 4.8186696113862633, 3.3059341765651107], 
[5.1026933838931949, 4.8883676480495204, 2.8619927715893199, 4.8664581961405453]]], [[[5.1336971847565973, 4.8153624609217802, 
4.9831868552841527, 5.2860863001531229], [5.1190341965356403, 4.9138351887800642, 5.2619209356708581, 3.9812337509996749], 
[5.1880808871830411, 2.2813448360566122, 4.7200128027313815, 4.6980145807006952]], [[5.1113887460984548, 5.174583875586106, 
3.5377558374706743, 5.1962936745874915], [4.4982751510718968, 2.8825036747979156, 4.9535886808707277, 2.2000230150801841], 
[3.4134785859353278, 4.4241599948938006, 4.7979547341963595, 3.5312118670180945]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atanh_float_rank0(self):
      arg=-0.592275223482
      res=atanh(arg)
      ref=-0.681163425555
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atanh_array_rank0(self):
      arg=numpy.array(-0.30414451933)
      res=atanh(arg)
      ref=numpy.array(-0.314080283997)
      if not isinstance(res,float):
         self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
         self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atanh_array_rank1(self):
      arg=numpy.array([-0.90096918623492839, 0.89206623716281741])
      res=atanh(arg)
      ref=numpy.array([-1.477344040629299, 1.4319535130794456])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atanh_array_rank2(self):
      arg=numpy.array([[0.98519290204234777, 0.51689917062873714, 0.54012700473460717, 0.84533750055363921, 
0.24888176173843712], [-0.9392510803025429, -0.05364983648229471, -0.057919784503267735, 0.11573973845788488, 
0.85794005699530351], [-0.92930387905336165, 0.55061929413156707, 0.40552427158099191, 0.84311800596832653, 
0.013895039908068174], [0.8401729689435331, -0.33703957780521365, -0.040912637679468289, -0.86122082649501897, 
0.48720415915218473]])
      res=atanh(arg)
      ref=numpy.array([[2.4491823553669319, 0.5720990612575908, 0.60433490425095848, 1.2395860734572051, 0.25422037941828374], 
[-1.7316539185692674, -0.053701399089427691, -0.057984683044005801, 0.11626073683047372, 1.2854872771886485], 
[-1.6532619136976088, 0.61926962449004286, 0.43024288539127159, 1.2318600462709175, 0.013895934260011041], [1.2217613382355634, 
-0.35074894261940537, -0.040935487755641473, -1.2980519598542541, 0.53238771906683224]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atanh_array_rank3(self):
      arg=numpy.array([[[-0.70873389223852024, 0.29216983907022298], [-0.34764082484643954, 0.4609463865990242]], 
[[-0.3157953216783046, 0.69690123663956305], [-0.84266027572661573, 0.68400866208510114]], [[0.60027950228619553, 
0.20819865362890644], [0.69265477257086649, 0.92679830194964596]], [[-0.03174374275803038, 0.12834621876984587], 
[-0.79905908818095572, 0.31876953472907998]], [[0.53890714929507189, 0.82027600653491572], [0.77304714505073613, 
-0.13299531994323488]], [[-0.7609942801729046, -0.49162454054003873], [0.82363013796126672, -0.025268336515463563]]])
      res=atanh(arg)
      ref=numpy.array([[[-0.88463532619496632, 0.30093697543387704], [-0.36275775658661402, 0.49851234023280322]], 
[[-0.32696972948855829, 0.86125018032960232], [-1.2302791459943971, 0.8366087684977741]], [[0.69358401737573838, 
0.21128763913971543], [0.8530409058154681, 1.6351981876841166]], [[-0.031754411564134777, 0.1290580056518027], 
[-1.0960040923169947, 0.33027686972236175]], [[0.60261418297663583, 1.1576605580388457], [1.0278562939053173, 
-0.13378787779369869]], [[-0.99857318822928942, -0.53820041632737792], [1.1680005479882387, -0.025273716426830083]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atanh_array_rank4(self):
      arg=numpy.array([[[[-0.6730966529289274, 0.32223638760630924, -0.17694096774557, 0.82002187611552535], 
[-0.45364479114171785, 0.062396165630533407, -0.64757202687839666, 0.83555903656175823], [0.28219073181642984, 
0.13224764389607668, 0.24069950330112877, 0.87149678605670244]], [[-0.50959648829095783, -0.37501056473348637, 
0.68724018021680844, 0.83173427013664769], [0.29369720105615871, 0.52577993282276725, -0.15137437850112057, 
-0.13430814536470403], [-0.27723616014013963, 0.51107607380739539, 0.52186443855922038, 0.52987237119958541]]], 
[[[0.29055804089408821, -0.28060697243062671, 0.85843826667034073, 0.46597283712533288], [-0.85749647247273497, 
0.31644828935463032, -0.83556201674416708, 0.37962733591145192], [0.90312969568942747, 0.034920231241019506, 
0.62091690306413905, 0.98775237559246754]], [[-0.42686485519575723, -0.85630757323288975, -0.57027634279785699, 
-0.29083022560753868], [-0.92689389610889761, -0.63607689290972891, -0.049854912994010236, -0.70920425273425103], 
[0.28191953281908133, -0.82225370166938572, 0.37458400030257288, -0.2014864525272827]]], [[[0.35762523565286175, 
0.26944932820183864, -0.038029136133831853, -0.020340175259131676], [0.78569173643263079, 0.76841312259123917, 
-0.65091172928642171, 0.43181541566512727], [0.49063010048302669, -0.59380656063437987, -0.68610655213728944, 
-0.25253434349364534]], [[0.52449702292094469, 0.30148163708183695, 0.82822465187764793, -0.12834148634130405], 
[0.15835907415917694, 0.83897627025447941, -0.87763968653087643, 0.79306994361302574], [-0.67161515811495176, 
0.4704282595318765, -0.62512993136713035, -0.012121589647265507]]]])
      res=atanh(arg)
      ref=numpy.array([[[[-0.81638345952037117, 0.33414062108097176, -0.17882301245849291, 1.1568842451565404], 
[-0.48928000032645547, 0.06247733059269267, -0.77110585725573877, 1.2062763726259871], [0.29006075631094286, 
0.1330268158020784, 0.24551650236925326, 1.3392699473899439]], [[-0.56218456114820547, -0.39424097374686673, 
0.84270694828590376, 1.1937370152455524], [0.30260770308512697, 0.58429474948353344, -0.15254674726922496, 
-0.13512458065784855], [-0.2846856214396713, 0.56418520022690932, 0.5788985864331927, 0.58996769245517655]]], 
[[[0.29917565287489067, -0.28834080126133327, 1.2873779362729365, 0.50491382461901624], [-1.2838090617689726, 
0.3276951951601047, -1.2062862460561803, 0.39962416333117762], [1.4889410016933142, 0.034934435807764869, 0.72649591174081529, 
2.5447139138141215]], [[-0.45605668187042359, -1.2793347639735624, -0.64793227647466367, -0.29947296104682636], 
[-1.635876370684437, -0.75155695777511855, -0.049896279702681033, -0.88558103530778409], [0.28976611727378676, 
-1.1637360341554719, 0.39374469562548337, -0.20428142299155708]]], [[[0.37416021185569842, 0.27626994544184008, 
-0.038047484829673912, -0.020342981019924898], [1.0600728420389327, 1.0164413995080823, -0.77687908129428629, 
0.46212605059376072], [0.53688986252588011, -0.68352549976418031, -0.84056187222254519, -0.25811794640092417]], 
[[0.58252333534645739, 0.31114857393382944, 1.1824565659150161, -0.12905319396445158], [0.15970311098265275, 
1.2177062885300067, -1.3654003500073009, 1.0796518188965465], [-0.81367968976138194, 0.51062016365130891, -0.73338178352318573, 
-0.01212218338652719]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atanh_constData_rank0(self):
      arg=Data(0.252669468172,self.functionspace)
      res=atanh(arg)
      ref=Data(0.258262280724,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atanh_constData_rank1(self):
      arg=Data(numpy.array([0.60482296159426152, 0.29426855259200457]),self.functionspace)
      res=atanh(arg)
      ref=Data(numpy.array([0.70071743109589513, 0.30323310564330574]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atanh_constData_rank2(self):
      arg=Data(numpy.array([[-0.88554031078222395, 0.032110894615181795, 0.13444811204388007, -0.98340152230522226, 
0.58357358742723742], [-0.61968002577233527, 0.46712231596467957, -0.38396258852289167, 0.73154147245823387, 
0.016436624285061097], [-0.83800168454640289, 0.69351059207308263, -0.24426365738526223, 0.034920141309882968, 
-0.84248606871474907], [0.39877272199113833, 0.50494478873489168, -0.27971673841325517, 0.39607744093103947, 
0.67106719283254024]]),self.functionspace)
      res=atanh(arg)
      ref=Data(numpy.array([[-1.4008734970279102, 0.032121938064922161, 0.13526712126158572, -2.3916288021256644, 
0.66786478538877636], [-0.72448547895771087, 0.50638312049936984, -0.40469920489571476, 0.93203543654793064, 
0.016438104712190812], [-1.2144241413410173, 0.85468786507758143, -0.24930331594108604, 0.034934345766830605, 
-1.2296785779993376], [0.42218873726973483, 0.55592109851298055, -0.28737474039865496, 0.41898788948862298, 
0.8126821212820865]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atanh_constData_rank3(self):
      arg=Data(numpy.array([[[0.8920991316890774, -0.62891448197758826], [-0.34623194907156873, 0.94164892935611388]], 
[[-0.47420917153717523, -0.36435474866326867], [0.97486840820128307, -0.98982128304404293]], [[0.17674219623824228, 
0.82069527472952308], [-0.27015070838265076, 0.50916616462175845]], [[0.12733139488530609, -0.44523245947401635], 
[-0.58158836032782502, -0.84939752030168991]], [[0.30120758019186833, -0.30971957193542432], [0.91271738728116403, 
0.22041801309090348]], [[-0.77328769987422763, 0.14676688358830869], [-0.56761092118786527, 
0.75086748616054289]]]),self.functionspace)
      res=atanh(arg)
      ref=Data(numpy.array([[[1.4321146119124104, -0.73961828356445547], [-0.36115609685828731, 1.7524075737057769]], 
[[-0.51548675048628934, -0.38189812872415979], [2.1820656771199056, -2.6377505693757226]], [[0.17861782410632548, 
1.1589434959740343], [-0.27702638872024621, 0.56160345936335765]], [[0.12802632109925119, -0.47873813955476857], 
[-0.66485959027643227, -1.2539857098199254]], [[0.3108471446808504, -0.3202351967718452], [1.5435644749039263, 
0.22409542545583833]], [[-1.0284543734297438, 0.14783452807742692], [-0.64399110240843882, 
0.97494085573700717]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atanh_constData_rank4(self):
      arg=Data(numpy.array([[[[-0.88846480539084438, -0.44491279408665729, -0.077686051674781265, 0.47979809836292686], 
[-0.17029746090235398, 0.29580382307515629, 0.19011591036873665, 0.66855458553324931], [0.30801679590554865, 
-0.2879724929278199, 0.78576520537846095, 0.59773428543732754]], [[-0.52735574710678035, -0.3137741652764614, 
0.49310861863991051, -0.97143883707603007], [0.38553971682998411, -0.078643529600947781, -0.36594906760074153, 
-0.96973919595926328], [0.86800770402970784, 0.50582254506419244, -0.64640091021434842, -0.50954345228675424]]], 
[[[0.51415882021974713, -0.48482612240226264, -0.23713296711566911, -0.86725807976023939], [0.78195095222909727, 
-0.4225459339949339, 0.54731993777730792, 0.68493377829057711], [-0.22543428446311242, -0.67810909046049406, 
-0.75108158402250647, 0.23538509232764193]], [[0.72556003229385135, 0.83323501315639636, -0.46235293776476927, 
-0.50409240906096109], [0.41066703397967208, -0.94158308935916757, 0.34511298296575665, -0.81733489904768653], 
[0.37371934352749059, -0.26731869019880539, -0.38768200899174854, -0.73994263219110468]]], [[[-0.054675292166386669, 
-0.0297176440519199, -0.92045930365672723, 0.098300639288722458], [-0.46898188353977732, -0.7089669178254201, 
0.85614394846923303, 0.028784782616051441], [0.85064862284936127, 0.40408542721476848, -0.46333269224469198, 
0.82081566547599016]], [[0.16351845780989893, 0.59639700857148426, -0.053196690356647758, -0.36633442333444188], 
[-0.01023247310289721, -0.65615391605328877, -0.38393275055599618, -0.59877823224146831], [0.20639377993133756, 
-0.88780111517162141, 0.0048053861544190557, -0.75821276394777248]]]]),self.functionspace)
      res=atanh(arg)
      ref=Data(numpy.array([[[[-1.4145896589443596, -0.47833950971870176, -0.077842901649064863, 0.52272196441540397], 
[-0.17197299282014211, 0.30491475831321313, 0.19245742341360061, 0.80812493958192733], [0.31835284452889112, 
-0.2963540029098421, 1.0602648520711428, 0.68961449064156388]], [[-0.58647512593584161, -0.32472626872843585, 
0.54015940206906121, -2.1172355370100515], [0.40655036789798132, -0.078806265548285959, -0.38373773370652531, 
-2.0879015730238049], [1.3249420929857387, 0.55709994812645591, -0.76909158065316929, -0.56211292363623389]]], 
[[[0.56836677482415066, -0.52927424250288346, -0.2417340554754521, -1.3219097862663829], [1.0503720281964324, 
-0.4507872783325651, 0.61454701469436812, 0.83834934845827569], [-0.22937403642069484, -0.82560509960764361, 
-0.97543186395866532, 0.23988284884362557]], [[0.91928719651366653, 1.19862594752313, -0.50029984596893473, 
-0.55477767202020001], [0.43641330699597325, -1.7518267655233581, 0.35988528151692822, -1.1487359694393304], 
[0.39273929295660848, -0.27397392392486608, -0.40906913697399028, -0.95035258478210216]]], [[[-0.054729871975665194, 
-0.029726396954292158, -1.5920254313482096, 0.098619114670633334], [-0.50876435264914155, -0.88510369007421297, 
1.278721655664004, 0.028792736579406359], [1.2584948489941765, 0.42852205289512801, -0.50154670770732024, 1.1593123848744229]], 
[[0.16499969599771161, 0.68753639949866152, -0.053246955954725159, -0.38418274796123575], [-0.010232830250603205, 
-0.78602957412183538, -0.40466420780410445, -0.69124035022085617], [0.20940172380993008, -1.411447472858995, 
0.0048054231431678219, -0.9919974737413364]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atanh_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(-0.0073816096608)+(1.-msk_arg)*(0.378540181545)
      res=atanh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-0.0073817437353)+(1.-msk_ref)*(0.398354561225)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atanh_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([-0.36068399900471526, -0.66001542666967361])+(1.-msk_arg)*numpy.array([-0.83990128831485988, 
-0.011534470732304847])
      res=atanh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-0.37767196817629906, -0.7928409652294659])+(1.-msk_ref)*numpy.array([-1.2208383142386112, 
-0.011534982303908405])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atanh_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[0.40851633886459293, -0.63074171005432322, -0.42011494996643473, 0.62258527806476205, 
-0.70927634775089687], [0.57563872433050545, 0.89580874818428535, 0.59463305514359543, 0.78042503533665819, 
-0.84181561961798679], [0.21482755103998041, -0.75163716999324692, 0.030702031675672092, 0.59894677907918492, 
0.58481215813243059], [-0.095006330465277511, 0.59374075914377933, 0.96624965328866863, 0.53142579238221166, 
-0.25832948279377899]])+(1.-msk_arg)*numpy.array([[0.27379299145328129, -0.17891419108552042, -0.92089047996482387, 
-0.40656316823098104, 0.18734626051148218], [-0.82408676613390897, 0.66910336676105064, -0.13716901193460351, 
-0.98852541078703626, -0.036756041923461047], [0.86663909937034012, -0.29716862808377309, -0.55331696951439202, 
-0.2692767005677047, -0.77213657344228326], [0.22084616982685956, -0.33486767981833598, -0.018039138508949337, 
-0.28500381503978234, 0.25277670062263868]])
      res=atanh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.43382906358177242, -0.7426469276098554, -0.44783160184927723, 0.72921568292805794, 
-0.88572610206819546], [0.65591542530985392, 1.4505863182124148, 0.6848031178472832, 1.0464568499038447, -1.227372898390757], 
[0.21822701927032237, -0.97670772704401232, 0.030711683864531845, 0.69150314470066609, 0.66974505911929516], 
[-0.095293737397257217, 0.68342386529869115, 2.0324462939235532, 0.59212999384756426, 
-0.26431760766071033]])+(1.-msk_ref)*numpy.array([[0.28095961296185329, -0.18086074762134002, -1.5948554635967342, 
-0.4314868817981859, 0.18958547427929348], [-1.1694219272867736, 0.80911790738780032, -0.13803915019710869, 
-2.5775068432834156, -0.036772607905948919], [1.319417913980055, -0.30641109801107735, -0.62314933236368897, 
-0.27608381267756232, -1.0255973790455357], [0.224545490710478, -0.348300725550931, -0.018041095599547939, 
-0.29311987963383501, 0.2583768292710914]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atanh_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[0.57257162353515412, -0.32389028749278037], [0.13351791263938284, 0.78427597660230042]], 
[[-0.80935463400484675, 0.75459196912557913], [0.27301914295107976, -0.25961801580951827]], [[0.63408835436754973, 
-0.11789377533246037], [-0.91962437176541378, -0.29696275764250879]], [[-0.073535901242307244, -0.73836733694795464], 
[-0.68119163199649968, -0.6006271329474353]], [[-0.83967803109674632, -0.14078291952600208], [0.50419616484559127, 
-0.32337389200167632]], [[-0.61246360170511849, 0.088846889520762584], [0.64280146796580895, 
0.051459918031424801]]])+(1.-msk_arg)*numpy.array([[[0.67596035704940927, 0.54133927862067521], [0.47987783537812723, 
0.13800459298873613]], [[-0.48461497906344375, 0.3366901903651367], [0.59255236224005325, -0.76341773081402509]], 
[[0.15394232671115837, 0.28724595650747475], [0.23026676807433488, 0.8592705380164245]], [[0.1019687674962273, 
-0.47365826039743153], [-0.56827759355462681, 0.62810125820723739]], [[-0.84642329302616359, -0.484496129899461], 
[0.96609519692172574, -0.18040248950270954]], [[0.85255742610578933, -0.94931596594765721], [-0.12068134749277448, 
-0.79714440747925086]]])
      res=atanh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.65134040059411202, -0.33598725457473233], [0.13431991811616778, 1.0563840376335927]], 
[[-1.1251552626090238, 0.98353467762640612], [0.28012324517864956, -0.26569877202959735]], [[0.74822422375793407, 
-0.11844457559781346], [-1.5865869036778408, -0.30618530146967871]], [[-0.073668882135729849, -0.94688008453442785], 
[-0.83133396535261062, -0.69412765256021425]], [[-1.2200808766439255, -0.14172423705854545], [0.55491678495101304, 
-0.33541044537932196]], [[-0.71285435018173549, 0.089081782004568774], [0.76293326815515639, 
0.051505414408319738]]])+(1.-msk_ref)*numpy.array([[[0.82163788217855449, 0.60604810564056089], [0.52282555186969859, 
0.13889085417002639]], [[-0.52899825472633255, 0.35035483234748388], [0.68159041965165978, -1.0043565444052576]], 
[[0.15517596997344457, 0.29556194843564915], [0.2344711533444559, 1.2905500839798876]], [[0.10232439994392903, 
-0.51477625199033628], [-0.64497520895855776, 0.73827406240297311]], [[-1.243402764849896, -0.52884293814798533], 
[2.0301240165441263, -0.18239868660008168]], [[1.265441792748242, -1.8248114262659678], [-0.12127238811631265, 
-1.090729942429308]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_atanh_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[0.54543132108932202, 0.95857363916851623, -0.84193591366344922, -0.7004846040396161], 
[0.45509408474851365, 0.16115064544310309, 0.70046400503563211, 0.69241670794584809], [0.22095518579192674, 
0.46133081082335092, -0.48381701258387377, -0.4708433913343355]], [[0.73186475957686947, 0.83814883460657597, 
0.37549155530584577, 0.16606111814475333], [0.10064084559275632, -0.013443218099880139, -0.58391435092153987, 
-0.10346854092738589], [-0.91344968501119916, 0.58668002656118312, 0.27925779190981248, 0.35875938533298024]]], 
[[[-0.71264008915975041, -0.055810295615554062, 0.17272780023364964, -0.29290164462362311], [-0.83975806569683153, 
-0.029699629370412262, -0.32310271813647329, 0.54234927457538484], [0.71873354976624082, 0.92412623520963333, 
0.089694420694559573, -0.54206763259144863]], [[0.65277048961299622, -0.30944572115792723, 0.21592126467306971, 
0.41781008841634182], [-0.21328270445537123, 0.015794450161580986, -0.62840160997092775, 0.70057457643833909], 
[-0.74140733317829366, 0.91048513429638755, -0.42172328658118741, 0.70878973971822501]]], [[[0.29811402565337719, 
0.033340893494908874, 0.35206869783576078, -0.68187138115999746], [-0.45153489833512384, -0.85114789782211209, 
-0.59265496562576048, -0.64545611715268913], [0.68323576919213536, 0.12377344715524852, -0.23493951978904526, 
0.087301476711074688]], [[0.37969403740036878, 0.62730578369044787, 0.12483832022798635, 0.16254185035410673], 
[0.76174398144100808, -0.90534304427796486, 0.87924086058617945, 0.047091028378077882], [-0.41341689456711161, 
-0.20818830059120086, -0.19047018293666174, 0.072909624584449606]]]])+(1.-msk_arg)*numpy.array([[[[0.22208319766123275, 
0.46340740412220582, -0.097424291026602328, 0.11568698873852701], [-0.98624873002751534, -0.76549677165799812, 
-0.35021366573936352, -0.019976347821733187], [0.65672597023964219, -0.88422290570584372, 0.25144724149735898, 
-0.3732143459677395]], [[0.41556930444255769, -0.86104682526150178, -0.70065002685674083, -0.20580664538753868], 
[0.5201214448774818, -0.54973751181237152, -0.35977730890165827, 0.46690501522389116], [-0.5282172061774435, 
-0.80385799007156067, 0.39211607437822016, -0.66667896103084967]]], [[[0.29031943454074183, -0.46368441974437091, 
-0.018321221302155077, -0.21374906211140521], [0.76742319992955688, 0.36144293047594056, -0.74853954386634891, 
-0.6135123837318015], [-0.96251057553921437, 0.87849226466130492, -0.48373945689941766, 0.21741204692280403]], 
[[-0.20170280430214904, 0.93409473383700581, 0.20685166728592441, 0.25020186849363601], [-0.73216711252524247, 
-0.66034469875454371, -0.15169716876105921, -0.094166266992694037], [-0.50960832482416407, -0.88987393375969859, 
0.40119795755760879, 0.17953204356008734]]], [[[0.57184585120248976, 0.86681967445998409, -0.34416201799913571, 
0.3746996425320257], [0.58607921736120949, -0.23206307834020212, 0.66194050886578415, -0.95480331624840598], 
[0.839818804942841, -0.57022095352352742, -0.67990710725475789, 0.68901426206832861]], [[0.39437813726864102, 
-0.26672942898401153, -0.8756676590066178, -0.41091386876807812], [-0.11999075586896257, 0.73468939034243497, 
-0.025477034497746631, 0.33769809340537127], [0.95675019330951239, 0.82484077353764196, 0.22383682587103215, 
0.96653016432048378]]]])
      res=atanh(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.61185465584025234, 1.9280271694586071, -1.2277859319908841, -0.86825136442186024], 
[0.49110634668898889, 0.16256779570300137, 0.86821092154298474, 0.85258343604933162], [0.22466009924067468, 
0.49900059025434534, -0.52795588992421827, -0.51115340808945875]], [[0.93273126050763833, 1.2149185476140958, 
0.39480079458990719, 0.16761333297212469], [0.10098270850694156, -0.013444028007666106, -0.66838168662365327, 
-0.10384016671129655], [-1.5479685443885605, 0.67258851577017043, 0.28687690652036857, 0.37546129267966111]]], 
[[[-0.89252794791885171, -0.055868349917136607, 0.17447699265081723, -0.30173726778297477], [-1.2203522966755389, 
-0.029708366358924737, -0.33510763273758642, 0.6074778755218293], [0.90502027439788846, 1.6165781350879536, 
0.089936121677470504, -0.60707895483541885]], [[0.7801111344467011, -0.31993231762569901, 0.21937393398124189, 
0.44503603293543431], [-0.21660798898553624, 0.015795763743893625, -0.73877026545911462, 0.86842803783199995], 
[-0.95359738437833441, 1.5303538885337276, -0.44978624992427807, 0.88474754709177061]]], [[[0.30744838973463395, 
0.033353255822344691, 0.36780319588354826, -0.83260330352013057], [-0.48662658673554626, -1.2603040004020682, 
-0.68174855787831479, -0.76747038665799028], [0.83515776363180283, 0.12441138559949382, -0.23941119311568423, 
0.08752428729257955]], [[0.39970209858145894, 0.73696137240641402, 0.12549297139340418, 0.1639964226197704], 
[1.0003568461096584, -1.5010789916276233, 1.3724125991597682, 0.047125883904536134], [-0.43972551301321644, 
-0.21127681702249096, -0.19282500670767125, 0.073039229528423913]]]])+(1.-msk_ref)*numpy.array([[[[0.22584632000713087, 
0.50164184714498694, -0.097734292351408403, 0.1162072712290204], [-2.4864359484929865, -1.0093590160437009, 
-0.36568726874926771, -0.019979005675118622], [0.7870347889621544, -1.3948020108589234, 0.25695713337570875, 
-0.39215244302416297]], [[0.44232443052503223, -1.2973787068391707, -0.86857622902238441, -0.20878854298079286], 
[0.57650622343315872, -0.61800506428203683, -0.37663007559769091, 0.50610520614548593], [-0.58766920658699073, 
-1.1094220203612399, 0.41429813612496968, -0.80474108639907505]]], [[[0.29891506469220636, -0.50199467702650291, 
-0.018323271659159746, -0.21709662300423554], [1.0140287261090664, 0.37854467124300167, -0.96962521351160713, 
-0.71453442986278926], [-1.9789604798140263, 1.3691234105425401, -0.52785463682439626, 0.22093814771511264]], 
[[-0.20450693986196555, 1.6895881542206337, 0.20988003154243046, 0.25562814987152344], [-0.9333826689691338, 
-0.79342461308277257, -0.15287712400320616, -0.094446090341075054], [-0.56220054986503887, -1.4213198189069196, 
0.42507588512607902, 0.18149910474055941]]], [[[0.65026130883411193, 1.3201437560579912, -0.35880614157075813, 
0.39387921909822665], [0.67167286980455521, -0.2363688731075875, 0.79625964530022075, -1.8835105730199122], 
[1.2205583632989023, -0.64785019597412397, -0.82894126703556037, 0.84607665423762712]], [[0.41697400635766257, 
-0.27333942148962681, -1.356880905083754, -0.43671025067513475], [-0.12057164922760653, 0.93884119955001089, 
-0.02548254885058452, 0.35149203539390422], [1.9060237738461292, 1.1717762945438432, 0.2276916839854089, 2.0366906625593297]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_exp_float_rank0(self):
      arg=3.09787887997
      res=exp(arg)
      ref=22.1509166631
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_exp_array_rank0(self):
      arg=numpy.array(3.06107617426)
      res=exp(arg)
      ref=numpy.array(21.3505216848)
      if not isinstance(res,float):
         self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
         self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_exp_array_rank1(self):
      arg=numpy.array([2.9441216067800298, 2.2198899917643411])
      res=exp(arg)
      ref=numpy.array([18.993970881446675, 9.2063180393692186])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_exp_array_rank2(self):
      arg=numpy.array([[3.6056927875227256, -0.514598042378692, 0.27276644455950372, 1.2073866779129707, -2.4234400805066536], 
[-0.57101471476892129, 1.6765859686366458, 1.0960945354606153, -4.4908510808295503, -3.75848628424695], [2.4578611525294844, 
-0.24806530303971641, 4.4931916527206859, 2.5860570534517606, -0.011166085853192342], [-0.78483127839221467, 
4.6543717917849889, -0.11770845753128967, 4.9679755233877927, -4.54621768888933]])
      res=exp(arg)
      ref=numpy.array([[36.807174577692315, 0.59774081281151281, 1.3135934121069111, 3.3447323583010529, 0.088616245489775675], 
[0.56495188273175689, 5.3472690286144813, 2.9924562410242372, 0.011211098197367505, 0.023319012028386241], [11.679803487164589, 
0.78030898506554647, 89.406344991535917, 13.277316508041498, 0.98889602349572636], [0.45619666552760763, 105.04321025952407, 
0.88895518302765453, 143.73560325433471, 0.010607248520354761]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_exp_array_rank3(self):
      arg=numpy.array([[[-0.20330917011009753, -0.42711392199616949], [4.9722652802793323, -1.1462128361773316]], 
[[2.2627935053851207, 2.814345122878307], [4.3983006838999739, 0.5061063048639074]], [[3.1690672921062522, 2.9409598978320028], 
[2.0498711170829731, 0.37840926930352925]], [[4.953102187502969, -1.3042778317056678], [-0.48014120628515045, 
0.19916724476781233]], [[2.1395834252086416, -1.9421140583243721], [4.7146287165921752, -4.8194619668850507]], 
[[-1.9536227133941333, 2.9170900873219106], [-1.852813059291194, -2.114093795593103]]])
      res=exp(arg)
      ref=numpy.array([[[0.81602591160026805, 0.65238922651135023], [144.35351845437506, 0.31783819826500992]], 
[[9.6098970044439902, 16.682247380194553], [81.312575427725733, 1.6588196659765442]], [[23.785289280209586, 
18.934012309476497], [7.7669000210657932, 1.4599603375703405]], [[141.61359513665687, 0.27136843798163401], 
[0.6186960218708083, 1.2203860520151399]], [[8.4958977147998844, 0.14340047214416518], [111.56738036571392, 
0.0080711285055064166]], [[0.14175958588154447, 18.487412356239044], [0.15679547039168934, 0.12074265751915175]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_exp_array_rank4(self):
      arg=numpy.array([[[[1.3616029687426909, 1.5242782350750597, -1.4581088082674221, -3.967647158564358], 
[0.89858122671547314, 4.0718725193363365, 0.027089157450978618, -3.28329099908622], [2.6820626725832835, -1.728754625873278, 
-2.8222860873642928, 4.2104389038284129]], [[4.9532570426359186, -2.049233776234789, -1.8405206037463282, 3.726411178506142], 
[0.69779122302662611, 3.6346118792407882, -1.9654592898258882, 3.6318625288807542], [-2.9530027724584542, 0.45730679695317455, 
2.2969456291666779, 1.0756915259632018]]], [[[2.2638859808180136, -1.6789415582643521, 2.9944569452685617, 
-3.8272419139222769], [-4.1157705217804388, -2.9686535006386583, 1.2299156457017553, 4.0058714009168845], [-1.4849359560797026, 
-4.703135568949258, 3.9560110454430095, -1.2600583271694186]], [[4.2791683087873693, -4.041249844053393, 4.5594928457107784, 
0.393468568755452], [-4.2732919475081257, 1.7813879899496428, -0.24233240402343803, -0.66900895112367031], 
[-1.8536864679694984, -3.4414366250487016, -4.7232377101446055, -0.031643077558968002]]], [[[1.2898930412135892, 
1.9626261017118036, 3.3367764256689973, -0.139770043758908], [-0.25056111967697703, 0.27207952343728792, -2.4059946107416108, 
-4.8665798508936877], [-2.2719991718129684, -0.38243599805692519, -4.6151955592668559, 0.64221258182516561]], 
[[-0.22706809743194789, 0.30078596195309171, -3.8123838847472968, 3.8530287538281947], [-2.9919654928160853, 
-0.025860593390909514, -3.2792411147122702, -1.0095421467593635], [-1.6339679492714354, 0.2016544581344526, 
-3.4240431543939618, -2.7694864864245092]]]])
      res=exp(arg)
      ref=numpy.array([[[[3.9024437862006489, 4.5918281524730178, 0.23267589362331229, 0.018917891612028306], 
[2.4561159662912329, 58.66671436070866, 1.0274594043414222, 0.0375046257947661], [14.615208616292277, 0.17750533292443776, 
0.05946983395400568, 67.386109341923571]], [[141.63552642680182, 0.12883358113005655, 0.15873476667817829, 41.52979736995465], 
[2.0093096854888151, 37.887145255575298, 0.14009152922353346, 37.78312328107728], [0.052182777449128671, 1.5798134918734594, 
9.9437640809836214, 2.9320197674311874]]], [[[9.6204013176412868, 0.18657134646821219, 19.974509692001728, 
0.021769575267381127], [0.016313365761911405, 0.051372436735400637, 3.4209409530447954, 54.919660596388667], 
[0.22651684723884583, 0.0090668028979747316, 52.248492849487441, 0.28363748224580493]], [[72.180383245976856, 
0.017575492058240771, 95.535016548511038, 1.4821126983533104], [0.01393583153378508, 5.9380927143285485, 0.78479526508636543, 
0.51221595737502357], [0.15665858365486429, 0.032018653470916453, 0.0088863604618923155, 0.96885232551075318]]], 
[[[3.6323980180906448, 7.117995121493899, 28.128306714059971, 0.86955817273831892], [0.77836390520942356, 1.312691386891687, 
0.090175761211049893, 0.007699654271823635], [0.10310584760050394, 0.68219755154811657, 0.0099002471325744159, 
1.900681643984323]], [[0.79686651587912238, 1.3509201622783746, 0.022095443134077623, 47.135609411415643], 
[0.050188694198622243, 0.97447092782786238, 0.037656823175597487, 0.36438577659298604], [0.1951536758964274, 
1.2234251904422988, 0.032580440530655411, 0.062694190795536106]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_exp_constData_rank0(self):
      arg=Data(1.23522013514,self.functionspace)
      res=exp(arg)
      ref=Data(3.43913551195,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_exp_constData_rank1(self):
      arg=Data(numpy.array([4.8381641803247177, 0.78109836687548828]),self.functionspace)
      res=exp(arg)
      ref=Data(numpy.array([126.23738979118131, 2.1838696387310765]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_exp_constData_rank2(self):
      arg=Data(numpy.array([[-1.4888842712342951, 0.12354715392259497, -4.3530220999707741, -1.7914479402637804, 
0.96944251880197108], [-4.8824309192518536, -4.0194100460515392, -3.0541510717416775, 3.1106459418759123, -3.508945983171162], 
[-0.66843763828019487, 1.0870296233059236, -0.98100659338971408, -3.8350421489353694, 1.511920201301078], [3.05587297494327, 
-0.27902345077889468, -4.8539175228073326, -0.34270673412748831, 4.9692388432501708]]),self.functionspace)
      res=exp(arg)
      ref=Data(numpy.array([[0.22562425062406413, 1.131503358105918, 0.012867865782757783, 0.16671859624907695, 
2.6364742647576169], [0.0075785687285209562, 0.017963559486895746, 0.047162741564586962, 22.435531770722218, 
0.029928442900719995], [0.51250867653925203, 2.9654524664089768, 0.37493350325470781, 0.021600428015944233, 
4.5354313794391823], [21.239719178673294, 0.75652216197307587, 0.0077977696942620684, 0.70984635474864455, 
143.91730204453654]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_exp_constData_rank3(self):
      arg=Data(numpy.array([[[-3.606562877265981, 3.1540139800969289], [0.21215810027223192, -3.3166417709184115]], 
[[3.9285951981583569, 0.14563386995394811], [1.1601886242980495, -1.629701342077039]], [[2.5438530417112322, 
0.73013114118609401], [-1.1083082687907764, 3.2024491034151552]], [[0.64785370579110069, -3.903352408227482], 
[-1.6220249865999081, 1.8319914759175333]], [[-1.8932594628656609, 3.6540242645296122], [-4.8316957353546197, 
-3.5251833535082189]], [[-4.1114260464382717, 1.7053131747004544], [1.4411303873851988, 
-4.3796771970277391]]]),self.functionspace)
      res=exp(arg)
      ref=Data(numpy.array([[[0.027144987360170986, 23.429923327259825], [1.2363433358456559, 0.036274445333861399]], 
[[50.835513661356828, 1.156772581309484], [3.1905350317923036, 0.1959880987855562]], [[12.728620513726009, 2.0753527540506629], 
[0.33011695811234032, 24.592686535075185]], [[1.9114339237157674, 0.020174165914119708], [0.19749836234489754, 
6.2463136636113799]], [[0.15058019751508733, 38.629810244166492], [0.0079729897156910114, 0.029446407771935447]], 
[[0.016384392953436227, 5.5031088318316073], [4.2254695358723344, 0.012529402496718325]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_exp_constData_rank4(self):
      arg=Data(numpy.array([[[[-2.7422761934197073, 4.2430196246454219, -0.63976027199661889, -1.0273932223054025], 
[3.3770837230275461, -4.7974232230463256, 0.3983871246932944, -3.8971913735501396], [0.48875532345956341, -1.2648426336433838, 
-4.2450072644066079, 4.8507221504923663]], [[2.4012313070412681, 4.4784373077905713, -3.045887189357479, 0.27562437981842791], 
[0.6607445887647696, -2.139078320440253, -1.9227318829736371, -3.9047088576444375], [4.195015125432036, 3.1528123928938481, 
-4.8999384817204517, 3.4837033803147648]]], [[[0.50173393545535649, 2.6331676820449479, -1.1056898311675498, 
1.1836826357233585], [0.66662216353643888, 2.8438271931572823, 1.7681204270434829, 1.4151416266077979], [3.6302585746373346, 
-4.6005171626897567, -0.45294451273506908, -2.0464152035031646]], [[1.7383130024796376, -3.2945700495053032, 
2.2278115494829169, -0.52288189399962093], [-3.9732366700972976, -0.88894529138167044, 3.4631475693295055, 3.7160339868798875], 
[-1.6200399971280999, 1.9691278697821355, 0.63989456886169105, -3.8415832797393481]]], [[[3.6716121344923813, 
-0.19445772220817226, 4.6734954060426546, 0.51832290881861898], [4.3660850387764558, 2.3232412955572714, 4.4030871862379772, 
-2.8491829598825777], [4.1543746418877028, 1.9118062767284174, -4.9476356018429799, 3.5188104707267307]], 
[[-4.3956472917365774, 3.6063463339745798, -3.9628631196406983, -2.5248002989007166], [1.0403542258087137, -2.0782813886753662, 
-2.3410697784468604, -2.3728605007112566], [3.0690015943301212, 4.2386062262192734, 0.31999920459386466, 
3.4140540774946633]]]]),self.functionspace)
      res=exp(arg)
      ref=Data(numpy.array([[[[0.064423539438927493, 69.617754252510991, 0.52741884595589328, 0.35793881238836001], 
[29.285242582886383, 0.0082509806168858252, 1.4894205097733149, 0.020298843326454694], [1.6302857781336109, 
0.28228371460514518, 0.014335629538791281, 127.8326709478624]], [[11.036757654979686, 88.09689671880443, 0.047554103772187974, 
1.3173529467296146], [1.9362334955059897, 0.11776333307454952, 0.14620699564625547, 0.020146819229917911], [66.354735213742359, 
23.401787138647993, 0.0074470411859944583, 32.580155639621161]]], [[[1.6515825268672815, 13.917787273656725, 
0.33098248144043196, 3.2663809722959245], [1.9476473627217405, 17.181396388754248, 5.8598290279379306, 4.1170695112738711], 
[37.722569455925431, 0.010046638654209359, 0.63575340886557041, 0.12919722017956209]], [[5.6877401210958149, 
0.037083985898147799, 9.2795360365135764, 0.59280966924003609], [0.018812444811274653, 0.41108910340310106, 31.91728042805272, 
41.101063080771844], [0.19789078386229075, 7.1644254509541145, 1.8962809417070745, 0.021459597887150874]]], 
[[[39.315236246467386, 0.82328098402282079, 107.07134693541812, 1.6792091001037017], [78.734783904537025, 10.208710184594093, 
81.702711208178656, 0.057891601317937455], [63.712109202838569, 6.7652977687429443, 0.007100176743152233, 33.744264789879985]], 
[[0.01233089605451233, 36.831237638347808, 0.0190086123746563, 0.08007430211628104], [2.8302193735567638, 0.1251451032802868, 
0.096224644099550227, 0.093213706680998126], [21.520405850293248, 69.311180379408, 1.3771266689605197, 
30.388190954345838]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_exp_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(-3.77449373342)+(1.-msk_arg)*(0.850456784312)
      res=exp(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(0.0229487058644)+(1.-msk_ref)*(2.34071581003)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_exp_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([-3.7393995344613651, -3.2628252631869037])+(1.-msk_arg)*numpy.array([-2.6528849873100424, 
-2.6870752148490418])
      res=exp(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.023768370934870008, 0.038280093755891982])+(1.-msk_ref)*numpy.array([0.070447678945151443, 
0.068079767157810028])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_exp_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[-1.9817260046677521, -4.9579089227254238, -1.8080820818740708, 1.942246515863971, 
-0.89777849570800949], [-3.4475537532915999, 0.92286608790206515, -4.9742829192878535, -3.4345050115024311, 
-1.4440244909517617], [-2.2479698812238746, 3.7681392162757454, -2.278335200674988, 4.5958758018008545, 4.9314167432079099], 
[-0.0045528617050436537, -3.6644126073714824, 1.194478179565631, 4.7866389019556497, 
-0.09088030576391315]])+(1.-msk_arg)*numpy.array([[4.5574743974592575, -0.51086271852118781, -1.4071481921124409, 
-2.2815675667251045, -0.72703877271964856], [4.6927540382536712, 1.5791262837615854, -1.3892707624464085, -4.0889078016285136, 
0.11317827252295309], [-2.7999251521229862, -4.095006214564072, -3.2583077556554985, -0.14598982920555859, 
-1.1801660667943894], [-3.4444834039686922, 3.3362620324949397, -1.6021385615184993, 0.86098020615056203, -4.5258378013382359]])
      res=exp(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.13783113470503661, 0.0070276077496757745, 0.16396831322408212, 6.9744014880719511, 
0.40747385995517627], [0.031823389098959154, 2.5164925528515067, 0.0069134746817567649, 0.032241365387794589, 
0.23597616119320719], [0.10561341484893481, 43.299418972772479, 0.1024546312157716, 99.074867509504145, 138.57569951479095], 
[0.99545748685865898, 0.025619215403153886, 3.3018343562463994, 119.89770272867753, 
0.91312700039723671]])+(1.-msk_ref)*numpy.array([[95.342378541726404, 0.59997774355968292, 0.2448405266236971, 
0.10212399500066996, 0.48333814712336087], [109.15337881867579, 4.8507158100570384, 0.24925700594113484, 0.016757526109942217, 
1.1198315503506719], [0.060814614299645998, 0.016655642775027846, 0.038453415564263752, 0.86416649239970333, 
0.3072277140428874], [0.031921248174027389, 28.113841425830181, 0.20146521122296515, 2.3654782139907935, 0.010825640899623283]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_exp_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[4.8211686386003265, 0.33647211246578301], [4.5563602979133133, -3.4341307531273424]], 
[[2.2600398402096733, 4.1772605849086712], [-3.4452590779824632, 4.2782511171199289]], [[4.4823860777074707, 
0.78208223357321938], [-2.2832579645301596, 0.38038362869613529]], [[3.0247974385839242, 3.9138373508323863], 
[-3.2855941178596302, 0.65159570464059069]], [[-0.99609737044317637, 0.7041263241081186], [1.8883378287555637, 
4.4017114575255434]], [[-4.3000930525268544, 4.3690219280662816], [3.2371062155178159, 
1.4649610075412465]]])+(1.-msk_arg)*numpy.array([[[4.4502788272957634, 3.3900991552133171], [2.860270673171752, 
1.1195170895476894]], [[-0.77765196828685124, -1.4937317811365212], [-4.2142312072247625, -3.2665384651405027]], 
[[0.098031429433721229, 2.2058587376889163], [2.5866663202092068, 4.8535872754250295]], [[-1.8232821319135715, 
-1.109643067575111], [2.2465342933220871, -4.0144805461647524]], [[-1.7847898282722374, -1.2936444282648774], 
[-1.1904692261865488, 4.7201220865919318]], [[2.5707891273705687, -2.0121743624848754], [-3.7276218310247335, 
-2.9730999237984967]]])
      res=exp(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[124.11004585355214, 1.3999998261824089], [95.236216789444654, 0.032253434247110233]], 
[[9.583470966651543, 65.187034051042161], [0.031896497291880703, 72.114210351163806]], [[88.445458837222503, 
2.1860193326735424], [0.10195151064833471, 1.4628456713813345]], [[20.589833582908145, 50.090799616237007], 
[0.037418347579634484, 1.9185999064469186]], [[0.3693179434969916, 2.0220792710507358], [6.608375295740645, 
81.590387723522895]], [[0.013567296482240787, 78.966359137082307], [25.459939466084585, 
4.3273745020392527]]])+(1.-msk_ref)*numpy.array([[[85.650822460269438, 29.6688939500335], [17.466253943167619, 
3.063374510471744]], [[0.45948362780102198, 0.2245331814558067], [0.014783682987988234, 0.03813821561109227]], 
[[1.1029974511490235, 9.0780438800228946], [13.285408400428144, 128.19945270747877]], [[0.16149483279420984, 
0.32967661234983786], [9.4549110403248076, 0.018052329467398195]], [[0.16783233090464833, 0.27426940430229085], 
[0.30407854896854808, 112.1819477434759]], [[13.076139109878472, 0.13369765123108918], [0.024049962756334319, 
0.051144520225053301]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_exp_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[-2.424785209585878, 1.9551262421856341, 0.75203461888284195, -1.8993167709191749], 
[2.2670934161314005, -1.744013538076322, 1.8179292857384945, -3.4191611562654254], [-0.69415911402530561, 2.8101808900097165, 
0.26131048827439241, -4.3269884172744213]], [[-1.2996386037992269, -3.9681090462069091, 2.7009186202394257, 
-0.0050476608552019187], [-1.4865295967865144, 1.8338982255812821, 4.6863791148525245, 4.9738875283324617], 
[-3.9042351336997774, 0.62379784544002526, 2.5666053061507634, 3.9412931021179514]]], [[[1.7198321648721828, 
-4.3592617525370416, -1.5999991858171789, 2.5829468230466919], [3.0904702985924963, 0.628357119307136, 3.9637883496231492, 
-2.5057248442804605], [-1.0056222281587948, -0.75188558803935912, 3.9313391757120826, 2.5212281057625816]], 
[[1.1437836999455335, 4.9292564703590962, 3.0879967102874755, 2.2777418709576613], [-1.0877884148147521, 0.22263145770195791, 
-1.7979648008903015, 4.9637004071101636], [-1.0248274615765607, 1.947171172423432, 2.491741920179086, 0.57689311828301904]]], 
[[[-2.9651889955270292, -4.8498612332439954, -3.7214123172411231, -4.5287679390077695], [2.0388428137188868, 
-1.4755602958525946, 2.9779468886396678, -2.1190181822614074], [2.6135103995447615, -1.2576577235294453, 4.7203653885315013, 
4.5901888957440349]], [[4.855919022864958, 2.0015733360351273, -3.9505478894072557, 2.3702556185527097], [-4.4279822631618693, 
-0.094585386790021708, 0.0070055342843113877, -3.6462032433375469], [-2.5730325343218521, 3.6403497118475379, 
4.2617426258766073, -1.7869108938153189]]]])+(1.-msk_arg)*numpy.array([[[[4.8701091912402301, 0.33882876560754394, 
1.9892684578785502, 3.4122593715062415], [-3.4412340160092847, 3.1384693768182998, -0.55752437677080025, -1.1568669573910215], 
[1.957648065190809, -0.51835732939872337, -4.9157241942769678, -4.217191614199356]], [[-4.3522702324876095, 
-4.7086054442987555, -1.9695571497780606, -4.8872868853232108], [-4.8424235556262438, 0.8463840226861219, -2.7494674778040196, 
2.4259649971852379], [1.4181805547632278, -1.664640461070539, -1.7569058660056589, -1.5283773981070481]]], 
[[[-0.97808221900597569, 0.21643196976221191, -3.7964768937174864, -1.7420077495346464], [-3.997259942422926, 1.4610954106145, 
-2.6878989908620587, -0.66066513322009079], [3.7564741927384269, 0.4312674169390931, -0.76578819896386818, 
-1.3181836507023972]], [[-0.71955321425873642, 2.100100658526558, 0.53881649159204059, -4.5174712157213213], 
[0.14916949613176378, 1.5168479149320211, 0.68389330642199297, 0.40865151004778166], [4.0362778585285923, 1.5203939802104731, 
3.3291659173474848, 2.0732362980513415]]], [[[-0.71715987678536575, 4.3638182507176406, -4.4216637557716147, 
-1.6252197704591662], [-4.2778983483856639, 3.2890281576082145, -1.0378109703402405, -2.048661215580776], [3.7608251231256062, 
-1.1347030339042421, -3.9622844778106936, 1.5863353947668566]], [[-4.8982558412498234, 1.4662828107567982, 2.3525658564492913, 
3.3949779800909834], [-3.8373235748170376, 1.4353248042848854, 1.6062578841244282, 4.7726914394644719], [4.6392942181345838, 
2.5020853941974206, 3.8646258492359902, 0.55578302390160683]]]])
      res=exp(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.088497125335028778, 7.0648108445694282, 2.1213116896397204, 0.14967084377028211], 
[9.6513076711251102, 0.17481735460067765, 6.1590915174677994, 0.032739887072526438], [0.49949428918332384, 16.612923058819263, 
1.2986308124842763, 0.013207262421022459]], [[0.27263030278805822, 0.018909155689332269, 14.893406827744005, 
0.99496505717708328], [0.22615614825893463, 6.2582351821529594, 108.45974766155362, 144.5878857181703], [0.020156365521570803, 
1.8660013864662683, 13.021545166882007, 51.485133810164463]]], [[[5.5835912628773121, 0.012787824744825952, 0.2018966823753989, 
13.236085147294125], [21.987416195437135, 1.8745284215968578, 52.656429538973484, 0.08161641734620563], [0.36581694237053264, 
0.47147670322742347, 50.975196725728068, 12.44386967127565]], [[3.1386215287451615, 138.27666131197435, 21.933095590981772, 
9.7546283111709311], [0.33696088796549906, 1.2493600468566288, 0.16563564694101338, 143.1224284678392], [0.35885837692333256, 
7.0088327315468035, 12.08230421001412, 1.7804980317425978]]], [[[0.051550725467849501, 0.0078294639431589937, 
0.024199765952728419, 0.010793966708958169], [7.6817148786058693, 0.22865057917700912, 19.647436831060325, 
0.12014953556566858], [13.646872841787458, 0.28431920135942007, 112.20924514956785, 98.513037097945244]], [[128.4987302373379, 
7.4006907173388594, 0.019244155217681774, 10.700127085562588], [0.011938554247334804, 0.90975005067293646, 1.0070301304424787, 
0.026089998343179471], [0.076303800320660656, 38.105160221552808, 70.93348634217638, 
0.16747672479580386]]]])+(1.-msk_ref)*numpy.array([[[[130.33514757592133, 1.4033030308963479, 7.3101840982124351, 
30.333701996517831], [0.032025141396773642, 23.068530594672602, 0.57262491410879357, 0.3144698866359889], [7.0826495306350283, 
0.59549795195281086, 0.0073304073306697795, 0.014739981988049249]], [[0.012877544350661865, 0.0090173440066052355, 
0.13951862839167825, 0.0075418566643153271], [0.0078879140690343141, 2.3312020193499996, 0.063961913277678806, 
11.313141302977847], [4.1296000197436813, 0.1892586916182544, 0.17257801808476661, 0.2168873037054711]]], 
[[[0.37603155396036814, 1.2416386135104178, 0.022449725462343549, 0.17516835314417231], [0.018365893612826624, 
4.3106789064747035, 0.068023707772012484, 0.51650767380202078], [42.79726474585469, 1.5392071049786642, 0.46496729976021928, 
0.26762095390720336]], [[0.48696977851683243, 8.1669919485704447, 1.713977154960046, 0.010916594506075501], 
[1.1608697354643929, 4.55783584252283, 1.9815776223713002, 1.5047872258718735], [56.615220275952524, 4.5740269162848044, 
27.91504853264696, 7.9505117530819911]]], [[[0.48813665734657447, 78.556510966387933, 0.012014226907948362, 
0.1968684045898948], [0.013871785132128656, 26.816789295217589, 0.35422925221029378, 0.12890736729103774], [42.98387834251767, 
0.32151758698118288, 0.019019614735812828, 4.8858115111921609]], [[0.0074595824271214053, 4.3330982215530183, 
10.51250873747037, 29.813996966209348], [0.021551204411766171, 4.2010092930324658, 4.9841251131244801, 118.2370419399661], 
[103.47129364359584, 12.207925771377807, 47.685427555407657, 1.7433055003547806]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sqrt_float_rank0(self):
      arg=86.3180961935
      res=sqrt(arg)
      ref=9.29075326298
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sqrt_array_rank0(self):
      arg=numpy.array(34.7398492007)
      res=sqrt(arg)
      ref=numpy.array(5.89405201883)
      if not isinstance(res,float):
         self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
         self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sqrt_array_rank1(self):
      arg=numpy.array([65.473964247138113, 58.816786944192444])
      res=sqrt(arg)
      ref=numpy.array([8.0915983740629454, 7.6692103207691762])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sqrt_array_rank2(self):
      arg=numpy.array([[9.7466006081638401, 40.976601804960112, 73.4862116207523, 69.501421671385955, 96.836156379934053], 
[5.0404948844892488, 89.444034143640067, 79.742481259351564, 33.841814163546566, 60.15927529061647], [63.83618936242592, 
49.731491842411955, 83.406390140676535, 41.500601498963128, 93.542164296312535], [39.395427576692711, 95.294739606968903, 
61.067252506011364, 28.490707389286872, 74.691727660902984]])
      res=sqrt(arg)
      ref=numpy.array([[3.1219546134054927, 6.4012968846133136, 8.5724099074153184, 8.3367512660139962, 9.8405363867999629], 
[2.2451046488948458, 9.4574856142444155, 8.929864571165206, 5.8173717573786332, 7.7562410541844606], [7.9897552755028682, 
7.0520558592804665, 9.1327099012656987, 6.4420960485670449, 9.6717198210200728], [6.2765776962205058, 9.7619024583822238, 
7.8145538904029168, 5.3376687223250254, 8.6424375994798464]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sqrt_array_rank3(self):
      arg=numpy.array([[[74.854954492301175, 23.557527882811947], [18.731116465023806, 61.448833781473425]], 
[[25.111378585579899, 3.3431334533192674], [65.811971154744285, 33.103737097745658]], [[40.058621666007284, 
78.813244474198498], [13.546056898438673, 92.91407408669734]], [[25.064225317778558, 94.523669611127445], [73.376019350557755, 
41.402828382167243]], [[22.21585756283973, 35.769511800360974], [16.242034866556565, 22.645056407332177]], 
[[50.516130970033942, 69.522118938327679], [83.389736959275552, 2.2753142363880206]]])
      res=sqrt(arg)
      ref=numpy.array([[[8.6518757788297656, 4.8536097785887096], [4.3279459868422352, 7.8389306529317775]], 
[[5.0111254809254078, 1.8284237619652801], [8.1124577752210385, 5.7535847171781223]], [[6.3291880732055423, 
8.8776823819169444], [3.680496827663172, 9.6391946804023689]], [[5.0064184121763695, 9.7223284048178211], [8.5659803496481217, 
6.4345029631019086]], [[4.713370085495062, 5.9807618076931446], [4.0301407998426759, 4.7586822133162219]], 
[[7.1074700822468424, 8.3379925004960089], [9.1317981230026959, 1.5084144776512922]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sqrt_array_rank4(self):
      arg=numpy.array([[[[71.46136332279454, 80.887398562414646, 86.735688921961057, 52.300236290793997], [2.7023069183704087, 
16.545501813426721, 10.45857409667561, 9.4410763253402425], [92.639467504221528, 20.770519212685478, 32.237201980032232, 
98.522614837908691]], [[78.514595416639366, 91.993818863993511, 46.977442638259063, 48.727376831133093], [54.301935499310368, 
73.882736618887463, 0.59827333650743164, 15.697299083334212], [0.88940858615458018, 7.521060219066233, 77.94644711245175, 
79.654838083035173]]], [[[49.680923109992627, 14.284037926053889, 14.549783884445, 10.51233575353886], [23.753791275088378, 
29.745138959267745, 77.905973362246712, 88.364835484713623], [22.930503013939816, 19.016299314287121, 57.009321091343118, 
23.095858867391023]], [[80.276955121739618, 71.605962163449888, 7.4220585527970417, 52.596490687887822], [69.005031582312256, 
66.713135957427994, 68.187426164093381, 96.693602764175139], [69.184776852318322, 22.35344754339399, 95.213012823801662, 
78.536999135178007]]], [[[40.02191359012, 99.962431101420378, 73.302984262572721, 34.462848057517569], [25.968251587852286, 
66.876554510906743, 59.381608464104069, 28.01468377700013], [49.376157524758554, 50.195957417607509, 95.88016523287672, 
5.2666076745667514]], [[1.9613653407399201, 21.902422168300358, 40.135192438340155, 60.930637525483007], [40.532540336073787, 
54.936949875603915, 20.260744521866418, 58.3907552034483], [91.748935363386806, 83.967371655385122, 13.502183427962146, 
78.597184065772723]]]])
      res=sqrt(arg)
      ref=numpy.array([[[[8.4534823193045447, 8.9937421890120159, 9.3131997144891638, 7.2318902294485916], [1.6438694955410569, 
4.0676162323192093, 3.233971876296331, 3.0726334511848696], [9.6249398701613469, 4.5574685092368412, 5.6777814311606107, 
9.9258558743268424]], [[8.8608462020644154, 9.5913408272250198, 6.8540092382677065, 6.9804997551130317], [7.3689846993537964, 
8.5955067691723368, 0.77348130973374629, 3.9619817116355058], [0.94308461240473018, 2.7424551444036842, 8.8287285105190403, 
8.9249559149070965]]], [[[7.0484695579957375, 3.7794229620477635, 3.8144178958846395, 3.2422732385687145], [4.8737861334991281, 
5.45391042824025, 8.8264360509917434, 9.4002572031149025], [4.7885804800524987, 4.3607682023110472, 7.5504517143905447, 
4.8058151095720509]], [[8.9597407954549446, 8.4620306170239008, 2.7243455274243469, 7.2523438065144035], [8.306926723061439, 
8.1678109648441293, 8.2575678116557647, 9.8332905359383709], [8.3177386862246596, 4.7279432677850517, 9.7577155535402689, 
8.8621103093551028]]], [[[6.3262875045416642, 9.9981213786101026, 8.5617161984366614, 5.8705066269886421], [5.0959053746956764, 
8.1778086619159005, 7.7059463055554751, 5.2928899267791438], [7.0268170265603578, 7.0849105440794036, 9.7918417691911621, 
2.2949090776252445]], [[1.4004875368027807, 4.6800023684075587, 6.335234205484447, 7.8058079354723438], [6.3665171276038981, 
7.4119464296231872, 4.5011936774445083, 7.6413843774180279], [9.5785664565939506, 9.1633711948924734, 3.6745317290727191, 
8.8655052910577368]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sqrt_constData_rank0(self):
      arg=Data(81.1885184289,self.functionspace)
      res=sqrt(arg)
      ref=Data(9.0104671593,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sqrt_constData_rank1(self):
      arg=Data(numpy.array([31.608199510469642, 58.266383154892708]),self.functionspace)
      res=sqrt(arg)
      ref=Data(numpy.array([5.6221169954448333, 7.633241981942712]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sqrt_constData_rank2(self):
      arg=Data(numpy.array([[1.8937448628391651, 45.117522376733682, 98.405119064635343, 51.590032151364696, 
7.5547011363264076], [57.216223146991382, 7.819675959857336, 62.730366829591304, 5.8519846411770544, 60.511471292557616], 
[54.906822712101345, 66.07733272267555, 64.003457767881173, 97.974422290698897, 57.233707543449995], [91.868492867217, 
54.921631224129897, 96.337085692613627, 18.032950834434121, 84.766018920765816]]),self.functionspace)
      res=sqrt(arg)
      ref=Data(numpy.array([[1.376134027934476, 6.7169578215687551, 9.9199354365154679, 7.182620145278789, 2.7485816590245973], 
[7.5641406086211394, 2.7963683519624762, 7.9202504272018635, 2.4190875637680116, 7.7789119607151758], [7.409913812730978, 
8.1287965113339844, 8.0002161075736673, 9.8982029829004254, 7.5652962627679026], [9.5848053119099408, 7.4109129818214639, 
9.8151457295657938, 4.2465222046321767, 9.2068463070025128]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sqrt_constData_rank3(self):
      arg=Data(numpy.array([[[87.02198222813827, 33.570737951021194], [52.525450531750558, 39.684397564383453]], 
[[75.510062497587867, 35.026989774648499], [55.509249826279124, 68.511254064729783]], [[72.555270511448654, 
13.956303298059117], [10.759604579613761, 16.32642464315532]], [[59.835090291025239, 68.866332495849889], [1.8025353235179187, 
47.895800506159119]], [[71.509624790591118, 71.710108451236664], [96.342933268841293, 8.6234602282293746]], 
[[64.367969688809822, 78.448665712031442], [74.892632534275066, 8.2842315302036038]]]),self.functionspace)
      res=sqrt(arg)
      ref=Data(numpy.array([[[9.3285573497802048, 5.7940260571575957], [7.2474444138434455, 6.2995553465608545]], 
[[8.6896526108693131, 5.9183603958062996], [7.4504529947030154, 8.277152533615034]], [[8.5179381608138396, 3.7358136059042235], 
[3.2801836198014525, 4.0405970651817436]], [[7.735314492056883, 8.2985741242607389], [1.3425853133108223, 6.9206791939923873]], 
[[8.4563363692908471, 8.468182121992692], [9.8154436103948601, 2.936572871261562]], [[8.0229651431880118, 8.857125138103866], 
[8.6540529542102451, 2.8782340992705238]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sqrt_constData_rank4(self):
      arg=Data(numpy.array([[[[21.795295499572095, 4.084894020517047, 29.981684018437836, 36.175898886164298], 
[92.038955722571828, 88.023304564848971, 35.159239538624213, 18.765068438143594], [65.993071025016945, 67.144658154770184, 
4.3549549384179151, 51.266321276269288]], [[70.207341980760489, 7.9633753590335425, 4.0666402122653604, 59.84875303002196], 
[23.368135780644838, 67.589350326724968, 88.410292282143644, 54.749271087775], [44.262021954292344, 16.29586925251364, 
64.42754080647218, 92.456969315491207]]], [[[56.82367308212428, 45.192248940060928, 27.82600645219588, 13.743334093988349], 
[51.252774880390291, 48.972849245552716, 9.4966751019603599, 5.5756087747698482], [99.023365548748004, 35.59013770422807, 
56.316117717520385, 8.4121043606021484]], [[3.3159509243607235, 79.747315050571387, 4.9738702104291548, 95.34998056645442], 
[68.081784817271384, 88.626595016315335, 27.398093893823344, 65.933424384182445], [55.307663614437693, 13.002752003162881, 
18.781170919454972, 71.553240971667549]]], [[[48.240139581729025, 17.261289447842351, 16.43259019545037, 87.531298111358524], 
[81.230709782876204, 83.536650945050937, 54.480030779685258, 33.059253312196411], [91.650322668380753, 42.463041081800853, 
24.498590807336498, 50.800356050788565]], [[37.405037211573024, 21.44834026340579, 15.48459838808721, 3.4277084042771802], 
[18.442845377165781, 73.240870896820113, 70.762795637307633, 17.77983605809187], [24.294517062681759, 10.85074078857421, 
65.088793217360191, 71.762599695727516]]]]),self.functionspace)
      res=sqrt(arg)
      ref=Data(numpy.array([[[[4.6685431881446799, 2.0211120751994547, 5.4755533070583686, 6.0146403787894336], 
[9.5936935391209897, 9.3820735749006445, 5.9295227074887071, 4.3318666228478913], [8.1236119445119321, 8.1941844105908537, 
2.086852878958628, 7.160050368277397]], [[8.3789821566083127, 2.8219453146780755, 2.0165912357900795, 7.736197582147315], 
[4.8340599686645218, 8.2212742520077118, 9.4026747408460132, 7.39927503798683], [6.6529709118778166, 4.0368142454804188, 
8.0266768221021696, 9.6154547118423483]]], [[[7.5381478548861249, 6.7225180505567206, 5.2750361564823312, 3.7072003040014372], 
[7.1591043350680605, 6.9980603916765904, 3.0816675845977222, 2.3612727023302176], [9.9510484647974664, 5.9657470365603054, 
7.504406553320548, 2.900362798099946]], [[1.8209752673665625, 8.9301352201728381, 2.2302175253614061, 9.7647314641240612], 
[8.2511686940257967, 9.4141699058554984, 5.2343188567208383, 8.1199399249121562], [7.4369122365695359, 3.6059328894424643, 
4.3337248319955632, 8.4589148814530315]]], [[[6.9455121900209074, 4.1546707989734095, 4.0537131367982084, 9.3558162717829454], 
[9.0128080964190183, 9.1398386717190441, 7.3810589199440253, 5.7497176723902204], [9.5734175020407815, 6.5163671690444867, 
4.9496051163033705, 7.1274368500035523]], [[6.1159657627862032, 4.6312352848247507, 3.9350474441977457, 1.8514071416836384], 
[4.2945134040035065, 8.5580880397913717, 8.4120625079291713, 4.2166142885129858], [4.9289468512737846, 3.2940462638788501, 
8.0677625905426957, 8.4712808769233661]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sqrt_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(54.8205914212)+(1.-msk_arg)*(38.8859377248)
      res=sqrt(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(7.40409288308)+(1.-msk_ref)*(6.2358590206)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sqrt_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([34.878225077708073, 24.414452439222732])+(1.-msk_arg)*numpy.array([23.98120273255164, 
81.768597684709349])
      res=sqrt(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([5.9057789560487342, 4.9410983029305067])+(1.-msk_ref)*numpy.array([4.8970606216945729, 
9.0425990558417073])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sqrt_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[20.317318590352926, 12.867420890558268, 22.016741419835256, 16.345623974345799, 
13.102237185921794], [53.92570230350389, 71.900156715861073, 9.4252149639806948, 60.84441930191781, 50.117408366318287], 
[61.905020666663972, 22.693877901696609, 9.4721991480585661, 68.967231237960291, 71.000835450129358], [6.5997037370555809, 
63.501958858703219, 43.973917777580162, 69.07274315702081, 70.028404173419162]])+(1.-msk_arg)*numpy.array([[45.027472873392824, 
4.7155944524916205, 61.188316066858214, 64.035258976578206, 57.221264404299795], [82.682817093270899, 88.313626452422909, 
78.555687426229909, 67.983911001161687, 0.026627056947911624], [66.579275596020452, 44.754583966935748, 20.260755873056699, 
78.383131578470142, 24.069605944462701], [43.362163173007445, 20.612208542421353, 50.626164383988531, 87.478509390698861, 
50.605714697529599]])
      res=sqrt(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[4.5074736372332698, 3.5871187449760105, 4.6922000617871413, 4.0429721708596755, 
3.6197012564466968], [7.343412170340426, 8.4793960112652531, 3.0700512966367017, 7.800283283440276, 7.0793649691422385], 
[7.867974368709139, 4.7638091798157287, 3.0776938034928958, 8.3046511809925097, 8.4261993478750163], [2.5689888549885889, 
7.9688116340332211, 6.631283267783104, 8.3110013329935644, 8.3682975672127693]])+(1.-msk_ref)*numpy.array([[6.7102513271406474, 
2.1715419527358022, 7.8222960866268805, 8.0022033826052059, 7.5644738352577967], [9.0930092429993099, 9.3975329982087796, 
8.8631646394631485, 8.2452356546772929, 0.16317799161624591], [8.1596124660439884, 6.6898866931313377, 4.5011949383532253, 
8.8534248502187083, 4.9060784690486452], [6.5849953054658625, 4.5400670195957851, 7.11520655947447, 9.3529946750064425, 
7.1137693733722909]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sqrt_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[11.684656696818585, 30.357257715884291], [85.350052353498484, 59.443901291115083]], 
[[18.389152959801454, 73.643955704011702], [53.984062439631494, 12.85516848676594]], [[6.9547545786425422, 16.452928207817298], 
[43.337018415658228, 48.015159376348123]], [[78.613768164511356, 77.318961155168324], [77.003339185791077, 
44.054275892995008]], [[68.769672855584389, 14.183877776643417], [1.0506243096367072, 83.074034158738783]], 
[[68.603749320615762, 62.181249258951887], [38.154924618191941, 
9.1944260234661481]]])+(1.-msk_arg)*numpy.array([[[40.391269307955973, 88.221006808310889], [97.091581134289328, 
50.521886193874188]], [[57.677989772970157, 30.811952486964394], [63.061271724983328, 79.854113231189942]], 
[[77.994260135883607, 63.180012960905167], [12.44106965996682, 52.114160340799486]], [[13.817876878807546, 62.985185841043162], 
[68.003559939530518, 46.932424756146055]], [[60.235244348430399, 24.99938634776942], [39.701707256745749, 33.324822817140266]], 
[[11.565408606014044, 49.311140529717512], [97.866164528901933, 3.6646241076566093]]])
      res=sqrt(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[3.4182827116577976, 5.5097420734444809], [9.2385092062246965, 7.70998711355052]], 
[[4.2882575668680927, 8.5816056600155957], [7.3473847346951615, 3.5854105046376405]], [[2.6371868683585058, 
4.0562209268994822], [6.5830857821889444, 6.9292971776615353]], [[8.8664405577723997, 8.7931201035336901], [8.7751546530982054, 
6.6373395191895233]], [[8.2927482088620295, 3.7661489318192687], [1.0249996632373628, 9.1144958258116926]], 
[[8.2827380328376776, 7.8855088142079888], [6.1769672670487692, 
3.0322311955829075]]])+(1.-msk_ref)*numpy.array([[[6.3554125993483677, 9.3926038353755175], [9.8535060325900918, 
7.1078749421943401]], [[7.5946026738052703, 5.5508515100806282], [7.9411127510559458, 8.9361128703251023]], 
[[8.8314359045335102, 7.9485855949914237], [3.527190051580269, 7.2190138066635869]], [[3.7172404924631315, 7.9363206740304513], 
[8.2464271014500898, 6.8507243964522511]], [[7.7611367948535994, 4.999938634400368], [6.3009290788538284, 5.7727656125240578]], 
[[3.4007952902246328, 7.0221891550795972], [9.8927329150696242, 1.9143207953884347]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sqrt_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[48.903559440371922, 74.83741208239141, 38.372431802981161, 96.248460927030706], 
[53.646878234472524, 94.510926650586029, 64.954817931787673, 30.868058963264321], [97.807409749537811, 73.388649361949149, 
86.373009563958377, 21.722617815276955]], [[65.389805742363407, 26.908173035974336, 23.241307099326256, 79.784918499476987], 
[88.165003103011685, 0.94587320080294524, 85.085866509426083, 20.148625146109431], [19.475365737551268, 50.235255083818743, 
66.670698298843362, 79.592082504132151]]], [[[18.983720519021755, 33.767160434207248, 33.648290197834861, 71.642750553814366], 
[11.620389515753942, 26.70839752656941, 45.845350936019628, 53.101702350780371], [47.580174509101269, 11.670115621363312, 
95.783800758948871, 90.605146589691373]], [[91.927849959428841, 11.970115640953882, 86.250444118615064, 34.183636640083861], 
[93.254675822828546, 44.327499129776491, 2.5273196359699144, 80.261909475569439], [36.616343579298942, 42.798010808821694, 
9.2315825982138033, 74.116212909516534]]], [[[97.307895977837902, 27.247193208428488, 57.71288820331646, 35.481037870800272], 
[10.386055244808556, 94.57708994266082, 37.894262013114322, 56.92860660303068], [68.023973652582171, 17.346638823152951, 
37.319078202247489, 6.7885967112243453]], [[74.44137263347794, 68.414420972601675, 66.211882002293919, 4.9175305887501555], 
[4.3005977638467723, 76.703108953685586, 43.861854506206427, 60.263317561208062], [78.476096974816173, 62.689184698921373, 
73.710453403098924, 27.492356009093722]]]])+(1.-msk_arg)*numpy.array([[[[3.190759736440032, 15.036103644004152, 
21.376479877496934, 59.579680762937635], [44.681334506324276, 63.847024085751514, 27.294603709268873, 44.966901274458365], 
[66.318481486537124, 24.832740518681611, 15.477237505771496, 56.52161360483359]], [[66.595305617471226, 24.18206622207947, 
26.115410328215514, 43.60836450058801], [60.757299343347022, 93.191364219053796, 80.849211937598938, 1.6869662760441873], 
[59.381831240364406, 47.095384182630283, 67.313927649463253, 17.499539516261652]]], [[[24.839658005483209, 52.504803666295729, 
87.87459573824647, 78.949929895519332], [97.713889013878827, 39.556876728750815, 20.651187751790982, 54.413595464170712], 
[8.4691180615773316, 8.8714364520982212, 52.15187490161172, 49.061429019212426]], [[86.581942153363585, 64.744203587159802, 
43.152956073155366, 97.716282181355226], [30.092930763693868, 74.763531272001316, 68.448714834568008, 38.651646618226806], 
[20.123115597233305, 66.906359271148062, 53.131773343770476, 91.480773519914734]]], [[[23.26242869945904, 7.1120366984988594, 
90.862762458254664, 35.918574234144955], [70.434390352354825, 84.281045642995352, 89.890625227965145, 92.800342512459139], 
[17.964067325224214, 65.114121093236093, 25.992385079346878, 12.514493689478991]], [[0.81420372076404934, 54.212437788996965, 
89.382294307529321, 40.609274874950771], [56.863423177120637, 35.420018043668335, 88.134436653862082, 75.036163443845382], 
[52.541571065623508, 15.657472191419401, 47.951668037430387, 96.172543997545844]]]])
      res=sqrt(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[6.9931079957606777, 8.6508619271371696, 6.1945485552202397, 9.8106299964391024], 
[7.3244029268243107, 9.7216730376302021, 8.0594551882734393, 5.5559030736023756], [9.8897628763048608, 8.5667175371871078, 
9.2937080631983697, 4.6607529236462382]], [[8.0863963384417055, 5.1873088433188874, 4.8209238844153361, 8.9322403964222197], 
[9.3896220958573018, 0.9725601270887807, 9.2242000471274519, 4.4887219947452115], [4.4130902707231439, 7.0876833368752266, 
8.1652126915863832, 8.921439486099322]]], [[[4.3570311588307185, 5.8109517666392012, 5.8007146282018445, 8.4642040708984787], 
[3.4088692429827727, 5.1680167885340129, 6.7709195044705437, 7.2870914877460109], [6.8978383939536645, 3.4161550932829896, 
9.7869198810937892, 9.5186735730190559]], [[9.5879012280805664, 3.4597854905982079, 9.2871117210150462, 5.8466774017457013], 
[9.6568460598079611, 6.6578899908136426, 1.5897545835662543, 8.9589011310299345], [6.0511439893047445, 6.5420188633801484, 
3.038351954302497, 8.6090773552986803]]], [[[9.8644764674988146, 5.2198844056577043, 7.5968999074172654, 5.9565961648243597], 
[3.2227403315825116, 9.725075318097069, 6.1558315452190797, 7.5451048106060581], [8.2476647393418077, 4.1649296300361369, 
6.1089342934956736, 2.6054935638424337]], [[8.6279413902435582, 8.2713010447330273, 8.1370683911525479, 2.217550583132244], 
[2.0737882639861698, 8.7580311117103022, 6.622828286027536, 7.7629451602602515], [8.8586735448833522, 7.9176502005911686, 
8.5854792180226571, 5.2433153642608339]]]])+(1.-msk_ref)*numpy.array([[[[1.7862697826588323, 3.8776415053488571, 
4.6234705446771187, 7.7187875189655033], [6.6844098098728413, 7.9904332852325046, 5.2244237681555727, 6.7057364453472497], 
[8.143615995768533, 4.9832459821567721, 3.9341120352338081, 7.5180857673235941]], [[8.1605946852831277, 4.9175264332873159, 
5.1103238966053333, 6.6036629608564983], [7.7946968730892303, 9.6535674348426141, 8.9916189831197215, 1.2988326589842847], 
[7.7059607603701439, 6.8626076809497336, 8.2045065451532952, 4.1832450939744916]]], [[[4.9839400082147067, 7.2460198499794171, 
9.3741450670579276, 8.8853773074371656], [9.8850335868867347, 6.289425786886337, 4.5443577931090529, 7.3765571552161591], 
[2.9101749194124622, 2.9784956693099658, 7.2216255027252503, 7.0043864127568254]], [[9.3049418135399211, 8.0463782900855332, 
6.5690909624662197, 9.8851546361883091], [5.485702394743436, 8.6465907311495496, 8.2733738483503814, 6.2170448460845771], 
[4.4858795789937682, 8.1796307539612112, 7.289154501296462, 9.564558197842425]]], [[[4.8231140044020355, 2.6668402086549654, 
9.5321960983948841, 5.9932106782712848], [8.3925199047934829, 9.1804708835111146, 9.48106667142285, 9.6332934405871367], 
[4.2384038652804445, 8.0693321342745641, 5.0982727545068514, 3.5375830293406532]], [[0.9023323782088557, 7.3629096007622534, 
9.454220978352966, 6.3725406922946179], [7.540783989554444, 5.9514719224464407, 9.3879942827987541, 8.6623416836237404], 
[7.2485564815088193, 3.9569523868021714, 6.9247142928376757, 9.8067601172632877]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log_float_rank0(self):
      arg=24.892231195
      res=log(arg)
      ref=3.21455575459
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log_array_rank0(self):
      arg=numpy.array(15.4265888534)
      res=log(arg)
      ref=numpy.array(2.73609256957)
      if not isinstance(res,float):
         self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
         self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log_array_rank1(self):
      arg=numpy.array([41.217592088139703, 73.500998090089979])
      res=log(arg)
      ref=numpy.array([3.7188651576507299, 4.2972989855835975])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log_array_rank2(self):
      arg=numpy.array([[31.875079721399224, 34.832226119089448, 93.641274834929035, 86.566403944415129, 70.233384556907083], 
[52.057970259166993, 68.74836253109244, 33.259194838017777, 61.146850394126993, 62.642704645544988], [67.726931221936539, 
77.911923166743605, 15.822918482945735, 39.773746643504957, 38.23625410032259], [89.495981515837059, 13.418462409130685, 
6.4977278270486307, 68.658843752645794, 76.363034456528709]])
      res=log(arg)
      ref=numpy.array([[3.4618245045389675, 3.5505429961767203, 4.5394712567432904, 4.4609117950835708, 4.2518237614759933], 
[3.9523579103155106, 4.2304529185334694, 3.5043312653544354, 4.1132783545781866, 4.1374472283612933], [4.215483903207808, 
4.3555789985040789, 2.7614594259158105, 3.6832070625646973, 3.6437841257574917], [4.494193725008417, 2.5966315503989414, 
1.871452550720039, 4.229149947699077, 4.3354987368910392]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log_array_rank3(self):
      arg=numpy.array([[[64.476599328684586, 66.886199125806456], [4.2506759234075693, 44.562708995657353]], 
[[40.880798701214729, 68.728901925279473], [10.85436368365343, 57.718517632721031]], [[70.71656488621737, 28.229645974764676], 
[97.192610440790062, 48.106569039850342]], [[45.701870353038935, 40.401420493359559], [76.615933687268878, 
0.93993584483392745]], [[47.59129323364337, 38.444271194895634], [19.459247729772727, 20.447276160585719]], 
[[50.758556758448997, 34.455683509869992], [97.671298763293379, 2.2250738386870004]]])
      res=log(arg)
      ref=numpy.array([[[4.1663023568678179, 4.2029926547591741], [1.4470780110924588, 3.796897388108142]], 
[[3.7106604834055381, 4.2301698083622004], [2.3845671819578351, 4.0555780515370907]], [[4.2586798437081637, 
3.3403727014416216], [4.5766946843046945, 3.8734187382618761]], [[3.822139223825022, 3.6988649450757021], [4.3388050666982805, 
-0.061943656223909682]], [[3.8626498292467448, 3.6492096912201966], [2.968322418753337, 3.0178496785361753]], 
[[3.9270802097873023, 3.5396739623680284], [4.5816077468348162, 0.79979010099430459]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log_array_rank4(self):
      arg=numpy.array([[[[7.940748330359841, 98.274339093554488, 9.7823697075113696, 16.014459973229165], [62.324770937938801, 
29.791675471244069, 96.638584692994172, 25.915985691021351], [44.151706402245154, 29.876914547758364, 29.884858201733593, 
27.061337090957885]], [[60.968778123619735, 40.607659998490469, 2.6549838244538226, 6.3143180278454558], [35.01369318176684, 
71.357122020093144, 71.338556015839075, 38.232118105247167], [34.886496207888641, 82.958784085343154, 28.975440375695332, 
85.215987664288036]]], [[[79.796340637484377, 61.276664763571077, 21.575162399502645, 15.414637553082256], [22.396911842915799, 
92.891007582841368, 7.6865042152980045, 29.857694632745535], [81.459761582856117, 87.536793846486432, 57.418476028580322, 
69.553303173753946]], [[89.850571917036916, 11.660419181286768, 4.3597209272971993, 37.707598444286653], [93.936620413128722, 
2.6392362890203347, 61.221732668741396, 68.606040555475573], [61.459463992767013, 52.627535834012221, 26.353930439525314, 
12.986714132899071]]], [[[19.22299083157252, 66.743248932943914, 87.638183236930828, 96.037233106635213], [87.034402721537873, 
78.389287194337783, 77.284996603370715, 19.00184151220035], [30.388533155549258, 36.06136632840694, 40.370148182231041, 
11.204902056961361]], [[34.233871583033732, 43.682627695516324, 84.231011435892668, 44.509740171221324], [25.243721328530146, 
2.9896832791631351, 0.73262591991285908, 14.893056900210631], [85.989381953571865, 21.465681458032964, 21.923368855128366, 
88.78513136953984]]]])
      res=log(arg)
      ref=numpy.array([[[[2.0720075189741172, 4.5877629462133642, 2.2805817560780177, 2.7734920624319672], [4.1323589541190939, 
3.3942290078842809, 4.5709780889322662, 3.254859986458424], [3.787631576410551, 3.3970860934789155, 3.397351937467691, 
3.298106033604606]], [[4.1103618991025357, 3.7039567187253155, 0.97643856219090697, 1.8428197575473304], [3.5557392187422066, 
4.2676971570225453, 4.2674369388489559, 3.643675950443694], [3.5520998261916468, 4.4183439071762436, 3.3664485875812544, 
4.4451890648733441]]], [[[4.3794776467311856, 4.1153990977692434, 3.0715427641487008, 2.7353175484494439], [3.1089230852013134, 
4.5314268443945505, 2.0394660917731513, 3.3964425832613125], [4.4001091754157882, 4.4720592060804591, 4.0503661335636298, 
4.2420934107535899]], [[4.498147978451791, 2.4562001306437948, 1.4724080478117332, 3.6298616243917734], [4.5426203039364603, 
0.97048959084540898, 4.1145022354483771, 4.2283805855997896], [4.1183778354271947, 3.9632394777111863, 3.2716174266540428, 
2.563926845096907]]], [[[2.9561070018196922, 4.2008531525840418, 4.473216784673637, 4.5647359611362432], [4.4663034740722249, 
4.3616872750905014, 4.347499843654882, 2.9445358961645915], [3.4140653384241864, 3.5852221074735136, 3.698090605460814, 
2.4163513662027607]], [[3.5332155511135985, 3.7769504874916224, 4.4335631602795464, 3.7957080454735714], [3.2285774645440495, 
1.0951674550906731, -0.31112004854687247, 2.700895124497428], [4.4542238229748552, 3.0664554485768702, 3.0875531389172917, 
4.4862191964463518]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log_constData_rank0(self):
      arg=Data(56.8034993935,self.functionspace)
      res=log(arg)
      ref=Data(4.03959793287,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log_constData_rank1(self):
      arg=Data(numpy.array([70.593499720144223, 56.988510922628677]),self.functionspace)
      res=log(arg)
      ref=Data(numpy.array([4.256938068307683, 4.0428496847571465]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log_constData_rank2(self):
      arg=Data(numpy.array([[0.74657762933205751, 18.060743238575597, 1.1033016992614968, 78.479142956402399, 
68.906204955586361], [72.717785467391963, 44.865362067234358, 18.009334263559456, 9.6343642038969843, 92.092624207623615], 
[39.296132639892448, 47.853575664192441, 13.232730936974001, 96.144004426643605, 66.900994586995111], [7.2266886703089108, 
5.8193972022966829, 0.82904390011205842, 87.251279783567981, 48.052884763695999]]),self.functionspace)
      res=log(arg)
      ref=Data(numpy.array([[-0.29225567634186217, 2.893740700994512, 0.098307228945628114, 4.3628328946623407, 
4.2327462313752298], [4.2865859965368784, 3.8036660508664459, 2.890890193682853, 2.2653363115101852, 4.5227948554465396], 
[3.6711261079234099, 3.8681458416314736, 2.5826933768534506, 4.5658471136333789, 4.2032138337949005], [1.9777809326831437, 
1.7611966828906997, -0.18748216974584908, 4.4687922289814725, 3.8723021703196023]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log_constData_rank3(self):
      arg=Data(numpy.array([[[88.723141574053372, 86.14344495920983], [60.995524537090297, 72.072873685369672]], 
[[23.431949875857839, 30.706074649360215], [23.581243700165654, 10.549386564446014]], [[24.78789514529339, 66.5346760742339], 
[87.058189497609717, 7.2011781538311359]], [[22.066499336856548, 21.690405075870103], [87.95770277352625, 91.977432402112427]], 
[[43.107399425465857, 87.231558429057856], [90.194317750879136, 66.60854781234535]], [[77.587908320752433, 48.19366507238783], 
[81.3290332190698, 16.398208317051651]]]),self.functionspace)
      res=log(arg)
      ref=Data(numpy.array([[[4.4855207524022687, 4.4560138713873449], [4.110800493237309, 4.2776777416722895]], 
[[3.154100470391096, 3.4244605063338698], [3.1604516375760627, 2.3560677126814165]], [[3.2103554350317483, 4.1977232564820719], 
[4.4665767398776985, 1.9742446451122229]], [[3.0940605913241939, 3.0768700003614944], [4.4768560486279849, 
4.5215432469810413]], [[3.7636946627795482, 4.4685661740319294], [4.501966428974252, 4.1988329148414714]], 
[[4.3514115944390204, 3.8752275823854236], [4.3985030649700727, 2.7971720799017854]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log_constData_rank4(self):
      arg=Data(numpy.array([[[[94.313527361847164, 11.189007556617248, 14.573062591242451, 93.331862769194132], 
[42.869863127749241, 14.279469143790667, 60.15229302555592, 18.879945169044326], [85.19134504809378, 4.0019786329677878, 
92.099179379982701, 21.232745256261421]], [[37.587941975940197, 3.1234976808413446, 45.34108294597327, 66.081602530358026], 
[49.50023371347649, 24.871949038044743, 11.22282494816106, 38.45019357343029], [3.9491380489005601, 38.355711133531258, 
49.561030023610151, 78.981970140502369]]], [[[2.1464040558374125, 9.6598661432437378, 8.2430433337516593, 73.623271087698072], 
[34.195461239450623, 51.820292654762042, 66.396580345512376, 15.61683468515216], [28.622942622063704, 1.9500884779576764, 
65.945800573959019, 44.455211702202313]], [[7.6231156052037221, 71.180591451578749, 37.879260539829907, 98.296535021827779], 
[66.420681286189975, 43.255917127461117, 96.503994968420074, 56.819815368622997], [7.2893702235738251, 39.26455997672435, 
18.819065077545645, 75.523938522964428]]], [[[24.754727101503232, 13.017563602488718, 34.312826557769732, 94.819726987205456], 
[17.600727975127921, 13.035767464690739, 37.241820668299908, 14.265529437050695], [65.706609740068558, 20.717853074403475, 
12.746969205328828, 59.849414122028769]], [[85.142353104425297, 56.151120866394862, 46.260084073682769, 38.732246545848191], 
[69.974344971587485, 45.871181830732986, 39.663904486977408, 93.56463268965129], [54.199136191616333, 65.62152414576741, 
79.900479063866285, 89.526702672199804]]]]),self.functionspace)
      res=log(arg)
      ref=Data(numpy.array([[[[4.5466246296361028, 2.4149318281866261, 2.6791747965565715, 4.5361615583326635], 
[3.7581690878767562, 2.6588227814158629, 4.0968795634821991, 2.9381002565311634], [4.4448998447327108, 1.3867888970587798, 
4.5228660331236581, 3.0555445775890155]], [[3.6266833068338817, 1.1389534254640483, 3.8142135297957567, 4.1908903802860662], 
[3.9019773910478719, 3.2137406237854225, 2.4179496463147792, 3.6493637303584938], [1.3734973396323584, 3.646903438069292, 
3.9032048399107926, 4.3692196003498989]]], [[[0.76379390972747918, 2.2679797913213537, 2.1093696123537469, 4.2989611590162058], 
[3.532092922964738, 3.9477818226433361, 4.1956455543355791, 2.7483494788796077], [3.3542085858263588, 0.66787474485794673, 
4.1888332012293166, 3.7944822037303001]], [[2.0311851582071254, 4.2652201892857766, 3.6344037470035233, 4.5879887775168831], 
[4.1960084731406502, 3.7671340362155861, 4.5695844061245179, 4.039885127012508], [1.9864171532042916, 3.6703223302626911, 
2.9348704558534489, 4.3244496725293393]]], [[[3.2090164647777981, 2.5662994919649105, 3.5355192363907926, 4.5519774781954698], 
[2.8679402634118905, 2.5676969228820918, 3.6174323413399891, 2.6578460983861829], [4.1851995253019014, 3.0309957959225451, 
2.5452935339611065, 4.0918316428504511]], [[4.4443245980780457, 4.0280466429869035, 3.8342794742180439, 3.6566724971593723], 
[4.2481286744657778, 3.8258370730631559, 3.6804415672367234, 4.538652456198232], [3.9926649708945456, 4.1839037541112418, 
4.3807818485473238, 4.4945369346614692]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(37.4196705674)+(1.-msk_arg)*(47.0594752283)
      res=log(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(3.62219651716)+(1.-msk_ref)*(3.85141223212)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([60.711508827996475, 98.076830520157273])+(1.-msk_arg)*numpy.array([96.379528847746016, 
24.559653458431836])
      res=log(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([4.1061332818724914, 4.585751156414501])+(1.-msk_ref)*numpy.array([4.5682938227145149, 
3.201104993180691])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[96.892989215451109, 31.390984424097773, 3.5451858406969468, 31.10565027093195, 
68.859465663487711], [62.389925982107371, 65.697847959211373, 22.955742607604762, 56.713302864287101, 18.396161229780429], 
[28.332038415285993, 54.907943558576022, 4.7856242995956517, 91.756453188368468, 98.070679093995807], [16.390514285904892, 
44.658368855238038, 2.7242204115623436, 45.165413040870689, 
7.9633228376744301]])+(1.-msk_arg)*numpy.array([[97.190153160606712, 34.025695735850618, 24.017923414292255, 
18.923109035381529, 49.082977917315738], [89.873747149799129, 39.113868325010408, 2.5484271022905762, 5.1456829853715975, 
7.0556644887527158], [21.77243802201825, 64.168490961123297, 7.0507216576158047, 1.8422050123073481, 65.766409444869112], 
[34.466259574910239, 71.167082993236349, 86.545161965190019, 94.876338416355352, 96.868715620081716]])
      res=log(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[4.5736071655615422, 3.4465207314356334, 1.2655905819378068, 3.4373894834206036, 
4.2320676981284224], [4.1334038197476906, 4.1850661693865305, 3.1335681277596965, 4.0380088016328717, 2.9121420140326402], 
[3.3439932637878873, 4.0056580294617348, 1.5656164865293623, 4.5191378189394564, 4.5856884339646919], [2.7967027702930429, 
3.7990417221274093, 1.0021822996843144, 3.8103315957137749, 2.074846354670552]])+(1.-msk_ref)*numpy.array([[4.5766694014026728, 
3.5271159961124772, 3.1788003605537614, 2.9403838754610931, 3.8935122927640062], [4.498405876025263, 3.6664770927233734, 
0.9354763462258453, 1.638158107875584, 1.9538307677940197], [3.0806448589631037, 4.1615122952185493, 1.9531299743678487, 
0.61096323039521461, 4.186109213225631], [3.5399808621636497, 4.2650303940055148, 4.4606663813329632, 4.5525743427715497, 
4.573356614550784]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[52.971158196331238, 34.180434442487673], [98.522424632684974, 83.605491117100584]], 
[[86.26947899473133, 59.724476499590459], [15.899545532318665, 57.431865375058763]], [[27.507523531761176, 39.197978838477525], 
[70.555671388724733, 66.24336963487147]], [[46.98439151941156, 55.055109769033969], [92.895400256384789, 54.087661875189326]], 
[[96.53103335673724, 21.94879105856657], [46.076478441721179, 90.076199966468636]], [[16.0518718356522, 56.909889962102199], 
[30.179734386339224, 5.3711111095459998]]])+(1.-msk_arg)*numpy.array([[[75.645509713228464, 11.238199062117102], 
[5.0239375880340269, 11.733814694393388]], [[58.085232620507888, 1.6814901769202175], [24.907507751507758, 11.66611707169247]], 
[[56.445951087319393, 37.040834521235254], [3.218989655101109, 82.263048513578781]], [[92.08928685804419, 58.066496660334849], 
[84.930642838249184, 53.284286730475728]], [[53.550447295801078, 47.036305731736995], [90.880947864006188, 
65.449229864137209]], [[69.41453390553923, 61.934555620902032], [24.153090577703221, 72.305842233882245]]])
      res=log(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[3.9697475804548565, 3.5316533880363834], [4.5902841835126873, 4.4261092011525145]], 
[[4.4574758738298588, 4.0897419279928888], [2.766290525944648, 4.0505992951959851]], [[3.314459550228662, 3.6686251852242218], 
[4.2564020632850816, 4.1933353788748233]], [[3.8498154512225233, 4.0083346793703774], [4.531474131745604, 3.9906060983857081]], 
[[4.5698645458352072, 3.0887120609459129], [3.8303025907615686, 4.5006559784046347]], [[2.7758254680501069, 
4.0414691390707667], [3.4071706523181784, 1.6810347976477233]]])+(1.-msk_ref)*numpy.array([[[4.326058082401719, 
2.4193186058377285], [1.6142140063248067, 2.4624748181847811]], [[4.0619114597866846, 0.51968041032576673], 
[3.2151692741486197, 2.4566886635710352]], [[4.0332835623779033, 3.612020939795106], [1.1690675386119449, 4.4099220216047579]], 
[[4.5227586157359134, 4.0615888479676174], [4.4418349567995286, 3.9756414796487438]], [[3.9806241497216379, 
3.8509197658919327], [4.5095503847600682, 4.1812747255049718]], [[4.2400962678569938, 4.1260782730535164], [3.1844123461425955, 
4.2809049313480223]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_log_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[95.735938578949501, 96.363927736963021, 69.704505187942857, 50.543901455188937], 
[93.515666485660105, 10.539787789926832, 47.599161217544747, 80.153990322118474], [54.288338855132451, 81.248264382905461, 
92.720351999340139, 10.15049400871084]], [[12.494937681820341, 74.867938633652031, 65.313394964655771, 80.305395430969298], 
[31.151858549718707, 28.319221346737468, 78.164432127440293, 41.114835918566584], [77.801897905082143, 32.807983084079609, 
62.376884098288585, 22.386170416820519]]], [[[47.064850151954893, 30.573629257977686, 23.966327941058037, 63.207646395461587], 
[55.743265235998329, 58.951730860515568, 30.284828742672243, 26.638942174879102], [54.272839631316366, 7.4006770686770142, 
89.757752786941722, 33.97960190261589]], [[80.635300600927152, 15.806774253524907, 84.540100453519088, 17.538608646723283], 
[90.831276564235566, 39.832275103168868, 13.411504901684284, 1.8417884904364463], [11.167249944595623, 69.363131107953791, 
51.87048846140241, 95.581629192166872]]], [[[38.263867578803385, 58.788951625117114, 14.240207890418008, 32.422684315501293], 
[19.604473478861589, 71.279482467313159, 69.171076895815077, 91.000324009256062], [95.06969517921921, 83.87881614996769, 
75.645783962025305, 28.076748538969213]], [[13.171973533284012, 98.126387049845704, 56.622562690772448, 78.525039093726662], 
[34.9329555383813, 74.657812774208821, 88.378248077817531, 41.371266049738082], [23.692005182018868, 43.188623841946267, 
50.66359543105812, 73.206577568347029]]]])+(1.-msk_arg)*numpy.array([[[[88.151993687138244, 83.555103277656286, 
92.543624479208233, 60.476252939306647], [78.914064301351758, 43.612192007892659, 69.244551339502678, 7.8637269909154455], 
[64.822147341278921, 8.9926844782833584, 55.23622927222651, 97.914859308121649]], [[72.114050762383997, 53.593769904594367, 
26.745083719336051, 17.112580305244741], [0.42296301264405495, 44.156007562546087, 36.764551542515086, 32.650178585562166], 
[67.094272152424466, 8.1169853932058302, 43.822462245115965, 63.668486047351742]]], [[[34.455716614217629, 58.267469286214158, 
17.669759884926485, 23.350497046716622], [69.165539132271633, 99.718171003775552, 63.179967492579372, 43.633679828042638], 
[4.3143912930981676, 86.616565656483544, 43.61506072647034, 26.908148793299638]], [[40.445222681931874, 33.449716824686504, 
39.169088020863711, 44.296437158080508], [41.054977449945383, 94.579911510685946, 54.119472955568682, 75.768954879893343], 
[89.578659299267514, 58.925704009470593, 18.57213872566069, 87.688636672391837]]], [[[27.085452948972669, 57.808491752966411, 
95.786887787139619, 37.94076103608149], [15.043395302137192, 1.5078054549753312, 87.595528650925729, 19.853212662533814], 
[25.991839415040136, 76.181672354948716, 58.050801945363403, 72.370645902864212]], [[66.045878087052714, 24.290083223459632, 
26.19431977043747, 82.401480878146046], [48.34781699553298, 51.540185992217751, 67.950217869033011, 59.670446286015526], 
[2.3506761988729745, 86.586978546948657, 7.7294623906957662, 64.893830268014213]]]])
      res=log(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[4.5615937617040911, 4.5681319379886363, 4.2442649525204033, 3.9228422943741657], 
[4.5381289782550676, 2.355157409127496, 3.8628151396019619, 4.3839496634946302], [3.9943094498278109, 4.3975094595405242, 
4.52958799539336, 2.3175223751123606]], [[2.5253235768251256, 4.3157257432557383, 4.179197144873374, 4.3858368096166691], 
[3.4388739079710602, 3.3435407736289373, 4.3588147119538379, 3.7163690276224011], [4.3541658255536664, 3.4906718724861019, 
4.1331947596000775, 3.1084433760897801]]], [[[3.8515264411410373, 3.4201378484497735, 3.1766498427625511, 4.1464252811044044], 
[4.0207566001012314, 4.0767189880538757, 3.4106468855604661, 3.282374136733615], [3.9940239108508488, 2.0015714917918057, 
4.497114405664882, 3.5257604005370142]], [[4.3899365263467454, 2.760438598368903, 4.4372259833745016, 2.8644046592094932], 
[4.5090036818493084, 3.6846775159029179, 2.5961129133449519, 0.61073710520125513], [2.4129853826431553, 4.2393554743016164, 
3.9487500053500546, 4.5599806383297778]]], [[[3.6445060456466098, 4.0739539397176063, 2.6560695049234733, 3.4788583110138616], 
[2.9757577789114058, 4.2666085225766164, 4.2365828113190451, 4.5108630670517869], [4.5546102560885213, 4.4293730923190484, 
4.3260617078422863, 3.3349417796813583]], [[2.5780913551956712, 4.5862563115424955, 4.0364075398882564, 4.3634175432752924], 
[3.5534306684193591, 4.3129151772390486, 4.4816258768971338, 3.7225865830927538], [3.1651376571391108, 3.7655771235788063, 
3.9252076138486389, 4.293285274421403]]]])+(1.-msk_ref)*numpy.array([[[[4.4790625255721039, 4.4255063336985803, 
4.5276801493339516, 4.1022507745833572], [4.3683594667363188, 3.7753367444734574, 4.2376444609539226, 2.0622606659293519], 
[4.1716473249227723, 2.1964114110598274, 4.0116190653898274, 4.5840983184900841]], [[4.2782489041170324, 3.9814328281979239, 
3.2863506702582188, 2.8398138833863023], [-0.86047054431679726, 3.7877289894249122, 3.6045341076277841, 3.4858503254031068], 
[4.2060986774789448, 2.0939588282462909, 3.7801465225731121, 4.1536897156301906]]], [[[3.5396749231480324, 4.0650439493934778, 
2.8718546973839549, 3.1506182706856314], [4.2365027491676726, 4.6023479171692188, 4.1459872808530625, 3.7758293252714314], 
[1.4619562469362497, 4.4614910866107547, 3.7754025202141528, 3.2924291698529777]], [[3.6999485322510721, 3.5100433209087027, 
3.6678878648145008, 3.7909042484913038], [3.7149120818960872, 4.5494451016124788, 3.9911940648140871, 4.3276886425248504], 
[4.4951171141485959, 4.0762773963022969, 2.9216625397432114, 4.4737923205429331]]], [[[3.298996792210668, 4.0571356810285, 
4.5621258049112496, 3.6360260232429988], [2.7109390445155523, 0.41065525262638608, 4.4727299538419665, 2.9883658410164236], 
[3.257782620102105, 4.3331209134866961, 4.0613185227834201, 4.2818007738466957]], [[4.1903496230717625, 3.1900681692519179, 
3.2655425846068997, 4.4116034085760951], [3.8784210708393267, 3.9423618139142103, 4.2187753469059643, 4.0888368607420782], 
[0.85470303096691036, 4.4611494410635029, 2.0450393117584458, 4.1727525539832673]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sign_float_rank0(self):
      arg=-18.2126891998
      res=sign(arg)
      ref=-1.0
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sign_array_rank0(self):
      arg=numpy.array(-96.6649118156)
      res=sign(arg)
      ref=numpy.array(-1.0)
      if not isinstance(res,float):
         self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
         self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sign_array_rank1(self):
      arg=numpy.array([73.120920394425923, -51.017956554686286])
      res=sign(arg)
      ref=numpy.array([1.0, -1.0])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sign_array_rank2(self):
      arg=numpy.array([[-11.98795701430582, -5.1224317246436897, 20.604467494578074, 44.734787729653391, -36.956220563888166], 
[-59.756609587211074, -93.889070878089512, -16.414713350588244, -6.0655879637917138, 90.990417679571522], [-88.398254957551984, 
83.868063121871444, -22.556040259388581, -53.90526531306881, -59.021139413580379], [99.750840899829655, 39.332495839759162, 
54.187597319917415, -90.665487013280895, -35.135555269515194]])
      res=sign(arg)
      ref=numpy.array([[-1.0, -1.0, 1.0, 1.0, -1.0], [-1.0, -1.0, -1.0, -1.0, 1.0], [-1.0, 1.0, -1.0, -1.0, -1.0], [1.0, 1.0, 
1.0, -1.0, -1.0]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sign_array_rank3(self):
      arg=numpy.array([[[28.666954534473319, -21.849928530629256], [-41.053478289995951, -69.480904899026697]], 
[[-32.669212235787114, -30.333880041906141], [-50.773862521394975, 49.906591235140098]], [[13.932514227107262, 
-95.675501270320737], [-15.998108666310145, 4.9292149009110489]], [[-91.444614673056691, -59.523111501043971], 
[1.8224642975728926, -17.636340087084946]], [[37.118207829150549, 93.838718884698721], [-55.209716984736602, 
40.21190018582277]], [[-91.698429710755349, -81.065525573606294], [66.582789340196541, -70.189513202053661]]])
      res=sign(arg)
      ref=numpy.array([[[1.0, -1.0], [-1.0, -1.0]], [[-1.0, -1.0], [-1.0, 1.0]], [[1.0, -1.0], [-1.0, 1.0]], [[-1.0, -1.0], 
[1.0, -1.0]], [[1.0, 1.0], [-1.0, 1.0]], [[-1.0, -1.0], [1.0, -1.0]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sign_array_rank4(self):
      arg=numpy.array([[[[77.684035876858701, 62.495415597195517, 89.748782943865564, 39.593402447890611], [-77.03246406337351, 
-9.1741547977780726, -48.777374031465051, 4.0823852724476808], [70.70958279193232, -6.6218766406500151, -61.585881895358227, 
70.928340502178571]], [[9.3484396683088704, -52.530944573493478, 26.037510325833679, 24.469329809843046], [-37.228028838834206, 
-18.283255913563281, 77.191687353006131, 50.807452983730798], [-62.822410351789301, -70.366926568058673, -10.468880931314814, 
94.92737012704805]]], [[[77.265457687000008, 88.320143256323547, 33.809728612209739, -49.6610058214793], [-79.519385456671074, 
-99.548708985368734, -23.075864175377305, -16.835460556078473], [-15.977924925095536, 71.243523019214251, -18.165598120205033, 
66.240687540362245]], [[-71.001083534791732, 36.847492133039367, 49.708115443721084, -74.772989691898786], [62.977132978246431, 
-50.880354682296947, 83.702940880511647, -91.927971222587246], [-88.176224794906744, -47.616096801509691, 99.992567501312948, 
-94.722451464031934]]], [[[-27.178688978757108, 68.541351794047927, 32.006935309689339, -71.46525107669504], 
[8.7903550734994553, 95.206440221681902, -93.354544396954523, -61.52507953871271], [-32.994778513188621, -2.0966144518114191, 
-99.505886259195009, -59.601410327863057]], [[-92.387120190008147, 15.56603709767576, 31.254114537361744, -76.455032496853349], 
[-68.554892137329105, 71.503378521340522, 74.355614635822008, 77.090099403680114], [19.329909700223297, -81.018574351829415, 
24.268193763520159, -11.750808179363474]]]])
      res=sign(arg)
      ref=numpy.array([[[[1.0, 1.0, 1.0, 1.0], [-1.0, -1.0, -1.0, 1.0], [1.0, -1.0, -1.0, 1.0]], [[1.0, -1.0, 1.0, 1.0], [-1.0, 
-1.0, 1.0, 1.0], [-1.0, -1.0, -1.0, 1.0]]], [[[1.0, 1.0, 1.0, -1.0], [-1.0, -1.0, -1.0, -1.0], [-1.0, 1.0, -1.0, 1.0]], [[-1.0, 
1.0, 1.0, -1.0], [1.0, -1.0, 1.0, -1.0], [-1.0, -1.0, 1.0, -1.0]]], [[[-1.0, 1.0, 1.0, -1.0], [1.0, 1.0, -1.0, -1.0], [-1.0, 
-1.0, -1.0, -1.0]], [[-1.0, 1.0, 1.0, -1.0], [-1.0, 1.0, 1.0, 1.0], [1.0, -1.0, 1.0, -1.0]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sign_constData_rank0(self):
      arg=Data(-34.8696007917,self.functionspace)
      res=sign(arg)
      ref=Data(-1.0,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sign_constData_rank1(self):
      arg=Data(numpy.array([98.1767032951511, -3.9025857857026551]),self.functionspace)
      res=sign(arg)
      ref=Data(numpy.array([1.0, -1.0]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sign_constData_rank2(self):
      arg=Data(numpy.array([[20.273152095302777, -51.657042758324742, 47.853995410367901, -33.622293848272463, 
-12.594528793089594], [-91.618901987461328, -61.141534524627936, -6.3134018449959513, -17.743445623622804, 
-68.302263040025139], [-22.394249061954397, -76.086628495087339, -55.356405111848161, -55.001895364224396, 
-20.441632463538454], [-1.3457494894788766, 98.662958471584261, -64.453967116855154, 49.118661652213945, 
-74.986652012776773]]),self.functionspace)
      res=sign(arg)
      ref=Data(numpy.array([[1.0, -1.0, 1.0, -1.0, -1.0], [-1.0, -1.0, -1.0, -1.0, -1.0], [-1.0, -1.0, -1.0, -1.0, -1.0], 
[-1.0, 1.0, -1.0, 1.0, -1.0]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sign_constData_rank3(self):
      arg=Data(numpy.array([[[29.474761424983853, 72.834452767423869], [25.741812872325127, -28.828076368884027]], 
[[-88.16808078420722, -53.444141947066903], [29.043188930534001, -50.266419364188877]], [[38.204326793346098, 
-50.239916456565737], [-12.876742166648157, -46.558028743784298]], [[-45.196593487459083, -68.775263260391881], 
[89.057884308693843, -30.948463759992407]], [[21.044897121544579, 6.3902825231764382], [-77.86279451461067, 
91.786000738058277]], [[-43.962746246636343, -49.964655138910132], [-96.712838846057522, 
58.532696291850783]]]),self.functionspace)
      res=sign(arg)
      ref=Data(numpy.array([[[1.0, 1.0], [1.0, -1.0]], [[-1.0, -1.0], [1.0, -1.0]], [[1.0, -1.0], [-1.0, -1.0]], [[-1.0, -1.0], 
[1.0, -1.0]], [[1.0, 1.0], [-1.0, 1.0]], [[-1.0, -1.0], [-1.0, 1.0]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sign_constData_rank4(self):
      arg=Data(numpy.array([[[[2.1438162991766632, 20.330192043649348, 60.913688646032654, 66.274230240723], 
[91.142696774305676, 20.679002197853407, 20.73373992367118, -89.22168064087009], [-41.855801149301143, -31.881673639456622, 
-12.140731767782469, -9.4010532397524287]], [[-75.666576216357882, 17.070185262399121, 87.905511004452364, 
-57.901755277470571], [40.173957785543678, 50.854378633226759, 57.982537444350498, -63.033010936860109], [28.802694238057569, 
-90.135727424882916, -32.513894291619508, -35.161003675716358]]], [[[63.323045034614069, -47.906497587320331, 
-44.897559301307524, 31.006153263223752], [25.4176196368402, -93.264763703283094, -42.350592203666103, -65.755508820166909], 
[-73.720605890366727, 26.298323008534254, 1.5498673240451808, 87.924102196651774]], [[-94.987438137925835, -11.292327666686973, 
-92.123750980319286, -80.249652955710502], [-0.66450593900655974, 62.96004562555467, -55.281171108479498, -79.699316939283761], 
[31.641551253275651, 41.786243102262205, 70.499606484945815, -68.47837649124655]]], [[[-59.810750292299055, 8.0597704193368571, 
97.310128000555864, -13.355086395199493], [58.138680921419677, 90.964885024042758, 79.846081517129079, 5.0078639646581991], 
[-83.119970137736331, 54.314975486973907, 2.6614613131056899, -56.076892926445467]], [[-67.719677183650703, 
-65.581040027594199, 32.587415105060614, -99.975557094748524], [-69.149901484217978, -87.321420902424279, 29.480371334831091, 
74.330952463211332], [71.719064142293462, -81.203419402152861, 69.192797161087441, -81.636781982795782]]]]),self.functionspace)
      res=sign(arg)
      ref=Data(numpy.array([[[[1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, -1.0], [-1.0, -1.0, -1.0, -1.0]], [[-1.0, 1.0, 1.0, -1.0], 
[1.0, 1.0, 1.0, -1.0], [1.0, -1.0, -1.0, -1.0]]], [[[1.0, -1.0, -1.0, 1.0], [1.0, -1.0, -1.0, -1.0], [-1.0, 1.0, 1.0, 1.0]], 
[[-1.0, -1.0, -1.0, -1.0], [-1.0, 1.0, -1.0, -1.0], [1.0, 1.0, 1.0, -1.0]]], [[[-1.0, 1.0, 1.0, -1.0], [1.0, 1.0, 1.0, 1.0], 
[-1.0, 1.0, 1.0, -1.0]], [[-1.0, -1.0, 1.0, -1.0], [-1.0, -1.0, 1.0, 1.0], [1.0, -1.0, 1.0, -1.0]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sign_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(-6.83056003598)+(1.-msk_arg)*(93.7743140716)
      res=sign(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-1.0)+(1.-msk_ref)*(1.0)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sign_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([-11.186851413013386, -71.696318278760486])+(1.-msk_arg)*numpy.array([-18.454967432967038, 
-86.210987232065236])
      res=sign(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-1.0, -1.0])+(1.-msk_ref)*numpy.array([-1.0, -1.0])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sign_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[-53.662299893860379, 5.3599562755663186, 38.296300160748331, -14.039266824115828, 
-42.969056287412677], [-84.752295267262738, 65.858268534283127, 0.30395094312700621, 51.989688475446087, 12.857708063158128], 
[84.468610667456801, 28.775892243353923, 59.213303779928509, -82.045043625679099, 88.463075125430464], [17.985642989336299, 
23.943548761518741, -20.333924318161792, -47.831981682704125, 
58.061164333528723]])+(1.-msk_arg)*numpy.array([[-49.891762439143349, -96.017427504998267, -81.119042329368256, 
75.643791712956897, 65.192004971511238], [38.239866340020683, 96.351188599787719, -19.636274286522323, -67.280508287899636, 
54.516043719202656], [-90.020113044293097, 62.718341408444047, 73.020369037408642, -47.043430168533696, -94.977036825689183], 
[96.603596261937895, -49.201432481025108, -84.787090895219364, -78.558574351296429, -39.622266437952746]])
      res=sign(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-1.0, 1.0, 1.0, -1.0, -1.0], [-1.0, 1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, -1.0, 1.0], [1.0, 1.0, 
-1.0, -1.0, 1.0]])+(1.-msk_ref)*numpy.array([[-1.0, -1.0, -1.0, 1.0, 1.0], [1.0, 1.0, -1.0, -1.0, 1.0], [-1.0, 1.0, 1.0, -1.0, 
-1.0], [1.0, -1.0, -1.0, -1.0, -1.0]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sign_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[-22.099080227702331, 50.244082727106388], [-98.102023954823991, 72.360186198533853]], 
[[-41.289358160120941, -92.594252820770535], [-55.202753589159045, 7.7345397087332088]], [[-16.873802405918781, 
-93.013471457304206], [46.033416851308999, -94.656387911273072]], [[78.397934112405153, 58.064205606720407], 
[-0.54174459157634658, -19.834273492350206]], [[52.861735730991313, 70.512697283136248], [82.341913846616251, 
73.803441980510286]], [[-17.268648148197869, 59.798648302932889], [-49.839046487539008, 
-46.18848048912767]]])+(1.-msk_arg)*numpy.array([[[68.332496555710776, 19.282542334765026], [-35.891572015042897, 
84.251675845930606]], [[-30.115709382530937, 45.96488792440158], [12.547381185696892, -35.474133606191486]], 
[[17.81382903589612, 39.052480179814154], [26.619916559214502, -9.6286154596784286]], [[67.350649437226025, 
-92.982743571652165], [71.645851124020936, 11.593140962418829]], [[-91.440086882735187, 12.865254010286549], 
[-64.59674279042855, -11.253005638765941]], [[-54.063574193853768, -26.181884153476716], [-77.036272048666035, 
16.470774903983013]]])
      res=sign(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-1.0, 1.0], [-1.0, 1.0]], [[-1.0, -1.0], [-1.0, 1.0]], [[-1.0, -1.0], [1.0, -1.0]], [[1.0, 
1.0], [-1.0, -1.0]], [[1.0, 1.0], [1.0, 1.0]], [[-1.0, 1.0], [-1.0, -1.0]]])+(1.-msk_ref)*numpy.array([[[1.0, 1.0], [-1.0, 
1.0]], [[-1.0, 1.0], [1.0, -1.0]], [[1.0, 1.0], [1.0, -1.0]], [[1.0, -1.0], [1.0, 1.0]], [[-1.0, 1.0], [-1.0, -1.0]], [[-1.0, 
-1.0], [-1.0, 1.0]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sign_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[15.002350237389976, -81.062794460130831, -13.062780410335463, 87.614382984683175], 
[-57.724092928435745, -56.634864152104747, 28.853537586824132, 71.016075342345943], [-78.609119381040671, 60.251575060044871, 
80.153834746297747, -32.999429644239882]], [[-42.820827767195865, -80.262523457372595, -29.351240042659853, 
12.308868445404968], [-86.77476565852011, 37.06702729924146, 5.2485654952869112, -3.6664701241923012], [-46.31155396120721, 
80.919256586083037, -47.674211521326157, 21.799442630666846]]], [[[-31.293320234412676, 3.0715092561689659, 2.7313520125245532, 
-49.695438509407076], [37.375659459396331, 14.449041110476074, 45.880315212015716, 40.414990838492457], [9.7983726252563343, 
68.626715156683758, 24.243125887368393, 92.600925622479537]], [[-21.118750800326296, 95.012831130732337, 69.779307988422772, 
-35.657378323940051], [-88.589184583860558, 90.418836827146862, 55.814975313955841, 61.878303195403845], [82.606253935511518, 
57.332107911880826, 85.799523542035331, -86.744941309590601]]], [[[76.202898648652052, -79.026757975788939, 70.291749583399906, 
-7.0849775470977363], [97.121706849519114, 52.398186895496167, 48.445096170529411, -63.718908548881473], [-76.873385615805432, 
8.874476673251479, -95.585592600639615, -35.175335964123832]], [[62.627453650672152, -34.17104389064724, -33.836079613736089, 
-83.295704399950495], [-20.429136239577232, 81.974299742518184, -15.24863610766613, 79.636609090314863], [-98.452000984593255, 
-80.323734618192816, 85.936271409813514, -7.5221373524393158]]]])+(1.-msk_arg)*numpy.array([[[[-72.003339458443861, 
70.568817160651236, -28.368226941811585, 24.983307778863235], [-2.2152682780227479, 25.038768987678566, 78.450048609198035, 
-95.048322419111571], [-91.916830433011242, -55.2095701226446, -4.3700464983362508, 39.382929873768973]], 
[[-59.241650798846202, 24.594269708302917, 72.706930900010093, -99.754324253699906], [-1.7913595307836232, -54.310939812100735, 
45.288109645938306, 66.036579184020582], [-78.60903697484116, -66.005330815121027, 33.191123399450561, 3.0532145005900162]]], 
[[[42.884405340420415, 35.087872935369717, 77.835020139016507, 90.824447187992774], [-30.783231977089144, -63.366317387226111, 
60.219217343436014, 70.732685467213287], [-22.939135172250033, -68.906447783007394, 52.264188231647751, 53.72512874877242]], 
[[69.495283239409247, -80.908475258250064, 73.264176116067318, -82.194314793802619], [96.473304308816097, -72.175046986052706, 
72.032286456844616, -80.607170399708224], [9.1866393286767192, -34.722724474202408, -40.157301950056912, 
-22.489561825300058]]], [[[92.33528659814877, -39.793300070294336, -86.572500540801371, -28.426595976361497], 
[81.77406932474068, -24.987767085288667, -15.327167379923367, -43.047482752797748], [-41.229689788126109, -29.993570970267953, 
93.285450086541687, 44.761383128191255]], [[-77.161988229855012, -56.985232159859663, -85.139006629642438, -31.88394687633847], 
[-34.449607443072082, 84.560551600135625, 4.3035823718425803, -56.041529215228358], [-83.419345552906577, 54.096335917556786, 
-6.4306132846383264, -48.057681049313807]]]])
      res=sign(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[1.0, -1.0, -1.0, 1.0], [-1.0, -1.0, 1.0, 1.0], [-1.0, 1.0, 1.0, -1.0]], [[-1.0, -1.0, -1.0, 
1.0], [-1.0, 1.0, 1.0, -1.0], [-1.0, 1.0, -1.0, 1.0]]], [[[-1.0, 1.0, 1.0, -1.0], [1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, 1.0]], 
[[-1.0, 1.0, 1.0, -1.0], [-1.0, 1.0, 1.0, 1.0], [1.0, 1.0, 1.0, -1.0]]], [[[1.0, -1.0, 1.0, -1.0], [1.0, 1.0, 1.0, -1.0], 
[-1.0, 1.0, -1.0, -1.0]], [[1.0, -1.0, -1.0, -1.0], [-1.0, 1.0, -1.0, 1.0], [-1.0, -1.0, 1.0, 
-1.0]]]])+(1.-msk_ref)*numpy.array([[[[-1.0, 1.0, -1.0, 1.0], [-1.0, 1.0, 1.0, -1.0], [-1.0, -1.0, -1.0, 1.0]], [[-1.0, 1.0, 
1.0, -1.0], [-1.0, -1.0, 1.0, 1.0], [-1.0, -1.0, 1.0, 1.0]]], [[[1.0, 1.0, 1.0, 1.0], [-1.0, -1.0, 1.0, 1.0], [-1.0, -1.0, 1.0, 
1.0]], [[1.0, -1.0, 1.0, -1.0], [1.0, -1.0, 1.0, -1.0], [1.0, -1.0, -1.0, -1.0]]], [[[1.0, -1.0, -1.0, -1.0], [1.0, -1.0, -1.0, 
-1.0], [-1.0, -1.0, 1.0, 1.0]], [[-1.0, -1.0, -1.0, -1.0], [-1.0, 1.0, 1.0, -1.0], [-1.0, 1.0, -1.0, -1.0]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_abs_float_rank0(self):
      arg=51.6201697729
      res=abs(arg)
      ref=51.6201697729
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_abs_array_rank0(self):
      arg=numpy.array(38.7009229865)
      res=abs(arg)
      ref=numpy.array(38.7009229865)
      if not isinstance(res,float):
         self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
         self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_abs_array_rank1(self):
      arg=numpy.array([-58.464470692535954, -22.054919540190255])
      res=abs(arg)
      ref=numpy.array([58.464470692535954, 22.054919540190255])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_abs_array_rank2(self):
      arg=numpy.array([[-96.901355307107508, -78.031647380731556, -50.276890956401488, 77.294230282916402, 
-1.2977621668211583], [38.923897170315655, -33.250122042266824, 84.914594517953873, -47.455465572727974, 61.07165482344621], 
[-87.258896142182763, 62.315569011026611, -34.443972912713733, -11.232072393212462, 80.224693442441662], [-87.424199747787043, 
44.669163056707106, 65.836520129945796, -4.866122371998955, 95.994956107230792]])
      res=abs(arg)
      ref=numpy.array([[96.901355307107508, 78.031647380731556, 50.276890956401488, 77.294230282916402, 1.2977621668211583], 
[38.923897170315655, 33.250122042266824, 84.914594517953873, 47.455465572727974, 61.07165482344621], [87.258896142182763, 
62.315569011026611, 34.443972912713733, 11.232072393212462, 80.224693442441662], [87.424199747787043, 44.669163056707106, 
65.836520129945796, 4.866122371998955, 95.994956107230792]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_abs_array_rank3(self):
      arg=numpy.array([[[58.051065040426408, 34.730343000663652], [-30.506938374862841, 63.230063915832716]], 
[[73.354399062470833, 13.257507592244451], [-54.465765196983142, -71.400334872612774]], [[87.90827748632671, 
-61.253955962782626], [41.671704867000301, -56.307742815500461]], [[86.877218743556568, -16.730993231591057], 
[-55.095918403819113, 58.108230545343758]], [[97.32884537797031, 41.382373278078148], [92.719362484637344, 
77.273361919828943]], [[92.629399023426231, -83.255755082138421], [43.758662700829177, -33.466137177034241]]])
      res=abs(arg)
      ref=numpy.array([[[58.051065040426408, 34.730343000663652], [30.506938374862841, 63.230063915832716]], 
[[73.354399062470833, 13.257507592244451], [54.465765196983142, 71.400334872612774]], [[87.90827748632671, 61.253955962782626], 
[41.671704867000301, 56.307742815500461]], [[86.877218743556568, 16.730993231591057], [55.095918403819113, 
58.108230545343758]], [[97.32884537797031, 41.382373278078148], [92.719362484637344, 77.273361919828943]], 
[[92.629399023426231, 83.255755082138421], [43.758662700829177, 33.466137177034241]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_abs_array_rank4(self):
      arg=numpy.array([[[[48.276976992744437, -19.074065786395806, -27.924516402683693, 99.661716639024888], 
[-82.065445323301532, 87.092700785274531, -68.929995849665417, 48.025105803520034], [-32.022151796386765, -96.539339050628897, 
76.601923308098975, -84.399338758996407]], [[79.742471417725426, -39.727968989983829, -94.139337785917633, 
-48.661234028703568], [-64.85136967574914, 95.006935598502878, 44.946215565740005, 33.315040532092496], [-5.4838910087016615, 
-93.363653946067089, 58.394670625162235, 74.627846038809906]]], [[[-12.701945942535957, 82.193285628769445, 36.251875534638032, 
-23.781275621379748], [-12.159236178993197, -57.328156205965655, 1.9077177398394412, 59.73930741572994], [1.9001733513139811, 
34.493131317340044, 29.730203285673213, -70.017180247120294]], [[-65.272618911019578, 33.903571522727333, -24.793465363843808, 
6.7217774821009186], [88.204946630472961, -2.2966802983844303, 84.91145756297712, -81.247158966944099], [-66.208928501532313, 
-43.505006973827243, 92.925201987956683, 53.92441066318699]]], [[[-13.23461623070645, 14.399479652180332, -39.038512903756597, 
79.361827385623172], [65.389870240596281, -99.864872805009128, 14.999754604352503, -9.5816714947056596], [47.298824174121648, 
96.701686198476693, -65.137295367790614, 87.995337640543738]], [[-79.290913416064782, -68.782288981156412, -43.976014968411924, 
64.628193462810799], [-42.000476402775533, 50.67289070522088, -53.034821394250933, -34.585011169830906], [82.851189268677388, 
17.361456978786904, 79.210129381343705, -41.671529048232415]]]])
      res=abs(arg)
      ref=numpy.array([[[[48.276976992744437, 19.074065786395806, 27.924516402683693, 99.661716639024888], [82.065445323301532, 
87.092700785274531, 68.929995849665417, 48.025105803520034], [32.022151796386765, 96.539339050628897, 76.601923308098975, 
84.399338758996407]], [[79.742471417725426, 39.727968989983829, 94.139337785917633, 48.661234028703568], [64.85136967574914, 
95.006935598502878, 44.946215565740005, 33.315040532092496], [5.4838910087016615, 93.363653946067089, 58.394670625162235, 
74.627846038809906]]], [[[12.701945942535957, 82.193285628769445, 36.251875534638032, 23.781275621379748], [12.159236178993197, 
57.328156205965655, 1.9077177398394412, 59.73930741572994], [1.9001733513139811, 34.493131317340044, 29.730203285673213, 
70.017180247120294]], [[65.272618911019578, 33.903571522727333, 24.793465363843808, 6.7217774821009186], [88.204946630472961, 
2.2966802983844303, 84.91145756297712, 81.247158966944099], [66.208928501532313, 43.505006973827243, 92.925201987956683, 
53.92441066318699]]], [[[13.23461623070645, 14.399479652180332, 39.038512903756597, 79.361827385623172], [65.389870240596281, 
99.864872805009128, 14.999754604352503, 9.5816714947056596], [47.298824174121648, 96.701686198476693, 65.137295367790614, 
87.995337640543738]], [[79.290913416064782, 68.782288981156412, 43.976014968411924, 64.628193462810799], [42.000476402775533, 
50.67289070522088, 53.034821394250933, 34.585011169830906], [82.851189268677388, 17.361456978786904, 79.210129381343705, 
41.671529048232415]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_abs_constData_rank0(self):
      arg=Data(-11.9524872096,self.functionspace)
      res=abs(arg)
      ref=Data(11.9524872096,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_abs_constData_rank1(self):
      arg=Data(numpy.array([8.4650064949724424, -63.650042114851367]),self.functionspace)
      res=abs(arg)
      ref=Data(numpy.array([8.4650064949724424, 63.650042114851367]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_abs_constData_rank2(self):
      arg=Data(numpy.array([[-91.706524146764238, -2.6172952132933887, 56.142839080842037, -38.14947638601376, 
-11.547601406915447], [-32.184039965696726, -32.78567744028507, 0.070569196384411725, -69.604499146046777, 86.052417914549096], 
[60.198511646325841, -80.423517598256041, -71.684126498057168, -96.098199789075807, -26.548187161353027], [12.693952839102877, 
-14.881345730989494, 28.002897927179674, -93.087907709145213, -24.254288851878641]]),self.functionspace)
      res=abs(arg)
      ref=Data(numpy.array([[91.706524146764238, 2.6172952132933887, 56.142839080842037, 38.14947638601376, 
11.547601406915447], [32.184039965696726, 32.78567744028507, 0.070569196384411725, 69.604499146046777, 86.052417914549096], 
[60.198511646325841, 80.423517598256041, 71.684126498057168, 96.098199789075807, 26.548187161353027], [12.693952839102877, 
14.881345730989494, 28.002897927179674, 93.087907709145213, 24.254288851878641]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_abs_constData_rank3(self):
      arg=Data(numpy.array([[[-73.812195265911896, 32.073476552424268], [-44.296822076592335, -13.144780004349684]], 
[[4.4701035555407316, -21.761880919446114], [-48.136124348847311, -85.48241733947583]], [[-31.926106783378927, 
9.3613913473251245], [-8.0340667739837102, -93.454156177116189]], [[-13.316350325087839, -94.921676999336583], 
[51.867551367989563, 21.176458409325761]], [[11.62655968484421, -68.729330126326033], [-14.205717030652181, 
2.9738019581715633]], [[30.607758752435785, -44.705274484441901], [60.09135559215386, 
-56.574857098221187]]]),self.functionspace)
      res=abs(arg)
      ref=Data(numpy.array([[[73.812195265911896, 32.073476552424268], [44.296822076592335, 13.144780004349684]], 
[[4.4701035555407316, 21.761880919446114], [48.136124348847311, 85.48241733947583]], [[31.926106783378927, 9.3613913473251245], 
[8.0340667739837102, 93.454156177116189]], [[13.316350325087839, 94.921676999336583], [51.867551367989563, 
21.176458409325761]], [[11.62655968484421, 68.729330126326033], [14.205717030652181, 2.9738019581715633]], 
[[30.607758752435785, 44.705274484441901], [60.09135559215386, 56.574857098221187]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_abs_constData_rank4(self):
      arg=Data(numpy.array([[[[-44.486016287107532, 46.071179308260582, 19.743573401086849, 2.5927142600415891], 
[39.96550282125483, 86.568707600262712, 33.010186602833357, -77.105228368552986], [79.539273747447908, -35.834770995225611, 
8.7410624024383878, -60.194497048928937]], [[-97.154071528646213, 72.358313392351192, -50.723940580431929, 
-89.053587622890305], [-37.454283519008413, -19.77626282002683, 68.136321424001778, -36.513840622473424], [38.952936344431805, 
-81.997037812570127, 20.52160401409094, 24.687951106479915]]], [[[-19.101501464619687, 43.684588939268139, -75.101417270929147, 
39.812732182907297], [5.9933540067843865, 4.3177881993364764, -91.473571231835933, 92.687794849605325], [70.69957112305201, 
-77.669471358419912, 56.333109185544146, -43.127607169149094]], [[51.97312038995176, -78.132631742016343, 1.9422989994757387, 
-0.76776144016645276], [32.907977971816166, 62.882471290077035, 69.386905332788984, -89.639924645307659], [54.430405906502131, 
-58.985557321722816, 20.567941026119342, -53.340585056570397]]], [[[88.709593546322964, 2.3787578151244873, 93.743258876140146, 
41.769438388067783], [-49.141551858025821, -33.512807274691525, -74.107748282012778, -37.81042799373229], [54.872014385280721, 
12.814600171729154, -94.666379276612361, -61.233731705499395]], [[52.486313292141006, -45.458325728634108, -85.953888595961118, 
-42.505839729050287], [1.5053591034486402, -42.962361694071127, -32.939436192226367, -21.646835936471987], [7.1877095754630176, 
-0.18913903353067951, 9.0191441490823792, -67.059442579565442]]]]),self.functionspace)
      res=abs(arg)
      ref=Data(numpy.array([[[[44.486016287107532, 46.071179308260582, 19.743573401086849, 2.5927142600415891], 
[39.96550282125483, 86.568707600262712, 33.010186602833357, 77.105228368552986], [79.539273747447908, 35.834770995225611, 
8.7410624024383878, 60.194497048928937]], [[97.154071528646213, 72.358313392351192, 50.723940580431929, 89.053587622890305], 
[37.454283519008413, 19.77626282002683, 68.136321424001778, 36.513840622473424], [38.952936344431805, 81.997037812570127, 
20.52160401409094, 24.687951106479915]]], [[[19.101501464619687, 43.684588939268139, 75.101417270929147, 39.812732182907297], 
[5.9933540067843865, 4.3177881993364764, 91.473571231835933, 92.687794849605325], [70.69957112305201, 77.669471358419912, 
56.333109185544146, 43.127607169149094]], [[51.97312038995176, 78.132631742016343, 1.9422989994757387, 0.76776144016645276], 
[32.907977971816166, 62.882471290077035, 69.386905332788984, 89.639924645307659], [54.430405906502131, 58.985557321722816, 
20.567941026119342, 53.340585056570397]]], [[[88.709593546322964, 2.3787578151244873, 93.743258876140146, 41.769438388067783], 
[49.141551858025821, 33.512807274691525, 74.107748282012778, 37.81042799373229], [54.872014385280721, 12.814600171729154, 
94.666379276612361, 61.233731705499395]], [[52.486313292141006, 45.458325728634108, 85.953888595961118, 42.505839729050287], 
[1.5053591034486402, 42.962361694071127, 32.939436192226367, 21.646835936471987], [7.1877095754630176, 0.18913903353067951, 
9.0191441490823792, 67.059442579565442]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_abs_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(-76.4306673681)+(1.-msk_arg)*(96.1336589528)
      res=abs(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(76.4306673681)+(1.-msk_ref)*(96.1336589528)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_abs_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([-89.010231014696714, -52.339610868174759])+(1.-msk_arg)*numpy.array([-12.895599386616226, 
-26.936784819431963])
      res=abs(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([89.010231014696714, 52.339610868174759])+(1.-msk_ref)*numpy.array([12.895599386616226, 
26.936784819431963])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_abs_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[91.761531083145172, -98.424247601533878, 97.60893276874188, 62.767793038696254, 
82.118208308316099], [-97.129354491970574, 3.7694710640934659, 25.9063373642296, 61.770408112654877, -14.095301495513638], 
[-77.915977666023608, -28.983537622054783, 82.487578771865401, 35.780837269172423, -5.070284940289298], [78.692137387496501, 
-99.862720128025188, -44.919957285824339, 56.712139209025167, 
18.983752066426035]])+(1.-msk_arg)*numpy.array([[-83.251509012547359, 25.403825263135445, -63.110101823808208, 
-72.010155109337191, 69.153764809842755], [-4.7294008367999112, 70.277237754283334, 82.614217633540591, -66.976281439509179, 
-67.189203880863687], [94.248756936404646, 86.337200166692696, -54.959327721483284, 57.446363330102855, -84.982468321375947], 
[56.081685037072077, 16.063793682047802, -86.256189500326656, -13.662660875283919, 50.726414883068429]])
      res=abs(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[91.761531083145172, 98.424247601533878, 97.60893276874188, 62.767793038696254, 
82.118208308316099], [97.129354491970574, 3.7694710640934659, 25.9063373642296, 61.770408112654877, 14.095301495513638], 
[77.915977666023608, 28.983537622054783, 82.487578771865401, 35.780837269172423, 5.070284940289298], [78.692137387496501, 
99.862720128025188, 44.919957285824339, 56.712139209025167, 
18.983752066426035]])+(1.-msk_ref)*numpy.array([[83.251509012547359, 25.403825263135445, 63.110101823808208, 
72.010155109337191, 69.153764809842755], [4.7294008367999112, 70.277237754283334, 82.614217633540591, 66.976281439509179, 
67.189203880863687], [94.248756936404646, 86.337200166692696, 54.959327721483284, 57.446363330102855, 84.982468321375947], 
[56.081685037072077, 16.063793682047802, 86.256189500326656, 13.662660875283919, 50.726414883068429]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_abs_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[26.991497379606017, -47.446645742076001], [23.687776050984269, -1.4505437166293547]], 
[[-54.44622856053136, -5.2916172451853924], [-64.356893051962686, 22.725083164070043]], [[0.45637239544325325, 
-33.885431357243007], [63.57799586867057, -14.417024645572084]], [[82.829852200187929, -1.0907690239340866], 
[52.786578147113801, 46.284524780543393]], [[-75.985081333728942, 85.295301132783976], [7.4429378477530292, 
30.168120819807285]], [[-7.3627226199651972, 47.239879399809752], [-85.672117813975063, 
-18.654775009176376]]])+(1.-msk_arg)*numpy.array([[[-6.2425757550023349, -7.7101315156318009], [-80.762556516625864, 
49.210032643193585]], [[57.820568353076567, 97.539767771441234], [54.841202593739155, 80.985707032722729]], 
[[-4.8315730018797325, -68.757343089359324], [-46.141936783719409, 73.53681481299526]], [[-2.1679967375372087, 
58.488731629445198], [-37.370562966921185, -46.292117686461488]], [[-9.6684507220311957, -92.4962744366469], 
[18.72674947931354, -29.933451246999596]], [[19.744183697847561, 36.099800510930493], [-83.461061663424005, 
-38.200545618145789]]])
      res=abs(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[26.991497379606017, 47.446645742076001], [23.687776050984269, 1.4505437166293547]], 
[[54.44622856053136, 5.2916172451853924], [64.356893051962686, 22.725083164070043]], [[0.45637239544325325, 
33.885431357243007], [63.57799586867057, 14.417024645572084]], [[82.829852200187929, 1.0907690239340866], [52.786578147113801, 
46.284524780543393]], [[75.985081333728942, 85.295301132783976], [7.4429378477530292, 30.168120819807285]], 
[[7.3627226199651972, 47.239879399809752], [85.672117813975063, 
18.654775009176376]]])+(1.-msk_ref)*numpy.array([[[6.2425757550023349, 7.7101315156318009], [80.762556516625864, 
49.210032643193585]], [[57.820568353076567, 97.539767771441234], [54.841202593739155, 80.985707032722729]], 
[[4.8315730018797325, 68.757343089359324], [46.141936783719409, 73.53681481299526]], [[2.1679967375372087, 58.488731629445198], 
[37.370562966921185, 46.292117686461488]], [[9.6684507220311957, 92.4962744366469], [18.72674947931354, 29.933451246999596]], 
[[19.744183697847561, 36.099800510930493], [83.461061663424005, 38.200545618145789]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_abs_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[-62.570165039466907, 12.819304823613265, -77.722735359537012, -94.79394036612949], 
[-8.4070619162705924, -77.968504372530575, -6.9213591389462295, -5.871069519721587], [-19.131263689737253, 19.746621027790596, 
58.751824610938343, -3.2982390587958008]], [[-26.511723136531032, -12.387662966957464, 60.863635102984432, 7.2889223766712519], 
[-8.4592434302686854, -63.527048327013944, -19.53543567138334, 29.633775718978484], [-69.182736740958148, -74.544598690879909, 
-54.424904085858714, -22.51903046701338]]], [[[66.902078865728612, -3.6355091404891766, 60.825664861830063, 
71.215972886589782], [-30.948178706069186, -66.369768372369379, -11.504632745333552, 61.221439460571105], [-71.836581857616011, 
-13.8973968443225, -78.237540915154682, -5.553100834949376]], [[-22.50285821644556, 24.173834031076851, -30.803940675785739, 
-56.607070089268149], [-54.202371622344181, 50.758303777623524, -1.588075144091647, 67.358009030721263], [14.371131110703999, 
-23.41663775927978, 33.373189595290313, -33.231714025364383]]], [[[-50.28127692235347, 69.694809178044551, 17.226452655935518, 
-27.695092362515084], [-71.05379300352709, -15.741664267790313, -69.938939285450772, -81.841463115537394], [53.709579500833257, 
37.427058337465411, 80.716603161388434, -71.714647150780152]], [[64.36825135396117, 57.866797935257068, 34.79471978033456, 
69.581737894568505], [-85.209053832817375, 55.107460386755747, 74.800612891725763, 38.146516713798377], [-16.371912746162238, 
-27.889617414407212, -28.339824164626989, -56.216729281028435]]]])+(1.-msk_arg)*numpy.array([[[[96.095924726687059, 
41.965893908882208, 66.115602824884121, -68.28843789101262], [9.0695847158511356, -30.551130169940649, -51.209197122669003, 
97.672353329626418], [-58.557164267624451, -4.7520089718760659, 1.1534556630673194, 94.015485292630018]], 
[[-26.168977850636793, -11.999488915280111, -17.584836225941174, 25.009018688447185], [38.045496522616389, -65.795995243174929, 
32.436602914978863, 81.497996895572413], [55.344697964163913, -88.858601772214939, 65.93281652965598, -94.411246120270434]]], 
[[[-7.0968022097972892, -84.525344437924119, 76.209280395458848, -24.898956140883129], [-33.980965428705588, 
-71.466573540839534, -70.00064205412437, 48.444187742207902], [51.728587585697909, -54.127583159802064, -8.7956546053162157, 
4.7674074887925002]], [[-36.744056798998194, -88.402583637240497, 53.956586347976867, 47.675087427384852], [54.862933675125817, 
19.341066747052182, -44.370791708722088, 93.675747750235359], [83.593942685422377, -66.690381282339615, 91.98204184857272, 
22.795412667428835]]], [[[24.361058692054627, 52.839580737091438, 39.334245159917458, 75.613243304926016], 
[-29.439745726674289, 52.865455927727197, 16.581278051305432, 48.115518748268414], [74.649169011770908, 89.706934852361627, 
46.114651878098925, -16.85746329874334]], [[-83.118050821004076, 41.836096965814193, -86.529486181819706, 81.484892877203322], 
[42.599335257946734, 73.790438523913963, 59.534866829066715, 88.274096952870224], [-26.157942375868799, 17.934422576686188, 
-5.108310193988828, 94.937029691693112]]]])
      res=abs(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[62.570165039466907, 12.819304823613265, 77.722735359537012, 94.79394036612949], 
[8.4070619162705924, 77.968504372530575, 6.9213591389462295, 5.871069519721587], [19.131263689737253, 19.746621027790596, 
58.751824610938343, 3.2982390587958008]], [[26.511723136531032, 12.387662966957464, 60.863635102984432, 7.2889223766712519], 
[8.4592434302686854, 63.527048327013944, 19.53543567138334, 29.633775718978484], [69.182736740958148, 74.544598690879909, 
54.424904085858714, 22.51903046701338]]], [[[66.902078865728612, 3.6355091404891766, 60.825664861830063, 71.215972886589782], 
[30.948178706069186, 66.369768372369379, 11.504632745333552, 61.221439460571105], [71.836581857616011, 13.8973968443225, 
78.237540915154682, 5.553100834949376]], [[22.50285821644556, 24.173834031076851, 30.803940675785739, 56.607070089268149], 
[54.202371622344181, 50.758303777623524, 1.588075144091647, 67.358009030721263], [14.371131110703999, 23.41663775927978, 
33.373189595290313, 33.231714025364383]]], [[[50.28127692235347, 69.694809178044551, 17.226452655935518, 27.695092362515084], 
[71.05379300352709, 15.741664267790313, 69.938939285450772, 81.841463115537394], [53.709579500833257, 37.427058337465411, 
80.716603161388434, 71.714647150780152]], [[64.36825135396117, 57.866797935257068, 34.79471978033456, 69.581737894568505], 
[85.209053832817375, 55.107460386755747, 74.800612891725763, 38.146516713798377], [16.371912746162238, 27.889617414407212, 
28.339824164626989, 56.216729281028435]]]])+(1.-msk_ref)*numpy.array([[[[96.095924726687059, 41.965893908882208, 
66.115602824884121, 68.28843789101262], [9.0695847158511356, 30.551130169940649, 51.209197122669003, 97.672353329626418], 
[58.557164267624451, 4.7520089718760659, 1.1534556630673194, 94.015485292630018]], [[26.168977850636793, 11.999488915280111, 
17.584836225941174, 25.009018688447185], [38.045496522616389, 65.795995243174929, 32.436602914978863, 81.497996895572413], 
[55.344697964163913, 88.858601772214939, 65.93281652965598, 94.411246120270434]]], [[[7.0968022097972892, 84.525344437924119, 
76.209280395458848, 24.898956140883129], [33.980965428705588, 71.466573540839534, 70.00064205412437, 48.444187742207902], 
[51.728587585697909, 54.127583159802064, 8.7956546053162157, 4.7674074887925002]], [[36.744056798998194, 88.402583637240497, 
53.956586347976867, 47.675087427384852], [54.862933675125817, 19.341066747052182, 44.370791708722088, 93.675747750235359], 
[83.593942685422377, 66.690381282339615, 91.98204184857272, 22.795412667428835]]], [[[24.361058692054627, 52.839580737091438, 
39.334245159917458, 75.613243304926016], [29.439745726674289, 52.865455927727197, 16.581278051305432, 48.115518748268414], 
[74.649169011770908, 89.706934852361627, 46.114651878098925, 16.85746329874334]], [[83.118050821004076, 41.836096965814193, 
86.529486181819706, 81.484892877203322], [42.599335257946734, 73.790438523913963, 59.534866829066715, 88.274096952870224], 
[26.157942375868799, 17.934422576686188, 5.108310193988828, 94.937029691693112]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_length_float_rank0(self):
      arg=-0.510102533262
      res=length(arg)
      ref=0.510102533262
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_length_array_rank0(self):
      arg=numpy.array(0.991109436523)
      res=length(arg)
      ref=numpy.array(0.991109436523)
      if not isinstance(res,float):
         self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
         self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_length_array_rank1(self):
      arg=numpy.array([-0.76294483500740484, 0.011677546682929307])
      res=length(arg)
      ref=numpy.array(0.763034197373)
      if not isinstance(res,float):
         self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
         self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_length_array_rank2(self):
      arg=numpy.array([[0.43339000667591221, -0.56813106354058163, -0.70722435943768658, 0.31944814927174714, 
0.79536111956010869], [-0.58324348156333672, -0.43307121633826595, 0.32581231177090197, 0.71482009238312316, 
-0.21002576061225442], [-0.49917603582374914, -0.63591527310476903, -0.87332002545344389, -0.91612221194664523, 
-0.93005892936163725], [-0.19058313727475529, -0.099980978412387289, 0.065023988386496923, -0.50908214426833687, 
0.50919445067665525]])
      res=length(arg)
      ref=numpy.array(2.57369396272)
      if not isinstance(res,float):
          self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
          self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_length_array_rank3(self):
      arg=numpy.array([[[-0.9803545808261207, 0.16576061210577353], [-0.30011375458769374, 0.36782112416630941]], 
[[0.71430934303489013, 0.031917178828326564], [0.62738715091020936, -0.53865697329064433]], [[-0.39594881663888426, 
0.51031034827379074], [-0.65321242269094326, 0.066960424832211762]], [[0.21295172167752496, 0.043385792033981962], 
[0.21039492093814283, -0.45875763326068775]], [[-0.14049283840925275, -0.43861027040015732], [-0.23520058218222872, 
-0.82264226361451076]], [[0.54845190061311566, 0.51238745343929559], [0.77453965957753224, 0.63412987812923882]]])
      res=length(arg)
      ref=numpy.array(2.46349793439)
      if not isinstance(res,float):
          self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
          self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_length_array_rank4(self):
      arg=numpy.array([[[[-0.0089365634034204167, -0.84148551298030028, -0.86751121336543813, -0.15969541886143923], 
[-0.094576010082227269, -0.29859956741061855, -0.9668492649640319, 0.85706900506152772], [-0.53010678888957452, 
0.23317712237751809, -0.54934453799322802, 0.63239069375979762]], [[-0.13286296817699417, 0.38304157353272172, 
0.94879997873099398, -0.41152562922857516], [-0.084699118599110834, -0.77799719953735957, -0.73697310382304981, 
-0.57686761933311148], [-0.691418224706843, 0.86481171901741027, -0.65768637189088874, -0.36467431513366044]]], 
[[[-0.84165953650117875, -0.030676018280178408, 0.68829493775738415, -0.3155228468519129], [-0.7871478173248736, 
-0.079032190579141126, -0.2177526168512447, -0.22662606238170069], [-0.36280488027671387, -0.30025872969028122, 
-0.1153990482492997, -0.64384164007636602]], [[0.43765759938201887, -0.83664343730186874, 0.20723950229710386, 
-0.56690864703827692], [-0.5752004595601341, 0.45715366983278116, -0.60791501104508927, -0.12584705515500483], 
[0.013022887283654638, -0.82203757609388917, 0.2276716578033573, -0.82649410580263871]]], [[[-0.73587736736778697, 
-0.075026122599631151, 0.47107705394337485, -0.5188872506774429], [-0.91844772159081667, -0.17389147261342175, 
-0.53243380583847233, -0.25336755215727336], [-0.7085054821135035, 0.92376667162885795, 0.31427206042606071, 
0.58305001351683972]], [[0.021860457140477907, -0.21893493890938442, 0.78089201272561026, -0.30413016651663138], 
[0.57250768478386815, 0.12062954871057352, -0.39418583967206589, 0.81081960816290599], [-0.62197759509633688, 
0.26606514367555634, -0.71790752119252188, -0.75228782860426513]]]])
      res=length(arg)
      ref=numpy.array(4.74889961553)
      if not isinstance(res,float):
          self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
          self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_length_constData_rank0(self):
      arg=Data(-0.621348324338,self.functionspace)
      res=length(arg)
      ref=Data(0.621348324338,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_length_constData_rank1(self):
      arg=Data(numpy.array([0.32933097146312651, 0.074695318351896134]),self.functionspace)
      res=length(arg)
      ref=Data(0.337695542388,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_length_constData_rank2(self):
      arg=Data(numpy.array([[-0.78192059954553383, -0.90936771512644765, -0.62922561123663545, 0.017784281736960006, 
0.9748747532171671], [0.058303280563049231, -0.2728591332910737, 0.031130134154037092, -0.77308681021304904, 
-0.11934107563523066], [0.16559555223913947, 0.42636646304641856, -0.57573933774337149, 0.94979031593856034, 
0.50631731019815573], [0.93403604481436431, -0.93439497144943484, -0.41539043658961994, -0.19785740903293636, 
-0.37401756164552413]]),self.functionspace)
      res=length(arg)
      ref=Data(2.69665449585,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_length_constData_rank3(self):
      arg=Data(numpy.array([[[-0.56969059522354204, 0.080095572899512968], [0.10882275113239914, 0.5219995791238794]], 
[[-0.45702291442137533, -0.48209810378998008], [-0.45061700485329181, 0.13409269350080355]], [[0.82959328262385434, 
0.43146697783249], [-0.17051774940195052, -0.11306038728528978]], [[0.1343766654439873, 0.26708535580201764], 
[-0.72182400606958974, 0.68503438438868658]], [[0.63507734978497932, 0.20722711779440695], [0.096695049458556781, 
-0.72783199691610112]], [[0.15899265871592516, -0.59393453523719764], [0.81362670777065538, 
0.44781346438355851]]]),self.functionspace)
      res=length(arg)
      ref=Data(2.34509551447,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_length_constData_rank4(self):
      arg=Data(numpy.array([[[[-0.15764527853259791, -0.30365168015204125, 0.47795233787343117, -0.98256953311171458], 
[0.0042986463256691199, 0.8035927446211617, -0.16701915572918757, 0.71275425093914158], [-0.96088340875110778, 
0.44246392270280177, -0.86258499735816296, -0.83799786987315006]], [[0.9570143548429173, 0.038448779070990424, 
-0.85758098571845975, 0.54765015129054828], [-0.87491827027518299, -0.45074431963780448, -0.3985454061322331, 
-0.96368920535242153], [0.68864777189463311, 0.08364027809503205, -0.15072701440063496, 0.37493446731949942]]], 
[[[-0.40037966034446559, -0.20439352375445297, -0.00083579211732121017, 0.71146162550419212], [0.57902779303828478, 
0.36965035149772829, -0.80710960825758327, -0.22291291503275867], [-0.050754745699356008, -0.74788880214762354, 
-0.29584218595779421, -0.15944687137497104]], [[-0.22559210423055043, -0.39361850034269019, -0.38362580334572582, 
0.45418102497613622], [0.42704140840065086, -0.16313142654556012, 0.27467683038933854, -0.92068758637216352], 
[-0.25578430722641832, -0.83470097064580551, 0.3737146950202388, 0.77547342867161784]]], [[[0.7869046231589758, 
0.42798876622613702, 0.063281236738654822, 0.90878329459830454], [0.9439304175181531, 0.50711996212079513, 
-0.23938857199714936, 0.55774844240445276], [0.5514271888482718, -0.71911109998895695, -0.079482513135118893, 
-0.50190162484878265]], [[-0.075988757174491495, 0.21390351071272762, -0.34834540192677776, -0.67335194629906114], 
[0.10253847169273911, 0.63514982365898098, 0.54802667295899443, -0.16433191310464568], [-0.74944750341855726, 
0.99070513266942384, -0.55535271544968623, 0.41204768838694661]]]]),self.functionspace)
      res=length(arg)
      ref=Data(4.80409161546,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_length_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(0.883776938505)+(1.-msk_arg)*(-0.0468902435277)
      res=length(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(0.883776938505)+(1.-msk_ref)*(0.0468902435277)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_length_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([-0.58845041644366813, 0.84032988075424919])+(1.-msk_arg)*numpy.array([0.24641851124094183, 
-0.83673496226594679])
      res=length(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(1.0258792332)+(1.-msk_ref)*(0.872265716259)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_length_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[-0.29586555942077419, 0.34274407148638875, -0.88727513948396664, 0.27160261149044485, 
0.21413449360849302], [0.28011027275430478, 0.75357944869733395, 0.24679947624917853, 0.23083266549318582, 
-0.031185959204752756], [0.80951928180904842, 0.1946069687658758, 0.19231558044922537, -0.78993705668305569, 
-0.34498118770326403], [-0.90798215978462582, 0.36396049398056696, -0.99479482070441749, 0.71319176756439284, 
-0.173316315027086]])+(1.-msk_arg)*numpy.array([[0.78326921791525206, 0.44780263366625617, 0.90004049588763002, 
0.86601571330532723, 0.89302226317470446], [-0.49280621379733769, 0.032223308649646842, -0.063785824564493776, 
-0.45830913844852361, 0.75573504780401213], [0.029363497650474857, 0.3732291937763248, -0.23856608969815407, 
-0.78387113786982909, 0.51309780260863525], [-0.055709228333384875, 0.47635746182366057, 0.69511257282026917, 
-0.70929759063656039, 0.65249778651689483]])
      res=length(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(2.41490921017)+(1.-msk_ref)*(2.63013072264)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_length_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[-0.46999515470673003, 0.35828008438348191], [-0.75807606560681484, 0.17336937881565895]], 
[[0.50255994290015416, 0.74389850916125733], [0.56251892098022993, -0.77887021071121221]], [[-0.94958377643932868, 
-0.55488659642221627], [-0.81161288797652253, 0.80725266426883979]], [[-0.38519127035487832, 0.85595943718503564], 
[-0.54032736868611431, -0.93807094048446449]], [[-0.18354593266618413, -0.14323695746161924], [-0.41586461136365327, 
0.83381306861717031]], [[-0.56427207094458476, 0.47224479851988166], [-0.93279101876654669, 
-0.48877724755416652]]])+(1.-msk_arg)*numpy.array([[[0.86959183228486547, -0.46716569229451088], [0.3766416916310138, 
-0.26875186058841827]], [[-0.85315616483044532, -0.44895823472349328], [-0.019719427383600552, 0.87331397676878764]], 
[[0.18450481536393126, -0.93079080072281672], [-0.51399520036784985, 0.02094393310881526]], [[0.34454271397516889, 
-0.057292572357173555], [0.88270644599519299, 0.55068444546064366]], [[0.79135120340267306, -0.21348692991823648], 
[0.56803236664975065, 0.91336197474998104]], [[0.16655314750502104, -0.42962978230839477], [0.44885045412539593, 
0.42912119101208823]]])
      res=length(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(3.13508828546)+(1.-msk_ref)*(2.77076207142)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_length_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[-0.93886781108675321, 0.75905716358672759, 0.37094825123593167, -0.19680193477839869], 
[-0.69130982226153326, -0.097833952605903773, -0.18032702039691939, -0.65126134878000252], [-0.50441371581635863, 
0.19853216992359735, 0.75211775926254854, 0.93313886601723994]], [[-0.81202720654711702, 0.39117544309224295, 
0.32422500396848175, -0.86970863899495798], [0.15860494885441168, -0.99388324869686295, -0.6126507983503755, 
0.45539570478611968], [-0.29981340221794484, 0.43378007384304418, -0.07804401429490726, 0.10520159837508403]]], 
[[[-0.067360545733815114, -0.75432265949843069, 0.23150713101298548, -0.62781393972269117], [0.16588303814612515, 
0.6338577465238171, 0.44543958860038324, -0.6116568441601038], [0.62085456412712414, -0.59591336867783595, 0.31657816819520201, 
0.96855037419698409]], [[0.89768670071243273, 0.047512605872325642, 0.38802965964728497, -0.047024004368128258], 
[0.28534214825759774, 0.79378823415801558, -0.41325077352252704, -0.41319182463438486], [-0.41172433227767002, 
-0.99248504691886752, -0.61435540813756728, -0.79377934237436421]]], [[[0.31056101760667421, 0.03997905342037944, 
-0.54839575574971211, -0.54963398878208447], [-0.19258715091848266, -0.67023816299126682, -0.61922833170488634, 
-0.3132139396514777], [0.95178737480931175, -0.84632696057445056, -0.043652087473059931, -0.39857277601740737]], 
[[-0.91850296338642701, 0.72749660991813925, 0.4827895084985423, -0.10560131182372157], [-0.91003622797953954, 
0.35676038745111271, -0.26082661035830013, -0.12361013193170001], [-0.22592227457768388, -0.20389666165635489, 
-0.008049373422560846, -0.50062341851266434]]]])+(1.-msk_arg)*numpy.array([[[[-0.021088077793533166, 0.62042392362261389, 
-0.90699620151616278, 0.83244686127694245], [-0.65935596202227198, 0.2773135818931769, -0.14671290779705948, 
-0.42303151477725387], [-0.67360170011508336, -0.94136391147369758, 0.60593006592988452, 0.53271662567479994]], 
[[-0.68567319305989449, -0.33687160212342526, 0.48445948721184595, 0.8639977742197007], [0.41684351239404371, 
-0.47583440168725621, 0.46055807978272645, 0.19279035992433058], [0.83098230075402957, -0.7973760655728046, 
0.22342511963387834, -0.89134877289705128]]], [[[0.35974286191438831, 0.21872434629252235, -0.63220544985048988, 
-0.52612039893949269], [-0.34493816049763404, -0.34015531655509457, 0.44937887672159205, -0.58847483438357262], 
[-0.34681966794346097, -0.91972234941106046, -0.51080277628557447, 0.45147645224503452]], [[0.70760788343718772, 
-0.031664619981034203, -0.16729804256437064, -0.80164830175689361], [0.97863767766204868, 0.91357020902462205, 
-0.0895749852663974, 0.50091501286651696], [-0.88377743658291208, 0.4536825024535025, 0.61591907060400253, 
0.57165024614346471]]], [[[-0.37160858934673713, 0.50843004647167334, 0.39647835504208251, 0.28802988212615177], 
[-0.9374130465768602, -0.97417359787823421, -0.54320852345322379, 0.48577720898699028], [-0.95423595276764428, 
0.033048674364561048, 0.69837226182049372, 0.037284034055920845]], [[-0.18095086863386811, 0.51687760740956334, 
-0.035166388385410308, -0.54024030399678025], [0.15723730135407932, 0.57872797253456132, -0.50187686605795045, 
-0.19174626179381882], [-0.36712116936754535, -0.94080998057238707, -0.84929322547542996, 0.46359997819244714]]]])
      res=length(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(4.72250379236)+(1.-msk_ref)*(4.95651187575)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maxval_float_rank0(self):
      arg=0.315854880825
      res=maxval(arg)
      ref=0.315854880825
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maxval_array_rank0(self):
      arg=numpy.array(-0.377472785293)
      res=maxval(arg)
      ref=numpy.array(-0.377472785293)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maxval_array_rank1(self):
      arg=numpy.array([0.67183156228324403, 0.98323683252360539])
      res=maxval(arg)
      ref=numpy.array(0.983236832524)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maxval_array_rank2(self):
      arg=numpy.array([[0.84034096955817894, 0.29665901063728572, -0.42190010384108523, -0.7150438773637906, 
0.14643629018234949], [0.22677764147443136, 0.14594960752890751, -0.74161102103554177, -0.042525371203158802, 
0.45739481907885904], [0.92281820938114612, -0.74136143299473378, -0.082616726724768386, 0.024796098776131537, 
-0.26761237274922411], [0.44638984793823289, 0.45566292575391332, -0.45703791845559705, 0.96294823838515708, 
0.20865001046980014]])
      res=maxval(arg)
      ref=numpy.array(0.962948238385)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maxval_array_rank3(self):
      arg=numpy.array([[[-0.69786738686010863, 0.22220015828286921], [0.39818252195561255, 0.62828383756907602]], 
[[-0.30663179429588383, 0.58142297666853637], [0.58551100004900891, -0.26454346793231442]], [[0.82280266855051476, 
-0.89955650739217519], [0.0091889934169460119, -0.10055418070372735]], [[0.40193434282716689, 0.77124273444864078], 
[-0.70873019803530313, -0.81529900334190519]], [[0.46864138606445871, 0.65526620123485468], [-0.8719674792548715, 
-0.94339546902627047]], [[0.56873047860501025, 0.38953302649374399], [-0.98526085240200989, -0.48033381512355788]]])
      res=maxval(arg)
      ref=numpy.array(0.822802668551)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maxval_array_rank4(self):
      arg=numpy.array([[[[0.85720742878416645, 0.27988827387323667, 0.49070838881477474, -0.45763132819743535], 
[-0.67237336183311225, -0.25445937371923955, 0.10361458853479233, 0.10413904498543669], [-0.93692345104386443, 
0.2678428054175328, 0.25533247056621211, -0.32305638067221998]], [[0.59340473127311899, -0.87227874511895909, 
0.6241736038770771, 0.79467283603250483], [-0.92762377016283226, -0.68487229864254506, -0.066576628979943431, 
-0.33180819533101635], [-0.50565694661802785, -0.27864825619146205, -0.10892529985569377, -0.38472639795925501]]], 
[[[0.89332236053146707, -0.53172955419735013, 0.041432658449450654, 0.46681521077857546], [0.021583966439083779, 
0.41850959174604618, 0.51949264416518059, -0.66312837589642548], [0.33246856848437045, -0.4801113671503936, 
0.55902212823529829, -0.63177305756143087]], [[-0.33448057589573166, -0.75739896498945303, 0.91642906573402105, 
-0.29316527400677339], [-0.55921776067113349, 0.11502888427705971, 0.60735749648784854, 0.41281299421768169], 
[0.90078770354412674, 0.93534572171226271, 0.67200314010782347, -0.83277182798924998]]], [[[0.50057755897525991, 
-0.7863579282681803, -0.78780741517599662, -0.80342487860441247], [0.71775356842357452, 0.60691302920938384, 
-0.61563688967106733, 0.89279476199536179], [0.60907733165373656, -0.97645380984070052, -0.68344826887442944, 
0.02838484569891464]], [[-0.39073286697774301, -0.37909899562646054, 0.16487244638410781, -0.20422979552499498], 
[0.42360937159006551, -0.21911732897340896, -0.77260176428047567, -0.39140596490426249], [-0.25978163449682445, 
0.69597575969217407, -0.82813780051302399, -0.2437763687760135]]]])
      res=maxval(arg)
      ref=numpy.array(0.935345721712)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maxval_constData_rank0(self):
      arg=Data(0.00425301467094,self.functionspace)
      res=maxval(arg)
      ref=Data(0.00425301467094,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maxval_constData_rank1(self):
      arg=Data(numpy.array([0.24481630771410834, 0.8482027633371616]),self.functionspace)
      res=maxval(arg)
      ref=Data(0.848202763337,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maxval_constData_rank2(self):
      arg=Data(numpy.array([[-0.62805848358264393, 0.47792569168847021, 0.63753766788606292, 0.56872613192380017, 
-0.62140547559582826], [-0.53300885909946549, 0.087728738264873707, 0.27815721899198342, -0.60846756436547444, 
0.76312701161889152], [-0.90031662952456015, -0.93562023084611834, -0.077605527954475173, -0.51077693482781039, 
0.95897780151638878], [-0.32875230962884427, -0.40568224632821437, 0.60125655094211172, 0.73095614466625314, 
0.77566626029390551]]),self.functionspace)
      res=maxval(arg)
      ref=Data(0.958977801516,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maxval_constData_rank3(self):
      arg=Data(numpy.array([[[-0.41133807237181697, 0.339475432953392], [0.751688023632332, -0.45416714276660897]], 
[[-0.82710852884284014, -0.54581937207379094], [0.96752580081305073, 0.017286990318454487]], [[0.95948104622292951, 
0.57403336395794513], [-0.4652813457699212, 0.87679325951992881]], [[-0.27378818188120158, 0.29628771669225618], 
[0.55915654739024712, -0.90003391143575473]], [[0.61472043944167232, 0.35735658086868294], [-0.20019759448385233, 
-0.0097721699054988953]], [[0.55933363921880153, 0.97230703622729164], [-0.97883751547481035, 
-0.16616084753658722]]]),self.functionspace)
      res=maxval(arg)
      ref=Data(0.972307036227,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maxval_constData_rank4(self):
      arg=Data(numpy.array([[[[-0.10987928269740532, 0.34460441718388068, -0.42251161876169707, -0.21841227109764882], 
[-0.52039679114287773, -0.067873360247190373, 0.71884821427626466, -0.77230257512959066], [-0.36304670040479614, 
-0.16015922187117293, 0.86853171882086122, -0.96567539865168328]], [[0.96601317514359786, 0.83752149178113511, 
0.65892376484600801, 0.31630886002000169], [-0.66226270610768667, -0.68350164369220967, -0.81054005335510793, 
0.3180480536844339], [-0.87422470417268672, -0.17129283972715248, 0.28039370045536738, 0.29695941930546255]]], 
[[[-0.47630810423711178, 0.52959745217192444, 0.040993980179728773, 0.5245591726930805], [-0.89383657059829846, 
0.029315208725834152, 0.88056186549081983, -0.63784231116158785], [-0.60183199925280473, -0.92921784953609343, 
0.01889309869387179, -0.49090749904730235]], [[-0.32390997756647799, 0.36642208468565207, -0.68111798476689711, 
-0.92004014931390654], [-0.0292412230442054, -0.71320760352104884, -0.57106096519639893, 0.012657128392972838], 
[0.70387931057541575, 0.25942878473151754, 0.78392882823935928, -0.26722465851260502]]], [[[0.26096223103983385, 
0.89553376726470413, -0.56049710274754716, 0.51051997594553389], [-0.87602756662141301, -0.29261382562617144, 
-0.12873590235943388, 0.69707757280626437], [0.63111654272128237, -0.56667864380284994, 0.04781100525711901, 
0.062962267029093688]], [[0.83354469589166058, 0.49147678083366575, -0.47313464330467436, -0.37984462029444055], 
[0.048709899932206424, -0.28197892047593331, 0.61145832786089915, -0.051745145517938074], [0.97194002325382467, 
-0.068565885935067961, 0.7719415422184297, -0.27187882484530701]]]]),self.functionspace)
      res=maxval(arg)
      ref=Data(0.971940023254,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maxval_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(0.332235320145)+(1.-msk_arg)*(-0.24604500324)
      res=maxval(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(0.332235320145)+(1.-msk_ref)*(-0.24604500324)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maxval_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([-0.59427677340505958, -0.84740257716903722])+(1.-msk_arg)*numpy.array([-0.72031138157212804, 
0.92918576704192746])
      res=maxval(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-0.594276773405)+(1.-msk_ref)*(0.929185767042)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maxval_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[0.41368801368105679, 0.29360526282114119, -0.30954301330224121, 0.26476660295182097, 
0.37942955464311723], [-0.14137058932732383, 0.56216536188087152, 0.85723969959744473, -0.34157407776672866, 
0.71187680163817757], [0.028323013942354924, -0.8271833359392009, -0.46852604935440745, 0.17851739216808915, 
0.66516251742568744], [-0.65769919956640965, 0.82148451636261277, -0.99401768066284557, 0.10732593805672952, 
-0.28231159743214285]])+(1.-msk_arg)*numpy.array([[-0.35422666104845879, -0.83499817780768693, 0.94898016702033416, 
-0.035772961794744429, 0.57477321877630838], [0.22489182670080199, -0.41146981529375215, -0.72758976996124791, 
-0.41066296298400817, 0.80718619480745124], [0.80235446968120638, -0.35840819741634955, -0.05618651096108862, 
-0.94078438157309785, -0.56658816600726691], [-0.73505106729271441, -0.33158257909652189, 0.44199211387292636, 
0.29529742850205309, 0.85346454421416262]])
      res=maxval(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(0.857239699597)+(1.-msk_ref)*(0.94898016702)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maxval_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[-0.496818524218406, 0.26161963352560824], [0.91044707618692389, -0.9432697535729444]], 
[[-0.10609846145317547, -0.72291144577158972], [-0.15568936492204277, -0.053141421374103048]], [[0.53303053127115252, 
0.24937055476184278], [-0.85544018056817905, 0.32023326034177413]], [[-0.19803538880229699, 0.071601808939763201], 
[0.78888291536029431, 0.10606346665910005]], [[-0.83691574508233346, 0.75194007447412869], [-0.46213522303320165, 
0.74118532899627509]], [[-0.27203854500308888, 0.0556603061557428], [-0.27568024558144644, 
-0.6596832871328886]]])+(1.-msk_arg)*numpy.array([[[-0.38387363639151295, -0.03233312164578428], [-0.7132988320484781, 
-0.12222778087194741]], [[-0.23514106160045301, 0.42068398992463907], [-0.73285598813751607, -0.85491869063637438]], 
[[-0.096049701961620437, 0.39603320496426764], [-0.86688149264271797, -0.48263249645534678]], [[0.61270540922521399, 
-0.27815399087641524], [-0.75568629133180254, 0.06586879881883867]], [[-0.98103416178316749, -0.34258493606401386], 
[-0.44344220314932414, -0.92896633412201046]], [[0.75303826320532785, -0.75939337307517341], [0.66523101128309792, 
0.42131106360540072]]])
      res=maxval(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(0.910447076187)+(1.-msk_ref)*(0.753038263205)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maxval_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[-0.39063884840579055, -0.062593055462244251, -0.022752676756311097, 0.82567766395652442], 
[-0.61782203411671532, -0.7174987171488576, 0.012416472497936981, 0.77732527522608708], [0.2581482129115249, 
-0.028970138751502228, 0.21400817374883752, 0.096518944579570443]], [[-0.9333652649577453, -0.3485799161416816, 
-0.22059438876339588, -0.32222484146513297], [0.049776858853060491, -0.39729968061906007, -0.15644245293308434, 
0.57933231238347394], [-0.91585792154697998, 0.6490525691757576, 0.13701512944986138, -0.50714811705008134]]], 
[[[0.94717926915302164, 0.44616257454360975, -0.40750067768073506, -0.50001686912458432], [-0.5640527606203567, 
-0.16806643428738455, -0.99809710887506187, -0.28997095902879977], [0.96080123261322714, 0.7072630535529012, 
0.060165698718575689, -0.48760045654442763]], [[0.8605264623824691, 0.37995771692823999, -0.58889533613098277, 
-0.12102927983038536], [-0.75913777624959944, 0.95883798144078658, -0.0170674547606533, 0.40302350028136646], 
[-0.41467348140561078, -0.34169615998854042, -0.094247808145228928, -0.77243636153593309]]], [[[0.76847611775181623, 
0.31114571658639933, 0.3375738255385885, 0.8252971351240701], [0.16843391508372885, -0.17487468243823523, 0.9500225050341351, 
0.15765605380472492], [0.84806367006401939, -0.8921145799188599, 0.70066703654989038, -0.29226492555032313]], 
[[0.26167372305421277, -0.14246109041907506, 0.67992861287405981, -0.85922091895242425], [-0.20238389587707561, 
-0.2880395272779086, -0.58172933182948428, 0.98397658132546684], [-0.70323630626752842, 0.047805048436355024, 
0.1334599526040372, 0.091415361986823429]]]])+(1.-msk_arg)*numpy.array([[[[0.45435533405178563, -0.24704909409153553, 
0.20906352762224722, 0.87632790903409052], [-0.27526942171841084, -0.20312196158302176, 0.62221965869732276, 
-0.51231649769277432], [-0.91064163780949081, -0.67582252480982086, 0.50052639822596001, -0.069909103979628684]], 
[[-0.83392522017960569, -0.87540703757185967, 0.02544719796251993, -0.68287237837241843], [-0.10314783966584584, 
0.33658223994757219, 0.2486432654565609, -0.50566553769425715], [0.3590841982402051, -0.69644955060860281, 
0.055034764880927289, -0.028554584782987558]]], [[[0.68013452135981023, -0.80066123330402261, 0.33556647382902027, 
0.58147090184653938], [0.55665998011202222, -0.69132338787736347, 0.91301301976970572, -0.36903333405262262], 
[0.84606547219969408, 0.1992186661680091, 0.74823224796946786, 0.43629167990068662]], [[-0.49014959276017844, 
0.774758193648448, 0.14961755035441104, 0.40547083361887415], [0.37831252161933504, -0.4349417215086282, 0.71836802753181694, 
-0.24652218870808684], [-0.38428296978779963, 0.99667296612807799, 0.16883895231650103, -0.42076923428686475]]], 
[[[-0.08342541607713061, -0.69745118901650538, -0.83706308088889525, 0.81201518967613806], [-0.38122402193435234, 
-0.46153944893815702, 0.43276508943432845, -0.086332781036586592], [0.7242527519061337, -0.60544717368864243, 
-0.29315538455633927, -0.97452928976637287]], [[-0.61595949771180325, 0.87520574402241502, 0.61694207224169428, 
-0.37854520619327103], [-0.14310710281978745, -0.61259903416672601, -0.66322044415369796, 0.89868948108915769], 
[0.95673622765714006, 0.17577298932819585, -0.11163387589198637, -0.61928756437099852]]]])
      res=maxval(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(0.983976581325)+(1.-msk_ref)*(0.996672966128)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minval_float_rank0(self):
      arg=-0.0813988872107
      res=minval(arg)
      ref=-0.0813988872107
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minval_array_rank0(self):
      arg=numpy.array(0.203135339689)
      res=minval(arg)
      ref=numpy.array(0.203135339689)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minval_array_rank1(self):
      arg=numpy.array([0.53544902048869591, -0.81095461710032879])
      res=minval(arg)
      ref=numpy.array(-0.8109546171)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minval_array_rank2(self):
      arg=numpy.array([[0.27214337670370048, 0.15560988073469084, 0.90015619199343311, -0.9025114039043387, 
-0.84476541550754214], [0.067681521831817637, 0.49725231481834298, -0.44690725788544317, 0.81611521497601935, 
-0.23818755255201474], [0.39409048948021819, -0.32989779963822996, 0.071598660625914912, -0.048721135674946847, 
-0.88486359278268867], [-0.047242209881730668, 0.5117999752695459, -0.065254537763000275, -0.98426384139999601, 
0.42640166196036344]])
      res=minval(arg)
      ref=numpy.array(-0.9842638414)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minval_array_rank3(self):
      arg=numpy.array([[[-0.57801369559578375, 0.8475709087853267], [-0.62156558628902858, 0.16317256776729372]], 
[[0.82280749191458802, 0.86757046029308138], [-0.80468648257487341, 0.50347496153235838]], [[0.57910271328476837, 
0.31832794067003101], [-0.43115546130737159, 0.96031608499179488]], [[0.49539799715856692, -0.10882931867648438], 
[-0.10954143472670097, -0.40160540217269403]], [[0.24326476418285869, 0.73974055647226611], [-0.062874816874243677, 
-0.084436498355318745]], [[0.022195220279687566, 0.77562819716638032], [-0.88624021250624607, 0.39848396848232337]]])
      res=minval(arg)
      ref=numpy.array(-0.886240212506)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minval_array_rank4(self):
      arg=numpy.array([[[[-0.14474876918789525, 0.54987363867145578, 0.041514287216928958, 0.50784202943730339], 
[0.22428027284721597, -0.90703539828428648, -0.57639520455060667, -0.48198988610144466], [0.94878712908709151, 
-0.34378540985741024, -0.85617090009829844, 0.38542149004545845]], [[0.40439396247960424, -0.33396533725982902, 
0.57102838967013425, -0.77292398584017619], [-0.30211593932962177, 0.31377477534575848, -0.65372193392871303, 
0.7598903854852137], [0.93212303414631692, -0.9918986741212521, -0.31650029155130377, -0.79547565383603569]]], 
[[[0.22810964805040368, 0.36801496715542159, -0.50290854627818193, -0.031218734373054335], [0.91491532038518919, 
0.3432896329141184, -0.67365848697820074, -0.56415601144850136], [-0.94529850756161604, 0.063750250741384518, 
-0.94341968293701362, 0.0086523607884236853]], [[-0.15793249822138344, 0.33733718444805083, -0.071528717305355149, 
-0.67498602448352707], [0.36887437066094186, -0.045164571795437425, 0.50187055239691203, -0.68431142493430408], 
[-0.5805689340418223, -0.1405830463965192, 0.49818568480317338, -0.49073491820057957]]], [[[-0.4547271777538564, 
0.0052825628200054986, 0.8386907168648059, -0.79277949820721361], [-0.17773231635682119, -0.73046788469917323, 
-0.32125176839388025, -0.81768823086027997], [0.97056323533228195, -0.12463448959269829, -0.057103597501829473, 
0.40417829046576115]], [[0.055559060913653591, -0.2205433998937556, -0.4707694934849973, 0.27520647582020419], 
[-0.7793261855641469, 0.2312443862224689, 0.49823564342243087, 0.50087939122723002], [0.13357337216768128, 
-0.21846566992619709, -0.67496935894945098, -0.047733069730594435]]]])
      res=minval(arg)
      ref=numpy.array(-0.991898674121)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minval_constData_rank0(self):
      arg=Data(-0.570106420108,self.functionspace)
      res=minval(arg)
      ref=Data(-0.570106420108,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minval_constData_rank1(self):
      arg=Data(numpy.array([-0.19847831559753981, 0.40610397160211087]),self.functionspace)
      res=minval(arg)
      ref=Data(-0.198478315598,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minval_constData_rank2(self):
      arg=Data(numpy.array([[-0.9639131333497899, -0.30622184254591556, -0.56534225119979475, 0.97113536551499657, 
0.72771392953144876], [-0.71538714174098872, 0.52973718939771963, -0.028651679335222946, 0.5929024511132337, 
-0.4866110022815282], [0.027770288586371983, -0.32978280221198308, 0.13790608674958271, -0.45983276751063973, 
0.31586441270296217], [-0.25006380493198432, -0.23317987015724517, -0.49457991065340834, -0.66183045720993428, 
0.6546653003791596]]),self.functionspace)
      res=minval(arg)
      ref=Data(-0.96391313335,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minval_constData_rank3(self):
      arg=Data(numpy.array([[[0.78772704902154156, 0.69698576470167839], [0.58349353871022425, -0.44429652744170367]], 
[[0.95122195248495611, -0.13329242810307163], [0.61489426950194725, -0.82627278543241833]], [[-0.20676898827641677, 
0.90746615695355337], [-0.21454902325466274, -0.06996179955865367]], [[-0.77386810656368321, -0.50452086581667466], 
[0.10741771738964179, 0.54945560673302318]], [[0.62059435342635272, 0.42764710981542731], [-0.69961537691414666, 
0.41473366544360712]], [[-0.69496424443778304, -0.71149599990967038], [0.41282921885173396, 
0.38102417382970177]]]),self.functionspace)
      res=minval(arg)
      ref=Data(-0.826272785432,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minval_constData_rank4(self):
      arg=Data(numpy.array([[[[-0.31617134142491587, -0.79150138292975791, -0.69179461153514898, 0.64957729213045989], 
[-0.81436417612892531, 0.42248652353980631, -0.25642022653882335, 0.27550856107884569], [0.0095865917205664175, 
0.13528115156489906, 0.72542268735015081, 0.30411944968409288]], [[0.43223908791872279, -0.031384323435708961, 
-0.27138787236554074, 0.94846267932468575], [-0.003171134402055964, -0.23874859540786719, -0.88544193420317718, 
0.81720893129705563], [0.9901501746767003, 0.23991422240922367, 0.10638468953085067, 0.4302537013800245]]], 
[[[-0.85020088072889277, 0.4552859449987976, -0.92342379862574431, 0.21936304106407478], [0.41100362018711367, 
0.69396272891408439, 0.4157617323953402, 0.79743336407104493], [-0.59192141405336196, -0.084339584940965606, 
-0.52187164926130802, -0.12126412509513074]], [[0.40596845427308348, 0.24314808354086681, 0.85019356732762463, 
0.68633031900498054], [0.13575440057734922, -0.056231914047657483, -0.71847761514904684, 0.16852540949725237], 
[-0.45562770062897218, 0.61253598275865895, -0.018428323215295084, 0.63750932492290602]]], [[[-0.95140029572331875, 
0.35903135903332606, 0.35062441928298194, 0.27289245766078674], [0.95038622171897491, 0.065120389529117251, 
0.39001097938148432, 0.41345777245129756], [0.42532849657704208, 0.19939457782565295, 0.77528674822056765, 
-0.050697957849086928]], [[-0.24970902580501475, -0.78386638935427833, -0.25378258399293552, -0.83330630180475596], 
[-0.030380655291931902, -0.96043242888909464, -0.14531326816314305, 0.39809472723243222], [-0.66171460089490175, 
-0.51291855133171671, 0.36594014782248796, 0.80253386528405168]]]]),self.functionspace)
      res=minval(arg)
      ref=Data(-0.960432428889,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minval_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(-0.778944214859)+(1.-msk_arg)*(0.953941502903)
      res=minval(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-0.778944214859)+(1.-msk_ref)*(0.953941502903)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minval_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([-0.73159504520188157, 0.68299192035807255])+(1.-msk_arg)*numpy.array([-0.3100681210597318, 
-0.65103554359904581])
      res=minval(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-0.731595045202)+(1.-msk_ref)*(-0.651035543599)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minval_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[-0.52277742513644765, -0.2277572152261802, -0.9511128526148902, 0.32017435807815242, 
-0.11136201958976244], [-0.80459257355669767, -0.95924143992579869, 0.50769588182449388, 0.86438625827388305, 
0.16860560966597427], [-0.72754696854920198, 0.67889538815250394, 0.6088408927519946, -0.72782010895545679, 
0.56616049103099786], [-0.11534623429520585, -0.092041379862312089, 0.56715923140761726, 0.0028734812988140401, 
-0.13593267576797796]])+(1.-msk_arg)*numpy.array([[0.32400946916673323, -0.99131325248594693, -0.29006290223719677, 
-0.81287830308772602, -0.14575320006496795], [0.23894418840621556, 0.99107715887800141, 0.69785171074135177, 
0.20540649144636669, -0.022178813617514059], [0.97125674513729066, 0.54103580977289178, -0.69594163458966074, 
0.045941422243358288, 0.51911179383180106], [-0.040941899113513314, 0.29908960491193759, 0.31070274901135231, 
-0.42386794845092379, -0.14020897842782176]])
      res=minval(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-0.959241439926)+(1.-msk_ref)*(-0.991313252486)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minval_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[0.2243593847183587, -0.55964269546917778], [0.086055585732637496, 0.85322676708491429]], 
[[-0.95462358187710294, -0.85991920026432767], [0.19704146031466507, 0.12266428022749887]], [[0.62491825811592427, 
0.74058421362755578], [-0.00909447859974577, -0.98100849454252903]], [[-0.35262739845910374, -0.92427752522606532], 
[0.26279257951943724, -0.94076289760274867]], [[0.2774731787257616, 0.66406607908393478], [-0.10466380765605798, 
0.9590095850576017]], [[-0.63686305073922522, 0.8055769733197482], [-0.18278593082199945, 
-0.34288116248154421]]])+(1.-msk_arg)*numpy.array([[[0.57238934623145776, -0.6008625917571444], [0.3693766407620318, 
0.70221455793260423]], [[-0.58848946803840674, -0.096161764675626982], [0.78217332385378691, 0.61375244943994911]], 
[[-0.30778069185951695, -0.55126352159863301], [0.6408299490321927, 0.99953805406121954]], [[0.26931958474181017, 
-0.60531989284036225], [-0.89886607437725319, 0.74907780969586568]], [[0.57734208195866099, -0.58661021170011907], 
[0.38922797525267239, 0.29493691914357334]], [[0.46695639713123049, 0.23171759126301761], [-0.052706998692634732, 
-0.78334544272850204]]])
      res=minval(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-0.981008494543)+(1.-msk_ref)*(-0.898866074377)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minval_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[0.46483848448463783, -0.87939053653747745, 0.18006027907711064, 0.59393412035549553], 
[-0.060446335653558059, -0.58654265140261042, -0.7103456625567921, -0.87737620348261225], [0.7915602685554104, 
0.058998481855049523, 0.68263476322383454, -0.346029376351082]], [[0.11758788080658089, 0.24948633626899586, 
-0.6136373781657769, 0.99819746973171686], [-0.5990037217573525, 0.81089434898012391, -0.79998545086485873, 
-0.99177417401308809], [-0.77652852159987606, -0.2482337515175721, 0.11649788623113655, -0.074557844494566483]]], 
[[[0.26806387309289925, 0.82459759920272702, 0.33106191528634432, 0.89462805607445128], [-0.19028989619203829, 
0.68615021175809709, 0.44076423865654846, 0.2756983480068973], [-0.77132677194140387, 0.99573172063081605, 
-0.57302930492670434, -0.50733002523701476]], [[-0.84471955813954591, -0.85176783029149705, -0.92684957977259796, 
0.59023847705979327], [0.68196420699586402, 0.58361004083768364, -0.71922201576501466, -0.20596576225294583], 
[0.79395039679328305, 0.472333500890604, -0.13097923185774984, 0.24331532019306423]]], [[[-0.83479197740948829, 
-0.0075589700651166325, -0.78975633265041978, 0.018667115085772901], [-0.049769243736694113, 0.5209394114068675, 
0.065925130190574111, -0.7431184749488009], [0.62313144993384539, -0.98514078085147361, -0.87222126612101603, 
0.90735220611080991]], [[0.79727858426486953, -0.85669952349479006, -0.74511584235611039, 0.026323551913185472], 
[-0.23933451992599641, 0.60640846085161648, -0.14396449329896144, 0.16035835222923356], [-0.050012848665722975, 
0.5731667028545524, 0.99860242361478924, 0.32146275555560866]]]])+(1.-msk_arg)*numpy.array([[[[0.62874848202922995, 
0.25537251444066023, -0.698374966993069, -0.38687588298399156], [0.49378024086811134, 0.7769539896932669, -0.10934606676333569, 
-0.21298971767267072], [-0.61708921394679916, -0.84745222820602195, 0.80837213654566931, 0.43012734925042673]], 
[[0.46908437808287862, 0.15847492833174037, 0.43970355055642618, -0.45249108496330326], [-0.99773656166168645, 
-0.30411024919059959, 0.29944789752031253, -0.92091659949825022], [0.14139835253841881, 0.65286819300206411, 
-0.33942314449708033, -0.25137887940591175]]], [[[0.12370235507304339, -0.26504015746942589, -0.77988713910590435, 
-0.90788635421965291], [-0.90705126108317047, 0.069929652428008193, 0.99718954015895278, -0.97450250453685361], 
[-0.70326152893105709, -0.091126066100986503, 0.78506550006102915, -0.72917498081256715]], [[0.42629906246786486, 
-0.91603749611465091, 0.31297245971177445, -0.8326590552190356], [0.3039549953318279, -0.71996259445943389, 0.9724420712657329, 
0.41592436425988466], [0.57718038790671078, -0.39375559879969191, 0.55089171490650224, -0.15686786098227512]]], 
[[[0.91433628175647907, -0.8335217876290042, 0.14319348718149993, -0.15460951345782847], [-0.1413557115515387, 
-0.7965552409354113, -0.6647827134700568, -0.6562005559403794], [0.8996333133589296, -0.60792710713831077, 0.32829660654943127, 
0.59993702653925007]], [[-0.20459622156626156, 0.54626282250284341, -0.12114642969601674, 0.31432342880341024], 
[0.44005219027624398, 0.45505989580530981, -0.30837994783093614, -0.88417181461185046], [0.2415549698483368, 
0.03180095496331381, 0.53977209810077453, -0.41688385798938143]]]])
      res=minval(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-0.991774174013)+(1.-msk_ref)*(-0.997736561662)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_clip_float_rank0(self):
      arg=-0.660844742681
      res=clip(arg,-0.3,0.5)
      ref=-0.3
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_clip_array_rank0(self):
      arg=numpy.array(-0.616432559809)
      res=clip(arg,-0.3,0.5)
      ref=numpy.array(-0.3)
      if isinstance(res,numpy.ndarray): 
         self.assertEqual(res.shape,(),"wrong shape of result.")
      else:
         self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_clip_array_rank1(self):
      arg=numpy.array([-0.51271199503186593, -0.822484111396796])
      res=clip(arg,-0.3,0.5)
      ref=numpy.array([-0.29999999999999999, -0.29999999999999999])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_clip_array_rank2(self):
      arg=numpy.array([[0.30100679449598511, 0.88767064649145766, 0.33865733182911906, 0.85531213080164914, 
0.99410402183813074], [0.095717025396767408, 0.79153699203511629, 0.69300596381099822, 0.52370716757069058, 
-0.59416147926326457], [-0.19974051297983286, -0.54785769539787643, 0.6908558361120738, 0.10853690318676268, 
-0.92243333197553667], [0.38961082223496413, 0.89291409313272374, 0.057863956684621076, 0.50395838385178759, 
-0.34645817452731187]])
      res=clip(arg,-0.3,0.5)
      ref=numpy.array([[0.30100679449598511, 0.5, 0.33865733182911906, 0.5, 0.5], [0.095717025396767408, 0.5, 0.5, 0.5, 
-0.29999999999999999], [-0.19974051297983286, -0.29999999999999999, 0.5, 0.10853690318676268, -0.29999999999999999], 
[0.38961082223496413, 0.5, 0.057863956684621076, 0.5, -0.29999999999999999]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_clip_array_rank3(self):
      arg=numpy.array([[[0.80651421777631804, 0.42731881222688806], [-0.97337503979172202, 0.55560652110255382]], 
[[0.57745817086577378, -0.74819163530551558], [0.65118109098829802, 0.45705605350544798]], [[0.84734044549687937, 
-0.97694579653116231], [0.11346644908007897, 0.97057408781664289]], [[-0.44110401079642081, -0.90446847999697488], 
[0.82981000049437226, 0.62852473157503597]], [[0.55643517187873859, 0.17159295755472526], [-0.3276983994906566, 
0.82111281587385077]], [[0.66037767196123465, -0.8138922973233893], [0.22428143590074856, 0.60032297288637171]]])
      res=clip(arg,-0.3,0.5)
      ref=numpy.array([[[0.5, 0.42731881222688806], [-0.29999999999999999, 0.5]], [[0.5, -0.29999999999999999], [0.5, 
0.45705605350544798]], [[0.5, -0.29999999999999999], [0.11346644908007897, 0.5]], [[-0.29999999999999999, 
-0.29999999999999999], [0.5, 0.5]], [[0.5, 0.17159295755472526], [-0.29999999999999999, 0.5]], [[0.5, -0.29999999999999999], 
[0.22428143590074856, 0.5]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_clip_array_rank4(self):
      arg=numpy.array([[[[0.46285917453321823, -0.91689578652051007, 0.85018159714116481, -0.25906072812293113], 
[0.81932949407078137, -0.60307967212356628, 0.56538931825524186, -0.40508251808655071], [0.77170764992626406, 
0.81299384258133656, 0.7404976524889133, 0.51860787224188187]], [[-0.62450454506954589, 0.94207725980470158, 
-0.12374258578480357, 0.66339457762270237], [0.35534578368047653, 0.42819081302985462, 0.063059236470800473, 
-0.73505630795988397], [0.24842113864491688, 0.44227807933837804, 0.36053730941174211, -0.34351746910574721]]], 
[[[-0.54014660449687857, -0.73963869205638022, -0.85829607205589942, 0.19478539963645392], [-0.94553140352926479, 
-0.29641607950976878, -0.57063497488586767, -0.48500827538538505], [-0.45756026282338103, -0.19137078199124091, 
-0.26560363911562557, -0.80865109063294827]], [[0.14311339149994962, 0.29424466150856587, 0.99821626133853081, 
-0.1461637145270076], [-0.72334486633204564, 0.033116994477579764, 0.47044975702162461, 0.40109158073490891], 
[-0.038219493414332506, 0.73839517283012568, 0.14208390499507262, 0.49550413385768977]]], [[[0.398910718898426, 
0.03446413720460928, -0.46028688211877444, 0.70797648020636439], [0.19363454901242005, 0.92111688093404243, 
0.79409098989516091, 0.71035510610026265], [0.12629228314430518, 0.88671129037390428, -0.91227198363005213, 
-0.94341232990645918]], [[0.57833363856424014, -0.47043185008904964, -0.63643237130510921, 0.28866935981066577], 
[0.39621689148915196, -0.89298541799113895, -0.76674863298496154, 0.86640350436457525], [0.33847977460676804, 
0.57610754346250648, 0.22238502772765689, -0.57901129898204906]]]])
      res=clip(arg,-0.3,0.5)
      ref=numpy.array([[[[0.46285917453321823, -0.29999999999999999, 0.5, -0.25906072812293113], [0.5, -0.29999999999999999, 
0.5, -0.29999999999999999], [0.5, 0.5, 0.5, 0.5]], [[-0.29999999999999999, 0.5, -0.12374258578480357, 0.5], 
[0.35534578368047653, 0.42819081302985462, 0.063059236470800473, -0.29999999999999999], [0.24842113864491688, 
0.44227807933837804, 0.36053730941174211, -0.29999999999999999]]], [[[-0.29999999999999999, -0.29999999999999999, 
-0.29999999999999999, 0.19478539963645392], [-0.29999999999999999, -0.29641607950976878, -0.29999999999999999, 
-0.29999999999999999], [-0.29999999999999999, -0.19137078199124091, -0.26560363911562557, -0.29999999999999999]], 
[[0.14311339149994962, 0.29424466150856587, 0.5, -0.1461637145270076], [-0.29999999999999999, 0.033116994477579764, 
0.47044975702162461, 0.40109158073490891], [-0.038219493414332506, 0.5, 0.14208390499507262, 0.49550413385768977]]], 
[[[0.398910718898426, 0.03446413720460928, -0.29999999999999999, 0.5], [0.19363454901242005, 0.5, 0.5, 0.5], 
[0.12629228314430518, 0.5, -0.29999999999999999, -0.29999999999999999]], [[0.5, -0.29999999999999999, -0.29999999999999999, 
0.28866935981066577], [0.39621689148915196, -0.29999999999999999, -0.29999999999999999, 0.5], [0.33847977460676804, 0.5, 
0.22238502772765689, -0.29999999999999999]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_clip_constData_rank0(self):
      arg=Data(0.873991409144,self.functionspace)
      res=clip(arg,-0.3,0.5)
      ref=Data(0.5,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_clip_constData_rank1(self):
      arg=Data(numpy.array([0.10029829613225805, 0.47403026690156014]),self.functionspace)
      res=clip(arg,-0.3,0.5)
      ref=Data(numpy.array([0.10029829613225805, 0.47403026690156014]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_clip_constData_rank2(self):
      arg=Data(numpy.array([[-0.28275534784898304, -0.13729948733728325, 0.013813296255671847, 0.73460682612929729, 
0.24551982471646627], [0.61067563404013048, 0.42590985220127653, 0.053795995066850733, -0.12813919742697655, 
-0.45354184407079257], [0.027592398296971021, -0.27728369179302592, 0.83461918535632207, 0.5996778630178683, 
0.64537419802208373], [0.46265837345465788, -0.013903448047151823, 0.43282947286633333, -0.7140268312200182, 
0.11491473517057038]]),self.functionspace)
      res=clip(arg,-0.3,0.5)
      ref=Data(numpy.array([[-0.28275534784898304, -0.13729948733728325, 0.013813296255671847, 0.5, 0.24551982471646627], [0.5, 
0.42590985220127653, 0.053795995066850733, -0.12813919742697655, -0.29999999999999999], [0.027592398296971021, 
-0.27728369179302592, 0.5, 0.5, 0.5], [0.46265837345465788, -0.013903448047151823, 0.43282947286633333, -0.29999999999999999, 
0.11491473517057038]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_clip_constData_rank3(self):
      arg=Data(numpy.array([[[-0.44382776839546945, -0.63662456777605092], [-0.54650947561527286, -0.84571189798389201]], 
[[-0.84094147647363049, 0.64826856634432373], [0.622621310911184, 0.23179746677715629]], [[0.72783765690686408, 
-0.14342314789815735], [0.38201531418904833, -0.69055788612963065]], [[-0.0064036502394755335, 0.88418526966764577], 
[0.38413648569626235, -0.48827873166332258]], [[-0.36805980178883502, -0.52366150491610441], [-0.17224428981836781, 
-0.11145730040193791]], [[0.0032312327392447493, 0.40509583911865898], [-0.078221409096661088, 
0.55983165731854423]]]),self.functionspace)
      res=clip(arg,-0.3,0.5)
      ref=Data(numpy.array([[[-0.29999999999999999, -0.29999999999999999], [-0.29999999999999999, -0.29999999999999999]], 
[[-0.29999999999999999, 0.5], [0.5, 0.23179746677715629]], [[0.5, -0.14342314789815735], [0.38201531418904833, 
-0.29999999999999999]], [[-0.0064036502394755335, 0.5], [0.38413648569626235, -0.29999999999999999]], [[-0.29999999999999999, 
-0.29999999999999999], [-0.17224428981836781, -0.11145730040193791]], [[0.0032312327392447493, 0.40509583911865898], 
[-0.078221409096661088, 0.5]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_clip_constData_rank4(self):
      arg=Data(numpy.array([[[[0.74565220604294113, 0.70707680004089535, -0.8773779384765521, 0.32828037279397537], 
[0.12930191501822041, 0.61037326219072852, -0.72378849864905304, -0.71546199305068181], [-0.99689819686806036, 
0.27574928269813848, -0.35838582444761657, -0.5263903369979559]], [[0.34209505072502133, 0.15911882981611325, 
0.35995945219157122, 0.66221816529573552], [0.43624456519819632, -0.86396951527158183, -0.39169318175569057, 
-0.42904934103954684], [-0.41672280174621079, 0.5747168458744496, 0.10190810106330472, -0.40269367788473454]]], 
[[[0.96435342278392056, -0.38598803996794429, -0.10569714455448587, 0.78395387940432859], [-0.038509327942198635, 
0.33381791489369883, -0.15711165198428034, 0.62080633262478568], [-0.85531025762383317, -0.51127155377851863, 
-0.98887178638798634, 0.74904701210248192]], [[-0.24723563589657527, 0.40962376639148879, -0.63040816326005822, 
-0.7114593474303792], [0.86532397329368793, 0.93027464092180923, 0.63743386509330668, 0.095944939045965238], 
[0.40272444381571271, -0.11827319884920562, 0.90658957997212819, 0.75580908924882162]]], [[[0.38931728355886186, 
-0.87926757063573313, 0.5065858460075825, -0.094306906597873885], [0.80847359053294654, -0.41252922807094805, 
-0.33611596899498153, 0.26443168557169749], [-0.017360832827441719, -0.67566709940607206, -0.64725079995155332, 
-0.65760481990092745]], [[0.31365664160961182, 0.18146342960739825, -0.91968636759286748, 0.95202972228558935], 
[0.29398556423299493, -0.19710497775305269, 0.051818367197931092, -0.58880901521639895], [-0.15383079570743585, 
0.058375088082090132, 0.86765927088269068, 0.54391547419416297]]]]),self.functionspace)
      res=clip(arg,-0.3,0.5)
      ref=Data(numpy.array([[[[0.5, 0.5, -0.29999999999999999, 0.32828037279397537], [0.12930191501822041, 0.5, 
-0.29999999999999999, -0.29999999999999999], [-0.29999999999999999, 0.27574928269813848, -0.29999999999999999, 
-0.29999999999999999]], [[0.34209505072502133, 0.15911882981611325, 0.35995945219157122, 0.5], [0.43624456519819632, 
-0.29999999999999999, -0.29999999999999999, -0.29999999999999999], [-0.29999999999999999, 0.5, 0.10190810106330472, 
-0.29999999999999999]]], [[[0.5, -0.29999999999999999, -0.10569714455448587, 0.5], [-0.038509327942198635, 0.33381791489369883, 
-0.15711165198428034, 0.5], [-0.29999999999999999, -0.29999999999999999, -0.29999999999999999, 0.5]], [[-0.24723563589657527, 
0.40962376639148879, -0.29999999999999999, -0.29999999999999999], [0.5, 0.5, 0.5, 0.095944939045965238], [0.40272444381571271, 
-0.11827319884920562, 0.5, 0.5]]], [[[0.38931728355886186, -0.29999999999999999, 0.5, -0.094306906597873885], [0.5, 
-0.29999999999999999, -0.29999999999999999, 0.26443168557169749], [-0.017360832827441719, -0.29999999999999999, 
-0.29999999999999999, -0.29999999999999999]], [[0.31365664160961182, 0.18146342960739825, -0.29999999999999999, 0.5], 
[0.29398556423299493, -0.19710497775305269, 0.051818367197931092, -0.29999999999999999], [-0.15383079570743585, 
0.058375088082090132, 0.5, 0.5]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_clip_expandedData_rank0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(-0.842028460005)+(1.-msk_arg)*(-0.293977917573)
      res=clip(arg,-0.3,0.5)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-0.3)+(1.-msk_ref)*(-0.293977917573)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_clip_expandedData_rank1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([0.68183730574045565, 0.11696629718283025])+(1.-msk_arg)*numpy.array([0.35288961662527463, 
-0.072592053270659651])
      res=clip(arg,-0.3,0.5)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.5, 0.11696629718283025])+(1.-msk_ref)*numpy.array([0.35288961662527463, -0.072592053270659651])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_clip_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[0.90716898369189836, -0.35803512117401981, 0.94018031372277644, 0.79669090472583703, 
-0.18693600910396135], [0.66131238434994155, -0.97051536409745509, 0.9041285176368381, -0.85346375259533169, 
0.080322075200497611], [-0.88411631255225487, -0.034767111898162728, -0.57932986353520599, -0.36438560630357641, 
0.46429678163635812], [0.27577795037496067, 0.63194239600641899, -0.25968796804973282, 0.57336606861296535, 
0.42681293935570541]])+(1.-msk_arg)*numpy.array([[-0.39043293288066505, -0.12586769834531109, 0.86941394855912812, 
0.87286911746074325, -0.31449584387107565], [0.067675865430771864, 0.1411821630632728, 0.36884686073023309, 
0.44919011697492928, -0.28594899701518273], [0.54855238117480076, 0.17688408664749944, 0.25657627305599928, 
-0.39115693595816903, 0.79599505168892248], [-0.63955125945377644, -0.94202055254588668, -0.68308189366965677, 
0.4578895128510021, -0.60069619882054903]])
      res=clip(arg,-0.3,0.5)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.5, -0.29999999999999999, 0.5, 0.5, -0.18693600910396135], [0.5, -0.29999999999999999, 0.5, 
-0.29999999999999999, 0.080322075200497611], [-0.29999999999999999, -0.034767111898162728, -0.29999999999999999, 
-0.29999999999999999, 0.46429678163635812], [0.27577795037496067, 0.5, -0.25968796804973282, 0.5, 
0.42681293935570541]])+(1.-msk_ref)*numpy.array([[-0.29999999999999999, -0.12586769834531109, 0.5, 0.5, -0.29999999999999999], 
[0.067675865430771864, 0.1411821630632728, 0.36884686073023309, 0.44919011697492928, -0.28594899701518273], [0.5, 
0.17688408664749944, 0.25657627305599928, -0.29999999999999999, 0.5], [-0.29999999999999999, -0.29999999999999999, 
-0.29999999999999999, 0.4578895128510021, -0.29999999999999999]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_clip_expandedData_rank3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[0.93020317689046128, -0.54447904233724853], [0.11461289830616783, -0.22468993486584155]], 
[[0.2580017724494863, 0.96989125372971774], [0.15057436976633753, -0.78151964173949384]], [[-0.71582456376904258, 
0.82703131516257322], [-0.24926797106114562, -0.1238935499896896]], [[0.32428682415562093, -0.91311839613385692], 
[0.30146899459086263, -0.12556147633489956]], [[-0.68665518227346567, -0.22144952551343633], [-0.50981084983177216, 
-0.030119060352006066]], [[0.18318336820339765, -0.38259427020750825], [0.19442151433667187, 
-0.99958596975615577]]])+(1.-msk_arg)*numpy.array([[[-0.4343470595053156, 0.96097360321015346], [-0.46568918744082421, 
0.48300612207019933]], [[0.81630000256950841, 0.38374999480003424], [0.79617490239235345, -0.097514887490353752]], 
[[0.039610536347541192, -0.11954981925174257], [-0.40947914780332573, -0.48354495774094963]], [[-0.078080941006748672, 
-0.46167291110337838], [0.26568933223987878, 0.96040591883607895]], [[0.64111624021647295, 0.67697706382199141], 
[0.64028160169512827, -0.98698793247338523]], [[-0.9114669507080646, 0.62094506194746613], [-0.42103142433552065, 
-0.63632370203732247]]])
      res=clip(arg,-0.3,0.5)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.5, -0.29999999999999999], [0.11461289830616783, -0.22468993486584155]], 
[[0.2580017724494863, 0.5], [0.15057436976633753, -0.29999999999999999]], [[-0.29999999999999999, 0.5], [-0.24926797106114562, 
-0.1238935499896896]], [[0.32428682415562093, -0.29999999999999999], [0.30146899459086263, -0.12556147633489956]], 
[[-0.29999999999999999, -0.22144952551343633], [-0.29999999999999999, -0.030119060352006066]], [[0.18318336820339765, 
-0.29999999999999999], [0.19442151433667187, -0.29999999999999999]]])+(1.-msk_ref)*numpy.array([[[-0.29999999999999999, 0.5], 
[-0.29999999999999999, 0.48300612207019933]], [[0.5, 0.38374999480003424], [0.5, -0.097514887490353752]], 
[[0.039610536347541192, -0.11954981925174257], [-0.29999999999999999, -0.29999999999999999]], [[-0.078080941006748672, 
-0.29999999999999999], [0.26568933223987878, 0.5]], [[0.5, 0.5], [0.5, -0.29999999999999999]], [[-0.29999999999999999, 0.5], 
[-0.29999999999999999, -0.29999999999999999]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_clip_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[0.47253941060193694, 0.19296787597619924, -0.48241297799548888, -0.75064623239139383], 
[-0.58524626791925782, 0.17426765589991278, 0.35312486983619285, 0.083669184154522291], [-0.84069384309987161, 
-0.15040140084518883, -0.41665586874162486, -0.52240956580940079]], [[-0.6154323811169542, -0.57957166937646276, 
0.79263239744786929, 0.75307658018532964], [-0.11379828349950394, -0.95566162645602204, 0.82201752038788811, 
-0.82419839021415009], [0.040955022200906965, 0.80044686051015956, -0.084301202894394134, 0.50298997540454393]]], 
[[[0.38713164665793709, -0.9893659106059931, -0.40195267367189746, 0.54085998487601983], [-0.36547517059197343, 
-0.97749720898537218, 0.55460176315663934, -0.51328259591874281], [-0.15014163110959977, 0.64227796852044805, 
0.67829045256375609, -0.53275577483557157]], [[0.69428959767000853, 0.27181575416077908, -0.67540666153511464, 
0.68537741704783373], [0.12635059570264229, 0.25790168831554028, 0.21053467373381252, 0.89625019365122061], 
[0.78533441113905633, 0.38651890253248178, 0.46866980084781207, 0.3966858407832079]]], [[[0.09213358476370237, 
0.32622677154765856, -0.52054474510218407, 0.11292458645612613], [0.60076852133813197, -0.59651506026789658, 
-0.310660307624816, -0.83164083985032011], [-0.11239584868207197, -0.21969260297894544, 0.66253117553533181, 
-0.47549870903007396]], [[0.61793844593902714, 0.049088591742032062, 0.73171791413069731, 0.10324087884362898], 
[0.70630497078056043, -0.39548178252243704, -0.20171780102887515, 0.4486867740825855], [0.3067826397434239, 
-0.69920342074020692, 0.55315007879538758, 0.76955263523522177]]]])+(1.-msk_arg)*numpy.array([[[[-0.98195387674416623, 
-0.71647041666207634, 0.96131746394220374, -0.2970268397734761], [-0.6798614682172941, 0.68011781692825046, 0.9901614342712215, 
-0.51224054060555768], [-0.49126801267038034, -0.2061185705955475, -0.94879172547184121, 0.32864682447432192]], 
[[0.81206407375573542, -0.47431758135493918, 0.35433905577893388, 0.015673991101298634], [0.48161105139319216, 
0.76475757327272964, 0.87767937074390434, -0.061736059585125069], [0.28109110620516442, -0.84666842157492495, 
-0.77512442918187219, 0.20772758208828335]]], [[[0.30772959794939792, -0.2885188792188953, -0.90570783131128163, 
-0.91845913691101799], [0.28634884391824733, -0.3690804749180665, 0.03915017349519978, -0.35281078034679925], 
[-0.086035338721760501, 0.67930968257225288, 0.67498622364084948, 0.93164079695122304]], [[0.68920407959864227, 
0.85164957425061294, -0.61676686381412349, -0.99625404634540149], [-0.79371359058265867, -0.94393926509038906, 
-0.35532528723257628, 0.0015878731596392637], [0.35679339163849177, -0.49580675506446492, 0.91497928940479811, 
0.14614297523964193]]], [[[-0.26376915847654825, -0.4289545189914894, 0.0096817410430425355, -0.4091406948939682], 
[-0.21537036900190487, 0.38987973408241117, -0.12828981462849209, -0.97396371660294467], [0.67689931380382329, 
-0.73005845261718227, -0.32903973852020085, -0.074856999663114099]], [[-0.40422874438770284, 0.95870358605131867, 
-0.048075373669192345, 0.036769265208584478], [0.25203988496541907, -0.62875034340309677, 0.41362004004453712, 
-0.40696408032236886], [0.81933944111362078, -0.68464153016036078, -0.43524094107963629, 0.19643875172460112]]]])
      res=clip(arg,-0.3,0.5)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.47253941060193694, 0.19296787597619924, -0.29999999999999999, -0.29999999999999999], 
[-0.29999999999999999, 0.17426765589991278, 0.35312486983619285, 0.083669184154522291], [-0.29999999999999999, 
-0.15040140084518883, -0.29999999999999999, -0.29999999999999999]], [[-0.29999999999999999, -0.29999999999999999, 0.5, 0.5], 
[-0.11379828349950394, -0.29999999999999999, 0.5, -0.29999999999999999], [0.040955022200906965, 0.5, -0.084301202894394134, 
0.5]]], [[[0.38713164665793709, -0.29999999999999999, -0.29999999999999999, 0.5], [-0.29999999999999999, -0.29999999999999999, 
0.5, -0.29999999999999999], [-0.15014163110959977, 0.5, 0.5, -0.29999999999999999]], [[0.5, 0.27181575416077908, 
-0.29999999999999999, 0.5], [0.12635059570264229, 0.25790168831554028, 0.21053467373381252, 0.5], [0.5, 0.38651890253248178, 
0.46866980084781207, 0.3966858407832079]]], [[[0.09213358476370237, 0.32622677154765856, -0.29999999999999999, 
0.11292458645612613], [0.5, -0.29999999999999999, -0.29999999999999999, -0.29999999999999999], [-0.11239584868207197, 
-0.21969260297894544, 0.5, -0.29999999999999999]], [[0.5, 0.049088591742032062, 0.5, 0.10324087884362898], [0.5, 
-0.29999999999999999, -0.20171780102887515, 0.4486867740825855], [0.3067826397434239, -0.29999999999999999, 0.5, 
0.5]]]])+(1.-msk_ref)*numpy.array([[[[-0.29999999999999999, -0.29999999999999999, 0.5, -0.2970268397734761], 
[-0.29999999999999999, 0.5, 0.5, -0.29999999999999999], [-0.29999999999999999, -0.2061185705955475, -0.29999999999999999, 
0.32864682447432192]], [[0.5, -0.29999999999999999, 0.35433905577893388, 0.015673991101298634], [0.48161105139319216, 0.5, 0.5, 
-0.061736059585125069], [0.28109110620516442, -0.29999999999999999, -0.29999999999999999, 0.20772758208828335]]], 
[[[0.30772959794939792, -0.2885188792188953, -0.29999999999999999, -0.29999999999999999], [0.28634884391824733, 
-0.29999999999999999, 0.03915017349519978, -0.29999999999999999], [-0.086035338721760501, 0.5, 0.5, 0.5]], [[0.5, 0.5, 
-0.29999999999999999, -0.29999999999999999], [-0.29999999999999999, -0.29999999999999999, -0.29999999999999999, 
0.0015878731596392637], [0.35679339163849177, -0.29999999999999999, 0.5, 0.14614297523964193]]], [[[-0.26376915847654825, 
-0.29999999999999999, 0.0096817410430425355, -0.29999999999999999], [-0.21537036900190487, 0.38987973408241117, 
-0.12828981462849209, -0.29999999999999999], [0.5, -0.29999999999999999, -0.29999999999999999, -0.074856999663114099]], 
[[-0.29999999999999999, 0.5, -0.048075373669192345, 0.036769265208584478], [0.25203988496541907, -0.29999999999999999, 
0.41362004004453712, -0.29999999999999999], [0.5, -0.29999999999999999, -0.29999999999999999, 0.19643875172460112]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_trace_array_rank2_offset0(self):
      arg=numpy.array([[0.26555691640939405, -0.18910413427150452, 0.12039085017309703, -0.17574192533077326], 
[-0.34124380166621093, 0.37425207868597998, -0.60605432527492908, -0.78351999526440919], [-0.008718872902747421, 
0.45919281265912248, -0.33985859983016309, 0.29866113913548786], [-0.41729887400737264, -0.28881644097258641, 
0.55984121129488917, 0.70121385950894766]])
      res=trace(arg,0)
      ref=numpy.array(1.0011642547741586)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_trace_array_rank3_offset0(self):
      arg=numpy.array([[[0.98480390026626874, 0.49765498235413208], [-0.29090485844190184, -0.75557539858330203], 
[-0.24147713943580618, 0.83189520938773742], [-0.39332790736632894, -0.23255430958687073], [0.61469056695908431, 
-0.54221652099612139], [-0.30688843565214863, 0.38245185105763024]], [[0.39352505237092705, -0.89859538741148248], 
[-0.19611389561895298, 0.46658392439701291], [-0.58513992115908198, -0.52572413506686666], [0.36528379930027022, 
0.35082214206594764], [-0.36197438572003837, -0.58168855905536532], [0.76595064870067198, -0.29898163768408659]], 
[[-0.21257388708161273, -0.82660354888963417], [-0.95912201085053916, -0.74140025039180379], [0.83542400169616804, 
0.71398525032328397], [0.35580921803956844, 0.9910319975254831], [-0.99833758049102572, 0.49906045783260655], 
[0.46148100303820816, 0.057489858386365444]], [[0.14448723439540556, -0.66497743959494815], [-0.65298722004566812, 
-0.80946208978135714], [-0.72729541818349297, 0.75942335665704896], [-0.22887401674150798, -0.45740224056161138], 
[0.32594979393943646, -0.14983354513871983], [-0.88654135390033595, 0.64170560178214209]], [[0.051660095426996566, 
0.072646454146782835], [0.80107524461478041, 0.75854236897011562], [-0.84218515265322691, 0.5416362111837385], 
[0.72274598019809133, -0.1904262266248522], [-0.70527606629361261, 0.71754633833989923], [0.9944261102138261, 
0.27642826392584841]], [[-0.71311099419488766, 0.096638813026104442], [-0.5881439560482713, 0.48983822134718635], 
[-0.36462007898555537, 0.31258183909269088], [-0.25934166771369438, -0.46384813583935425], [-0.40956274472953136, 
-0.45410860721145374], [-0.63285410497195316, 0.26725460398192613]]])
      res=trace(arg,0)
      ref=numpy.array([0.057109818336410045, 2.2056228588346429])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_trace_array_rank3_offset1(self):
      arg=numpy.array([[[-0.76875725167611453, 0.075414018110247394], [-0.46910058955171152, -0.78870866656070926]], 
[[0.1763412665571984, -0.38625105646597069], [-0.24103707165873289, 0.90747381262632065]], [[0.092043720781569416, 
-0.26846050280345235], [-0.7946162528668701, 0.0066516500151119207]], [[0.25018944598669868, -0.051585233002871211], 
[-0.47027454711245098, -0.25743603279448135]], [[0.89008264050297203, 0.083887666317993048], [0.85233581507994938, 
0.2788350650030873]], [[-0.30761996005339931, -0.4926586222583984], [0.69525691978757131, 0.88245916957253989]]])
      res=trace(arg,1)
      ref=numpy.array([-1.5574659182368238, 1.083815079183519, 0.098695370796681337, -0.0072465868077826734, 
1.1689177055060593, 0.57483920951914058])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_trace_array_rank4_offset0(self):
      arg=numpy.array([[[[-0.35914276251844779, -0.89177244154479451, 0.75156232311996729, 0.2689197071295113], 
[0.48550638042599714, 0.37876012371739165, 0.11909831579647689, -0.13698649685115294], [0.66469543775731199, 
-0.04713994250240372, 0.69016847677268345, 0.43203165345044803]], [[-0.84196908557528283, 0.31114408094237334, 
0.97829773043316881, 0.71705939156379173], [-0.59254857582824449, 0.99125198550959226, -0.27824457105332501, 
0.95639865727128393], [-0.88869987019695151, 0.71945513418541562, -0.63948935534342555, -0.60595079857487888]], 
[[0.20060003292491246, -0.51895689477941298, 0.77610596136013754, 0.89846794711535649], [-0.797787885912846, 
0.1126603408303033, 0.71040396312258625, -0.58371207496516675], [-0.39629389058593234, -0.224661339329151, -0.4415492883796388, 
-0.2843763466047633]]], [[[-0.14648357755460073, 0.10563951301739172, 0.074504969752375638, -0.11173828803978103], 
[-0.54814938162159921, -0.030096820913388944, 0.01245902120734721, -0.93883615158053946], [0.23918339542047629, 
-0.58141478071359765, -0.47847130713180896, -0.3854827811199637]], [[0.031675817270823137, -0.28496738108976349, 
0.75670930591992125, -0.78671398093094447], [0.3632307462902622, 0.086431371878165075, -0.73497382201547867, 
0.41464504948755421], [-0.100681389463116, 0.10486835825917429, -0.51283334572425865, -0.10649277312087135]], 
[[0.64697608810822471, -0.4521370777632252, -0.5134303970348042, 0.87851641270408254], [-0.54474768357062953, 
0.62303825302672777, -0.80583800146399409, 0.34594811232501099], [-0.31036396242882325, -0.56242948188853092, 
-0.22336487915091019, -0.39223978983887853]]], [[[-0.27938145518702884, 0.77804736964504406, -0.011111792974861645, 
-0.20087387832153847], [-0.31688601479435285, -0.82880241287501444, -0.5794350312586245, -0.92630428602849646], 
[-0.88496830540835147, -0.083138586748798371, -0.95091261792702308, 0.31621428161145904]], [[0.17190403532611431, 
-0.55658920241117382, 0.68690100106161078, -0.31096726230017713], [0.70230740428853178, -0.74569911664915844, 
-0.45156846173579446, -0.12422693360460091], [-0.98904485448914325, 0.57482038825465653, -0.46254001816981405, 
-0.82707533787151544]], [[0.70924494286527739, 0.050750820630569571, 0.7836413546922556, 0.32035567617899541], 
[0.76832143595594471, -0.36561424057317549, -0.26942425383212854, 0.8169272968127963], [0.73410520671439561, 
0.53990562381492802, -0.75655688584516367, 0.20038872707677813]]]])
      res=trace(arg,0)
      ref=numpy.array([[0.38177799761765274, -1.1259890020039884, 2.2919129837321441, -0.19743859762243776], 
[1.617058562672204, 0.099577255022381239, -0.88529976005113031, 1.0945858494491976], [1.2981192550085916, 0.59763403957169858, 
-0.57922175479673887, 0.5259276074063548]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_trace_array_rank4_offset1(self):
      arg=numpy.array([[[[-0.92784167492855762, 0.60248011777483357, 0.031225045884133662, -0.34745967830316826], 
[0.027129935696126095, 0.48013540111896069, -0.72517396078746899, 0.95371795346554977]], [[-0.72505453089243255, 
0.007597625514423223, -0.98625885351348153, 0.1682118209020349], [0.64064186297081394, 0.71427245750801327, 
0.92923384113278851, -0.95941382390217966]]], [[[-0.028684917730743331, 0.090779113645879095, 0.63429381954611386, 
0.10940719084580586], [-0.067228410948499295, -0.45721418427291693, -0.47873249885715663, 0.21835729813061455]], 
[[0.51965061080506958, 0.75785664872937386, -0.33550922445148745, 0.66896630346996044], [0.7176636344292453, 
0.56712006361198353, -0.0048458303867575658, 0.17607306572513104]]], [[[-0.76702410971539137, -0.14274699317299167, 
0.28278039945268163, -0.7080275942808707], [-0.20539126224132875, -0.9172867821862789, -0.49799499531251512, 
0.092624073961727005]], [[-0.59138766627665462, -0.46807518914541313, -0.9104444200453814, 0.11708065088439268], 
[-0.4668904319703393, -0.32397199958420608, 0.059672712734865518, 0.99088767923010379]]]])
      res=trace(arg,1)
      ref=numpy.array([[-0.28719981195774369, 1.3167525752828468, 0.96045888701692217, -1.3068735022053479], 
[0.68897871669850197, 0.65789917725786262, 0.62944798915935629, 0.2854802565709369], [-1.2339145416857307, 
-0.46671899275719775, 0.34245311218754715, 0.28286008494923309]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_trace_array_rank4_offset2(self):
      arg=numpy.array([[[[0.13031826439397065, -0.1789298694401773, -0.64989661091089923], [-0.27327904713332241, 
-0.61679868125863613, 0.18686962619451131], [0.66553888328001665, -0.99338065217851401, 0.70085830978616381]], 
[[0.41407307674031446, 0.70389105425033516, -0.85751604971482887], [0.60932599055523418, -0.027942855299541769, 
0.42976951506702132], [-0.45560078263789872, 0.87711330379085095, 0.35291250996606194]]], [[[0.0085795831135988543, 
0.18775995060321127, 0.92382753051777255], [0.2329781632292165, 0.19209684057591137, 0.76180170447446627], 
[-0.22107729339810578, -0.063955483745411046, -0.4596195798551701]], [[-0.87062584076293215, -0.97712503659797512, 
-0.36609561443173488], [0.025990661451154473, 0.13804237670844399, -0.3575230803781464], [-0.089217826694265812, 
0.33387178666740835, -0.016589404783220019]]], [[[-0.52969301393409496, 0.99628743319515278, -0.68831285828334776], 
[0.17295713720552652, -0.10633816939243634, -0.75799480026623534], [0.049894479253417279, 0.97106513893516566, 
0.19808838402211992]], [[-0.57109506688962441, -0.21527086508290272, -0.81800681445834766], [0.53496737929008442, 
-0.43086860643255731, -0.93046040310632749], [0.05573925281031844, 0.63980368481088545, -0.10083869982683513]]]])
      res=trace(arg,2)
      ref=numpy.array([[0.21437789292149834, 0.73904273140683463], [-0.25894315616565988, -0.74917286883770817], 
[-0.43794279930441138, -1.1028023731490169]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_trace_constData_rank2_offset0(self):
      arg=Data(numpy.array([[-0.29383963344234121, -0.015014018118957306, 0.99753580206723647, -0.50635236021974106], 
[-0.37905392076965927, 0.31528456608530631, 0.86721682590868698, 0.51278396705456841], [0.29008438489433841, 
0.10101024328890973, 0.66687277847571313, 0.6654399862684186], [-0.5902980567859355, -0.57191420388096859, 
-0.51058587062074134, 0.93677538124606463]]),self.functionspace)
      res=trace(arg,0)
      ref=Data(1.6250930923647429,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_trace_constData_rank3_offset0(self):
      arg=Data(numpy.array([[[-0.72642799948220493, 0.79232473269996073], [-0.67563422655155247, 0.20011212614972362], 
[-0.59028571211858649, 0.14506117798083507], [0.47240012720854407, -0.82029555113272101], [-0.58454362400865878, 
0.63215213979312868], [-0.35007656466177095, -0.1470415518683037]], [[0.42763460915772211, 0.8357573230966695], 
[-0.29914137624741732, -0.036306874859310279], [-0.51046716828256211, -0.85831582073872537], [-0.79470878322572314, 
-0.48987591183092638], [-0.86974326209750896, 0.70916592312129745], [0.83868372787049505, -0.10785599577865157]], 
[[-0.058672803278049468, -0.14146375774695574], [0.17727056643394556, -0.48058347796639467], [-0.83011490290698431, 
-0.63504062113632753], [0.086833789382744442, 0.70068497275513919], [0.49044532362179361, 0.40352559278906575], 
[-0.88547213401229574, 0.87226874572961344]], [[-0.33114461927211525, 0.0027119570097013934], [-0.37590722158855527, 
0.2213145859812935], [0.80071873391378312, -0.41643470039851649], [-0.58961378449755419, 0.93123715794003958], 
[0.02582176095827271, -0.53120064845162362], [-0.38118292706842039, 0.98328185978424587]], [[0.53642120847771513, 
-0.26939855882552544], [-0.22456707784987806, -0.22452517341877654], [0.89817688350629865, -0.63775048322698091], 
[-0.76799424555480655, 0.92000058433888654], [0.36620149364765475, 0.34379452633744956], [-0.59690996179249978, 
-0.59279936908563813]], [[-0.59055334860005648, 0.30644018478641044], [-0.25015804641416217, -0.49250326577772507], 
[-0.12727433609341832, -0.35398126798225626], [-0.56502814650267053, -0.96640875351734778], [-0.51485842473096888, 
-0.31753865090662758], [-0.63020882370649534, 0.61645627794582958]]]),self.functionspace)
      res=trace(arg,0)
      ref=Data(numpy.array([-2.7093053931930013, 2.0124651989276416]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_trace_constData_rank3_offset1(self):
      arg=Data(numpy.array([[[0.27352469692010906, 0.47078446143689212], [0.57479782614850539, -0.0051486191501395329]], 
[[0.74361396403981384, 0.25515510619301329], [-0.61255554527004286, -0.46948770334026335]], [[-0.028715409681372206, 
0.070901178629972694], [-0.47846874064507117, -0.91959320136358702]], [[-0.24835828088751577, 0.23671885472681842], 
[0.99388364258199013, -0.23721426553099789]], [[0.19787821531251604, -0.77414401933415267], [-0.6848568761717686, 
-0.22644337787735802]], [[-0.78546754720422962, -0.65168742945227121], [-0.26014163072111196, 
-0.26194458519933228]]]),self.functionspace)
      res=trace(arg,1)
      ref=Data(numpy.array([0.26837607776996952, 0.27412626069955048, -0.94830861104495923, -0.48557254641851366, 
-0.028565162564841984, -1.0474121324035619]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_trace_constData_rank4_offset0(self):
      arg=Data(numpy.array([[[[0.67837901384728561, 0.25230684578079843, -0.18006172651018115, -0.18742559018056171], 
[0.25980554035178893, -0.60710586338433759, 0.18529181621885771, 0.06454562228317684], [-0.89720749423697344, 
-0.809936419954433, -0.52546885113780628, -0.65826181914830206]], [[0.75126603048955443, 0.41791482287477688, 
-0.18708594285812397, -0.097592007674713432], [-0.30556054895669615, -0.25367425753154871, -0.65986245393338994, 
0.49830950398371265], [-0.74835162221267404, -0.98775159055059558, 0.53343998083071287, 0.58249266652470078]], 
[[0.72661613097074818, 0.22851198720172761, -0.38897158619501715, 0.15133995066275441], [0.76231929070207505, 
-0.87183861352936654, -0.9162704267412165, -0.075190375042774704], [-0.99789286156015344, 0.41046263149263118, 
-0.72945251192102378, -0.59631547494176385]]], [[[-0.27663864530792637, -0.91812683979135334, 0.5568519562913512, 
-0.31904133788107036], [-0.48634727640176045, 0.87978759058913414, -0.4631455206362578, -0.35771439543824557], 
[-0.0088863660929114907, 0.67411887052330854, 0.43694758100285558, 0.13263535729230225]], [[-0.047911420964740215, 
-0.1671659731804469, -0.79106159933094444, -0.60086616171961471], [-0.40660682701782869, -0.65857886957039002, 
0.97138661007671434, -0.38521124487194713], [-0.99504601002533355, 0.038715505103106729, 0.27375528366649449, 
0.51179190061643776]], [[0.17226508653462647, 0.41821768543572513, 0.99025384379339765, 0.42934653741475604], 
[0.94505216727849306, -0.90852488045982582, 0.99963241021520122, 0.32009293199423694], [0.64969595277507719, 
-0.81807980261861268, 0.63135375645829073, 0.31133912070438985]]], [[[-0.81843681081377451, 0.62297813580429251, 
-0.64834999047301412, 0.31879083594375723], [0.5745557058624462, -0.31209122247715793, -0.91449671557843981, 
-0.28521064904747151], [-0.11835330152389645, 0.23564571048634719, 0.21055665107048438, -0.84463421709998987]], 
[[-0.16113935663618451, -0.79196713229468929, -0.51417934429324008, 0.54945395167798505], [0.76643759834273495, 
-0.47795914006001694, 0.43614611442999918, -0.17337837227416975], [0.50429854624245429, -0.91686495814729718, 
-0.73068569106157333, 0.70282293932303208]], [[0.24655393516994573, -0.36508446162486963, -0.18165603004763731, 
-0.60679606785622919], [0.62995050624366744, 0.50604553586174239, 0.2139378361831199, 0.033810482605382219], 
[-0.21127570703083398, 0.61056026945659569, -0.94937237529332541, -0.64919134337137296]]]]),self.functionspace)
      res=trace(arg,0)
      ref=Data(numpy.array([[0.87702152805249112, -0.2799435890245181, -1.1527793558887629, -1.3950878197564056], 
[0.48314921957762769, -0.75963919709298522, 1.3706162624786919, -0.28685513998338807], [-2.1035292112931412, 
-0.16066064539473057, -1.2010859427646372, -0.79566126190323727]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_trace_constData_rank4_offset1(self):
      arg=Data(numpy.array([[[[-0.16696076135170923, 0.82974391265868208, -0.078680670490009375, -0.96835524907764126], 
[0.54969239234040534, 0.17623541766186968, 0.077019704254756016, 0.57636452671928362]], [[-0.4519311176106926, 
0.57880525532235882, -0.0015113586612311991, 0.31438699743750309], [-0.46119335345094092, 0.59471868696549524, 
0.64818472484469392, -0.7434166956705377]]], [[[-0.3104928944593035, 0.89709092416386893, 0.3374479306116227, 
-0.56706034715284614], [0.52964596335340208, 0.062265529064360114, -0.66426921214265189, 0.0099179068655548264]], 
[[0.0042797304397270697, -0.68728374808571213, 0.20538696492502284, -0.82057659071817879], [-0.12466108726578962, 
0.052484065740935115, -0.28252863583893451, 0.42076514415105537]]], [[[-0.65642721290351291, 0.72607023613257682, 
-0.075581205318460798, 0.76265280933396595], [0.9863819936467153, 0.46861308614825292, 0.25011512506209099, 
-0.088132594048707213]], [[0.12851311890909933, -0.26566989109689865, 0.44009914867510891, -0.30062749250691967], 
[0.89182785391555042, -0.54420395388411191, -0.9018886462510296, 0.95827128488931823]]]]),self.functionspace)
      res=trace(arg,1)
      ref=Data(numpy.array([[-0.62815411480265015, 1.4244625996241773, 0.56950405435468454, -1.711771944748179], 
[-0.43515398172509312, 0.94957498990480405, 0.054919294772688199, -0.14629520300179077], [0.2354006410120375, 
0.18186628224846491, -0.9774698515694904, 1.7209240942232842]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_trace_constData_rank4_offset2(self):
      arg=Data(numpy.array([[[[-0.98482921887938679, -0.2021932043246204, -0.43931510971992593], [0.82500558882646424, 
-0.94073688978501102, -0.85123670983501043], [0.31094599072729778, 0.12059062935396159, 0.45095349007500651]], 
[[0.10491489308253454, -0.87224687812665835, 0.85946921554980338], [0.20110899808432126, -0.04819131401865584, 
-0.65225345772055876], [0.43197594848373377, -0.8665825801549325, 0.68414135494343808]]], [[[0.82421905238462712, 
0.05713259985882635, 0.88279598225821787], [-0.2098145249261274, 0.022164492617607445, -0.54619059245669699], 
[0.0062526761224697136, -0.13250141161679396, 0.85642390101957422]], [[0.88057912670407634, -0.059756350737471209, 
0.35828573888292059], [0.49077984989887513, 0.8028691375260737, -0.93700752416276556], [0.030934159674997685, 
-0.60451701161112292, -0.65521990807624797]]], [[[0.0070660955019832272, 0.87011155990448241, -0.32286449245280857], 
[-0.24887688324443102, -0.18727119138654746, -0.90048655101564057], [0.30874988433738726, -0.69579883453454605, 
0.78507788657918276]], [[0.18716153278559133, 0.20495393697585418, -0.42114034048974114], [-0.72877450302661506, 
0.57308411806594006, 0.70745122427680229], [0.13182579049068832, -0.83070067184375551, 
-0.22593153722147408]]]]),self.functionspace)
      res=trace(arg,2)
      ref=Data(numpy.array([[-1.4746126185893913, 0.74086493400731679], [1.7028074460218088, 1.0282283561539021], 
[0.60487279069461852, 0.53431411363005732]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_trace_expandedData_rank2_offset0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[0.11566446254344553, -0.4745105734645696, -0.87728089123849706, -0.30742134665898813], 
[0.64561603368593645, -0.16643615643478848, -0.7068536821797744, -0.033045748559738408], [0.32071043030768398, 
0.22106761892996096, 0.48388048184729837, 0.74228108665057824], [0.41235289044686274, -0.80515202915608386, 
-0.50750243672304451, 0.19620812039387592]])+(1.-msk_arg)*numpy.array([[0.6982051620420755, -0.43997510892232849, 
0.31785539449789679, 0.77071019636847571], [0.25437948245168607, 0.87309054348917914, 0.40806383784241795, 
0.23697269987364522], [0.76503085553268702, 0.54594905359750445, -0.63787892882701391, -0.43923459263360387], 
[-0.22326896676087116, -0.68233388681771623, 0.44544487240308017, -0.13913483142044614]])
      res=trace(arg,0)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array(0.62931690834983134)+(1.-msk_ref)*numpy.array(0.79428194528379459)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_trace_expandedData_rank3_offset0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[-0.66027506839281647, 0.11166612285841016], [0.42749286897655625, -0.93865817184981437], 
[0.15068940524255603, 0.52349452094956739], [0.27687559884096458, 0.066986941961115765], [0.93180835331970946, 
-0.48756585147447229], [0.65280839067465135, -0.71921262071024761]], [[0.11618442368982751, -0.4763658244658322], 
[-0.50020128416710752, -0.25851957859869312], [0.35878292067616147, -0.38038146181290644], [0.93041110099841817, 
0.029846232912715598], [-0.45141831725121584, 0.28948945003451643], [0.32280124045787928, 0.053742832260912587]], 
[[-0.98592436891189683, 0.71413775868480056], [0.046660357647682282, 0.98162830306990978], [0.58933281142936988, 
0.22717987654775929], [0.71293710502335106, 0.73508206251418873], [-0.61306596901927679, -0.63773297908928717], 
[0.66040111118536293, -0.87407302149827326]], [[-0.67220137400111146, 0.9753868329755222], [0.37836258737909279, 
-0.76957121571717235], [-0.50213275436171911, -0.070066516163526771], [-0.58706972621765052, 0.77480496092018991], 
[0.17595133543567765, -0.67397634261638495], [-0.88890284486005799, -0.69276478680832065]], [[0.93447809528654435, 
0.21446672025978786], [-0.018401823547098628, -0.31000980913085052], [0.5413158316784219, -0.058395872945901983], 
[-0.54734531427605515, 0.28202816017233179], [-0.69286581846548523, 0.82432638767569344], [-0.87062165825132021, 
-0.93175755696284512]], [[-0.65710236849233516, -0.62035012520073329], [-0.50968063974239408, -0.93313959756872777], 
[-0.35391354855647905, 0.75699472909905396], [0.11178875277715772, 0.072002604708961249], [-0.83779828776596932, 
-0.90850093178213798], [-0.68468230379800277, -0.99110190358689487]]])+(1.-msk_arg)*numpy.array([[[-0.99848839209688167, 
-0.644631003682262], [-0.48778615498860289, 0.30351281670072061], [0.35476651624916067, 0.65677420217236393], 
[0.50513285317507672, 0.38249927932844585], [0.37184137033045817, -0.014062623855171763], [0.20346283701971224, 
-0.90065418094786609]], [[-0.44711449100012057, 0.06097628012933809], [0.80356090400302205, 0.27132926350242892], 
[0.98253752709557163, -0.59382185199137427], [-0.25845751522938087, -0.38353350312948487], [-0.33300811027660804, 
-0.93105133171802001], [-0.26330496922471625, -0.50428263033869603]], [[-0.81751144420930899, -0.30764780803870528], 
[-0.78647633040943976, -0.91372171625429766], [0.20200739094322584, -0.21172544950311645], [0.9090164133708083, 
0.63061458792814884], [-0.95363552021907294, -0.47103249447229723], [0.84223778501992297, 0.86715821546071359]], 
[[0.15707485130940202, -0.2899064141846559], [-0.84537923972525575, 0.015976015150186429], [0.74386525719705388, 
0.12827716540476786], [-0.48285012973111252, 0.66655440178479153], [0.35341753421674826, 0.96292028306601618], 
[0.90070502993119717, -0.45105023721624105]], [[0.28873136026507829, -0.080104489353869379], [0.063967147741553321, 
0.33578030011784255], [0.5987855411248364, 0.71129093318111458], [-0.2770293252781042, -0.69711328956064111], 
[-0.66489886266023213, -0.64595981816454051], [-0.96026557406741264, -0.78155062841685208]], [[0.55660561073469239, 
-0.54703296386818789], [0.19275274446148116, 0.29276600726776647], [0.012541645015802017, -0.30750896443491094], 
[0.18531754942814893, -0.19424254722285217], [0.36728317133983568, -0.86908257258996935], [0.6986276697478373, 
0.35789358454072695]]])
      res=trace(arg,0)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-2.5357613896116926, 0.6883558658164648])+(1.-msk_ref)*numpy.array([-0.44204141979414113, 
-0.20653902152197157])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_trace_expandedData_rank3_offset1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[0.54832323387675541, -0.8708353613360531], [-0.61626047688459518, -0.49478488575159019]], 
[[-0.83629746484611744, 0.47083140193399564], [-0.23408945961247651, 0.039929220485188655]], [[0.75625307461269897, 
0.14019030241332509], [-0.27268120332088674, -0.72257006486353137]], [[-0.55826466629659932, 0.73521559094994182], 
[0.91050139917670103, 0.24001369605640965]], [[0.25393441044315046, -0.90266415386637022], [0.633218691079662, 
-0.6608559941901635]], [[0.59063791740460214, -0.6151947887528777], [-0.11030206731241865, 
0.87604601489174794]]])+(1.-msk_arg)*numpy.array([[[-0.59125906679462714, -0.70711124409167958], [0.27856680992337113, 
-0.086397184806098037]], [[-0.86220968392839969, 0.26537880004255987], [0.93545519643136954, -0.48749383545450575]], 
[[0.06022485624330054, -0.15164028816670005], [-0.99602031320916007, -0.96448449281376236]], [[-0.57562468885884877, 
-0.44453327158851774], [-0.021996078817393627, -0.60252338148303708]], [[-0.043233633786327053, 0.82352268170410592], 
[0.74140809698310162, 0.89351886922223023]], [[0.5483633681991571, 0.85381405316771697], [-0.85793760080248904, 
0.12725561555427167]]])
      res=trace(arg,1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.053538348125165225, -0.79636824436092879, 0.033683009749167603, -0.31825097024018967, 
-0.40692158374701304, 1.4666839322963501])+(1.-msk_ref)*numpy.array([-0.67765625160072518, -1.3497035193829054, 
-0.90425963657046182, -1.1781480703418858, 0.85028523543590318, 0.67561898375342877])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_trace_expandedData_rank4_offset0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[0.80505741203337466, -0.80734097237269142, 0.71509731690044021, -0.34102290030005511], 
[-0.27163627689262082, 0.96803797983982598, 0.39024478352616399, 0.45615965347192255], [-0.2492201152095701, 
-0.53890105212209205, -0.26097899477118602, 0.098974826517917558]], [[0.4109524481147584, 0.32745334361464762, 
0.88135376430664136, -0.88007501076892969], [-0.2546833692939714, -0.62753769519110136, 0.2296060741039474, 
0.57476321487434467], [-0.50883771898767782, 0.0015685044689357497, -0.66998456995491273, 0.34968106764759765]], 
[[-0.64126116758172813, -0.55568939471539136, 0.23227193794020073, 0.7330369779999284], [-0.3126458847194391, 
0.94502653499662759, -0.5936869592079892, 0.25705352992342645], [-0.46308398732466771, 0.37874287044184518, 
-0.030587221468148629, -0.80649056741936587]]], [[[-0.55428203289059597, 0.39384118967627457, 0.21400763473662332, 
-0.05778739181646686], [0.89572023189653716, 0.64808964944115921, -0.76081286684220872, 0.29643887696780724], 
[0.92808629225161865, -0.5927287532507779, 0.080932430903517538, 0.6300942109115113]], [[-0.050786057311087207, 
0.82696470921887877, -0.52069224695178296, -0.19457105492027638], [0.20464920722411661, -0.5846772435204266, 
0.15252929823896588, -0.40702515404436856], [0.30048751596882983, -0.77039288688309648, -0.25964577407800782, 
-0.92201363813026638]], [[0.30524909950231782, -0.078384011294031719, 0.54945703430160053, -0.79337517722753814], 
[0.23862358165502573, -0.21132833792836037, 0.67284298273913512, 0.47737327757131043], [-0.57895539269247887, 
0.30865689803953855, 0.64471920728490284, 0.56866096983603387]]], [[[0.59411993702838983, -0.55286609932973252, 
-0.11704951317866552, -0.1799876691786706], [0.7044831381893395, 0.41444296685797211, 0.035873800421900803, 
-0.3077870122658044], [-0.5537875848354139, -0.09294785912808079, 0.60281458905112695, -0.25586177426534462]], 
[[0.073972334874764956, -0.17863943805405702, -0.40192876955035106, 0.49779070006291537], [-0.78351344470846263, 
0.066371129010451391, -0.68107079280677119, 0.27027467137931782], [-0.08733315269502917, 0.17609544147485212, 
-0.84985580086147139, -0.57706563708906877]], [[0.80900077345628585, 0.54299611298654993, 0.36197457335708672, 
0.78822507164521771], [-0.57823417843160185, 0.24045072531447076, -0.21646907149363126, 0.39109919937326909], 
[0.86048045599481848, 0.5316575326401245, 0.17357533828519278, 
-0.035528536537402955]]]])+(1.-msk_arg)*numpy.array([[[[0.70941815199826297, -0.11292617215886147, 0.77501725501558871, 
0.55963189951353853], [-0.7796391490563328, -0.73773608726012929, -0.97005387036695923, -0.89110745459054397], 
[0.61643890466549855, 0.36305011268742771, -0.35899006379107701, -0.89037252955545054]], [[-0.48513893441395584, 
-0.80463983453322796, -0.26992613069089511, 0.25377671510133326], [0.85296361003701948, -0.088353279464719492, 
-0.8833491655162129, -0.49061051639006448], [-0.0070639306921898015, -0.91051044523499236, 0.19421401294033869, 
-0.023694190336753351]], [[0.87600763033159823, -0.12842080556081559, 0.23748702101368124, -0.42712865548684986], 
[0.12798880486722353, -0.68743327370801688, 0.066429185783382527, 0.89836542344770587], [-0.96008406560328807, 
0.69958585815823149, -0.96646148853866998, -0.19840985556590041]]], [[[-0.50676183913854866, -0.95802357084856005, 
0.88072647796789671, 0.87404328671267173], [0.91057125659389504, 0.28205294652962043, -0.84518566395353356, 
0.43476689284070802], [0.72919748914970839, 0.16300515512838709, 0.69548292805402645, -0.40221177456763235]], 
[[0.015376288615946532, -0.65336549560515977, -0.013429002513120514, 0.92410715356068551], [0.93234891088019034, 
-0.25180012657781981, -0.85520698120760597, 0.030845786152130295], [0.2883277702478666, 0.98831889817695329, 
-0.18710978196618666, 0.63191491053115234]], [[-0.35727830132105409, -0.38880115020070694, -0.41377801087774246, 
0.26046673344155802], [0.46430510016515281, -0.37898634296018896, 0.80517984749417848, -0.89342861976109855], 
[0.39122847915190984, 0.839544668081603, 0.032449290839240641, 0.19743019769729941]]], [[[-0.40460228252750641, 
0.48245356602434275, -0.39051899259424205, -0.18353327424101074], [0.97419082853471006, -0.042098069664895599, 
0.89968753511941424, -0.42150619030266823], [0.7086498513346402, -0.55571035848915984, 0.5581268170564333, 
0.35555965546369439]], [[0.70102153566399905, -0.00085601991690098522, -0.003295266014190501, -0.044650933649731828], 
[-0.69526860140456903, -0.20761407547243449, 0.93251889119089992, -0.24412708149011197], [0.079410921553375147, 
-0.74409430021596035, 0.10338967121074294, 0.96093957200944913]], [[-0.2191453313682481, 0.60828504356886492, 
0.83999448435024604, -0.28751114677396172], [0.49008990050737888, -0.00067270376026429268, -0.37727354733847673, 
0.22111819218371975], [-0.21174513129074191, 0.33959243779929893, 0.73167011601901133, -0.084743053268589774]]]])
      res=trace(arg,0)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[1.5632721281785733, 0.56261984983273727, 0.55637964330574396, 0.25263111642488623], 
[-0.64522124810010606, 0.62381146163387013, 0.32630501027149861, 0.44023369880082308], [0.91174785675407821, 
-0.77763640636506404, -0.34704943056400106, -0.85856734814975177]])+(1.-msk_ref)*numpy.array([[0.5056491092459614, 
-0.15800662419515632, 1.6015827368527142, 1.1962279063002623], [0.64279966233123642, -0.99020891759821339, -2.2025343989130421, 
-0.63914347625469392], [0.69302154362262325, 1.6909614486636799, 0.18557027026174766, -0.34320067229288798]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_trace_expandedData_rank4_offset1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[0.08984283572763152, 0.55774397308602075, 0.87551033839309578, 0.99597647119622823], 
[0.42229013451029207, -0.69334370113115673, 0.24000324515885452, 0.42734821657622946]], [[-0.81493193917641937, 
-0.050594548082001589, 0.0038864544826278102, 0.052096885283608341], [0.90250995205525242, 0.70852431792468717, 
-0.51329731427030234, -0.35733423128402686]]], [[[0.40608243871683025, -0.23521357565080536, 0.89730267936680086, 
0.047309735490747906], [-0.59393261956767396, -0.29550705351553352, 0.30755167993031152, -0.39461060352402422]], 
[[0.53278112038096181, -0.18809691653542338, 0.31331369186508473, -0.072624780731609606], [0.22822350853600937, 
-0.77304798948100206, -0.92290799290383396, -0.77844654084572884]]], [[[0.52913787058447759, -0.0060496727934438166, 
-0.11597929750770342, 0.34450138717310375], [0.022084213363977367, 0.65407578801966126, -0.75425196792782034, 
-0.75722973429394558]], [[0.98740067425091804, -0.67549616735140949, 0.62950749508587323, -0.88707486447485584], 
[0.060245192990180829, 0.57744779530191237, 0.37664105332815967, 
0.88854593410198413]]]])+(1.-msk_arg)*numpy.array([[[[0.47417067087989362, 0.67736115892005633, -0.47715348159010551, 
-0.22133534252199838], [-0.65637448271923016, 0.96209293069951607, -0.36528071036921772, -0.48073633550727246]], 
[[-0.96698553584714331, -0.32663464055897684, -0.82564205136999669, 0.97180689930200814], [0.28291320059607195, 
-0.88357282368832246, 0.71368496471764575, 0.16389798613013395]]], [[[-0.46081508440668251, 0.37873602667271622, 
0.60832335684486516, -0.84502188519038723], [0.19274895184009821, 0.42741773658053939, 0.30785853018719433, 
0.99619445060780953]], [[0.74277555386059979, -0.49774986195020943, 0.48334133283298297, 0.75890613926618067], 
[-0.82416568056271866, 0.36258558577355227, 0.062026544439186537, -0.11407159191585503]]], [[[-0.36297821802040353, 
-0.65847019250544503, 0.60479755659167767, 0.32850256319429771], [-0.31926573823318805, 0.76520370780791147, 
0.23701043523043985, -0.84107360755418736]], [[-0.94264543949346069, -0.91307915369911474, -0.44258244810340419, 
-0.075654165697478071], [0.089419047834058185, -0.71629556686014295, 0.17553339506971932, 0.57290908432214582]]]])
      res=trace(arg,1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.99235278778288394, 1.2662682910107079, 0.36221302412279344, 0.63864223991220137], 
[0.63430594725283962, -1.0082615651318074, -0.025605313537033103, -0.73113680535498093], [0.58938306357465842, 
0.57139812250846855, 0.26066175582045625, 1.2330473212750879]])+(1.-msk_ref)*numpy.array([[0.75708387147596556, 
-0.20621166476826613, 0.23653148312754024, -0.057437356391864425], [-1.2849807649694012, 0.74132161244626849, 
0.6703499012840517, -0.95909347710624226], [-0.27355917018634535, -1.374765759365588, 0.78033095166139699, 
0.90141164751644354]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_trace_expandedData_rank4_offset2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[-0.66853129559502222, 0.291820898682152, 0.28763294566928921], [0.90955604295816017, 
0.15689918504537581, -0.84974098181736846], [-0.11864531145144963, 0.62639318585116577, 0.28878897761901667]], 
[[-0.81891488095094211, 0.087816450656737821, -0.61977148230317702], [-0.25672822143041052, 0.850129776941005, 
0.88131009407039174], [0.68042928359428401, -0.22600045838684846, 0.29289573548364345]]], [[[0.86947162304463177, 
0.83714818463465979, 0.48073227883036274], [0.74537740000749331, -0.724938999271586, -0.24964341400235268], 
[0.96015316466387812, -0.77803078685663474, -0.40930971981379272]], [[0.50513454444497397, 0.27333552102189596, 
-0.010414057530881538], [-0.56187171953000825, -0.44137302845768289, -0.29244303378144232], [-0.1881407405943325, 
0.4373804442340905, 0.079976247887080687]]], [[[-0.03923293421052132, -0.83834889486007658, -0.25537540031705741], 
[0.85687986615403067, 0.62632041906338909, -0.28611872462260002], [-0.080621198164744223, 0.92763342316339847, 
-0.96379876567397837]], [[-0.51229650531528792, 0.61168109929691394, -0.48562540039833846], [0.49244617296496918, 
-0.34501205541607938, -0.43139933921747198], [0.97335832979343806, 0.90952427744807229, 
-0.79476366666508969]]]])+(1.-msk_arg)*numpy.array([[[[-0.9049127988617065, -0.77513691833851217, -0.37363048230956908], 
[-0.19332122397693419, -0.76584280953452466, -0.36465847067381119], [-0.93936927771468182, -0.9052541144775863, 
-0.44538405341396325]], [[-0.91309628219582017, -0.87504920853740842, -0.6919306085080319], [0.7357120775715762, 
-0.81739945020223725, -0.2499946214554134], [-0.069828197774766787, 0.97127065995603989, 0.64641336042534037]]], 
[[[0.98354181283512387, 0.26145468296473728, 0.67122148396418746], [-0.85594120625088976, -0.90242375185857582, 
0.96277097499514519], [0.08105375151436145, 0.14214289843261119, -0.18712602171243975]], [[0.5561891002327306, 
-0.11458725485012078, 0.70506410495446925], [0.17668410374153187, 0.72063557456429894, -0.64688343152063466], 
[-0.50952512796965399, -0.097278116167972328, 0.41947753973499124]]], [[[0.47279413902758161, 0.81935636228453546, 
0.87589001940363165], [0.9561752822027425, -0.6004911198517342, -0.70023577057274666], [0.86945177086287462, 
0.53351985326887874, -0.89239035872981298]], [[-0.98966175755131802, -0.19919768852186159, -0.81359664155341171], 
[-0.28175871977250644, -0.059790145158654218, -0.0089004393110567737], [0.77373148023876048, -0.2581197388152523, 
0.89448508702586982]]]])
      res=trace(arg,2)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-0.22284313293062974, 0.32411063147370633], [-0.26477709604074695, 0.14373776387437176], 
[-0.37671128082111061, -1.652072227396457]])+(1.-msk_ref)*numpy.array([[-2.1161396618101946, -1.0840823719727171], 
[-0.10600796073589169, 1.6963022145320208], [-1.0200873395539656, -0.15496681568410242]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inverse_array_dim1(self):
      arg=numpy.array([[1.3914367298126584]])
      res=inverse(arg)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(1, 1),"wrong shape of result.")
      self.assertTrue(Lsup(matrix_mult(res,arg)-kronecker(1))<=self.RES_TOL,"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inverse_array_dim2(self):
      arg=numpy.array([[2.9418052544261108, 0.53874493129953072], [0.46782111529042547, 1.9101073069970917]])
      res=inverse(arg)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(matrix_mult(res,arg)-kronecker(2))<=self.RES_TOL,"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inverse_array_dim3(self):
      arg=numpy.array([[1.6020395384987041, -0.87670859867007334, -0.98070275415465935], [-0.8409158097058016, 
2.8045205591389903, 0.88187627481882669], [-0.45001235350647861, 0.11036032836309251, 1.8806880610894336]])
      res=inverse(arg)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 3),"wrong shape of result.")
      self.assertTrue(Lsup(matrix_mult(res,arg)-kronecker(3))<=self.RES_TOL,"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inverse_constData_dim1(self):
      arg=Data(numpy.array([[1.4764835684284772]]),self.functionspace)
      res=inverse(arg)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(1, 1),"wrong shape of result.")
      self.assertTrue(Lsup(matrix_mult(res,arg)-kronecker(1))<=self.RES_TOL,"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inverse_constData_dim2(self):
      arg=Data(numpy.array([[1.571080150835487, -0.082524281281854961], [-0.9371994394566685, 
2.5886052290160944]]),self.functionspace)
      res=inverse(arg)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(matrix_mult(res,arg)-kronecker(2))<=self.RES_TOL,"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inverse_constData_dim3(self):
      arg=Data(numpy.array([[2.7386874118759605, 0.24320192932450824, -0.98551433885644957], [-0.47237322912611623, 
2.0770157618999936, -0.70532850079694409], [-0.55077067435537064, -0.72886734893974148, 
1.8532894898622749]]),self.functionspace)
      res=inverse(arg)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 3),"wrong shape of result.")
      self.assertTrue(Lsup(matrix_mult(res,arg)-kronecker(3))<=self.RES_TOL,"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inverse_constData_dim4(self):
      arg=Data(numpy.array([[2.7386874118759605, 0.24320192932450824, -0.98551433885644957,1], [-0.47237322912611623, 
2.0770157618999936, -0.70532850079694409,2], [-0.55077067435537064, -0.72886734893974148, 
1.8532894898622749,3],[0,1,0,1]]),self.functionspace)
      try:
        res=inverse(arg)
      except RuntimeError:
        self.assertTrue(not haveLapack,'matrix>3x3 should have inverted')       #We expect no lapack versions to throw
      else:
        self.assertTrue(haveLapack,'matrix <=3x3 should have thrown')   #We should have thrown here if we don't have lapack
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(4, 4),"wrong shape of result.")
        self.assertTrue(Lsup(matrix_mult(res,arg)-kronecker(4))<=self.RES_TOL,"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inverse_constData_singular(self):
        arg=Data(0,self.functionspace)
        try:
           inverse(arg)
        except RuntimeError:
           pass
        else:
           self.fail('Singular matrix (1x1) did not fail to invert.')
        arg=Data([[0,0],[0,1]],self.functionspace)
        try:
          inverse(arg)
        except RuntimeError:
          pass
        else:
          self.fail('Singular matrix (2x2) did not fail to invert.')
        arg=Data([[0,0,0],[0,1,0],[1,1,1]],self.functionspace)
        try:
          inverse(arg)
        except RuntimeError:
          pass
        else:
          self.fail('Singular matrix (3x3) did not fail to invert.')
        #Unsupported matrix sizes are checked in the _dim4 tests so I won't check it here
        if haveLapack:
            arg=Data([[0,0,0,0],[1,4,5,8],[1.0007, 4.00005, 19.00001, 34.000],[-1,1,-243,0]], self.functionspace)
            try:
                inverse(arg)
            except RuntimeError:
                pass
            else:
                self.fail('Singular matrix (4x4) did not fail to invert.')
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inverse_expandedData_dim1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[2.7459817071296375]])+(1.-msk_arg)*numpy.array([[2.708920165934452]])
      res=inverse(arg)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(1, 1),"wrong shape of result.")
      self.assertTrue(Lsup(matrix_mult(res,arg)-kronecker(1))<=self.RES_TOL,"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inverse_expandedData_dim2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[1.940002512105073, 0.029227206714448339], [-0.16699889152420955, 
1.7415408052680506]])+(1.-msk_arg)*numpy.array([[3.6647888118613525, -0.79263077445508201], [-0.12552062258114849, 
3.8289111523004937]])
      res=inverse(arg)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(matrix_mult(res,arg)-kronecker(2))<=self.RES_TOL,"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inverse_expandedData_dim3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[2.2418112382624198, 0.81650662903581805, -0.094756021110331989], [-0.79442369492917742, 
1.9995946283413137, 0.99436683143019566], [0.20497870840231047, 0.56858349778979589, 
1.7344833021084756]])+(1.-msk_arg)*numpy.array([[2.3428233100558398, 0.5902214559572323, 0.7775760520103312], 
[0.30856300288555638, 2.8439402341090472, 0.60593451605418513], [0.99714433866199181, -0.38410207533093699, 
3.2417070638990513]])
      res=inverse(arg)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 3),"wrong shape of result.")
      self.assertTrue(Lsup(matrix_mult(res,arg)-kronecker(3))<=self.RES_TOL,"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inverse_expandedData_dim4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[2.2418112382624198, 0.81650662903581805, -0.094756021110331989,0], [-0.79442369492917742, 
1.9995946283413137, 0.99436683143019566,1], [0.20497870840231047, 0.56858349778979589, 
1.7344833021084756,2],[1,2,3,4]])+(1.-msk_arg)*numpy.array([[2.3428233100558398, 0.5902214559572323, 0.7775760520103312,0], 
[0.30856300288555638, 2.8439402341090472, 0.60593451605418513,1], [0.99714433866199181, -0.38410207533093699, 
3.2417070638990513,2],[2,3,4,5]])
      try:
        res=inverse(arg)
      except RuntimeError:
        self.assertTrue(not haveLapack,'matrix>3x3 should have inverted')       #We expect no lapack versions to throw
      else:
        self.assertTrue(haveLapack,'matrix <=3x3 should have thrown')   #We should have thrown here if we don't have lapack
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(4, 4),"wrong shape of result.")
        self.assertTrue(Lsup(matrix_mult(res,arg)-kronecker(4))<=self.RES_TOL,"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inverse_expandedData_singular(self):
        arg=Data([[0]],self.functionspace, True)
        try:
           inverse(arg)
        except RuntimeError:
           pass
        else:
           self.fail('Singular matrix (1x1) did not fail to invert.')
        arg=Data([[0,0],[0,1]],self.functionspace, True)
        try:
          inverse(arg)
        except RuntimeError:
          pass
        else:
          self.fail('Singular matrix (2x2) did not fail to invert.')
        arg=Data([[0,0,0],[0,1,0],[1,1,1]],self.functionspace, True)
        try:
          inverse(arg)
        except RuntimeError:
          pass
        else:
          self.fail('Singular matrix (3x3) did not fail to invert.')
        #Unsupported matrix sizes are checked in the _dim4 tests so I won't check it here
        if haveLapack:
            arg=Data([[0,0,0,0],[1,4,5,8],[1.0007, 4.00005, 19.00001, 34.000],[-1,1,-243,0]], self.functionspace, True)
            try:
                inverse(arg)
            except RuntimeError:
                pass
            else:
                self.fail('Singular matrix (4x4) did not fail to invert.')
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_array_rank0_offset0(self):
      arg=numpy.array(-0.991585801448)
      res=transpose(arg,0)
      ref=numpy.array(-0.991585801448)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_array_rank1_offset0(self):
      arg=numpy.array([-0.93789697591082799, -0.85164603706162878, -0.93876560479038851])
      res=transpose(arg,0)
      ref=numpy.array([-0.93789697591082799, -0.85164603706162878, -0.93876560479038851])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_array_rank1_offset1(self):
      arg=numpy.array([0.57950937147989445, 0.61922458089073573, -0.86689205664903102])
      res=transpose(arg,1)
      ref=numpy.array([0.57950937147989445, 0.61922458089073573, -0.86689205664903102])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_array_rank2_offset0(self):
      arg=numpy.array([[0.74554267093336457, -0.38983213164494734, -0.92162108007187671, -0.27814168115233451, 
0.10686669551838812], [-0.79094977940587818, -0.36771179185975589, -0.53004651017715187, 0.6072392220613505, 
0.37575091805538463], [0.69989105169549526, -0.4415467907807813, -0.36527282205210621, 0.34876929422929415, 
0.19840166525571834], [0.57437881461884266, 0.0025299568086414315, -0.97959099879387979, 0.67950898726549047, 
0.74830813263939988]])
      res=transpose(arg,0)
      ref=numpy.array([[0.74554267093336457, -0.38983213164494734, -0.92162108007187671, -0.27814168115233451, 
0.10686669551838812], [-0.79094977940587818, -0.36771179185975589, -0.53004651017715187, 0.6072392220613505, 
0.37575091805538463], [0.69989105169549526, -0.4415467907807813, -0.36527282205210621, 0.34876929422929415, 
0.19840166525571834], [0.57437881461884266, 0.0025299568086414315, -0.97959099879387979, 0.67950898726549047, 
0.74830813263939988]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_array_rank2_offset1(self):
      arg=numpy.array([[-0.30474580504453863, 0.98563144702907657, 0.50224838837262076, 0.62731733982431925, 
0.74993187664314243], [0.93457680986442626, 0.3601419236576926, 0.97798123267446213, 0.87881919384548279, 
-0.30000086024833261], [-0.34903366971777983, 0.85738257671029516, 0.052447427116289358, 0.49419334797884118, 
0.044321898264149118], [0.57682727492265862, 0.27716252486272297, -0.44079399571716715, -0.39038714855728962, 
-0.82332203527846648]])
      res=transpose(arg,1)
      ref=numpy.array([[-0.30474580504453863, 0.93457680986442626, -0.34903366971777983, 0.57682727492265862], 
[0.98563144702907657, 0.3601419236576926, 0.85738257671029516, 0.27716252486272297], [0.50224838837262076, 0.97798123267446213, 
0.052447427116289358, -0.44079399571716715], [0.62731733982431925, 0.87881919384548279, 0.49419334797884118, 
-0.39038714855728962], [0.74993187664314243, -0.30000086024833261, 0.044321898264149118, -0.82332203527846648]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(5, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_array_rank2_offset2(self):
      arg=numpy.array([[-0.58013368405730525, -0.92857980063928247, 0.70085030553713668, -0.25547806853804667, 
0.011814665793625778], [0.38418489713323667, -0.06080369512380579, 0.55459171728680867, -0.11124317531496764, 
-0.9833749344955669], [0.61582314984946973, -0.53700585175331406, -0.6091067375473096, 0.65203419280273178, 
0.45748564826212301], [0.43914360965543708, 0.6727820420440922, -0.35329412892227374, -0.25873365726002451, 
0.55429680617726484]])
      res=transpose(arg,2)
      ref=numpy.array([[-0.58013368405730525, -0.92857980063928247, 0.70085030553713668, -0.25547806853804667, 
0.011814665793625778], [0.38418489713323667, -0.06080369512380579, 0.55459171728680867, -0.11124317531496764, 
-0.9833749344955669], [0.61582314984946973, -0.53700585175331406, -0.6091067375473096, 0.65203419280273178, 
0.45748564826212301], [0.43914360965543708, 0.6727820420440922, -0.35329412892227374, -0.25873365726002451, 
0.55429680617726484]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_array_rank3_offset0(self):
      arg=numpy.array([[[-0.96601243442328921, 0.041409518807083767], [-0.40352002819278376, 0.43537792312447721]], 
[[-0.94222823486544405, 0.021427776174599122], [0.78052016765589483, 0.50008305969740552]], [[0.89896642789673376, 
0.41521291763962997], [0.24567573580506474, 0.38950507369775322]], [[0.85861796270240021, -0.51962633835954208], 
[-0.28880432723729177, -0.068216494118681803]], [[0.58798107312627623, 0.3478778666383906], [-0.88873273716635603, 
-0.3973973348469042]], [[-0.44865302574852262, 0.76024479470961959], [-0.72797111599142017, 0.26809687664724402]]])
      res=transpose(arg,0)
      ref=numpy.array([[[-0.96601243442328921, 0.041409518807083767], [-0.40352002819278376, 0.43537792312447721]], 
[[-0.94222823486544405, 0.021427776174599122], [0.78052016765589483, 0.50008305969740552]], [[0.89896642789673376, 
0.41521291763962997], [0.24567573580506474, 0.38950507369775322]], [[0.85861796270240021, -0.51962633835954208], 
[-0.28880432723729177, -0.068216494118681803]], [[0.58798107312627623, 0.3478778666383906], [-0.88873273716635603, 
-0.3973973348469042]], [[-0.44865302574852262, 0.76024479470961959], [-0.72797111599142017, 0.26809687664724402]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_array_rank3_offset1(self):
      arg=numpy.array([[[0.77541791069444232, 0.42785423356049179], [0.9706108086220151, 0.78757657833760475]], 
[[-0.85760178880953264, 0.88496107441065952], [-0.077487704301219207, -0.50835639837845803]], [[-0.51425610306958758, 
-0.13483837297404455], [-0.66770127309063043, -0.91712705977673425]], [[-0.22471523206337851, 0.88763208073630118], 
[-0.49221739211225013, -0.15845213992338847]], [[-0.052710828590955439, -0.013047493530843779], [0.25391285203956082, 
-0.48944179761585471]], [[0.81593949355430095, -0.57124525609145915], [-0.28343172397719396, 0.35854694093602135]]])
      res=transpose(arg,1)
      ref=numpy.array([[[0.77541791069444232, -0.85760178880953264, -0.51425610306958758, -0.22471523206337851, 
-0.052710828590955439, 0.81593949355430095], [0.42785423356049179, 0.88496107441065952, -0.13483837297404455, 
0.88763208073630118, -0.013047493530843779, -0.57124525609145915]], [[0.9706108086220151, -0.077487704301219207, 
-0.66770127309063043, -0.49221739211225013, 0.25391285203956082, -0.28343172397719396], [0.78757657833760475, 
-0.50835639837845803, -0.91712705977673425, -0.15845213992338847, -0.48944179761585471, 0.35854694093602135]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2, 2, 6),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_array_rank3_offset2(self):
      arg=numpy.array([[[0.4107387718286355, -0.25823001180782534], [-0.66910501721662996, 0.58404606599030728]], 
[[0.7258659867497157, -0.46058389378317077], [0.80501499591912729, -0.87682813212476685]], [[-0.78898004497687491, 
0.31447233731211255], [-0.021978427005372225, 0.61716588998675426]], [[-0.15671353391963794, 0.61658185514101738], 
[-0.75593475648502428, 0.41937823738331859]], [[0.84166997419402834, -0.038914108991471297], [-0.20779873870192778, 
-0.68028517664067056]], [[0.003205766919945896, -0.98945427368150418], [-0.83515699811314814, 0.22371237899798246]]])
      res=transpose(arg,2)
      ref=numpy.array([[[0.4107387718286355, -0.66910501721662996], [0.7258659867497157, 0.80501499591912729], 
[-0.78898004497687491, -0.021978427005372225], [-0.15671353391963794, -0.75593475648502428], [0.84166997419402834, 
-0.20779873870192778], [0.003205766919945896, -0.83515699811314814]], [[-0.25823001180782534, 0.58404606599030728], 
[-0.46058389378317077, -0.87682813212476685], [0.31447233731211255, 0.61716588998675426], [0.61658185514101738, 
0.41937823738331859], [-0.038914108991471297, -0.68028517664067056], [-0.98945427368150418, 0.22371237899798246]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2, 6, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_array_rank3_offset3(self):
      arg=numpy.array([[[0.724139520401202, -0.98731569923730378], [-0.94190785758979145, -0.72651112429905895]], 
[[0.78511586821969548, -0.59012022491307348], [0.85081912760423806, -0.77317254849388117]], [[0.10689774347744163, 
-0.54386373082137096], [-0.33232831195778578, 0.91539449878464962]], [[-0.17540031661874367, 0.65159405121611069], 
[-0.90082790488206199, 0.53019611561978897]], [[0.3618076700728019, -0.68986502491378521], [0.78524986751378645, 
0.095904671781718909]], [[-0.69770398784501775, -0.53269269560081178], [-0.0027961902797011273, -0.28489333292668384]]])
      res=transpose(arg,3)
      ref=numpy.array([[[0.724139520401202, -0.98731569923730378], [-0.94190785758979145, -0.72651112429905895]], 
[[0.78511586821969548, -0.59012022491307348], [0.85081912760423806, -0.77317254849388117]], [[0.10689774347744163, 
-0.54386373082137096], [-0.33232831195778578, 0.91539449878464962]], [[-0.17540031661874367, 0.65159405121611069], 
[-0.90082790488206199, 0.53019611561978897]], [[0.3618076700728019, -0.68986502491378521], [0.78524986751378645, 
0.095904671781718909]], [[-0.69770398784501775, -0.53269269560081178], [-0.0027961902797011273, -0.28489333292668384]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_array_rank4_offset0(self):
      arg=numpy.array([[[[0.09812401540939053, 0.80946426430131302, -0.45899387640842004, 0.77896016252927036], 
[-0.54315960491973758, -0.077939066420437797, 0.59063317069743682, -0.33352393778162703], [0.60294370077540971, 
0.107425370583355, 0.86297322995114545, 0.16121116397677393]], [[0.86682576315489324, 0.76888052067825274, 
-0.57922301748199367, 0.60905451241966935], [-0.11661748668576699, 0.66439251442446623, -0.52865134117803492, 
-0.72268552914190431], [-0.54639825544734832, -0.24531562968038023, 0.53793235406305606, -0.86636140894068392]]], 
[[[-0.076439807508062785, 0.83156169505435784, 0.91402573100389395, -0.88105081269257801], [-0.76067392638359776, 
0.80156058418636489, 0.97688764285623986, -0.12705354757834697], [-0.4600110704654008, 0.29781109471673051, 
-0.29570804307232956, 0.28705975558477692]], [[-0.70193541376907409, -0.035159748359987031, -0.98005270012329482, 
0.39680734748805913], [-0.37326444917914725, -0.67836076096735165, -0.86830151010735479, 0.77042437664500563], 
[-0.5924251866453718, 0.23723441799977163, 0.25103612608553294, -0.43090409496106696]]], [[[-0.43116385233731647, 
-0.46396933058788181, 0.24841810386905028, 0.42275908942297735], [0.26296489728983374, -0.43235942967927632, 
0.29754617701030539, 0.33023689393070943], [0.26470180547096622, -0.58578568767761285, 0.37388080111033339, 
-0.35637523566591689]], [[0.15324863417100421, -0.65254701224707223, 0.9226325140442817, 0.69961693627233168], 
[0.18808763144977725, -0.15618854481254441, 0.86825003440822957, -0.045404058373784251], [-0.96788605246572423, 
0.34068734851131643, -0.80896463232761828, 0.65232967047480317]]]])
      res=transpose(arg,0)
      ref=numpy.array([[[[0.09812401540939053, 0.80946426430131302, -0.45899387640842004, 0.77896016252927036], 
[-0.54315960491973758, -0.077939066420437797, 0.59063317069743682, -0.33352393778162703], [0.60294370077540971, 
0.107425370583355, 0.86297322995114545, 0.16121116397677393]], [[0.86682576315489324, 0.76888052067825274, 
-0.57922301748199367, 0.60905451241966935], [-0.11661748668576699, 0.66439251442446623, -0.52865134117803492, 
-0.72268552914190431], [-0.54639825544734832, -0.24531562968038023, 0.53793235406305606, -0.86636140894068392]]], 
[[[-0.076439807508062785, 0.83156169505435784, 0.91402573100389395, -0.88105081269257801], [-0.76067392638359776, 
0.80156058418636489, 0.97688764285623986, -0.12705354757834697], [-0.4600110704654008, 0.29781109471673051, 
-0.29570804307232956, 0.28705975558477692]], [[-0.70193541376907409, -0.035159748359987031, -0.98005270012329482, 
0.39680734748805913], [-0.37326444917914725, -0.67836076096735165, -0.86830151010735479, 0.77042437664500563], 
[-0.5924251866453718, 0.23723441799977163, 0.25103612608553294, -0.43090409496106696]]], [[[-0.43116385233731647, 
-0.46396933058788181, 0.24841810386905028, 0.42275908942297735], [0.26296489728983374, -0.43235942967927632, 
0.29754617701030539, 0.33023689393070943], [0.26470180547096622, -0.58578568767761285, 0.37388080111033339, 
-0.35637523566591689]], [[0.15324863417100421, -0.65254701224707223, 0.9226325140442817, 0.69961693627233168], 
[0.18808763144977725, -0.15618854481254441, 0.86825003440822957, -0.045404058373784251], [-0.96788605246572423, 
0.34068734851131643, -0.80896463232761828, 0.65232967047480317]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_array_rank4_offset1(self):
      arg=numpy.array([[[[-0.92133388466922783, 0.063522028160519994, -0.82474705047534602, 0.56999162597624964], 
[-0.44188686027052659, 0.10753292290441774, -0.2251876099140091, 0.09159533230980732], [0.37288894674283513, 
-0.73634256752976635, 0.9380719796957071, -0.80355477112019469]], [[-0.36801477459215048, 0.15558303622587877, 
-0.57148780339340521, -0.06356910221579648], [-0.99751491991049979, -0.37422131681857018, -0.056562648630222867, 
0.73798510625118352], [0.045097657973861116, 0.23631650272957283, -0.26465081430787452, -0.028776525260937769]]], 
[[[0.66329306904779006, -0.10111657511904149, 0.46292251671927986, -0.33246445082688902], [0.19658994498258653, 
-0.304313819796455, -0.89175885135836808, 0.076006728462423734], [0.066218462193249117, -0.88650135115481099, 
-0.79158588253120787, -0.062950910693769346]], [[-0.91033036042112703, 0.99963056319855248, -0.7066213988807124, 
0.4611608893490291], [0.65426307637009917, -0.0022845454711004631, 0.93184410848727861, 0.1451735233727538], 
[0.76422534498859096, 0.56731429017432955, -0.57208797779009113, 0.44545862507733647]]], [[[-0.0060653117656519484, 
-0.76438826059601639, 0.30727193474375536, -0.83119171226362898], [0.98159496161119986, -0.28252902649741318, 
0.24078298735252779, 0.5938057633461844], [0.36630989643702505, 0.82289575515265168, -0.67044348330078707, 
-0.68197628148255207]], [[0.23814043883864899, -0.57603447153257226, 0.7333127796436838, 0.14727979263146418], 
[-0.60404834234573301, 0.21594495697425575, -0.0028281444528621691, 0.3480910898118017], [-0.31464557699854945, 
-0.42876918982501344, 0.69457001887945435, -0.33144151954797074]]]])
      res=transpose(arg,1)
      ref=numpy.array([[[[-0.92133388466922783, 0.66329306904779006, -0.0060653117656519484], [0.063522028160519994, 
-0.10111657511904149, -0.76438826059601639], [-0.82474705047534602, 0.46292251671927986, 0.30727193474375536], 
[0.56999162597624964, -0.33246445082688902, -0.83119171226362898]], [[-0.44188686027052659, 0.19658994498258653, 
0.98159496161119986], [0.10753292290441774, -0.304313819796455, -0.28252902649741318], [-0.2251876099140091, 
-0.89175885135836808, 0.24078298735252779], [0.09159533230980732, 0.076006728462423734, 0.5938057633461844]], 
[[0.37288894674283513, 0.066218462193249117, 0.36630989643702505], [-0.73634256752976635, -0.88650135115481099, 
0.82289575515265168], [0.9380719796957071, -0.79158588253120787, -0.67044348330078707], [-0.80355477112019469, 
-0.062950910693769346, -0.68197628148255207]]], [[[-0.36801477459215048, -0.91033036042112703, 0.23814043883864899], 
[0.15558303622587877, 0.99963056319855248, -0.57603447153257226], [-0.57148780339340521, -0.7066213988807124, 
0.7333127796436838], [-0.06356910221579648, 0.4611608893490291, 0.14727979263146418]], [[-0.99751491991049979, 
0.65426307637009917, -0.60404834234573301], [-0.37422131681857018, -0.0022845454711004631, 0.21594495697425575], 
[-0.056562648630222867, 0.93184410848727861, -0.0028281444528621691], [0.73798510625118352, 0.1451735233727538, 
0.3480910898118017]], [[0.045097657973861116, 0.76422534498859096, -0.31464557699854945], [0.23631650272957283, 
0.56731429017432955, -0.42876918982501344], [-0.26465081430787452, -0.57208797779009113, 0.69457001887945435], 
[-0.028776525260937769, 0.44545862507733647, -0.33144151954797074]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2, 3, 4, 3),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_array_rank4_offset2(self):
      arg=numpy.array([[[[-0.39194818196315806, 0.0087374611715589356, 0.6980486110060975, -0.78097604030874423], 
[-0.31317301548159748, 0.14169322086274261, 0.62646323444670227, -0.16523556756442637], [0.41791292698281457, 
-0.64223581493619486, 0.13601195687550471, -0.6477931908651946]], [[0.89824128100380962, 0.015467897571568079, 
-0.60064074188271088, 0.90545042896487637], [-0.2506498730319755, 0.23563124098185217, 0.67787302547045547, 
-0.22636958760516146], [0.13225930154798626, 0.97644434105670408, -0.67563262960407644, -0.50154917524304565]]], 
[[[0.44989157343764963, -0.90278693571206192, 0.32346712346673789, 0.42946392962658453], [0.81727249231940435, 
-0.11481000915268225, -0.38398501019471731, -0.6782941659421664], [-0.41899749886411453, -0.2494099321196519, 
-0.8570831244085948, 0.92428527619298229]], [[0.16144141420919156, -0.479974665525988, -0.68731570908367212, 
0.63756508344748397], [0.074741672352118593, 0.72138784560494429, 0.33389831859955788, 0.73846404206748173], 
[0.36241294139543556, -0.74099444339101961, 0.34513222260751109, -0.30186247163429814]]], [[[-0.36127309603550084, 
-0.97059029039592803, -0.96854855153498809, 0.94503791608000975], [0.38046938959204923, -0.32534063493005627, 
0.17638917103468588, 0.5170553762404364], [-0.29768776334286629, 0.69376292750941038, -0.99216100053633061, 
-0.67755969419520001]], [[-0.20226504656709565, 0.23649297621850862, -0.099403356802471476, -0.11769460834269219], 
[-0.046778650986543324, 0.73705899628359273, -0.695785586035782, -0.43048297263482738], [0.24703382796554907, 
-0.16764511013834693, -0.7308358820066041, 0.52382084689504516]]]])
      res=transpose(arg,2)
      ref=numpy.array([[[[-0.39194818196315806, 0.89824128100380962], [0.44989157343764963, 0.16144141420919156], 
[-0.36127309603550084, -0.20226504656709565]], [[0.0087374611715589356, 0.015467897571568079], [-0.90278693571206192, 
-0.479974665525988], [-0.97059029039592803, 0.23649297621850862]], [[0.6980486110060975, -0.60064074188271088], 
[0.32346712346673789, -0.68731570908367212], [-0.96854855153498809, -0.099403356802471476]], [[-0.78097604030874423, 
0.90545042896487637], [0.42946392962658453, 0.63756508344748397], [0.94503791608000975, -0.11769460834269219]]], 
[[[-0.31317301548159748, -0.2506498730319755], [0.81727249231940435, 0.074741672352118593], [0.38046938959204923, 
-0.046778650986543324]], [[0.14169322086274261, 0.23563124098185217], [-0.11481000915268225, 0.72138784560494429], 
[-0.32534063493005627, 0.73705899628359273]], [[0.62646323444670227, 0.67787302547045547], [-0.38398501019471731, 
0.33389831859955788], [0.17638917103468588, -0.695785586035782]], [[-0.16523556756442637, -0.22636958760516146], 
[-0.6782941659421664, 0.73846404206748173], [0.5170553762404364, -0.43048297263482738]]], [[[0.41791292698281457, 
0.13225930154798626], [-0.41899749886411453, 0.36241294139543556], [-0.29768776334286629, 0.24703382796554907]], 
[[-0.64223581493619486, 0.97644434105670408], [-0.2494099321196519, -0.74099444339101961], [0.69376292750941038, 
-0.16764511013834693]], [[0.13601195687550471, -0.67563262960407644], [-0.8570831244085948, 0.34513222260751109], 
[-0.99216100053633061, -0.7308358820066041]], [[-0.6477931908651946, -0.50154917524304565], [0.92428527619298229, 
-0.30186247163429814], [-0.67755969419520001, 0.52382084689504516]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 4, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_array_rank4_offset3(self):
      arg=numpy.array([[[[0.39480405877991176, -0.6768910859264321, 0.26381359372034741, -0.70674598775632269], 
[-0.78564162589903908, -0.85653346456752755, -0.79335314738965312, 0.7516839508332358], [-0.47092897778843468, 
-0.26377958464054219, 0.73673022445723402, 0.86476155458289194]], [[0.71687405334341303, -0.1546458488006206, 
-0.045568700762531611, 0.93326011821354093], [-0.66809209484168086, -0.17858407495041728, 0.36611199155284635, 
-0.52537725117058343], [-0.36873708974096053, 0.80507229929062762, -0.88340398047283042, 0.24300994862310987]]], 
[[[0.059294573390584748, 0.74078576759379811, 0.13930103670146909, -0.35845987219889985], [0.46935953702804389, 
-0.56886695762170492, -0.70994202543541873, -0.9140498486316595], [0.76602070949823897, 0.4491596052404605, 
0.45390492897673429, 0.41919369500824466]], [[0.40215466505209307, 0.032551497530743267, -0.78204565023687156, 
-0.86230004009407302], [-0.67504381173709516, -0.17934625684306016, 0.78685133443957755, 0.1845593069537641], 
[0.81250654404527989, -0.27686485385000137, 0.57358698046907719, -0.54475235992207183]]], [[[-0.87787606164769105, 
-0.39280604930273344, 0.016982421799579317, -0.26908764110213612], [0.25487328595539327, 0.77690677151094367, 
-0.74002559444662253, -0.80324245458401067], [-0.62797579793917868, 0.46887176761116645, -0.34421953685472362, 
0.69446193514808341]], [[0.45322690217593453, -0.021648249468393166, 0.87810965645186467, 0.77212202086944126], 
[-0.81171139951349636, -0.6259263478058712, 0.123349154019325, 0.99346788281053011], [-0.32500306399852441, 
-0.70237364807998026, -0.30382784084127978, -0.26382321660062602]]]])
      res=transpose(arg,3)
      ref=numpy.array([[[[0.39480405877991176, -0.78564162589903908, -0.47092897778843468], [0.71687405334341303, 
-0.66809209484168086, -0.36873708974096053]], [[0.059294573390584748, 0.46935953702804389, 0.76602070949823897], 
[0.40215466505209307, -0.67504381173709516, 0.81250654404527989]], [[-0.87787606164769105, 0.25487328595539327, 
-0.62797579793917868], [0.45322690217593453, -0.81171139951349636, -0.32500306399852441]]], [[[-0.6768910859264321, 
-0.85653346456752755, -0.26377958464054219], [-0.1546458488006206, -0.17858407495041728, 0.80507229929062762]], 
[[0.74078576759379811, -0.56886695762170492, 0.4491596052404605], [0.032551497530743267, -0.17934625684306016, 
-0.27686485385000137]], [[-0.39280604930273344, 0.77690677151094367, 0.46887176761116645], [-0.021648249468393166, 
-0.6259263478058712, -0.70237364807998026]]], [[[0.26381359372034741, -0.79335314738965312, 0.73673022445723402], 
[-0.045568700762531611, 0.36611199155284635, -0.88340398047283042]], [[0.13930103670146909, -0.70994202543541873, 
0.45390492897673429], [-0.78204565023687156, 0.78685133443957755, 0.57358698046907719]], [[0.016982421799579317, 
-0.74002559444662253, -0.34421953685472362], [0.87810965645186467, 0.123349154019325, -0.30382784084127978]]], 
[[[-0.70674598775632269, 0.7516839508332358, 0.86476155458289194], [0.93326011821354093, -0.52537725117058343, 
0.24300994862310987]], [[-0.35845987219889985, -0.9140498486316595, 0.41919369500824466], [-0.86230004009407302, 
0.1845593069537641, -0.54475235992207183]], [[-0.26908764110213612, -0.80324245458401067, 0.69446193514808341], 
[0.77212202086944126, 0.99346788281053011, -0.26382321660062602]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 3, 2, 3),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_array_rank4_offset4(self):
      arg=numpy.array([[[[-0.9658926599408757, -0.23644944052085748, 0.64071824761020624, -0.56976474079226946], 
[0.67423324833611842, -0.99094554504056709, -0.34197552902169948, -0.85836966054000752], [-0.5708423398782716, 
0.25325450758971568, 0.49863354070788368, 0.63557984329430695]], [[0.39004930932542425, 0.088907860315799336, 
-0.048998546516065389, 0.44247073369591461], [0.5479877348819493, -0.75931613693902644, 0.33819785095157595, 
0.40519218936583123], [0.078364340146261702, 0.70711485529110996, -0.25869421030006912, -0.38625752652783385]]], 
[[[-0.85427665795211483, -0.087594882756076409, 0.055310164131721296, 0.070482940196875843], [0.65200754862401777, 
0.023329496112006742, 0.91186115531098411, 0.30950292694703352], [-0.99715036654259537, -0.65102073222971746, 
-0.41402754083185078, -0.43847208832440199]], [[0.11959466833146148, 0.19562739720210454, -0.58535684918614228, 
0.82001955188392861], [0.91767110054128187, 0.5431658034303195, 0.19948875136199584, -0.41918687469886984], 
[-0.79601244453208864, -0.25496246794653832, -0.57116715861978484, 0.95712178469931142]]], [[[0.41926169752472742, 
0.22363966150509551, 0.56478152300402629, 0.12593821398682792], [-0.48080793636533947, 0.45626337991851673, 
-0.81292341487440289, -0.29167169777042101], [-0.20685957393518706, -0.81484780826749881, -0.63164067984126726, 
-0.97740565329217399]], [[0.95723196868689442, 0.20635682802978472, 0.16233572909065708, 0.85221363470099076], 
[-0.39282226126993858, -0.19918405809467465, -0.69684444586336047, 0.10975215759145174], [-0.88828535872416392, 
-0.52806164035912806, 0.80084768712330789, 0.71904960268426499]]]])
      res=transpose(arg,4)
      ref=numpy.array([[[[-0.9658926599408757, -0.23644944052085748, 0.64071824761020624, -0.56976474079226946], 
[0.67423324833611842, -0.99094554504056709, -0.34197552902169948, -0.85836966054000752], [-0.5708423398782716, 
0.25325450758971568, 0.49863354070788368, 0.63557984329430695]], [[0.39004930932542425, 0.088907860315799336, 
-0.048998546516065389, 0.44247073369591461], [0.5479877348819493, -0.75931613693902644, 0.33819785095157595, 
0.40519218936583123], [0.078364340146261702, 0.70711485529110996, -0.25869421030006912, -0.38625752652783385]]], 
[[[-0.85427665795211483, -0.087594882756076409, 0.055310164131721296, 0.070482940196875843], [0.65200754862401777, 
0.023329496112006742, 0.91186115531098411, 0.30950292694703352], [-0.99715036654259537, -0.65102073222971746, 
-0.41402754083185078, -0.43847208832440199]], [[0.11959466833146148, 0.19562739720210454, -0.58535684918614228, 
0.82001955188392861], [0.91767110054128187, 0.5431658034303195, 0.19948875136199584, -0.41918687469886984], 
[-0.79601244453208864, -0.25496246794653832, -0.57116715861978484, 0.95712178469931142]]], [[[0.41926169752472742, 
0.22363966150509551, 0.56478152300402629, 0.12593821398682792], [-0.48080793636533947, 0.45626337991851673, 
-0.81292341487440289, -0.29167169777042101], [-0.20685957393518706, -0.81484780826749881, -0.63164067984126726, 
-0.97740565329217399]], [[0.95723196868689442, 0.20635682802978472, 0.16233572909065708, 0.85221363470099076], 
[-0.39282226126993858, -0.19918405809467465, -0.69684444586336047, 0.10975215759145174], [-0.88828535872416392, 
-0.52806164035912806, 0.80084768712330789, 0.71904960268426499]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_constData_rank0_offset0(self):
      arg=Data(0.225959396074,self.functionspace)
      res=transpose(arg,0)
      ref=Data(0.225959396074,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_constData_rank1_offset0(self):
      arg=Data(numpy.array([-0.10069916326121553, 0.92007057532607028, 0.23757652766451387]),self.functionspace)
      res=transpose(arg,0)
      ref=Data(numpy.array([-0.10069916326121553, 0.92007057532607028, 0.23757652766451387]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_constData_rank1_offset1(self):
      arg=Data(numpy.array([-0.96894085254637852, 0.99397138080599179, 0.75096727359933446]),self.functionspace)
      res=transpose(arg,1)
      ref=Data(numpy.array([-0.96894085254637852, 0.99397138080599179, 0.75096727359933446]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_constData_rank2_offset0(self):
      arg=Data(numpy.array([[-0.77040849666417954, -0.65569275146714245, -0.085115062942961117, 0.53031883260280233, 
0.45045866722167194], [-0.25138155122864525, -0.57389792702151166, 0.68860922394355262, -0.0065526746880473219, 
-0.35764306534130941], [0.1855120335364151, 0.20458186711393633, -0.92082337603524778, 0.44273448552707206, 
0.9132574415070196], [-0.66827173835450138, -0.10195695513805592, -0.62636668115005345, 0.42547131691714135, 
-0.26859658298750433]]),self.functionspace)
      res=transpose(arg,0)
      ref=Data(numpy.array([[-0.77040849666417954, -0.65569275146714245, -0.085115062942961117, 0.53031883260280233, 
0.45045866722167194], [-0.25138155122864525, -0.57389792702151166, 0.68860922394355262, -0.0065526746880473219, 
-0.35764306534130941], [0.1855120335364151, 0.20458186711393633, -0.92082337603524778, 0.44273448552707206, 
0.9132574415070196], [-0.66827173835450138, -0.10195695513805592, -0.62636668115005345, 0.42547131691714135, 
-0.26859658298750433]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_constData_rank2_offset1(self):
      arg=Data(numpy.array([[-0.33491601939065641, 0.69094680423719512, 0.59915028553776173, -0.17958818166304691, 
-0.36878341684202787], [0.070568238098461578, -0.93071350688047483, -0.51431930979320262, 0.79243870568471286, 
-0.30760978245682358], [0.48188891571355419, 0.27082538070485107, -0.61798546635554019, -0.09157099532428048, 
0.041573188753503576], [-0.46996472567292291, 0.091822116640003726, 0.82968798099977259, -0.90218837916197514, 
-0.2800509995815792]]),self.functionspace)
      res=transpose(arg,1)
      ref=Data(numpy.array([[-0.33491601939065641, 0.070568238098461578, 0.48188891571355419, -0.46996472567292291], 
[0.69094680423719512, -0.93071350688047483, 0.27082538070485107, 0.091822116640003726], [0.59915028553776173, 
-0.51431930979320262, -0.61798546635554019, 0.82968798099977259], [-0.17958818166304691, 0.79243870568471286, 
-0.09157099532428048, -0.90218837916197514], [-0.36878341684202787, -0.30760978245682358, 0.041573188753503576, 
-0.2800509995815792]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(5, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_constData_rank2_offset2(self):
      arg=Data(numpy.array([[-0.82520616823441428, 0.23178414157473859, 0.51336605638991473, -0.15382223373865966, 
0.38135571801585266], [0.32542008573228198, 0.16152008642250415, 0.70218192092285547, -0.59967809110321091, 
-0.27328553913693976], [0.6664714000556613, -0.81171782543448479, -0.64798918174611497, 0.0271087416779352, 
0.36283514673162354], [0.77384232927642294, 0.34982601937720692, 0.45691537494513046, 0.91177206590340187, 
0.30654445623290028]]),self.functionspace)
      res=transpose(arg,2)
      ref=Data(numpy.array([[-0.82520616823441428, 0.23178414157473859, 0.51336605638991473, -0.15382223373865966, 
0.38135571801585266], [0.32542008573228198, 0.16152008642250415, 0.70218192092285547, -0.59967809110321091, 
-0.27328553913693976], [0.6664714000556613, -0.81171782543448479, -0.64798918174611497, 0.0271087416779352, 
0.36283514673162354], [0.77384232927642294, 0.34982601937720692, 0.45691537494513046, 0.91177206590340187, 
0.30654445623290028]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_constData_rank3_offset0(self):
      arg=Data(numpy.array([[[-0.083961167763012767, 0.75255816882127125], [-0.74297953496476565, -0.78165934740603404]], 
[[0.83549977133305964, 0.20912684191235753], [-0.91601301962855231, 0.15145807325710625]], [[-0.31770291701539133, 
0.102506071837142], [-0.85609847709286746, 0.073777859252800893]], [[-0.71229101547579732, 0.42916964805672819], 
[0.6746565760958132, 0.63810376664450619]], [[-0.23285662388801653, -0.48801062112788451], [0.17936444574797683, 
0.11032910465573775]], [[-0.85718541874892162, 0.29088039452314662], [-0.13540243373731498, 
0.22760939006250824]]]),self.functionspace)
      res=transpose(arg,0)
      ref=Data(numpy.array([[[-0.083961167763012767, 0.75255816882127125], [-0.74297953496476565, -0.78165934740603404]], 
[[0.83549977133305964, 0.20912684191235753], [-0.91601301962855231, 0.15145807325710625]], [[-0.31770291701539133, 
0.102506071837142], [-0.85609847709286746, 0.073777859252800893]], [[-0.71229101547579732, 0.42916964805672819], 
[0.6746565760958132, 0.63810376664450619]], [[-0.23285662388801653, -0.48801062112788451], [0.17936444574797683, 
0.11032910465573775]], [[-0.85718541874892162, 0.29088039452314662], [-0.13540243373731498, 
0.22760939006250824]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_constData_rank3_offset1(self):
      arg=Data(numpy.array([[[-0.042269181435689029, 0.17793672760180068], [0.45087643293361235, -0.056364428732654792]], 
[[-0.81477257272391168, 0.14054428213249559], [0.35573867405158488, 0.76785646537954655]], [[-0.22083616952137142, 
-0.23664220804541358], [0.41657512286553944, 0.94253794934849355]], [[0.2311373052564818, -0.19762165243937391], 
[-0.95012581524794082, -0.19612170397872153]], [[0.88230819444997133, -0.59133465030314891], [0.58772016310959208, 
-0.60523154692164716]], [[0.61572404162018302, 0.11636730158627984], [-0.91159653308647148, 
-0.14380185308054139]]]),self.functionspace)
      res=transpose(arg,1)
      ref=Data(numpy.array([[[-0.042269181435689029, -0.81477257272391168, -0.22083616952137142, 0.2311373052564818, 
0.88230819444997133, 0.61572404162018302], [0.17793672760180068, 0.14054428213249559, -0.23664220804541358, 
-0.19762165243937391, -0.59133465030314891, 0.11636730158627984]], [[0.45087643293361235, 0.35573867405158488, 
0.41657512286553944, -0.95012581524794082, 0.58772016310959208, -0.91159653308647148], [-0.056364428732654792, 
0.76785646537954655, 0.94253794934849355, -0.19612170397872153, -0.60523154692164716, 
-0.14380185308054139]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 2, 6),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_constData_rank3_offset2(self):
      arg=Data(numpy.array([[[-0.94655642599340895, 0.98245908885666666], [-0.72990190255518717, 0.35843119603651763]], 
[[0.037401662839183336, -0.0045781922217276172], [0.10281748425147752, 0.20718092971734881]], [[-0.74962348813202762, 
-0.70912709764536541], [0.62287774759194026, -0.45625121314087513]], [[0.578381701983951, 0.60103694479786918], 
[-0.52967023703614613, 0.16147518679648654]], [[0.12967425251567466, -0.71363229423562569], [-0.1609519217368085, 
-0.60004506300397464]], [[0.91403980319630196, -0.84616417269132849], [-0.81542282397933752, 
-0.02174165953527174]]]),self.functionspace)
      res=transpose(arg,2)
      ref=Data(numpy.array([[[-0.94655642599340895, -0.72990190255518717], [0.037401662839183336, 0.10281748425147752], 
[-0.74962348813202762, 0.62287774759194026], [0.578381701983951, -0.52967023703614613], [0.12967425251567466, 
-0.1609519217368085], [0.91403980319630196, -0.81542282397933752]], [[0.98245908885666666, 0.35843119603651763], 
[-0.0045781922217276172, 0.20718092971734881], [-0.70912709764536541, -0.45625121314087513], [0.60103694479786918, 
0.16147518679648654], [-0.71363229423562569, -0.60004506300397464], [-0.84616417269132849, 
-0.02174165953527174]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 6, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_constData_rank3_offset3(self):
      arg=Data(numpy.array([[[-0.50094258685726012, -0.11804641099402491], [0.54676942661623795, 0.31600921256216274]], 
[[-0.58408147535921007, 0.34809360137770984], [0.50005428183149991, -0.25294841139527469]], [[0.059314096894031554, 
0.52874319292190664], [-0.1314313646550076, -0.87940671115560032]], [[0.46292409727943684, 0.5191235491178583], 
[-0.437616790136516, -0.47109591872500434]], [[-0.72208122346331116, -0.47535246418613242], [0.91974494685767372, 
-0.23086488372938296]], [[-0.69613100517423487, 0.54386522047707553], [0.8597484998147733, 
0.28629791272229932]]]),self.functionspace)
      res=transpose(arg,3)
      ref=Data(numpy.array([[[-0.50094258685726012, -0.11804641099402491], [0.54676942661623795, 0.31600921256216274]], 
[[-0.58408147535921007, 0.34809360137770984], [0.50005428183149991, -0.25294841139527469]], [[0.059314096894031554, 
0.52874319292190664], [-0.1314313646550076, -0.87940671115560032]], [[0.46292409727943684, 0.5191235491178583], 
[-0.437616790136516, -0.47109591872500434]], [[-0.72208122346331116, -0.47535246418613242], [0.91974494685767372, 
-0.23086488372938296]], [[-0.69613100517423487, 0.54386522047707553], [0.8597484998147733, 
0.28629791272229932]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_constData_rank4_offset0(self):
      arg=Data(numpy.array([[[[0.43160274169330548, 0.64574673937769389, -0.22090587850110532, 0.27327990716134543], 
[0.4357184869287225, -0.1057167094295115, 0.088376391082047645, 0.98890168268659351], [-0.26892091808757956, 
-0.46449597655854058, -0.30767377671695084, 0.48381430522270263]], [[0.53268114144817602, 0.8914423078395235, 
-0.98279699443410662, -0.074270373952304514], [0.61245177480132051, 0.38414589123621901, -0.59664346034600069, 
-0.43578722171684037], [0.61008123791889957, 0.55624350594847893, -0.51828071056523184, -0.94828145590534518]]], 
[[[0.098790655455476273, 0.59777914100459784, 0.85324921191791403, 0.58793186703765188], [0.57583199823949371, 
0.84582314809452841, -0.24387545604263217, -0.49022805643629686], [-0.8755050156388906, -0.69414546093848251, 
-0.83631843020297181, -0.16501548428890045]], [[0.011484769832809771, -0.67529896836916636, 0.48307712186937435, 
0.84232426864036714], [0.76787908935376636, -0.92147846282378132, -0.73025447906715946, -0.64382697566715974], 
[-0.005449284426912504, 0.85962202832752466, 0.15071959625479225, -0.96316320443521364]]], [[[-0.91108923040744427, 
0.77314517473264477, -0.23191867503149455, -0.58593465344895135], [-0.72007957060415384, 0.95515846407330152, 
-0.10252164430284805, -0.41514590942875462], [0.88047846832877275, 0.95787212628912477, 0.58694365764216649, 
-0.13020535549536283]], [[-0.31152431684213489, 0.67808804984728899, -0.32133868411563782, -0.038594688259944387], 
[0.7991166706322419, -0.12765107971293377, 0.15512221968538764, -0.41710652523509806], [0.74696018231245098, 
0.60113942409165722, -0.27277443196119178, 0.47852441965314441]]]]),self.functionspace)
      res=transpose(arg,0)
      ref=Data(numpy.array([[[[0.43160274169330548, 0.64574673937769389, -0.22090587850110532, 0.27327990716134543], 
[0.4357184869287225, -0.1057167094295115, 0.088376391082047645, 0.98890168268659351], [-0.26892091808757956, 
-0.46449597655854058, -0.30767377671695084, 0.48381430522270263]], [[0.53268114144817602, 0.8914423078395235, 
-0.98279699443410662, -0.074270373952304514], [0.61245177480132051, 0.38414589123621901, -0.59664346034600069, 
-0.43578722171684037], [0.61008123791889957, 0.55624350594847893, -0.51828071056523184, -0.94828145590534518]]], 
[[[0.098790655455476273, 0.59777914100459784, 0.85324921191791403, 0.58793186703765188], [0.57583199823949371, 
0.84582314809452841, -0.24387545604263217, -0.49022805643629686], [-0.8755050156388906, -0.69414546093848251, 
-0.83631843020297181, -0.16501548428890045]], [[0.011484769832809771, -0.67529896836916636, 0.48307712186937435, 
0.84232426864036714], [0.76787908935376636, -0.92147846282378132, -0.73025447906715946, -0.64382697566715974], 
[-0.005449284426912504, 0.85962202832752466, 0.15071959625479225, -0.96316320443521364]]], [[[-0.91108923040744427, 
0.77314517473264477, -0.23191867503149455, -0.58593465344895135], [-0.72007957060415384, 0.95515846407330152, 
-0.10252164430284805, -0.41514590942875462], [0.88047846832877275, 0.95787212628912477, 0.58694365764216649, 
-0.13020535549536283]], [[-0.31152431684213489, 0.67808804984728899, -0.32133868411563782, -0.038594688259944387], 
[0.7991166706322419, -0.12765107971293377, 0.15512221968538764, -0.41710652523509806], [0.74696018231245098, 
0.60113942409165722, -0.27277443196119178, 0.47852441965314441]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_constData_rank4_offset1(self):
      arg=Data(numpy.array([[[[-0.86619024662118393, 0.77612208466864563, 0.01032463102377279, -0.38055879688655758], 
[-0.76325416572146532, 0.52930284656375926, -0.33395466552432773, -0.52901256263270868], [0.50540410887764753, 
-0.58977464856070116, -0.13411386892344201, 0.9082622556330453]], [[0.73821183656589584, -0.57794756275774684, 
-0.42792298780429583, 0.07270908658219688], [0.87197306795180629, -0.14896474573021479, -0.87552810567779904, 
0.25274597077059013], [-0.44755310665820369, -0.59858983399272736, 0.41742224877641743, -0.62800871201009878]]], 
[[[-0.93586002663127976, 0.78565035995205079, 0.73570402336212593, 0.5006428539322989], [0.092412867825900191, 
-0.95348826554521304, -0.30899515577249881, 0.79470430322061003], [0.59590624583087837, 0.57396077391852351, 
0.48632583736616786, 0.95612341051252536]], [[-0.65283497592741901, 0.6279985524035312, -0.39634295071715342, 
-0.0095363457098334159], [0.64338532378786528, 0.96737632439281795, -0.57890272176670976, -0.16717012024502775], 
[0.97037747341473812, 0.38268766961993972, -0.37098741976903682, 0.14499586626138461]]], [[[-0.61437183199444911, 
-0.55366743717712352, 0.01702633409230403, -0.79534955631554172], [-0.50449304819837759, 0.50669985742674628, 
0.18309694323838688, -0.23103415009029082], [0.95915689329110876, -0.66668316006912609, 0.57141073520479302, 
0.93148753597082279]], [[0.84247788447581407, 0.43798733751020591, 0.1836536952502783, 0.18227598980498261], 
[-0.87846592343467211, 0.50449580150763484, -0.76562309090549308, -0.12438599387823968], [-0.40127928144782588, 
0.33005242608613328, 0.047231350853066845, 0.011416343030370646]]]]),self.functionspace)
      res=transpose(arg,1)
      ref=Data(numpy.array([[[[-0.86619024662118393, -0.93586002663127976, -0.61437183199444911], [0.77612208466864563, 
0.78565035995205079, -0.55366743717712352], [0.01032463102377279, 0.73570402336212593, 0.01702633409230403], 
[-0.38055879688655758, 0.5006428539322989, -0.79534955631554172]], [[-0.76325416572146532, 0.092412867825900191, 
-0.50449304819837759], [0.52930284656375926, -0.95348826554521304, 0.50669985742674628], [-0.33395466552432773, 
-0.30899515577249881, 0.18309694323838688], [-0.52901256263270868, 0.79470430322061003, -0.23103415009029082]], 
[[0.50540410887764753, 0.59590624583087837, 0.95915689329110876], [-0.58977464856070116, 0.57396077391852351, 
-0.66668316006912609], [-0.13411386892344201, 0.48632583736616786, 0.57141073520479302], [0.9082622556330453, 
0.95612341051252536, 0.93148753597082279]]], [[[0.73821183656589584, -0.65283497592741901, 0.84247788447581407], 
[-0.57794756275774684, 0.6279985524035312, 0.43798733751020591], [-0.42792298780429583, -0.39634295071715342, 
0.1836536952502783], [0.07270908658219688, -0.0095363457098334159, 0.18227598980498261]], [[0.87197306795180629, 
0.64338532378786528, -0.87846592343467211], [-0.14896474573021479, 0.96737632439281795, 0.50449580150763484], 
[-0.87552810567779904, -0.57890272176670976, -0.76562309090549308], [0.25274597077059013, -0.16717012024502775, 
-0.12438599387823968]], [[-0.44755310665820369, 0.97037747341473812, -0.40127928144782588], [-0.59858983399272736, 
0.38268766961993972, 0.33005242608613328], [0.41742224877641743, -0.37098741976903682, 0.047231350853066845], 
[-0.62800871201009878, 0.14499586626138461, 0.011416343030370646]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 3, 4, 3),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_constData_rank4_offset2(self):
      arg=Data(numpy.array([[[[-0.020623148720276596, 0.98817517733714388, -0.96844017850217767, 0.48166130295864429], 
[0.90089248798594501, -0.19956489345294481, 0.94049453942339656, 0.23506161663644898], [-0.15660691951419681, 
-0.46495501679844131, -0.091698985972450853, 0.049525910723174027]], [[-0.032229935637019036, -0.88814308599722147, 
0.39414768367069453, -0.65002599890113144], [-0.73013457839377804, 0.63429806306930359, -0.56994284243904447, 
0.096527566156159539], [0.75614698527591218, -0.54061679138185648, 0.48758775780312913, -0.94944624152301027]]], 
[[[-0.31675293498881008, 0.40032179478230479, -0.99790640009238163, -0.70056231587973605], [0.5712106930511196, 
-0.4874962644886589, 0.59326823122811145, 0.13022434812901329], [-0.072826502308464214, 0.81908281754326562, 
-0.77203842477023277, 0.45180068025520881]], [[-0.011901043078094586, -0.71554336414688269, -0.98178621273359634, 
-0.032041726870593479], [0.33213233012322663, -0.39783730802603889, -0.95103310118353623, 0.41750247945599828], 
[0.62496483297284411, 0.8146196435785853, 0.65406659823571522, 0.63961536124475726]]], [[[0.044264946155937679, 
-0.15567810759246359, 0.13443381061365201, 0.10897985607493266], [-0.83600415194649869, 0.53742333585330848, 
0.0042639147861471827, -0.46552390496676921], [-0.3142114095880375, -0.98204264176975697, 0.080925914517088859, 
-0.57795077282453455]], [[-0.84501525652759168, 0.42616929215003285, -0.4379776100099424, -0.87675319820159325], 
[0.7132274646561414, 0.82482062775536691, -0.13933556022696036, 0.91560258175534237], [-0.48443700594914252, 
0.97756918085699551, -0.16672083508601676, 0.61656316598467287]]]]),self.functionspace)
      res=transpose(arg,2)
      ref=Data(numpy.array([[[[-0.020623148720276596, -0.032229935637019036], [-0.31675293498881008, -0.011901043078094586], 
[0.044264946155937679, -0.84501525652759168]], [[0.98817517733714388, -0.88814308599722147], [0.40032179478230479, 
-0.71554336414688269], [-0.15567810759246359, 0.42616929215003285]], [[-0.96844017850217767, 0.39414768367069453], 
[-0.99790640009238163, -0.98178621273359634], [0.13443381061365201, -0.4379776100099424]], [[0.48166130295864429, 
-0.65002599890113144], [-0.70056231587973605, -0.032041726870593479], [0.10897985607493266, -0.87675319820159325]]], 
[[[0.90089248798594501, -0.73013457839377804], [0.5712106930511196, 0.33213233012322663], [-0.83600415194649869, 
0.7132274646561414]], [[-0.19956489345294481, 0.63429806306930359], [-0.4874962644886589, -0.39783730802603889], 
[0.53742333585330848, 0.82482062775536691]], [[0.94049453942339656, -0.56994284243904447], [0.59326823122811145, 
-0.95103310118353623], [0.0042639147861471827, -0.13933556022696036]], [[0.23506161663644898, 0.096527566156159539], 
[0.13022434812901329, 0.41750247945599828], [-0.46552390496676921, 0.91560258175534237]]], [[[-0.15660691951419681, 
0.75614698527591218], [-0.072826502308464214, 0.62496483297284411], [-0.3142114095880375, -0.48443700594914252]], 
[[-0.46495501679844131, -0.54061679138185648], [0.81908281754326562, 0.8146196435785853], [-0.98204264176975697, 
0.97756918085699551]], [[-0.091698985972450853, 0.48758775780312913], [-0.77203842477023277, 0.65406659823571522], 
[0.080925914517088859, -0.16672083508601676]], [[0.049525910723174027, -0.94944624152301027], [0.45180068025520881, 
0.63961536124475726], [-0.57795077282453455, 0.61656316598467287]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 4, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_constData_rank4_offset3(self):
      arg=Data(numpy.array([[[[-0.52141273307449154, -0.83929977632806896, -0.31902270452335513, 0.79665228423666079], 
[-0.30077572285955645, 0.24229518193839339, 0.91790350449627489, -0.25374653431343952], [-0.23356898033553941, 
0.56721098117836743, 0.97316407264831328, -0.166067557334723]], [[-0.040908002983164549, 0.85581284787935563, 
-0.42981166479977229, 0.19934426615375656], [0.60764048518567293, 0.60051343907877519, -0.29480724585301155, 
-0.50949402827825008], [0.41819323746109816, -0.40641240097303255, 0.72766403087786813, 0.75161850950077302]]], 
[[[0.84909119687341494, 0.096983123743885402, 0.78889022930311126, 0.92856811779567971], [0.79843424835198196, 
0.86978258523723206, -0.0066347882980821726, -0.19013205984594506], [-0.609421546922853, -0.9856086946541267, 
0.24658983362852505, -0.92045223577964919]], [[-0.92473724556297165, 0.80123098001545623, 0.98058353955443978, 
0.90668805371963557], [0.098685567085265768, -0.48364223799443584, 0.75585568137986692, 0.73201973380740815], 
[-0.1944600998500754, 0.5871839497933653, -0.44282997348843978, 0.68118761183728993]]], [[[-0.99507882663539826, 
-0.26609145039959969, 0.030024200934443801, -0.52303304466657075], [0.24431774917342652, 0.79035097476813365, 
0.48069117907454184, -0.18360982112260693], [-0.31923284946487951, 0.4736754145799531, 0.047841289025573719, 
0.18825437106135445]], [[-0.86362123608515495, -0.97264093101084081, 0.28867583994022872, 0.77433272790107366], 
[0.6346898892416557, 0.16172500867356354, -0.86962243714176268, -0.58276863176407767], [0.98135568353256208, 
-0.45535575203409961, -0.29359155368758638, 0.28638172754798341]]]]),self.functionspace)
      res=transpose(arg,3)
      ref=Data(numpy.array([[[[-0.52141273307449154, -0.30077572285955645, -0.23356898033553941], [-0.040908002983164549, 
0.60764048518567293, 0.41819323746109816]], [[0.84909119687341494, 0.79843424835198196, -0.609421546922853], 
[-0.92473724556297165, 0.098685567085265768, -0.1944600998500754]], [[-0.99507882663539826, 0.24431774917342652, 
-0.31923284946487951], [-0.86362123608515495, 0.6346898892416557, 0.98135568353256208]]], [[[-0.83929977632806896, 
0.24229518193839339, 0.56721098117836743], [0.85581284787935563, 0.60051343907877519, -0.40641240097303255]], 
[[0.096983123743885402, 0.86978258523723206, -0.9856086946541267], [0.80123098001545623, -0.48364223799443584, 
0.5871839497933653]], [[-0.26609145039959969, 0.79035097476813365, 0.4736754145799531], [-0.97264093101084081, 
0.16172500867356354, -0.45535575203409961]]], [[[-0.31902270452335513, 0.91790350449627489, 0.97316407264831328], 
[-0.42981166479977229, -0.29480724585301155, 0.72766403087786813]], [[0.78889022930311126, -0.0066347882980821726, 
0.24658983362852505], [0.98058353955443978, 0.75585568137986692, -0.44282997348843978]], [[0.030024200934443801, 
0.48069117907454184, 0.047841289025573719], [0.28867583994022872, -0.86962243714176268, -0.29359155368758638]]], 
[[[0.79665228423666079, -0.25374653431343952, -0.166067557334723], [0.19934426615375656, -0.50949402827825008, 
0.75161850950077302]], [[0.92856811779567971, -0.19013205984594506, -0.92045223577964919], [0.90668805371963557, 
0.73201973380740815, 0.68118761183728993]], [[-0.52303304466657075, -0.18360982112260693, 0.18825437106135445], 
[0.77433272790107366, -0.58276863176407767, 0.28638172754798341]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 3, 2, 3),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_constData_rank4_offset4(self):
      arg=Data(numpy.array([[[[0.11669032343097374, 0.28341863178991522, 0.36241214828868151, -0.08560215446885544], 
[0.17010320280477753, -0.23484785243370876, 0.52537017022974086, 0.62537416226011411], [0.45730727091713796, 
-0.0083202340695580634, -0.82551004556823471, 0.89643686628458519]], [[-0.28008564822305959, 0.32394601069303208, 
0.45330078548847519, -0.48036164776646229], [-0.39006023397659173, 0.3153683404849541, -0.26898131503979106, 
0.80195481437101712], [-0.74384820956423736, -0.21747831692486153, 0.59908167627432674, -0.041350975612570728]]], 
[[[0.65057968212347927, -0.35439970026212975, -0.94710836622812988, 0.11230875969455778], [0.31908697775133166, 
-0.95209327810136535, -0.67586428755001382, 0.50625307288857213], [-0.37665219594286015, -0.83900587129744575, 
0.35137418052980207, 0.41253459721425978]], [[0.36222029822986723, -0.1682017102331097, 0.16724749033810471, 
-0.045560191691116358], [0.1408126135381178, 0.33279136286718436, 0.098208087565660573, -0.55465634242608441], 
[0.44936996356416992, -0.9035330938984163, -0.080691409187981789, -0.25890210880784648]]], [[[0.3430200098156595, 
0.95669331413565128, -0.54005650467695854, -0.69638636975878887], [0.23184666203843474, 0.35457799800815737, 
-0.11035729576745301, 0.21677731652893417], [0.64330152381940198, -0.058702985238521466, -0.94443456222587918, 
-0.41151451705295461]], [[-0.056223161671889788, 0.37074211517923872, -0.41652840869344776, -0.17907498921595488], 
[0.27601775720460786, 0.017853516785225176, -0.48540226282117449, 0.056532709460667352], [0.81356629339054698, 
0.31342153937788741, -0.91433991650147006, -0.98552878390061083]]]]),self.functionspace)
      res=transpose(arg,4)
      ref=Data(numpy.array([[[[0.11669032343097374, 0.28341863178991522, 0.36241214828868151, -0.08560215446885544], 
[0.17010320280477753, -0.23484785243370876, 0.52537017022974086, 0.62537416226011411], [0.45730727091713796, 
-0.0083202340695580634, -0.82551004556823471, 0.89643686628458519]], [[-0.28008564822305959, 0.32394601069303208, 
0.45330078548847519, -0.48036164776646229], [-0.39006023397659173, 0.3153683404849541, -0.26898131503979106, 
0.80195481437101712], [-0.74384820956423736, -0.21747831692486153, 0.59908167627432674, -0.041350975612570728]]], 
[[[0.65057968212347927, -0.35439970026212975, -0.94710836622812988, 0.11230875969455778], [0.31908697775133166, 
-0.95209327810136535, -0.67586428755001382, 0.50625307288857213], [-0.37665219594286015, -0.83900587129744575, 
0.35137418052980207, 0.41253459721425978]], [[0.36222029822986723, -0.1682017102331097, 0.16724749033810471, 
-0.045560191691116358], [0.1408126135381178, 0.33279136286718436, 0.098208087565660573, -0.55465634242608441], 
[0.44936996356416992, -0.9035330938984163, -0.080691409187981789, -0.25890210880784648]]], [[[0.3430200098156595, 
0.95669331413565128, -0.54005650467695854, -0.69638636975878887], [0.23184666203843474, 0.35457799800815737, 
-0.11035729576745301, 0.21677731652893417], [0.64330152381940198, -0.058702985238521466, -0.94443456222587918, 
-0.41151451705295461]], [[-0.056223161671889788, 0.37074211517923872, -0.41652840869344776, -0.17907498921595488], 
[0.27601775720460786, 0.017853516785225176, -0.48540226282117449, 0.056532709460667352], [0.81356629339054698, 
0.31342153937788741, -0.91433991650147006, -0.98552878390061083]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_expandedData_rank0_offset0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*(-0.464998323008)+(1.-msk_arg)*(0.842661504847)
      res=transpose(arg,0)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-0.464998323008)+(1.-msk_ref)*(0.842661504847)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_expandedData_rank1_offset0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([-0.93515248951280916, -0.63645299662894583, 
-0.72032006106062596])+(1.-msk_arg)*numpy.array([0.85731330443781228, 0.27720960254947258, -0.94190591247287814])
      res=transpose(arg,0)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-0.93515248951280916, -0.63645299662894583, 
-0.72032006106062596])+(1.-msk_ref)*numpy.array([0.85731330443781228, 0.27720960254947258, -0.94190591247287814])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_expandedData_rank1_offset1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([0.45790263146770438, 0.33433219849137918, 
0.37200574526317576])+(1.-msk_arg)*numpy.array([-0.75528597960251331, 0.47992806140429534, 0.5673613000647284])
      res=transpose(arg,1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.45790263146770438, 0.33433219849137918, 
0.37200574526317576])+(1.-msk_ref)*numpy.array([-0.75528597960251331, 0.47992806140429534, 0.5673613000647284])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_expandedData_rank2_offset0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[-0.53704721440444847, 0.61203953798930444, 0.85727280004308493, 0.4377187379856915, 
-0.063934854409582575], [0.62806224461388704, 0.83664645182853792, -0.72466823082490794, -0.2037872685625377, 
0.19137745291903485], [0.96277548564078264, -0.091657304567904996, 0.36279787910690731, 0.6629159707207255, 
-0.44957509633007775], [0.27694065166979032, -0.40566467935404371, -0.68230951496025294, -0.28933816472920992, 
0.8200698367653243]])+(1.-msk_arg)*numpy.array([[0.45375160446492946, 0.4757763372891195, 0.73822215623688892, 
0.90125927956355345, 0.51284267833736674], [-0.36203552633408687, -0.47778674559695666, -0.15474129256023361, 
0.57238037773730244, 0.94217455361631708], [-0.6488839277687366, 0.44180578636353141, 0.545573933661059, 0.64824452832079249, 
-0.2705135007008721], [0.086042778893174798, 0.86395549252612658, 0.54598774936213412, 0.18350957802415713, 
-0.84816627288593338]])
      res=transpose(arg,0)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-0.53704721440444847, 0.61203953798930444, 0.85727280004308493, 0.4377187379856915, 
-0.063934854409582575], [0.62806224461388704, 0.83664645182853792, -0.72466823082490794, -0.2037872685625377, 
0.19137745291903485], [0.96277548564078264, -0.091657304567904996, 0.36279787910690731, 0.6629159707207255, 
-0.44957509633007775], [0.27694065166979032, -0.40566467935404371, -0.68230951496025294, -0.28933816472920992, 
0.8200698367653243]])+(1.-msk_ref)*numpy.array([[0.45375160446492946, 0.4757763372891195, 0.73822215623688892, 
0.90125927956355345, 0.51284267833736674], [-0.36203552633408687, -0.47778674559695666, -0.15474129256023361, 
0.57238037773730244, 0.94217455361631708], [-0.6488839277687366, 0.44180578636353141, 0.545573933661059, 0.64824452832079249, 
-0.2705135007008721], [0.086042778893174798, 0.86395549252612658, 0.54598774936213412, 0.18350957802415713, 
-0.84816627288593338]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_expandedData_rank2_offset1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[-0.32580216273561802, 0.88258181859689411, -0.31772866396073574, 0.27627930667308154, 
-0.088429805468721812], [0.99335510106820202, 0.19340185580001501, 0.091626824903613624, -0.83323166636627199, 
-0.60695824987136238], [0.14618868726964851, -0.40313708009836602, -0.41526874325398389, -0.71664007199572732, 
0.53157204714956818], [0.65271418059585451, 0.7010524959749973, -0.73193701119154175, -0.19024209620515653, 
0.84128736160113271]])+(1.-msk_arg)*numpy.array([[-0.20206353179328307, 0.35713518346467343, 0.6228740908405952, 
0.30894459291684706, 0.047716301598745847], [-0.52923853056818992, -0.21887853121329703, 0.060697225764464902, 
0.44776948837240549, 0.28926865496068621], [0.45738136318010469, 0.93446336677291897, 0.79568100664793207, 0.09096636683455861, 
0.60967354629925286], [-0.88687370297859092, 0.76071587077257563, 0.55798705838929408, -0.72961818946498069, 
-0.43832456514333074]])
      res=transpose(arg,1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-0.32580216273561802, 0.99335510106820202, 0.14618868726964851, 0.65271418059585451], 
[0.88258181859689411, 0.19340185580001501, -0.40313708009836602, 0.7010524959749973], [-0.31772866396073574, 
0.091626824903613624, -0.41526874325398389, -0.73193701119154175], [0.27627930667308154, -0.83323166636627199, 
-0.71664007199572732, -0.19024209620515653], [-0.088429805468721812, -0.60695824987136238, 0.53157204714956818, 
0.84128736160113271]])+(1.-msk_ref)*numpy.array([[-0.20206353179328307, -0.52923853056818992, 0.45738136318010469, 
-0.88687370297859092], [0.35713518346467343, -0.21887853121329703, 0.93446336677291897, 0.76071587077257563], 
[0.6228740908405952, 0.060697225764464902, 0.79568100664793207, 0.55798705838929408], [0.30894459291684706, 
0.44776948837240549, 0.09096636683455861, -0.72961818946498069], [0.047716301598745847, 0.28926865496068621, 
0.60967354629925286, -0.43832456514333074]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(5, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_expandedData_rank2_offset2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[-0.97403318737416011, 0.54001074266784244, -0.67257310766949741, -0.96103153788957196, 
-0.090600621848506036], [0.8012530387117236, -0.46233200285581977, -0.93711608690024262, -0.84456531095899678, 
0.25368308422261365], [0.17120901507941344, -0.039509505335912243, 0.26212105744531833, 0.44137638362107534, 
0.068065142614604168], [0.88476743417179837, -0.94424338152961318, 0.32121853448672133, -0.41052967402945106, 
0.63834170539248025]])+(1.-msk_arg)*numpy.array([[0.83834016279533419, 0.87758908559916393, -0.87142410558147954, 
0.75081011516190665, 0.38788552560538037], [0.56389050328386947, 0.68360899469026748, 0.15224375501252863, 0.3878626742528275, 
0.808658442259099], [0.95751078857047456, -0.4485866232585427, -0.057922467649792919, 0.67058305132721951, 0.4170761862507304], 
[-0.83468036775004095, -0.42590581548764872, 0.33806350887148717, 0.92324742714838015, 0.86739002484873473]])
      res=transpose(arg,2)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-0.97403318737416011, 0.54001074266784244, -0.67257310766949741, -0.96103153788957196, 
-0.090600621848506036], [0.8012530387117236, -0.46233200285581977, -0.93711608690024262, -0.84456531095899678, 
0.25368308422261365], [0.17120901507941344, -0.039509505335912243, 0.26212105744531833, 0.44137638362107534, 
0.068065142614604168], [0.88476743417179837, -0.94424338152961318, 0.32121853448672133, -0.41052967402945106, 
0.63834170539248025]])+(1.-msk_ref)*numpy.array([[0.83834016279533419, 0.87758908559916393, -0.87142410558147954, 
0.75081011516190665, 0.38788552560538037], [0.56389050328386947, 0.68360899469026748, 0.15224375501252863, 0.3878626742528275, 
0.808658442259099], [0.95751078857047456, -0.4485866232585427, -0.057922467649792919, 0.67058305132721951, 0.4170761862507304], 
[-0.83468036775004095, -0.42590581548764872, 0.33806350887148717, 0.92324742714838015, 0.86739002484873473]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_expandedData_rank3_offset0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[-0.028166702102059515, -0.65351279092816994], [0.95298016917902295, -0.94116303047578986]], 
[[0.95567274481534303, -0.92003120295127805], [0.42765976432858976, -0.31983322718682206]], [[0.6594701854374565, 
0.93364717548894505], [0.43942332156329322, -0.61290767453893036]], [[-0.018980262942142057, 0.58083925333713826], 
[0.34100232100453409, -0.66544629598076988]], [[0.74749657732448926, -0.076590201414883596], [-0.080101944121035551, 
0.46279559877354726]], [[-0.30894045650367352, 0.32520823197623239], [-0.76166037545395082, 
-0.85094509752834857]]])+(1.-msk_arg)*numpy.array([[[0.21529172013404407, 0.17618688196541799], [-0.15693875160198978, 
0.2300388702556555]], [[-0.14355628056799907, 0.097284390186505698], [0.70840494470572057, -0.39686655898757484]], 
[[0.39869343017412806, -0.1175308671433728], [-0.68320739412614229, -0.80868375226463374]], [[-0.098983144871312101, 
-0.084826511475350808], [-0.67115518288588816, 0.88295863571073507]], [[-0.66948047549080814, 0.76395211978036848], 
[0.71116659904241653, -0.62357374862249104]], [[-0.47845456014444587, -0.18838396364199839], [-0.89361492901920725, 
-0.60664644922995437]]])
      res=transpose(arg,0)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.028166702102059515, -0.65351279092816994], [0.95298016917902295, -0.94116303047578986]], 
[[0.95567274481534303, -0.92003120295127805], [0.42765976432858976, -0.31983322718682206]], [[0.6594701854374565, 
0.93364717548894505], [0.43942332156329322, -0.61290767453893036]], [[-0.018980262942142057, 0.58083925333713826], 
[0.34100232100453409, -0.66544629598076988]], [[0.74749657732448926, -0.076590201414883596], [-0.080101944121035551, 
0.46279559877354726]], [[-0.30894045650367352, 0.32520823197623239], [-0.76166037545395082, 
-0.85094509752834857]]])+(1.-msk_ref)*numpy.array([[[0.21529172013404407, 0.17618688196541799], [-0.15693875160198978, 
0.2300388702556555]], [[-0.14355628056799907, 0.097284390186505698], [0.70840494470572057, -0.39686655898757484]], 
[[0.39869343017412806, -0.1175308671433728], [-0.68320739412614229, -0.80868375226463374]], [[-0.098983144871312101, 
-0.084826511475350808], [-0.67115518288588816, 0.88295863571073507]], [[-0.66948047549080814, 0.76395211978036848], 
[0.71116659904241653, -0.62357374862249104]], [[-0.47845456014444587, -0.18838396364199839], [-0.89361492901920725, 
-0.60664644922995437]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_expandedData_rank3_offset1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[-0.65212850444869774, -0.67427825613144465], [0.032515841309460392, -0.070453441121938409]], 
[[-0.44405818865607904, -0.88066476407580097], [-0.93943889067471176, -0.18440508636169772]], [[0.50229407654290581, 
-0.46720545488262188], [-0.12305085433257457, 0.43760534985372135]], [[-0.34034328160142824, -0.33974758283591244], 
[-0.10359700187580767, -0.47619370131541672]], [[0.99560014491745208, 0.60433461317408632], [-0.01200102407520931, 
-0.56878679528667964]], [[0.071244629477601462, 0.47885889090888267], [0.55063383124273013, 
-0.98807819331889735]]])+(1.-msk_arg)*numpy.array([[[-0.61574370477685192, -0.33843489935062609], [-0.24744346254201832, 
-0.99244383369104638]], [[-0.47705335491139178, 0.59873718107100449], [0.95001518342138969, -0.45469199010386396]], 
[[0.82537151930411401, 0.78306864563067369], [-0.29781496093360582, -0.1389059510802535]], [[-0.15187600029441217, 
0.19148463548052108], [-0.068458630473290594, -0.17373178799506772]], [[0.9606079373215417, 0.53866671148384992], 
[0.12671344872674917, 0.63991537162253498]], [[0.14421305024721365, 0.34646429257341271], [0.060890878638466983, 
-0.12837501000317642]]])
      res=transpose(arg,1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.65212850444869774, -0.44405818865607904, 0.50229407654290581, -0.34034328160142824, 
0.99560014491745208, 0.071244629477601462], [-0.67427825613144465, -0.88066476407580097, -0.46720545488262188, 
-0.33974758283591244, 0.60433461317408632, 0.47885889090888267]], [[0.032515841309460392, -0.93943889067471176, 
-0.12305085433257457, -0.10359700187580767, -0.01200102407520931, 0.55063383124273013], [-0.070453441121938409, 
-0.18440508636169772, 0.43760534985372135, -0.47619370131541672, -0.56878679528667964, 
-0.98807819331889735]]])+(1.-msk_ref)*numpy.array([[[-0.61574370477685192, -0.47705335491139178, 0.82537151930411401, 
-0.15187600029441217, 0.9606079373215417, 0.14421305024721365], [-0.33843489935062609, 0.59873718107100449, 
0.78306864563067369, 0.19148463548052108, 0.53866671148384992, 0.34646429257341271]], [[-0.24744346254201832, 
0.95001518342138969, -0.29781496093360582, -0.068458630473290594, 0.12671344872674917, 0.060890878638466983], 
[-0.99244383369104638, -0.45469199010386396, -0.1389059510802535, -0.17373178799506772, 0.63991537162253498, 
-0.12837501000317642]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 2, 6),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_expandedData_rank3_offset2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[-0.53903371687770951, 0.50880456508237493], [-0.97321323339217702, -0.70984704429690315]], 
[[-0.29984792204722388, -0.29952593289024687], [-0.51539132133232002, 0.80947522826389551]], [[-0.25606147050458139, 
-0.76851985240107257], [0.18298989771828822, 0.21458019480730894]], [[-0.85320820766677063, 0.130626211216055], 
[-0.52862240159366403, 0.054039476584147428]], [[-0.12379893481755899, 0.75923360070874324], [0.43042980136306119, 
0.82098581908338986]], [[-0.1063388172840285, -0.19723460169468043], [0.56520147884081684, 
0.11926890514299293]]])+(1.-msk_arg)*numpy.array([[[0.87041473035143047, -0.61864214268461115], [-0.51123703040419999, 
0.79067050331554145]], [[-0.93665451783569065, -0.84017435553820263], [0.15157488183009526, -0.65788579051674634]], 
[[-0.50181333219235946, -0.0813086873482185], [0.75325943197873957, -0.89460545835219318]], [[0.39727223404279655, 
0.21557493453719245], [0.48497226618136491, -0.55181527987004331]], [[0.56856268337500926, -0.59250828867398297], 
[0.97866925940743577, 0.28718934799498919]], [[0.451728712560445, -0.052194155928735775], [0.62325926405792464, 
0.92529013965526552]]])
      res=transpose(arg,2)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.53903371687770951, -0.97321323339217702], [-0.29984792204722388, -0.51539132133232002], 
[-0.25606147050458139, 0.18298989771828822], [-0.85320820766677063, -0.52862240159366403], [-0.12379893481755899, 
0.43042980136306119], [-0.1063388172840285, 0.56520147884081684]], [[0.50880456508237493, -0.70984704429690315], 
[-0.29952593289024687, 0.80947522826389551], [-0.76851985240107257, 0.21458019480730894], [0.130626211216055, 
0.054039476584147428], [0.75923360070874324, 0.82098581908338986], [-0.19723460169468043, 
0.11926890514299293]]])+(1.-msk_ref)*numpy.array([[[0.87041473035143047, -0.51123703040419999], [-0.93665451783569065, 
0.15157488183009526], [-0.50181333219235946, 0.75325943197873957], [0.39727223404279655, 0.48497226618136491], 
[0.56856268337500926, 0.97866925940743577], [0.451728712560445, 0.62325926405792464]], [[-0.61864214268461115, 
0.79067050331554145], [-0.84017435553820263, -0.65788579051674634], [-0.0813086873482185, -0.89460545835219318], 
[0.21557493453719245, -0.55181527987004331], [-0.59250828867398297, 0.28718934799498919], [-0.052194155928735775, 
0.92529013965526552]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 6, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_expandedData_rank3_offset3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[0.95280346387411785, -0.8597936044370964], [0.75874743984327697, -0.72169276926585768]], 
[[-0.3937382062339374, 0.30993043294385125], [0.71442785036443235, -0.028619926104225257]], [[0.8839593043072036, 
0.48825441905213762], [-0.56975067316864059, 0.326471333947842]], [[0.90769090486825044, -0.96229536718804543], 
[-0.63035004993415056, 0.69308286490365645]], [[0.43681916733085657, -0.83959070890106169], [-0.85007498666880044, 
0.48184432125333387]], [[-0.42218969748124069, -0.12552895993038238], [0.09899649175538161, 
0.73382708590795431]]])+(1.-msk_arg)*numpy.array([[[0.6981920443190599, -0.81522673360912545], [0.74644719606925092, 
-0.92980760296799514]], [[-0.56060385477886832, -0.48194379298251722], [0.052156709810996515, 0.63054816998533902]], 
[[0.19473488250147031, 0.47827111373305309], [-0.70026589874588874, -0.9657868039697437]], [[0.38957118187307449, 
-0.99428804994986497], [-0.73816985888014042, 0.77756988638402524]], [[0.65927654078500075, -0.74578044179329828], 
[0.77276372440199648, 0.92719242860572515]], [[0.6227560684724025, -0.57739027506080065], [0.76208509145720682, 
-0.54371480995667154]]])
      res=transpose(arg,3)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.95280346387411785, -0.8597936044370964], [0.75874743984327697, -0.72169276926585768]], 
[[-0.3937382062339374, 0.30993043294385125], [0.71442785036443235, -0.028619926104225257]], [[0.8839593043072036, 
0.48825441905213762], [-0.56975067316864059, 0.326471333947842]], [[0.90769090486825044, -0.96229536718804543], 
[-0.63035004993415056, 0.69308286490365645]], [[0.43681916733085657, -0.83959070890106169], [-0.85007498666880044, 
0.48184432125333387]], [[-0.42218969748124069, -0.12552895993038238], [0.09899649175538161, 
0.73382708590795431]]])+(1.-msk_ref)*numpy.array([[[0.6981920443190599, -0.81522673360912545], [0.74644719606925092, 
-0.92980760296799514]], [[-0.56060385477886832, -0.48194379298251722], [0.052156709810996515, 0.63054816998533902]], 
[[0.19473488250147031, 0.47827111373305309], [-0.70026589874588874, -0.9657868039697437]], [[0.38957118187307449, 
-0.99428804994986497], [-0.73816985888014042, 0.77756988638402524]], [[0.65927654078500075, -0.74578044179329828], 
[0.77276372440199648, 0.92719242860572515]], [[0.6227560684724025, -0.57739027506080065], [0.76208509145720682, 
-0.54371480995667154]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_expandedData_rank4_offset0(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[0.81869402324079421, -0.010426790790318163, 0.81602756109052454, -0.62442093894574113], 
[0.7078561438909603, -0.82727300588647346, -0.32523384907570807, -0.79543723372576491], [-0.84666126535799702, 
0.36665427129032513, -0.20239160816273793, -0.24432094759297329]], [[0.12528190952985629, -0.6654993988168012, 
-0.8633381499204662, -0.3440703159080063], [-0.072645065869830994, 0.66722151643468619, 0.52271358494400388, 
-0.10066051348077676], [-0.23843351249380373, -0.46559440824373644, 0.48690780786997512, 0.93026389486527483]]], 
[[[0.7731713364251056, -0.70940874010004218, 0.58775645365261608, -0.71235361850702072], [0.48924929297027386, 
-0.58429249738685018, 0.51174147131730363, -0.39624667123742441], [0.55054179653627267, -0.93110569532221255, 
-0.040048710470845794, -0.73129453854897242]], [[-0.29357609501371429, -0.10177896805633635, 0.68084634100814423, 
0.67164277915926496], [-0.30707764379481284, 0.52277794839314251, -0.63944100965934192, -0.69075328601591424], 
[0.76020217943844881, 0.84392878869840304, -0.63894762575037611, 0.09148097151184742]]], [[[-0.45340171975626831, 
0.018489889194636566, -0.2475409727065343, 0.92146815788657377], [-0.43178988675210483, -0.63868466704965488, 
-0.70615203684716543, -0.081156395091610545], [0.03023740091439775, 0.038082297458140957, 0.2727792900915651, 
-0.31207470093614775]], [[0.76156061599481517, -0.67078395293486448, 0.7748140039581779, -0.22794668580099264], 
[-0.9013816625854838, 0.17470357313303775, 0.39122278674109046, 0.93137139452775419], [-0.090956547472199167, 
0.59685604960025573, 0.73105963808354812, -0.15603573236847224]]]])+(1.-msk_arg)*numpy.array([[[[-0.76725648877324404, 
0.015245757133742321, 0.99314477995261696, -0.61973001840063491], [0.7001718430223165, 0.33363829219788377, 
-0.85614009511728661, -0.40303672744138597], [-0.74271016814121671, -0.16466169430980071, 0.53923902460068729, 
-0.87899110008395831]], [[0.57954472900890774, 0.41148609583801266, -0.43921064372789376, 0.087038409738692923], 
[-0.35892851493259581, -0.87654055619137194, 0.42100369956297712, -0.23061186655103505], [-0.044855086158764435, 
-0.50562286197909656, -0.28948250699820388, -0.44514000463666825]]], [[[-0.1817422694873263, -0.74014543691996804, 
-0.51641677925651042, -0.52013880734377516], [-0.02135821913146585, -0.071010347161462262, 0.96069451747209866, 
-0.91732353712862746], [0.90794388971530604, -0.93990200249118661, -0.22015779472787766, -0.015902129610877669]], 
[[0.92570703703005042, 0.0257172426216985, 0.48500303890414487, -0.019695182871451111], [-0.33500929720249673, 
-0.026389937578426448, 0.13852674725578984, -0.31028487034205465], [0.30692937773320339, -0.10312813725950276, 
0.65945732601603391, -0.94530407645713321]]], [[[-0.88901000279791886, 0.26375878520993168, 0.87307830915304141, 
-0.99476930566962785], [0.96889278203485585, 0.24278528683413647, 0.66840991781409631, -0.45990095836976974], 
[0.59021226623571255, 0.4865764356278135, 0.17396249549551235, 0.0099840509722524118]], [[-0.72549311651874393, 
-0.18428823210680001, 0.46813884602522293, 0.84243096795311501], [-0.2928051310524864, -0.3915081581700719, 
0.80673877426477048, -0.34983079791056682], [0.62283520201605325, -0.281811909243179, -0.97932300041287434, 
0.98834191128607873]]]])
      res=transpose(arg,0)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.81869402324079421, -0.010426790790318163, 0.81602756109052454, -0.62442093894574113], 
[0.7078561438909603, -0.82727300588647346, -0.32523384907570807, -0.79543723372576491], [-0.84666126535799702, 
0.36665427129032513, -0.20239160816273793, -0.24432094759297329]], [[0.12528190952985629, -0.6654993988168012, 
-0.8633381499204662, -0.3440703159080063], [-0.072645065869830994, 0.66722151643468619, 0.52271358494400388, 
-0.10066051348077676], [-0.23843351249380373, -0.46559440824373644, 0.48690780786997512, 0.93026389486527483]]], 
[[[0.7731713364251056, -0.70940874010004218, 0.58775645365261608, -0.71235361850702072], [0.48924929297027386, 
-0.58429249738685018, 0.51174147131730363, -0.39624667123742441], [0.55054179653627267, -0.93110569532221255, 
-0.040048710470845794, -0.73129453854897242]], [[-0.29357609501371429, -0.10177896805633635, 0.68084634100814423, 
0.67164277915926496], [-0.30707764379481284, 0.52277794839314251, -0.63944100965934192, -0.69075328601591424], 
[0.76020217943844881, 0.84392878869840304, -0.63894762575037611, 0.09148097151184742]]], [[[-0.45340171975626831, 
0.018489889194636566, -0.2475409727065343, 0.92146815788657377], [-0.43178988675210483, -0.63868466704965488, 
-0.70615203684716543, -0.081156395091610545], [0.03023740091439775, 0.038082297458140957, 0.2727792900915651, 
-0.31207470093614775]], [[0.76156061599481517, -0.67078395293486448, 0.7748140039581779, -0.22794668580099264], 
[-0.9013816625854838, 0.17470357313303775, 0.39122278674109046, 0.93137139452775419], [-0.090956547472199167, 
0.59685604960025573, 0.73105963808354812, -0.15603573236847224]]]])+(1.-msk_ref)*numpy.array([[[[-0.76725648877324404, 
0.015245757133742321, 0.99314477995261696, -0.61973001840063491], [0.7001718430223165, 0.33363829219788377, 
-0.85614009511728661, -0.40303672744138597], [-0.74271016814121671, -0.16466169430980071, 0.53923902460068729, 
-0.87899110008395831]], [[0.57954472900890774, 0.41148609583801266, -0.43921064372789376, 0.087038409738692923], 
[-0.35892851493259581, -0.87654055619137194, 0.42100369956297712, -0.23061186655103505], [-0.044855086158764435, 
-0.50562286197909656, -0.28948250699820388, -0.44514000463666825]]], [[[-0.1817422694873263, -0.74014543691996804, 
-0.51641677925651042, -0.52013880734377516], [-0.02135821913146585, -0.071010347161462262, 0.96069451747209866, 
-0.91732353712862746], [0.90794388971530604, -0.93990200249118661, -0.22015779472787766, -0.015902129610877669]], 
[[0.92570703703005042, 0.0257172426216985, 0.48500303890414487, -0.019695182871451111], [-0.33500929720249673, 
-0.026389937578426448, 0.13852674725578984, -0.31028487034205465], [0.30692937773320339, -0.10312813725950276, 
0.65945732601603391, -0.94530407645713321]]], [[[-0.88901000279791886, 0.26375878520993168, 0.87307830915304141, 
-0.99476930566962785], [0.96889278203485585, 0.24278528683413647, 0.66840991781409631, -0.45990095836976974], 
[0.59021226623571255, 0.4865764356278135, 0.17396249549551235, 0.0099840509722524118]], [[-0.72549311651874393, 
-0.18428823210680001, 0.46813884602522293, 0.84243096795311501], [-0.2928051310524864, -0.3915081581700719, 
0.80673877426477048, -0.34983079791056682], [0.62283520201605325, -0.281811909243179, -0.97932300041287434, 
0.98834191128607873]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_expandedData_rank4_offset1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[-0.048819050540667597, 0.073412941081060001, -0.40916233029580451, 0.21381417778795631], 
[-0.58854827092304496, -0.10595136154093288, 0.65807232040559627, -0.78658005066180747], [0.37852953741585571, 
0.12509013527419599, -0.36227259064010608, -0.27531657009900079]], [[0.056710591592378679, -0.20311477289364799, 
0.39216009496062365, -0.91696789909839316], [0.43891932098900677, 0.85835144280886588, 0.21708783687303357, 
0.47944129991865925], [-0.77859295994328326, -0.89854616004745735, -0.70928105672056652, -0.22941633500484326]]], 
[[[-0.35619990493958698, -0.38033219286105613, -0.49978678208123384, 0.83945371576602823], [-0.10429686318649911, 
-0.81465699109225698, 0.59393092517497026, -0.79906537251314713], [0.7420938505437078, 0.73261882071497886, 
-0.10903051376573414, -0.5686807497125923]], [[-0.094547564433882236, -0.97404312569854379, -0.16451419129417855, 
-0.70240379827910004], [0.055913546460434782, -0.83760832378688788, -0.24958459029717406, -0.44869628045710996], 
[-0.048521229840842395, 0.61004675252622254, -0.13860530037680796, 0.5892536901326737]]], [[[-0.43323685996283157, 
-0.74007855630164698, -0.12315471559872337, 0.71630276794437697], [-0.13423126080114467, 0.0043841611942150305, 
-0.98228190092937928, 0.50642602679381654], [0.72255474223581939, 0.6626322112516827, 0.045475320133810948, 
0.26634612620024245]], [[0.7931643774350805, -0.18761453634299019, -0.64809591125487187, 0.70505567194231911], 
[-0.97131926769839128, 0.75199848500163458, -0.10828884981536269, 0.15623566644399367], [0.081763341002275336, 
0.47364100065457126, 0.2408074003101881, -0.66981995170023922]]]])+(1.-msk_arg)*numpy.array([[[[0.47400462735905013, 
-0.39852798401855583, -0.74339114670731865, -0.41503683234739563], [-0.88150113600440716, 0.2018961746493857, 
0.12340831119303108, -0.76286870406502727], [0.17829549553710033, -0.18035617671767579, -0.16380181203650523, 
0.52789007152254586]], [[0.086111180954433042, -0.014802214581463469, -0.38619388566642199, 0.38393628724858408], 
[0.0097951751768969864, 0.44723053766673959, 0.6727209440204629, 0.18396480207247068], [0.79274034237746438, 
-0.61295051071113882, 0.205553795490661, -0.69552317318207568]]], [[[0.51700025913065595, -0.8876467818799203, 
-0.33322253102932264, -0.47916390987562218], [-0.88418537566938427, -0.77615995413851047, -0.45080523080120249, 
0.3149523527324456], [0.23109729798985224, 0.58566732186982895, 0.64258132305459936, -0.22365920215162993]], 
[[-0.81644226040156931, 0.22207689422893795, -0.13844945472623937, -0.69450532786607311], [-0.77892116009689705, 
0.10954889018386771, -0.54347347121331158, 0.91839991786786768], [0.79380220203641194, -0.01500796749763933, 
0.91768540519214126, 0.70863353126169581]]], [[[-0.37257475168970489, 0.2687751640717484, -0.77942457666399778, 
-0.6977200237953074], [-0.31231185864363842, -0.84898191509909737, 0.42669863567131583, -0.69386713636951214], 
[-0.50773887932678496, -0.68528044041981206, -0.62466973412337401, 0.74029103511113625]], [[-0.073842497656119388, 
0.20375191445595964, 0.390412002575941, -0.99051909465152188], [0.58963517415554501, 0.27807727878994792, 0.4699163088851368, 
0.84953208550268866], [-0.49056115579247717, 0.33082241761331832, 0.8370840519299958, -0.65193906496406462]]]])
      res=transpose(arg,1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.048819050540667597, -0.35619990493958698, -0.43323685996283157], [0.073412941081060001, 
-0.38033219286105613, -0.74007855630164698], [-0.40916233029580451, -0.49978678208123384, -0.12315471559872337], 
[0.21381417778795631, 0.83945371576602823, 0.71630276794437697]], [[-0.58854827092304496, -0.10429686318649911, 
-0.13423126080114467], [-0.10595136154093288, -0.81465699109225698, 0.0043841611942150305], [0.65807232040559627, 
0.59393092517497026, -0.98228190092937928], [-0.78658005066180747, -0.79906537251314713, 0.50642602679381654]], 
[[0.37852953741585571, 0.7420938505437078, 0.72255474223581939], [0.12509013527419599, 0.73261882071497886, 
0.6626322112516827], [-0.36227259064010608, -0.10903051376573414, 0.045475320133810948], [-0.27531657009900079, 
-0.5686807497125923, 0.26634612620024245]]], [[[0.056710591592378679, -0.094547564433882236, 0.7931643774350805], 
[-0.20311477289364799, -0.97404312569854379, -0.18761453634299019], [0.39216009496062365, -0.16451419129417855, 
-0.64809591125487187], [-0.91696789909839316, -0.70240379827910004, 0.70505567194231911]], [[0.43891932098900677, 
0.055913546460434782, -0.97131926769839128], [0.85835144280886588, -0.83760832378688788, 0.75199848500163458], 
[0.21708783687303357, -0.24958459029717406, -0.10828884981536269], [0.47944129991865925, -0.44869628045710996, 
0.15623566644399367]], [[-0.77859295994328326, -0.048521229840842395, 0.081763341002275336], [-0.89854616004745735, 
0.61004675252622254, 0.47364100065457126], [-0.70928105672056652, -0.13860530037680796, 0.2408074003101881], 
[-0.22941633500484326, 0.5892536901326737, -0.66981995170023922]]]])+(1.-msk_ref)*numpy.array([[[[0.47400462735905013, 
0.51700025913065595, -0.37257475168970489], [-0.39852798401855583, -0.8876467818799203, 0.2687751640717484], 
[-0.74339114670731865, -0.33322253102932264, -0.77942457666399778], [-0.41503683234739563, -0.47916390987562218, 
-0.6977200237953074]], [[-0.88150113600440716, -0.88418537566938427, -0.31231185864363842], [0.2018961746493857, 
-0.77615995413851047, -0.84898191509909737], [0.12340831119303108, -0.45080523080120249, 0.42669863567131583], 
[-0.76286870406502727, 0.3149523527324456, -0.69386713636951214]], [[0.17829549553710033, 0.23109729798985224, 
-0.50773887932678496], [-0.18035617671767579, 0.58566732186982895, -0.68528044041981206], [-0.16380181203650523, 
0.64258132305459936, -0.62466973412337401], [0.52789007152254586, -0.22365920215162993, 0.74029103511113625]]], 
[[[0.086111180954433042, -0.81644226040156931, -0.073842497656119388], [-0.014802214581463469, 0.22207689422893795, 
0.20375191445595964], [-0.38619388566642199, -0.13844945472623937, 0.390412002575941], [0.38393628724858408, 
-0.69450532786607311, -0.99051909465152188]], [[0.0097951751768969864, -0.77892116009689705, 0.58963517415554501], 
[0.44723053766673959, 0.10954889018386771, 0.27807727878994792], [0.6727209440204629, -0.54347347121331158, 
0.4699163088851368], [0.18396480207247068, 0.91839991786786768, 0.84953208550268866]], [[0.79274034237746438, 
0.79380220203641194, -0.49056115579247717], [-0.61295051071113882, -0.01500796749763933, 0.33082241761331832], 
[0.205553795490661, 0.91768540519214126, 0.8370840519299958], [-0.69552317318207568, 0.70863353126169581, 
-0.65193906496406462]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 3, 4, 3),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_expandedData_rank4_offset2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[0.82691378813207361, 0.75601412583334637, 0.094339614810351469, 0.65763396579143341], 
[-0.44936985041624555, -0.17289606057712148, 0.12086007969034407, 0.62148756054888588], [0.48578428300240306, 
0.89194298916819159, 0.86381272926337838, 0.44735883641203489]], [[0.47162319912329953, 0.29778028138343382, 
-0.30446934052440056, -0.63178513980850015], [0.05409344608353428, -0.61756272108637722, -0.58202921087939874, 
-0.86637478190029915], [-0.67829385566459632, -0.39253263965340746, -0.59914668025155882, 0.12700127706133868]]], 
[[[0.3588888654790745, 0.66774960005889228, 0.85521153905186287, 0.51165736484563196], [-0.65874992512735631, 
0.084203961656897031, 0.74961673296491238, -0.58075019586619026], [-0.34501529484131233, 0.66465972348152458, 
-0.24819651654841346, -0.45708840913271875]], [[0.29245976088440084, 0.57662066073591745, 0.40824164114989991, 
0.72407884827867219], [0.28191567140363394, 0.063388931447958097, -0.60945304700175229, 0.78234647400698476], 
[0.37494417075446784, -0.94654572471817167, -0.41391209388823347, -0.97295964185973172]]], [[[-0.63131900288607357, 
-0.57105931030840895, -0.56374452835856648, -0.6412471468563643], [-0.19540229837369916, -0.50813816060939221, 
0.51389925106455236, 0.044949365624884718], [-0.83486749693392781, -0.35141697629531454, -0.44109075785843688, 
0.69773867500860876]], [[-0.94987265337847737, 0.78768487656758968, -0.98329526131696277, 0.072370998294981614], 
[-0.8732103855505835, -0.83495361675593815, -0.70019896952723015, -0.52071543626147299], [-0.11199016199767331, 
-0.93486594678371682, -0.76284657045107118, -0.59436275187568688]]]])+(1.-msk_arg)*numpy.array([[[[0.38319749702248007, 
0.47697398105712696, -0.90942599512602085, -0.13951650547575722], [-0.64858191935095366, -0.86686852820081595, 
0.091845111612018471, 0.018036867830121261], [-0.68470765521525001, -0.40811641836047685, -0.95877787795297631, 
-0.43074264171436316]], [[-0.510152208291617, -0.78210832331247904, -0.12147896847076889, -0.15736275503966723], 
[-0.99101028268449998, 0.93282518490434141, 0.28437255063834943, -0.076952791938787124], [0.43250510058194269, 
0.38741829151287588, 0.27260121458743569, 0.74443017737146389]]], [[[0.47620394999223437, -0.56840424059294237, 
0.16562963252734408, -0.78125201221835283], [-0.70565615599014198, -0.067105949665969167, 0.67986244720361122, 
-0.70787219438310989], [-0.63318035645221338, -0.12799625797341685, -0.67916617194550066, -0.50063000672793567]], 
[[0.2317865255298881, -0.99283720434349632, -0.93836356873230242, -0.086856634026029278], [0.22040654164816265, 
-0.61546887103786863, 0.62985234823171865, -0.62551669097947116], [-0.93016620410572459, 0.74271380835971024, 
-0.60113645890759382, -0.09055205949663736]]], [[[0.54017035868478769, 0.4607223920914798, -0.53300577805949878, 
0.23732350893260268], [-0.044918040347948107, -0.19688990696396824, -0.38804666415807398, -0.97641750204548328], 
[-0.85641375056218627, -0.11642557377137686, 0.32552481775546593, 0.065240488269599606]], [[-0.30438043357237166, 
-0.16435006242306049, -0.4687624402457613, -0.44571103372306298], [0.83549765131007603, -0.86568475111010135, 
-0.64334402729419216, -0.71457042181838926], [-0.86715709080052794, 0.81179382479499007, 0.4823914387567243, 
-0.66559992391241551]]]])
      res=transpose(arg,2)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.82691378813207361, 0.47162319912329953], [0.3588888654790745, 0.29245976088440084], 
[-0.63131900288607357, -0.94987265337847737]], [[0.75601412583334637, 0.29778028138343382], [0.66774960005889228, 
0.57662066073591745], [-0.57105931030840895, 0.78768487656758968]], [[0.094339614810351469, -0.30446934052440056], 
[0.85521153905186287, 0.40824164114989991], [-0.56374452835856648, -0.98329526131696277]], [[0.65763396579143341, 
-0.63178513980850015], [0.51165736484563196, 0.72407884827867219], [-0.6412471468563643, 0.072370998294981614]]], 
[[[-0.44936985041624555, 0.05409344608353428], [-0.65874992512735631, 0.28191567140363394], [-0.19540229837369916, 
-0.8732103855505835]], [[-0.17289606057712148, -0.61756272108637722], [0.084203961656897031, 0.063388931447958097], 
[-0.50813816060939221, -0.83495361675593815]], [[0.12086007969034407, -0.58202921087939874], [0.74961673296491238, 
-0.60945304700175229], [0.51389925106455236, -0.70019896952723015]], [[0.62148756054888588, -0.86637478190029915], 
[-0.58075019586619026, 0.78234647400698476], [0.044949365624884718, -0.52071543626147299]]], [[[0.48578428300240306, 
-0.67829385566459632], [-0.34501529484131233, 0.37494417075446784], [-0.83486749693392781, -0.11199016199767331]], 
[[0.89194298916819159, -0.39253263965340746], [0.66465972348152458, -0.94654572471817167], [-0.35141697629531454, 
-0.93486594678371682]], [[0.86381272926337838, -0.59914668025155882], [-0.24819651654841346, -0.41391209388823347], 
[-0.44109075785843688, -0.76284657045107118]], [[0.44735883641203489, 0.12700127706133868], [-0.45708840913271875, 
-0.97295964185973172], [0.69773867500860876, -0.59436275187568688]]]])+(1.-msk_ref)*numpy.array([[[[0.38319749702248007, 
-0.510152208291617], [0.47620394999223437, 0.2317865255298881], [0.54017035868478769, -0.30438043357237166]], 
[[0.47697398105712696, -0.78210832331247904], [-0.56840424059294237, -0.99283720434349632], [0.4607223920914798, 
-0.16435006242306049]], [[-0.90942599512602085, -0.12147896847076889], [0.16562963252734408, -0.93836356873230242], 
[-0.53300577805949878, -0.4687624402457613]], [[-0.13951650547575722, -0.15736275503966723], [-0.78125201221835283, 
-0.086856634026029278], [0.23732350893260268, -0.44571103372306298]]], [[[-0.64858191935095366, -0.99101028268449998], 
[-0.70565615599014198, 0.22040654164816265], [-0.044918040347948107, 0.83549765131007603]], [[-0.86686852820081595, 
0.93282518490434141], [-0.067105949665969167, -0.61546887103786863], [-0.19688990696396824, -0.86568475111010135]], 
[[0.091845111612018471, 0.28437255063834943], [0.67986244720361122, 0.62985234823171865], [-0.38804666415807398, 
-0.64334402729419216]], [[0.018036867830121261, -0.076952791938787124], [-0.70787219438310989, -0.62551669097947116], 
[-0.97641750204548328, -0.71457042181838926]]], [[[-0.68470765521525001, 0.43250510058194269], [-0.63318035645221338, 
-0.93016620410572459], [-0.85641375056218627, -0.86715709080052794]], [[-0.40811641836047685, 0.38741829151287588], 
[-0.12799625797341685, 0.74271380835971024], [-0.11642557377137686, 0.81179382479499007]], [[-0.95877787795297631, 
0.27260121458743569], [-0.67916617194550066, -0.60113645890759382], [0.32552481775546593, 0.4823914387567243]], 
[[-0.43074264171436316, 0.74443017737146389], [-0.50063000672793567, -0.09055205949663736], [0.065240488269599606, 
-0.66559992391241551]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 4, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_expandedData_rank4_offset3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[0.39592710128050279, -0.55774104314693762, -0.48423470649302747, -0.76422840781865142], 
[-0.2591670352652331, 0.99541668323759569, -0.83698239192598356, 0.74084794787250785], [0.97038843848187195, 
-0.66014441118024125, 0.11601860624471194, 0.29021218171036289]], [[-0.29112522641519867, 0.90741737147807222, 
-0.78517346787461673, -0.97664265278956308], [0.601744254232355, 0.57871285223392999, 0.29960791614101967, 
0.98160587761504958], [0.095396343619639445, 0.34782416505858937, 0.98084238518207001, 0.024124785450889474]]], 
[[[0.96051202538975611, 0.0019461682597774477, 0.71371988422553634, -0.78702872554519643], [0.47726434688438446, 
-0.59697676977046155, 0.35304103751251792, -0.67589923066636959], [-0.24624962841590459, -0.1912365839852217, 
-0.73438097074611286, 0.80727872707496418]], [[-0.28840574751009385, 0.031539850580448503, 0.35788167669274817, 
-0.64808386133645568], [0.41225386549101795, -0.90820473251969558, -0.42165815233957415, 0.52738280337139676], 
[-0.26361766398849196, -0.96851439908289927, 0.7135268472416858, -0.0033944998896155987]]], [[[0.49705705693032343, 
-0.68418107882504242, 0.75534107818619245, -0.43706485262378525], [-0.12966068338916825, 0.40785296483872568, 
-0.56030259413712269, 0.70234770172700389], [-0.75927569829449792, 0.59550553155661468, 0.32470137197014548, 
0.37521249883666252]], [[-0.28366624151933939, -0.84030769669812355, -0.048771152278094254, -0.40535417224416603], 
[0.117387083061266, 0.63902057135317447, -0.15131536537197232, 0.48974233899313036], [-0.76796256562696219, 
0.30313617306578777, -0.12947794961755998, 0.87242810449971042]]]])+(1.-msk_arg)*numpy.array([[[[-0.83273561728916246, 
0.6586806237836822, 0.18717011633395564, 0.54134700314137163], [-0.17261622655109443, -0.7698271628278921, 
0.082522134120451174, -0.9686456668158625], [0.85659322772667745, 0.88258417769265685, -0.90078421567685973, 
-0.16079236971446131]], [[-0.25699388456241978, 0.59186308580113955, 0.9264685280104632, -0.04327095044831708], 
[0.5877520302531849, -0.44709173568573402, -0.047572457043194127, -0.033943337768506554], [0.55037040402299797, 
-0.82189006847061807, 0.60414396988862573, -0.75744106716307802]]], [[[0.56239368345061203, 0.8624470912009059, 
0.82182985581989865, -0.047317268453788364], [0.10771485975207495, 0.43130536883992177, 0.35241499148644229, 
0.67530416534041149], [-0.91399404846452037, 0.11770920968273502, 0.75330471839809721, 0.36386110162155516]], 
[[-0.53076510188047532, -0.71107046927406747, 0.10648505106425143, -0.52171084969585468], [0.099351602448291887, 
-0.98635073173793009, -0.17446423890589258, -0.78813890020047728], [-0.098452537069638968, 0.16796348435665065, 
0.60563862866662577, -0.22964447365612894]]], [[[-0.80865191845915207, 0.86173993844442487, 0.58683718047690014, 
0.87226774195942069], [0.49208787197621606, 0.042775033668111861, -0.52422854054434409, -0.9492660405434501], 
[0.98019752165811513, -0.33587600943403872, -0.95433847119032134, 0.38950832668615698]], [[0.94021086847410951, 
-0.84626715926659335, 0.079749886363350342, 0.93232757800797561], [0.14116136886760611, 0.11850640222575848, -0.52697310897756, 
0.34303872023440096], [0.11961662549168994, 0.98413976116020851, -0.01193045612675192, -0.84123115238322432]]]])
      res=transpose(arg,3)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.39592710128050279, -0.2591670352652331, 0.97038843848187195], [-0.29112522641519867, 
0.601744254232355, 0.095396343619639445]], [[0.96051202538975611, 0.47726434688438446, -0.24624962841590459], 
[-0.28840574751009385, 0.41225386549101795, -0.26361766398849196]], [[0.49705705693032343, -0.12966068338916825, 
-0.75927569829449792], [-0.28366624151933939, 0.117387083061266, -0.76796256562696219]]], [[[-0.55774104314693762, 
0.99541668323759569, -0.66014441118024125], [0.90741737147807222, 0.57871285223392999, 0.34782416505858937]], 
[[0.0019461682597774477, -0.59697676977046155, -0.1912365839852217], [0.031539850580448503, -0.90820473251969558, 
-0.96851439908289927]], [[-0.68418107882504242, 0.40785296483872568, 0.59550553155661468], [-0.84030769669812355, 
0.63902057135317447, 0.30313617306578777]]], [[[-0.48423470649302747, -0.83698239192598356, 0.11601860624471194], 
[-0.78517346787461673, 0.29960791614101967, 0.98084238518207001]], [[0.71371988422553634, 0.35304103751251792, 
-0.73438097074611286], [0.35788167669274817, -0.42165815233957415, 0.7135268472416858]], [[0.75534107818619245, 
-0.56030259413712269, 0.32470137197014548], [-0.048771152278094254, -0.15131536537197232, -0.12947794961755998]]], 
[[[-0.76422840781865142, 0.74084794787250785, 0.29021218171036289], [-0.97664265278956308, 0.98160587761504958, 
0.024124785450889474]], [[-0.78702872554519643, -0.67589923066636959, 0.80727872707496418], [-0.64808386133645568, 
0.52738280337139676, -0.0033944998896155987]], [[-0.43706485262378525, 0.70234770172700389, 0.37521249883666252], 
[-0.40535417224416603, 0.48974233899313036, 0.87242810449971042]]]])+(1.-msk_ref)*numpy.array([[[[-0.83273561728916246, 
-0.17261622655109443, 0.85659322772667745], [-0.25699388456241978, 0.5877520302531849, 0.55037040402299797]], 
[[0.56239368345061203, 0.10771485975207495, -0.91399404846452037], [-0.53076510188047532, 0.099351602448291887, 
-0.098452537069638968]], [[-0.80865191845915207, 0.49208787197621606, 0.98019752165811513], [0.94021086847410951, 
0.14116136886760611, 0.11961662549168994]]], [[[0.6586806237836822, -0.7698271628278921, 0.88258417769265685], 
[0.59186308580113955, -0.44709173568573402, -0.82189006847061807]], [[0.8624470912009059, 0.43130536883992177, 
0.11770920968273502], [-0.71107046927406747, -0.98635073173793009, 0.16796348435665065]], [[0.86173993844442487, 
0.042775033668111861, -0.33587600943403872], [-0.84626715926659335, 0.11850640222575848, 0.98413976116020851]]], 
[[[0.18717011633395564, 0.082522134120451174, -0.90078421567685973], [0.9264685280104632, -0.047572457043194127, 
0.60414396988862573]], [[0.82182985581989865, 0.35241499148644229, 0.75330471839809721], [0.10648505106425143, 
-0.17446423890589258, 0.60563862866662577]], [[0.58683718047690014, -0.52422854054434409, -0.95433847119032134], 
[0.079749886363350342, -0.52697310897756, -0.01193045612675192]]], [[[0.54134700314137163, -0.9686456668158625, 
-0.16079236971446131], [-0.04327095044831708, -0.033943337768506554, -0.75744106716307802]], [[-0.047317268453788364, 
0.67530416534041149, 0.36386110162155516], [-0.52171084969585468, -0.78813890020047728, -0.22964447365612894]], 
[[0.87226774195942069, -0.9492660405434501, 0.38950832668615698], [0.93232757800797561, 0.34303872023440096, 
-0.84123115238322432]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 3, 2, 3),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_transpose_expandedData_rank4_offset4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[-0.2887881272484023, 0.74426268753942648, -0.38207556675833709, 0.93962570999638517], 
[0.89914452045123117, -0.52205150253636878, -0.87339146333207962, 0.53102175678165375], [0.97580653435389775, 
-0.9924440563388508, -0.37428004205712595, -0.85181014506736719]], [[-0.10775161595962346, -0.37249464097912033, 
0.28208994801539844, 0.061130088056791765], [0.30195631702268266, -0.16583763888354164, -0.97867087128715458, 
-0.58962041494134931], [-0.065300617573456732, 0.53129715538152977, -0.59354562708913816, 0.5955350885659898]]], 
[[[-0.12768828171755642, -0.64344564637443225, 0.00041882187634478818, -0.90422814190515499], [0.17818119430387558, 
-0.39592286679222899, -0.83256288489578312, -0.25353409099106661], [-0.84036525941403162, 0.88554835214508199, 
-0.18343024186529222, -0.14450972118065053]], [[-0.34296136916207498, 0.048038866964399629, 0.39647892864482737, 
0.35134637853215223], [-0.58816156012573217, 0.81975064772632855, 0.67476942636192705, -0.71470871095361899], 
[0.43140215734124698, 0.59194835107785315, -0.5254525872432696, 0.53323435175949152]]], [[[0.37404874333837612, 
0.15221366089615307, 0.43521784267772845, 0.93586074353369186], [-0.90888998779870955, 0.5873848040141183, 
-0.096913663065009814, -0.17799517388146757], [-0.89339508566080172, 0.86762334671283337, -0.63949213166531083, 
-0.18193114583501435]], [[0.82251084728840196, -0.31016097089441619, 0.78686218528495977, 0.14291170408246434], 
[0.23319449634742218, 0.21560170960572589, 0.53962105610944455, 0.60254315510124368], [0.10366999328239435, 
0.89482068783616908, -0.58836794833524442, -0.022649100704941638]]]])+(1.-msk_arg)*numpy.array([[[[-0.86440938734279071, 
-0.92148805703114012, -0.97051028558980423, 0.07731340266916753], [-0.082715775651232004, -0.87945548490614667, 
-0.53848213872234818, 0.47652773294521356], [0.28234520244924721, -0.12575283406927817, -0.28522031046177254, 
0.84033965844064062]], [[0.70162141253130694, -0.36972128524914005, 0.21909956046008516, -0.41729088103127121], 
[0.61232899687452136, 0.60630914772742117, -0.14216066581562137, 0.57629325797362307], [-0.34650193429693932, 
0.022112391216752014, -0.10598185708451169, 0.8108202951807153]]], [[[0.94781931334254965, 0.48874515270778129, 
0.3393589193061628, -0.92046912555111193], [0.77516056791542987, 0.18072948492888896, 0.74564033949784325, 0.3231517237261794], 
[-0.049803680160861452, -0.17206431542097467, 0.44534178424231086, 0.81847192196951046]], [[0.57009609070660883, 
-0.048698196884315426, 0.17414550399780304, -0.95590354378347953], [-0.93036754049019588, -0.14906649502473979, 
0.31851415471056588, 0.45310922579005752], [0.094408068484628682, -0.030156046538578174, 0.72909476175084365, 
-0.50299358271168582]]], [[[-0.027782396450896307, 0.8553622518488293, 0.51679349337720648, 0.74381861370213342], 
[0.94475939941939169, -0.58336416771290089, 0.48903975537195077, 0.45310262303724458], [0.32662445807114171, 
-0.20292565811781316, -0.86994585276450453, 0.92686367679200621]], [[-0.68390887094225561, -0.75725275402307712, 
0.18026387890297046, 0.322381828072136], [-0.33014455949624377, -0.57395906759854931, 0.90470887939455569, 
-0.99921393713390683], [0.70224434842276695, 0.78298323516401869, -0.70889293233578865, 0.095901983614461672]]]])
      res=transpose(arg,4)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.2887881272484023, 0.74426268753942648, -0.38207556675833709, 0.93962570999638517], 
[0.89914452045123117, -0.52205150253636878, -0.87339146333207962, 0.53102175678165375], [0.97580653435389775, 
-0.9924440563388508, -0.37428004205712595, -0.85181014506736719]], [[-0.10775161595962346, -0.37249464097912033, 
0.28208994801539844, 0.061130088056791765], [0.30195631702268266, -0.16583763888354164, -0.97867087128715458, 
-0.58962041494134931], [-0.065300617573456732, 0.53129715538152977, -0.59354562708913816, 0.5955350885659898]]], 
[[[-0.12768828171755642, -0.64344564637443225, 0.00041882187634478818, -0.90422814190515499], [0.17818119430387558, 
-0.39592286679222899, -0.83256288489578312, -0.25353409099106661], [-0.84036525941403162, 0.88554835214508199, 
-0.18343024186529222, -0.14450972118065053]], [[-0.34296136916207498, 0.048038866964399629, 0.39647892864482737, 
0.35134637853215223], [-0.58816156012573217, 0.81975064772632855, 0.67476942636192705, -0.71470871095361899], 
[0.43140215734124698, 0.59194835107785315, -0.5254525872432696, 0.53323435175949152]]], [[[0.37404874333837612, 
0.15221366089615307, 0.43521784267772845, 0.93586074353369186], [-0.90888998779870955, 0.5873848040141183, 
-0.096913663065009814, -0.17799517388146757], [-0.89339508566080172, 0.86762334671283337, -0.63949213166531083, 
-0.18193114583501435]], [[0.82251084728840196, -0.31016097089441619, 0.78686218528495977, 0.14291170408246434], 
[0.23319449634742218, 0.21560170960572589, 0.53962105610944455, 0.60254315510124368], [0.10366999328239435, 
0.89482068783616908, -0.58836794833524442, -0.022649100704941638]]]])+(1.-msk_ref)*numpy.array([[[[-0.86440938734279071, 
-0.92148805703114012, -0.97051028558980423, 0.07731340266916753], [-0.082715775651232004, -0.87945548490614667, 
-0.53848213872234818, 0.47652773294521356], [0.28234520244924721, -0.12575283406927817, -0.28522031046177254, 
0.84033965844064062]], [[0.70162141253130694, -0.36972128524914005, 0.21909956046008516, -0.41729088103127121], 
[0.61232899687452136, 0.60630914772742117, -0.14216066581562137, 0.57629325797362307], [-0.34650193429693932, 
0.022112391216752014, -0.10598185708451169, 0.8108202951807153]]], [[[0.94781931334254965, 0.48874515270778129, 
0.3393589193061628, -0.92046912555111193], [0.77516056791542987, 0.18072948492888896, 0.74564033949784325, 0.3231517237261794], 
[-0.049803680160861452, -0.17206431542097467, 0.44534178424231086, 0.81847192196951046]], [[0.57009609070660883, 
-0.048698196884315426, 0.17414550399780304, -0.95590354378347953], [-0.93036754049019588, -0.14906649502473979, 
0.31851415471056588, 0.45310922579005752], [0.094408068484628682, -0.030156046538578174, 0.72909476175084365, 
-0.50299358271168582]]], [[[-0.027782396450896307, 0.8553622518488293, 0.51679349337720648, 0.74381861370213342], 
[0.94475939941939169, -0.58336416771290089, 0.48903975537195077, 0.45310262303724458], [0.32662445807114171, 
-0.20292565811781316, -0.86994585276450453, 0.92686367679200621]], [[-0.68390887094225561, -0.75725275402307712, 
0.18026387890297046, 0.322381828072136], [-0.33014455949624377, -0.57395906759854931, 0.90470887939455569, 
-0.99921393713390683], [0.70224434842276695, 0.78298323516401869, -0.70889293233578865, 0.095901983614461672]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_constData_dim1(self):
      arg=Data(numpy.array([[0.18644177194421618]]),self.functionspace)
      res=eigenvalues(arg)
      ref=Data(numpy.array([0.18644177194421618]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(1,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_constData_dim2(self):
      arg=Data(numpy.array([[-0.81762199211802677, 0.45850501297427182], [0.45850501297427182, 
-0.47124343125083223]]),self.functionspace)
      res=eigenvalues(arg)
      ref=Data(numpy.array([-1.1345565468318792, -0.15430887653697989]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_constData_dim3(self):
      arg=Data(numpy.array([[0.94061304998883721, 0.26537839728361134, 0.23504527017785981], [0.26537839728361134, 
-0.5459884098403629, 0.2630475033998162], [0.23504527017785981, 0.2630475033998162, -0.17078741356007954]]),self.functionspace)
      res=eigenvalues(arg)
      ref=Data(numpy.array([-0.69269722708749937, -0.13414705681961808, 1.0506815104955125]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_constData_zero_dim1(self):
      arg=Data(numpy.array([[0.]]),self.functionspace)
      res=eigenvalues(arg)
      ref=Data(numpy.array([0.]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(1,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_constData_zero_dim2(self):
      arg=Data(numpy.array([[0., 0.], [0.,0.]]),self.functionspace)
      res=eigenvalues(arg)
      ref=Data(numpy.array([0., 0.]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_constData_zero_dim3(self):
      arg=Data(numpy.array([[0., 0., 0.], [0., 0., 0.], [0., 0., 0.]]),self.functionspace)
      res=eigenvalues(arg)
      ref=Data(numpy.array([0., 0., 0.]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_expandedData_dim1(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[-0.23193456556887959]])+(1.-msk_arg)*numpy.array([[-0.96174181563514316]])
      res=eigenvalues(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-0.23193456556887959])+(1.-msk_ref)*numpy.array([-0.96174181563514316])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(1,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_expandedData_dim2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[0.50169800187705049, -0.25726673908249542], [-0.25726673908249542, 
0.31046458149750822]])+(1.-msk_arg)*numpy.array([[-0.20084507113079897, 0.043607881117532221], [0.043607881117532221, 
-0.26818550790921791]])
      res=eigenvalues(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.13162049247242708, 0.68054209090213169])+(1.-msk_ref)*numpy.array([-0.28960912676100919, 
-0.17942145227900769])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_expandedData_dim3(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[0.77521278873012034, -0.068861371980342923, -0.17496467817334527], [-0.068861371980342923, 
0.51501928025302446, 0.17204057959868801], [-0.17496467817334527, 0.17204057959868801, 
-0.87053006022942903]])+(1.-msk_arg)*numpy.array([[-0.76245139703149523, -0.32739735199553155, -0.40491609402756901], 
[-0.32739735199553155, 0.031612520130607225, 0.027781414946789096], [-0.40491609402756901, 0.027781414946789096, 
-0.80918893919540325]])
      res=eigenvalues(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-0.90786283754763031, 0.50806128358627256, 
0.81950356271507419])+(1.-msk_ref)*numpy.array([-1.2268520410033508, -0.49329882636432321, 0.18012305127138231])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_symmetric_array_rank2(self):
      arg=numpy.array([[0.0, 1.0, 2.0], [3.0, 4.0, 5.0], [6.0, 7.0, 8.0]])
      res=symmetric(arg)
      ref=numpy.array([[0.0, 2.0, 4.0], [2.0, 4.0, 6.0], [4.0, 6.0, 8.0]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 3),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_symmetric_array_rank4(self):
      arg=numpy.array([[[[0.0, 1.0, 2.0], [3.0, 4.0, 5.0]], [[6.0, 7.0, 8.0], [9.0, 10.0, 11.0]], [[12.0, 13.0, 14.0], [15.0, 
16.0, 17.0]]], [[[18.0, 19.0, 20.0], [21.0, 22.0, 23.0]], [[24.0, 25.0, 26.0], [27.0, 28.0, 29.0]], [[30.0, 31.0, 32.0], [33.0, 
34.0, 35.0]]]])
      res=symmetric(arg)
      ref=numpy.array([[[[0.0, 3.5, 7.0], [10.5, 14.0, 17.5]], [[3.5, 7.0, 10.5], [14.0, 17.5, 21.0]], [[7.0, 10.5, 14.0], 
[17.5, 21.0, 24.5]]], [[[10.5, 14.0, 17.5], [21.0, 24.5, 28.0]], [[14.0, 17.5, 21.0], [24.5, 28.0, 31.5]], [[17.5, 21.0, 24.5], 
[28.0, 31.5, 35.0]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2, 3, 2, 3),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_symmetric_constData_rank2(self):
      arg=Data(numpy.array([[0.0, 1.0, 2.0], [3.0, 4.0, 5.0], [6.0, 7.0, 8.0]]),self.functionspace)
      res=symmetric(arg)
      ref=Data(numpy.array([[0.0, 2.0, 4.0], [2.0, 4.0, 6.0], [4.0, 6.0, 8.0]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 3),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_symmetric_constData_rank4(self):
      arg=Data(numpy.array([[[[0.0, 1.0, 2.0], [3.0, 4.0, 5.0]], [[6.0, 7.0, 8.0], [9.0, 10.0, 11.0]], [[12.0, 13.0, 14.0], 
[15.0, 16.0, 17.0]]], [[[18.0, 19.0, 20.0], [21.0, 22.0, 23.0]], [[24.0, 25.0, 26.0], [27.0, 28.0, 29.0]], [[30.0, 31.0, 32.0], 
[33.0, 34.0, 35.0]]]]),self.functionspace)
      res=symmetric(arg)
      ref=Data(numpy.array([[[[0.0, 3.5, 7.0], [10.5, 14.0, 17.5]], [[3.5, 7.0, 10.5], [14.0, 17.5, 21.0]], [[7.0, 10.5, 14.0], 
[17.5, 21.0, 24.5]]], [[[10.5, 14.0, 17.5], [21.0, 24.5, 28.0]], [[14.0, 17.5, 21.0], [24.5, 28.0, 31.5]], [[17.5, 21.0, 24.5], 
[28.0, 31.5, 35.0]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 3, 2, 3),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_symmetric_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[0.0, 1.0, 2.0], [3.0, 4.0, 5.0], [6.0, 7.0, 8.0]])+(1.-msk_arg)*numpy.array([[-0.0, -1.0, 
-2.0], [-3.0, -4.0, -5.0], [-6.0, -7.0, -8.0]])
      res=symmetric(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.0, 2.0, 4.0], [2.0, 4.0, 6.0], [4.0, 6.0, 8.0]])+(1.-msk_ref)*numpy.array([[-0.0, -2.0, 
-4.0], [-2.0, -4.0, -6.0], [-4.0, -6.0, -8.0]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 3),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_symmetric_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[0.0, 1.0, 2.0], [3.0, 4.0, 5.0]], [[6.0, 7.0, 8.0], [9.0, 10.0, 11.0]], [[12.0, 13.0, 14.0], 
[15.0, 16.0, 17.0]]], [[[18.0, 19.0, 20.0], [21.0, 22.0, 23.0]], [[24.0, 25.0, 26.0], [27.0, 28.0, 29.0]], [[30.0, 31.0, 32.0], 
[33.0, 34.0, 35.0]]]])+(1.-msk_arg)*numpy.array([[[[-0.0, -1.0, -2.0], [-3.0, -4.0, -5.0]], [[-6.0, -7.0, -8.0], [-9.0, -10.0, 
-11.0]], [[-12.0, -13.0, -14.0], [-15.0, -16.0, -17.0]]], [[[-18.0, -19.0, -20.0], [-21.0, -22.0, -23.0]], [[-24.0, -25.0, 
-26.0], [-27.0, -28.0, -29.0]], [[-30.0, -31.0, -32.0], [-33.0, -34.0, -35.0]]]])
      res=symmetric(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.0, 3.5, 7.0], [10.5, 14.0, 17.5]], [[3.5, 7.0, 10.5], [14.0, 17.5, 21.0]], [[7.0, 10.5, 
14.0], [17.5, 21.0, 24.5]]], [[[10.5, 14.0, 17.5], [21.0, 24.5, 28.0]], [[14.0, 17.5, 21.0], [24.5, 28.0, 31.5]], [[17.5, 21.0, 
24.5], [28.0, 31.5, 35.0]]]])+(1.-msk_ref)*numpy.array([[[[-0.0, -3.5, -7.0], [-10.5, -14.0, -17.5]], [[-3.5, -7.0, -10.5], 
[-14.0, -17.5, -21.0]], [[-7.0, -10.5, -14.0], [-17.5, -21.0, -24.5]]], [[[-10.5, -14.0, -17.5], [-21.0, -24.5, -28.0]], 
[[-14.0, -17.5, -21.0], [-24.5, -28.0, -31.5]], [[-17.5, -21.0, -24.5], [-28.0, -31.5, -35.0]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 3, 2, 3),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_antisymmetric_array_rank2(self):
      arg=numpy.array([[0.0, 1.0, 2.0], [3.0, 4.0, 5.0], [6.0, 7.0, 8.0]])
      res=antisymmetric(arg)
      ref=numpy.array([[0.0, -1.0, -2.0], [1.0, 0.0, -1.0], [2.0, 1.0, 0.0]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 3),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_antisymmetric_array_rank4(self):
      arg=numpy.array([[[[0.0, 1.0, 2.0], [3.0, 4.0, 5.0]], [[6.0, 7.0, 8.0], [9.0, 10.0, 11.0]], [[12.0, 13.0, 14.0], [15.0, 
16.0, 17.0]]], [[[18.0, 19.0, 20.0], [21.0, 22.0, 23.0]], [[24.0, 25.0, 26.0], [27.0, 28.0, 29.0]], [[30.0, 31.0, 32.0], [33.0, 
34.0, 35.0]]]])
      res=antisymmetric(arg)
      ref=numpy.array([[[[0.0, -2.5, -5.0], [-7.5, -10.0, -12.5]], [[2.5, 0.0, -2.5], [-5.0, -7.5, -10.0]], [[5.0, 2.5, 0.0], 
[-2.5, -5.0, -7.5]]], [[[7.5, 5.0, 2.5], [0.0, -2.5, -5.0]], [[10.0, 7.5, 5.0], [2.5, 0.0, -2.5]], [[12.5, 10.0, 7.5], [5.0, 
2.5, 0.0]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2, 3, 2, 3),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_antisymmetric_constData_rank2(self):
      arg=Data(numpy.array([[0.0, 1.0, 2.0], [3.0, 4.0, 5.0], [6.0, 7.0, 8.0]]),self.functionspace)
      res=antisymmetric(arg)
      ref=Data(numpy.array([[0.0, -1.0, -2.0], [1.0, 0.0, -1.0], [2.0, 1.0, 0.0]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 3),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_antisymmetric_constData_rank4(self):
      arg=Data(numpy.array([[[[0.0, 1.0, 2.0], [3.0, 4.0, 5.0]], [[6.0, 7.0, 8.0], [9.0, 10.0, 11.0]], [[12.0, 13.0, 14.0], 
[15.0, 16.0, 17.0]]], [[[18.0, 19.0, 20.0], [21.0, 22.0, 23.0]], [[24.0, 25.0, 26.0], [27.0, 28.0, 29.0]], [[30.0, 31.0, 32.0], 
[33.0, 34.0, 35.0]]]]),self.functionspace)
      res=antisymmetric(arg)
      ref=Data(numpy.array([[[[0.0, -2.5, -5.0], [-7.5, -10.0, -12.5]], [[2.5, 0.0, -2.5], [-5.0, -7.5, -10.0]], [[5.0, 2.5, 
0.0], [-2.5, -5.0, -7.5]]], [[[7.5, 5.0, 2.5], [0.0, -2.5, -5.0]], [[10.0, 7.5, 5.0], [2.5, 0.0, -2.5]], [[12.5, 10.0, 7.5], 
[5.0, 2.5, 0.0]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 3, 2, 3),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_antisymmetric_expandedData_rank2(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[0.0, 1.0, 2.0], [3.0, 4.0, 5.0], [6.0, 7.0, 8.0]])+(1.-msk_arg)*numpy.array([[-0.0, -1.0, 
-2.0], [-3.0, -4.0, -5.0], [-6.0, -7.0, -8.0]])
      res=antisymmetric(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.0, -1.0, -2.0], [1.0, 0.0, -1.0], [2.0, 1.0, 0.0]])+(1.-msk_ref)*numpy.array([[0.0, 1.0, 
2.0], [-1.0, 0.0, 1.0], [-2.0, -1.0, 0.0]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 3),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_antisymmetric_expandedData_rank4(self):
      msk_arg=whereNegative(self.functionspace.getX()[0]-0.5)
      arg=msk_arg*numpy.array([[[[0.0, 1.0, 2.0], [3.0, 4.0, 5.0]], [[6.0, 7.0, 8.0], [9.0, 10.0, 11.0]], [[12.0, 13.0, 14.0], 
[15.0, 16.0, 17.0]]], [[[18.0, 19.0, 20.0], [21.0, 22.0, 23.0]], [[24.0, 25.0, 26.0], [27.0, 28.0, 29.0]], [[30.0, 31.0, 32.0], 
[33.0, 34.0, 35.0]]]])+(1.-msk_arg)*numpy.array([[[[-0.0, -1.0, -2.0], [-3.0, -4.0, -5.0]], [[-6.0, -7.0, -8.0], [-9.0, -10.0, 
-11.0]], [[-12.0, -13.0, -14.0], [-15.0, -16.0, -17.0]]], [[[-18.0, -19.0, -20.0], [-21.0, -22.0, -23.0]], [[-24.0, -25.0, 
-26.0], [-27.0, -28.0, -29.0]], [[-30.0, -31.0, -32.0], [-33.0, -34.0, -35.0]]]])
      res=antisymmetric(arg)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.0, -2.5, -5.0], [-7.5, -10.0, -12.5]], [[2.5, 0.0, -2.5], [-5.0, -7.5, -10.0]], [[5.0, 2.5, 
0.0], [-2.5, -5.0, -7.5]]], [[[7.5, 5.0, 2.5], [0.0, -2.5, -5.0]], [[10.0, 7.5, 5.0], [2.5, 0.0, -2.5]], [[12.5, 10.0, 7.5], 
[5.0, 2.5, 0.0]]]])+(1.-msk_ref)*numpy.array([[[[0.0, 2.5, 5.0], [7.5, 10.0, 12.5]], [[-2.5, 0.0, 2.5], [5.0, 7.5, 10.0]], 
[[-5.0, -2.5, 0.0], [2.5, 5.0, 7.5]]], [[[-7.5, -5.0, -2.5], [0.0, 2.5, 5.0]], [[-10.0, -7.5, -5.0], [-2.5, 0.0, 2.5]], 
[[-12.5, -10.0, -7.5], [-5.0, -2.5, 0.0]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 3, 2, 3),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim2_case0(self):
      arg=Data(numpy.array([[0.0, 0.0], [0.0, 0.0]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([0.0, 0.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(2,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(2, 2),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(2))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim2_case3(self):
      arg=Data(numpy.array([[-1.0, 0.0], [0.0, -1.0]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([-1.0, -1.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(2,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(2, 2),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(2))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim2_case5(self):
      arg=Data(numpy.array([[-0.99999999999999967, -6.4606252205695602e-16], [-6.4606252205695602e-16, 
-0.99999999999999967]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([-1.0, -1.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(2,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(2, 2),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(2))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim2_case6(self):
      arg=Data(numpy.array([[0.0, 0.0], [0.0, 0.0001]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([0.0, 0.0001]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(2,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(2, 2),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(2))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim2_case7(self):
      arg=Data(numpy.array([[0.0001, 0.0], [0.0, 0.0]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([0.0, 0.0001]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(2,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(2, 2),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(2))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim2_case8(self):
      arg=Data(numpy.array([[6.0598371831785722e-06, 2.3859213977648625e-05], [2.3859213977648629e-05, 
9.3940162816821425e-05]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([0.0, 0.0001]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(2,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(2, 2),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(2))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim2_case9(self):
      arg=Data(numpy.array([[1.0, 0.0], [0.0, 2.0]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([1.0, 2.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(2,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(2, 2),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(2))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim2_case10(self):
      arg=Data(numpy.array([[2.0, 0.0], [0.0, 1.0]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([1.0, 2.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(2,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(2, 2),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(2))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim2_case11(self):
      arg=Data(numpy.array([[1.0605983718317855, 0.23859213977648688], [0.23859213977648688, 
1.9394016281682138]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([1.0, 2.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(2,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(2, 2),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(2))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim2_case12(self):
      arg=Data(numpy.array([[1.0, 0.0], [0.0, 1000000.0]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([1.0, 1000000.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(2,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(2, 2),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(2))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim2_case13(self):
      arg=Data(numpy.array([[1000000.0, 0.0], [0.0, 1.0]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([1.0, 1000000.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(2,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(2, 2),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(2))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim2_case14(self):
      arg=Data(numpy.array([[60599.311233413886, 238591.90118434647], [238591.90118434647, 
939401.68876658613]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([1.0, 1000000.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(2,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(2, 2),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(2))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_expandedData_dim2_case0(self):
      msk_arg=whereZero(self.functionspace.getX()[0],1.e-8)
      arg=msk_arg*numpy.array([[0.0, 0.0], [0.0, 0.0]])+(1.-msk_arg)*numpy.array([[-1.0, 0.0], [0.0, -1.0]])
      res=eigenvalues_and_eigenvectors(arg)
      msk_ref_ev=whereZero(self.functionspace.getX()[0],1.e-8)
      ref_ev=msk_ref_ev*numpy.array([0.0, 0.0])+(1.-msk_ref_ev)*numpy.array([-1.0, -1.0])
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(2,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(2, 2),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(2))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_expandedData_dim2_case5(self):
      msk_arg=whereZero(self.functionspace.getX()[0],1.e-8)
      arg=msk_arg*numpy.array([[-0.99999999999999967, -6.4606252205695602e-16], [-6.4606252205695602e-16, 
-0.99999999999999967]])+(1.-msk_arg)*numpy.array([[0.0, 0.0], [0.0, 0.0001]])
      res=eigenvalues_and_eigenvectors(arg)
      msk_ref_ev=whereZero(self.functionspace.getX()[0],1.e-8)
      ref_ev=msk_ref_ev*numpy.array([-1.0, -1.0])+(1.-msk_ref_ev)*numpy.array([0.0, 0.0001])
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(2,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(2, 2),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(2))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_expandedData_dim2_case7(self):
      msk_arg=whereZero(self.functionspace.getX()[0],1.e-8)
      arg=msk_arg*numpy.array([[0.0001, 0.0], [0.0, 0.0]])+(1.-msk_arg)*numpy.array([[6.0598371831785722e-06, 
2.3859213977648625e-05], [2.3859213977648629e-05, 9.3940162816821425e-05]])
      res=eigenvalues_and_eigenvectors(arg)
      msk_ref_ev=whereZero(self.functionspace.getX()[0],1.e-8)
      ref_ev=msk_ref_ev*numpy.array([0.0, 0.0001])+(1.-msk_ref_ev)*numpy.array([0.0, 0.0001])
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(2,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(2, 2),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(2))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_expandedData_dim2_case9(self):
      msk_arg=whereZero(self.functionspace.getX()[0],1.e-8)
      arg=msk_arg*numpy.array([[1.0, 0.0], [0.0, 2.0]])+(1.-msk_arg)*numpy.array([[2.0, 0.0], [0.0, 1.0]])
      res=eigenvalues_and_eigenvectors(arg)
      msk_ref_ev=whereZero(self.functionspace.getX()[0],1.e-8)
      ref_ev=msk_ref_ev*numpy.array([1.0, 2.0])+(1.-msk_ref_ev)*numpy.array([1.0, 2.0])
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(2,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(2, 2),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(2))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_expandedData_dim2_case11(self):
      msk_arg=whereZero(self.functionspace.getX()[0],1.e-8)
      arg=msk_arg*numpy.array([[1.0605983718317855, 0.23859213977648688], [0.23859213977648688, 
1.9394016281682138]])+(1.-msk_arg)*numpy.array([[1.0, 0.0], [0.0, 1000000.0]])
      res=eigenvalues_and_eigenvectors(arg)
      msk_ref_ev=whereZero(self.functionspace.getX()[0],1.e-8)
      ref_ev=msk_ref_ev*numpy.array([1.0, 2.0])+(1.-msk_ref_ev)*numpy.array([1.0, 1000000.0])
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(2,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(2, 2),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(2))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_expandedData_dim2_case13(self):
      msk_arg=whereZero(self.functionspace.getX()[0],1.e-8)
      arg=msk_arg*numpy.array([[1000000.0, 0.0], [0.0, 1.0]])+(1.-msk_arg)*numpy.array([[60599.311233413886, 
238591.90118434647], [238591.90118434647, 939401.68876658613]])
      res=eigenvalues_and_eigenvectors(arg)
      msk_ref_ev=whereZero(self.functionspace.getX()[0],1.e-8)
      ref_ev=msk_ref_ev*numpy.array([1.0, 1000000.0])+(1.-msk_ref_ev)*numpy.array([1.0, 1000000.0])
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(2,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(2, 2),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(2))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim3_case0(self):
      arg=Data(numpy.array([[0.0, 0.0, 0.0], [0.0, 0.0, 0.0], [0.0, 0.0, 0.0]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([0.0, 0.0, 0.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim3_case5(self):
      arg=Data(numpy.array([[10.0, 0.0, 0.0], [0.0, 10.0, 0.0], [0.0, 0.0, 10.0]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([10.0, 10.0, 10.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim3_case10(self):
      arg=Data(numpy.array([[0.9, 0.0, 0.0], [0.0, 0.9, 0.0], [0.0, 0.0, 1.0]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([0.9, 0.9, 1.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim3_case11(self):
      arg=Data(numpy.array([[0.9, 0.0, 0.0], [0.0, 0.97060899725040983, -0.045555123008643325], [0.0, -0.045555123008643339, 
0.92939100274959041]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([0.9, 0.9, 1.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim3_case12(self):
      arg=Data(numpy.array([[0.92694799760252555, 0.0, 0.044368966468320177], [0.0, 0.9, 0.0], [0.044368966468320184, 0.0, 
0.97305200239747425]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([0.9, 0.9, 1.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim3_case13(self):
      arg=Data(numpy.array([[1.0, 0.0, 0.0], [0.0, 0.9, 0.0], [0.0, 0.0, 0.9]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([0.9, 0.9, 1.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim3_case14(self):
      arg=Data(numpy.array([[0.92379770619813639, 0.041031106298491521, -0.011396846732439278], [0.041031106298491535, 
0.97074428392640366, -0.019650012730342326], [-0.011396846732439236, -0.019650012730342337, 
0.90545800987545966]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([0.9, 0.9, 1.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim3_case15(self):
      arg=Data(numpy.array([[1.0, 0.0, 0.0], [0.0, 1.1, 0.0], [0.0, 0.0, 1.1]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([1.0, 1.1, 1.1]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim3_case17(self):
      arg=Data(numpy.array([[1.0269479976025255, 0.0, 0.044368966468320309], [0.0, 1.1, 0.0], [0.044368966468320295, 0.0, 
1.0730520023974743]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([1.0, 1.1, 1.1]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim3_case18(self):
      arg=Data(numpy.array([[1.1, 0.0, 0.0], [0.0, 1.0153410887977139, -0.036038311201720394], [0.0, -0.036038311201720373, 
1.084658911202286]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([1.0, 1.1, 1.1]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim3_case19(self):
      arg=Data(numpy.array([[1.035487967756175, 0.026317079185831614, -0.039960133424212368], [0.026317079185831618, 
1.0892641940924184, 0.016301362071911414], [-0.039960133424212355, 0.016301362071911431, 
1.0752478381514063]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([1.0, 1.1, 1.1]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim3_case20(self):
      arg=Data(numpy.array([[1.0, 0.0, 0.0], [0.0, 2.0, 0.0], [0.0, 0.0, 3.0]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([1.0, 2.0, 3.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim3_case21(self):
      arg=Data(numpy.array([[1.0, 0.0, 0.0], [0.0, 2.7060899725040968, -0.45555123008643206], [0.0, -0.45555123008643228, 
2.2939100274959037]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([1.0, 2.0, 3.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim3_case22(self):
      arg=Data(numpy.array([[1.5389599520505153, 0.0, 0.88737932936638753], [0.0, 2.0, 0.0], [0.88737932936638753, 0.0, 
2.4610400479494858]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([1.0, 2.0, 3.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim3_case23(self):
      arg=Data(numpy.array([[3.0, 0.0, 0.0], [0.0, 1.153410887977139, -0.36038311201720391], [0.0, -0.36038311201720391, 
1.8465891120228608]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([1.0, 2.0, 3.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim3_case24(self):
      arg=Data(numpy.array([[1.5928567395431172, 0.67348185484323142, -0.51356980156651744], [0.67348185484323153, 
2.6000847801882254, -0.033486506584313548], [-0.51356980156651744, -0.033486506584313541, 
1.8070584802686565]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([1.0, 2.0, 3.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim3_case25(self):
      arg=Data(numpy.array([[1.0, 0.0, 0.0], [0.0, 2.0, 0.0], [0.0, 0.0, 30000.0]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([1.0, 2.0, 30000.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim3_case26(self):
      arg=Data(numpy.array([[1.0, 0.0, 0.0], [0.0, 21183.286995177881, -13665.625800132779], [0.0, -13665.625800132779, 
8818.7130048221279]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([1.0, 2.0, 30000.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim3_case27(self):
      arg=Data(numpy.array([[8085.1298007817086, 0.0, 13310.246250831115], [0.0, 2.0, 0.0], [13310.246250831115, 0.0, 
21915.870199218316]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([1.0, 2.0, 30000.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim3_case28(self):
      arg=Data(numpy.array([[30000.0, 0.0, 0.0], [0.0, 1.153410887977139, -0.36038311201720391], [0.0, -0.36038311201720391, 
1.8465891120228608]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([1.0, 2.0, 30000.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim3_case29(self):
      arg=Data(numpy.array([[7140.1907849945546, 12308.774438213351, -3419.2256841313947], [12308.774438213351, 
21223.762934183575, -5894.4478052274408], [-3419.2256841313947, -5894.4478052274408, 1639.0462808218595]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([1.0, 2.0, 30000.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_expandedData_dim3_case0(self):
      msk_arg=whereZero(self.functionspace.getX()[0],1.e-8)
      arg=msk_arg*numpy.array([[0.0, 0.0, 0.0], [0.0, 0.0, 0.0], [0.0, 0.0, 0.0]])+(1.-msk_arg)*numpy.array([[10.0, 0.0, 0.0], 
[0.0, 10.0, 0.0], [0.0, 0.0, 10.0]])
      res=eigenvalues_and_eigenvectors(arg)
      msk_ref_ev=whereZero(self.functionspace.getX()[0],1.e-8)
      ref_ev=msk_ref_ev*numpy.array([0.0, 0.0, 0.0])+(1.-msk_ref_ev)*numpy.array([10.0, 10.0, 10.0])
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_expandedData_dim3_case10(self):
      msk_arg=whereZero(self.functionspace.getX()[0],1.e-8)
      arg=msk_arg*numpy.array([[0.9, 0.0, 0.0], [0.0, 0.9, 0.0], [0.0, 0.0, 1.0]])+(1.-msk_arg)*numpy.array([[0.9, 0.0, 0.0], 
[0.0, 0.97060899725040983, -0.045555123008643325], [0.0, -0.045555123008643339, 0.92939100274959041]])
      res=eigenvalues_and_eigenvectors(arg)
      msk_ref_ev=whereZero(self.functionspace.getX()[0],1.e-8)
      ref_ev=msk_ref_ev*numpy.array([0.9, 0.9, 1.0])+(1.-msk_ref_ev)*numpy.array([0.9, 0.9, 1.0])
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_expandedData_dim3_case12(self):
      msk_arg=whereZero(self.functionspace.getX()[0],1.e-8)
      arg=msk_arg*numpy.array([[0.92694799760252555, 0.0, 0.044368966468320177], [0.0, 0.9, 0.0], [0.044368966468320184, 0.0, 
0.97305200239747425]])+(1.-msk_arg)*numpy.array([[1.0, 0.0, 0.0], [0.0, 0.9, 0.0], [0.0, 0.0, 0.9]])
      res=eigenvalues_and_eigenvectors(arg)
      msk_ref_ev=whereZero(self.functionspace.getX()[0],1.e-8)
      ref_ev=msk_ref_ev*numpy.array([0.9, 0.9, 1.0])+(1.-msk_ref_ev)*numpy.array([0.9, 0.9, 1.0])
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_expandedData_dim3_case14(self):
      msk_arg=whereZero(self.functionspace.getX()[0],1.e-8)
      arg=msk_arg*numpy.array([[0.92379770619813639, 0.041031106298491521, -0.011396846732439278], [0.041031106298491535, 
0.97074428392640366, -0.019650012730342326], [-0.011396846732439236, -0.019650012730342337, 
0.90545800987545966]])+(1.-msk_arg)*numpy.array([[1.0, 0.0, 0.0], [0.0, 1.1, 0.0], [0.0, 0.0, 1.1]])
      res=eigenvalues_and_eigenvectors(arg)
      msk_ref_ev=whereZero(self.functionspace.getX()[0],1.e-8)
      ref_ev=msk_ref_ev*numpy.array([0.9, 0.9, 1.0])+(1.-msk_ref_ev)*numpy.array([1.0, 1.1, 1.1])
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_expandedData_dim3_case17(self):
      msk_arg=whereZero(self.functionspace.getX()[0],1.e-8)
      arg=msk_arg*numpy.array([[1.0269479976025255, 0.0, 0.044368966468320309], [0.0, 1.1, 0.0], [0.044368966468320295, 0.0, 
1.0730520023974743]])+(1.-msk_arg)*numpy.array([[1.1, 0.0, 0.0], [0.0, 1.0153410887977139, -0.036038311201720394], [0.0, 
-0.036038311201720373, 1.084658911202286]])
      res=eigenvalues_and_eigenvectors(arg)
      msk_ref_ev=whereZero(self.functionspace.getX()[0],1.e-8)
      ref_ev=msk_ref_ev*numpy.array([1.0, 1.1, 1.1])+(1.-msk_ref_ev)*numpy.array([1.0, 1.1, 1.1])
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_expandedData_dim3_case19(self):
      msk_arg=whereZero(self.functionspace.getX()[0],1.e-8)
      arg=msk_arg*numpy.array([[1.035487967756175, 0.026317079185831614, -0.039960133424212368], [0.026317079185831618, 
1.0892641940924184, 0.016301362071911414], [-0.039960133424212355, 0.016301362071911431, 
1.0752478381514063]])+(1.-msk_arg)*numpy.array([[1.0, 0.0, 0.0], [0.0, 2.0, 0.0], [0.0, 0.0, 3.0]])
      res=eigenvalues_and_eigenvectors(arg)
      msk_ref_ev=whereZero(self.functionspace.getX()[0],1.e-8)
      ref_ev=msk_ref_ev*numpy.array([1.0, 1.1, 1.1])+(1.-msk_ref_ev)*numpy.array([1.0, 2.0, 3.0])
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_expandedData_dim3_case21(self):
      msk_arg=whereZero(self.functionspace.getX()[0],1.e-8)
      arg=msk_arg*numpy.array([[1.0, 0.0, 0.0], [0.0, 2.7060899725040968, -0.45555123008643206], [0.0, -0.45555123008643228, 
2.2939100274959037]])+(1.-msk_arg)*numpy.array([[1.5389599520505153, 0.0, 0.88737932936638753], [0.0, 2.0, 0.0], 
[0.88737932936638753, 0.0, 2.4610400479494858]])
      res=eigenvalues_and_eigenvectors(arg)
      msk_ref_ev=whereZero(self.functionspace.getX()[0],1.e-8)
      ref_ev=msk_ref_ev*numpy.array([1.0, 2.0, 3.0])+(1.-msk_ref_ev)*numpy.array([1.0, 2.0, 3.0])
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_expandedData_dim3_case23(self):
      msk_arg=whereZero(self.functionspace.getX()[0],1.e-8)
      arg=msk_arg*numpy.array([[3.0, 0.0, 0.0], [0.0, 1.153410887977139, -0.36038311201720391], [0.0, -0.36038311201720391, 
1.8465891120228608]])+(1.-msk_arg)*numpy.array([[1.5928567395431172, 0.67348185484323142, -0.51356980156651744], 
[0.67348185484323153, 2.6000847801882254, -0.033486506584313548], [-0.51356980156651744, -0.033486506584313541, 
1.8070584802686565]])
      res=eigenvalues_and_eigenvectors(arg)
      msk_ref_ev=whereZero(self.functionspace.getX()[0],1.e-8)
      ref_ev=msk_ref_ev*numpy.array([1.0, 2.0, 3.0])+(1.-msk_ref_ev)*numpy.array([1.0, 2.0, 3.0])
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_expandedData_dim3_case25(self):
      msk_arg=whereZero(self.functionspace.getX()[0],1.e-8)
      arg=msk_arg*numpy.array([[1.0, 0.0, 0.0], [0.0, 2.0, 0.0], [0.0, 0.0, 30000.0]])+(1.-msk_arg)*numpy.array([[1.0, 0.0, 
0.0], [0.0, 21183.286995177881, -13665.625800132779], [0.0, -13665.625800132779, 8818.7130048221279]])
      res=eigenvalues_and_eigenvectors(arg)
      msk_ref_ev=whereZero(self.functionspace.getX()[0],1.e-8)
      ref_ev=msk_ref_ev*numpy.array([1.0, 2.0, 30000.0])+(1.-msk_ref_ev)*numpy.array([1.0, 2.0, 30000.0])
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_expandedData_dim3_case27(self):
      msk_arg=whereZero(self.functionspace.getX()[0],1.e-8)
      arg=msk_arg*numpy.array([[8085.1298007817086, 0.0, 13310.246250831115], [0.0, 2.0, 0.0], [13310.246250831115, 0.0, 
21915.870199218316]])+(1.-msk_arg)*numpy.array([[30000.0, 0.0, 0.0], [0.0, 1.153410887977139, -0.36038311201720391], [0.0, 
-0.36038311201720391, 1.8465891120228608]])
      res=eigenvalues_and_eigenvectors(arg)
      msk_ref_ev=whereZero(self.functionspace.getX()[0],1.e-8)
      ref_ev=msk_ref_ev*numpy.array([1.0, 2.0, 30000.0])+(1.-msk_ref_ev)*numpy.array([1.0, 2.0, 30000.0])
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_expandedData_dim3_case29(self):
      msk_arg=whereZero(self.functionspace.getX()[0],1.e-8)
      arg=msk_arg*numpy.array([[7140.1907849945546, 12308.774438213351, -3419.2256841313947], [12308.774438213351, 
21223.762934183575, -5894.4478052274408], [-3419.2256841313947, -5894.4478052274408, 
1639.0462808218595]])+(1.-msk_arg)*numpy.array([[0.0, 0.0, 0.0], [0.0, 0.0, 0.0], [0.0, 0.0, 0.0]])
      res=eigenvalues_and_eigenvectors(arg)
      msk_ref_ev=whereZero(self.functionspace.getX()[0],1.e-8)
      ref_ev=msk_ref_ev*numpy.array([1.0, 2.0, 30000.0])+(1.-msk_ref_ev)*numpy.array([0.0, 0.0, 0.0])
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(3,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(3, 3),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,1])-res[0][1]*res[1][:,1])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 1")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,2])-res[0][2]*res[1][:,2])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 2")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(3))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim1_case0(self):
      arg=Data(numpy.array([[0.0]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([0.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(1,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(1, 1),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(1))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_constData_dim1_case1(self):
      arg=Data(numpy.array([[1.0]]),self.functionspace)
      res=eigenvalues_and_eigenvectors(arg)
      ref_ev=Data(numpy.array([1.0]),self.functionspace)
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(1,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(1, 1),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(1))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_eigenvalues_and_eigenvectors_expandedData_dim1_case0(self):
      msk_arg=whereZero(self.functionspace.getX()[0],1.e-8)
      arg=msk_arg*numpy.array([[0.0]])+(1.-msk_arg)*numpy.array([[1.0]])
      res=eigenvalues_and_eigenvectors(arg)
      msk_ref_ev=whereZero(self.functionspace.getX()[0],1.e-8)
      ref_ev=msk_ref_ev*numpy.array([0.0])+(1.-msk_ref_ev)*numpy.array([1.0])
      self.assertTrue(isinstance(res[0],Data),"wrong type of result for eigenvalues.")
      self.assertEqual(res[0].getShape(),(1,),"wrong shape of result for eigenvalues.")
      self.assertTrue(isinstance(res[1],Data),"wrong type of result for eigenvectors.")
      self.assertEqual(res[1].getShape(),(1, 1),"wrong shape of result for eigenvectors.")
      self.assertTrue(Lsup(res[0]-ref_ev)<=self.RES_TOL*Lsup(ref_ev),"wrong eigenvalues")
      self.assertTrue(Lsup(matrix_mult(arg,res[1][:,0])-res[0][0]*res[1][:,0])<=self.RES_TOL*Lsup(res[0]),"wrong eigenvector 0")
      self.assertTrue(Lsup(matrix_mult(transpose(res[1]),res[1])-kronecker(1))<=self.RES_TOL,"eigenvectors are not orthonormal")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_array_rank2_axes_01(self):
      arg=numpy.array([[1.0, 2.0, -1.0, 7.0, -3.0], [0.0, -5.0, 3.0, 7.0, 5.0], [-3.0, -3.0, 4.0, 6.0, -6.0], [-6.0, 7.0, 6.0, 
4.0, -7.0]])
      res=swap_axes(arg,axis0=0,axis1=1)
      ref=numpy.array([[1.0, 0.0, -3.0, -6.0], [2.0, -5.0, -3.0, 7.0], [-1.0, 3.0, 4.0, 6.0], [7.0, 7.0, 6.0, 4.0], [-3.0, 5.0, 
-6.0, -7.0]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(5, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_array_rank3_axes_01(self):
      arg=numpy.array([[[6.0, 3.0], [7.0, 0.0]], [[-5.0, 3.0], [5.0, 3.0]], [[-7.0, -4.0], [3.0, 4.0]], [[1.0, 3.0], [-7.0, 
-7.0]], [[-7.0, 3.0], [-6.0, -5.0]], [[-2.0, 7.0], [-1.0, 1.0]]])
      res=swap_axes(arg,axis0=0,axis1=1)
      ref=numpy.array([[[6.0, 3.0], [-5.0, 3.0], [-7.0, -4.0], [1.0, 3.0], [-7.0, 3.0], [-2.0, 7.0]], [[7.0, 0.0], [5.0, 3.0], 
[3.0, 4.0], [-7.0, -7.0], [-6.0, -5.0], [-1.0, 1.0]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2, 6, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_array_rank3_axes_02(self):
      arg=numpy.array([[[0.0, 4.0], [-3.0, -1.0]], [[2.0, 3.0], [-6.0, 2.0]], [[-6.0, -2.0], [0.0, 5.0]], [[4.0, -7.0], [5.0, 
-1.0]], [[2.0, -7.0], [-3.0, 2.0]], [[6.0, -6.0], [2.0, -1.0]]])
      res=swap_axes(arg,axis0=0,axis1=2)
      ref=numpy.array([[[0.0, 2.0, -6.0, 4.0, 2.0, 6.0], [-3.0, -6.0, 0.0, 5.0, -3.0, 2.0]], [[4.0, 3.0, -2.0, -7.0, -7.0, 
-6.0], [-1.0, 2.0, 5.0, -1.0, 2.0, -1.0]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2, 2, 6),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_array_rank3_axes_12(self):
      arg=numpy.array([[[7.0, 1.0], [-6.0, -3.0]], [[0.0, 1.0], [-3.0, 2.0]], [[-3.0, -6.0], [-3.0, 1.0]], [[-6.0, 1.0], [-7.0, 
-3.0]], [[2.0, 3.0], [5.0, -2.0]], [[-1.0, 0.0], [-2.0, 7.0]]])
      res=swap_axes(arg,axis0=1,axis1=2)
      ref=numpy.array([[[7.0, -6.0], [1.0, -3.0]], [[0.0, -3.0], [1.0, 2.0]], [[-3.0, -3.0], [-6.0, 1.0]], [[-6.0, -7.0], [1.0, 
-3.0]], [[2.0, 5.0], [3.0, -2.0]], [[-1.0, -2.0], [0.0, 7.0]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_array_rank4_axes_01(self):
      arg=numpy.array([[[[7.0, -5.0, 0.0, 1.0], [-6.0, 0.0, -6.0, 1.0], [0.0, 6.0, -6.0, -6.0]], [[7.0, -3.0, 4.0, 7.0], [5.0, 
2.0, -4.0, -3.0], [-6.0, 6.0, -4.0, 1.0]]], [[[2.0, -1.0, -1.0, 1.0], [-2.0, 6.0, -6.0, -2.0], [1.0, 2.0, 0.0, 2.0]], [[-6.0, 
-7.0, 6.0, 1.0], [-5.0, 4.0, 2.0, -6.0], [-4.0, 0.0, 3.0, -7.0]]], [[[-5.0, 1.0, 6.0, 0.0], [-7.0, -6.0, 1.0, -2.0], [-1.0, 
-5.0, 0.0, -2.0]], [[6.0, 5.0, -1.0, 3.0], [2.0, 5.0, -3.0, 0.0], [3.0, 2.0, -4.0, -2.0]]]])
      res=swap_axes(arg,axis0=0,axis1=1)
      ref=numpy.array([[[[7.0, -5.0, 0.0, 1.0], [-6.0, 0.0, -6.0, 1.0], [0.0, 6.0, -6.0, -6.0]], [[2.0, -1.0, -1.0, 1.0], 
[-2.0, 6.0, -6.0, -2.0], [1.0, 2.0, 0.0, 2.0]], [[-5.0, 1.0, 6.0, 0.0], [-7.0, -6.0, 1.0, -2.0], [-1.0, -5.0, 0.0, -2.0]]], 
[[[7.0, -3.0, 4.0, 7.0], [5.0, 2.0, -4.0, -3.0], [-6.0, 6.0, -4.0, 1.0]], [[-6.0, -7.0, 6.0, 1.0], [-5.0, 4.0, 2.0, -6.0], 
[-4.0, 0.0, 3.0, -7.0]], [[6.0, 5.0, -1.0, 3.0], [2.0, 5.0, -3.0, 0.0], [3.0, 2.0, -4.0, -2.0]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2, 3, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_array_rank4_axes_02(self):
      arg=numpy.array([[[[5.0, -7.0, 5.0, 1.0], [-3.0, -1.0, 7.0, -2.0], [-3.0, -6.0, -4.0, -6.0]], [[-2.0, 5.0, -4.0, 0.0], 
[-3.0, -1.0, -3.0, 0.0], [3.0, -2.0, 1.0, -5.0]]], [[[0.0, 6.0, -7.0, 6.0], [7.0, -5.0, 0.0, -2.0], [-5.0, -4.0, 4.0, -4.0]], 
[[-4.0, -4.0, -4.0, -4.0], [-7.0, 3.0, 0.0, 5.0], [-6.0, -7.0, 7.0, 0.0]]], [[[-7.0, -3.0, 1.0, 0.0], [0.0, 6.0, 2.0, 0.0], 
[7.0, -3.0, 4.0, 3.0]], [[3.0, -4.0, 2.0, 2.0], [1.0, -5.0, -4.0, -3.0], [1.0, 0.0, -3.0, -5.0]]]])
      res=swap_axes(arg,axis0=0,axis1=2)
      ref=numpy.array([[[[5.0, -7.0, 5.0, 1.0], [0.0, 6.0, -7.0, 6.0], [-7.0, -3.0, 1.0, 0.0]], [[-2.0, 5.0, -4.0, 0.0], [-4.0, 
-4.0, -4.0, -4.0], [3.0, -4.0, 2.0, 2.0]]], [[[-3.0, -1.0, 7.0, -2.0], [7.0, -5.0, 0.0, -2.0], [0.0, 6.0, 2.0, 0.0]], [[-3.0, 
-1.0, -3.0, 0.0], [-7.0, 3.0, 0.0, 5.0], [1.0, -5.0, -4.0, -3.0]]], [[[-3.0, -6.0, -4.0, -6.0], [-5.0, -4.0, 4.0, -4.0], [7.0, 
-3.0, 4.0, 3.0]], [[3.0, -2.0, 1.0, -5.0], [-6.0, -7.0, 7.0, 0.0], [1.0, 0.0, -3.0, -5.0]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_array_rank4_axes_03(self):
      arg=numpy.array([[[[7.0, 0.0, -3.0, 5.0], [0.0, -5.0, -2.0, 4.0], [1.0, 4.0, 4.0, 1.0]], [[1.0, 3.0, -4.0, 2.0], [-3.0, 
3.0, 7.0, -2.0], [-6.0, -2.0, 0.0, -6.0]]], [[[-2.0, -2.0, 4.0, -6.0], [3.0, -5.0, 7.0, -2.0], [7.0, -3.0, -1.0, 3.0]], [[-5.0, 
-6.0, -6.0, 1.0], [7.0, 1.0, 0.0, -1.0], [2.0, -7.0, 0.0, -3.0]]], [[[0.0, -6.0, -1.0, -1.0], [-7.0, 7.0, -2.0, 1.0], [-5.0, 
-3.0, -5.0, -3.0]], [[1.0, 0.0, 0.0, -2.0], [-6.0, -1.0, -4.0, -6.0], [-4.0, 0.0, 6.0, -6.0]]]])
      res=swap_axes(arg,axis0=0,axis1=3)
      ref=numpy.array([[[[7.0, -2.0, 0.0], [0.0, 3.0, -7.0], [1.0, 7.0, -5.0]], [[1.0, -5.0, 1.0], [-3.0, 7.0, -6.0], [-6.0, 
2.0, -4.0]]], [[[0.0, -2.0, -6.0], [-5.0, -5.0, 7.0], [4.0, -3.0, -3.0]], [[3.0, -6.0, 0.0], [3.0, 1.0, -1.0], [-2.0, -7.0, 
0.0]]], [[[-3.0, 4.0, -1.0], [-2.0, 7.0, -2.0], [4.0, -1.0, -5.0]], [[-4.0, -6.0, 0.0], [7.0, 0.0, -4.0], [0.0, 0.0, 6.0]]], 
[[[5.0, -6.0, -1.0], [4.0, -2.0, 1.0], [1.0, 3.0, -3.0]], [[2.0, 1.0, -2.0], [-2.0, -1.0, -6.0], [-6.0, -3.0, -6.0]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 2, 3, 3),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_array_rank4_axes_12(self):
      arg=numpy.array([[[[6.0, 4.0, 7.0, -6.0], [6.0, -7.0, -2.0, 1.0], [-4.0, 0.0, -5.0, 2.0]], [[0.0, -4.0, -5.0, 7.0], 
[-7.0, 0.0, -6.0, 2.0], [7.0, -3.0, 2.0, -1.0]]], [[[0.0, -1.0, -7.0, 1.0], [4.0, 4.0, 1.0, -7.0], [-1.0, 3.0, 0.0, -6.0]], 
[[3.0, 3.0, -4.0, -3.0], [-6.0, 0.0, 6.0, 0.0], [-6.0, 6.0, 6.0, 2.0]]], [[[-1.0, 5.0, 4.0, 4.0], [-4.0, -4.0, 5.0, -3.0], 
[2.0, -1.0, 2.0, 2.0]], [[2.0, 0.0, 5.0, 3.0], [-7.0, 0.0, -4.0, 0.0], [0.0, 3.0, -6.0, -1.0]]]])
      res=swap_axes(arg,axis0=1,axis1=2)
      ref=numpy.array([[[[6.0, 4.0, 7.0, -6.0], [0.0, -4.0, -5.0, 7.0]], [[6.0, -7.0, -2.0, 1.0], [-7.0, 0.0, -6.0, 2.0]], 
[[-4.0, 0.0, -5.0, 2.0], [7.0, -3.0, 2.0, -1.0]]], [[[0.0, -1.0, -7.0, 1.0], [3.0, 3.0, -4.0, -3.0]], [[4.0, 4.0, 1.0, -7.0], 
[-6.0, 0.0, 6.0, 0.0]], [[-1.0, 3.0, 0.0, -6.0], [-6.0, 6.0, 6.0, 2.0]]], [[[-1.0, 5.0, 4.0, 4.0], [2.0, 0.0, 5.0, 3.0]], 
[[-4.0, -4.0, 5.0, -3.0], [-7.0, 0.0, -4.0, 0.0]], [[2.0, -1.0, 2.0, 2.0], [0.0, 3.0, -6.0, -1.0]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 3, 2, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_array_rank4_axes_13(self):
      arg=numpy.array([[[[-1.0, 5.0, 4.0, -2.0], [-7.0, 1.0, 1.0, -2.0], [-5.0, -2.0, 6.0, -1.0]], [[7.0, -5.0, -2.0, 6.0], 
[4.0, 4.0, 0.0, -2.0], [1.0, 1.0, 3.0, -5.0]]], [[[0.0, 0.0, 3.0, 0.0], [3.0, 4.0, 4.0, 3.0], [7.0, -3.0, 4.0, -3.0]], [[1.0, 
0.0, 4.0, 0.0], [5.0, -2.0, -5.0, -7.0], [-2.0, -5.0, -7.0, -1.0]]], [[[1.0, 4.0, -6.0, 0.0], [0.0, -3.0, 0.0, -2.0], [3.0, 
-3.0, 1.0, 1.0]], [[6.0, 0.0, -7.0, 1.0], [-4.0, 6.0, -7.0, 7.0], [-4.0, 0.0, 4.0, -5.0]]]])
      res=swap_axes(arg,axis0=1,axis1=3)
      ref=numpy.array([[[[-1.0, 7.0], [-7.0, 4.0], [-5.0, 1.0]], [[5.0, -5.0], [1.0, 4.0], [-2.0, 1.0]], [[4.0, -2.0], [1.0, 
0.0], [6.0, 3.0]], [[-2.0, 6.0], [-2.0, -2.0], [-1.0, -5.0]]], [[[0.0, 1.0], [3.0, 5.0], [7.0, -2.0]], [[0.0, 0.0], [4.0, 
-2.0], [-3.0, -5.0]], [[3.0, 4.0], [4.0, -5.0], [4.0, -7.0]], [[0.0, 0.0], [3.0, -7.0], [-3.0, -1.0]]], [[[1.0, 6.0], [0.0, 
-4.0], [3.0, -4.0]], [[4.0, 0.0], [-3.0, 6.0], [-3.0, 0.0]], [[-6.0, -7.0], [0.0, -7.0], [1.0, 4.0]], [[0.0, 1.0], [-2.0, 7.0], 
[1.0, -5.0]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 4, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_array_rank4_axes_23(self):
      arg=numpy.array([[[[1.0, 1.0, -2.0, -1.0], [2.0, 4.0, 5.0, 5.0], [6.0, -2.0, 0.0, 3.0]], [[7.0, 7.0, 1.0, 7.0], [-7.0, 
7.0, -4.0, 6.0], [-3.0, 0.0, -4.0, 4.0]]], [[[0.0, 0.0, -1.0, 7.0], [-6.0, 6.0, -4.0, -4.0], [-7.0, -4.0, -5.0, -1.0]], [[-5.0, 
-5.0, -3.0, 0.0], [-1.0, 1.0, 7.0, 7.0], [-6.0, 6.0, 7.0, 0.0]]], [[[-3.0, -4.0, 0.0, 4.0], [3.0, -4.0, -7.0, 0.0], [4.0, 2.0, 
-4.0, 6.0]], [[4.0, 0.0, 0.0, 2.0], [4.0, 5.0, 2.0, -7.0], [3.0, -7.0, -4.0, 1.0]]]])
      res=swap_axes(arg,axis0=2,axis1=3)
      ref=numpy.array([[[[1.0, 2.0, 6.0], [1.0, 4.0, -2.0], [-2.0, 5.0, 0.0], [-1.0, 5.0, 3.0]], [[7.0, -7.0, -3.0], [7.0, 7.0, 
0.0], [1.0, -4.0, -4.0], [7.0, 6.0, 4.0]]], [[[0.0, -6.0, -7.0], [0.0, 6.0, -4.0], [-1.0, -4.0, -5.0], [7.0, -4.0, -1.0]], 
[[-5.0, -1.0, -6.0], [-5.0, 1.0, 6.0], [-3.0, 7.0, 7.0], [0.0, 7.0, 0.0]]], [[[-3.0, 3.0, 4.0], [-4.0, -4.0, 2.0], [0.0, -7.0, 
-4.0], [4.0, 0.0, 6.0]], [[4.0, 4.0, 3.0], [0.0, 5.0, -7.0], [0.0, 2.0, -4.0], [2.0, -7.0, 1.0]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 4, 3),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_constData_rank2_axes_01(self):
      arg=Data(numpy.array([[-6.0, 4.0, 7.0, 7.0, -5.0], [-5.0, 2.0, 7.0, 6.0, 1.0], [0.0, 3.0, -6.0, 4.0, -7.0], [-4.0, 5.0, 
0.0, 0.0, 3.0]]),self.functionspace)
      res=swap_axes(arg,axis0=0,axis1=1)
      ref=Data(numpy.array([[-6.0, -5.0, 0.0, -4.0], [4.0, 2.0, 3.0, 5.0], [7.0, 7.0, -6.0, 0.0], [7.0, 6.0, 4.0, 0.0], [-5.0, 
1.0, -7.0, 3.0]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(5, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_constData_rank3_axes_01(self):
      arg=Data(numpy.array([[[6.0, 1.0], [-2.0, -1.0]], [[0.0, 5.0], [-4.0, -1.0]], [[-7.0, -5.0], [3.0, 1.0]], [[0.0, 7.0], 
[1.0, -6.0]], [[-7.0, 0.0], [-3.0, 0.0]], [[0.0, -3.0], [6.0, -7.0]]]),self.functionspace)
      res=swap_axes(arg,axis0=0,axis1=1)
      ref=Data(numpy.array([[[6.0, 1.0], [0.0, 5.0], [-7.0, -5.0], [0.0, 7.0], [-7.0, 0.0], [0.0, -3.0]], [[-2.0, -1.0], [-4.0, 
-1.0], [3.0, 1.0], [1.0, -6.0], [-3.0, 0.0], [6.0, -7.0]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 6, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_constData_rank3_axes_02(self):
      arg=Data(numpy.array([[[-1.0, 7.0], [7.0, 2.0]], [[0.0, -6.0], [4.0, 0.0]], [[-7.0, 0.0], [-1.0, -6.0]], [[-3.0, -4.0], 
[1.0, -7.0]], [[5.0, 2.0], [-1.0, -7.0]], [[0.0, 3.0], [-5.0, 0.0]]]),self.functionspace)
      res=swap_axes(arg,axis0=0,axis1=2)
      ref=Data(numpy.array([[[-1.0, 0.0, -7.0, -3.0, 5.0, 0.0], [7.0, 4.0, -1.0, 1.0, -1.0, -5.0]], [[7.0, -6.0, 0.0, -4.0, 
2.0, 3.0], [2.0, 0.0, -6.0, -7.0, -7.0, 0.0]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 2, 6),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_constData_rank3_axes_12(self):
      arg=Data(numpy.array([[[-1.0, 4.0], [-7.0, -2.0]], [[0.0, 1.0], [4.0, 1.0]], [[4.0, 3.0], [5.0, 6.0]], [[-3.0, 3.0], 
[-2.0, -5.0]], [[0.0, 3.0], [-5.0, -6.0]], [[1.0, -6.0], [1.0, 6.0]]]),self.functionspace)
      res=swap_axes(arg,axis0=1,axis1=2)
      ref=Data(numpy.array([[[-1.0, -7.0], [4.0, -2.0]], [[0.0, 4.0], [1.0, 1.0]], [[4.0, 5.0], [3.0, 6.0]], [[-3.0, -2.0], 
[3.0, -5.0]], [[0.0, -5.0], [3.0, -6.0]], [[1.0, 1.0], [-6.0, 6.0]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_constData_rank4_axes_01(self):
      arg=Data(numpy.array([[[[0.0, 0.0, 4.0, -7.0], [-5.0, 5.0, -3.0, -1.0], [0.0, 3.0, 7.0, -2.0]], [[2.0, 0.0, 2.0, -3.0], 
[0.0, -2.0, 2.0, 7.0], [3.0, -5.0, 4.0, -1.0]]], [[[5.0, -2.0, -2.0, -4.0], [-4.0, 0.0, -1.0, 4.0], [-2.0, -3.0, 4.0, 3.0]], 
[[-4.0, -1.0, -6.0, 0.0], [-3.0, 4.0, -6.0, -3.0], [1.0, -2.0, -5.0, -2.0]]], [[[-5.0, 6.0, 2.0, 6.0], [0.0, 6.0, -5.0, 0.0], 
[7.0, -6.0, 4.0, 3.0]], [[-6.0, -7.0, 4.0, 4.0], [3.0, -7.0, -6.0, 3.0], [3.0, 3.0, -6.0, 4.0]]]]),self.functionspace)
      res=swap_axes(arg,axis0=0,axis1=1)
      ref=Data(numpy.array([[[[0.0, 0.0, 4.0, -7.0], [-5.0, 5.0, -3.0, -1.0], [0.0, 3.0, 7.0, -2.0]], [[5.0, -2.0, -2.0, -4.0], 
[-4.0, 0.0, -1.0, 4.0], [-2.0, -3.0, 4.0, 3.0]], [[-5.0, 6.0, 2.0, 6.0], [0.0, 6.0, -5.0, 0.0], [7.0, -6.0, 4.0, 3.0]]], 
[[[2.0, 0.0, 2.0, -3.0], [0.0, -2.0, 2.0, 7.0], [3.0, -5.0, 4.0, -1.0]], [[-4.0, -1.0, -6.0, 0.0], [-3.0, 4.0, -6.0, -3.0], 
[1.0, -2.0, -5.0, -2.0]], [[-6.0, -7.0, 4.0, 4.0], [3.0, -7.0, -6.0, 3.0], [3.0, 3.0, -6.0, 4.0]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 3, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_constData_rank4_axes_02(self):
      arg=Data(numpy.array([[[[-4.0, 0.0, 5.0, 0.0], [6.0, 4.0, 0.0, -7.0], [-3.0, 7.0, 2.0, -5.0]], [[4.0, -3.0, 5.0, -4.0], 
[-7.0, -6.0, 0.0, 3.0], [4.0, 6.0, -1.0, 7.0]]], [[[-3.0, -4.0, -4.0, 5.0], [0.0, 5.0, -3.0, 0.0], [-4.0, -7.0, 0.0, 4.0]], 
[[5.0, 4.0, 5.0, 5.0], [-2.0, 4.0, 2.0, 5.0], [6.0, 0.0, -5.0, -7.0]]], [[[6.0, 1.0, 7.0, -5.0], [2.0, 7.0, 5.0, -7.0], [-7.0, 
-6.0, 1.0, 6.0]], [[6.0, -4.0, 0.0, 2.0], [-6.0, -6.0, -1.0, 1.0], [-4.0, 4.0, 0.0, 1.0]]]]),self.functionspace)
      res=swap_axes(arg,axis0=0,axis1=2)
      ref=Data(numpy.array([[[[-4.0, 0.0, 5.0, 0.0], [-3.0, -4.0, -4.0, 5.0], [6.0, 1.0, 7.0, -5.0]], [[4.0, -3.0, 5.0, -4.0], 
[5.0, 4.0, 5.0, 5.0], [6.0, -4.0, 0.0, 2.0]]], [[[6.0, 4.0, 0.0, -7.0], [0.0, 5.0, -3.0, 0.0], [2.0, 7.0, 5.0, -7.0]], [[-7.0, 
-6.0, 0.0, 3.0], [-2.0, 4.0, 2.0, 5.0], [-6.0, -6.0, -1.0, 1.0]]], [[[-3.0, 7.0, 2.0, -5.0], [-4.0, -7.0, 0.0, 4.0], [-7.0, 
-6.0, 1.0, 6.0]], [[4.0, 6.0, -1.0, 7.0], [6.0, 0.0, -5.0, -7.0], [-4.0, 4.0, 0.0, 1.0]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_constData_rank4_axes_03(self):
      arg=Data(numpy.array([[[[-4.0, 5.0, 7.0, -2.0], [4.0, 1.0, -1.0, 0.0], [-6.0, 7.0, -7.0, 3.0]], [[0.0, 0.0, -2.0, 0.0], 
[3.0, 3.0, 1.0, -3.0], [-4.0, 0.0, 0.0, 2.0]]], [[[6.0, -3.0, -3.0, -3.0], [3.0, 7.0, -7.0, 5.0], [6.0, 0.0, 5.0, 5.0]], [[4.0, 
-6.0, 4.0, 0.0], [0.0, 1.0, 5.0, 0.0], [-3.0, -4.0, 7.0, 7.0]]], [[[-3.0, -1.0, -2.0, -7.0], [-2.0, 7.0, -5.0, -7.0], [-2.0, 
-2.0, -2.0, -2.0]], [[5.0, 5.0, 1.0, 5.0], [-6.0, 2.0, -4.0, 3.0], [5.0, 4.0, -3.0, 2.0]]]]),self.functionspace)
      res=swap_axes(arg,axis0=0,axis1=3)
      ref=Data(numpy.array([[[[-4.0, 6.0, -3.0], [4.0, 3.0, -2.0], [-6.0, 6.0, -2.0]], [[0.0, 4.0, 5.0], [3.0, 0.0, -6.0], 
[-4.0, -3.0, 5.0]]], [[[5.0, -3.0, -1.0], [1.0, 7.0, 7.0], [7.0, 0.0, -2.0]], [[0.0, -6.0, 5.0], [3.0, 1.0, 2.0], [0.0, -4.0, 
4.0]]], [[[7.0, -3.0, -2.0], [-1.0, -7.0, -5.0], [-7.0, 5.0, -2.0]], [[-2.0, 4.0, 1.0], [1.0, 5.0, -4.0], [0.0, 7.0, -3.0]]], 
[[[-2.0, -3.0, -7.0], [0.0, 5.0, -7.0], [3.0, 5.0, -2.0]], [[0.0, 0.0, 5.0], [-3.0, 0.0, 3.0], [2.0, 7.0, 
2.0]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 2, 3, 3),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_constData_rank4_axes_12(self):
      arg=Data(numpy.array([[[[2.0, 6.0, -7.0, -4.0], [0.0, 0.0, 0.0, 0.0], [3.0, -1.0, 0.0, -1.0]], [[-2.0, 3.0, 1.0, -1.0], 
[0.0, 0.0, 5.0, 4.0], [3.0, 0.0, 0.0, 5.0]]], [[[-2.0, 1.0, -3.0, 7.0], [-7.0, 7.0, 5.0, 2.0], [3.0, 3.0, 3.0, -1.0]], [[5.0, 
4.0, -4.0, 3.0], [0.0, 0.0, 6.0, -1.0], [-6.0, 5.0, -7.0, -5.0]]], [[[-1.0, 0.0, 6.0, -3.0], [5.0, -6.0, 7.0, -5.0], [-3.0, 
-4.0, 4.0, 1.0]], [[-6.0, 5.0, -3.0, -6.0], [7.0, 0.0, 0.0, -2.0], [5.0, -5.0, 3.0, 0.0]]]]),self.functionspace)
      res=swap_axes(arg,axis0=1,axis1=2)
      ref=Data(numpy.array([[[[2.0, 6.0, -7.0, -4.0], [-2.0, 3.0, 1.0, -1.0]], [[0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 5.0, 4.0]], 
[[3.0, -1.0, 0.0, -1.0], [3.0, 0.0, 0.0, 5.0]]], [[[-2.0, 1.0, -3.0, 7.0], [5.0, 4.0, -4.0, 3.0]], [[-7.0, 7.0, 5.0, 2.0], 
[0.0, 0.0, 6.0, -1.0]], [[3.0, 3.0, 3.0, -1.0], [-6.0, 5.0, -7.0, -5.0]]], [[[-1.0, 0.0, 6.0, -3.0], [-6.0, 5.0, -3.0, -6.0]], 
[[5.0, -6.0, 7.0, -5.0], [7.0, 0.0, 0.0, -2.0]], [[-3.0, -4.0, 4.0, 1.0], [5.0, -5.0, 3.0, 0.0]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 3, 2, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_constData_rank4_axes_13(self):
      arg=Data(numpy.array([[[[5.0, 0.0, 1.0, -2.0], [0.0, -1.0, -6.0, 0.0], [-2.0, -4.0, 7.0, -1.0]], [[-3.0, -1.0, 2.0, 
-3.0], [-1.0, 7.0, -6.0, 3.0], [3.0, -6.0, 0.0, -2.0]]], [[[2.0, -1.0, 3.0, -2.0], [-1.0, -5.0, 1.0, 6.0], [4.0, -5.0, -7.0, 
-6.0]], [[0.0, -5.0, -6.0, -7.0], [-1.0, -5.0, -4.0, 1.0], [-1.0, 0.0, 5.0, 6.0]]], [[[3.0, -5.0, 7.0, -5.0], [3.0, 3.0, -1.0, 
-1.0], [6.0, 5.0, 2.0, 3.0]], [[6.0, 1.0, -4.0, -3.0], [7.0, -2.0, -4.0, -4.0], [3.0, -4.0, -3.0, -3.0]]]]),self.functionspace)
      res=swap_axes(arg,axis0=1,axis1=3)
      ref=Data(numpy.array([[[[5.0, -3.0], [0.0, -1.0], [-2.0, 3.0]], [[0.0, -1.0], [-1.0, 7.0], [-4.0, -6.0]], [[1.0, 2.0], 
[-6.0, -6.0], [7.0, 0.0]], [[-2.0, -3.0], [0.0, 3.0], [-1.0, -2.0]]], [[[2.0, 0.0], [-1.0, -1.0], [4.0, -1.0]], [[-1.0, -5.0], 
[-5.0, -5.0], [-5.0, 0.0]], [[3.0, -6.0], [1.0, -4.0], [-7.0, 5.0]], [[-2.0, -7.0], [6.0, 1.0], [-6.0, 6.0]]], [[[3.0, 6.0], 
[3.0, 7.0], [6.0, 3.0]], [[-5.0, 1.0], [3.0, -2.0], [5.0, -4.0]], [[7.0, -4.0], [-1.0, -4.0], [2.0, -3.0]], [[-5.0, -3.0], 
[-1.0, -4.0], [3.0, -3.0]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 4, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_constData_rank4_axes_23(self):
      arg=Data(numpy.array([[[[-1.0, -1.0, 0.0, -1.0], [3.0, 2.0, 1.0, -2.0], [2.0, 7.0, -7.0, 0.0]], [[-1.0, -5.0, 7.0, 1.0], 
[-5.0, 1.0, 3.0, -7.0], [-1.0, 4.0, -5.0, 4.0]]], [[[2.0, 1.0, 7.0, 7.0], [-1.0, 2.0, -3.0, 0.0], [2.0, 7.0, -6.0, -1.0]], 
[[-2.0, -6.0, 6.0, -1.0], [4.0, 0.0, 3.0, 5.0], [0.0, 4.0, 2.0, 6.0]]], [[[5.0, 6.0, -6.0, 3.0], [4.0, 3.0, -4.0, 4.0], [-6.0, 
6.0, 4.0, 0.0]], [[3.0, 0.0, -6.0, -5.0], [-6.0, 0.0, -3.0, 2.0], [6.0, -2.0, 0.0, -2.0]]]]),self.functionspace)
      res=swap_axes(arg,axis0=2,axis1=3)
      ref=Data(numpy.array([[[[-1.0, 3.0, 2.0], [-1.0, 2.0, 7.0], [0.0, 1.0, -7.0], [-1.0, -2.0, 0.0]], [[-1.0, -5.0, -1.0], 
[-5.0, 1.0, 4.0], [7.0, 3.0, -5.0], [1.0, -7.0, 4.0]]], [[[2.0, -1.0, 2.0], [1.0, 2.0, 7.0], [7.0, -3.0, -6.0], [7.0, 0.0, 
-1.0]], [[-2.0, 4.0, 0.0], [-6.0, 0.0, 4.0], [6.0, 3.0, 2.0], [-1.0, 5.0, 6.0]]], [[[5.0, 4.0, -6.0], [6.0, 3.0, 6.0], [-6.0, 
-4.0, 4.0], [3.0, 4.0, 0.0]], [[3.0, -6.0, 6.0], [0.0, 0.0, -2.0], [-6.0, -3.0, 0.0], [-5.0, 2.0, -2.0]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 4, 3),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_expandedData_rank2_axes_01(self):
      msk_arg=1-whereZero(self.functionspace.getX()[0],1.e-8)
      arg=msk_arg*numpy.array([[6.0, 5.0, 1.0, 5.0, 7.0], [-4.0, -6.0, -4.0, -2.0, -7.0], [7.0, -1.0, 4.0, 5.0, -1.0], [1.0, 
-7.0, -7.0, 2.0, -7.0]])+(1.-msk_arg)*numpy.array([[4.0, -3.0, -6.0, 6.0, -4.0], [7.0, 6.0, 2.0, 7.0, -6.0], [-4.0, -5.0, 4.0, 
-3.0, 6.0], [5.0, -4.0, 4.0, -3.0, -7.0]])
      res=swap_axes(arg,axis0=0,axis1=1)
      msk_ref=1-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[6.0, -4.0, 7.0, 1.0], [5.0, -6.0, -1.0, -7.0], [1.0, -4.0, 4.0, -7.0], [5.0, -2.0, 5.0, 2.0], 
[7.0, -7.0, -1.0, -7.0]])+(1.-msk_ref)*numpy.array([[4.0, 7.0, -4.0, 5.0], [-3.0, 6.0, -5.0, -4.0], [-6.0, 2.0, 4.0, 4.0], 
[6.0, 7.0, -3.0, -3.0], [-4.0, -6.0, 6.0, -7.0]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(5, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_expandedData_rank3_axes_01(self):
      msk_arg=1-whereZero(self.functionspace.getX()[0],1.e-8)
      arg=msk_arg*numpy.array([[[-6.0, 3.0], [3.0, -3.0]], [[0.0, -4.0], [4.0, -7.0]], [[5.0, -4.0], [6.0, 4.0]], [[6.0, 0.0], 
[1.0, -7.0]], [[3.0, 7.0], [5.0, -5.0]], [[-1.0, 2.0], [0.0, 0.0]]])+(1.-msk_arg)*numpy.array([[[0.0, -4.0], [-5.0, 4.0]], 
[[0.0, -4.0], [7.0, 0.0]], [[-5.0, 1.0], [4.0, 0.0]], [[-4.0, -2.0], [0.0, 2.0]], [[-6.0, 4.0], [-2.0, -3.0]], [[-2.0, 0.0], 
[-1.0, 3.0]]])
      res=swap_axes(arg,axis0=0,axis1=1)
      msk_ref=1-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[-6.0, 3.0], [0.0, -4.0], [5.0, -4.0], [6.0, 0.0], [3.0, 7.0], [-1.0, 2.0]], [[3.0, -3.0], 
[4.0, -7.0], [6.0, 4.0], [1.0, -7.0], [5.0, -5.0], [0.0, 0.0]]])+(1.-msk_ref)*numpy.array([[[0.0, -4.0], [0.0, -4.0], [-5.0, 
1.0], [-4.0, -2.0], [-6.0, 4.0], [-2.0, 0.0]], [[-5.0, 4.0], [7.0, 0.0], [4.0, 0.0], [0.0, 2.0], [-2.0, -3.0], [-1.0, 3.0]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 6, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_expandedData_rank3_axes_02(self):
      msk_arg=1-whereZero(self.functionspace.getX()[0],1.e-8)
      arg=msk_arg*numpy.array([[[0.0, -6.0], [-3.0, -7.0]], [[4.0, -3.0], [-2.0, 6.0]], [[-2.0, 4.0], [5.0, -6.0]], [[-2.0, 
-4.0], [-4.0, -2.0]], [[2.0, -7.0], [-5.0, 4.0]], [[7.0, 4.0], [-6.0, 7.0]]])+(1.-msk_arg)*numpy.array([[[-4.0, -1.0], [0.0, 
2.0]], [[5.0, 7.0], [-1.0, -3.0]], [[0.0, -3.0], [-1.0, -1.0]], [[7.0, 5.0], [4.0, -5.0]], [[-1.0, -6.0], [3.0, 3.0]], [[0.0, 
7.0], [3.0, -4.0]]])
      res=swap_axes(arg,axis0=0,axis1=2)
      msk_ref=1-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[0.0, 4.0, -2.0, -2.0, 2.0, 7.0], [-3.0, -2.0, 5.0, -4.0, -5.0, -6.0]], [[-6.0, -3.0, 4.0, 
-4.0, -7.0, 4.0], [-7.0, 6.0, -6.0, -2.0, 4.0, 7.0]]])+(1.-msk_ref)*numpy.array([[[-4.0, 5.0, 0.0, 7.0, -1.0, 0.0], [0.0, -1.0, 
-1.0, 4.0, 3.0, 3.0]], [[-1.0, 7.0, -3.0, 5.0, -6.0, 7.0], [2.0, -3.0, -1.0, -5.0, 3.0, -4.0]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 2, 6),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_expandedData_rank3_axes_12(self):
      msk_arg=1-whereZero(self.functionspace.getX()[0],1.e-8)
      arg=msk_arg*numpy.array([[[3.0, -2.0], [-2.0, -7.0]], [[-5.0, -2.0], [2.0, 2.0]], [[1.0, 1.0], [7.0, -1.0]], [[-4.0, 
0.0], [0.0, 3.0]], [[-6.0, -2.0], [-7.0, -1.0]], [[7.0, 3.0], [4.0, 2.0]]])+(1.-msk_arg)*numpy.array([[[3.0, 6.0], [3.0, 3.0]], 
[[2.0, -3.0], [6.0, 0.0]], [[4.0, -5.0], [2.0, -4.0]], [[-6.0, -7.0], [0.0, 0.0]], [[-2.0, -2.0], [5.0, 7.0]], [[-6.0, -3.0], 
[0.0, 4.0]]])
      res=swap_axes(arg,axis0=1,axis1=2)
      msk_ref=1-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[3.0, -2.0], [-2.0, -7.0]], [[-5.0, 2.0], [-2.0, 2.0]], [[1.0, 7.0], [1.0, -1.0]], [[-4.0, 
0.0], [0.0, 3.0]], [[-6.0, -7.0], [-2.0, -1.0]], [[7.0, 4.0], [3.0, 2.0]]])+(1.-msk_ref)*numpy.array([[[3.0, 3.0], [6.0, 3.0]], 
[[2.0, 6.0], [-3.0, 0.0]], [[4.0, 2.0], [-5.0, -4.0]], [[-6.0, 0.0], [-7.0, 0.0]], [[-2.0, 5.0], [-2.0, 7.0]], [[-6.0, 0.0], 
[-3.0, 4.0]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_expandedData_rank4_axes_01(self):
      msk_arg=1-whereZero(self.functionspace.getX()[0],1.e-8)
      arg=msk_arg*numpy.array([[[[-4.0, 5.0, 4.0, 0.0], [5.0, -5.0, 1.0, -7.0], [0.0, 4.0, -6.0, 0.0]], [[7.0, 0.0, -4.0, 
-4.0], [2.0, -7.0, -3.0, 2.0], [-3.0, 4.0, -4.0, 3.0]]], [[[-2.0, 3.0, -7.0, -5.0], [2.0, 6.0, -6.0, 6.0], [0.0, -7.0, -2.0, 
-7.0]], [[1.0, 0.0, -6.0, 0.0], [4.0, -1.0, 1.0, 2.0], [-1.0, -2.0, 1.0, 7.0]]], [[[4.0, 3.0, -7.0, -1.0], [0.0, 0.0, 2.0, 
-2.0], [-3.0, -1.0, 1.0, 3.0]], [[2.0, -7.0, 3.0, -4.0], [-5.0, 5.0, -2.0, -7.0], [2.0, 3.0, 0.0, 
6.0]]]])+(1.-msk_arg)*numpy.array([[[[4.0, 7.0, -2.0, -1.0], [-6.0, -1.0, 5.0, -7.0], [2.0, 4.0, -5.0, -2.0]], [[5.0, -7.0, 
-5.0, -6.0], [-1.0, 0.0, -5.0, 1.0], [7.0, -3.0, 5.0, -7.0]]], [[[-5.0, -2.0, 7.0, -3.0], [-1.0, 0.0, -7.0, 2.0], [4.0, -4.0, 
7.0, 2.0]], [[1.0, 6.0, 6.0, 3.0], [-1.0, 7.0, 4.0, 7.0], [4.0, 2.0, -6.0, -2.0]]], [[[0.0, 1.0, -5.0, 1.0], [4.0, -3.0, 0.0, 
-3.0], [-5.0, -4.0, -7.0, 1.0]], [[-4.0, 5.0, -2.0, -1.0], [3.0, 6.0, -5.0, -1.0], [0.0, -4.0, -6.0, 0.0]]]])
      res=swap_axes(arg,axis0=0,axis1=1)
      msk_ref=1-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[-4.0, 5.0, 4.0, 0.0], [5.0, -5.0, 1.0, -7.0], [0.0, 4.0, -6.0, 0.0]], [[-2.0, 3.0, -7.0, 
-5.0], [2.0, 6.0, -6.0, 6.0], [0.0, -7.0, -2.0, -7.0]], [[4.0, 3.0, -7.0, -1.0], [0.0, 0.0, 2.0, -2.0], [-3.0, -1.0, 1.0, 
3.0]]], [[[7.0, 0.0, -4.0, -4.0], [2.0, -7.0, -3.0, 2.0], [-3.0, 4.0, -4.0, 3.0]], [[1.0, 0.0, -6.0, 0.0], [4.0, -1.0, 1.0, 
2.0], [-1.0, -2.0, 1.0, 7.0]], [[2.0, -7.0, 3.0, -4.0], [-5.0, 5.0, -2.0, -7.0], [2.0, 3.0, 0.0, 
6.0]]]])+(1.-msk_ref)*numpy.array([[[[4.0, 7.0, -2.0, -1.0], [-6.0, -1.0, 5.0, -7.0], [2.0, 4.0, -5.0, -2.0]], [[-5.0, -2.0, 
7.0, -3.0], [-1.0, 0.0, -7.0, 2.0], [4.0, -4.0, 7.0, 2.0]], [[0.0, 1.0, -5.0, 1.0], [4.0, -3.0, 0.0, -3.0], [-5.0, -4.0, -7.0, 
1.0]]], [[[5.0, -7.0, -5.0, -6.0], [-1.0, 0.0, -5.0, 1.0], [7.0, -3.0, 5.0, -7.0]], [[1.0, 6.0, 6.0, 3.0], [-1.0, 7.0, 4.0, 
7.0], [4.0, 2.0, -6.0, -2.0]], [[-4.0, 5.0, -2.0, -1.0], [3.0, 6.0, -5.0, -1.0], [0.0, -4.0, -6.0, 0.0]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 3, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_expandedData_rank4_axes_02(self):
      msk_arg=1-whereZero(self.functionspace.getX()[0],1.e-8)
      arg=msk_arg*numpy.array([[[[-1.0, -7.0, 6.0, 1.0], [2.0, 4.0, 4.0, 3.0], [7.0, -7.0, 7.0, -1.0]], [[-6.0, -6.0, -2.0, 
1.0], [0.0, 1.0, -2.0, 5.0], [6.0, 4.0, 4.0, 4.0]]], [[[-5.0, 2.0, -6.0, 0.0], [-7.0, -6.0, 3.0, 4.0], [7.0, 7.0, 6.0, 0.0]], 
[[-4.0, 4.0, 5.0, -3.0], [-2.0, -1.0, 6.0, 5.0], [-4.0, -2.0, 0.0, -7.0]]], [[[1.0, -3.0, -3.0, 6.0], [6.0, -6.0, 1.0, -7.0], 
[7.0, -3.0, -2.0, 0.0]], [[0.0, 7.0, -1.0, -7.0], [-2.0, 0.0, 4.0, -6.0], [-4.0, 7.0, -7.0, 
6.0]]]])+(1.-msk_arg)*numpy.array([[[[-6.0, -5.0, 4.0, 7.0], [0.0, -1.0, 4.0, 7.0], [3.0, 2.0, 5.0, 2.0]], [[7.0, 7.0, 0.0, 
5.0], [4.0, 3.0, -5.0, -3.0], [0.0, 0.0, -6.0, 6.0]]], [[[4.0, 1.0, 6.0, -3.0], [-2.0, 5.0, 3.0, 4.0], [0.0, -4.0, 1.0, 2.0]], 
[[-4.0, 7.0, 1.0, -6.0], [6.0, 0.0, 0.0, 5.0], [-4.0, 0.0, 0.0, 4.0]]], [[[-2.0, -1.0, -2.0, 0.0], [1.0, -3.0, 4.0, 7.0], [4.0, 
-6.0, -5.0, 3.0]], [[-4.0, 2.0, 7.0, 2.0], [4.0, 1.0, -6.0, 2.0], [4.0, 1.0, 3.0, 0.0]]]])
      res=swap_axes(arg,axis0=0,axis1=2)
      msk_ref=1-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[-1.0, -7.0, 6.0, 1.0], [-5.0, 2.0, -6.0, 0.0], [1.0, -3.0, -3.0, 6.0]], [[-6.0, -6.0, -2.0, 
1.0], [-4.0, 4.0, 5.0, -3.0], [0.0, 7.0, -1.0, -7.0]]], [[[2.0, 4.0, 4.0, 3.0], [-7.0, -6.0, 3.0, 4.0], [6.0, -6.0, 1.0, 
-7.0]], [[0.0, 1.0, -2.0, 5.0], [-2.0, -1.0, 6.0, 5.0], [-2.0, 0.0, 4.0, -6.0]]], [[[7.0, -7.0, 7.0, -1.0], [7.0, 7.0, 6.0, 
0.0], [7.0, -3.0, -2.0, 0.0]], [[6.0, 4.0, 4.0, 4.0], [-4.0, -2.0, 0.0, -7.0], [-4.0, 7.0, -7.0, 
6.0]]]])+(1.-msk_ref)*numpy.array([[[[-6.0, -5.0, 4.0, 7.0], [4.0, 1.0, 6.0, -3.0], [-2.0, -1.0, -2.0, 0.0]], [[7.0, 7.0, 0.0, 
5.0], [-4.0, 7.0, 1.0, -6.0], [-4.0, 2.0, 7.0, 2.0]]], [[[0.0, -1.0, 4.0, 7.0], [-2.0, 5.0, 3.0, 4.0], [1.0, -3.0, 4.0, 7.0]], 
[[4.0, 3.0, -5.0, -3.0], [6.0, 0.0, 0.0, 5.0], [4.0, 1.0, -6.0, 2.0]]], [[[3.0, 2.0, 5.0, 2.0], [0.0, -4.0, 1.0, 2.0], [4.0, 
-6.0, -5.0, 3.0]], [[0.0, 0.0, -6.0, 6.0], [-4.0, 0.0, 0.0, 4.0], [4.0, 1.0, 3.0, 0.0]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_expandedData_rank4_axes_03(self):
      msk_arg=1-whereZero(self.functionspace.getX()[0],1.e-8)
      arg=msk_arg*numpy.array([[[[-6.0, -7.0, -7.0, -5.0], [7.0, -2.0, -2.0, 1.0], [4.0, -7.0, -4.0, 7.0]], [[7.0, 3.0, 2.0, 
6.0], [-3.0, -3.0, 2.0, 2.0], [-5.0, 0.0, 1.0, -4.0]]], [[[-7.0, 5.0, 3.0, -1.0], [-3.0, 0.0, 0.0, 7.0], [2.0, -1.0, -2.0, 
-5.0]], [[-5.0, 2.0, -4.0, 1.0], [2.0, 1.0, -7.0, 0.0], [-6.0, -2.0, 7.0, -6.0]]], [[[0.0, -2.0, 0.0, 3.0], [-2.0, 2.0, -4.0, 
0.0], [1.0, 0.0, 4.0, 5.0]], [[-2.0, -7.0, -4.0, -1.0], [-2.0, 0.0, -6.0, 1.0], [1.0, -5.0, -2.0, 
5.0]]]])+(1.-msk_arg)*numpy.array([[[[-7.0, -2.0, 0.0, 7.0], [7.0, 0.0, 0.0, -5.0], [-2.0, -7.0, -3.0, -3.0]], [[6.0, -2.0, 
-6.0, 6.0], [3.0, 4.0, -6.0, -6.0], [7.0, -2.0, 5.0, -6.0]]], [[[-5.0, 5.0, 3.0, -6.0], [-4.0, -4.0, 6.0, 0.0], [1.0, 0.0, 4.0, 
5.0]], [[0.0, -5.0, -4.0, -2.0], [5.0, -6.0, 5.0, -1.0], [-1.0, 3.0, 1.0, -4.0]]], [[[-1.0, -4.0, 2.0, -3.0], [-4.0, 5.0, -5.0, 
0.0], [-6.0, 6.0, 4.0, 4.0]], [[-7.0, -2.0, 3.0, -1.0], [5.0, 1.0, -6.0, -6.0], [-5.0, -2.0, 3.0, -3.0]]]])
      res=swap_axes(arg,axis0=0,axis1=3)
      msk_ref=1-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[-6.0, -7.0, 0.0], [7.0, -3.0, -2.0], [4.0, 2.0, 1.0]], [[7.0, -5.0, -2.0], [-3.0, 2.0, -2.0], 
[-5.0, -6.0, 1.0]]], [[[-7.0, 5.0, -2.0], [-2.0, 0.0, 2.0], [-7.0, -1.0, 0.0]], [[3.0, 2.0, -7.0], [-3.0, 1.0, 0.0], [0.0, 
-2.0, -5.0]]], [[[-7.0, 3.0, 0.0], [-2.0, 0.0, -4.0], [-4.0, -2.0, 4.0]], [[2.0, -4.0, -4.0], [2.0, -7.0, -6.0], [1.0, 7.0, 
-2.0]]], [[[-5.0, -1.0, 3.0], [1.0, 7.0, 0.0], [7.0, -5.0, 5.0]], [[6.0, 1.0, -1.0], [2.0, 0.0, 1.0], [-4.0, -6.0, 
5.0]]]])+(1.-msk_ref)*numpy.array([[[[-7.0, -5.0, -1.0], [7.0, -4.0, -4.0], [-2.0, 1.0, -6.0]], [[6.0, 0.0, -7.0], [3.0, 5.0, 
5.0], [7.0, -1.0, -5.0]]], [[[-2.0, 5.0, -4.0], [0.0, -4.0, 5.0], [-7.0, 0.0, 6.0]], [[-2.0, -5.0, -2.0], [4.0, -6.0, 1.0], 
[-2.0, 3.0, -2.0]]], [[[0.0, 3.0, 2.0], [0.0, 6.0, -5.0], [-3.0, 4.0, 4.0]], [[-6.0, -4.0, 3.0], [-6.0, 5.0, -6.0], [5.0, 1.0, 
3.0]]], [[[7.0, -6.0, -3.0], [-5.0, 0.0, 0.0], [-3.0, 5.0, 4.0]], [[6.0, -2.0, -1.0], [-6.0, -1.0, -6.0], [-6.0, -4.0, -3.0]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 2, 3, 3),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_expandedData_rank4_axes_12(self):
      msk_arg=1-whereZero(self.functionspace.getX()[0],1.e-8)
      arg=msk_arg*numpy.array([[[[3.0, -4.0, 5.0, -6.0], [5.0, 0.0, 7.0, 2.0], [6.0, 0.0, 5.0, -6.0]], [[-1.0, 7.0, 0.0, 7.0], 
[-4.0, -5.0, 5.0, 1.0], [6.0, 3.0, 4.0, -3.0]]], [[[-5.0, -4.0, 7.0, 6.0], [0.0, 4.0, -1.0, -4.0], [2.0, -3.0, -7.0, -4.0]], 
[[5.0, 0.0, 7.0, -7.0], [2.0, 2.0, 6.0, -3.0], [-1.0, 4.0, -4.0, 5.0]]], [[[0.0, 6.0, -7.0, -3.0], [4.0, -6.0, -7.0, 0.0], 
[5.0, 6.0, 0.0, -6.0]], [[-2.0, 3.0, 0.0, 0.0], [-6.0, 0.0, 0.0, -5.0], [5.0, -1.0, 1.0, 
-2.0]]]])+(1.-msk_arg)*numpy.array([[[[0.0, 0.0, 6.0, -2.0], [-5.0, 5.0, 1.0, 6.0], [4.0, 4.0, 7.0, 1.0]], [[5.0, -7.0, 7.0, 
7.0], [0.0, -6.0, 1.0, 0.0], [-4.0, -5.0, 5.0, -1.0]]], [[[-5.0, -6.0, 7.0, 7.0], [1.0, 5.0, 7.0, 7.0], [6.0, 0.0, 0.0, 7.0]], 
[[1.0, 6.0, 1.0, 0.0], [-3.0, 0.0, 1.0, -4.0], [-5.0, 1.0, 2.0, 6.0]]], [[[0.0, 0.0, 2.0, -4.0], [-3.0, 1.0, 0.0, 0.0], [0.0, 
3.0, 1.0, -6.0]], [[1.0, 0.0, 0.0, -5.0], [-6.0, -7.0, 4.0, 5.0], [-7.0, -4.0, 2.0, -2.0]]]])
      res=swap_axes(arg,axis0=1,axis1=2)
      msk_ref=1-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[3.0, -4.0, 5.0, -6.0], [-1.0, 7.0, 0.0, 7.0]], [[5.0, 0.0, 7.0, 2.0], [-4.0, -5.0, 5.0, 
1.0]], [[6.0, 0.0, 5.0, -6.0], [6.0, 3.0, 4.0, -3.0]]], [[[-5.0, -4.0, 7.0, 6.0], [5.0, 0.0, 7.0, -7.0]], [[0.0, 4.0, -1.0, 
-4.0], [2.0, 2.0, 6.0, -3.0]], [[2.0, -3.0, -7.0, -4.0], [-1.0, 4.0, -4.0, 5.0]]], [[[0.0, 6.0, -7.0, -3.0], [-2.0, 3.0, 0.0, 
0.0]], [[4.0, -6.0, -7.0, 0.0], [-6.0, 0.0, 0.0, -5.0]], [[5.0, 6.0, 0.0, -6.0], [5.0, -1.0, 1.0, 
-2.0]]]])+(1.-msk_ref)*numpy.array([[[[0.0, 0.0, 6.0, -2.0], [5.0, -7.0, 7.0, 7.0]], [[-5.0, 5.0, 1.0, 6.0], [0.0, -6.0, 1.0, 
0.0]], [[4.0, 4.0, 7.0, 1.0], [-4.0, -5.0, 5.0, -1.0]]], [[[-5.0, -6.0, 7.0, 7.0], [1.0, 6.0, 1.0, 0.0]], [[1.0, 5.0, 7.0, 
7.0], [-3.0, 0.0, 1.0, -4.0]], [[6.0, 0.0, 0.0, 7.0], [-5.0, 1.0, 2.0, 6.0]]], [[[0.0, 0.0, 2.0, -4.0], [1.0, 0.0, 0.0, -5.0]], 
[[-3.0, 1.0, 0.0, 0.0], [-6.0, -7.0, 4.0, 5.0]], [[0.0, 3.0, 1.0, -6.0], [-7.0, -4.0, 2.0, -2.0]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 3, 2, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_swapaxes_expandedData_rank4_axes_13(self):
      msk_arg=1-whereZero(self.functionspace.getX()[0],1.e-8)
      arg=msk_arg*numpy.array([[[[5.0, -4.0, -2.0, 6.0], [-7.0, 0.0, 3.0, 3.0], [-3.0, 7.0, -2.0, -3.0]], [[-1.0, -2.0, 6.0, 
0.0], [4.0, -1.0, -1.0, -6.0], [5.0, -7.0, -3.0, -6.0]]], [[[-7.0, 0.0, 7.0, 4.0], [-5.0, 6.0, 1.0, 2.0], [-2.0, -3.0, -1.0, 
2.0]], [[4.0, -4.0, 7.0, -2.0], [-2.0, -6.0, -4.0, -6.0], [-5.0, -2.0, 0.0, 2.0]]], [[[1.0, -6.0, -3.0, 0.0], [-5.0, 0.0, 6.0, 
-5.0], [0.0, 0.0, 0.0, -7.0]], [[-7.0, -4.0, -4.0, 0.0], [-1.0, -4.0, -1.0, 0.0], [7.0, 3.0, 1.0, 
6.0]]]]