# Accessibility

Creating and publishing content that does not exclude disabled users is a
complex and iterative task.

While there is no one-size-fits-all solution to maintaining accessible content,
the PyData Sphinx Theme and this documentation site use some techniques to avoid
common content shortcomings.

:::{note}
Issues and pull requests to identify or fix accessibility issues on this theme
or site are heartily welcomed!
:::

## What We've Done

### Metadata

Several of our documentation pages contain metadata (i.e., `.. meta::` directives
in reStructuredText) giving summaries of the page contents. If you notice a
page that lacks metadata, please open a pull request to add it!

### Colors

- Our default code highlighting styles are `a11y-high-contrast-light` and
  `a11y-high-contrast-dark` from https://github.com/Quansight-Labs/accessible-pygments.
  These styles are designed to meet WCAG 2 AA or AAA contrast requirements.
  If you don't like the look of our default code highlighting styles, there are several more
  to choose from at https://github.com/Quansight-Labs/accessible-pygments.
- We recently revisited the PyData Sphinx theme color palette to ensure that
  the colors we use meet WCAG 2 AA or AAA contrast requirements.
- We also re-defined our `primary` and `secondary` colors to be more accessible
  and distinct from semantic colors used to denote success, warning, info, and
  danger contexts or information.
- We simplified the color palette and removed some colors that were problematic
  in meeting WCAG 2 AA or AAA contrast requirements and for certain types of
  colorblindness.
- We have improved how we assign text colors to interactive elements such as
  buttons and dropdowns to ensure that they meet WCAG 2 AA or AAA contrast
  requirements.

To learn more about the PyData Sphinx Theme colors, check the [](../community/design-system.md)
section.

### Keyboard Navigation Support

For all buttons, dropdowns, tabbed panels, hamburger menus, modals, overlays,
links and other interactive elements, we have worked to ensure they:

- Have a visible focus indicator (WCAG 2.4.7)
- Can be accessed via keyboard navigation (WCAG 2.1.1)

### Testing and auditing

We have also added automated tests and conducted manual audits. See
[](../community/topics/accessibility.md).

## What You Can Do

### Site configuration

The following sections include recommendations for settings in the `conf.py` file that can positively impact the
accessibility of content generated by this theme and Sphinx in general.

### Natural Language

If not using a more robust [internationalization approach](https://www.sphinx-doc.org/en/master/usage/advanced/intl.html),
specifying at least the baseline natural language will help assistive technology
identify if the content is in a language the reader understands.

:::{hint}
In your `conf.py` file,
[specifying the language your docs are written in](https://www.sphinx-doc.org/en/master/usage/configuration.html#confval-language)
will propagate to the top-level `HTML` tag.

```python
language = "en"
```

:::

### Add a Site Map

Site maps, usually served from a file called `sitemap.xml` are a broadly-employed
approach to telling programs like search engines and assistive technologies where
different content appears on a website.

If using a service like [ReadTheDocs](https://about.readthedocs.com/), these files
will be created for you _automatically_, but for some other approaches below,
it's handy to generate a `sitemap.xml` locally or in CI with a tool like
[sphinx-sitemap](https://pypi.org/project/sphinx-sitemap/).

:::{hint}

For a simple site (no extra languages or versions), ensure `sphinx-sitemap`
is installed in your documentation environment, and modify your `conf.py`:

```python
extensions += ["sphinx_sitemap"]

html_baseurl = os.environ.get("SPHINX_HTML_BASE_URL", "http://127.0.0.1:8000/")
sitemap_locales = [None]
sitemap_url_scheme = "{link}"
```

:::

### Logo best practices

We recommend that you support dark mode by providing either a single version of
your logo that works well in both light and dark modes, or two separate
versions. We also recommend that you provide alt text for your logo if you do
not provide visible text.

These recommendations are covered in detail at the page [](./branding.rst)

### Test and inspect your site

Several browser tools exist for interactively debugging the accessibility
of a single page at a time and can be useful during the content development
cycle.

Non-interactive tools also exist for use with CI (continuous integration).

#### Browser tools

Most major browsers, including Firefox and Chrome, have built-in accessibility
tools as part of their web developer tools. These tools can help to quickly
identify accessibility issues and often include links to standards:

- [Firefox Accessibility Inspector](https://developer.mozilla.org/en-US/docs/Tools/Accessibility_inspector)
- [Chrome DevTools Accessibility](https://developer.chrome.com/docs/devtools/accessibility/reference)

There are also a few browser extensions that some of the maintainers of this
theme use when working to make websites more accessible. Some of these include:

- [Accessibility Insights for Web](https://accessibilityinsights.io/docs/web/overview/)
- [Axe DevTools](https://www.deque.com/axe/browser-extensions/)
- [WAVE](https://wave.webaim.org/extension/)

We have also found [Polypane](https://polypane.app/) to be a great tool (but it
is not free and requires a license).

#### Continuous Integration tools

Two accessibility testing tools designed for continuous integration are
[Lighthouse CI](https://github.com/GoogleChrome/lighthouse-ci/blob/main/docs/getting-started.md)
and [Pa11y CI](https://github.com/pa11y/pa11y-ci).

The [foo-software/lighthouse-check-action](https://github.com/foo-software/lighthouse-check-action)
may be helpful if the code for your site is hosted on GitHub.

If you are curious about how we do accessibility CI for this theme, refer to the
page [](../community/topics/accessibility.md).

:::{warning}
Note that automated testing and extensions such as the ones mentioned above will
at best catch 30-40% of accessibility issues. They are not a replacement for
manual testing. Having a perfect score on any of these tools does not mean that
your site can be used by disabled users. Instead, a good score signals that your
site follows some best accessibility practices.
:::
