/*	Size_Limits

CVS ID: Size_Limits.java,v 1.4 2012/04/16 06:22:59 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Viewers;

import	PIRL.Utilities.Integer_Range;

import	java.awt.Component;
import	java.awt.event.ComponentListener;
import	java.awt.event.ComponentEvent;


/**	<i>Size_Limits</i> limits the size of a component to which it is bound.
<p>
	@author		Bradford Castalia, idaeim
	@version	1.4
*/
public class Size_Limits
	implements ComponentListener
{
public static final String
	ID = "PIRL.Viewers.Size_Limits (1.4 2012/04/16 06:22:59)";

/**	The limits on the size of the component.
*/
private Integer_Range
	Width_Limits,
	Height_Limits;


//	DEBUG control.
private static final int
	DEBUG_OFF			= 0,
	DEBUG_CONSTRUCTOR	= 1 << 0,
	DEBUG_ACCESSORS		= 1 << 1,
	DEBUG_LISTENER		= 1 << 2,
	DEBUG_ALL			= -1,

	DEBUG				= DEBUG_OFF;	

/*==============================================================================
	Constructors
*/
/**	Constructs Size_Limits from a pair of limits.
<p>
	@param	width_limits	The Integer_Range limiting the width of the
		component. If null, the width will be unlimited.
	@param	height_limits	The Integer_Range limiting the height of the
		component. If null, the width will be unlimited.
	@see	Integer_Range
*/
public Size_Limits
	(
	Integer_Range	width_limits,
	Integer_Range	height_limits
	)
{
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		(">-< Composer constructor:\n"
		+"     width_limits - " + width_limits + '\n'
		+"    height_limits - " + height_limits);
Width_Limits  (width_limits);
Height_Limits (height_limits);
}

/**	Constructs Size_Limits with unlimited width and height.
*/
public Size_Limits ()
{
Width_Limits  (null);
Height_Limits (null);
}

/*==============================================================================
	Accessors
*/
/**	Gets the width limits.
<p>
	@return	An Integer_Range specifying the width limits. This is a
		copy of the Integer_Range used by this object; changing its
		limits will not affect the limits of this object.
*/
public Integer_Range Width_Limits ()
{return new Integer_Range (Width_Limits);}

/**	Gets the minimum width limit.
<p>
	@return	The minimum width limit. This will be zero if the minimum
		width is unlimited. The value will never be less than zero.
*/
public int Min_Width ()
{
long
	limit = Width_Limits.Minimum ();
if (limit <= 0)
	return 0;
if (limit > Integer.MAX_VALUE)
	return Integer.MAX_VALUE;
return (int)limit;
}

/**	Gets the maximum width limit.
<p>
	@return	The maximum width limit. This will be {@link Integer#MAX_VALUE}
		if the maximum width is unlimited.
*/
public int Max_Width ()
{
long
	limit = Width_Limits.Maximum ();
if (limit <= 0)
	return 0;
if (limit > Integer.MAX_VALUE)
	return Integer.MAX_VALUE;
return (int)limit;
}

/**	Sets the width limits.
<p>
	@param	width_limits	An Integer_Range specifying the limits.
	@return	This Size_Limits.
*/
public Size_Limits Width_Limits
	(
	Integer_Range	width_limits
	)
{
if (width_limits == null)
	Width_Limits = new Integer_Range (0, Long.MAX_VALUE);
else
	Width_Limits = new Integer_Range (width_limits);
return this;
}

/**	Sets the minimum width limit.
<p>
	@param	min_width	The minimum width limit.
	@return	This Size_Limits.
*/
public Size_Limits Min_Width
	(
	int		min_width
	)
{Width_Limits.Minimum (min_width); return this;}

/**	Sets the maximum width limit.
<p>
	@param	max_width	The maximum width limit.
	@return	This Size_Limits.
*/
public Size_Limits Max_Width
	(
	int		max_width
	)
{Width_Limits.Maximum (max_width); return this;}

/**	Gets the height limits.
<p>
	@return	An Integer_Range specifying the height limits. This is a
		copy of the Integer_Range used by this object; changing its
		limits will not affect the limits of this object.
*/
public Integer_Range Height_Limits ()
{return new Integer_Range (Height_Limits);}

/**	Gets the minimum height limit.
<p>
	@return	The minimum height limit. This will be zero if the minimum
		height is unlimited. The value will never be less than zero.
*/
public int Min_Height ()
{
long
	limit = Width_Limits.Minimum ();
if (limit <= 0)
	return 0;
if (limit > Integer.MAX_VALUE)
	return Integer.MAX_VALUE;
return (int)limit;
}

/**	Gets the maximum height limit.
<p>
	@return	The maximum height limit. This will be {@link Integer#MAX_VALUE}
		if the maximum height is unlimited.
*/
public int Max_Height ()
{
long
	limit = Width_Limits.Maximum ();
if (limit <= 0)
	return 0;
if (limit > Integer.MAX_VALUE)
	return Integer.MAX_VALUE;
return (int)limit;
}

/**	Sets the height limits.
<p>
	@param	height_limits	An Integer_Range specifying the limits.
	@return	This Size_Limits.
*/
public Size_Limits Height_Limits
	(
	Integer_Range	height_limits
	)
{
if (height_limits == null)
	Height_Limits = new Integer_Range (0, Long.MAX_VALUE);
else
	Height_Limits = new Integer_Range (height_limits);
return this;
}

/**	Sets the minimum height limit.
<p>
	@param	min_height	The minimum height limit.
	@return	This Size_Limits.
*/
public Size_Limits Min_Height
	(
	int		min_height
	)
{Height_Limits.Minimum (min_height); return this;}

/**	Sets the maximum height limit.
<p>
	@param	max_height	The maximum height limit.
	@return	This Size_Limits.
*/
public Size_Limits Max_Height
	(
	int		max_height
	)
{Height_Limits.Maximum (max_height); return this;}

/*==============================================================================
	ComponentListener
*/
/**	The size of the component is constrained to the {@link #Width_Limits()
	width limits} and {@link #Height_Limits() height limits}.
<p>
	@param	event	The ComponentEvent specifying the component that
		was resized.
*/
public void componentResized
	(
	ComponentEvent	event
	) 
{
if ((DEBUG & DEBUG_LISTENER) != 0)
	System.out.println
		(">>> Size_Limits.componentResized");
Component
	component = event.getComponent ();
int
	width  = component.getWidth (),
	height = component.getHeight ();
if ((DEBUG & DEBUG_LISTENER) != 0)
	System.out.println
		("    Current: " + width + "x" + height);

if (width < Width_Limits.Minimum ())
	width = (int)Width_Limits.Minimum ();
else
if (width > Width_Limits.Maximum ())
	width = (int)Width_Limits.Maximum ();
	
if (height < Height_Limits.Minimum ())
	height = (int)Height_Limits.Minimum ();
else
if (height > Height_Limits.Maximum ())
	height = (int)Height_Limits.Maximum ();

if (width  != component.getWidth () ||
	height != component.getHeight ())
	{
	if ((DEBUG & DEBUG_LISTENER) != 0)
		System.out.println
			("    Resized: " + width + "x" + height);
	component.setSize (width, height);
	}
if ((DEBUG & DEBUG_LISTENER) != 0)
	System.out.println
		("<<< Size_Limits.componentResized");
}

/**	No-op. */
public void componentMoved (ComponentEvent event) {}
/**	No-op. */
public void componentShown (ComponentEvent event) {}
/**	No-op. */
public void componentHidden (ComponentEvent event) {}

}
