@rem = '--*-Perl-*--
@echo off
if "%OS%" == "Windows_NT" goto WinNT
perl -x -S "%0" %1 %2 %3 %4 %5 %6 %7 %8 %9
goto endofperl
:WinNT
perl -x -S %0 %*
if NOT "%COMSPEC%" == "%SystemRoot%\system32\cmd.exe" goto endofperl
if %errorlevel% == 9009 echo You do not have Perl in your PATH.
if errorlevel 1 goto script_failed_so_exit_with_non_zero_val 2>nul
goto endofperl
@rem ';
#!/usr/bin/perl -w
#line 15
#
# ecaccess-event-send: Create an ECaccess Event
#
# Laurent.Gougeon@ecmwf.int - 2010-10-15

use ECMWF::ECaccess;
use Getopt::Long;
use Pod::Usage;
use Term::ReadKey;

my %opt = ( comment => '', public => 0, overwrite => 0, version => 0, help => 0, manual => 0, retry => 0, debug => 0 );

pod2usage( -noperldoc => 1, -exit => 1, verbose => 1 ) if !GetOptions(
	\%opt,
	qw(
	  comment:s
	  public
	  overwrite
	  version
	  help|?
	  manual
	  retry=i
	  debug
	  )
);

# Display version if requested
die ECMWF::ECaccess->VERSION . "\n" if ( $opt{version} );

my $eventname = $ARGV[0];

pod2usage( -noperldoc => 1, -exit => 1, verbose => 1 ) if ( $opt{help} );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 2 ) if ( $opt{manual} );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "No event-name specified!\n" ) if not($eventname);

# Create the ECaccess Controler
my $ecaccess = ECMWF::ECaccess->new( $opt{retry}, $opt{debug});

# Get the Token (using the Certificate in $HOME)
my $token = $ecaccess->getToken();

# Get the Control Channel
my $controlChannel = $ecaccess->getControlChannel();

# Create the event
my $event = $controlChannel->createEvent(
	$token,
	SOAP::Data->name(
		"request" => \SOAP::Data->value(
			SOAP::Data->name( 'name'      => $eventname ),
			SOAP::Data->name( 'comment'   => $opt{comment} ),
			SOAP::Data->name( 'overwrite' => $opt{overwrite} ? 'true' : 'false' )->type('xsd:boolean'),
			SOAP::Data->name( 'isPublic'  => $opt{public} ? 'true' : 'false' )->type('xsd:boolean')
		)
	)
);
print $event->valueof('//createEventResponse/return')->{eventId}, "\n";

# Logout
$ecaccess->releaseToken($token);

__END__

=head1 NAME

ecaccess-event-create - Create an ECaccess Event

=head1 SYNOPSIS

B<ecaccess-event-create -version|-help|-manual>

B<ecaccess-event-create [-debug] [-overwrite] [-public] [-comment> I<string>B<]> I<event-name>

=head1 DESCRIPTION

Allow creating an ECaccess Event. As a result the I<event-id> is displayed.

If an event already exists with the same I<event-name> then by default the request is
rejected (please check the B<-overwrite> option).

By default an event is private and restricted to its owner. Access can however be shared with
other users thanks to the B<ecaccess-event-grant> command or with the B<-public> option which
make it visible to all.

=head1 ARGUMENTS

=over 8

=item I<event-name>

The name of the Event to create.

=back

=head1 OPTIONS

=over 8

=item B<-overwrite>

Allow deleting an existing event with the same name and create a new one (the
identifier will be however different to the original one).

=item B<-public>

Allow creating a public event which can be used by everybody. In order to have
a better tunning of the permissions please keep the event private and use the
B<ecaccess-event-grant> command instead.

=item B<-comment> I<string>

Specify the I<string> to display as a comment for the event.

=item B<-version>

Display version number and exits.

=item B<-help>

Print a brief help message and exits.

=item B<-manual>

Prints the manual page and exits.

=item B<-retry> I<count>

Number of SSL connection retries per 5s to ECMWF. This parameter only apply to the
initial SSL connection initiated by the command to the ECMWF server. It does not
apply to all the subsequent requests made afteward as it is mainly targeting errors
that can happen from time to time during the SSL handshake. Default is no retry.

=item B<-debug>

Display the SOAP and SSL messages exchanged.

=back

=head1 EXAMPLES

B<ecaccess-event-create -public -comment> I<"My Comment"> I<"My Event Name">

Create the new event with the name I<"My Event Name"> and the comment I<"My Comment">. This
new event is public and other users can subscribe to it.

=head1 SEE ALSO

B<ecaccess-event-grant>, B<ecaccess-event-clear>, B<ecaccess-event-send>, B<ecaccess-event-list>,
B<ecaccess-event-delete> and B<ecaccess>.

=cut

__END__
:endofperl
