/*
 * Copyright (c) 2009 Cyrille Berger <cberger@cberger.net>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either 
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>. */

#include "MessagesModel.h"
#include <KIcon>

struct MessagesModel::Private {
  static MessagesModel* s_instance;
  QList<Message> messages;
};

MessagesModel* MessagesModel::Private::s_instance = 0;

MessagesModel::MessagesModel() : d(new Private) {
}

MessagesModel::~MessagesModel() {
  delete d;
}

MessagesModel* MessagesModel::instance() {
  if(not Private::s_instance) {
    Private::s_instance = new MessagesModel;
  }
  return Private::s_instance;
}

void MessagesModel::tellError( const QString& _who, const QString& _err ) {
  Message msg;
  msg.date = QDateTime::currentDateTime();
  msg.type = Message::Error;
  msg.who = _who;
  msg.what = _err;
  beginInsertRows(QModelIndex(), 0,0);
  d->messages.prepend(msg);
  endInsertRows();
  emit(errorReceived());
  emit(messageReceived());
}

void MessagesModel::tellInfo( const QString& _who, const QString& _info ) {
  Message msg;
  msg.date = QDateTime::currentDateTime();
  msg.type = Message::Information;
  msg.who = _who;
  msg.what = _info;
  beginInsertRows(QModelIndex(), 0,0);
  d->messages.prepend(msg);
  endInsertRows();
  emit(infoReceived());
  emit(messageReceived());
}

int MessagesModel::rowCount(const QModelIndex &parent) const {
  Q_UNUSED(parent);
  return d->messages.count();
}

int MessagesModel::columnCount(const QModelIndex &parent ) const {
  Q_UNUSED(parent);
  return 4;
}

QVariant MessagesModel::data(const QModelIndex &index, int role ) const {
  if(index.isValid())
  {
    Q_ASSERT(index.row() < d->messages.count());
    Message msg = d->messages[index.row()];
    switch(role) {
      case Qt::DisplayRole:
      {
        switch(index.column()) {
          case 0:
            break;
          case 1:
            return msg.date;
          case 2:
            return msg.who;
          case 3:
            return msg.what;
          default:
            qFatal("missing column");
        }
      }
        break;
      case Qt::DecorationRole:
        if( index.column() == 0 ) {
          return icon( msg.type );
        }
        break;
      case Qt::ForegroundRole:
        if( msg.type == Message::Error ) {
          return Qt::black;
        }
        break;
      case Qt::BackgroundRole:
        if( msg.type == Message::Error ) {
          return QColor(255, 125, 125);
        }
        break;
    }
  }
  return QVariant();
}

QPixmap MessagesModel::icon(Message::Type _type) const {
  QString name;
  switch(_type) {
    case Message::Information:
      name = "help-about";
      break;
    case Message::Error:
      name = "window-close";
      break;
  }
  return KIcon(name).pixmap(22);
}

#include "MessagesModel.moc"
