% test_fec.m
% David Rowe Oct 2014
%

% Simulation to test FDM QPSK combined with FEC.  A low rate Codec
% (e.g. 450 bit/s) is transmitted on Nc=4 FDM carriers.  FEC parity
% bits are repeated on a seperate block of 4 carriers that is delayed
% in time by Rs symbols (1 second).  A (n,k) Read Solomon code that can
% correct (n-k)/2 errors is simulated.
  
1;

% main test function 

function sim_out = ber_test(sim_in, modulation)
    Fs = 8000;

    verbose          = sim_in.verbose;
    Ntrials          = sim_in.Ntrials;
    Esvec            = sim_in.Esvec;
    phase_offset     = sim_in.phase_offset;
    w_offset         = sim_in.w_offset;
    plot_scatter     = sim_in.plot_scatter;
    Rs               = sim_in.Rs;
    hf_sim           = sim_in.hf_sim;
    nhfdelay         = sim_in.hf_delay_ms*Rs/1000;
    hf_mag_only      = sim_in.hf_mag_only;
    n                = sim_in.n;
    k                = sim_in.k;  % k message bits 
    Nc               = sim_in.Nc; % number of carriers

    bps              = 2;
    Nsymb            = n/bps;      % total number of symbols
    Nb               = Nsymb/Nc;   % length of block of symbols 
    assert(Nb == floor(Nb), "Nb must be an integer");

    Nck              = (k/n)*Nc;      % Number of carriers for data symbols
    Ncp              = ((n-k)/n)*Nc;  % Number of carriers for parity symbols 
    assert(Nck == floor(Nck), "Number of carriers for data symbols must be an integer");
    assert(Ncp == floor(Ncp), "Number of carriers for parity symbols must be an integer");

    printf("(n,k)=(%d,%d) Nsymb: %d Nc: %d Nb: %d Nck: %d Ncp: %d\n",n,k,Nsymb,Nc,Nb,Nck,Ncp);

    prev_sym_tx      = qpsk_mod([0 0])*ones(1,Nc);
    prev_sym_rx      = qpsk_mod([0 0])*ones(1,Nc);

    % Init HF channel model from stored sample files of spreading signal ----------------------------------

    % convert "spreading" samples from 1kHz carrier at Fs to complex
    % baseband, generated by passing a 1kHz sine wave through PathSim
    % with the ccir-poor model, enabling one path at a time.
    
    Fc = 1000; M = Fs/Rs;
    fspread = fopen("../raw/sine1k_2Hz_spread.raw","rb");
    spread1k = fread(fspread, "int16")/10000;
    fclose(fspread);
    fspread = fopen("../raw/sine1k_2ms_delay_2Hz_spread.raw","rb");
    spread1k_2ms = fread(fspread, "int16")/10000;
    fclose(fspread);

    % down convert to complex baseband
    spreadbb = spread1k.*exp(-j*(2*pi*Fc/Fs)*(1:length(spread1k))');
    spreadbb_2ms = spread1k_2ms.*exp(-j*(2*pi*Fc/Fs)*(1:length(spread1k_2ms))');

    % remove -2000 Hz image
    b = fir1(50, 5/Fs);
    spread = filter(b,1,spreadbb);
    spread_2ms = filter(b,1,spreadbb_2ms);
   
    % discard first 1000 samples as these were near 0, probably as
    % PathSim states were ramping up

    spread    = spread(1000:length(spread));
    spread_2ms = spread_2ms(1000:length(spread_2ms));

    % decimate down to Rs

    spread = spread(1:M:length(spread));
    spread_2ms = spread_2ms(1:M:length(spread_2ms));

    % Determine "gain" of HF channel model, so we can normalise
    % carrier power during HF channel sim to calibrate SNR.  I imagine
    % different implementations of ccir-poor would do this in
    % different ways, leading to different BER results.  Oh Well!

    hf_gain = 1.0/sqrt(var(spread)+var(spread_2ms));

    % Start Simulation ----------------------------------------------------------------

    for ne = 1:length(Esvec)
        EsNodB = Esvec(ne);
        EsNo = 10^(EsNodB/10);
    
        variance = 1/EsNo;
         if verbose > 1
            printf("EsNo (dB): %f EsNo: %f variance: %f\n", EsNodB, EsNo, variance);
        end
        
        Terrs = 0;  Tbits = 0;

        tx_symb_log      = [];
        rx_symb_log      = [];
        errors_log       = [];
        Nerrs_log        = [];

        % init HF channel

        hf_n = 1;

        % simulation starts here-----------------------------------
 
        for nn = 1: Ntrials
                  
            tx_bits = round(rand(1,n));                       

            % modulate --------------------------------------------

            tx_symb=zeros(Nc,Nb);

            for b=1:Nb
              for c=1:Nc
                  i = (b-1)*Nc + c;
                  tx_symb(c,b) = qpsk_mod(tx_bits(2*(i-1)+1:2*i));
                  if strcmp(modulation,'dqpsk')
                    tx_symb(c,b) *= prev_sym_tx(c);
                    prev_sym_tx(c) = tx_symb(c,b);
                  end
              end
            end
            s_ch = tx_symb;

            % HF channel simulation  ------------------------------------
            
            if hf_sim

                % separation between carriers.  Note this effectively
                % under samples at Rs, I dont think this matters.
                % Equivalent to doing freq shift at Fs, then
                % decimating to Rs.

                wsep = 2*pi*(1+0.5);  % e.g. 75Hz spacing at Rs=50Hz, alpha=0.5 filters

                for b=1:Nb

                    % apply HF model to data symbol carriers

                    for c=1:Nck
                        ahf_model = hf_gain*(spread(hf_n) + exp(-j*k*wsep*nhfdelay)*spread_2ms(hf_n));
                        if hf_mag_only
                             s_ch(c,b) *= abs(ahf_model);
                        else
                             s_ch(c,b) *= ahf_model;
                        end
                        hf_model(hf_n, c) = ahf_model;
                    end

                    % apply HF model (time shifted into the future) to parity symbol carriers 

                    for c=1:Ncp
                        ahf_model = hf_gain*(spread(hf_n+Rs) + exp(-j*k*wsep*nhfdelay)*spread_2ms(hf_n+Rs));
                        if hf_mag_only
                             s_ch(Nck+c,b) *= abs(ahf_model);
                        else
                             s_ch(Nck+c,b) *= ahf_model;
                        end
                        hf_model(hf_n, Nck+c) = ahf_model;
                   end

                    hf_n++;
                end
            end
           
            for b=1:Nb
              for c=1:Nc
                tx_symb_log = [tx_symb_log s_ch(c,b)];
              end
            end

            % AWGN noise and phase/freq offset channel simulation
            % 0.5 factor ensures var(noise) == variance , i.e. splits power between Re & Im

            noise = sqrt(variance*0.5)*(randn(Nc,Nb) + j*randn(Nc,Nb));

            s_ch = s_ch + noise;

            % de-modulate

            rx_bits = zeros(1, n);
            for b=1:Nb
              for c=1:Nc
                rx_symb(c,b) = s_ch(c,b);
                if strcmp(modulation,'dqpsk')
                    tmp = rx_symb(c,b);
                    rx_symb(c,b) *= conj(prev_sym_rx(c)/abs(prev_sym_rx(c)));
                    prev_sym_rx(c) = tmp;
                end
                i = (b-1)*Nc + c;
                rx_bits((2*(i-1)+1):(2*i)) = qpsk_demod(rx_symb(c,b));
                rx_symb_log = [rx_symb_log rx_symb(c,b)];
              end
            end

            % Measure BER

            error_positions = xor(rx_bits, tx_bits);
            Nerrs = sum(error_positions);
            Terrs += Nerrs;
            Tbits += length(tx_bits);
            errors_log = [errors_log error_positions];
            Nerrs_log = [Nerrs_log Nerrs];
        end
    
        TERvec(ne) = Terrs;
        BERvec(ne) = Terrs/Tbits;

        if verbose 
            av_tx_pwr = (tx_symb_log * tx_symb_log')/length(tx_symb_log)

            printf("EsNo (dB): %3.1f  Terrs: %d BER %4.3f QPSK BER theory %4.3f av_tx_pwr: %3.2f", EsNodB, Terrs,
                   Terrs/Tbits, 0.5*erfc(sqrt(EsNo/2)), av_tx_pwr);
            printf("\n");
        end
        if verbose > 1
            printf("Terrs: %d BER %f BER theory %f C %f N %f Es %f No %f Es/No %f\n\n", Terrs,
                   Terrs/Tbits, 0.5*erfc(sqrt(EsNo/2)), var(tx_symb_log), var(noise_log),
                   var(tx_symb_log), var(noise_log), var(tx_symb_log)/var(noise_log));
        end
    end
    
    Ebvec = Esvec - 10*log10(bps);
    sim_out.BERvec          = BERvec;
    sim_out.Ebvec           = Ebvec;
    sim_out.TERvec          = TERvec;
    sim_out.errors_log      = errors_log;

    if plot_scatter
        figure(2);
        clf;
        scat = rx_symb_log .* exp(j*pi/4);
        plot(real(scat), imag(scat),'+');
        title('Scatter plot');

        if hf_sim
          figure(3);
          clf;
        
          y = 1:(hf_n-1);
          x = 1:Nc;
          EsNodBSurface = 20*log10(abs(hf_model(y,:))) - 10*log10(variance);
          EsNodBSurface(find(EsNodBSurface < -5)) = -5;
          mesh(x,y,EsNodBSurface);
          grid
          axis([1 Nc 1 Rs*5 -5 15])
          title('HF Channel Es/No');

          if verbose 
            av_hf_pwr = sum(abs(hf_model(y)).^2)/(hf_n-1);
            printf("average HF power: %3.2f over %d symbols\n", av_hf_pwr, hf_n-1);
          end
        end

        figure(4)
        clf
        stem(Nerrs_log)
   end

endfunction

% Gray coded QPSK modulation function

function symbol = qpsk_mod(two_bits)
    two_bits_decimal = sum(two_bits .* [2 1]); 
    switch(two_bits_decimal)
        case (0) symbol =  1;
        case (1) symbol =  j;
        case (2) symbol = -j;
        case (3) symbol = -1;
    endswitch
endfunction

% Gray coded QPSK demodulation function

function two_bits = qpsk_demod(symbol)
    if isscalar(symbol) == 0
        printf("only works with scalars\n");
        return;
    end
    bit0 = real(symbol*exp(j*pi/4)) < 0;
    bit1 = imag(symbol*exp(j*pi/4)) < 0;
    two_bits = [bit1 bit0];
endfunction

function sim_in = standard_init
  sim_in.verbose          = 1;
  sim_in.plot_scatter     = 0;

  sim_in.Esvec            = 5; 
  sim_in.Ntrials          = 30;
  sim_in.Rs               = 50;
  sim_in.Nc               = 8;

  sim_in.phase_offset     = 0;
  sim_in.w_offset         = 0;
  sim_in.phase_noise_amp  = 0;

  sim_in.hf_delay_ms      = 2;
  sim_in.hf_sim           = 0;
  sim_in.hf_mag_only      = 0;
endfunction

function test_curves

  sim_in = standard_init();

  sim_in.verbose          = 1;
  sim_in.plot_scatter     = 1;

  sim_in.Esvec            = 50; 
  sim_in.hf_sim           = 0;
  sim_in.Ntrials          = 1000;

  sim_qpsk_hf             = ber_test(sim_in, 'qpsk');

  sim_in.hf_sim           = 0;
  sim_in.plot_scatter     = 0;
  sim_in.Esvec            = 5:15; 
  Ebvec = sim_in.Esvec - 10*log10(2);
  BER_theory = 0.5*erfc(sqrt(10.^(Ebvec/10)));
  sim_dqpsk               = ber_test(sim_in, 'dqpsk');
  sim_in.hf_sim           = 1;
  sim_in.hf_mag_only      = 1;
  sim_qpsk_hf             = ber_test(sim_in, 'qpsk');
  sim_in.hf_mag_only      = 0;
  sim_dqpsk_hf            = ber_test(sim_in, 'dqpsk');
  sim_in.Nchip            = 4;
  sim_dqpsk_hf_dsss       = ber_test(sim_in, 'dqpsk');

  figure(1); 
  clf;
  semilogy(Ebvec, BER_theory,'r;QPSK theory;')
  hold on;
  semilogy(sim_dqpsk.Ebvec, sim_dqpsk.BERvec,'c;DQPSK AWGN;')
  semilogy(sim_qpsk_hf.Ebvec, sim_qpsk_hf.BERvec,'b;QPSK HF;')
  semilogy(sim_dqpsk_hf.Ebvec, sim_dqpsk_hf.BERvec,'k;DQPSK HF;')
  semilogy(sim_dqpsk_hf_dsss.Ebvec, sim_dqpsk_hf_dsss.BERvec,'g;DQPSK DSSS HF;')
  hold off;

  xlabel('Eb/N0')
  ylabel('BER')
  grid("minor")
  axis([min(Ebvec) max(Ebvec) 1E-3 1])
endfunction

function test_1600_v_450

  sim_in = standard_init();

  sim_in.verbose          = 1;
  sim_in.plot_scatter     = 1;
  sim_in.Ntrials          = 500;
  sim_in.hf_sim           = 1;

  sim_in.framesize        = 32;
  sim_in.Nc               = 16;
  sim_in.Esvec            = 7; 
  sim_in.Nchip            = 1;
  
  sim_dqpsk_hf_1600        = ber_test(sim_in, 'dqpsk');

  sim_in.framesize        = 8;
  sim_in.Nc               = 4;
  sim_in.Esvec            = sim_in.Esvec + 10*log10(1600/450); 
  sim_in.Nchip            = 4;
  
  sim_dqpsk_hf_450         = ber_test(sim_in, 'dqpsk');
  
  fep=fopen("errors_1600.bin","wb"); fwrite(fep, sim_dqpsk_hf_1600.errors_log, "short"); fclose(fep);
  fep=fopen("errors_450.bin","wb"); fwrite(fep, sim_dqpsk_hf_450.errors_log, "short"); fclose(fep);

endfunction

function test_single

  sim_in = standard_init();

  sim_in.verbose          = 1;
  sim_in.plot_scatter     = 1;

  sim_in.n                = 48*8;
  sim_in.k                = 24*8;
  snr                     = 1;
  sim_in.Esvec            = snr + 10*log10(3000/400); 
  sim_in.hf_sim           = 1;
  sim_in.Ntrials          = 100;
  
  sim_qpsk_hf             = ber_test(sim_in, 'dqpsk');
endfunction


% Start simulations ---------------------------------------

more off;

%test_1600_v_450();
%test_curves();
test_single();
