//
// anyRemote
// a bluetooth remote for your PC.
//
// Copyright (C) 2006-2013 Mikhail Fedotov <anyremote@mail.ru>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <time.h>
#include <unistd.h>
#ifdef __FreeBSD__
#include <signal.h>
#endif

#include "lib_wrapper.h"
#include "atsend.h"
#include "btio.h"
#include "conf.h"
#include "common.h"
#include "utils.h"
#include "executor.h"
#include "dispatcher.h"
#include "security.h"
#include "list.h"
#include "alarm.h"

#include "pr_frontend.h"
#include "pr_btspp.h"
#include "pr_socket.h"
#ifdef USE_L2CAP
#include "pr_l2cap.h"
#endif
#include "pr_web.h"
#include "pr_stdin.h"
#include "queue.h"
#include "peer.h"

#define TMPDISCONN_TIME 60

extern CONF    conf;
extern char    tmp[MAXMAXLEN];

extern char callerId[MAXLEN];

boolean_t dispIsJoinable = BOOL_YES;


int  gotExitSignal  = 0;

static int         _callTimer     = 0;
static int         _initialized  = 0;
static string_t*   _readBuffer   = NULL; // store statically, not to reallocate

// Then port will be closed from a forked child use 0
void closePort(int final)
{
    if (final) {
        logger(L_DBG,"[DS]: closePort");
    }

    closePeers(final);

    if (getFrontEnd() > 0) {
        if (final)  {
            sendToFrontEnd("Exiting");
        }
        disconnectFrontEnd();
    }
}

static void handleActiveCall(void)
{
    if (_callTimer >= 20) {  // about a second

        // Will set global callerId [MAXLEN];
    
    boolean_t stillActive = checkActiveCall();
    if (!stillActive) {
        strcpy(callerId, "NO CALLER ID");
        
        eMessage* em = (eMessage*) malloc(sizeof(eMessage));
        em->type  = EM_KEY;
        em->value = strdup("Msg:EndCall(,)");

        DEBUG2("[DS]: Send message to executor thread %s", (char*) em->value);
        sendToExecutor(em);
        }
    
        _callTimer = 0;
    } else {
        _callTimer++;
    }
}

char* readFromFile(char *cmdTag, char* file, int* size)
{
    FILE *fp;
    struct stat buf;

    long fLen = 0;
    if(stat(file, &buf) == -1) {
        logger(L_ERR,"can't get file size!");
    } else {
        fLen = (int) buf.st_size;

        if (!S_ISREG (buf.st_mode)) {
            logger(L_ERR,"not regular file");
        } else {

            fp=fopen(file,"r");
            if (fp == NULL) {
                logger(L_ERR,"can't open file!");
            } else {

                int prefixSz = strlen(cmdTag);

                uint32_t szh32 = (uint32_t) buf.st_size;
                uint32_t szi32 = htonl(szh32);

                char * fBuffer = (char*) calloc(fLen+prefixSz+7,1);    // 7 =   <size of bin data=4bytes>    ");\0"
                if (fBuffer == NULL) {
                    logger(L_ERR,"no more memory!");

                    *size = -1;
                    return NULL;
                }

                strcat(fBuffer, cmdTag);
                memcpy((void*)fBuffer+prefixSz, (const void *) &szi32, 4);    // length on binary data

                if (fp) {
#ifdef __cplusplus
                    size_t dummy =
#endif
                        fread(fBuffer+prefixSz+4, sizeof(char), fLen, fp);

                    fclose(fp);
                }
                strcpy(fBuffer+prefixSz+fLen+4,");");

                *size = fLen+prefixSz+7;
                return fBuffer;
            }
        }
    }

    if (strncmp(cmdTag, "Set(cover", 9) == 0) {  // cleanup the cover on client

        char * fBuffer = (char*) malloc(12);
        strcpy(fBuffer, "Set(cover);");

        *size = 11;
        return fBuffer;
    }

    *size = -1;
    return NULL;
}

dMessage* getDFinalizer()
{
    dMessage* dm = (dMessage*) malloc(sizeof(dMessage));
    dm->type  = DM_SET;
    dm->size  = 6;
    dm->value = (void*) strdup("End();");
    dm->file  = NULL;
    return dm;
}

void freeDMessage(void *ptr)
{
    dMessage* dm = (dMessage*) ptr;
    if (dm) {
        if (dm->value != NULL) {
            free(dm->value);
        }
        if (dm->file != NULL) {
            free(dm->file);
        }
        free(dm);
    }
}

static int dispatchEventInit(dMessage*  dm)
{
    logger(L_INF, "[DS]: Got init OK event");
    return EXIT_INITOK;
}

static int dispatchEventExit(dMessage* dm)
{
    logger(L_INF, "[DS]: Got exit event");

    gotExitSignal = 1;

    // in server mode wait client to close connection and then exit
    if (disconnectPeers() == 1) {
        logger(L_INF, "[DS]: Got exit event: exit immediately");
        closePort(1);
        return EXIT_ABORT;
    }
    return EXIT_OK;
}

static int dispatchDisconnect(dMessage* dm)
{
    logger(L_INF, "[DS]: Got disconnect event");
    disconnectPeers();
    closePort(0);
    return EXIT_DISCON;
}

static int dispatchEventSendString(dMessage* dm)
{
    DEBUG2("[DS]: Send(string) %s", (char*) dm->value);
    return writePeers(dm);
}

static int dispatchEventSendBytes(dMessage* dm)
{
    DEBUG2("[DS]: Send(bytes) %s", (char*) dm->value);
    return writeBytesPeers((char*) dm->value);
}

static int dispatchEventSet(dMessage* dm)
{
    logger(L_DBG, "[DS]: Set(...)");
    return writePeers(dm);
}

static struct {
    int id;
    int (*hook)(dMessage* p);
} _dispMsgHooks[] = {
    {DM_SET,              dispatchEventSet       },
    {DM_SETFILE,          writeFilePeers         },
    {DM_SENDB,            dispatchEventSendBytes },
    {DM_SENDS,            dispatchEventSendString},
    {DM_CKPD,             writeCKPD              },
    {DM_CMER,             writeCMER              },
    {DM_EVENT_INIT,       dispatchEventInit      },
    {DM_EVENT_DISCONNECT, dispatchDisconnect     },
    {DM_EVENT_EXIT,       dispatchEventExit      }
};

static int checkMsgQueue()
{
    //logger(L_DBG,"[DS]: checkMsgQueue");

    int ret = EXIT_OK;

    // Verify commands from queue (timeout about 1/2 sec)
    dMessage* dm = (dMessage*) queuePop(Q_DISP);
    if (dm != NULL) {

        //DEBUG2("[DS]: Got event %p %d", dm, dm->type);

        if (connected() == EXIT_NOK &&
                !(dm->type == DM_EVENT_INIT ||     // can process these 3 even if no connection
                  dm->type == DM_EVENT_EXIT ||
                  dm->type == DM_EVENT_DISCONNECT)) {

            logger(L_DBG, "[DS]: No connection. Skip event");

        } else {

            ret = (_dispMsgHooks[dm->type].hook)(dm);

        }

        freeDMessage(dm);
    }
    if (ret == EXIT_ABORT) { DEBUG2("[DS]: checkMsgQueue ret %d", ret); }
    return ret;
}

void parseCommand(char* cmd)
{
    //DEBUG2("[DS]: parseCommand >%s<", cmd);

    char *prev, *next;

    if (cmd == NULL) {
        return ;
    }

    //skip lines starting with \n and \r
    if (cmd[0] == '\r') {
        cmd++;
    }
    if (cmd[0] == '\n') {
        cmd++;
    }

    // most common case
    if (!cmd[0]) {
        return;
    }

    // if recieved multiline command - handle line by line and return
    prev = cmd;
    next = strchr(cmd, '\r');

    if (next == NULL) {    // Java client will send +CKEV: 1,1; +CKEV: 1,0
        next = strchr(cmd, ';');
    }

    if (next) {
        logger(L_DBG, "[DS]: parseCommand multiline");

        char copy[1024];
        int len;

        do {
            len = next-prev;

            if (len >= 2) {
                memcpy(copy, prev, len);
                copy[len] = 0;

                // use recursion
                parseCommand(copy);
            }
            prev = next+1;
            next = strchr(prev, '\r');
            if (next == NULL) { // Java client will send +CKEV: 1,1; +CKEV: 1,0
                next = strchr(prev, ';');
            }

            /* handle in reader
            if (next == NULL && getIViewer()) {
                next = strchr(prev, '\3');        // end-of-text marker in CommandFusion
            }*/

        } while (next) ;

        // now return
        return;
    }

    logger(L_DBG,"[DS]: -------------------- Command read --------------------");
    DEBUG2("[DS]: parseCommand >%s<", cmd);

    if (IS_OK(cmd)) {   // OK - nothing to do
        return;
    }

    if (strncmp(cmd, "AT+CKPD=", 8) == 0) {        // This is echo of sent message in AT mode; nothing to do
        return;
    }

    if (strncmp(cmd, "Msg:_DEBUG_(", 12) == 0) {   // This is debug message from java client; nothing to do
        return;
    }
    
    // This is keepalive message, handle it internally
    if (strncmp(cmd, "Msg:Ping", 8) == 0 && useKeepalive()) {
        setKeepalive();
    return;
    }
    
    eMessage* em = (eMessage*) malloc(sizeof(eMessage));
    em->type  = EM_KEY;
    em->value = strdup(cmd);

    DEBUG2("[DS]: Send message to executor thread %s", (char*) em->value);
    sendToExecutor(em);

    return;
}

static int read_command(char* buf, int l)
{
    if (!_readBuffer) {
        _readBuffer = stringNew("");
    }
    stringTruncate(_readBuffer,0);

    int rc = readPeers(_readBuffer);
    if (rc == EOF) {
        return EOF;
    }

    char* start = _readBuffer->str;
    int   sz = rc;
    while (*start == '\r' && sz >= 0) {
        start++;
        sz--;
    }
    strncpy(buf, start, sz);
    buf[sz] = '\0';

    return rc;
}

static void hookInitOnce()
{
    if (_initialized == 0) {

        // setgid
        if(conf.uid && getuid()==0) {
            DEBUG2("[DS]: setuid/setgid %d,%d",conf.uid,conf.gid);
            setgid(conf.gid);
            setuid(conf.uid);
        }

        _initialized++;
    }
}

static int doDisconnect()
{
    logger(L_INF, "[DS]: Got disconnected");
    //printf("Got disconnected\n");

    freeBtAddress();

    if (gotExitSignal) {
        logger(L_INF, "[DS]: Got signal, exiting");
        closePort(1);
        return EXIT_ABORT;

    }
    sendToFrontEnd("Disconnected");
    sendDisconnect();
    sendEventToExecutor(ID_EVT_DISCONNECT);

    if (disconnectPeers() == 1) {
        logger(L_INF, "[DS]: Closing the port");
        closePort(0);
    }

    return EXIT_DISCON;
}

void sendFinalizer()
{
    if (needFinalizer() == EXIT_OK) {
        eMessage* em = (eMessage*) malloc(sizeof(eMessage));
        em->type  = EM_AS_IS;
        em->value = strdup("End();");
        sendToExecutor(em);
    }
}

void dispatcherCleanup()
{
    logger(L_DBG,"[DS]: dispatcherCleanup()");
    freePeers();
    
    if (_readBuffer) {
        stringFree(_readBuffer, BOOL_YES);
        _readBuffer = NULL;
    }
}

static int doReadPeers()
{
    char answer[MAXCMDLEN];

    int retRead = read_command(answer, sizeof(answer));
    //if (retRead != 0) printf("GOT %d\n",retRead);

    if (retRead == EOF) {
        logger(L_DBG,"[DS]: doReadPeers() EOF");
        return doDisconnect();

    } else if (retRead > 0) {

        parseCommand(answer);
        sendFinalizer();

    }
    return EXIT_OK;
}

static int doMessageLoop()
{
    int ret = EXIT_DISCON;

    while (1) {
 
        // read from sockets, etc.
        ret = doReadPeers();

        if (ret == EXIT_ABORT) {
            logger(L_DBG,"[DS]: doMessageLoop abort on read ");
            break;
        } else if (ret == EXIT_DISCON) {
            logger(L_DBG,"[DS]: doMessageLoop disconnect on read");
            break;
        }

        // Is call still active ? (timeout about 1 seconds, check it inside)
        if (hasActiveCall()) {
            handleActiveCall();
        }
    
        ret = checkMsgQueue();
        if (ret == EXIT_ABORT) {
            logger(L_DBG,"[DS]: doMessageLoop abort on check queue");
            break;
        } else if (ret == EXIT_DISCON) {
            logger(L_DBG,"[DS]: doMessageLoop disconnect on check queue");
            break;
        }

        // Main loop timer (1/50 of second)
        usleep(20000);
    } // while (forever)

    return ret;
}

static int doConnectionLoop()
{
    if (setupPeers() == 1) {  // Init modem: AT, ATE0, AT+CMER, in server mode waits for connection

        logger(L_DBG,"[DS]: Init connection OK");
        
        if (connected() == EXIT_OK) {
            connectNotify();
        }

        dispIsJoinable = BOOL_YES;

        logger(L_DBG,"[DS]: Start message loop");

        int ret = doMessageLoop();
        
        DEBUG2("[DS]: Stop message loop %d", ret);
        
        return ret;

        // EXIT_DISCON / EXIT_NOK is OK here

    } else {        // init port
        logger(L_DBG,"[DS]: Init connection error");
    }
    return EXIT_OK;
}

pointer_t dispatcherRoutine(pointer_t thread)
{
    int ret = EXIT_OK;

    logger(L_DBG,"[DS]: start dispatcher thread");

    setCmdFiles();

    // wait init ok event
    while (1) {
        ret = checkMsgQueue();
        if (ret == EXIT_ABORT) {
            dispatcherCleanup();
            return NULL;
        } else if (ret == EXIT_INITOK) {
            break;
        }
        //logger(L_DBG,"[DS]: wait init OK event");
        usleep(50000);
    }
    logger(L_DBG,"[DS]: got init event");

    int dmn = autoConnect();
    int rs  = getRetrySecs();
    strcpy(callerId, "NO CALLER ID");

    if (definePeers() == EXIT_ABORT) {
        logger(L_DBG,"[DS]: Incorrect device specification");
        dispatcherCleanup();
        sendAbort();
        return NULL;
    }

    while (1) {

        logger(L_DBG,"[DS]: ************ outer loop **********");

        if(openPeers() == EXIT_OK) {   // Open device

            logger(L_DBG,"[DS]: Device open OK");

            hookInitOnce();

            dispIsJoinable = BOOL_NO;

            DEBUG2("[DS]: Start connection loop");
            int retLoop = doConnectionLoop();
            DEBUG2("[DS]: Stop connection loop %d", retLoop);
            
            if (retLoop == EXIT_ABORT) {
                DEBUG2("[DS]: Dispatcher abort");
                dispatcherCleanup();
                return NULL;
            }
           
        } else {        // open port
            logger(L_DBG,"[DS]: Device open error");
        }

        //printf("Connection closed or lost\n");
        logger(L_INF, "[DS]: Connection closed or lost");

        // Can't open port or it closed again

        int isServer = isServerMode();

        if (!gotExitSignal &&
                (dmn || isServer == EXIT_OK || ret == EXIT_DISCON)) {

            int timeout;

            if (isServer == EXIT_OK) {

                timeout = 2;    // wait only 2 seconds

            } else if (ret == EXIT_DISCON) {

                timeout = TMPDISCONN_TIME;
                ret = EXIT_OK;

            } else {
                timeout = rs;
            }

            INFO2("[DS]: Wait %d seconds to connect/open server socket ...", timeout);
            //printf("Wait %d seconds to connect/open server socket ...\n", timeout);

            dispIsJoinable = BOOL_NO;
            sleep(timeout);
            dispIsJoinable = BOOL_YES;

        } else {    // Proceed to exit
            printf("Exiting ...\n");
            sendAbort();
            break;
        }
    }

    // Finish all
    logger(L_INF, "[DS]: Stop dispatcher ...");
    closePort(1);
    dispatcherCleanup();

    return NULL;
}

void sendToDispatcher(dMessage *buf)
{
    if (queuePush(Q_DISP, (void*) buf) == RC_OK) {
        DEBUG2("send to dispatcher %d", buf->type);
    }
}
