% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eta_squared-main.R, R/eta_squared_posterior.R
\name{eta_squared}
\alias{eta_squared}
\alias{omega_squared}
\alias{epsilon_squared}
\alias{cohens_f}
\alias{cohens_f_squared}
\alias{eta_squared_posterior}
\title{\eqn{\eta^2} and Other Effect Size for ANOVA}
\usage{
eta_squared(
  model,
  partial = TRUE,
  generalized = FALSE,
  ci = 0.95,
  alternative = "greater",
  verbose = TRUE,
  ...
)

omega_squared(
  model,
  partial = TRUE,
  ci = 0.95,
  alternative = "greater",
  verbose = TRUE,
  ...
)

epsilon_squared(
  model,
  partial = TRUE,
  ci = 0.95,
  alternative = "greater",
  verbose = TRUE,
  ...
)

cohens_f(
  model,
  partial = TRUE,
  generalized = FALSE,
  squared = FALSE,
  method = c("eta", "omega", "epsilon"),
  model2 = NULL,
  ci = 0.95,
  alternative = "greater",
  verbose = TRUE,
  ...
)

cohens_f_squared(
  model,
  partial = TRUE,
  generalized = FALSE,
  squared = TRUE,
  method = c("eta", "omega", "epsilon"),
  model2 = NULL,
  ci = 0.95,
  alternative = "greater",
  verbose = TRUE,
  ...
)

eta_squared_posterior(
  model,
  partial = TRUE,
  generalized = FALSE,
  ss_function = stats::anova,
  draws = 500,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{model}{An ANOVA table (or an ANOVA-like table, e.g., outputs from
\code{parameters::model_parameters}), or a statistical model for which such a
table can be extracted. See details.}

\item{partial}{If \code{TRUE}, return partial indices.}

\item{generalized}{A character vector of observed (non-manipulated) variables
to be used in the estimation of a generalized Eta Squared. Can also be
\code{TRUE}, in which case generalized Eta Squared is estimated assuming \emph{none}
of the variables are observed (all are manipulated). (For \code{afex_aov}
models, when \code{TRUE}, the observed variables are extracted automatically
from the fitted model, if they were provided during fitting.}

\item{ci}{Confidence Interval (CI) level}

\item{alternative}{a character string specifying the alternative hypothesis;
Controls the type of CI returned: \code{"greater"} (default) or \code{"less"}
(one-sided CI), or \code{"two.sided"} (default, two-sided CI). Partial matching
is allowed (e.g., \code{"g"}, \code{"l"}, \code{"two"}...). See \emph{One-Sided CIs} in
\link{effectsize_CIs}.}

\item{verbose}{Toggle warnings and messages on or off.}

\item{...}{Arguments passed to or from other methods.
\itemize{
\item Can be \code{include_intercept = TRUE} to include the effect size for the intercept (when it is included in the ANOVA table).
\item For Bayesian models, arguments passed to \code{ss_function}.
}}

\item{squared}{Return Cohen's \emph{f} or Cohen's \emph{f}-squared?}

\item{method}{What effect size should be used as the basis for Cohen's \emph{f}?}

\item{model2}{Optional second model for Cohen's f (/squared). If specified,
returns the effect size for R-squared-change between the two models.}

\item{ss_function}{For Bayesian models, the function used to extract
sum-of-squares. Uses \code{\link[=anova]{anova()}} by default, but can also be \code{car::Anova()}
for simple linear models.}

\item{draws}{For Bayesian models, an integer indicating the number of draws
from the posterior predictive distribution to return. Larger numbers take
longer to run, but provide estimates that are more stable.}
}
\value{
A data frame with the effect size(s) between 0-1 (\code{Eta2}, \code{Epsilon2},
\code{Omega2}, \code{Cohens_f} or \code{Cohens_f2}, possibly with the \code{partial} or
\code{generalized} suffix), and their CIs (\code{CI_low} and \code{CI_high}).
\cr\cr
For \code{eta_squared_posterior()}, a data frame containing the ppd of the Eta
squared for each fixed effect, which can then be passed to
\code{\link[bayestestR:describe_posterior]{bayestestR::describe_posterior()}} for summary stats.

A data frame containing the effect size values and their confidence
intervals.
}
\description{
Functions to compute effect size measures for ANOVAs, such as Eta-
(\eqn{\eta}), Omega- (\eqn{\omega}) and Epsilon- (\eqn{\epsilon}) squared,
and Cohen's f (or their partialled versions) for ANOVA tables. These indices
represent an estimate of how much variance in the response variables is
accounted for by the explanatory variable(s).
\cr\cr
When passing models, effect sizes are computed using the sums of squares
obtained from \code{anova(model)} which might not always be appropriate. See
details.
}
\details{
For \code{aov} (or \code{lm}), \code{aovlist} and \code{afex_aov} models, and for \code{anova} objects
that provide Sums-of-Squares, the effect sizes are computed directly using
Sums-of-Squares. (For \code{maov} (or \code{mlm}) models, effect sizes are computed for
each response separately.)
\cr\cr
For other ANOVA tables and models (converted to ANOVA-like tables via
\code{anova()} methods), effect sizes are approximated via test statistic
conversion of the omnibus \emph{F} statistic provided by the (see \code{\link[=F_to_eta2]{F_to_eta2()}}
for more details.)
\subsection{Type of Sums of Squares}{

When \code{model} is a statistical model, the sums of squares (or \emph{F} statistics)
used for the computation of the effect sizes are based on those returned by
\code{anova(model)}. Different models have different default output type. For
example, for \code{aov} and \code{aovlist} these are \emph{type-1} sums of squares, but for
\code{lmerMod} (and \code{lmerModLmerTest}) these are \emph{type-3} sums of squares. Make
sure these are the sums of squares you are interested in. You might want to
convert your model to an ANOVA(-like) table yourself and then pass the result
to \code{eta_squared()}. See examples below for use of \code{car::Anova()} and the
\code{afex} package.
\cr\cr
For type 3 sum of squares, it is generally recommended to fit models with
\emph{orthogonal factor weights} (e.g., \code{contr.sum}) and \emph{centered covariates},
for sensible results. See examples and the \code{afex} package.
}

\subsection{Un-Biased Estimate of Eta}{

Both \emph{\strong{Omega}} and \emph{\strong{Epsilon}} are unbiased estimators of the
population's \emph{\strong{Eta}}, which is especially important is small samples. But
which to choose?
\cr\cr
Though Omega is the more popular choice (Albers and Lakens, 2018), Epsilon is
analogous to adjusted R2 (Allen, 2017, p. 382), and has been found to be less
biased (Carroll & Nordholm, 1975).
}

\subsection{Cohen's f}{

Cohen's f can take on values between zero, when the population means are all
equal, and an indefinitely large number as standard deviation of means
increases relative to the average standard deviation within each group.
\cr\cr
When comparing two models in a sequential regression analysis, Cohen's f for
R-square change is the ratio between the increase in R-square
and the percent of unexplained variance.
\cr\cr
Cohen has suggested that the values of 0.10, 0.25, and 0.40 represent small,
medium, and large effect sizes, respectively.
}

\subsection{Eta Squared from Posterior Predictive Distribution}{

For Bayesian models (fit with \code{brms} or \code{rstanarm}),
\code{eta_squared_posterior()} simulates data from the posterior predictive
distribution (ppd) and for each simulation the Eta Squared is computed for
the model's fixed effects. This means that the returned values are the
population level effect size as implied by the posterior model (and not the
effect size in the sample data). See \code{\link[rstantools:posterior_predict]{rstantools::posterior_predict()}} for
more info.
}
}
\section{Confidence (Compatibility) Intervals (CIs)}{
Unless stated otherwise, confidence (compatibility) intervals (CIs) are
estimated using the noncentrality parameter method (also called the "pivot
method"). This method finds the noncentrality parameter ("\emph{ncp}") of a
noncentral \emph{t}, \emph{F}, or \eqn{\chi^2} distribution that places the observed
\emph{t}, \emph{F}, or \eqn{\chi^2} test statistic at the desired probability point of
the distribution. For example, if the observed \emph{t} statistic is 2.0, with 50
degrees of freedom, for which cumulative noncentral \emph{t} distribution is \emph{t} =
2.0 the .025 quantile (answer: the noncentral \emph{t} distribution with \emph{ncp} =
.04)? After estimating these confidence bounds on the \emph{ncp}, they are
converted into the effect size metric to obtain a confidence interval for the
effect size (Steiger, 2004).
\cr\cr
For additional details on estimation and troubleshooting, see \link{effectsize_CIs}.
}

\section{CIs and Significance Tests}{
"Confidence intervals on measures of effect size convey all the information
in a hypothesis test, and more." (Steiger, 2004). Confidence (compatibility)
intervals and p values are complementary summaries of parameter uncertainty
given the observed data. A dichotomous hypothesis test could be performed
with either a CI or a p value. The 100 (1 - \eqn{\alpha})\% confidence
interval contains all of the parameter values for which \emph{p} > \eqn{\alpha}
for the current data and model. For example, a 95\% confidence interval
contains all of the values for which p > .05.
\cr\cr
Note that a confidence interval including 0 \emph{does not} indicate that the null
(no effect) is true. Rather, it suggests that the observed data together with
the model and its assumptions combined do not provided clear evidence against
a parameter value of 0 (same as with any other value in the interval), with
the level of this evidence defined by the chosen \eqn{\alpha} level (Rafi &
Greenland, 2020; Schweder & Hjort, 2016; Xie & Singh, 2013). To infer no
effect, additional judgments about what parameter values are "close enough"
to 0 to be negligible are needed ("equivalence testing"; Bauer & Kiesser,
1996).
}

\section{Plotting with \code{see}}{

The \code{see} package contains relevant plotting functions. See the \href{https://easystats.github.io/see/articles/effectsize.html}{plotting vignette in the \code{see} package}.
}

\examples{
data(mtcars)
mtcars$am_f <- factor(mtcars$am)
mtcars$cyl_f <- factor(mtcars$cyl)

model <- aov(mpg ~ am_f * cyl_f, data = mtcars)

(eta2 <- eta_squared(model))

# More types:
eta_squared(model, partial = FALSE)
eta_squared(model, generalized = "cyl_f")
omega_squared(model)
epsilon_squared(model)
cohens_f(model)

model0 <- aov(mpg ~ am_f + cyl_f, data = mtcars) # no interaction
cohens_f_squared(model0, model2 = model)

## Interpretation of effect sizes
## ------------------------------

interpret_omega_squared(0.10, rules = "field2013")
interpret_eta_squared(0.10, rules = "cohen1992")
interpret_epsilon_squared(0.10, rules = "cohen1992")

interpret(eta2, rules = "cohen1992")

\dontshow{if (require("see") && interactive()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
plot(eta2) # Requires the {see} package
\dontshow{\}) # examplesIf}
\dontshow{if (require("car")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
# Recommended: Type-2 or -3 effect sizes + effects coding
# -------------------------------------------------------
contrasts(mtcars$am_f) <- contr.sum
contrasts(mtcars$cyl_f) <- contr.sum

model <- aov(mpg ~ am_f * cyl_f, data = mtcars)
model_anova <- car::Anova(model, type = 3)

epsilon_squared(model_anova)
\dontshow{\}) # examplesIf}
\dontshow{if (require("car") && require("afex")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
# afex takes care of both type-3 effects and effects coding:
data(obk.long, package = "afex")
model <- afex::aov_car(value ~ gender + Error(id / (phase * hour)),
  data = obk.long, observed = "gender"
)

omega_squared(model)
eta_squared(model, generalized = TRUE) # observed vars are pulled from the afex model.
\dontshow{\}) # examplesIf}
\dontshow{if (require("lmerTest") && require("lme4") && FALSE) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
## Approx. effect sizes for mixed models
## -------------------------------------
model <- lme4::lmer(mpg ~ am_f * cyl_f + (1 | vs), data = mtcars)
omega_squared(model)
\dontshow{\}) # examplesIf}
\dontshow{if (require(rstanarm) && require(bayestestR) && require(car) && interactive()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
## Bayesian Models (PPD)
## ---------------------
fit_bayes <- rstanarm::stan_glm(
  mpg ~ factor(cyl) * wt + qsec,
  data = mtcars, family = gaussian(),
  refresh = 0
)

es <- eta_squared_posterior(fit_bayes,
  verbose = FALSE,
  ss_function = car::Anova, type = 3
)
bayestestR::describe_posterior(es, test = NULL)


# compare to:
fit_freq <- lm(mpg ~ factor(cyl) * wt + qsec,
  data = mtcars
)
aov_table <- car::Anova(fit_freq, type = 3)
eta_squared(aov_table)
\dontshow{\}) # examplesIf}
}
\references{
\itemize{
\item Albers, C., and Lakens, D. (2018). When power analyses based on pilot data
are biased: Inaccurate effect size estimators and follow-up bias. Journal of
experimental social psychology, 74, 187-195.
\item Allen, R. (2017). Statistics and Experimental Design for Psychologists: A
Model Comparison Approach. World Scientific Publishing Company.
\item Carroll, R. M., & Nordholm, L. A. (1975). Sampling Characteristics of
Kelley's epsilon and Hays' omega. Educational and Psychological Measurement,
35(3), 541-554.
\item Kelley, T. (1935) An unbiased correlation ratio measure. Proceedings of the
National Academy of Sciences. 21(9). 554-559.
\item Olejnik, S., & Algina, J. (2003). Generalized eta and omega squared
statistics: measures of effect size for some common research designs.
Psychological methods, 8(4), 434.
\item Steiger, J. H. (2004). Beyond the F test: Effect size confidence intervals
and tests of close fit in the analysis of variance and contrast analysis.
Psychological Methods, 9, 164-182.
}
}
\seealso{
\code{\link[=F_to_eta2]{F_to_eta2()}}

Other effect sizes for ANOVAs: 
\code{\link{rank_epsilon_squared}()}
}
\concept{effect sizes for ANOVAs}
