
#include "image.h"
#include "moon.h"
#include "pngwrite.h"
#include "stars.h"
#include "fail.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#define VERSION "1.2"

static inline void usage( char *name )
{
   char *base = name;
   char *c;
   
   for( c = name+1; *c; c++)
   {
      if( *(c-1) == '/' ) base = c; 
   }
   
   printf( "this is pngphoon version " VERSION "\n\n"
           "usage: %s options (* = mandatory)\n"
           "\t-w # width*\n"
           "\t-h # height*\n"
           "\t-f $ filename*\n"
           "\t-x # number of moons in x axis (for multihead configurations)\n"
           "\t-y # number of moons in y axis (for multihead configurations)\n"
           "\t-s # star density\n"
           "\t-b   black (no earthlight)\n"
           "\t-q   quiet (disable warnings)\n", base );
   exit(0);
}

int main( int argc, char *argv[] )
{
   int     width      = 0;
   int     height     = 0;
   int     drawheight = 0;
   int     xmoons     = 1;
   int     ymoons     = 1;
   int     stars      = 50;
   char    *filename  = NULL;
   int     black      = 0;
   image_t *image     = NULL;
   moon_t  *moon      = NULL;
   int     c;
   int     x;
   int     y;
   
   opterr = 0;
   
   if( argc == 1 )
   {
      usage( argv[0] );
   }
   
   while ((c = getopt (argc, argv, "w:h:x:y:s:f:bq")) != EOF)
   {
      switch (c)
      {
      case 'w':
         width    = strtol( optarg, NULL, 0 );
         break;
      case 'h':
         height   = strtol( optarg, NULL, 0 );
         break;
      case 'x':
         xmoons   = strtol( optarg, NULL, 0 );
         break;
      case 'y':
         ymoons   = strtol( optarg, NULL, 0 );
         break;
      case 's':
         stars    = strtol( optarg, NULL, 0 );
         break;
      case 'f':
         filename = optarg;
         break;
      case 'b':
         black    = 1;
         break;
      case 'q':
         disablewarn();
         break;
      default:
         fail("unknown option: '%c'\n", c);
      }
   }
   
   if( !width || !height || !filename )
   {
      fail( "you need to specify at least width, height and filename\n" );
   }
   
   moon = mooncreate();

   if( width < moon->width )
   {
      warn( "image not wide enough for anything, just creating stars\n" );
      xmoons = 0;
   }

   if( width < moon->width * xmoons )
   {
      warn( "decreasing number of moons in x axis from %d ", xmoons );
      while( width < moon->width * --xmoons );
      warn( "to %d moons.\n", xmoons );
   }

   drawheight = height;
   if( height < moon->height )
   {
      warn( "image not high enough for moon, using truncate mode\n" );
      drawheight = moon->height;
      ymoons = 1;
   }
   else
   {
      if( height < moon->height * ymoons )
      {
         warn( "decreasing number of moons in y axis from %d ", ymoons );
         /* this is not nice, but works: decreasing number of moons until they fit */
         while( width < (moon->width * --ymoons) );
         warn( "to %d moons.\n", ymoons );
      }
   }
   
   image = imagecreate( width, drawheight );
   if( xmoons * ymoons )
   {
      scarymonster( image, stars );
      for( y = (drawheight/ymoons)/2; y < height; y += (drawheight/ymoons) )
      {
         for( x = (width/xmoons)/2; x < width; x += (width/xmoons) )
         {
            mooncopy( image, moon, x, y, black );
         }
      }
   }
   else
   {
      scarymonster( image, stars * 2 );
   }
   pngwrite( image, filename, height );
   imagedestroy( image );
   moondestroy( moon );
   
   return 0;
}
