### Copyright (C) 2010 Peter Williams <peter_ono@users.sourceforge.net>

### This program is free software; you can redistribute it and/or modify
### it under the terms of the GNU General Public License as published by
### the Free Software Foundation; version 2 of the License only.

### This program is distributed in the hope that it will be useful,
### but WITHOUT ANY WARRANTY; without even the implied warranty of
### MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
### GNU General Public License for more details.

### You should have received a copy of the GNU General Public License
### along with this program; if not, write to the Free Software
### Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import gtk, pango, time, os

from gquilt_pkg import cmd_result, utils
from gquilt_pkg import dialogue

class Console(gtk.Frame):
    def __init__(self):
        gtk.Frame.__init__(self)
        self.view = gtk.TextView()
        self.text = self.view.get_buffer()
        self.dat_tag = self.text.create_tag("DAT", weight=pango.WEIGHT_BOLD)
        self.stderr_tag = self.text.create_tag("STDERR", foreground="red")
        self.stdout_tag = self.text.create_tag("STDOUT", foreground="black")
        self.cmd_tag = self.text.create_tag("CMD", foreground="black")
        self.eobuf = self.text.create_mark("eobuf", self.text.get_end_iter(), False)
        scr_win = gtk.ScrolledWindow()
        scr_win.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        scr_win.add(self.view)
        self.add(scr_win)
        self.view.set_cursor_visible(False)
        self.view.set_editable(False)
        self.show_all()
        self._finish_entry()
    def _append_to_console_tagged(self, msg, tag):
        self.text.insert_with_tags(self.text.get_end_iter(), msg, tag)
        self.view.scroll_to_mark(self.eobuf, 0.001)
    def _append_dat(self):
        self._append_to_console_tagged(time.strftime("%Y-%m-%d %H:%M:%S") + ": ", self.dat_tag)
    def _finish_entry(self):
        self._append_to_console_tagged("%", self.dat_tag)        
    def _log_entry(self, entry, tag):
        entry = entry.rstrip()
        if entry:
            self._append_dat()
            self._append_to_console_tagged(entry, tag)
            self._append_newline()
            self._finish_entry()
    def log_entry(self, entry):
        self._log_entry(entry, self.cmd_tag)
    def log_entry_bold(self, entry):
        self._log_entry(entry, self.dat_tag)
    def _append_with_newline(self, entry, tag):
        entry = entry.rstrip()
        if entry:
            self._append_to_console_tagged(entry, tag)
            self._append_newline()
    def _append_stdout(self, msg):
        self._append_with_newline(msg, self.stdout_tag)
    def _append_stderr(self, msg):
        self._append_with_newline(msg, self.stderr_tag)
    def append_stderr(self, msg):
        self._append_stderr(msg)
        self._finish_entry()
    def _append_newline(self):
        self._append_to_console_tagged(os.linesep, self.stdout_tag)
    def exec_cmd(self, cmd):
        self._append_dat()
        self._append_with_newline(cmd, self.cmd_tag)
        while gtk.events_pending():
            gtk.main_iteration()
        res, sout, err = utils.run_cmd(cmd)
        self._append_stdout(sout)
        self._append_stderr(err)
        self._finish_entry()
        return cmd_result.Result(res, sout, err)
    def exec_cmd_with_alert(self, cmd):
        res, sout, err = self.exec_cmd(cmd)
        if res != 0:
            if len(err) > 0:
                dialogue.inform_user(err)
            else:
                dialogue.inform_user(sout)

LOG = Console()

# run a command and log the result to the provided console
def exec_console_cmd(cmd, error=cmd_result.ERROR):
    if LOG is not None:
        res, sout, serr = LOG.exec_cmd(cmd)
    else:
        res, sout, serr = utils.run_cmd(cmd)
    if res != 0:
        return cmd_result.Result(error, sout, serr)
    else:
        return cmd_result.Result(cmd_result.OK, sout, serr)
