# -*- coding: utf-8 -*-
from __future__ import print_function

"""Check GW calculations.

Compare to data from:
    
    Predictive GW calculations using plane waves and pseudopotentials

    Jiří Klimeš, Merzuk Kaltak, Georg Kresse

    http://arxiv.org/abs/1404.3101
"""

import ase.db


# Table II:
lda = {
    'C': [-21.32, 5.54, -2.79, 8.38, -6.29, 4.70, None],
    'SiC': [-15.33, 6.28, -1.06, 5.38, -3.20, 1.30, None],
    'Si': [-11.96, 2.52, -1.20, 1.42, -2.85, 0.60, None],
    'Ge': [-12.81, -0.15, -1.41, 0.06, -3.08, 0.66, None],
    'BN': [-20.07, 8.68, -1.94, 10.19, -4.91, 4.34, None],
    'AlP': [-11.50, 3.10, -0.77, 2.66, -2.12, 1.44, None],
    'AlAs': [-11.90, 1.86, -0.82, 2.02, -2.17, 1.35, None],
    'AlSb': [-10.77, 1.46, -0.91, 1.26, -2.19, 1.15, None],
    'GaN': [-15.64, 1.63, -0.95, 4.42, -2.66, 3.24, -13.50],
    'GaN-wz': [None, 1.94, None, None, None, None, -13.30],
    'GaP': [-12.51, 1.63, -1.13, 1.51, -2.71, 1.49, -14.68],
    'GaAs': [-12.76, 0.32, -1.15, 0.86, -2.69, 1.34, -14.91],
    'GaSb': [-11.59, -0.06, -1.17, 0.32, -2.61, 0.83, -15.14],
    'InP': [-11.52, 0.48, -0.98, 1.31, -2.35, 1.60, -14.13],
    'InAs': [-11.88, -0.43, -0.99, 0.79, -2.33, 1.43, -14.30],
    'InSb': [-10.77, -0.38, -1.01, 0.42, -2.27, 1.25, -14.51],
    'MgO': [-17.00, 4.68, -0.67, 7.75, -1.37, 8.89, None],
    'ZnO': [-17.68, 0.62, -0.80, 5.32, -2.21, 5.13, -5.30],
    'ZnO-wz': [None, 0.75, None, None, None, None, -5.24],
    'ZnS': [-13.08, 1.87, -0.87, 3.10, -2.23, 3.19, -6.31],
    'ZnSe': [-13.28, 1.05, -0.87, 2.36, -2.20, 2.81, -6.55],
    'ZnTe': [-11.82, 1.06, -0.90, 1.65, -2.18, 2.17, -6.94],
    'CdO': [-15.73, 0.92, 1.42, 5.68, -0.98, 5.10, -5.14],
    'CdS': [-12.40, 0.90, -0.78, 2.79, -1.95, 3.30, -7.60],
    'CdSe': [-12.65, 0.36, -0.77, 2.19, -1.89, 2.94, -7.80],
    'CdTe': [-11.23, 0.55, -0.79, 1.66, -1.89, 2.45, -8.18]}

# Table III:
gw = {
    'C': [-1.07, -22.06, 7.43, -2.92, 10.37, -6.55, 6.21, None],
    'SiC': [-0.92, -15.61, 7.30, -1.09, 6.58, -3.27, 2.43, None],
    'Si': [-0.60, -11.82, 3.21, -1.21, 2.06, -2.86, 1.22, None],
    'Ge': [-0.54, -12.69, 0.43, -1.42, 0.56, -3.09, 1.10, None],
    'BN': [-1.35, -21.09, 11.29, -2.08, 12.25, -5.19, 6.37, None],
    'AlP': [-0.83, -11.36, 4.11, -0.78, 3.70, -2.13, 2.42, None],
    'AlAs': [-0.82, -11.81, 2.82, -0.83, 2.96, -2.17, 2.23, None],
    'AlSb': [-0.65, -10.65, 2.27, -0.91, 1.99, -2.20, 1.82, None],
    'GaN': [-1.04, -15.86, 2.88, -0.97, 5.95, -2.68, 4.59, -15.87],
    'GaN-wz': [-1.08, None, 3.23, None, None, None, None, -15.66],
    'GaP': [-0.70, -12.34, 2.50, -1.14, 2.36, -2.71, 2.25, -16.86],
    'GaAs': [-0.63, -12.59, 1.08, -1.15, 1.57, -2.68, 1.96, -17.10],
    'GaSb': [-0.54, -11.45, 0.54, -1.17, 0.84, -2.61, 1.29, -17.32],
    'InP': [-0.61, -11.32, 1.13, -0.99, 2.00, -2.35, 2.21, -15.73],
    'InAs': [-0.60, -11.69, 0.13, -1.00, 1.43, -2.33, 2.02, -15.84],
    'InSb': [-0.54, -10.61, 0.13, -1.03, 0.93, -2.28, 1.72, -16.06],
    'MgO': [-2.01, -17.47, 7.55, -0.71, 10.86, -1.44, 11.91, None],
    'ZnO': [-1.53, -17.75, 2.46, -0.79, 7.43, -2.15, 7.00, -6.22],
    'ZnO-wz': [-1.78, None, 2.83, None, None, None, None, -6.09],
    'ZnS': [-1.15, -12.62, 3.36, -0.86, 4.70, -2.18, 4.62, -7.55],
    'ZnSe': [-1.09, -12.97, 2.38, -0.85, 3.75, -2.12, 4.06, -7.82],
    'ZnTe': [-0.84, -11.59, 2.17, -0.88, 2.71, -2.13, 3.15, -8.31],
    'CdO': [-0.81, -16.16, 1.57, 1.14, 6.83, -0.98, 6.46, -6.78],
    'CdS': [-0.96, -11.98, 2.05, -0.77, 4.09, -1.91, 4.48, -8.67],
    'CdSe': [-0.90, -12.37, 1.33, -0.75, 3.27, -1.83, 3.93, -8.84],
    'CdTe': [-0.75, -10.99, 1.46, -0.78, 2.58, -1.85, 3.28, -9.34]}

c = ase.db.connect('gw.db')
for d in c.select(calculator='gpaw'):
    if d.name not in lda:
        continue
    e1 = d.data.eps[0] - d.data.eps[0, 0, 0]
    r = list(e1.ravel()[1:] - lda[d.name][1:6])
    e2 = d.data.qp[0] - d.data.qp[0, 0, 0]
    r.extend(e2.ravel()[1:] - gw[d.name][2:7])
    print('{0:7}'.format(d.name), ' '.join('{0:5.2f}'.format(x) for x in r))
