/*
 * Copyright © 2005 Ondra Kamenik
 * Copyright © 2019 Dynare Team
 *
 * This file is part of Dynare.
 *
 * Dynare is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Dynare is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Dynare.  If not, see <https://www.gnu.org/licenses/>.
 */

// The file contains one dimensional quadrature points and weights for a few
// quadratures. The format of data is clear. There is a class
// OneDPrecalcQuadrature which implements an interface OneDQuadrature using the
// data of this format.

// Gauss-Hermite quadrature; prefix gh

// Number of levels
static const int gh_num_levels = 26;

// Number of points in each level
static const int gh_num_points[] =
  {
   1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
   30, 32, 40, 50, 60, 64
  };

// Weights, starting with the first level
static const double gh_weights[] =
  {
   // weights 1 = √π
   1.77245385090551588191942755656782537698745727539062,
   // weights 2
   0.886226925452758013649083741671e+00,
   0.886226925452758013649083741671e+00,
   // weights 3
   0.295408975150919337883027913890e+00,
   0.118163590060367735153211165556e+01,
   0.295408975150919337883027913890e+00,
   // weights 4
   0.813128354472451771430345571899e-01,
   0.804914090005512836506049184481e+00,
   0.804914090005512836506049184481e+00,
   0.813128354472451771430345571899e-01,
   // weights 5
   0.199532420590459132077434585942e-01,
   0.393619323152241159828495620852e+00,
   0.945308720482941881225689324449e+00,
   0.393619323152241159828495620852e+00,
   0.199532420590459132077434585942e-01,
   // weights 6
   0.453000990550884564085747256463e-02,
   0.157067320322856643916311563508e+00,
   0.724629595224392524091914705598e+00,
   0.724629595224392524091914705598e+00,
   0.157067320322856643916311563508e+00,
   0.453000990550884564085747256463e-02,
   // weights 7
   0.971781245099519154149424255939e-03,
   0.545155828191270305921785688417e-01,
   0.425607252610127800520317466666e+00,
   0.810264617556807326764876563813e+00,
   0.425607252610127800520317466666e+00,
   0.545155828191270305921785688417e-01,
   0.971781245099519154149424255939e-03,
   // weights 8
   0.199604072211367619206090452544e-03,
   0.170779830074134754562030564364e-01,
   0.207802325814891879543258620286e+00,
   0.661147012558241291030415974496e+00,
   0.661147012558241291030415974496e+00,
   0.207802325814891879543258620286e+00,
   0.170779830074134754562030564364e-01,
   0.199604072211367619206090452544e-03,
   // weights 9
   0.396069772632643819045862946425e-04,
   0.494362427553694721722456597763e-02,
   0.884745273943765732879751147476e-01,
   0.432651559002555750199812112956e+00,
   0.720235215606050957124334723389e+00,
   0.432651559002555750199812112956e+00,
   0.884745273943765732879751147476e-01,
   0.494362427553694721722456597763e-02,
   0.396069772632643819045862946425e-04,
   // weights 10
   0.764043285523262062915936785960e-05,
   0.134364574678123269220156558585e-02,
   0.338743944554810631361647312776e-01,
   0.240138611082314686416523295006e+00,
   0.610862633735325798783564990433e+00,
   0.610862633735325798783564990433e+00,
   0.240138611082314686416523295006e+00,
   0.338743944554810631361647312776e-01,
   0.134364574678123269220156558585e-02,
   0.764043285523262062915936785960e-05,
   // weights 11
   0.143956039371425822033088366032e-05,
   0.346819466323345510643413772940e-03,
   0.119113954449115324503874202916e-01,
   0.117227875167708503381788649308e+00,
   0.429359752356125028446073598601e+00,
   0.654759286914591779203940657627e+00,
   0.429359752356125028446073598601e+00,
   0.117227875167708503381788649308e+00,
   0.119113954449115324503874202916e-01,
   0.346819466323345510643413772940e-03,
   0.143956039371425822033088366032e-05,
   // weights 12
   0.265855168435630160602311400877e-06,
   0.857368704358785865456906323153e-04,
   0.390539058462906185999438432620e-02,
   0.516079856158839299918734423606e-01,
   0.260492310264161129233396139765e+00,
   0.570135236262479578347113482275e+00,
   0.570135236262479578347113482275e+00,
   0.260492310264161129233396139765e+00,
   0.516079856158839299918734423606e-01,
   0.390539058462906185999438432620e-02,
   0.857368704358785865456906323153e-04,
   0.265855168435630160602311400877e-06,
   // weights 13
   0.482573185007313108834997332342e-07,
   0.204303604027070731248669432937e-04,
   0.120745999271938594730924899224e-02,
   0.208627752961699392166033805050e-01,
   0.140323320687023437762792268873e+00,
   0.421616296898543221746893558568e+00,
   0.604393187921161642342099068579e+00,
   0.421616296898543221746893558568e+00,
   0.140323320687023437762792268873e+00,
   0.208627752961699392166033805050e-01,
   0.120745999271938594730924899224e-02,
   0.204303604027070731248669432937e-04,
   0.482573185007313108834997332342e-07,
   // weights 14
   0.862859116812515794532041783429e-08,
   0.471648435501891674887688950105e-05,
   0.355092613551923610483661076691e-03,
   0.785005472645794431048644334608e-02,
   0.685055342234652055387163312367e-01,
   0.273105609064246603352569187026e+00,
   0.536405909712090149794921296776e+00,
   0.536405909712090149794921296776e+00,
   0.273105609064246603352569187026e+00,
   0.685055342234652055387163312367e-01,
   0.785005472645794431048644334608e-02,
   0.355092613551923610483661076691e-03,
   0.471648435501891674887688950105e-05,
   0.862859116812515794532041783429e-08,
   // weights 15
   0.152247580425351702016062666965e-08,
   0.105911554771106663577520791055e-05,
   0.100004441232499868127296736177e-03,
   0.277806884291277589607887049229e-02,
   0.307800338725460822286814158758e-01,
   0.158488915795935746883839384960e+00,
   0.412028687498898627025891079568e+00,
   0.564100308726417532852625797340e+00,
   0.412028687498898627025891079568e+00,
   0.158488915795935746883839384960e+00,
   0.307800338725460822286814158758e-01,
   0.277806884291277589607887049229e-02,
   0.100004441232499868127296736177e-03,
   0.105911554771106663577520791055e-05,
   0.152247580425351702016062666965e-08,
   // weights 16
   0.265480747401118224470926366050e-09,
   0.232098084486521065338749423185e-06,
   0.271186009253788151201891432244e-04,
   0.932284008624180529914277305537e-03,
   0.128803115355099736834642999312e-01,
   0.838100413989858294154207349001e-01,
   0.280647458528533675369463335380e+00,
   0.507929479016613741913517341791e+00,
   0.507929479016613741913517341791e+00,
   0.280647458528533675369463335380e+00,
   0.838100413989858294154207349001e-01,
   0.128803115355099736834642999312e-01,
   0.932284008624180529914277305537e-03,
   0.271186009253788151201891432244e-04,
   0.232098084486521065338749423185e-06,
   0.265480747401118224470926366050e-09,
   // weights 17
   0.458057893079863330580889281222e-10,
   0.497707898163079405227863353715e-07,
   0.711228914002130958353327376218e-05,
   0.298643286697753041151336643059e-03,
   0.506734995762753791170069495879e-02,
   0.409200341495762798094994877854e-01,
   0.172648297670097079217645196219e+00,
   0.401826469470411956577635085257e+00,
   0.530917937624863560331883103379e+00,
   0.401826469470411956577635085257e+00,
   0.172648297670097079217645196219e+00,
   0.409200341495762798094994877854e-01,
   0.506734995762753791170069495879e-02,
   0.298643286697753041151336643059e-03,
   0.711228914002130958353327376218e-05,
   0.497707898163079405227863353715e-07,
   0.458057893079863330580889281222e-10,
   // weights 18
   0.782819977211589102925147471012e-11,
   0.104672057957920824443559608435e-07,
   0.181065448109343040959702385911e-05,
   0.918112686792940352914675407371e-04,
   0.188852263026841789438175325426e-02,
   0.186400423875446519219315221973e-01,
   0.973017476413154293308537234155e-01,
   0.284807285669979578595606820713e+00,
   0.483495694725455552876410522141e+00,
   0.483495694725455552876410522141e+00,
   0.284807285669979578595606820713e+00,
   0.973017476413154293308537234155e-01,
   0.186400423875446519219315221973e-01,
   0.188852263026841789438175325426e-02,
   0.918112686792940352914675407371e-04,
   0.181065448109343040959702385911e-05,
   0.104672057957920824443559608435e-07,
   0.782819977211589102925147471012e-11,
   // weights 19
   0.132629709449851575185289154385e-11,
   0.216305100986355475019693077221e-08,
   0.448824314722312295179447915594e-06,
   0.272091977631616257711941025214e-04,
   0.670877521407181106194696282100e-03,
   0.798886677772299020922211491861e-02,
   0.508103869090520673569908110358e-01,
   0.183632701306997074156148485766e+00,
   0.391608988613030244504042313621e+00,
   0.502974888276186530840731361096e+00,
   0.391608988613030244504042313621e+00,
   0.183632701306997074156148485766e+00,
   0.508103869090520673569908110358e-01,
   0.798886677772299020922211491861e-02,
   0.670877521407181106194696282100e-03,
   0.272091977631616257711941025214e-04,
   0.448824314722312295179447915594e-06,
   0.216305100986355475019693077221e-08,
   0.132629709449851575185289154385e-11,
   // weights 20
   0.222939364553415129252250061603e-12,
   0.439934099227318055362885145547e-09,
   0.108606937076928169399952456345e-06,
   0.780255647853206369414599199965e-05,
   0.228338636016353967257145917963e-03,
   0.324377334223786183218324713235e-02,
   0.248105208874636108821649525589e-01,
   0.109017206020023320013755033535e+00,
   0.286675505362834129719659706228e+00,
   0.462243669600610089650328639861e+00,
   0.462243669600610089650328639861e+00,
   0.286675505362834129719659706228e+00,
   0.109017206020023320013755033535e+00,
   0.248105208874636108821649525589e-01,
   0.324377334223786183218324713235e-02,
   0.228338636016353967257145917963e-03,
   0.780255647853206369414599199965e-05,
   0.108606937076928169399952456345e-06,
   0.439934099227318055362885145547e-09,
   0.222939364553415129252250061603e-12,
   // weights 30
   0.290825470013122622941102747365e-20,
   0.281033360275090370876277491534e-16,
   0.287860708054870606219239791142e-13,
   0.810618629746304420399344796173e-11,
   0.917858042437852820850075742492e-09,
   0.510852245077594627738963204403e-07,
   0.157909488732471028834638794022e-05,
   0.293872522892298764150118423412e-04,
   0.348310124318685523420995323183e-03,
   0.273792247306765846298942568953e-02,
   0.147038297048266835152773557787e-01,
   0.551441768702342511680754948183e-01,
   0.146735847540890099751693643152e+00,
   0.280130930839212667413493211293e+00,
   0.386394889541813862555601849165e+00,
   0.386394889541813862555601849165e+00,
   0.280130930839212667413493211293e+00,
   0.146735847540890099751693643152e+00,
   0.551441768702342511680754948183e-01,
   0.147038297048266835152773557787e-01,
   0.273792247306765846298942568953e-02,
   0.348310124318685523420995323183e-03,
   0.293872522892298764150118423412e-04,
   0.157909488732471028834638794022e-05,
   0.510852245077594627738963204403e-07,
   0.917858042437852820850075742492e-09,
   0.810618629746304420399344796173e-11,
   0.287860708054870606219239791142e-13,
   0.281033360275090370876277491534e-16,
   0.290825470013122622941102747365e-20,
   // weights 32
   0.731067642736e-22,
   0.923173653649e-18,
   0.119734401709e-14,
   0.421501021125e-12,
   0.593329146300e-10,
   0.409883216476e-08,
   0.157416779254e-06,
   0.365058512955e-05,
   0.541658406172e-04,
   0.536268365526e-03,
   0.365489032664e-02,
   0.175534288315e-01,
   0.604581309557e-01,
   0.151269734076e+00,
   0.277458142302e+00,
   0.375238352592e+00,
   0.375238352592e+00,
   0.277458142302e+00,
   0.151269734076e+00,
   0.604581309557e-01,
   0.175534288315e-01,
   0.365489032664e-02,
   0.536268365526e-03,
   0.541658406172e-04,
   0.365058512955e-05,
   0.157416779254e-06,
   0.409883216476e-08,
   0.593329146300e-10,
   0.421501021125e-12,
   0.119734401709e-14,
   0.923173653649e-18,
   0.731067642736e-22,
   // weights 40
   0.259104371384e-28,
   0.854405696375e-24,
   0.256759336540e-20,
   0.198918101211e-17,
   0.600835878947e-15,
   0.880570764518e-13,
   0.715652805267e-11,
   0.352562079135e-09,
   0.112123608322e-07,
   0.241114416359e-06,
   0.363157615067e-05,
   0.393693398108e-04,
   0.313853594540e-03,
   0.187149682959e-02,
   0.846088800823e-02,
   0.293125655361e-01,
   0.784746058652e-01,
   0.163378732713e+00,
   0.265728251876e+00,
   0.338643277425e+00,
   0.338643277425e+00,
   0.265728251876e+00,
   0.163378732713e+00,
   0.784746058652e-01,
   0.293125655361e-01,
   0.846088800823e-02,
   0.187149682959e-02,
   0.313853594540e-03,
   0.393693398108e-04,
   0.363157615067e-05,
   0.241114416359e-06,
   0.112123608322e-07,
   0.352562079135e-09,
   0.715652805267e-11,
   0.880570764518e-13,
   0.600835878947e-15,
   0.198918101211e-17,
   0.256759336540e-20,
   0.854405696375e-24,
   0.259104371384e-28,
   // weights 50
   0.183379404857e-36,
   0.167380166790e-31,
   0.121524412340e-27,
   0.213765830835e-24,
   0.141709359957e-21,
   0.447098436530e-19,
   0.774238295702e-17,
   0.809426189344e-15,
   0.546594403180e-13,
   0.250665552389e-11,
   0.811187736448e-10,
   0.190904054379e-08,
   0.334679340401e-07,
   0.445702996680e-06,
   0.458168270794e-05,
   0.368401905377e-04,
   0.234269892109e-03,
   0.118901178175e-02,
   0.485326382616e-02,
   0.160319410684e-01,
   0.430791591566e-01,
   0.945489354768e-01,
   0.170032455676e+00,
   0.251130856331e+00,
   0.305085129203e+00,
   0.305085129203e+00,
   0.251130856331e+00,
   0.170032455676e+00,
   0.945489354768e-01,
   0.430791591566e-01,
   0.160319410684e-01,
   0.485326382616e-02,
   0.118901178175e-02,
   0.234269892109e-03,
   0.368401905377e-04,
   0.458168270794e-05,
   0.445702996680e-06,
   0.334679340401e-07,
   0.190904054379e-08,
   0.811187736448e-10,
   0.250665552389e-11,
   0.546594403180e-13,
   0.809426189344e-15,
   0.774238295702e-17,
   0.447098436530e-19,
   0.141709359957e-21,
   0.213765830835e-24,
   0.121524412340e-27,
   0.167380166790e-31,
   0.183379404857e-36,
   // weights 60
   0.110958724796e-44,
   0.243974758810e-39,
   0.377162672698e-35,
   0.133255961176e-31,
   0.171557314767e-28,
   0.102940599693e-25,
   0.334575695574e-23,
   0.651256725748e-21,
   0.815364047300e-19,
   0.692324790956e-17,
   0.415244410968e-15,
   0.181662457614e-13,
   0.594843051597e-12,
   0.148895734905e-10,
   0.289935901280e-09,
   0.445682277521e-08,
   0.547555461926e-07,
   0.543351613419e-06,
   0.439428693625e-05,
   0.291874190415e-04,
   0.160277334681e-03,
   0.731773556963e-03,
   0.279132482894e-02,
   0.893217836028e-02,
   0.240612727660e-01,
   0.547189709320e-01,
   0.105298763697e+00,
   0.171776156918e+00,
   0.237868904958e+00,
   0.279853117522e+00,
   0.279853117522e+00,
   0.237868904958e+00,
   0.171776156918e+00,
   0.105298763697e+00,
   0.547189709320e-01,
   0.240612727660e-01,
   0.893217836028e-02,
   0.279132482894e-02,
   0.731773556963e-03,
   0.160277334681e-03,
   0.291874190415e-04,
   0.439428693625e-05,
   0.543351613419e-06,
   0.547555461926e-07,
   0.445682277521e-08,
   0.289935901280e-09,
   0.148895734905e-10,
   0.594843051597e-12,
   0.181662457614e-13,
   0.415244410968e-15,
   0.692324790956e-17,
   0.815364047300e-19,
   0.651256725748e-21,
   0.334575695574e-23,
   0.102940599693e-25,
   0.171557314767e-28,
   0.133255961176e-31,
   0.377162672698e-35,
   0.243974758810e-39,
   0.110958724796e-44,
   // weights 64
   0.553570653584e-48,
   0.167974799010e-42,
   0.342113801099e-38,
   0.155739062462e-34,
   0.254966089910e-31,
   0.192910359546e-28,
   0.786179778889e-26,
   0.191170688329e-23,
   0.298286278427e-21,
   0.315225456649e-19,
   0.235188471067e-17,
   0.128009339117e-15,
   0.521862372645e-14,
   0.162834073070e-12,
   0.395917776693e-11,
   0.761521725012e-10,
   0.117361674232e-08,
   0.146512531647e-07,
   0.149553293672e-06,
   0.125834025103e-05,
   0.878849923082e-05,
   0.512592913577e-04,
   0.250983698512e-03,
   0.103632909950e-02,
   0.362258697852e-02,
   0.107560405098e-01,
   0.272031289536e-01,
   0.587399819634e-01,
   0.108498349306e+00,
   0.171685842349e+00,
   0.232994786062e+00,
   0.271377424940e+00,
   0.271377424940e+00,
   0.232994786062e+00,
   0.171685842349e+00,
   0.108498349306e+00,
   0.587399819634e-01,
   0.272031289536e-01,
   0.107560405098e-01,
   0.362258697852e-02,
   0.103632909950e-02,
   0.250983698512e-03,
   0.512592913577e-04,
   0.878849923082e-05,
   0.125834025103e-05,
   0.149553293672e-06,
   0.146512531647e-07,
   0.117361674232e-08,
   0.761521725012e-10,
   0.395917776693e-11,
   0.162834073070e-12,
   0.521862372645e-14,
   0.128009339117e-15,
   0.235188471067e-17,
   0.315225456649e-19,
   0.298286278427e-21,
   0.191170688329e-23,
   0.786179778889e-26,
   0.192910359546e-28,
   0.254966089910e-31,
   0.155739062462e-34,
   0.342113801099e-38,
   0.167974799010e-42,
   0.553570653584e-48
  };

// Points, starting with the first level
static const double gh_points[] =
  {
   // points 1
   0.0,
   // points 2
   -0.707106781186547524400844362105e+00,
   0.707106781186547524400844362105e+00,
   // points 3
   -0.122474487139158904909864203735e+01,
   0.0e+00,
   0.122474487139158904909864203735e+01,
   // points 4
   -0.165068012388578455588334111112e+01,
   -0.524647623275290317884060253835e+00,
   0.524647623275290317884060253835e+00,
   0.165068012388578455588334111112e+01,
   // points 5
   -0.202018287045608563292872408814e+01,
   -0.958572464613818507112770593893e+00,
   0.0e+00,
   0.958572464613818507112770593893e+00,
   0.202018287045608563292872408814e+01,
   // points 6
   -0.235060497367449222283392198706e+01,
   -0.133584907401369694971489528297e+01,
   -0.436077411927616508679215948251e+00,
   0.436077411927616508679215948251e+00,
   0.133584907401369694971489528297e+01,
   0.235060497367449222283392198706e+01,
   // points 7
   -0.265196135683523349244708200652e+01,
   -0.167355162876747144503180139830e+01,
   -0.816287882858964663038710959027e+00,
   0.0e+00,
   0.816287882858964663038710959027e+00,
   0.167355162876747144503180139830e+01,
   0.265196135683523349244708200652e+01,
   // points 8
   -0.293063742025724401922350270524e+01,
   -0.198165675669584292585463063977e+01,
   -0.115719371244678019472076577906e+01,
   -0.381186990207322116854718885584e+00,
   0.381186990207322116854718885584e+00,
   0.115719371244678019472076577906e+01,
   0.198165675669584292585463063977e+01,
   0.293063742025724401922350270524e+01,
   // points 9
   -0.319099320178152760723004779538e+01,
   -0.226658058453184311180209693284e+01,
   -0.146855328921666793166701573925e+01,
   -0.723551018752837573322639864579e+00,
   0.0e+00,
   0.723551018752837573322639864579e+00,
   0.146855328921666793166701573925e+01,
   0.226658058453184311180209693284e+01,
   0.319099320178152760723004779538e+01,
   // points 10
   -0.343615911883773760332672549432e+01,
   -0.253273167423278979640896079775e+01,
   -0.175668364929988177345140122011e+01,
   -0.103661082978951365417749191676e+01,
   -0.342901327223704608789165025557e+00,
   0.342901327223704608789165025557e+00,
   0.103661082978951365417749191676e+01,
   0.175668364929988177345140122011e+01,
   0.253273167423278979640896079775e+01,
   0.343615911883773760332672549432e+01,
   // points 11
   -0.366847084655958251845837146485e+01,
   -0.278329009978165177083671870152e+01,
   -0.202594801582575533516591283121e+01,
   -0.132655708449493285594973473558e+01,
   -0.656809566882099765024611575383e+00,
   0.0e+00,
   0.656809566882099765024611575383e+00,
   0.132655708449493285594973473558e+01,
   0.202594801582575533516591283121e+01,
   0.278329009978165177083671870152e+01,
   0.366847084655958251845837146485e+01,
   // points 12
   -0.388972489786978191927164274724e+01,
   -0.302063702512088977171067937518e+01,
   -0.227950708050105990018772856942e+01,
   -0.159768263515260479670966277090e+01,
   -0.947788391240163743704578131060e+00,
   -0.314240376254359111276611634095e+00,
   0.314240376254359111276611634095e+00,
   0.947788391240163743704578131060e+00,
   0.159768263515260479670966277090e+01,
   0.227950708050105990018772856942e+01,
   0.302063702512088977171067937518e+01,
   0.388972489786978191927164274724e+01,
   // points 13
   -0.410133759617863964117891508007e+01,
   -0.324660897837240998812205115236e+01,
   -0.251973568567823788343040913628e+01,
   -0.185310765160151214200350644316e+01,
   -0.122005503659074842622205526637e+01,
   -0.605763879171060113080537108602e+00,
   0.0e+00,
   0.605763879171060113080537108602e+00,
   0.122005503659074842622205526637e+01,
   0.185310765160151214200350644316e+01,
   0.251973568567823788343040913628e+01,
   0.324660897837240998812205115236e+01,
   0.410133759617863964117891508007e+01,
   // points 14
   -0.430444857047363181262129810037e+01,
   -0.346265693360227055020891736115e+01,
   -0.274847072498540256862499852415e+01,
   -0.209518325850771681573497272630e+01,
   -0.147668273114114087058350654421e+01,
   -0.878713787329399416114679311861e+00,
   -0.291745510672562078446113075799e+00,
   0.291745510672562078446113075799e+00,
   0.878713787329399416114679311861e+00,
   0.147668273114114087058350654421e+01,
   0.209518325850771681573497272630e+01,
   0.274847072498540256862499852415e+01,
   0.346265693360227055020891736115e+01,
   0.430444857047363181262129810037e+01,
   // points 15
   -0.449999070730939155366438053053e+01,
   -0.366995037340445253472922383312e+01,
   -0.296716692790560324848896036355e+01,
   -0.232573248617385774545404479449e+01,
   -0.171999257518648893241583152515e+01,
   -0.113611558521092066631913490556e+01,
   -0.565069583255575748526020337198e+00,
   0.0e+00,
   0.565069583255575748526020337198e+00,
   0.113611558521092066631913490556e+01,
   0.171999257518648893241583152515e+01,
   0.232573248617385774545404479449e+01,
   0.296716692790560324848896036355e+01,
   0.366995037340445253472922383312e+01,
   0.449999070730939155366438053053e+01,
   // points 16
   -0.468873893930581836468849864875e+01,
   -0.386944790486012269871942409801e+01,
   -0.317699916197995602681399455926e+01,
   -0.254620215784748136215932870545e+01,
   -0.195178799091625397743465541496e+01,
   -0.138025853919888079637208966969e+01,
   -0.822951449144655892582454496734e+00,
   -0.273481046138152452158280401965e+00,
   0.273481046138152452158280401965e+00,
   0.822951449144655892582454496734e+00,
   0.138025853919888079637208966969e+01,
   0.195178799091625397743465541496e+01,
   0.254620215784748136215932870545e+01,
   0.317699916197995602681399455926e+01,
   0.386944790486012269871942409801e+01,
   0.468873893930581836468849864875e+01,
   // points 17
   -0.487134519367440308834927655662e+01,
   -0.406194667587547430689245559698e+01,
   -0.337893209114149408338327069289e+01,
   -0.275776291570388873092640349574e+01,
   -0.217350282666662081927537907149e+01,
   -0.161292431422123133311288254454e+01,
   -0.106764872574345055363045773799e+01,
   -0.531633001342654731349086553718e+00,
   0.0e+00,
   0.531633001342654731349086553718e+00,
   0.106764872574345055363045773799e+01,
   0.161292431422123133311288254454e+01,
   0.217350282666662081927537907149e+01,
   0.275776291570388873092640349574e+01,
   0.337893209114149408338327069289e+01,
   0.406194667587547430689245559698e+01,
   0.487134519367440308834927655662e+01,
   // points 18
   -0.504836400887446676837203757885e+01,
   -0.424811787356812646302342016090e+01,
   -0.357376906848626607950067599377e+01,
   -0.296137750553160684477863254906e+01,
   -0.238629908916668600026459301424e+01,
   -0.183553160426162889225383944409e+01,
   -0.130092085838961736566626555439e+01,
   -0.776682919267411661316659462284e+00,
   -0.258267750519096759258116098711e+00,
   0.258267750519096759258116098711e+00,
   0.776682919267411661316659462284e+00,
   0.130092085838961736566626555439e+01,
   0.183553160426162889225383944409e+01,
   0.238629908916668600026459301424e+01,
   0.296137750553160684477863254906e+01,
   0.357376906848626607950067599377e+01,
   0.424811787356812646302342016090e+01,
   0.504836400887446676837203757885e+01,
   // points 19
   -0.522027169053748216460967142500e+01,
   -0.442853280660377943723498532226e+01,
   -0.376218735196402009751489394104e+01,
   -0.315784881834760228184318034120e+01,
   -0.259113378979454256492128084112e+01,
   -0.204923170985061937575050838669e+01,
   -0.152417061939353303183354859367e+01,
   -0.101036838713431135136859873726e+01,
   -0.503520163423888209373811765050e+00,
   0.0e+00,
   0.503520163423888209373811765050e+00,
   0.101036838713431135136859873726e+01,
   0.152417061939353303183354859367e+01,
   0.204923170985061937575050838669e+01,
   0.259113378979454256492128084112e+01,
   0.315784881834760228184318034120e+01,
   0.376218735196402009751489394104e+01,
   0.442853280660377943723498532226e+01,
   0.522027169053748216460967142500e+01,
   // points 20
   -0.538748089001123286201690041068e+01,
   -0.460368244955074427307767524898e+01,
   -0.394476404011562521037562880052e+01,
   -0.334785456738321632691492452300e+01,
   -0.278880605842813048052503375640e+01,
   -0.225497400208927552308233334473e+01,
   -0.173853771211658620678086566214e+01,
   -0.123407621539532300788581834696e+01,
   -0.737473728545394358705605144252e+00,
   -0.245340708300901249903836530634e+00,
   0.245340708300901249903836530634e+00,
   0.737473728545394358705605144252e+00,
   0.123407621539532300788581834696e+01,
   0.173853771211658620678086566214e+01,
   0.225497400208927552308233334473e+01,
   0.278880605842813048052503375640e+01,
   0.334785456738321632691492452300e+01,
   0.394476404011562521037562880052e+01,
   0.460368244955074427307767524898e+01,
   0.538748089001123286201690041068e+01,
   // points 30
   -6.86334529352989158106110835756e+00,
   -6.13827922012393462039499237854e+00,
   -5.53314715156749572511833355558e+00,
   -4.98891896858994394448649710633e+00,
   -4.48305535709251834188703761971e+00,
   -4.00390860386122881522787601332e+00,
   -3.54444387315534988692540090217e+00,
   -3.09997052958644174868873332237e+00,
   -2.66713212453561720057110646422e+00,
   -2.24339146776150407247297999483e+00,
   -1.82674114360368803883588048351e+00,
   -1.41552780019818851194072510555e+00,
   -1.00833827104672346180498960870e+00,
   -0.603921058625552307778155678757e+00,
   -0.201128576548871485545763013244e+00,
   0.201128576548871485545763013244e+00,
   0.603921058625552307778155678757e+00,
   1.00833827104672346180498960870e+00,
   1.41552780019818851194072510555e+00,
   1.82674114360368803883588048351e+00,
   2.24339146776150407247297999483e+00,
   2.66713212453561720057110646422e+00,
   3.09997052958644174868873332237e+00,
   3.54444387315534988692540090217e+00,
   4.00390860386122881522787601332e+00,
   4.48305535709251834188703761971e+00,
   4.98891896858994394448649710633e+00,
   5.53314715156749572511833355558e+00,
   6.13827922012393462039499237854e+00,
   6.86334529352989158106110835756e+00,
   // points 32
   -7.12581390983e+00,
   -6.40949814927e+00,
   -5.81222594952e+00,
   -5.27555098652e+00,
   -4.77716450350e+00,
   -4.30554795335e+00,
   -3.85375548547e+00,
   -3.41716749282e+00,
   -2.99249082500e+00,
   -2.57724953773e+00,
   -2.16949918361e+00,
   -1.76765410946e+00,
   -1.37037641095e+00,
   -0.976500463590e+00,
   -0.584978765436e+00,
   -0.194840741569e+00,
   0.194840741569e+00,
   0.584978765436e+00,
   0.976500463590e+00,
   1.37037641095e+00,
   1.76765410946e+00,
   2.16949918361e+00,
   2.57724953773e+00,
   2.99249082500e+00,
   3.41716749282e+00,
   3.85375548547e+00,
   4.30554795335e+00,
   4.77716450350e+00,
   5.27555098652e+00,
   5.81222594952e+00,
   6.40949814927e+00,
   7.12581390983e+00,
   // points 40
   -8.09876113925e+00,
   -7.41158253149e+00,
   -6.84023730525e+00,
   -6.32825535122e+00,
   -5.85409505603e+00,
   -5.40665424797e+00,
   -4.97926097855e+00,
   -4.56750207284e+00,
   -4.16825706683e+00,
   -3.77920675344e+00,
   -3.39855826586e+00,
   -3.02487988390e+00,
   -2.65699599844e+00,
   -2.29391714188e+00,
   -1.93479147228e+00,
   -1.57886989493e+00,
   -1.22548010905e+00,
   -0.874006612357e+00,
   -0.523874713832e+00,
   -0.174537214598e+00,
   0.174537214598e+00,
   0.523874713832e+00,
   0.874006612357e+00,
   1.22548010905e+00,
   1.57886989493e+00,
   1.93479147228e+00,
   2.29391714188e+00,
   2.65699599844e+00,
   3.02487988390e+00,
   3.39855826586e+00,
   3.77920675344e+00,
   4.16825706683e+00,
   4.56750207284e+00,
   4.97926097855e+00,
   5.40665424797e+00,
   5.85409505603e+00,
   6.32825535122e+00,
   6.84023730525e+00,
   7.41158253149e+00,
   8.09876113925e+00,
   // points 50
   -9.18240695813e+00,
   -8.52277103092e+00,
   -7.97562236821e+00,
   -7.48640942986e+00,
   -7.03432350977e+00,
   -6.60864797386e+00,
   -6.20295251927e+00,
   -5.81299467542e+00,
   -5.43578608722e+00,
   -5.06911758492e+00,
   -4.71129366617e+00,
   -4.36097316045e+00,
   -4.01706817286e+00,
   -3.67867706252e+00,
   -3.34503831394e+00,
   -3.01549776957e+00,
   -2.68948470227e+00,
   -2.36649390430e+00,
   -2.04607196869e+00,
   -1.72780654752e+00,
   -1.41131775490e+00,
   -1.09625112896e+00,
   -0.782271729555e+00,
   -0.469059056678e+00,
   -0.156302546889e+00,
   0.156302546889e+00,
   0.469059056678e+00,
   0.782271729555e+00,
   1.09625112896e+00,
   1.41131775490e+00,
   1.72780654752e+00,
   2.04607196869e+00,
   2.36649390430e+00,
   2.68948470227e+00,
   3.01549776957e+00,
   3.34503831394e+00,
   3.67867706252e+00,
   4.01706817286e+00,
   4.36097316045e+00,
   4.71129366617e+00,
   5.06911758492e+00,
   5.43578608722e+00,
   5.81299467542e+00,
   6.20295251927e+00,
   6.60864797386e+00,
   7.03432350977e+00,
   7.48640942986e+00,
   7.97562236821e+00,
   8.52277103092e+00,
   9.18240695813e+00,
   // points 60
   -10.1591092462e+00,
   -9.52090367701e+00,
   -8.99239800140e+00,
   -8.52056928412e+00,
   -8.08518865425e+00,
   -7.67583993750e+00,
   -7.28627659440e+00,
   -6.91238153219e+00,
   -6.55125916706e+00,
   -6.20077355799e+00,
   -5.85929019639e+00,
   -5.52552108614e+00,
   -5.19842653458e+00,
   -4.87715007747e+00,
   -4.56097375794e+00,
   -4.24928643596e+00,
   -3.94156073393e+00,
   -3.63733587617e+00,
   -3.33620465355e+00,
   -3.03780333823e+00,
   -2.74180374807e+00,
   -2.44790690231e+00,
   -2.15583787123e+00,
   -1.86534153123e+00,
   -1.57617901198e+00,
   -1.28812467487e+00,
   -1.00096349956e+00,
   -0.714488781673e+00,
   -0.428500064221e+00,
   -0.142801238703e+00,
   0.142801238703e+00,
   0.428500064221e+00,
   0.714488781673e+00,
   1.00096349956e+00,
   1.28812467487e+00,
   1.57617901198e+00,
   1.86534153123e+00,
   2.15583787123e+00,
   2.44790690231e+00,
   2.74180374807e+00,
   3.03780333823e+00,
   3.33620465355e+00,
   3.63733587617e+00,
   3.94156073393e+00,
   4.24928643596e+00,
   4.56097375794e+00,
   4.87715007747e+00,
   5.19842653458e+00,
   5.52552108614e+00,
   5.85929019639e+00,
   6.20077355799e+00,
   6.55125916706e+00,
   6.91238153219e+00,
   7.28627659440e+00,
   7.67583993750e+00,
   8.08518865425e+00,
   8.52056928412e+00,
   8.99239800140e+00,
   9.52090367701e+00,
   10.1591092462e+00,
   // points 64
   -10.5261231680e+00,
   -9.89528758683e+00,
   -9.37315954965e+00,
   -8.90724909996e+00,
   -8.47752908338e+00,
   -8.07368728501e+00,
   -7.68954016404e+00,
   -7.32101303278e+00,
   -6.96524112055e+00,
   -6.62011226264e+00,
   -6.28401122877e+00,
   -5.95566632680e+00,
   -5.63405216435e+00,
   -5.31832522463e+00,
   -5.00777960220e+00,
   -4.70181564741e+00,
   -4.39991716823e+00,
   -4.10163447457e+00,
   -3.80657151395e+00,
   -3.51437593574e+00,
   -3.22473129199e+00,
   -2.93735082300e+00,
   -2.65197243543e+00,
   -2.36835458863e+00,
   -2.08627287988e+00,
   -1.80551717147e+00,
   -1.52588914021e+00,
   -1.24720015694e+00,
   -0.969269423071e+00,
   -0.691922305810e+00,
   -0.414988824121e+00,
   -0.138302244987e+00,
   0.138302244987e+00,
   0.414988824121e+00,
   0.691922305810e+00,
   0.969269423071e+00,
   1.24720015694e+00,
   1.52588914021e+00,
   1.80551717147e+00,
   2.08627287988e+00,
   2.36835458863e+00,
   2.65197243543e+00,
   2.93735082300e+00,
   3.22473129199e+00,
   3.51437593574e+00,
   3.80657151395e+00,
   4.10163447457e+00,
   4.39991716823e+00,
   4.70181564741e+00,
   5.00777960220e+00,
   5.31832522463e+00,
   5.63405216435e+00,
   5.95566632680e+00,
   6.28401122877e+00,
   6.62011226264e+00,
   6.96524112055e+00,
   7.32101303278e+00,
   7.68954016404e+00,
   8.07368728501e+00,
   8.47752908338e+00,
   8.90724909996e+00,
   9.37315954965e+00,
   9.89528758683e+00,
   10.5261231680e+00
  };

// Gauss-Legendre quadrature; prefix gl

// Number of levels
static const int gl_num_levels = 22;

// Number of points in each level
static const int gl_num_points[] =
  {
   1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
   32, 64
  };

// Weights, starting with the first level
static const double gl_weights[] =
  {
   // weight 1
   2.0e+00,
   // weights 2
   1.0e+00,
   1.0e+00,
   // weights 3
   0.555555555555555555555555555555e+00,
   0.888888888888888888888888888888e+00,
   0.555555555555555555555555555555e+00,
   // weights 4
   0.347854845137453857373063949222e+00,
   0.652145154862546142626936050778e+00,
   0.652145154862546142626936050778e+00,
   0.347854845137453857373063949222e+00,
   // weights 5
   0.236926885056189087514264040720e+00,
   0.478628670499366468041291514836e+00,
   0.568888888888888888888888888889e+00,
   0.478628670499366468041291514836e+00,
   0.236926885056189087514264040720e+00,
   // weights 6
   0.171324492379170345040296142173e+00,
   0.360761573048138607569833513838e+00,
   0.467913934572691047389870343990e+00,
   0.467913934572691047389870343990e+00,
   0.360761573048138607569833513838e+00,
   0.171324492379170345040296142173e+00,
   // weights 7
   0.129484966168869693270611432679e+00,
   0.279705391489276667901467771424e+00,
   0.381830050505118944950369775489e+00,
   0.417959183673469387755102040816e+00,
   0.381830050505118944950369775489e+00,
   0.279705391489276667901467771424e+00,
   0.129484966168869693270611432679e+00,
   // weights 8
   0.101228536290376259152531354310e+00,
   0.222381034453374470544355994426e+00,
   0.313706645877887287337962201987e+00,
   0.362683783378361982965150449277e+00,
   0.362683783378361982965150449277e+00,
   0.313706645877887287337962201987e+00,
   0.222381034453374470544355994426e+00,
   0.101228536290376259152531354310e+00,
   // weights 9
   0.812743883615744119718921581105e-01,
   0.180648160694857404058472031243e+00,
   0.260610696402935462318742869419e+00,
   0.312347077040002840068630406584e+00,
   0.330239355001259763164525069287e+00,
   0.312347077040002840068630406584e+00,
   0.260610696402935462318742869419e+00,
   0.180648160694857404058472031243e+00,
   0.812743883615744119718921581105e-01,
   // weights 10
   0.666713443086881375935688098933e-01,
   0.149451349150580593145776339658e+00,
   0.219086362515982043995534934228e+00,
   0.269266719309996355091226921569e+00,
   0.295524224714752870173892994651e+00,
   0.295524224714752870173892994651e+00,
   0.269266719309996355091226921569e+00,
   0.219086362515982043995534934228e+00,
   0.149451349150580593145776339658e+00,
   0.666713443086881375935688098933e-01,
   // weights 11
   0.556685671161736664827537204425e-01,
   0.125580369464904624634694299224e+00,
   0.186290210927734251426097641432e+00,
   0.233193764591990479918523704843e+00,
   0.262804544510246662180688869891e+00,
   0.272925086777900630714483528336e+00,
   0.262804544510246662180688869891e+00,
   0.233193764591990479918523704843e+00,
   0.186290210927734251426097641432e+00,
   0.125580369464904624634694299224e+00,
   0.556685671161736664827537204425e-01,
   // weights 12
   0.471753363865118271946159614850e-01,
   0.106939325995318430960254718194e+00,
   0.160078328543346226334652529543e+00,
   0.203167426723065921749064455810e+00,
   0.233492536538354808760849898925e+00,
   0.249147045813402785000562436043e+00,
   0.249147045813402785000562436043e+00,
   0.233492536538354808760849898925e+00,
   0.203167426723065921749064455810e+00,
   0.160078328543346226334652529543e+00,
   0.106939325995318430960254718194e+00,
   0.471753363865118271946159614850e-01,
   // weights 13
   0.404840047653158795200215922010e-01,
   0.921214998377284479144217759538e-01,
   0.138873510219787238463601776869e+00,
   0.178145980761945738280046691996e+00,
   0.207816047536888502312523219306e+00,
   0.226283180262897238412090186040e+00,
   0.232551553230873910194589515269e+00,
   0.226283180262897238412090186040e+00,
   0.207816047536888502312523219306e+00,
   0.178145980761945738280046691996e+00,
   0.138873510219787238463601776869e+00,
   0.921214998377284479144217759538e-01,
   0.404840047653158795200215922010e-01,
   // weights 14
   0.351194603317518630318328761382e-01,
   0.801580871597602098056332770629e-01,
   0.121518570687903184689414809072e+00,
   0.157203167158193534569601938624e+00,
   0.185538397477937813741716590125e+00,
   0.205198463721295603965924065661e+00,
   0.215263853463157790195876443316e+00,
   0.215263853463157790195876443316e+00,
   0.205198463721295603965924065661e+00,
   0.185538397477937813741716590125e+00,
   0.157203167158193534569601938624e+00,
   0.121518570687903184689414809072e+00,
   0.801580871597602098056332770629e-01,
   0.351194603317518630318328761382e-01,
   // weights 15
   0.307532419961172683546283935772e-01,
   0.703660474881081247092674164507e-01,
   0.107159220467171935011869546686e+00,
   0.139570677926154314447804794511e+00,
   0.166269205816993933553200860481e+00,
   0.186161000015562211026800561866e+00,
   0.198431485327111576456118326444e+00,
   0.202578241925561272880620199968e+00,
   0.198431485327111576456118326444e+00,
   0.186161000015562211026800561866e+00,
   0.166269205816993933553200860481e+00,
   0.139570677926154314447804794511e+00,
   0.107159220467171935011869546686e+00,
   0.703660474881081247092674164507e-01,
   0.307532419961172683546283935772e-01,
   // weights 16
   0.271524594117540948517805724560e-01,
   0.622535239386478928628438369944e-01,
   0.951585116824927848099251076022e-01,
   0.124628971255533872052476282192e+00,
   0.149595988816576732081501730547e+00,
   0.169156519395002538189312079030e+00,
   0.182603415044923588866763667969e+00,
   0.189450610455068496285396723208e+00,
   0.189450610455068496285396723208e+00,
   0.182603415044923588866763667969e+00,
   0.169156519395002538189312079030e+00,
   0.149595988816576732081501730547e+00,
   0.124628971255533872052476282192e+00,
   0.951585116824927848099251076022e-01,
   0.622535239386478928628438369944e-01,
   0.271524594117540948517805724560e-01,
   // weights 17
   0.241483028685479319601100262876e-01,
   0.554595293739872011294401653582e-01,
   0.850361483171791808835353701911e-01,
   0.111883847193403971094788385626e+00,
   0.135136368468525473286319981702e+00,
   0.154045761076810288081431594802e+00,
   0.168004102156450044509970663788e+00,
   0.176562705366992646325270990113e+00,
   0.179446470356206525458265644262e+00,
   0.176562705366992646325270990113e+00,
   0.168004102156450044509970663788e+00,
   0.154045761076810288081431594802e+00,
   0.135136368468525473286319981702e+00,
   0.111883847193403971094788385626e+00,
   0.850361483171791808835353701911e-01,
   0.554595293739872011294401653582e-01,
   0.241483028685479319601100262876e-01,
   // weights 18
   0.216160135264833103133427102665e-01,
   0.497145488949697964533349462026e-01,
   0.764257302548890565291296776166e-01,
   0.100942044106287165562813984925e+00,
   0.122555206711478460184519126800e+00,
   0.140642914670650651204731303752e+00,
   0.154684675126265244925418003836e+00,
   0.164276483745832722986053776466e+00,
   0.169142382963143591840656470135e+00,
   0.169142382963143591840656470135e+00,
   0.164276483745832722986053776466e+00,
   0.154684675126265244925418003836e+00,
   0.140642914670650651204731303752e+00,
   0.122555206711478460184519126800e+00,
   0.100942044106287165562813984925e+00,
   0.764257302548890565291296776166e-01,
   0.497145488949697964533349462026e-01,
   0.216160135264833103133427102665e-01,
   // weights 19
   0.194617882297264770363120414644e-01,
   0.448142267656996003328381574020e-01,
   0.690445427376412265807082580060e-01,
   0.914900216224499994644620941238e-01,
   0.111566645547333994716023901682e+00,
   0.128753962539336227675515784857e+00,
   0.142606702173606611775746109442e+00,
   0.152766042065859666778855400898e+00,
   0.158968843393954347649956439465e+00,
   0.161054449848783695979163625321e+00,
   0.158968843393954347649956439465e+00,
   0.152766042065859666778855400898e+00,
   0.142606702173606611775746109442e+00,
   0.128753962539336227675515784857e+00,
   0.111566645547333994716023901682e+00,
   0.914900216224499994644620941238e-01,
   0.690445427376412265807082580060e-01,
   0.448142267656996003328381574020e-01,
   0.194617882297264770363120414644e-01,
   // weights 20
   0.176140071391521183118619623519e-01,
   0.406014298003869413310399522749e-01,
   0.626720483341090635695065351870e-01,
   0.832767415767047487247581432220e-01,
   0.101930119817240435036750135480e+00,
   0.118194531961518417312377377711e+00,
   0.131688638449176626898494499748e+00,
   0.142096109318382051329298325067e+00,
   0.149172986472603746787828737002e+00,
   0.152753387130725850698084331955e+00,
   0.152753387130725850698084331955e+00,
   0.149172986472603746787828737002e+00,
   0.142096109318382051329298325067e+00,
   0.131688638449176626898494499748e+00,
   0.118194531961518417312377377711e+00,
   0.101930119817240435036750135480e+00,
   0.832767415767047487247581432220e-01,
   0.626720483341090635695065351870e-01,
   0.406014298003869413310399522749e-01,
   0.176140071391521183118619623519e-01,
   // weights 32
   0.701861000947009660040706373885e-02,
   0.162743947309056706051705622064e-01,
   0.253920653092620594557525897892e-01,
   0.342738629130214331026877322524e-01,
   0.428358980222266806568786466061e-01,
   0.509980592623761761961632446895e-01,
   0.586840934785355471452836373002e-01,
   0.658222227763618468376500637069e-01,
   0.723457941088485062253993564785e-01,
   0.781938957870703064717409188283e-01,
   0.833119242269467552221990746043e-01,
   0.876520930044038111427714627518e-01,
   0.911738786957638847128685771116e-01,
   0.938443990808045656391802376681e-01,
   0.956387200792748594190820022041e-01,
   0.965400885147278005667648300636e-01,
   0.965400885147278005667648300636e-01,
   0.956387200792748594190820022041e-01,
   0.938443990808045656391802376681e-01,
   0.911738786957638847128685771116e-01,
   0.876520930044038111427714627518e-01,
   0.833119242269467552221990746043e-01,
   0.781938957870703064717409188283e-01,
   0.723457941088485062253993564785e-01,
   0.658222227763618468376500637069e-01,
   0.586840934785355471452836373002e-01,
   0.509980592623761761961632446895e-01,
   0.428358980222266806568786466061e-01,
   0.342738629130214331026877322524e-01,
   0.253920653092620594557525897892e-01,
   0.162743947309056706051705622064e-01,
   0.701861000947009660040706373885e-02,
   // weights 64
   0.178328072169643294729607914497e-02,
   0.414703326056246763528753572855e-02,
   0.650445796897836285611736039998e-02,
   0.884675982636394772303091465973e-02,
   0.111681394601311288185904930192e-01,
   0.134630478967186425980607666860e-01,
   0.157260304760247193219659952975e-01,
   0.179517157756973430850453020011e-01,
   0.201348231535302093723403167285e-01,
   0.222701738083832541592983303842e-01,
   0.243527025687108733381775504091e-01,
   0.263774697150546586716917926252e-01,
   0.283396726142594832275113052002e-01,
   0.302346570724024788679740598195e-01,
   0.320579283548515535854675043479e-01,
   0.338051618371416093915654821107e-01,
   0.354722132568823838106931467152e-01,
   0.370551285402400460404151018096e-01,
   0.385501531786156291289624969468e-01,
   0.399537411327203413866569261283e-01,
   0.412625632426235286101562974736e-01,
   0.424735151236535890073397679088e-01,
   0.435837245293234533768278609737e-01,
   0.445905581637565630601347100309e-01,
   0.454916279274181444797709969713e-01,
   0.462847965813144172959532492323e-01,
   0.469681828162100173253262857546e-01,
   0.475401657148303086622822069442e-01,
   0.479993885964583077281261798713e-01,
   0.483447622348029571697695271580e-01,
   0.485754674415034269347990667840e-01,
   0.486909570091397203833653907347e-01,
   0.486909570091397203833653907347e-01,
   0.485754674415034269347990667840e-01,
   0.483447622348029571697695271580e-01,
   0.479993885964583077281261798713e-01,
   0.475401657148303086622822069442e-01,
   0.469681828162100173253262857546e-01,
   0.462847965813144172959532492323e-01,
   0.454916279274181444797709969713e-01,
   0.445905581637565630601347100309e-01,
   0.435837245293234533768278609737e-01,
   0.424735151236535890073397679088e-01,
   0.412625632426235286101562974736e-01,
   0.399537411327203413866569261283e-01,
   0.385501531786156291289624969468e-01,
   0.370551285402400460404151018096e-01,
   0.354722132568823838106931467152e-01,
   0.338051618371416093915654821107e-01,
   0.320579283548515535854675043479e-01,
   0.302346570724024788679740598195e-01,
   0.283396726142594832275113052002e-01,
   0.263774697150546586716917926252e-01,
   0.243527025687108733381775504091e-01,
   0.222701738083832541592983303842e-01,
   0.201348231535302093723403167285e-01,
   0.179517157756973430850453020011e-01,
   0.157260304760247193219659952975e-01,
   0.134630478967186425980607666860e-01,
   0.111681394601311288185904930192e-01,
   0.884675982636394772303091465973e-02,
   0.650445796897836285611736039998e-02,
   0.414703326056246763528753572855e-02,
   0.178328072169643294729607914497e-02
  };

// Points, starting with the first level
static const double gl_points[] =
  {
   // points 1
   0.0e+00,
   // points 2
   -0.577350269189625764509148780502e+00,
   0.577350269189625764509148780502e+00,
   // points 3
   -0.774596669241483377035853079956e+00,
   0.0e+00,
   0.774596669241483377035853079956e+00,
   // points 4
   -0.861136311594052575223946488893e+00,
   -0.339981043584856264802665759103e+00,
   0.339981043584856264802665759103e+00,
   0.861136311594052575223946488893e+00,
   // points 5
   -0.906179845938663992797626878299e+00,
   -0.538469310105683091036314420700e+00,
   0.0e+00,
   0.538469310105683091036314420700e+00,
   0.906179845938663992797626878299e+00,
   // points 6
   -0.932469514203152027812301554494e+00,
   -0.661209386466264513661399595020e+00,
   -0.238619186083196908630501721681e+00,
   0.238619186083196908630501721681e+00,
   0.661209386466264513661399595020e+00,
   0.932469514203152027812301554494e+00,
   // points 7
   -0.949107912342758524526189684048e+00,
   -0.741531185599394439863864773281e+00,
   -0.405845151377397166906606412077e+00,
   0.0e+00,
   0.405845151377397166906606412077e+00,
   0.741531185599394439863864773281e+00,
   0.949107912342758524526189684048e+00,
   // points 8
   -0.960289856497536231683560868569e+00,
   -0.796666477413626739591553936476e+00,
   -0.525532409916328985817739049189e+00,
   -0.183434642495649804939476142360e+00,
   0.183434642495649804939476142360e+00,
   0.525532409916328985817739049189e+00,
   0.796666477413626739591553936476e+00,
   0.960289856497536231683560868569e+00,
   // points 9
   -0.968160239507626089835576202904e+00,
   -0.836031107326635794299429788070e+00,
   -0.613371432700590397308702039341e+00,
   -0.324253423403808929038538014643e+00,
   0.0e+00,
   0.324253423403808929038538014643e+00,
   0.613371432700590397308702039341e+00,
   0.836031107326635794299429788070e+00,
   0.968160239507626089835576202904e+00,
   // points 10
   -0.973906528517171720077964012084e+00,
   -0.865063366688984510732096688423e+00,
   -0.679409568299024406234327365115e+00,
   -0.433395394129247190799265943166e+00,
   -0.148874338981631210884826001130e+00,
   0.148874338981631210884826001130e+00,
   0.433395394129247190799265943166e+00,
   0.679409568299024406234327365115e+00,
   0.865063366688984510732096688423e+00,
   0.973906528517171720077964012084e+00,
   // points 11
   -0.978228658146056992803938001123e+00,
   -0.887062599768095299075157769304e+00,
   -0.730152005574049324093416252031e+00,
   -0.519096129206811815925725669459e+00,
   -0.269543155952344972331531985401e+00,
   0.0e+00,
   0.269543155952344972331531985401e+00,
   0.519096129206811815925725669459e+00,
   0.730152005574049324093416252031e+00,
   0.887062599768095299075157769304e+00,
   0.978228658146056992803938001123e+00,
   // points 12
   -0.981560634246719250690549090149e+00,
   -0.904117256370474856678465866119e+00,
   -0.769902674194304687036893833213e+00,
   -0.587317954286617447296702418941e+00,
   -0.367831498998180193752691536644e+00,
   -0.125233408511468915472441369464e+00,
   0.125233408511468915472441369464e+00,
   0.367831498998180193752691536644e+00,
   0.587317954286617447296702418941e+00,
   0.769902674194304687036893833213e+00,
   0.904117256370474856678465866119e+00,
   0.981560634246719250690549090149e+00,
   // points 13
   -0.984183054718588149472829448807e+00,
   -0.917598399222977965206547836501e+00,
   -0.801578090733309912794206489583e+00,
   -0.642349339440340220643984606996e+00,
   -0.448492751036446852877912852128e+00,
   -0.230458315955134794065528121098e+00,
   0.0e+00,
   0.230458315955134794065528121098e+00,
   0.448492751036446852877912852128e+00,
   0.642349339440340220643984606996e+00,
   0.801578090733309912794206489583e+00,
   0.917598399222977965206547836501e+00,
   0.984183054718588149472829448807e+00,
   // points 14
   -0.986283808696812338841597266704e+00,
   -0.928434883663573517336391139378e+00,
   -0.827201315069764993189794742650e+00,
   -0.687292904811685470148019803019e+00,
   -0.515248636358154091965290718551e+00,
   -0.319112368927889760435671824168e+00,
   -0.108054948707343662066244650220e+00,
   0.108054948707343662066244650220e+00,
   0.319112368927889760435671824168e+00,
   0.515248636358154091965290718551e+00,
   0.687292904811685470148019803019e+00,
   0.827201315069764993189794742650e+00,
   0.928434883663573517336391139378e+00,
   0.986283808696812338841597266704e+00,
   // points 15
   -0.987992518020485428489565718587e+00,
   -0.937273392400705904307758947710e+00,
   -0.848206583410427216200648320774e+00,
   -0.724417731360170047416186054614e+00,
   -0.570972172608538847537226737254e+00,
   -0.394151347077563369897207370981e+00,
   -0.201194093997434522300628303395e+00,
   0.0e+00,
   0.201194093997434522300628303395e+00,
   0.394151347077563369897207370981e+00,
   0.570972172608538847537226737254e+00,
   0.724417731360170047416186054614e+00,
   0.848206583410427216200648320774e+00,
   0.937273392400705904307758947710e+00,
   0.987992518020485428489565718587e+00,
   // points 16
   -0.989400934991649932596154173450e+00,
   -0.944575023073232576077988415535e+00,
   -0.865631202387831743880467897712e+00,
   -0.755404408355003033895101194847e+00,
   -0.617876244402643748446671764049e+00,
   -0.458016777657227386342419442984e+00,
   -0.281603550779258913230460501460e+00,
   -0.950125098376374401853193354250e-01,
   0.950125098376374401853193354250e-01,
   0.281603550779258913230460501460e+00,
   0.458016777657227386342419442984e+00,
   0.617876244402643748446671764049e+00,
   0.755404408355003033895101194847e+00,
   0.865631202387831743880467897712e+00,
   0.944575023073232576077988415535e+00,
   0.989400934991649932596154173450e+00,
   // points 17
   -0.990575475314417335675434019941e+00,
   -0.950675521768767761222716957896e+00,
   -0.880239153726985902122955694488e+00,
   -0.781514003896801406925230055520e+00,
   -0.657671159216690765850302216643e+00,
   -0.512690537086476967886246568630e+00,
   -0.351231763453876315297185517095e+00,
   -0.178484181495847855850677493654e+00,
   0.0e+00,
   0.178484181495847855850677493654e+00,
   0.351231763453876315297185517095e+00,
   0.512690537086476967886246568630e+00,
   0.657671159216690765850302216643e+00,
   0.781514003896801406925230055520e+00,
   0.880239153726985902122955694488e+00,
   0.950675521768767761222716957896e+00,
   0.990575475314417335675434019941e+00,
   // points 18
   -0.991565168420930946730016004706e+00,
   -0.955823949571397755181195892930e+00,
   -0.892602466497555739206060591127e+00,
   -0.803704958972523115682417455015e+00,
   -0.691687043060353207874891081289e+00,
   -0.559770831073947534607871548525e+00,
   -0.411751161462842646035931793833e+00,
   -0.251886225691505509588972854878e+00,
   -0.847750130417353012422618529358e-01,
   0.847750130417353012422618529358e-01,
   0.251886225691505509588972854878e+00,
   0.411751161462842646035931793833e+00,
   0.559770831073947534607871548525e+00,
   0.691687043060353207874891081289e+00,
   0.803704958972523115682417455015e+00,
   0.892602466497555739206060591127e+00,
   0.955823949571397755181195892930e+00,
   0.991565168420930946730016004706e+00,
   // points 19
   -0.992406843843584403189017670253e+00,
   -0.960208152134830030852778840688e+00,
   -0.903155903614817901642660928532e+00,
   -0.822714656537142824978922486713e+00,
   -0.720966177335229378617095860824e+00,
   -0.600545304661681023469638164946e+00,
   -0.464570741375960945717267148104e+00,
   -0.316564099963629831990117328850e+00,
   -0.160358645640225375868096115741e+00,
   0.0e+00,
   0.160358645640225375868096115741e+00,
   0.316564099963629831990117328850e+00,
   0.464570741375960945717267148104e+00,
   0.600545304661681023469638164946e+00,
   0.720966177335229378617095860824e+00,
   0.822714656537142824978922486713e+00,
   0.903155903614817901642660928532e+00,
   0.960208152134830030852778840688e+00,
   0.992406843843584403189017670253e+00,
   // points 20
   -0.993128599185094924786122388471e+00,
   -0.963971927277913791267666131197e+00,
   -0.912234428251325905867752441203e+00,
   -0.839116971822218823394529061702e+00,
   -0.746331906460150792614305070356e+00,
   -0.636053680726515025452836696226e+00,
   -0.510867001950827098004364050955e+00,
   -0.373706088715419560672548177025e+00,
   -0.227785851141645078080496195369e+00,
   -0.765265211334973337546404093988e-01,
   0.765265211334973337546404093988e-01,
   0.227785851141645078080496195369e+00,
   0.373706088715419560672548177025e+00,
   0.510867001950827098004364050955e+00,
   0.636053680726515025452836696226e+00,
   0.746331906460150792614305070356e+00,
   0.839116971822218823394529061702e+00,
   0.912234428251325905867752441203e+00,
   0.963971927277913791267666131197e+00,
   0.993128599185094924786122388471e+00,
   // points 32
   -0.997263861849481563544981128665e+00,
   -0.985611511545268335400175044631e+00,
   -0.964762255587506430773811928118e+00,
   -0.934906075937739689170919134835e+00,
   -0.896321155766052123965307243719e+00,
   -0.849367613732569970133693004968e+00,
   -0.794483795967942406963097298970e+00,
   -0.732182118740289680387426665091e+00,
   -0.663044266930215200975115168663e+00,
   -0.587715757240762329040745476402e+00,
   -0.506899908932229390023747474378e+00,
   -0.421351276130635345364119436172e+00,
   -0.331868602282127649779916805730e+00,
   -0.239287362252137074544603209166e+00,
   -0.144471961582796493485186373599e+00,
   -0.483076656877383162348125704405e-01,
   0.483076656877383162348125704405e-01,
   0.144471961582796493485186373599e+00,
   0.239287362252137074544603209166e+00,
   0.331868602282127649779916805730e+00,
   0.421351276130635345364119436172e+00,
   0.506899908932229390023747474378e+00,
   0.587715757240762329040745476402e+00,
   0.663044266930215200975115168663e+00,
   0.732182118740289680387426665091e+00,
   0.794483795967942406963097298970e+00,
   0.849367613732569970133693004968e+00,
   0.896321155766052123965307243719e+00,
   0.934906075937739689170919134835e+00,
   0.964762255587506430773811928118e+00,
   0.985611511545268335400175044631e+00,
   0.997263861849481563544981128665e+00,
   // points 64
   -0.999305041735772139456905624346e+00,
   -0.996340116771955279346924500676e+00,
   -0.991013371476744320739382383443e+00,
   -0.983336253884625956931299302157e+00,
   -0.973326827789910963741853507352e+00,
   -0.961008799652053718918614121897e+00,
   -0.946411374858402816062481491347e+00,
   -0.929569172131939575821490154559e+00,
   -0.910522137078502805756380668008e+00,
   -0.889315445995114105853404038273e+00,
   -0.865999398154092819760783385070e+00,
   -0.840629296252580362751691544696e+00,
   -0.813265315122797559741923338086e+00,
   -0.783972358943341407610220525214e+00,
   -0.752819907260531896611863774886e+00,
   -0.719881850171610826848940217832e+00,
   -0.685236313054233242563558371031e+00,
   -0.648965471254657339857761231993e+00,
   -0.611155355172393250248852971019e+00,
   -0.571895646202634034283878116659e+00,
   -0.531279464019894545658013903544e+00,
   -0.489403145707052957478526307022e+00,
   -0.446366017253464087984947714759e+00,
   -0.402270157963991603695766771260e+00,
   -0.357220158337668115950442615046e+00,
   -0.311322871990210956157512698560e+00,
   -0.264687162208767416373964172510e+00,
   -0.217423643740007084149648748989e+00,
   -0.169644420423992818037313629748e+00,
   -0.121462819296120554470376463492e+00,
   -0.729931217877990394495429419403e-01,
   -0.243502926634244325089558428537e-01,
   0.243502926634244325089558428537e-01,
   0.729931217877990394495429419403e-01,
   0.121462819296120554470376463492e+00,
   0.169644420423992818037313629748e+00,
   0.217423643740007084149648748989e+00,
   0.264687162208767416373964172510e+00,
   0.311322871990210956157512698560e+00,
   0.357220158337668115950442615046e+00,
   0.402270157963991603695766771260e+00,
   0.446366017253464087984947714759e+00,
   0.489403145707052957478526307022e+00,
   0.531279464019894545658013903544e+00,
   0.571895646202634034283878116659e+00,
   0.611155355172393250248852971019e+00,
   0.648965471254657339857761231993e+00,
   0.685236313054233242563558371031e+00,
   0.719881850171610826848940217832e+00,
   0.752819907260531896611863774886e+00,
   0.783972358943341407610220525214e+00,
   0.813265315122797559741923338086e+00,
   0.840629296252580362751691544696e+00,
   0.865999398154092819760783385070e+00,
   0.889315445995114105853404038273e+00,
   0.910522137078502805756380668008e+00,
   0.929569172131939575821490154559e+00,
   0.946411374858402816062481491347e+00,
   0.961008799652053718918614121897e+00,
   0.973326827789910963741853507352e+00,
   0.983336253884625956931299302157e+00,
   0.991013371476744320739382383443e+00,
   0.996340116771955279346924500676e+00,
   0.999305041735772139456905624346e+00
  };
