// Copyright 2013 Dolphin Emulator Project
// Licensed under GPLv2
// Refer to the license.txt file included.

#include <map>

#include "Common.h"
#include "MemoryUtil.h"
#include "x64ABI.h"
#include "Thunk.h"

#define THUNK_ARENA_SIZE 1024*1024*1

namespace
{

static u8 GC_ALIGNED32(saved_fp_state[16 * 4 * 4]);
static u8 GC_ALIGNED32(saved_gpr_state[16 * 8]);
static u16 saved_mxcsr;

}  // namespace

using namespace Gen;

void ThunkManager::Init()
{
	AllocCodeSpace(THUNK_ARENA_SIZE);
	save_regs = GetCodePtr();
	for (int i = 2; i < ABI_GetNumXMMRegs(); i++)
		MOVAPS(M(saved_fp_state + i * 16), (X64Reg)(XMM0 + i));
	STMXCSR(M(&saved_mxcsr));
#ifdef _M_X64
	MOV(64, M(saved_gpr_state + 0 ), R(RCX));
	MOV(64, M(saved_gpr_state + 8 ), R(RDX));
	MOV(64, M(saved_gpr_state + 16), R(R8) );
	MOV(64, M(saved_gpr_state + 24), R(R9) );
	MOV(64, M(saved_gpr_state + 32), R(R10));
	MOV(64, M(saved_gpr_state + 40), R(R11));
#ifndef _WIN32
	MOV(64, M(saved_gpr_state + 48), R(RSI));
	MOV(64, M(saved_gpr_state + 56), R(RDI));
#endif
	MOV(64, M(saved_gpr_state + 64), R(RBX));
#else
	MOV(32, M(saved_gpr_state + 0 ), R(RCX));
	MOV(32, M(saved_gpr_state + 4 ), R(RDX));
#endif
	RET();
	load_regs = GetCodePtr();
	LDMXCSR(M(&saved_mxcsr));
	for (int i = 2; i < ABI_GetNumXMMRegs(); i++)
		MOVAPS((X64Reg)(XMM0 + i), M(saved_fp_state + i * 16));
#ifdef _M_X64
	MOV(64, R(RCX), M(saved_gpr_state + 0 ));
	MOV(64, R(RDX), M(saved_gpr_state + 8 ));
	MOV(64, R(R8) , M(saved_gpr_state + 16));
	MOV(64, R(R9) , M(saved_gpr_state + 24));
	MOV(64, R(R10), M(saved_gpr_state + 32));
	MOV(64, R(R11), M(saved_gpr_state + 40));
#ifndef _WIN32
	MOV(64, R(RSI), M(saved_gpr_state + 48));
	MOV(64, R(RDI), M(saved_gpr_state + 56));
#endif
	MOV(64, R(RBX), M(saved_gpr_state + 64));
#else
	MOV(32, R(RCX), M(saved_gpr_state + 0 ));
	MOV(32, R(RDX), M(saved_gpr_state + 4 ));
#endif
	RET();
}

void ThunkManager::Reset()
{
	thunks.clear();
	ResetCodePtr();
}

void ThunkManager::Shutdown()
{
	Reset();
	FreeCodeSpace();
}

void *ThunkManager::ProtectFunction(void *function, int num_params)
{
	std::map<void *, const u8 *>::iterator iter;
	iter = thunks.find(function);
	if (iter != thunks.end())
		return (void *)iter->second;
	if (!region)
		PanicAlert("Trying to protect functions before the emu is started. Bad bad bad.");

	const u8 *call_point = GetCodePtr();
#ifdef _M_X64
	// Make sure to align stack.
	ABI_AlignStack(0, true);
	CALL((void*)save_regs);
	CALL((void*)function);
	CALL((void*)load_regs);
	ABI_RestoreStack(0, true);
	RET();
#else
	CALL((void*)save_regs);
	// Since parameters are in the previous stack frame, not in registers, this takes some
	// trickery : we simply re-push the parameters. might not be optimal, but that doesn't really
	// matter.
	ABI_AlignStack(num_params * 4, true);
	unsigned int alignedSize = ABI_GetAlignedFrameSize(num_params * 4, true);
	for (int i = 0; i < num_params; i++) {
		// dst-arg1 dst-arg2 | dst-arg3 padding return-addr orig-arg1 | orig-arg2 orig-arg3
		//                   ^ ESP                                    ^ target
		// The offset is just alignedSize (return address makes up for the
		// missing argument).
		PUSH(32, MDisp(ESP, alignedSize));
	}
	CALL(function);
	ABI_RestoreStack(num_params * 4, true);
	CALL((void*)load_regs);
	RET();
#endif

	thunks[function] = call_point;
	return (void *)call_point;
}
