#ifndef H_CDW_DEBUG
#define H_CDW_DEBUG

#include <assert.h>
#include <stdio.h>

/**
 * \file cdw_debug.h
 * \brief Utilities used during development phase to print debug messages
 *
 * Some utilities that can replace calls to fprintf(stderr, ...).
 */







/**
 * \brief Print debug message - verbose version
 *
 * This macro behaves much like fprintf(stderr, ...) function,
 * caller only have to provide format string with converesion specifiers
 * and list of arguments for this format string.
 *
 * Each message is preceeded with name of function that called the macro.
 *
 * See "C: A Reference Manual", chapter 3.3.10 for more information
 * on variable argument lists in macros (it requires C99).
 */
#ifndef NDEBUG
#define cdw_vdm(...) fprintf(stderr, "%s():%d: ", __func__, __LINE__); fprintf(stderr, __VA_ARGS__);
#else
#define cdw_vdm(...)
#endif





/** \brief Print debug message - silent version */
#define cdw_sdm(...)





/**
 * \brief Print literal cdw return value - verbose version
 *
 * Print literal corresponding to integer value of given variable of type
 * cdw_rv_t.
 *
 * \param func - name of function that returns a cdw_rv_t value ('char *' string)
 * \param val - value of type cdw_rv_t that should be interpreted by this macro
 */
#ifndef NDEBUG
#define cdw_vcrv(func, val) \
if (val == CDW_OK) { \
	cdw_vdm("%s() returns CDW_OK\n", func); \
} else if (val == CDW_NO) { \
	cdw_vdm("%s() returns CDW_NO\n", func); \
} else if (val == CDW_CANCEL) { \
	cdw_vdm("%s() returns CDW_CANCEL\n", func); \
} else if (val == CDW_GEN_ERROR) { \
	cdw_vdm("%s() returns CDW_GEN_ERROR\n", func); \
} else if (val == CDW_MEM_ERROR) { \
	cdw_vdm("%s() returns CDW_MEM_ERROR\n", func); \
} else if (val == CDW_SYS_ERROR) { \
	cdw_vdm("%s() returns CDW_SYS_ERROR\n", func); \
} else if (val == CDW_FILE_INVALID) { \
	cdw_vdm("%s() returns CDW_FILE_INVALID\n", func); \
} else if (val == CDW_ARG_ERROR) { \
	cdw_vdm("%s() returns CDW_ARG_ERROR\n", func); \
} else if (val == CDW_FAILSAFE_MODE) { \
	cdw_vdm("%s() returns CDW_FAILSAFE_MODE\n", func); \
} else { \
	cdw_vdm("%s() returns unknown crv value: %d\n", func, val); \
}
#else
#define cdw_vcrv(func, val)
#endif




/** \brief Print literal cdw return value - silent version of cdw_vcrv */
#define cdw_scrv(func, val)





/**
 * \brief Assert macro with message
 */
#ifndef NDEBUG
#define cdw_assert(expr, ...)                   \
if (! (expr)) {                                 \
 fprintf(stderr, "\n\nassertion failed in:\n"); \
 fprintf(stderr, "file %s\n", __FILE__);        \
 fprintf(stderr, "line %d\n", __LINE__);        \
 cdw_vdm (__VA_ARGS__);                         \
 fprintf(stderr, "\n\n");                       \
assert (expr);                                  \
}
#else
#define cdw_assert(expr, ...)
#endif


/**
 * \brief Assert macro with message - version for unit tests
 */
#ifdef CDW_UNIT_TEST_CODE
#define cdw_assert_test(expr, ...)				\
	if (! (expr)) {						\
		fprintf(stderr, "\n\nassertion failed in:\n");	\
		fprintf(stderr, "file %s\n", __FILE__);			\
		fprintf(stderr, "line %d\n", __LINE__);			\
		fprintf(stderr, "%s():%d: ", __func__, __LINE__); fprintf(stderr, __VA_ARGS__); \
		fprintf(stderr, "\n\n");				\
		exit(0);						\
	}
#else
#define cdw_assert_test(expr, ...)
#endif


#endif /* H_CDW_DEBUG */
