#ifndef H_CDW_TASK
#define H_CDW_TASK

#include <stdbool.h>

#include "main.h"
#include "cdw_disc.h"
#include "cdw_config.h"
#include "cdw_iso9660.h"
#include "cdw_ext_tools.h"

/* forward declaration */
typedef struct cdw_task cdw_task_t;

cdw_task_t *cdw_task_new(cdw_id_t id, cdw_disc_t *disc);
void        cdw_task_save(cdw_task_t *task);
void        cdw_task_delete(cdw_task_t **task);

cdw_rv_t cdw_task_check_tool_status(cdw_task_t *task);
void     cdw_task_reset_tool_status(cdw_task_t *task);

const char *cdw_task_get_session_mode_label(cdw_id_t mode);
const char *cdw_task_get_disc_mode_label(cdw_id_t mode);

void cdw_task_debug_print_iso9660_options(cdw_task_t *task);
void cdw_task_debug_print_burn_options(cdw_task_t *task);





#define CDW_TOOL_STATUS_OK                                       0x00000000

/* mkisofs */
#define CDW_TOOL_STATUS_MKISOFS_THE_SAME_JOLIET_NAME             0x00000001
#define CDW_TOOL_STATUS_MKISOFS_JOLIET_TREE_SORT                 0x00000002
#define CDW_TOOL_STATUS_MKISOFS_CANT_FIND_BOOT_IMAGE             0x00000004
#define CDW_TOOL_STATUS_MKISOFS_NO_SUCH_FILE_OR_DIR              0x00000008
#define CDW_TOOL_STATUS_MKISOFS_BAD_OPTION                       0x00000010
#define CDW_TOOL_STATUS_MKISOFS_INODE_NUMBER_OVERFLOW            0x00000020
#define CDW_TOOL_STATUS_MKISOFS_NO_SPACE_LEFT                    0x00000040
#define CDW_TOOL_STATUS_MKISOFS_ISO_LEVEL_MISMATCH               0x00000080
#define CDW_TOOL_STATUS_MKISOFS_THE_SAME_RR_NAME                 0x00000100
#define CDW_TOOL_STATUS_MKISOFS_FILE_DID_GROW                    0x00000200
#define CDW_TOOL_STATUS_MKISOFS_FATAL_ERROR                      0x00000300
#define CDW_TOOL_STATUS_MKISOFS_ALLOW_LIMITED_SIZE               0x00000800
#define CDW_TOOL_STATUS_MKISOFS_LOW_ISO_LEVEL                    0x00001000
#define CDW_TOOL_STATUS_MKISOFS_SHORT_READ                       0x00002000
#define CDW_TOOL_STATUS_MKISOFS_IMPLEMENTATION_BOTCH             0x00004000

/* cdrecord */
#define CDW_TOOL_STATUS_CDRECORD_CANNOT_OPEN_DEVICE              0x00000001
#define CDW_TOOL_STATUS_CDRECORD_NO_SUCH_FILE_OR_DIR             0x00000002
#define CDW_TOOL_STATUS_CDRECORD_CANNOT_SEND_CUE_SHEET           0x00000004
#define CDW_TOOL_STATUS_CDRECORD_CANNOT_BLANK_DVD_RWP_NON_RICOH  0x00000008
#define CDW_TOOL_STATUS_CDRECORD_CANNOT_OPEN_NEXT                0x00000010
#define CDW_TOOL_STATUS_CDRECORD_ERROR_AFTER                     0x00000020
#define CDW_TOOL_STATUS_CDRECORD_DATA_MAY_NOT_FIT                0x00000040
#define CDW_TOOL_STATUS_CDRECORD_BAD_OPTION                      0x00000080
#define CDW_TOOL_STATUS_CDRECORD_NO_DISK_WRONG_DISK              0x00000100

/* growisofs */
#define CDW_TOOL_STATUS_GROWISOFS_NO_SPACE_FOR_FILES             0x00000001
#define CDW_TOOL_STATUS_GROWISOFS_MEDIA_NOT_APPENDABLE           0x00000002
#define CDW_TOOL_STATUS_GROWISOFS_USE_Z_OPTION                   0x00000004
#define CDW_TOOL_STATUS_GROWISOFS_IO_ERROR                       0x00000008
#define CDW_TOOL_STATUS_GROWISOFS_NO_SPACE_FOR_IMAGE             0x00000010
#define CDW_TOOL_STATUS_GROWISOFS_4GB_BOUNDARY                   0x00000020


/* in the process of verification there may be used two different tools:
   standard code reading files, or library code (from libcdio
   or similar library) reading optical disc; in the first case
   there should be displayed some generic system message (e.g.
   can't find some file in user's system), in the second case
   there should be displayed information about problems with
   accessing optical disc or about problems with optical drive */
#define CDW_TOOL_STATUS_CALCULATE_DIGEST_SETUP_FAIL              0x00000001
#define CDW_TOOL_STATUS_CALCULATE_DIGEST_READ_FAIL               0x00000002
#define CDW_TOOL_STATUS_CALCULATE_DIGEST_WRONG_TRACK             0x00000004

/* libcdio */
#define CDW_TOOL_STATUS_LIBCDIO_SETUP_FAIL                       0x00080001
#define CDW_TOOL_STATUS_LIBCDIO_READ_FAIL                        0x00080002


/* xorriso */
#define CDW_TOOL_STATUS_XORRISO_SIMULATION_NOT_SUPPORTED         0x00000001
#define CDW_TOOL_STATUS_XORRISO_NOT_ENOUGH_SPACE                 0x00000002
#define CDW_TOOL_STATUS_XORRISO_CANNOT_READ_ISO_IMAGE_TREE       0x00000004
#define CDW_TOOL_STATUS_XORRISO_NOT_A_KNOWN_OPTION               0x00000008


/* mode of writing files to a CD/DVD disc */
typedef enum {
	CDW_SESSION_MODE_INIT = -1,
	CDW_SESSION_MODE_START_MULTI = 0,  /**< \brief Start writing to disc in mode allowing further appending of data (don't close disc after writing) */
	CDW_SESSION_MODE_CREATE_SINGLE,    /**< \brief Start writing to disc in mode NOT allowing further appending of data (close disc after writing)*/
	CDW_SESSION_MODE_CONTINUE_MULTI,   /**< \brief Continue writing to disc in mode allowing further appending of data (don't close disc after writing) */
	CDW_SESSION_MODE_WRITE_FINAL,      /**< \brief Write to disc in mode NOT allowing further appending of data (close disc after writing) */
	CDW_SESSION_MODE_ERROR,
	CDW_SESSION_MODE_N_MAX
} cdw_session_mode_t;


/* mode of writing files to a CD/DVD disc */
#define CDW_DISC_MODE_INIT        0x100000000000
#define CDW_DISC_MODE_UNSPECIFIED 0x000000000000   /**< \brief Let a tool select its own mode */
#define CDW_DISC_MODE_TAO         0x000000000001   /**< \brief Track at once */
#define CDW_DISC_MODE_DAO         0x000000000010   /**< \brief Disc at once */
#define CDW_DISC_MODE_SAO         0x000000000100   /**< \brief Session at once */
#if 0 /* currently unsupported modes */
#define CDW_DISC_MODE_RAW         0x000000001000   /**< \brief Raw */
#define CDW_DISC_MODE_RAW96P      0x000000010000
#define CDW_DISC_MODE_RAW96R      0x000000100000
#define CDW_DISC_MODE_RAW16       0x000001000000
#endif
#define CDW_DISC_MODE_ERROR       0x010000000000

#define CDW_DISC_MODE_N_MODES    5 /* includes "Error" */
// #define CDW_DISC_MODE_MODES_TABLE_SIZE     CDW_DISC_MODE_SAO


enum task_id {
	CDW_TASK_NONE,               /**< no task currently assigned */

	CDW_TASK_BURN_FROM_IMAGE,    /**< burn image to optical disc */
	CDW_TASK_BURN_FROM_FILES,    /**< burn files to optical disc */
	CDW_TASK_READ_DISC,          /**< read content of optical disc */
	CDW_TASK_ERASE_DISC,         /**< erase optical disc - this includes formatting */
	CDW_TASK_CLONE_DISC,         /**< clone one optical disc to another optical disc */

	CDW_TASK_CREATE_IMAGE,       /**< create iso image on hdd, using files */

	CDW_TASK_CALCULATE_DIGEST,   /**< calculate digest of specified object */

	CDW_TASK_MEDIA_INFO          /**< get check media information */
};


struct cdw_task {

	cdw_id_t id;

	struct {
		bool ok;
		size_t mkisofs;
		size_t cdrecord;
		size_t calculate_digest;
		size_t growisofs;
		size_t dvd_rw_mediainfo;
		size_t libcdio;
		size_t xorriso;
	} tool_status;


	struct {
		/* "input" parameters of erasing */
		cdw_id_t format_mode; /* format of DVD-RW: Sequential or Restricted Overwrite */
		cdw_id_t speed_id;
		cdw_id_t erase_mode;  /* erasing disc: all or fast */
		cdw_id_label_t tool;

		bool eject;

		/* "output" parameter of erasing (but only for CD-RW): if
		   after attempting to perform erasing of CD-RW value of
		   erase_time is still zero, then erasing failed */
		size_t erase_time;
	} erase;

	struct {
		/* single sesion, append, finalize, etc. */
		int session_modes[CDW_SESSION_MODE_N_MAX];
		int session_modes_n_labels;
		cdw_session_mode_t session_mode;

		/* allowed disc modes: tao, sao, etc.; I'm not using
		   single int + ORing, because the order of modes
		   is significant: first mode on the list is most
		   appropriate/probable */
		long long int disc_modes[CDW_DISC_MODE_N_MODES];
		int disc_modes_n_labels;
		long long int disc_mode;

		cdw_id_t speed_id;
		bool verify;
		bool dummy;
		bool eject;

		cdw_id_label_t tool;

		/* size of data being burned to disc; this variable should not store
		   _exact_ size of data, value stored in this variable should be
		   used only when approximate data size needs to be known, e.g. in
		   progress window when displaying approximate progress of task;
		   depending on type of task this variable stores size of iso image
		   file being burned to optical disc, or size of files selected for
		   burning to optical disc */
		double data_size_mb;
		struct {
			char *other_growisofs_options;
		} growisofs;

		struct {
			bool burnproof;
			bool pad;
			int pad_size;
			char *other_cdrecord_options;
		} cdrecord;

		struct {
			char *other_xorriso_burn_options;
		} xorriso;

		/* path to source ISO9660 file */
		char *iso9660_file_fullpath;
	} burn;

	struct {
		cdw_id_label_t tool;

		/* mkisofs: -J
		   xorriso: -joliet on|off */
		bool joliet_information;

		/* mkisofs:
		   xorriso: -follow link */
		bool follow_symlinks;

		/* mkisofs:
		   xorriso: -volid text */
		char volume_id[CDW_ISO9660_VOLI_LEN + 1]; /* 1 for ending '\0' */
		char volume_set_id[CDW_ISO9660_VOLS_LEN + 1];
		char preparer[CDW_ISO9660_PREP_LEN + 1];
		char publisher[CDW_ISO9660_PUBL_LEN + 1];
		char system_id[CDW_ISO9660_SYSI_LEN + 1];
		char copyright[CDW_ISO9660_COPY_LEN + 1];
		char abstract[CDW_ISO9660_ABST_LEN + 1];

		/* mkisofs: -pad (hardwired value 300k);
		   xorriso: -padding Xk|m */
		bool pad;

		/* none/useful/full
		   mkisofs: -r for "useful", -R for "full"
		   xorriso: full is always on, "none is not allowed", and for "useful":
		   "-find / -true -exec mkisofs_r --" */
		cdw_id_t rock_ridge;

		/* path to resulting ISO9660 file */
		char *iso9660_file_fullpath;

		/* mkisofs-specific options */
		struct {
			bool joliet_long;
			cdw_id_t iso_level;
			char *root_dir;
			char *other_mkisofs_options;
		} mkisofs;

		/* xorriso-specific options */
		struct {
			char *other_xorriso_iso_options;
		} xorriso;

	} create_image;

	struct {
		cdw_id_label_t tool;
	} media_info;

	struct {
		cdw_id_label_t tool;

		cdw_id_t mode;

		long long int data_size_sectors;

		/* size of file to be verified */
		long long int file_size;

		int source_fd;
		int target_fd;

		void * (*read_and_write)(void *task);

	} calculate_digest;

	cdw_disc_t *disc;

}; /** \brief Data structure describing task that is currently
       performed or is about to be performed */




#endif /* H_CDW_TASK */
