/*
 *  $Id: app-image-window.c 29071 2026-01-02 17:13:37Z yeti-dn $
 *  Copyright (C) 2025-2026 David Necas (Yeti)
 *  E-mail: yeti@gwyddion.net
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
#define DEBUG 1
#include "config.h"
#include <glib/gi18n-lib.h>
#include <gtk/gtk.h>

#include "libgwyddion/macros.h"
#include "libgwyui/utils.h"

#include "libgwyapp/gwyapp.h"
#include "libgwyapp/sanity.h"
#include "libgwyapp/gwyappinternal.h"

struct _GwyAppImageWindowPrivate {
    GtkWidget *menubar;
    GtkWidget *infobar;
    GtkWidget *info_label;
    GwySensitivityGroup *sens_group;

    GtkWidget *meta_widget;
    gulong meta_destroyed_id;

    GtkWidget *log_widget;
    gulong log_destroyed_id;

    GwyFile *file;
    gulong item_changed_id;

    GObject *object;
    gulong object_changed_id;

    GObject *mask;
    gulong mask_changed_id;

    GwyDataKind data_kind;
    gint id;

    GwyParams *params;
    /* FIXME: This is probably something to store in File. */
    gboolean xyz_is_density;
};

static void       finalize                 (GObject *object);
static void       dispose                  (GObject *object);
static gboolean   configured               (GtkWidget *widget,
                                            GdkEventConfigure *event);
static void       destroyed                (GtkWidget *widget);
static gboolean   key_pressed              (GtkWidget *widget,
                                            GdkEventKey *event);
static void       size_allocated           (GtkWidget *widget,
                                            GdkRectangle *allocation);
static void       save_size                (GwyAppImageWindow *window,
                                            const GdkRectangle *allocation);
static void       add_menubar              (GwyAppImageWindow *window,
                                            GtkBox *vbox);
static void       add_infobar              (GwyAppImageWindow *window,
                                            GtkBox *vbox);
static GtkWidget* create_view_menu         (GwyAppImageWindow *window,
                                            GtkAccelGroup *accel_group);
static void       change_mask_color        (GwyAppImageWindow *window);
static void       show_meta_browser        (GwyAppImageWindow *window);
static void       meta_destroyed           (GwyAppImageWindow *window);
static void       show_log_browser         (GwyAppImageWindow *window);
static void       log_destroyed            (GwyAppImageWindow *window);
static void       reset_zoom               (GwyDataWindow *window);
static void       zoom_in                  (GwyDataWindow *window);
static void       zoom_out                 (GwyDataWindow *window);
static void       realsquare_toggled       (GwyAppImageWindow *window,
                                            GtkCheckMenuItem *item);
static void       item_changed             (GwyContainer *container,
                                            GQuark key,
                                            GwyAppImageWindow *window);
static void       update_color_mapping     (GwyAppImageWindow *window);
static void       dataview_gradient_changed(GwyDataView *dataview,
                                            const GParamSpec *pspec,
                                            GwyAppImageWindow *window);
static void       object_changed           (GObject *object,
                                            GwyAppImageWindow *window);
static void       mask_changed             (GObject *object,
                                            GwyAppImageWindow *window);
static void       initial_setup            (GwyAppImageWindow *window);
static void       update_infobar           (GwyAppImageWindow *windiw);
static void       update_window_title      (GwyAppImageWindow *windiw);
static void       xyz_density_map_toggled  (GtkCheckMenuItem *toggle,
                                            GwyAppImageWindow *window);
static void       update_xyz_preview       (GwyAppImageWindow *window);
static void       maybe_add_xyz_preview    (GwyAppImageWindow *window);
static void       restore_zoom             (GwyAppImageWindow *window);
static gboolean   data_kind_has_pixels     (GwyDataKind data_kind);

static GtkWindowClass *parent_class = NULL;

G_DEFINE_TYPE_WITH_CODE(GwyAppImageWindow, gwy_app_image_window, GWY_TYPE_DATA_WINDOW,
                        G_ADD_PRIVATE(GwyAppImageWindow))

static void
gwy_app_image_window_class_init(GwyAppImageWindowClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
    GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(klass);

    parent_class = gwy_app_image_window_parent_class;

    gobject_class->finalize = finalize;
    gobject_class->dispose = dispose;

    widget_class->configure_event = configured;
    widget_class->key_press_event = key_pressed;
    widget_class->size_allocate = size_allocated;
    widget_class->destroy = destroyed;
    /* TODO: Various button/key press events, popup-menu event (which we might not need if we have a menubar). */
}

static void
gwy_app_image_window_init(GwyAppImageWindow *window)
{
    GwyAppImageWindowPrivate *priv;

    window->priv = priv = gwy_app_image_window_get_instance_private(window);

    priv->sens_group = gwy_sensitivity_group_new();
    gwy_app_add_main_accel_group(GTK_WINDOW(window));
}

static void
dispose(GObject *object)
{
    GwyAppImageWindow *window = GWY_APP_IMAGE_WINDOW(object);
    GwyAppImageWindowPrivate *priv = window->priv;

    gwy_set_member_object(window, NULL, GWY_TYPE_FIELD, &priv->object,
                          "data-changed", G_CALLBACK(object_changed), &priv->object_changed_id, 0,
                          NULL);
    gwy_set_member_object(window, NULL, GWY_TYPE_FIELD, &priv->mask,
                          "data-changed", G_CALLBACK(mask_changed), &priv->mask_changed_id, 0,
                          NULL);
    g_clear_signal_handler(&priv->item_changed_id, priv->file);

    G_OBJECT_CLASS(parent_class)->dispose(object);
}

static void
finalize(GObject *object)
{
    GwyAppImageWindow *window = GWY_APP_IMAGE_WINDOW(object);
    GwyAppImageWindowPrivate *priv = window->priv;

    g_clear_object(&priv->params);
    g_clear_object(&priv->sens_group);

    G_OBJECT_CLASS(parent_class)->finalize(object);
}

/**
 * gwy_app_image_window_new:
 * @file: (transfer none): A data file container.
 * @data_kind: Type of the data object.
 * @id: Data item id.
 *
 * Creates a new app image data window.
 *
 * The arguments must correspond to a valid data object in @file which can be visualised as an image. Possible data
 * kinds include @GWY_FILE_IMAGE, @GWY_FILE_VOLUME, @GWY_FILE_XYZ and @GWY_FILE_CMAP.
 *
 * Returns: (transfer full) (constructor): A newly created window.
 **/
GtkWindow*
gwy_app_image_window_new(GwyFile *file,
                         GwyDataKind data_kind,
                         gint id)
{
    static const GwyEnum help_sections[] = {
        { "data-windows", GWY_FILE_IMAGE,  },
        { "volume-data",  GWY_FILE_VOLUME, },
        { "xyz-data",     GWY_FILE_XYZ,    },
        { "curve-maps",   GWY_FILE_CMAP,   },
    };

    GwyAppImageWindow *window = g_object_new(GWY_TYPE_APP_IMAGE_WINDOW, NULL);
    GwyAppImageWindowPrivate *priv = window->priv;

    g_assert(GWY_IS_FILE(file));
    g_assert(data_kind == GWY_FILE_IMAGE || data_kind == GWY_FILE_VOLUME
             || data_kind == GWY_FILE_XYZ || data_kind == GWY_FILE_CMAP);
    g_assert(id >= 0);
    priv->file = file;
    priv->data_kind = data_kind;
    priv->id = id;

    const gchar *helploc = gwy_enum_to_string(data_kind, help_sections, G_N_ELEMENTS(help_sections));
    if (helploc)
        gwy_help_add_to_window(GTK_WINDOW(window), helploc, NULL, GWY_HELP_DEFAULT);

    GtkBox *vbox = GTK_BOX(gtk_bin_get_child(GTK_BIN(window)));
    g_assert(GTK_IS_BOX(vbox));
    add_infobar(window, vbox);
    add_menubar(window, vbox);

    /* TODO: There are various popups and stuff. */

    initial_setup(window);
    priv->item_changed_id = g_signal_connect(file, "item-changed", G_CALLBACK(item_changed), window);

    GwyDataView *dataview = GWY_DATA_VIEW(gwy_data_window_get_data_view(GWY_DATA_WINDOW(window)));
    g_signal_connect(dataview, "notify::gradient", G_CALLBACK(dataview_gradient_changed), window);

    return (GtkWindow*)window;
}

/**
 * gwy_app_image_window_get_file:
 * @window: App image data window.
 *
 * Gets the data file containing data displayed in an app image data window.
 *
 * Returns: (transfer none): The file data container.
 **/
GwyFile*
gwy_app_image_window_get_file(GwyAppImageWindow *window)
{
    g_return_val_if_fail(GWY_IS_APP_IMAGE_WINDOW(window), NULL);
    return window->priv->file;
}

/**
 * gwy_app_image_window_get_data_kind:
 * @window: App image data window.
 *
 * Gets the kind of data displayed in an app image data window.
 *
 * Returns: The data kind.
 **/
GwyDataKind
gwy_app_image_window_get_data_kind(GwyAppImageWindow *window)
{
    g_return_val_if_fail(GWY_IS_APP_IMAGE_WINDOW(window), GWY_FILE_NONE);
    return window->priv->data_kind;
}

/**
 * gwy_app_image_window_get_id:
 * @window: App image data window.
 *
 * Gets the numerical id of data item displayed in an app image data window.
 *
 * Returns: The numerical id.
 **/
gint
gwy_app_image_window_get_id(GwyAppImageWindow *window)
{
    g_return_val_if_fail(GWY_IS_APP_IMAGE_WINDOW(window), -1);
    return window->priv->id;
}

static void
item_changed(GwyContainer *container, GQuark key, GwyAppImageWindow *window)
{
    GwyAppImageWindowPrivate *priv = window->priv;
    GwyFileKeyParsed parsed;

    const gchar *strkey = g_quark_to_string(key);
    gwy_debug("<%s>", strkey);
    gboolean parsed_ok = gwy_file_parse_key(strkey, &parsed);
    g_return_if_fail(parsed_ok);

    GwyDataKind data_kind = parsed.data_kind;
    gint id = parsed.id;
    GwyFilePiece piece = parsed.piece;
    if (data_kind == GWY_FILE_NONE && piece == GWY_FILE_PIECE_FILENAME) {
        update_window_title(window);
        return;
    }

    if (data_kind != priv->data_kind || id != priv->id)
        return;

    GwyFile *file = GWY_FILE(container);
    g_assert(file == priv->file);
    GwyDataWindow *datawindow = GWY_DATA_WINDOW(window);
    GwyDataView *dataview = GWY_DATA_VIEW(gwy_data_window_get_data_view(datawindow));
    if (data_kind == GWY_FILE_IMAGE && piece == GWY_FILE_PIECE_MASK) {
        GwyMenuSensFlags sensflags = GWY_MENU_FLAG_IMAGE_MASK;
        GwyField *mask = gwy_file_get_image_mask(file, id);
        gwy_set_member_object(window, mask, GWY_TYPE_FIELD, &priv->object,
                              "data-changed", G_CALLBACK(mask_changed), &priv->mask_changed_id, 0,
                              NULL);
        gwy_data_view_set_mask(dataview, mask);
        gwy_sensitivity_group_set_state(priv->sens_group, sensflags, mask ? sensflags : 0);
    }
    else if (data_kind == GWY_FILE_IMAGE && piece == GWY_FILE_PIECE_MASK_COLOR) {
        GwyRGBA color;
        if (!gwy_container_gis_boxed(container, GWY_TYPE_RGBA, key, &color))
            gwy_app_settings_get_default_mask_color(&color);
        gwy_data_view_set_mask_color(dataview, &color);
    }
    else if (data_kind == GWY_FILE_IMAGE && (piece == GWY_FILE_PIECE_NONE || piece == GWY_FILE_PIECE_PICTURE)) {
        GwyField *show = NULL, *field = gwy_file_get_image(file, id);
        gwy_container_gis_object(container, gwy_file_key_image_picture(id), &show);
        if (show)
            gwy_data_view_set_field(dataview, show);
        else if (field)
            gwy_data_view_set_field(dataview, field);
        else
            gwy_data_view_set_field(dataview, NULL);
        gwy_sensitivity_group_set_state(priv->sens_group,
                                        GWY_MENU_FLAG_IMAGE | GWY_MENU_FLAG_IMAGE_SHOW,
                                        (field ? GWY_MENU_FLAG_IMAGE : 0) | (show ? GWY_MENU_FLAG_IMAGE_SHOW : 0));
        update_color_mapping(window);
    }
    else if (piece == GWY_FILE_PIECE_NONE) {
        GType type = 0;
        GwyMenuSensFlags sensflags = 0;
        if (data_kind == GWY_FILE_VOLUME) {
            type = GWY_TYPE_BRICK;
            sensflags = GWY_MENU_FLAG_VOLUME;
        }
        else if (data_kind == GWY_FILE_XYZ) {
            type = GWY_TYPE_SURFACE;
            sensflags = GWY_MENU_FLAG_XYZ;
            update_xyz_preview(window);
        }
        else if (data_kind == GWY_FILE_CMAP) {
            type = GWY_TYPE_LAWN;
            sensflags = GWY_MENU_FLAG_CMAP;
        }
        else {
            g_return_if_reached();
        }

        GObject *object = NULL;
        gwy_container_gis_object(container, key, &object);
        if (gwy_set_member_object(window, object, type, &priv->object,
                                  "data-changed", G_CALLBACK(object_changed), &priv->object_changed_id, 0,
                                  NULL))
            update_infobar(window);

        gwy_sensitivity_group_set_state(priv->sens_group, sensflags, priv->object ? sensflags : 0);
    }
    else if (piece == GWY_FILE_PIECE_PICTURE) {
        GwyField *preview = NULL;
        gwy_container_gis_object(container, key, &preview);
        gwy_data_view_set_field(dataview, preview);
    }
    else if (piece == GWY_FILE_PIECE_PALETTE) {
        GwyColorAxis *coloraxis = GWY_COLOR_AXIS(gwy_data_window_get_color_axis(datawindow));
        const gchar *name = NULL;
        gwy_container_gis_string(container, key, &name);
        GwyGradient *gradient = name ? gwy_inventory_get_item(gwy_gradients(), name) : NULL;
        /* XXX: This is somewhat inconsistent. When someone sets the name to a non-existent gradient we set NULL here
         * because it does not exist. But if it is created later we do not notice. This would not happen if widgets
         * like GwyColorAxis held the name and got the object from the inventory each time they were redrawing
         * themselves. But on the other hand the widget still needs to watch the gradient object because otherwise it
         * does not know if its data change. Anyway, this is an edge case. Worry about it later. */
        gwy_data_view_set_gradient(dataview, gradient);
        gwy_color_axis_set_gradient(coloraxis, gradient);
    }
    else if (piece == GWY_FILE_PIECE_COLOR_MAPPING)
        update_color_mapping(window);
    else if (piece == GWY_FILE_PIECE_RANGE) {
        gdouble min, max;
        if (gwy_file_get_fixed_range(file, data_kind, id, &min, &max))
            gwy_data_view_set_fixed_color_range(dataview, min, max);
        else
            gwy_data_view_unset_fixed_color_range(dataview);
    }
    else if (piece == GWY_FILE_PIECE_REAL_SQUARE) {
        gboolean realsquare = FALSE;
        gwy_container_gis_boolean(container, key, &realsquare);
        gwy_data_view_set_real_square(dataview, realsquare);
    }
    else if (piece == GWY_FILE_PIECE_TITLE) {
        update_window_title(window);
    }
    /* TODO: There are lots of other things: selections, … */

    /* TODO: If we get file piece NONE, update infobar. We should also connect to the main data object and update
     * infobar when it changes. */
}

static void
adaptive_color_axis_map_func(G_GNUC_UNUSED GwyColorAxis *axis,
                             const gdouble *z,
                             gdouble *mapped,
                             guint n,
                             gpointer user_data)
{
    GwyField *field = gwy_data_view_get_field(GWY_DATA_VIEW(user_data));
    if (!field) {
        gwy_clear(mapped, n);
        return;
    }

    gwy_map_field_adaptive(field, z, mapped, n);
}

static void
update_color_mapping(GwyAppImageWindow *window)
{
    GwyAppImageWindowPrivate *priv = window->priv;
    GwyDataWindow *datawindow = GWY_DATA_WINDOW(window);
    GwyDataView *dataview = GWY_DATA_VIEW(gwy_data_window_get_data_view(datawindow));
    GwyColorAxis *coloraxis = GWY_COLOR_AXIS(gwy_data_window_get_color_axis(datawindow));
    GwyContainer *container = GWY_CONTAINER(priv->file);

    GwyColorMappingType color_mapping = GWY_COLOR_MAPPING_FULL;
    GwyColorAxisMapFunc map_func = NULL;
    gpointer map_func_data = NULL;
    GwyTicksStyle ticks_style = GWY_TICKS_STYLE_AUTO;
    gboolean show_labels = TRUE;

    if (gwy_container_contains(container, gwy_file_key_image_picture(priv->id))) {
        ticks_style = GWY_TICKS_STYLE_CENTER;
        show_labels = FALSE;
    }
    else {
        color_mapping = gwy_file_get_color_mapping(priv->file, priv->data_kind, priv->id);
        if (color_mapping == GWY_COLOR_MAPPING_ADAPT) {
            ticks_style = GWY_TICKS_STYLE_UNLABELLED;
            map_func = adaptive_color_axis_map_func;
            map_func_data = dataview;
        }
    }

    gwy_data_view_set_color_mapping(dataview, color_mapping);
    gwy_color_axis_set_ticks_style(coloraxis, ticks_style);
    gwy_color_axis_set_labels_visible(coloraxis, show_labels);
    gwy_color_axis_set_tick_map_func(coloraxis, map_func, map_func_data);
}

static void
dataview_gradient_changed(GwyDataView *dataview,
                          G_GNUC_UNUSED const GParamSpec *pspec,
                          GwyAppImageWindow *window)
{
    GwyGradient *gradient = gwy_data_view_get_gradient(dataview);
    GwyAppImageWindowPrivate *priv = window->priv;
    gwy_file_set_palette(priv->file, priv->data_kind, priv->id,
                         gradient ? gwy_resource_get_name(GWY_RESOURCE(gradient)) : NULL);
}

static void
object_changed(G_GNUC_UNUSED GObject *object,
               GwyAppImageWindow *window)
{
    update_infobar(window);
}

static void
mask_changed(G_GNUC_UNUSED GObject *object,
             GwyAppImageWindow *window)
{
    update_infobar(window);
}

static void
initial_setup(GwyAppImageWindow *window)
{
    static const GwyFilePiece common_pieces[] = {
        GWY_FILE_PIECE_PICTURE,
        GWY_FILE_PIECE_PALETTE,
        GWY_FILE_PIECE_COLOR_MAPPING,
    };

    GwyAppImageWindowPrivate *priv = window->priv;
    GwyDataKind data_kind = priv->data_kind;
    gint id = priv->id;
    GwyFileKeyParsed parsed = { .data_kind = data_kind, .id = id, .piece = GWY_FILE_PIECE_NONE, .suffix = NULL };
    GwyContainer *container = GWY_CONTAINER(priv->file);

    item_changed(container, gwy_file_form_key(&parsed), window);
    for (guint i = 0; i < G_N_ELEMENTS(common_pieces); i++) {
        parsed.piece = common_pieces[i];
        if (data_kind == GWY_FILE_XYZ && parsed.piece == GWY_FILE_PIECE_PICTURE)
            maybe_add_xyz_preview(window);
        item_changed(container, gwy_file_form_key(&parsed), window);
    }
    if (data_kind_has_pixels(data_kind)) {
        parsed.piece = GWY_FILE_PIECE_REAL_SQUARE;
        item_changed(container, gwy_file_form_key(&parsed), window);
    }
    parsed.piece = GWY_FILE_PIECE_RANGE;
    parsed.suffix = "min";
    item_changed(container, gwy_file_form_key(&parsed), window);
    parsed.suffix = "max";
    item_changed(container, gwy_file_form_key(&parsed), window);

    if (data_kind == GWY_FILE_IMAGE) {
        item_changed(container, gwy_file_key_image_mask(id), window);
        item_changed(container, gwy_file_key_image_mask_color(id), window);
    }
    update_window_title(window);
    /* TODO: There are lots of other things: selections, … */

    restore_zoom(window);
}

static void
add_menubar(GwyAppImageWindow *window, GtkBox *vbox)
{
    GwyAppImageWindowPrivate *priv = window->priv;

    GtkAccelGroup *accel_group = NULL;
    GtkWidget *main_window = gwy_app_main_window_get();
    if (main_window)
        accel_group = GTK_ACCEL_GROUP(g_object_get_data(G_OBJECT(main_window), "accel_group"));

    GwyDataKind data_kind = priv->data_kind;
    GtkWidget *funcmenu = NULL, *item;
    const gchar *label = NULL;
    if (data_kind == GWY_FILE_IMAGE) {
        funcmenu = gwy_app_image_menu(accel_group, priv->sens_group);
        label = _("_Image");
    }
    else if (data_kind == GWY_FILE_VOLUME) {
        funcmenu = gwy_app_volume_menu(accel_group, priv->sens_group);
        label = _("_Volume");
    }
    else if (data_kind == GWY_FILE_XYZ) {
        funcmenu = gwy_app_xyz_menu(accel_group, priv->sens_group);
        label = _("_XYZ");
    }
    else if (data_kind == GWY_FILE_CMAP) {
        funcmenu = gwy_app_curve_map_menu(accel_group, priv->sens_group);
        label = _("_Curve Map");
    }

    priv->menubar = gtk_menu_bar_new();
    gtk_box_pack_start(vbox, priv->menubar, FALSE, FALSE, 0);
    gtk_box_reorder_child(vbox, priv->menubar, 0);

    if (funcmenu) {
        item = gtk_menu_item_new_with_mnemonic(label);
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), funcmenu);
        gtk_menu_shell_append(GTK_MENU_SHELL(priv->menubar), item);
    }

    if (funcmenu) {
        GtkWidget *recentmenu = gwy_app_create_recent_func_menu(data_kind, GTK_WINDOW(window), priv->sens_group);
        item = gtk_menu_item_new_with_mnemonic(("_Recent"));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), recentmenu);
        gtk_menu_shell_append(GTK_MENU_SHELL(priv->menubar), item);
    }

    item = gtk_menu_item_new_with_mnemonic(("Vie_w"));
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), create_view_menu(window, accel_group));
    gtk_menu_shell_append(GTK_MENU_SHELL(priv->menubar), item);
}

/* FIXME: This should probably be a small toolbar, which could include the ‘Change Preview’ functions for volume
 * and curve map data. */
static GtkWidget*
create_view_menu(GwyAppImageWindow *window, GtkAccelGroup *accel_group)
{
    GtkAccelFlags accel_flags = GTK_ACCEL_VISIBLE | GTK_ACCEL_LOCKED;
    GwyAppImageWindowPrivate *priv = window->priv;
    GwyDataKind data_kind = priv->data_kind;
    gint id = priv->id;
    GtkWidget *item;

    GtkWidget *menu = gtk_menu_new();
    GtkMenuShell *shell = GTK_MENU_SHELL(menu);
    if (accel_group)
        gtk_menu_set_accel_group(GTK_MENU(menu), accel_group);

    if (data_kind == GWY_FILE_IMAGE || data_kind == GWY_FILE_VOLUME || data_kind == GWY_FILE_CMAP) {
        item = gtk_menu_item_new_with_mnemonic(_("Zoom _1:1 (/)"));
        gtk_menu_shell_append(shell, item);
        g_signal_connect_swapped(item, "activate", G_CALLBACK(reset_zoom), window);
    }

    item = gwy_create_image_menu_item(_("Zoom In (+)"), GWY_ICON_ZOOM_IN, FALSE);
    gtk_menu_shell_append(shell, item);
    g_signal_connect_swapped(item, "activate", G_CALLBACK(zoom_in), window);

    item = gwy_create_image_menu_item(_("Zoom Out (−)"), GWY_ICON_ZOOM_OUT, FALSE);
    gtk_menu_shell_append(shell, item);
    g_signal_connect_swapped(item, "activate", G_CALLBACK(zoom_out), window);

    if (data_kind_has_pixels(data_kind)) {
        item = gtk_check_menu_item_new_with_mnemonic(_("_Physical Aspect Ratio"));
        GwyFileKeyParsed parsed = {
            .data_kind = data_kind, .id = id, .piece = GWY_FILE_PIECE_REAL_SQUARE, .suffix = NULL
        };
        gboolean realsquare = FALSE;
        if (gwy_container_gis_boolean(GWY_CONTAINER(priv->file), gwy_file_form_key(&parsed), &realsquare) && realsquare)
            gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), TRUE);
        gtk_menu_shell_append(shell, item);
        g_signal_connect_swapped(item, "activate", G_CALLBACK(realsquare_toggled), window);
    }

    if (data_kind == GWY_FILE_IMAGE) {
        item = gwy_create_image_menu_item(_("Mask _Color"), GWY_ICON_MASK, FALSE);
        gtk_menu_shell_append(shell, item);
        gwy_sensitivity_group_add_widget(priv->sens_group, item, GWY_MENU_FLAG_IMAGE_MASK);
        g_signal_connect_swapped(item, "activate", G_CALLBACK(change_mask_color), window);
    }

    if (data_kind == GWY_FILE_VOLUME || data_kind == GWY_FILE_CMAP) {
        item = gtk_menu_item_new_with_mnemonic(_("_Change Preview"));
        gtk_menu_shell_append(shell, item);
        if (data_kind == GWY_FILE_VOLUME)
            g_signal_connect_swapped(item, "activate", G_CALLBACK(_gwy_app_change_volume_preview), window);
        else
            g_signal_connect_swapped(item, "activate", G_CALLBACK(_gwy_app_change_cmap_preview), window);
    }

    if (data_kind == GWY_FILE_XYZ) {
        item = gtk_menu_item_new_with_mnemonic(_("_Update Preview"));
        gtk_menu_shell_append(shell, item);
        g_signal_connect_swapped(item, "activate", G_CALLBACK(update_xyz_preview), window);

        item = gtk_check_menu_item_new_with_mnemonic(_("_Density Map"));
        gtk_menu_shell_append(shell, item);
        g_signal_connect(item, "toggled", G_CALLBACK(xyz_density_map_toggled), window);
    }

    gtk_menu_shell_append(shell, gtk_separator_menu_item_new());

    item = gtk_menu_item_new_with_mnemonic(_("_Metadata..."));
    gtk_widget_add_accelerator(item, "activate", accel_group,
                               GDK_KEY_B, GDK_CONTROL_MASK | GDK_SHIFT_MASK, accel_flags);
    gtk_menu_shell_append(shell, item);
    g_signal_connect_swapped(item, "activate", G_CALLBACK(show_meta_browser), window);

    item = gtk_menu_item_new_with_mnemonic(_("_Log..."));
    gtk_menu_shell_append(shell, item);
    g_signal_connect_swapped(item, "activate", G_CALLBACK(show_log_browser), window);

    return menu;
}

static void
zoom_in(GwyDataWindow *window)
{
    gwy_data_window_set_zoom(window, 1);
}

static void
zoom_out(GwyDataWindow *window)
{
    gwy_data_window_set_zoom(window, -1);
}

static void
reset_zoom(GwyDataWindow *window)
{
    gwy_data_window_set_zoom(window, 10000);
}

static void
realsquare_toggled(GwyAppImageWindow *window, GtkCheckMenuItem *item)
{
    GwyAppImageWindowPrivate *priv = window->priv;
    GwyDataKind data_kind = priv->data_kind;
    gint id = priv->id;
    GwyFileKeyParsed parsed = { .data_kind = data_kind, .id = id, .piece = GWY_FILE_PIECE_REAL_SQUARE, .suffix = NULL };
    GQuark key = gwy_file_form_key(&parsed);
    gboolean active = gtk_check_menu_item_get_active(item);
    if (active)
        gwy_container_set_boolean(GWY_CONTAINER(priv->file), key, TRUE);
    else
        gwy_container_remove(GWY_CONTAINER(priv->file), key);
}

static void
change_mask_color(GwyAppImageWindow *window)
{
    GwyAppImageWindowPrivate *priv = window->priv;
    GwyContainer *data = GWY_CONTAINER(priv->file);

    GwyRGBA rgba;
    GQuark quark = gwy_file_key_image_mask_color(priv->id);
    if (!gwy_container_gis_boxed(data, GWY_TYPE_RGBA, quark, &rgba)) {
        gwy_app_settings_get_default_mask_color(&rgba);
        gwy_container_set_boxed(data, GWY_TYPE_RGBA, quark, &rgba);
    }
    gwy_mask_color_selector_run(NULL, NULL, NULL, GWY_CONTAINER(data), quark);
}

static void
show_meta_browser(GwyAppImageWindow *window)
{
    GwyAppImageWindowPrivate *priv = window->priv;
    if (!priv->meta_widget) {
        priv->meta_widget = gwy_meta_browser_new(priv->file, priv->data_kind, priv->id);
        priv->meta_destroyed_id = g_signal_connect_swapped(priv->meta_widget, "destroy",
                                                           G_CALLBACK(meta_destroyed), window);
    }
    gtk_window_present(GTK_WINDOW(priv->meta_widget));
}

static void
meta_destroyed(GwyAppImageWindow *window)
{
    GwyAppImageWindowPrivate *priv = window->priv;
    g_clear_signal_handler(&priv->meta_destroyed_id, priv->meta_widget);
    priv->meta_widget = NULL;
}

static void
show_log_browser(GwyAppImageWindow *window)
{
    GwyAppImageWindowPrivate *priv = window->priv;
    if (!priv->log_widget) {
        priv->log_widget = gwy_log_browser_new(priv->file, priv->data_kind, priv->id);
        priv->log_destroyed_id = g_signal_connect_swapped(priv->log_widget, "destroy",
                                                          G_CALLBACK(log_destroyed), window);
    }
    gtk_window_present(GTK_WINDOW(priv->log_widget));
}

static void
log_destroyed(GwyAppImageWindow *window)
{
    GwyAppImageWindowPrivate *priv = window->priv;
    g_clear_signal_handler(&priv->log_destroyed_id, priv->log_widget);
    priv->log_widget = NULL;
}

static void
add_infobar(GwyAppImageWindow *window, GtkBox *vbox)
{
    GwyAppImageWindowPrivate *priv = window->priv;
    GwyDataKind data_kind = priv->data_kind;

    if (data_kind == GWY_FILE_IMAGE)
        return;

    priv->infobar = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 0);

    priv->info_label = gtk_label_new(NULL);
    GtkLabel *label = GTK_LABEL(priv->info_label);
    gtk_label_set_xalign(GTK_LABEL(label), 0.0);
    gtk_label_set_ellipsize(GTK_LABEL(label), PANGO_ELLIPSIZE_END);
    gtk_box_pack_start(GTK_BOX(priv->infobar), priv->info_label, TRUE, TRUE, 0);

    gtk_box_pack_start(vbox, priv->infobar, FALSE, FALSE, 0);
    gtk_box_reorder_child(vbox, priv->infobar, 0);
}

static void
update_infobar(GwyAppImageWindow *window)
{
    GwyAppImageWindowPrivate *priv = window->priv;
    if (!priv->infobar || !priv->info_label)
        return;
    GwyDataKind data_kind = priv->data_kind;
    GwyFile *file = priv->file;

    if (!file || !priv->object) {
        gtk_label_set_text(GTK_LABEL(priv->info_label), NULL);
        return;
    }

    gchar *text = NULL;
    if (data_kind == GWY_FILE_VOLUME) {
        GwyBrick *brick = GWY_BRICK(priv->object);
        if (brick) {
            gchar *unit = gwy_unit_get_string(gwy_brick_get_unit_z(brick), GWY_UNIT_FORMAT_MARKUP);
            text = g_strdup_printf(_("Z levels: %d, Z unit: %s"), gwy_brick_get_zres(brick), *unit ? unit : "1");
            g_free(unit);
        }
    }
    else if (data_kind == GWY_FILE_XYZ) {
        GwySurface *surface = GWY_SURFACE(priv->object);
        if (surface)
            text = g_strdup_printf(_("Points: %d"), gwy_surface_get_npoints(surface));
    }
    else if (data_kind == GWY_FILE_CMAP) {
        GwyLawn *lawn = GWY_LAWN(priv->object);
        if (lawn) {
            GString *str = g_string_new(_("Curves:"));
            gint n = gwy_lawn_get_n_curves(lawn);
            for (gint i = 0; i < n; i++) {
                if (i)
                    g_string_append_c(str, ',');
                g_string_append_c(str, ' ');
                const gchar *name = gwy_lawn_get_curve_label(lawn, i);
                g_string_append(str, name ? name : _("Untitled"));
            }

            if ((n = gwy_lawn_get_n_segments(lawn))) {
                g_string_append(str, "   ");
                g_string_append(str, _("Segments:"));
                for (gint i = 0; i < n; i++) {
                    if (i)
                        g_string_append_c(str, ',');
                    g_string_append_c(str, ' ');
                    const gchar *name = gwy_lawn_get_segment_label(lawn, i);
                    g_string_append(str, name ? name : _("Untitled"));
                }
            }
            text = g_string_free(str, FALSE);
        }
    }
    gtk_label_set_text(GTK_LABEL(priv->info_label), text);
    g_free(text);
}

static void
update_window_title(GwyAppImageWindow *window)
{
    GwyAppImageWindowPrivate *priv = window->priv;
    gchar *basename, *dataname = gwy_file_get_display_title(priv->file, priv->data_kind, priv->id);
    const gchar *filename;
    if (gwy_container_gis_string(GWY_CONTAINER(priv->file), gwy_file_key_filename(), &filename))
        basename = g_path_get_basename(filename);
    else
        basename = g_strdup(_("Untitled"));

    gchar *title = g_strdup_printf("%s [%s]", basename, dataname);
    g_free(dataname);
    g_free(basename);
    gwy_data_window_set_data_name(GWY_DATA_WINDOW(window), title);
    g_free(title);
}

static void
xyz_density_map_toggled(GtkCheckMenuItem *toggle, GwyAppImageWindow *window)
{
    window->priv->xyz_is_density = gtk_check_menu_item_get_active(toggle);
    update_xyz_preview(window);
}

static void
maybe_add_xyz_preview(GwyAppImageWindow *window)
{
    GwyAppImageWindowPrivate *priv = window->priv;
    GwySurface *surface = GWY_SURFACE(priv->object);
    GwyContainer *container = GWY_CONTAINER(priv->file);
    GQuark key = gwy_file_key_xyz_picture(priv->id);
    if (gwy_container_contains(container, key)) {
        /* The caller does not do this when calling us. */
        item_changed(container, key, window);
        return;
    }

    GwyField *raster = gwy_field_new(1, 1, 1.0, 1.0, FALSE);
    gint res = GWY_ROUND(sqrt(gwy_surface_get_npoints(surface)));
    res = CLAMP(res, 2, 1024);
    gwy_preview_surface_to_field(surface, raster, res, res, GWY_PREVIEW_SURFACE_FILL);
    gwy_container_pass_object(container, key, raster);
}

static void
update_xyz_preview(GwyAppImageWindow *window)
{
    GwyAppImageWindowPrivate *priv = window->priv;
    GtkWidget *dataview = gwy_data_window_get_data_view(GWY_DATA_WINDOW(window));
    if (!priv->object) {
        /* We might be being destroyed anyway here? */
        gwy_data_view_set_field(GWY_DATA_VIEW(dataview), NULL);
        return;
    }

    g_return_if_fail(GWY_IS_SURFACE(priv->object));
    GwySurface *surface = GWY_SURFACE(priv->object);
    GQuark key = gwy_file_key_xyz_picture(priv->id);
    GwyField *raster = NULL;
    if (!gwy_container_gis_object(GWY_CONTAINER(priv->file), key, &raster)) {
        raster = gwy_field_new(1, 1, 1.0, 1.0, TRUE);
        gwy_container_pass_object(GWY_CONTAINER(priv->file), key, raster);
    }

    GwyPreviewSurfaceFlags flags = GWY_PREVIEW_SURFACE_FILL;
    if (priv->xyz_is_density)
        flags |= GWY_PREVIEW_SURFACE_DENSITY;

    GdkRectangle allocation;
    gtk_widget_get_allocation(dataview, &allocation);
    gwy_preview_surface_to_field(surface, raster, allocation.width, allocation.height, flags);
    gwy_data_view_set_zoom(GWY_DATA_VIEW(dataview), 1.0);
    gwy_field_data_changed(raster);
}

static void
destroyed(GtkWidget *widget)
{
    GwyAppImageWindow *window = GWY_APP_IMAGE_WINDOW(widget);
    GwyAppImageWindowPrivate *priv = window->priv;
    if (priv->meta_widget)
        gtk_widget_destroy(priv->meta_widget);
    if (priv->log_widget)
        gtk_widget_destroy(priv->log_widget);

    GTK_WIDGET_CLASS(parent_class)->destroy(widget);
}

/* FIXME: This should probably logically go somewhere else. */
static gboolean
key_pressed(GtkWidget *widget,
            GdkEventKey *event)
{
    if (event->keyval != GDK_KEY_F3 || (event->state & (GDK_SHIFT_MASK | GDK_CONTROL_MASK | GDK_MOD1_MASK))) {
        return GTK_WIDGET_CLASS(parent_class)->key_press_event(widget, event);
    }

    GwyTool *current_tool = gwy_app_current_tool();
    if (current_tool) {
        if (!gwy_tool_is_visible(current_tool))
            gwy_tool_show(current_tool);
        else
            gwy_tool_hide(current_tool);
    }
    return TRUE;
}

static gboolean
configured(GtkWidget *widget, GdkEventConfigure *event)
{
    GdkRectangle allocation = { .x = event->x, .y = event->y, .width = event->width, .height = event->height };
    save_size(GWY_APP_IMAGE_WINDOW(widget), &allocation);
    return GTK_WIDGET_CLASS(parent_class)->configure_event(widget, event);
}

static void
size_allocated(GtkWidget *widget, GdkRectangle *allocation)
{
    save_size(GWY_APP_IMAGE_WINDOW(widget), allocation);

    GTK_WIDGET_CLASS(parent_class)->size_allocate(widget, allocation);
}

static void
save_size(GwyAppImageWindow *window, const GdkRectangle *allocation)
{
    GwyAppImageWindowPrivate *priv = window->priv;
    /* Someone created us with plain g_object_new(). */
    g_return_if_fail(priv->file);
    GtkWidget *dataview = gwy_data_window_get_data_view(GWY_DATA_WINDOW(window));
    gboolean have_pixels = data_kind_has_pixels(priv->data_kind);
    if (have_pixels) {
        gdouble zoom = gwy_data_view_get_real_zoom(GWY_DATA_VIEW(dataview));
        GwyFileKeyParsed parsed = {
            .id = priv->id, .data_kind = priv->data_kind, .piece = GWY_FILE_PIECE_VIEW, .suffix = "scale"
        };
        gwy_container_set_double(GWY_CONTAINER(priv->file), gwy_file_form_key(&parsed), zoom);
    }
    _gwy_save_widget_screen_size(dataview, allocation, priv->file, priv->data_kind, priv->id, !have_pixels);
}

/* FIXME: Where should it go? */
void
_gwy_save_widget_screen_size(GtkWidget *widget, const GdkRectangle *allocation,
                             GwyFile *file, GwyDataKind data_kind, gint id,
                             gboolean absolute_too)
{
    gdouble scw = gwy_get_screen_width(widget);
    gdouble sch = gwy_get_screen_height(widget);
    gint w, h;
    if (allocation) {
        w = allocation->width;
        h = allocation->height;
    }
    else {
        GdkRectangle alloc;
        gtk_widget_get_allocation(widget, &alloc);
        w = alloc.width;
        h = alloc.height;
    }
    gdouble relsize = MAX(w/scw, h/sch);
    GwyContainer *container = GWY_CONTAINER(file);
    GwyFileKeyParsed parsed = { .id = id, .data_kind = data_kind, .piece = GWY_FILE_PIECE_VIEW };

    parsed.suffix = "relative-size";
    gwy_container_set_double(container, gwy_file_form_key(&parsed), relsize);
    if (absolute_too) {
        parsed.suffix = "width";
        gwy_container_set_int32(container, gwy_file_form_key(&parsed), w);
        parsed.suffix = "height";
        gwy_container_set_int32(container, gwy_file_form_key(&parsed), h);
    }
}

static void
restore_zoom(GwyAppImageWindow *window)
{
    GwyAppImageWindowPrivate *priv = window->priv;
    /* Someone created us with plain g_object_new(). */
    g_return_if_fail(priv->file);
    GwyContainer *container = GWY_CONTAINER(priv->file);

    GwyFileKeyParsed parsed = { .id = priv->id, .data_kind = priv->data_kind, .piece = GWY_FILE_PIECE_VIEW };
    gdouble scale, relsize;

    parsed.suffix = "relative-size";
    if (!gwy_container_gis_double(container, gwy_file_form_key(&parsed), &relsize))
        return;
    parsed.suffix = "scale";
    if (!gwy_container_gis_double(container, gwy_file_form_key(&parsed), &scale))
        return;
    if (scale <= 0.0 || relsize <= 0.0)
        return;

    GtkRequisition req;
    GtkWidget *dataview = gwy_data_window_get_data_view(GWY_DATA_WINDOW(window));
    gtk_widget_get_preferred_size(dataview, NULL, &req);
    gdouble scw = gwy_get_screen_width(GTK_WIDGET(window));
    gdouble sch = gwy_get_screen_height(GTK_WIDGET(window));
    gdouble newrelsize = MAX(scale*req.width/scw, scale*req.height/sch);
    gwy_debug("restoring data window: relsize %g, zoom %g, request %dx%d, newrelsize %g",
              relsize, scale, req.width, req.height, newrelsize);

    /* If the data view will be small we can just apply the saved zoom.  Should it be larger though, we must check if
     * it is not too large and better show it at defaut size than huge. */
    if (newrelsize > 1.2*relsize || newrelsize > 0.85) {
        gwy_debug("window too large for this screen; do not trying to restore the size");
        return;
    }
    gwy_data_view_set_zoom(GWY_DATA_VIEW(dataview), scale);
}

GwyParams*
_gwy_app_image_window_get_params(GwyAppImageWindow *window)
{
    GwyAppImageWindowPrivate *priv = window->priv;
    if (!priv->params)
        priv->params = gwy_params_new();
    return priv->params;
}

static gboolean
data_kind_has_pixels(GwyDataKind data_kind)
{
    return data_kind == GWY_FILE_IMAGE || data_kind == GWY_FILE_VOLUME || data_kind == GWY_FILE_CMAP;
}

/**
 * SECTION: app-image-window
 * @title: GwyAppImageWindow
 * @short_description: Windows for data shown as images
 *
 * #GwyAppImageWindow wraps #GwyDataWindow and updates it to changes in the file.
 **/

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
