/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_case.hpp
 *
 * @brief [LEVEL: beta] @ref diagnostics::unittest::Test_Case interface 
 *
 * $Id: test_case.hpp,v 1.20 2005/06/23 09:54:24 esdentem Exp $
 * 
 * @author Christian Schallhart
 *
 * @test none
 */

#ifndef DIAGNOSTICS__UNITTEST__TEST_CASE_HPP__INCLUDE_GUARD
#define DIAGNOSTICS__UNITTEST__TEST_CASE_HPP__INCLUDE_GUARD

#include <diagnostics/unittest/namespace.hpp>

// used in the interface by value
#include <diagnostics/frame/level.hpp>

// used in the interface by reference
#include <string> 


DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;

// used in the interface by reference
// <diagnostics/unittest/test_data.hpp>
class Test_Data; 

/**
 * @class Test_Case diagnostics/unittest/test_case.hpp
 *
 * @brief The Test_Case interface is used to specify Test_Cases to the
 * test system. 
 *
 * A Test_Case comes with the following data:
 * @arg its name,
 * @arg its timeout in seconds (0 means no timeout at all)
 * @arg whether it is rerunable
 * @arg and the build levels which it is testable at
 * Finally, the Test_Case comes with the @ref run method which is
 * called to run the Test_Case at hand.
 *
 * @nosubgrouping
 */
class Test_Case
{
public:
    /**
     * @throw never
     */
    virtual ~Test_Case();

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Hooks
     * @{
     */
public:
    /**
     * @brief The hook method called by the testing environment to invoke the Test_Case
     * 
     * This method is supposed to terminate within @ref timeout() seconds, and to produce
     * no error in the build levels which are described by @ref is_testable_at().
     * This method is only allowed to be invoked twice within the same process iff
     * @ref is_rerunable() returns true. The latter is useful for testcases which
     * test static entities. 
     *
     * @throw anything
     */
    virtual void run(Test_Data & test_data) const=0;
    // @}
    
    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Accessors
     * @{
     */
public:
    /**
     * @brief returns true for @a level if the Test_Case is supposed
     * to succeed at build level @a level
     *
     * @pre @a level != @ref LEVEL_TEST
     *
     * @throw Test_System_Exception
     */
    virtual bool is_testable_at(Level_t const level) const=0;

    /**
     * @brief the name of the Test_Case
     *
     * @throw never
     */
    virtual ::std::string const & name() const=0;

    /**
     * @brief the timeout of the Test_Case in seconds
     *
     * @throw never
     */
    virtual int timeout() const=0;
    
    /**
     * @brief returns true if the Test_Case can be run more than once
     *
     * @throw never
     */
    virtual bool is_rerunable() const=0;
    // @}
};


UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;


#endif
// vim:ts=4:sw=4
